/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { EMPTY_ARRAY } from './empty';
import { NG_COMP_DEF, NG_DIR_DEF, NG_LOC_ID_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
let _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const type = componentDefinition.type;
        const declaredInputs = {};
        const def = {
            type: type,
            providersResolver: null,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            factory: null,
            template: componentDefinition.template || null,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            hostBindings: componentDefinition.hostBindings || null,
            hostVars: componentDefinition.hostVars || 0,
            hostAttrs: componentDefinition.hostAttrs || null,
            contentQueries: componentDefinition.contentQueries || null,
            declaredInputs: declaredInputs,
            inputs: null,
            outputs: null,
            exportAs: componentDefinition.exportAs || null,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            selectors: componentDefinition.selectors || EMPTY_ARRAY,
            viewQuery: componentDefinition.viewQuery || null,
            features: componentDefinition.features || null,
            data: componentDefinition.data || {},
            // TODO(misko): convert ViewEncapsulation into const enum so that it can be used
            // directly in the next line. Also `None` should be 0 not 2.
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: 'c',
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            setInput: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        const directiveTypes = componentDefinition.directives;
        const feature = componentDefinition.features;
        const pipeTypes = componentDefinition.pipes;
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach((fn) => fn(def));
        def.directiveDefs = directiveTypes ?
            () => (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef) :
            null;
        def.pipeDefs = pipeTypes ?
            () => (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef) :
            null;
        return def;
    });
}
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = () => directives.map(extractDirectiveDef);
    def.pipeDefs = () => pipes.map(extractPipeDef);
}
export function extractDirectiveDef(type) {
    const def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is neither 'ComponentType' or 'DirectiveType'.`);
    }
    return def;
}
export function extractPipeDef(type) {
    const def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is not a 'PipeType'.`);
    }
    return def;
}
export const autoRegisterModuleById = {};
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    const res = {
        type: def.type,
        bootstrap: def.bootstrap || EMPTY_ARRAY,
        declarations: def.declarations || EMPTY_ARRAY,
        imports: def.imports || EMPTY_ARRAY,
        exports: def.exports || EMPTY_ARRAY,
        transitiveCompileScopes: null,
        schemas: def.schemas || null,
        id: def.id || null,
    };
    if (def.id != null) {
        noSideEffects(() => {
            autoRegisterModuleById[def.id] = def.type;
        });
    }
    return res;
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export const ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
export function getNgLocaleIdDef(type) {
    return type[NG_LOC_ID_DEF] || null;
}
//# sourceMappingURL=data:application/json;base64,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