/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { autoRegisterModuleById, getNgModuleDef } from '../render3/definition';
import { maybeUnwrapFn } from '../render3/util/misc_utils';
import { stringify } from '../util/stringify';
/**
 * Map of module-id to the corresponding NgModule.
 * - In pre Ivy we track NgModuleFactory,
 * - In post Ivy we track the NgModuleType
 */
const modules = new Map();
/**
 * Registers a loaded module. Should only be called from generated NgModuleFactory code.
 * @publicApi
 */
export function registerModuleFactory(id, factory) {
    const existing = modules.get(id);
    assertSameOrNotExisting(id, existing && existing.moduleType, factory.moduleType);
    modules.set(id, factory);
}
function assertSameOrNotExisting(id, type, incoming) {
    if (type && type !== incoming) {
        throw new Error(`Duplicate module registered for ${id} - ${stringify(type)} vs ${stringify(type.name)}`);
    }
}
export function registerNgModuleType(ngModuleType) {
    const visited = new Set();
    recurse(ngModuleType);
    function recurse(ngModuleType) {
        // The imports array of an NgModule must refer to other NgModules,
        // so an error is thrown if no module definition is available.
        const def = getNgModuleDef(ngModuleType, /* throwNotFound */ true);
        const id = def.id;
        if (id !== null) {
            const existing = modules.get(id);
            assertSameOrNotExisting(id, existing, ngModuleType);
            modules.set(id, ngModuleType);
        }
        const imports = maybeUnwrapFn(def.imports);
        for (const i of imports) {
            if (!visited.has(i)) {
                visited.add(i);
                recurse(i);
            }
        }
    }
}
export function clearModulesForTest() {
    modules.clear();
}
export function getRegisteredNgModuleType(id) {
    return modules.get(id) || autoRegisterModuleById[id];
}
//# sourceMappingURL=data:application/json;base64,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