/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createTokenForExternalReference, Identifiers } from '../identifiers';
import { LifecycleHooks } from '../lifecycle_reflector';
import * as o from '../output/output_ast';
import { convertValueToOutputAst } from '../output/value_util';
import { ProviderAstType } from '../template_parser/template_ast';
export function providerDef(ctx, providerAst) {
    let flags = 0 /* None */;
    if (!providerAst.eager) {
        flags |= 4096 /* LazyProvider */;
    }
    if (providerAst.providerType === ProviderAstType.PrivateService) {
        flags |= 8192 /* PrivateProvider */;
    }
    if (providerAst.isModule) {
        flags |= 1073741824 /* TypeModuleProvider */;
    }
    providerAst.lifecycleHooks.forEach((lifecycleHook) => {
        // for regular providers, we only support ngOnDestroy
        if (lifecycleHook === LifecycleHooks.OnDestroy ||
            providerAst.providerType === ProviderAstType.Directive ||
            providerAst.providerType === ProviderAstType.Component) {
            flags |= lifecycleHookToNodeFlag(lifecycleHook);
        }
    });
    const { providerExpr, flags: providerFlags, depsExpr } = providerAst.multiProvider ?
        multiProviderDef(ctx, flags, providerAst.providers) :
        singleProviderDef(ctx, flags, providerAst.providerType, providerAst.providers[0]);
    return {
        providerExpr,
        flags: providerFlags,
        depsExpr,
        tokenExpr: tokenExpr(ctx, providerAst.token),
    };
}
function multiProviderDef(ctx, flags, providers) {
    const allDepDefs = [];
    const allParams = [];
    const exprs = providers.map((provider, providerIndex) => {
        let expr;
        if (provider.useClass) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useClass.diDeps);
            expr = ctx.importExpr(provider.useClass.reference).instantiate(depExprs);
        }
        else if (provider.useFactory) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useFactory.diDeps);
            expr = ctx.importExpr(provider.useFactory.reference).callFn(depExprs);
        }
        else if (provider.useExisting) {
            const depExprs = convertDeps(providerIndex, [{ token: provider.useExisting }]);
            expr = depExprs[0];
        }
        else {
            expr = convertValueToOutputAst(ctx, provider.useValue);
        }
        return expr;
    });
    const providerExpr = o.fn(allParams, [new o.ReturnStatement(o.literalArr(exprs))], o.INFERRED_TYPE);
    return {
        providerExpr,
        flags: flags | 1024 /* TypeFactoryProvider */,
        depsExpr: o.literalArr(allDepDefs)
    };
    function convertDeps(providerIndex, deps) {
        return deps.map((dep, depIndex) => {
            const paramName = `p${providerIndex}_${depIndex}`;
            allParams.push(new o.FnParam(paramName, o.DYNAMIC_TYPE));
            allDepDefs.push(depDef(ctx, dep));
            return o.variable(paramName);
        });
    }
}
function singleProviderDef(ctx, flags, providerType, providerMeta) {
    let providerExpr;
    let deps;
    if (providerType === ProviderAstType.Directive || providerType === ProviderAstType.Component) {
        providerExpr = ctx.importExpr(providerMeta.useClass.reference);
        flags |= 16384 /* TypeDirective */;
        deps = providerMeta.deps || providerMeta.useClass.diDeps;
    }
    else {
        if (providerMeta.useClass) {
            providerExpr = ctx.importExpr(providerMeta.useClass.reference);
            flags |= 512 /* TypeClassProvider */;
            deps = providerMeta.deps || providerMeta.useClass.diDeps;
        }
        else if (providerMeta.useFactory) {
            providerExpr = ctx.importExpr(providerMeta.useFactory.reference);
            flags |= 1024 /* TypeFactoryProvider */;
            deps = providerMeta.deps || providerMeta.useFactory.diDeps;
        }
        else if (providerMeta.useExisting) {
            providerExpr = o.NULL_EXPR;
            flags |= 2048 /* TypeUseExistingProvider */;
            deps = [{ token: providerMeta.useExisting }];
        }
        else {
            providerExpr = convertValueToOutputAst(ctx, providerMeta.useValue);
            flags |= 256 /* TypeValueProvider */;
            deps = [];
        }
    }
    const depsExpr = o.literalArr(deps.map(dep => depDef(ctx, dep)));
    return { providerExpr, flags, depsExpr };
}
function tokenExpr(ctx, tokenMeta) {
    return tokenMeta.identifier ? ctx.importExpr(tokenMeta.identifier.reference) :
        o.literal(tokenMeta.value);
}
export function depDef(ctx, dep) {
    // Note: the following fields have already been normalized out by provider_analyzer:
    // - isAttribute, isHost
    const expr = dep.isValue ? convertValueToOutputAst(ctx, dep.value) : tokenExpr(ctx, dep.token);
    let flags = 0 /* None */;
    if (dep.isSkipSelf) {
        flags |= 1 /* SkipSelf */;
    }
    if (dep.isOptional) {
        flags |= 2 /* Optional */;
    }
    if (dep.isSelf) {
        flags |= 4 /* Self */;
    }
    if (dep.isValue) {
        flags |= 8 /* Value */;
    }
    return flags === 0 /* None */ ? expr : o.literalArr([o.literal(flags), expr]);
}
export function lifecycleHookToNodeFlag(lifecycleHook) {
    let nodeFlag = 0 /* None */;
    switch (lifecycleHook) {
        case LifecycleHooks.AfterContentChecked:
            nodeFlag = 2097152 /* AfterContentChecked */;
            break;
        case LifecycleHooks.AfterContentInit:
            nodeFlag = 1048576 /* AfterContentInit */;
            break;
        case LifecycleHooks.AfterViewChecked:
            nodeFlag = 8388608 /* AfterViewChecked */;
            break;
        case LifecycleHooks.AfterViewInit:
            nodeFlag = 4194304 /* AfterViewInit */;
            break;
        case LifecycleHooks.DoCheck:
            nodeFlag = 262144 /* DoCheck */;
            break;
        case LifecycleHooks.OnChanges:
            nodeFlag = 524288 /* OnChanges */;
            break;
        case LifecycleHooks.OnDestroy:
            nodeFlag = 131072 /* OnDestroy */;
            break;
        case LifecycleHooks.OnInit:
            nodeFlag = 65536 /* OnInit */;
            break;
    }
    return nodeFlag;
}
export function componentFactoryResolverProviderDef(reflector, ctx, flags, entryComponents) {
    const entryComponentFactories = entryComponents.map((entryComponent) => ctx.importExpr(entryComponent.componentFactory));
    const token = createTokenForExternalReference(reflector, Identifiers.ComponentFactoryResolver);
    const classMeta = {
        diDeps: [
            { isValue: true, value: o.literalArr(entryComponentFactories) },
            { token: token, isSkipSelf: true, isOptional: true },
            { token: createTokenForExternalReference(reflector, Identifiers.NgModuleRef) },
        ],
        lifecycleHooks: [],
        reference: reflector.resolveExternalReference(Identifiers.CodegenComponentFactoryResolver)
    };
    const { providerExpr, flags: providerFlags, depsExpr } = singleProviderDef(ctx, flags, ProviderAstType.PrivateService, {
        token,
        multi: false,
        useClass: classMeta,
    });
    return { providerExpr, flags: providerFlags, depsExpr, tokenExpr: tokenExpr(ctx, token) };
}
//# sourceMappingURL=data:application/json;base64,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