/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../output/output_ast';
/**
 * Creates an array literal expression from the given array, mapping all values to an expression
 * using the provided mapping function. If the array is empty or null, then null is returned.
 *
 * @param values The array to transfer into literal array expression.
 * @param mapper The logic to use for creating an expression for the array's values.
 * @returns An array literal expression representing `values`, or null if `values` is empty or
 * is itself null.
 */
export function toOptionalLiteralArray(values, mapper) {
    if (values === null || values.length === 0) {
        return null;
    }
    return o.literalArr(values.map(value => mapper(value)));
}
/**
 * Creates an object literal expression from the given object, mapping all values to an expression
 * using the provided mapping function. If the object has no keys, then null is returned.
 *
 * @param object The object to transfer into an object literal expression.
 * @param mapper The logic to use for creating an expression for the object's values.
 * @returns An object literal expression representing `object`, or null if `object` does not have
 * any keys.
 */
export function toOptionalLiteralMap(object, mapper) {
    const entries = Object.keys(object).map(key => {
        const value = object[key];
        return { key, value: mapper(value), quoted: true };
    });
    if (entries.length > 0) {
        return o.literalMap(entries);
    }
    else {
        return null;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvbXBpbGVyL3NyYy9yZW5kZXIzL3BhcnRpYWwvdXRpbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFDSCxPQUFPLEtBQUssQ0FBQyxNQUFNLHlCQUF5QixDQUFDO0FBRTdDOzs7Ozs7OztHQVFHO0FBQ0gsTUFBTSxVQUFVLHNCQUFzQixDQUNsQyxNQUFnQixFQUFFLE1BQWtDO0lBQ3RELElBQUksTUFBTSxLQUFLLElBQUksSUFBSSxNQUFNLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUMxQyxPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsT0FBTyxDQUFDLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQzFELENBQUM7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILE1BQU0sVUFBVSxvQkFBb0IsQ0FDaEMsTUFBMEIsRUFBRSxNQUFrQztJQUNoRSxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUM1QyxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDMUIsT0FBTyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLEtBQUssQ0FBQyxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUMsQ0FBQztJQUNuRCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDdEIsT0FBTyxDQUFDLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQzlCO1NBQU07UUFDTCxPQUFPLElBQUksQ0FBQztLQUNiO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuaW1wb3J0ICogYXMgbyBmcm9tICcuLi8uLi9vdXRwdXQvb3V0cHV0X2FzdCc7XG5cbi8qKlxuICogQ3JlYXRlcyBhbiBhcnJheSBsaXRlcmFsIGV4cHJlc3Npb24gZnJvbSB0aGUgZ2l2ZW4gYXJyYXksIG1hcHBpbmcgYWxsIHZhbHVlcyB0byBhbiBleHByZXNzaW9uXG4gKiB1c2luZyB0aGUgcHJvdmlkZWQgbWFwcGluZyBmdW5jdGlvbi4gSWYgdGhlIGFycmF5IGlzIGVtcHR5IG9yIG51bGwsIHRoZW4gbnVsbCBpcyByZXR1cm5lZC5cbiAqXG4gKiBAcGFyYW0gdmFsdWVzIFRoZSBhcnJheSB0byB0cmFuc2ZlciBpbnRvIGxpdGVyYWwgYXJyYXkgZXhwcmVzc2lvbi5cbiAqIEBwYXJhbSBtYXBwZXIgVGhlIGxvZ2ljIHRvIHVzZSBmb3IgY3JlYXRpbmcgYW4gZXhwcmVzc2lvbiBmb3IgdGhlIGFycmF5J3MgdmFsdWVzLlxuICogQHJldHVybnMgQW4gYXJyYXkgbGl0ZXJhbCBleHByZXNzaW9uIHJlcHJlc2VudGluZyBgdmFsdWVzYCwgb3IgbnVsbCBpZiBgdmFsdWVzYCBpcyBlbXB0eSBvclxuICogaXMgaXRzZWxmIG51bGwuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB0b09wdGlvbmFsTGl0ZXJhbEFycmF5PFQ+KFxuICAgIHZhbHVlczogVFtdfG51bGwsIG1hcHBlcjogKHZhbHVlOiBUKSA9PiBvLkV4cHJlc3Npb24pOiBvLkxpdGVyYWxBcnJheUV4cHJ8bnVsbCB7XG4gIGlmICh2YWx1ZXMgPT09IG51bGwgfHwgdmFsdWVzLmxlbmd0aCA9PT0gMCkge1xuICAgIHJldHVybiBudWxsO1xuICB9XG4gIHJldHVybiBvLmxpdGVyYWxBcnIodmFsdWVzLm1hcCh2YWx1ZSA9PiBtYXBwZXIodmFsdWUpKSk7XG59XG5cbi8qKlxuICogQ3JlYXRlcyBhbiBvYmplY3QgbGl0ZXJhbCBleHByZXNzaW9uIGZyb20gdGhlIGdpdmVuIG9iamVjdCwgbWFwcGluZyBhbGwgdmFsdWVzIHRvIGFuIGV4cHJlc3Npb25cbiAqIHVzaW5nIHRoZSBwcm92aWRlZCBtYXBwaW5nIGZ1bmN0aW9uLiBJZiB0aGUgb2JqZWN0IGhhcyBubyBrZXlzLCB0aGVuIG51bGwgaXMgcmV0dXJuZWQuXG4gKlxuICogQHBhcmFtIG9iamVjdCBUaGUgb2JqZWN0IHRvIHRyYW5zZmVyIGludG8gYW4gb2JqZWN0IGxpdGVyYWwgZXhwcmVzc2lvbi5cbiAqIEBwYXJhbSBtYXBwZXIgVGhlIGxvZ2ljIHRvIHVzZSBmb3IgY3JlYXRpbmcgYW4gZXhwcmVzc2lvbiBmb3IgdGhlIG9iamVjdCdzIHZhbHVlcy5cbiAqIEByZXR1cm5zIEFuIG9iamVjdCBsaXRlcmFsIGV4cHJlc3Npb24gcmVwcmVzZW50aW5nIGBvYmplY3RgLCBvciBudWxsIGlmIGBvYmplY3RgIGRvZXMgbm90IGhhdmVcbiAqIGFueSBrZXlzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gdG9PcHRpb25hbExpdGVyYWxNYXA8VD4oXG4gICAgb2JqZWN0OiB7W2tleTogc3RyaW5nXTogVH0sIG1hcHBlcjogKHZhbHVlOiBUKSA9PiBvLkV4cHJlc3Npb24pOiBvLkxpdGVyYWxNYXBFeHByfG51bGwge1xuICBjb25zdCBlbnRyaWVzID0gT2JqZWN0LmtleXMob2JqZWN0KS5tYXAoa2V5ID0+IHtcbiAgICBjb25zdCB2YWx1ZSA9IG9iamVjdFtrZXldO1xuICAgIHJldHVybiB7a2V5LCB2YWx1ZTogbWFwcGVyKHZhbHVlKSwgcXVvdGVkOiB0cnVlfTtcbiAgfSk7XG5cbiAgaWYgKGVudHJpZXMubGVuZ3RoID4gMCkge1xuICAgIHJldHVybiBvLmxpdGVyYWxNYXAoZW50cmllcyk7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cbn1cbiJdfQ==