(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/imports/src/emitter", ["require", "exports", "tslib", "@angular/compiler", "typescript", "@angular/compiler-cli/src/ngtsc/file_system", "@angular/compiler-cli/src/ngtsc/file_system/src/util", "@angular/compiler-cli/src/ngtsc/util/src/typescript", "@angular/compiler-cli/src/ngtsc/imports/src/find_export"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.UnifiedModulesStrategy = exports.RelativePathStrategy = exports.LogicalProjectStrategy = exports.AbsoluteModuleStrategy = exports.LocalIdentifierStrategy = exports.ReferenceEmitter = exports.ImportFlags = void 0;
    var tslib_1 = require("tslib");
    /**
     * @license
     * Copyright Google LLC All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    var compiler_1 = require("@angular/compiler");
    var ts = require("typescript");
    var file_system_1 = require("@angular/compiler-cli/src/ngtsc/file_system");
    var util_1 = require("@angular/compiler-cli/src/ngtsc/file_system/src/util");
    var typescript_1 = require("@angular/compiler-cli/src/ngtsc/util/src/typescript");
    var find_export_1 = require("@angular/compiler-cli/src/ngtsc/imports/src/find_export");
    /**
     * Flags which alter the imports generated by the `ReferenceEmitter`.
     */
    var ImportFlags;
    (function (ImportFlags) {
        ImportFlags[ImportFlags["None"] = 0] = "None";
        /**
         * Force the generation of a new import when generating a reference, even if an identifier already
         * exists in the target file which could be used instead.
         *
         * This is sometimes required if there's a risk TypeScript might remove imports during emit.
         */
        ImportFlags[ImportFlags["ForceNewImport"] = 1] = "ForceNewImport";
        /**
         * Don't make use of any aliasing information when emitting a reference.
         *
         * This is sometimes required if emitting into a context where generated references will be fed
         * into TypeScript and type-checked (such as in template type-checking).
         */
        ImportFlags[ImportFlags["NoAliasing"] = 2] = "NoAliasing";
        /**
         * Indicates that an import to a type-only declaration is allowed.
         *
         * For references that occur in type-positions, the referred declaration may be a type-only
         * declaration that is not retained during emit. Including this flag allows to emit references to
         * type-only declarations as used in e.g. template type-checking.
         */
        ImportFlags[ImportFlags["AllowTypeImports"] = 4] = "AllowTypeImports";
    })(ImportFlags = exports.ImportFlags || (exports.ImportFlags = {}));
    /**
     * Generates `Expression`s which refer to `Reference`s in a given context.
     *
     * A `ReferenceEmitter` uses one or more `ReferenceEmitStrategy` implementations to produce an
     * `Expression` which refers to a `Reference` in the context of a particular file.
     */
    var ReferenceEmitter = /** @class */ (function () {
        function ReferenceEmitter(strategies) {
            this.strategies = strategies;
        }
        ReferenceEmitter.prototype.emit = function (ref, context, importFlags) {
            var e_1, _a;
            if (importFlags === void 0) { importFlags = ImportFlags.None; }
            try {
                for (var _b = tslib_1.__values(this.strategies), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var strategy = _c.value;
                    var emitted = strategy.emit(ref, context, importFlags);
                    if (emitted !== null) {
                        return emitted;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            throw new Error("Unable to write a reference to " + typescript_1.nodeNameForError(ref.node) + " in " + ref.node.getSourceFile().fileName + " from " + context.fileName);
        };
        return ReferenceEmitter;
    }());
    exports.ReferenceEmitter = ReferenceEmitter;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations by any local `ts.Identifier`s, if
     * such identifiers are available.
     */
    var LocalIdentifierStrategy = /** @class */ (function () {
        function LocalIdentifierStrategy() {
        }
        LocalIdentifierStrategy.prototype.emit = function (ref, context, importFlags) {
            var refSf = typescript_1.getSourceFile(ref.node);
            // If the emitter has specified ForceNewImport, then LocalIdentifierStrategy should not use a
            // local identifier at all, *except* in the source file where the node is actually declared.
            if (importFlags & ImportFlags.ForceNewImport && refSf !== context) {
                return null;
            }
            // If referenced node is not an actual TS declaration (e.g. `class Foo` or `function foo() {}`,
            // etc) and it is in the current file then just use it directly.
            // This is important because the reference could be a property access (e.g. `exports.foo`). In
            // such a case, the reference's `identities` property would be `[foo]`, which would result in an
            // invalid emission of a free-standing `foo` identifier, rather than `exports.foo`.
            if (!typescript_1.isDeclaration(ref.node) && refSf === context) {
                return {
                    expression: new compiler_1.WrappedNodeExpr(ref.node),
                    importedFile: null,
                };
            }
            // A Reference can have multiple identities in different files, so it may already have an
            // Identifier in the requested context file.
            var identifier = ref.getIdentityIn(context);
            if (identifier !== null) {
                return {
                    expression: new compiler_1.WrappedNodeExpr(identifier),
                    importedFile: null,
                };
            }
            else {
                return null;
            }
        };
        return LocalIdentifierStrategy;
    }());
    exports.LocalIdentifierStrategy = LocalIdentifierStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations that come from `node_modules` using
     * an absolute import.
     *
     * Part of this strategy involves looking at the target entry point and identifying the exported
     * name of the targeted declaration, as it might be different from the declared name (e.g. a
     * directive might be declared as FooDirImpl, but exported as FooDir). If no export can be found
     * which maps back to the original directive, an error is thrown.
     */
    var AbsoluteModuleStrategy = /** @class */ (function () {
        function AbsoluteModuleStrategy(program, checker, moduleResolver, reflectionHost) {
            this.program = program;
            this.checker = checker;
            this.moduleResolver = moduleResolver;
            this.reflectionHost = reflectionHost;
            /**
             * A cache of the exports of specific modules, because resolving a module to its exports is a
             * costly operation.
             */
            this.moduleExportsCache = new Map();
        }
        AbsoluteModuleStrategy.prototype.emit = function (ref, context, importFlags) {
            if (ref.bestGuessOwningModule === null) {
                // There is no module name available for this Reference, meaning it was arrived at via a
                // relative path.
                return null;
            }
            else if (!typescript_1.isDeclaration(ref.node)) {
                // It's not possible to import something which isn't a declaration.
                throw new Error("Debug assert: unable to import a Reference to non-declaration of type " + ts.SyntaxKind[ref.node.kind] + ".");
            }
            else if ((importFlags & ImportFlags.AllowTypeImports) === 0 && typescript_1.isTypeDeclaration(ref.node)) {
                throw new Error("Importing a type-only declaration of type " + ts.SyntaxKind[ref.node.kind] + " in a value position is not allowed.");
            }
            // Try to find the exported name of the declaration, if one is available.
            var _a = ref.bestGuessOwningModule, specifier = _a.specifier, resolutionContext = _a.resolutionContext;
            var exports = this.getExportsOfModule(specifier, resolutionContext);
            if (exports === null || !exports.exportMap.has(ref.node)) {
                // TODO(alxhub): make this error a ts.Diagnostic pointing at whatever caused this import to be
                // triggered.
                throw new Error("Symbol " + ref.debugName + " declared in " + typescript_1.getSourceFile(ref.node).fileName + " is not exported from " + specifier + " (import into " + context.fileName + ")");
            }
            var symbolName = exports.exportMap.get(ref.node);
            return {
                expression: new compiler_1.ExternalExpr(new compiler_1.ExternalReference(specifier, symbolName)),
                importedFile: exports.module,
            };
        };
        AbsoluteModuleStrategy.prototype.getExportsOfModule = function (moduleName, fromFile) {
            if (!this.moduleExportsCache.has(moduleName)) {
                this.moduleExportsCache.set(moduleName, this.enumerateExportsOfModule(moduleName, fromFile));
            }
            return this.moduleExportsCache.get(moduleName);
        };
        AbsoluteModuleStrategy.prototype.enumerateExportsOfModule = function (specifier, fromFile) {
            // First, resolve the module specifier to its entry point, and get the ts.Symbol for it.
            var entryPointFile = this.moduleResolver.resolveModule(specifier, fromFile);
            if (entryPointFile === null) {
                return null;
            }
            var exports = this.reflectionHost.getExportsOfModule(entryPointFile);
            if (exports === null) {
                return null;
            }
            var exportMap = new Map();
            exports.forEach(function (declaration, name) {
                exportMap.set(declaration.node, name);
            });
            return { module: entryPointFile, exportMap: exportMap };
        };
        return AbsoluteModuleStrategy;
    }());
    exports.AbsoluteModuleStrategy = AbsoluteModuleStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations via relative paths, provided they're
     * both in the logical project "space" of paths.
     *
     * This is trickier than it sounds, as the two files may be in different root directories in the
     * project. Simply calculating a file system relative path between the two is not sufficient.
     * Instead, `LogicalProjectPath`s are used.
     */
    var LogicalProjectStrategy = /** @class */ (function () {
        function LogicalProjectStrategy(reflector, logicalFs) {
            this.reflector = reflector;
            this.logicalFs = logicalFs;
        }
        LogicalProjectStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            // Compute the relative path from the importing file to the file being imported. This is done
            // as a logical path computation, because the two files might be in different rootDirs.
            var destPath = this.logicalFs.logicalPathOfSf(destSf);
            if (destPath === null) {
                // The imported file is not within the logical project filesystem.
                return null;
            }
            var originPath = this.logicalFs.logicalPathOfSf(context);
            if (originPath === null) {
                throw new Error("Debug assert: attempt to import from " + context.fileName + " but it's outside the program?");
            }
            // There's no way to emit a relative reference from a file to itself.
            if (destPath === originPath) {
                return null;
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                // The target declaration isn't exported from the file it's declared in. This is an issue!
                return null;
            }
            // With both files expressed as LogicalProjectPaths, getting the module specifier as a relative
            // path is now straightforward.
            var moduleName = file_system_1.LogicalProjectPath.relativePathBetween(originPath, destPath);
            return {
                expression: new compiler_1.ExternalExpr({ moduleName: moduleName, name: name }),
                importedFile: destSf,
            };
        };
        return LogicalProjectStrategy;
    }());
    exports.LogicalProjectStrategy = LogicalProjectStrategy;
    /**
     * A `ReferenceEmitStrategy` which constructs relatives paths between `ts.SourceFile`s.
     *
     * This strategy can be used if there is no `rootDir`/`rootDirs` structure for the project which
     * necessitates the stronger logic of `LogicalProjectStrategy`.
     */
    var RelativePathStrategy = /** @class */ (function () {
        function RelativePathStrategy(reflector) {
            this.reflector = reflector;
        }
        RelativePathStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var relativePath = file_system_1.relative(file_system_1.dirname(file_system_1.absoluteFromSourceFile(context)), file_system_1.absoluteFromSourceFile(destSf));
            var moduleName = file_system_1.toRelativeImport(util_1.stripExtension(relativePath));
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            return { expression: new compiler_1.ExternalExpr({ moduleName: moduleName, name: name }), importedFile: destSf };
        };
        return RelativePathStrategy;
    }());
    exports.RelativePathStrategy = RelativePathStrategy;
    /**
     * A `ReferenceEmitStrategy` which uses a `UnifiedModulesHost` to generate absolute import
     * references.
     */
    var UnifiedModulesStrategy = /** @class */ (function () {
        function UnifiedModulesStrategy(reflector, unifiedModulesHost) {
            this.reflector = reflector;
            this.unifiedModulesHost = unifiedModulesHost;
        }
        UnifiedModulesStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                return null;
            }
            var moduleName = this.unifiedModulesHost.fileNameToModuleName(destSf.fileName, context.fileName);
            return {
                expression: new compiler_1.ExternalExpr({ moduleName: moduleName, name: name }),
                importedFile: destSf,
            };
        };
        return UnifiedModulesStrategy;
    }());
    exports.UnifiedModulesStrategy = UnifiedModulesStrategy;
});
//# sourceMappingURL=data:application/json;base64,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