/* This file is part of BmiCalc.
 * BmiCalc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * BmiCalc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
 *********************************************************************/
package com.ei.bmicalc;

import org.junit.Test;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.UUID;
import java.util.stream.Stream;

import android.content.Context;
import android.content.res.Resources;

import static org.junit.Assert.*;

@RunWith(MockitoJUnitRunner.class)
public class TestBmiDataSet{

    private enum Errors{
        BAD_HEADER("file_header_err"),
        BAD_WEIGHT("file_invalid_weight"),
        BAD_BMI("file_invalid_bmi"),
        BAD_DATE("file_invalid_date"),
        BAD_FILE("file_open_err");
        public final String text;
        Errors(String text){
            this.text = text;
        }
    }

    private static String m_dataString;
    private static double[] m_Weights;
    private static double[] m_BMIs;
    private static Date[] m_dates;
    private static Context m_context;

    @BeforeAll
    public static void createDataAndString(){
        m_Weights = new double[]{174.1, 180.2, 170.3};
        m_BMIs = new double[]{10.1, 16.2, 25.0};
        m_dates = new Date[]{
                new Date(101, 7, 1), //year = 1900 + year
                new Date(102, 8, 11),
                new Date(103, 9, 24)
        };
        m_dataString = createDateString(m_dates, m_Weights, m_BMIs);
        m_context = Mockito.mock(Context.class);
        Resources res = Mockito.mock(Resources.class);
        Mockito.when(res.getString(R.string.date)).thenReturn("Date");
        Mockito.when(res.getString(R.string.weight)).thenReturn("Weight");
        Mockito.when(res.getString(R.string.bmi)).thenReturn("BMI");
        Mockito.when(res.getString(R.string.file_header_err)).thenReturn(Errors.BAD_HEADER.text);
        Mockito.when(res.getString(R.string.file_invalid_weight)).thenReturn(Errors.BAD_WEIGHT.text);
        Mockito.when(res.getString(R.string.file_invalid_bmi)).thenReturn(Errors.BAD_BMI.text);
        Mockito.when(res.getString(R.string.file_invalid_date)).thenReturn(Errors.BAD_DATE.text);
        Mockito.when(res.getString(R.string.file_open_err)).thenReturn(Errors.BAD_FILE.text);
        Mockito.when(res.getString(R.string.file_not_writable)).thenReturn("file_not_writable");
        Mockito.when(m_context.getResources()).thenReturn(res);
    }

    private static String createDateString(Date[] dates, double[] weights, double[] bmis){
        StringBuilder sbData = new StringBuilder("Date,Weight,BMI\n");
        for(int i=0; i<weights.length; i++)
        {
            sbData.append(BmiDataSet.SERIALIZATION_DATE_FORMAT.format(dates[i]));
            sbData.append(',');
            sbData.append(weights[i]);
            sbData.append(',');
            sbData.append(bmis[i]);
            sbData.append('\n');
        }
        return sbData.toString();
    }

    private File createTestFile(String content){
        File fleTemp = null;
        PrintWriter pw = null;
        try {
            fleTemp = File.createTempFile(UUID.randomUUID().toString(), ".tmp");
            pw = new PrintWriter(fleTemp.toString());
            pw.print(content);
        }
        catch(IOException e){
            fail(e.getMessage());
        }
        finally{
            closeStreamMayBe(pw);
        }
        fleTemp.deleteOnExit();
        return fleTemp;
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void outputDataCorrect(boolean is_imperial) {
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        for(int i=0; i<m_Weights.length; i++) {
            double weight = is_imperial ? BmiUtils.kgToLb(m_Weights[i]) : m_Weights[i];
            bdSet.addEntry(m_dates[i], weight, m_BMIs[i], is_imperial);
        }
        File fleExpected = createTestFile(m_dataString);
        File fleOut = createTestFile("");
        bdSet.writeCsv(fleOut);
        assertFilesEqual(fleExpected, fleOut);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testDeleteData(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        for(int i=0; i<m_Weights.length; i++) {
            double weight=is_imperial?BmiUtils.kgToLb(m_Weights[i]):m_Weights[i];
            bdSet.addEntry(m_dates[i], weight, m_BMIs[i], is_imperial);
        }
        bdSet.removeEntry(m_dates[1]);
        Date[] dates ={m_dates[0], m_dates[2]};
        double[] weights = {m_Weights[0], m_Weights[2]};
        double[] bmis = {m_BMIs[0], m_BMIs[2]};
        String newData = createDateString(dates, weights, bmis);
        File fleExpected = createTestFile(newData);
        File fleOut = createTestFile("");
        bdSet.writeCsv(fleOut);
        assertFilesEqual(fleExpected, fleOut);
    }

    @Test
    public void testDeleteNonExistingNotRaises(){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        bdSet.removeEntry(m_dates[0]);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testOutputSorted(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        int[] intNewOrder = {1, 0, 2};
        Date[] dtUnexpected = new Date[3];
        double[] dblWtUnexpected = new double[3];
        double[] dblBMIUnexpected = new double[3];
        int k=0;
        for(int i: intNewOrder) {
            double weight = is_imperial?BmiUtils.kgToLb(m_Weights[i]):m_Weights[i];
            bdSet.addEntry(m_dates[i], weight, m_BMIs[i], is_imperial);
            dtUnexpected[k] = m_dates[i];
            dblWtUnexpected[k] = m_Weights[i];
            dblBMIUnexpected[k] = m_BMIs[i];
            k++;
        }
        String strUnexpected = createDateString(dtUnexpected, dblWtUnexpected, dblBMIUnexpected);
        assertNotEquals(strUnexpected, m_dataString);
        File fleExpected = createTestFile(m_dataString);
        File fleOut = createTestFile("");
        bdSet.writeCsv(fleOut);
        assertFilesEqual(fleExpected, fleOut);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testReadAndSave(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        File fleExpected = createTestFile(m_dataString);
        assertNull(bdSet.readData(fleExpected, is_imperial));
        File fleOut = createTestFile("");
        bdSet.writeCsv(fleOut);
        assertFilesEqual(fleExpected, fleOut);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testReadAndSaveStreams(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        File fleExpected = createTestFile(m_dataString);
        InputStream fis = null;
        try {
            fis = new FileInputStream(fleExpected);
        }
        catch(FileNotFoundException e){
            fail("The test file was not created.");
        }
        assertNull(bdSet.readData(fis, is_imperial));
        File fleOut = createTestFile("");
        FileOutputStream os = null;
        try{
            os = new FileOutputStream(fleOut);
        }
        catch(IOException e){
            fail("Unable to create the output file.");
        }
        bdSet.writeCsv(os);
        assertFilesEqual(fleExpected, fleOut);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testNoFileThrows(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        File fleExpected = new File("not_there.csv");
        assertEquals(Errors.BAD_FILE.text, bdSet.readData(fleExpected, is_imperial));
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testBadHeader(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        String strBadHeader = m_dataString.replace("Date", "Time");
        File fleExpected = createTestFile(strBadHeader);
        assertEquals(Errors.BAD_HEADER.text, bdSet.readData(fleExpected, is_imperial));
    }

    /**
     * Create file with the bad field(s), try to readit in and get results.
     * @param date break date.
     * @param weight break weight.
     * @param bmi break BMI.
     * @param is_imperial use Lb instead of kg.
     * @return The status string from BmiDataSet after reading of data.
     */
    private String do_test_bad_field(boolean date, boolean weight, boolean bmi, boolean is_imperial){
        StringBuilder sbData = new StringBuilder("Date,Weight,BMI\n");
        DateFormat df = DateFormat.getDateInstance();
        if(date)
            sbData.append("bad_date");
        else
            sbData.append(df.format(m_dates[0]));
        sbData.append(',');
        if(weight)
            sbData.append("bad_weight");
        else
            sbData.append(m_Weights[0]);
        sbData.append(',');
        if(bmi)
            sbData.append("bad_bmi");
        else
            sbData.append(m_BMIs[0]);
        sbData.append('\n');
        File fleBroken = createTestFile(sbData.toString());
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        return bdSet.readData(fleBroken, is_imperial);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testBadDate(boolean is_imperial){
        String strBadDate = do_test_bad_field(true, false, false, is_imperial);
        assertEquals(Errors.BAD_DATE.text, strBadDate);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testBadWeight(boolean is_imperial){
        String strBadWeight = do_test_bad_field(false, true, false, is_imperial);
        assertEquals(Errors.BAD_WEIGHT.text, strBadWeight);
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testBadBMI(boolean is_imperial){
        String strBadBMI = do_test_bad_field(false, false, true, is_imperial);
        assertEquals(Errors.BAD_BMI.text, strBadBMI);
    }

    @Test
    public void testNonWritableFile(){
        //Test the error returned if file can not be written.
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        assertEquals("file_not_writable", bdSet.writeCsv(new File("not_there/TestNonExistent12678.csv")));
    }

    @ParameterizedTest
    @MethodSource("getBooleans")
    public void testIsEmpty(boolean is_imperial){
        BmiDataSet bdSet = new BmiDataSet(null, m_context);
        assertTrue(bdSet.isEmpty());
        double weight = is_imperial ? BmiUtils.kgToLb(m_Weights[0]) : m_Weights[0];
        bdSet.addEntry(m_dates[0], weight, m_BMIs[0], is_imperial);
        assertFalse(bdSet.isEmpty());
        bdSet.removeEntry(m_dates[0]);
        assertTrue(bdSet.isEmpty());
    }

    private void assertFilesEqual(File first, File second){
        InputStream is1=null, is2=null;
        InputStreamReader isr1=null, isr2=null;
        BufferedReader br1=null, br2=null;
        try{
            is1 = new FileInputStream(first);
            is2 = new FileInputStream(second);
            isr1 = new InputStreamReader(is1);
            isr2 = new InputStreamReader(is2);
            br1 = new BufferedReader(isr1);
            br2 = new BufferedReader(isr2);
            String s1=br1.readLine(), s2 = br2.readLine();
            assertNotNull(s1);
            assertNotNull(s2);
            while(s1!=null || s2!=null){
                assertEquals(s1, s2);
                if(s1!=null)
                    s1 = br1.readLine();
                if(s2!=null)
                    s2 = br2.readLine();
            }

        }
        catch(IOException e){
            fail("The IO exception was thrown: "+e.getMessage());
        }
        finally{
            closeStreamMayBe(is1);
            closeStreamMayBe(is2);
            closeStreamMayBe(isr1);
            closeStreamMayBe(isr2);
            closeStreamMayBe(br1);
            closeStreamMayBe(br2);
        }
    }

    private void closeStreamMayBe(Closeable cl){
        if (cl!=null) {
            try {
                cl.close();
            }
            catch(IOException e){
                //Nothing here.
            }
        }
    }

    private static Stream<Boolean> getBooleans(){
        //return Stream.of();
        Boolean[] bools = {true, false};
        return Arrays.stream(bools);
    }
}
