////////////////////////////////////////////////////////////////////////////
// This file is part of BmiCalc.
// BmiCalc is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// BmiCalc is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
////////////////////////////////////////////////////////////////////////////
package com.ei.bmicalc;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * The class to get and set the weight category resource IDs based on the BMI values.
 */
public class WeightCategories {

    private final Map<WeightCategory, Integer> m_mapCategoryNameID;
    private final Map<WeightCategory, Integer> m_mapCategoryStyleID;

    public enum WeightCategory{
        VERY_SEVERELY_UNDERWEIGHT,
        SEVERELY_UNDERWEIGHT,
        UNDERWEIGHT,
        HEALTHY,
        OVERWEIGHT,
        OBESE,
        SEVERELY_OBESE,
        VERY_SEVERELY_OBESE
    }

    /**
     * Get the weight category based on BMI.
     * @param bmi bmi.
     * @return the text representing weight category.
     */
    public static WeightCategory getCategory(double bmi){
        if(bmi <= 15)
            return WeightCategory.VERY_SEVERELY_UNDERWEIGHT;
        else if (bmi > 15 && bmi <= 16)
            return WeightCategory.SEVERELY_UNDERWEIGHT;
        else if (bmi > 16 && bmi <= 18.5)
            return WeightCategory.UNDERWEIGHT;
        else if (bmi > 18.5 && bmi <=25)
            return WeightCategory.HEALTHY;
        else if (bmi > 25 && bmi <= 30)
            return WeightCategory.OVERWEIGHT;
        else if (bmi > 30 && bmi <= 35)
            return WeightCategory.OBESE;
        else if (bmi > 35 && bmi <= 40)
            return WeightCategory.SEVERELY_OBESE;
        else
            return WeightCategory.VERY_SEVERELY_OBESE;
    }

    /**
     * Create an instance of Weight categories.
     */
    public WeightCategories(){
        Map<WeightCategory, Integer> catNames = new HashMap<>();
        catNames.put(WeightCategory.VERY_SEVERELY_UNDERWEIGHT, R.string.very_sev_underweight);
        catNames.put(WeightCategory.SEVERELY_UNDERWEIGHT, R.string.sev_underweight);
        catNames.put(WeightCategory.UNDERWEIGHT, R.string.underweight);
        catNames.put(WeightCategory.HEALTHY, R.string.healthy);
        catNames.put(WeightCategory.OVERWEIGHT, R.string.overweight);
        catNames.put(WeightCategory.OBESE, R.string.obese);
        catNames.put(WeightCategory.SEVERELY_OBESE, R.string.sev_obese);
        catNames.put(WeightCategory.VERY_SEVERELY_OBESE, R.string.very_sev_obese);
        m_mapCategoryNameID = Collections.unmodifiableMap(catNames);
        Map<WeightCategory, Integer> catStyles = new HashMap<>();
        catStyles.put(WeightCategory.VERY_SEVERELY_UNDERWEIGHT, R.drawable.body_cell_vs_under);
        catStyles.put(WeightCategory.SEVERELY_UNDERWEIGHT, R.drawable.body_cell_s_under);
        catStyles.put(WeightCategory.UNDERWEIGHT, R.drawable.body_cell_under);
        catStyles.put(WeightCategory.HEALTHY, R.drawable.body_cell_healthy);
        catStyles.put(WeightCategory.OVERWEIGHT, R.drawable.body_cell_over);
        catStyles.put(WeightCategory.OBESE, R.drawable.body_cell_obese);
        catStyles.put(WeightCategory.SEVERELY_OBESE, R.drawable.body_cell_s_obese);
        catStyles.put(WeightCategory.VERY_SEVERELY_OBESE, R.drawable.body_cell_vs_obese);
        m_mapCategoryStyleID = Collections.unmodifiableMap(catStyles);
    }

    /**
     * Get the recource ID for the weight category. corresponding to given BMI.
     * @param bmi bmi.
     * @return the text representing weight category.
     */
    public int getCategoryNameID(double bmi){
        return m_mapCategoryNameID.get(getCategory(bmi));
    }

    /**
     * Return the style for the cell, containing category.
     * @param bmi bmi.
     * @return The ID of a style resource to set for a cell, containing category.
     */
    public int getCategoryStyleID(double bmi){
        return m_mapCategoryStyleID.get(getCategory(bmi));
    }

}
