////////////////////////////////////////////////////////////////////////////
// This file is part of BmiCalc.
// BmiCalc is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// BmiCalc is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
////////////////////////////////////////////////////////////////////////////
package com.ei.bmicalc;

import android.os.Bundle;

import androidx.fragment.app.Fragment;

import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import java.text.DecimalFormat;
import java.text.NumberFormat;

/**
 * A simple {@link Fragment} subclass.
 * Use the {@link ImperialHeightFragment#newInstance} factory method to
 * create an instance of this fragment.
 */
public class ImperialHeightFragment extends Fragment {

    private EditText m_txtHeightCm;
    private EditText m_txtHeightFeet;
    private EditText m_txtHeightInch;
    private TextView m_lblCm;
    private TextView m_lblFeet;
    private TextView m_lblInch;

    private boolean m_blnIsImperial;
    private final NumberFormat m_nfIntFormatter = new DecimalFormat("#");


    public ImperialHeightFragment() {
        // Required empty public constructor
    }

    /**
     * Use this factory method to create a new instance of
     * this fragment using the provided parameters.
     * @return A new instance of fragment ImperialHeightFragment.
     */
    public static ImperialHeightFragment newInstance() {
        return new ImperialHeightFragment();
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        // Inflate the layout for this fragment
        View vw =  inflater.inflate(R.layout.fragment_height, container, false);
        m_txtHeightCm = vw.findViewById(R.id.txt_cm);
        m_txtHeightFeet = vw.findViewById(R.id.txt_feet);
        m_txtHeightInch = vw.findViewById(R.id.txt_inch);
        m_lblCm = vw.findViewById(R.id.lbl_cm);
        m_lblFeet = vw.findViewById(R.id.lbl_feet);
        m_lblInch = vw.findViewById(R.id.lbl_inch);
        return vw;
    }

    /**
     * Set the measurement system.
     * @param is_imperial if true the metric system is imperial/
     */
    public void setMetricType(boolean is_imperial){
        if (is_imperial!=m_blnIsImperial) {
            double height = getHeight();
            height = is_imperial?BmiUtils.cmToIn(height):BmiUtils.inToCm(height);
            int intMetricVisibility = is_imperial ? View.GONE : View.VISIBLE;
            int intImperialVisibility = is_imperial ? View.VISIBLE : View.GONE;
            //Metric views.
            m_txtHeightCm.setVisibility(intMetricVisibility);
            m_lblCm.setVisibility(intMetricVisibility);
            //Imperial views.
            m_txtHeightFeet.setVisibility(intImperialVisibility);
            m_txtHeightInch.setVisibility(intImperialVisibility);
            m_lblFeet.setVisibility(intImperialVisibility);
            m_lblInch.setVisibility(intImperialVisibility);
            m_blnIsImperial = is_imperial;
            updateUI(height);
        }
    }

    /**
     * Return the height in cm or inches.
     * @return the height.
     */
    public double getHeight(){
        if(m_blnIsImperial){
            double inches = 0;
            try{
                if(!m_txtHeightFeet.getText().toString().isEmpty())
                    inches += BmiUtils.feetToInches(Double.parseDouble(m_txtHeightFeet.getText().toString()));
                if(!m_txtHeightInch.getText().toString().isEmpty())
                    inches += Double.parseDouble(m_txtHeightInch.getText().toString());
            }
            catch(NumberFormatException e){
                Toast.makeText(getContext(), getString(R.string.height_type), Toast.LENGTH_SHORT).show();
            }
            return inches;
        }
        else{
            double cm = 0;
            try{
                if(!m_txtHeightCm.getText().toString().isEmpty())
                    cm = Double.parseDouble(m_txtHeightCm.getText().toString());
            }
            catch(NumberFormatException e){
                Toast.makeText(getContext(), getString(R.string.height_type), Toast.LENGTH_SHORT).show();
            }
            return cm;
        }
    }

    /**
     * Set the height in cm or inches to the control.
     * @param height The height.
     */
    public void setHeight(int height, boolean is_imperial){
        if(is_imperial!=m_blnIsImperial){
            setMetricType(is_imperial);
        }
        updateUI(height);
    }

    /**
     * Update user interface and show the correct views.
     * @param height the height read from the text fields.
     */
    private void updateUI(double height){
        if(height==0)
        {
            m_txtHeightFeet.setText("");
            m_txtHeightInch.setText("");
            m_txtHeightCm.setText("");
        }
        else {
            if (m_blnIsImperial) {
                //int will implicitly return floor of height in inchec
                m_txtHeightFeet.setText(m_nfIntFormatter.format((int)BmiUtils.inchToFeet(height)));
                m_txtHeightInch.setText(m_nfIntFormatter.format(height % BmiUtils.INCHES_IN_FEET));
            } else {
                m_txtHeightCm.setText(m_nfIntFormatter.format(height));
            }
        }
    }
}