////////////////////////////////////////////////////////////////////////////
// This file is part of BmiCalc.
// BmiCalc is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// BmiCalc is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with BmiCalc.  If not, see <https://www.gnu.org/licenses/>.
////////////////////////////////////////////////////////////////////////////
package com.ei.bmicalc;

import android.content.Context;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * The class which can store and read the data on BMI.
 */
public class BmiDataSet {
    public static final DateFormat SERIALIZATION_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd", Locale.US);

    private final String SEPARATOR = ",";
    private final TableLayout m_tableLayout;
    private final Context m_context;
    private final Map<Date, BMIRecord> m_dataMap = new HashMap<>();
    private final Map<Date, View> m_viewMap = new HashMap<>();
    private final LayoutInflater m_inflater;
    private final DateFormat m_dateFormat = DateFormat.getDateInstance();
    private final WeightCategories m_wCat = new WeightCategories();
    private final NumberFormat m_ftWeight = new DecimalFormat("###.#");
    private final NumberFormat m_ftBMI = new DecimalFormat("###.##");

    public enum TableCols{
        DATE(0),
        WEIGHT(1),
        BMI(2),
        CATEGORY(3);
        public final int order;
        TableCols(int ord){
            order = ord;
        }
    }

    /**
     * Create the BmiDataSet object.
     * @param layout The table layout to be used to output data.
     * @param context the context to be used to take the resources from.
     */
    public BmiDataSet(@Nullable TableLayout layout, @NonNull Context context){
        m_tableLayout = layout;
        m_context = context;
        if(m_tableLayout!=null)
            m_inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE );
        else
            m_inflater = null;
    }

    /**
     * Consume the data from the file.
     * @param csv the file with the data on weight.
     * @param is_imperial set to True is weight is measured in Lb.
     * @return null if the there is no error or error string otherwise.
     */
    public @Nullable String readData(@NonNull File csv, boolean is_imperial){

        BufferedReader br;
        FileReader fr;
        try{
            fr = new FileReader(csv);
            br = new BufferedReader(fr);
        }
        catch(IOException e)
        {
            return m_context.getResources().getString(R.string.file_open_err);
        }
        String err = readData(br, is_imperial);
        closeStreamMayBe(fr);
        return err;
    }

    /**
     * The mrthod allowing to read in the data from the input stream.
     * @param is The input stream to read the data from.
     * @param is_imperial set to True is weight is measured in Lb.
     * @return null if the there is no error or error string otherwise.
     */
    public @Nullable String readData(@NonNull InputStream is, boolean is_imperial){
        BufferedReader br;
        InputStreamReader isr = new InputStreamReader(is);
        br = new BufferedReader(isr);
        String err = readData(br, is_imperial);
        closeStreamMayBe(isr);
        return err;
    }

    /**
     * Consume the data from the file.
     * @param br the buffered reader to be used to read in the data.
     * @param is_imperial set to True is weight is measured in Lb.
     * @return null if the there is no error or error string otherwise.
     */
    private @Nullable String readData(@NonNull BufferedReader br, boolean is_imperial){
        try{
            String s = br.readLine();
            if(s==null)
                return m_context.getResources().getString(R.string.file_empty);
            List<String> lstHeader = Arrays.asList(s.split(SEPARATOR));
            int intDate = lstHeader.indexOf(m_context.getResources().getString(R.string.date));
            int intWeight = lstHeader.indexOf(m_context.getResources().getString(R.string.weight));
            int intBMI = lstHeader.indexOf(m_context.getResources().getString(R.string.bmi));
            if (intDate == -1 || intWeight == -1 || intBMI == -1){
                return m_context.getResources().getString(R.string.file_header_err);
            }
            int intMinFields = Math.max(intDate, Math.max(intWeight, intBMI)) + 1;
            while((s = br.readLine()) != null){
                String[] values = s.split(SEPARATOR);
                if(values.length >= intMinFields){
                    double weight, bmi;
                    Date dtDate;
                    try{
                        weight = Double.parseDouble(values[intWeight]);
                    }
                    catch(NumberFormatException e) {
                        return m_context.getResources().getString(R.string.file_invalid_weight);
                    }
                    try{
                        bmi = Double.parseDouble(values[intBMI]);
                    }
                    catch(NumberFormatException e){
                        return m_context.getResources().getString(R.string.file_invalid_bmi);
                    }
                    try{
                        dtDate = SERIALIZATION_DATE_FORMAT.parse(values[intDate]);
                    }
                    catch(ParseException e) {
                        return m_context.getResources().getString(R.string.file_invalid_date);
                    }
                    // We save the data always in kg, because we do not store metric type in the file.
                    addEntry(dtDate, weight, bmi, false);
                }
            }
            // Now, when data are loaded, we are setting the correct metric system.
            if(is_imperial){
                setMetricType(is_imperial);
            }
        }
        catch(IOException e){
            return m_context.getResources().getString(R.string.file_open_err);
        }
        finally{
            closeStreamMayBe(br);
        }
        return null;
    }

    /**
     * Return the unmodifiable map with the BMI records.
     * @return unmodifiable map.
     */
    public Bundle getData(){
        Bundle bdData = new Bundle();
        String[] strDates = new String[m_dataMap.size()];
        double[] dblBmis = new double[m_dataMap.size()];
        double[] dblWeights = new double[m_dataMap.size()];
        int i = 0;
        for(Date dt: m_dataMap.keySet()){
            BMIRecord rc = m_dataMap.get(dt);
            if(rc!=null) {
                strDates[i] = SERIALIZATION_DATE_FORMAT.format(rc.date);
                dblBmis[i] = rc.BMI;
                dblWeights[i] = rc.weight;
                i++;
            }
        }
        bdData.putStringArray(BmiDataContract.PAST_DATES.value, strDates);
        bdData.putDoubleArray(BmiDataContract.PAST_BMIS.value, dblBmis);
        bdData.putDoubleArray(BmiDataContract.PAST_WEIGHTS.value, dblWeights);
        return bdData;
    }

    /**
     * Set the data and draw the table if possible.
     * @param data The bundle with data about BMI to be added to the table.
     * @param is_imperial set to True is weight is measured in Lb.
     */
    public void setData(@Nullable Bundle data, boolean is_imperial){
        if(data==null)
            return;
        String[] strDates = data.getStringArray(BmiDataContract.PAST_DATES.value);
        double[] dblBmis = data.getDoubleArray(BmiDataContract.PAST_BMIS.value);
        double[] dblWeights = data.getDoubleArray(BmiDataContract.PAST_WEIGHTS.value);
        if (dblBmis==null || dblWeights==null || strDates==null){
            return;
        }
        try {
            for (int i=0; i<strDates.length; i++) {
                Date dtPast = SERIALIZATION_DATE_FORMAT.parse(strDates[i]);
                addEntry(dtPast, dblWeights[i], dblBmis[i], is_imperial);
            }
        }
        catch(ParseException e){
            //Nothing here.
            //We do not expect non parsable dates.
        }
    }

    /**
     * Set the type of a metric and update it in the UI.
     * @param is_imperial set to True is weight is measured in Lb.
     */
    public void setMetricType(boolean is_imperial){
        for(Date dt:m_viewMap.keySet()){
            TableRow row  = (TableRow)m_viewMap.get(dt);
            BMIRecord recFotDt = m_dataMap.get(dt);
            if(row!=null && recFotDt!=null){
                double weight = is_imperial?BmiUtils.kgToLb(recFotDt.weight): recFotDt.weight;
                TextView txtWeight = (TextView)row.getChildAt(TableCols.WEIGHT.order);
                txtWeight.setText(m_ftWeight.format(weight));
            }
        }
    }
    /**
     * Add the entry with BMI for the specific date.
     * @param date the date of measurement.
     * @param weight the weight.
     * @param BMI the BMI to be stored.
     * @param is_imperial set to True is weight is measured in Lb.
     */
    public void addEntry(@NonNull Date date, double weight, double BMI, boolean is_imperial){
        if(is_imperial){
            weight = BmiUtils.lbToKg(weight);
        }
        m_dataMap.put(date, new BMIRecord(date, weight, BMI));
        if(m_tableLayout != null && m_context != null){
            //Add the entry to the table in the UI.
            boolean blnEntered = false;
            String strDate = m_dateFormat.format(date);
            for(int i=0; i<m_tableLayout.getChildCount(); i++){
                TableRow trChild = (TableRow)m_tableLayout.getChildAt(i);
                String strCurrentDate = ((TextView)trChild.getChildAt(TableCols.DATE.order)).getText().toString();
                if(strDate.equals(strCurrentDate)){
                    //Reuse the old Views to enter the new data.
                    ((TextView)trChild.getChildAt(TableCols.WEIGHT.order)).setText(m_ftWeight.format(weight));
                    ((TextView)trChild.getChildAt(TableCols.BMI.order)).setText(m_ftBMI.format(BMI));
                    ((TextView)trChild.getChildAt(TableCols.CATEGORY.order)).setText(m_context.getString(m_wCat.getCategoryNameID(BMI)));
                    trChild.getChildAt(TableCols.CATEGORY.order).setBackgroundResource(m_wCat.getCategoryStyleID(BMI));
                    blnEntered = true;
                    break;
                }
                Date currentDate = null;
                try {
                    currentDate = m_dateFormat.parse(strCurrentDate);
                }
                catch(ParseException e){
                    //Nothing here. We have populated this field in compliance with date format.
                }
                if(currentDate != null && date.before(currentDate)){
                    //Create a new row and insert it into the table.
                    addView(date, weight, BMI, i);
                    blnEntered = true;
                    break;
                }
            }
            if(!blnEntered){
                addView(date, weight, BMI, -1);
            }
        }
    }

    /**
     * Draw the views. Used after the phone was flipped.
     */
    public void redraw(){
        for(View vw:m_viewMap.values())
            m_tableLayout.removeView(vw);
        List<Date> lstDates = new ArrayList<>(m_viewMap.keySet());
        Collections.sort(lstDates);
        for(Date dt:lstDates){
            m_tableLayout.addView(m_viewMap.get(dt));
        }
    }

    private void addView(Date date, double weight, double BMI, int index){
        View vwNew = getNewRow(date, weight, BMI);
        m_viewMap.put(date, vwNew);
        if(index==-1)
            m_tableLayout.addView(vwNew);
        else
            m_tableLayout.addView(vwNew, index);

    }
    /**
     * Generate a new row with a meassurement
     * @param date the date the measurements were taken.
     * @param weight the weight at the date.
     * @param BMI the BMI at the date.
     * @return the newly created row.
     */
    private TableRow getNewRow(@NonNull Date date, double weight, double BMI){
        TableRow rwNew = new TableRow(m_tableLayout.getContext());
        TextView lblDate = (TextView)m_inflater.inflate(R.layout.body_cell, rwNew, false);
        lblDate.setText(m_dateFormat.format(date));
        lblDate.setTextAppearance(m_context, R.style.table_body);
        TextView lblWeight = (TextView)m_inflater.inflate(R.layout.body_cell, rwNew, false);
        lblWeight.setText(m_ftWeight.format(weight));
        lblWeight.setTextAppearance(m_context, R.style.table_body);
        TextView lblBMI = (TextView)m_inflater.inflate(R.layout.body_cell, rwNew, false);
        lblBMI.setText(m_ftBMI.format(BMI));
        lblBMI.setTextAppearance(m_context, R.style.table_body);
        TextView lblCat = (TextView)m_inflater.inflate(R.layout.body_cell, rwNew, false);
        lblCat.setText(m_context.getString(m_wCat.getCategoryNameID(BMI)));
        lblCat.setBackgroundResource(m_wCat.getCategoryStyleID(BMI));
        rwNew.addView(lblDate);
        rwNew.addView(lblWeight);
        rwNew.addView(lblBMI);
        rwNew.addView(lblCat);
        rwNew.setLayoutParams(new TableRow.LayoutParams(TableRow.LayoutParams.MATCH_PARENT));
        return rwNew;
    }

    /**
     * Remove all records in the table.
     */
    public void clearTable(){
        m_dataMap.clear();
        for(View view: m_viewMap.values())
            m_tableLayout.removeView(view);
        m_viewMap.clear();
    }
    /**
     * Remove the entry from the data table.
     * @param date the date for which the entry needs to be removed.
     */
    public void removeEntry(@NonNull Date date){
        if(m_dataMap.containsKey(date)) {
            m_dataMap.remove(date);
            if (m_tableLayout!=null) {
                //Create a new row and insert it into the table.
                m_tableLayout.removeView(m_viewMap.get(date));
                m_viewMap.remove(date);
            }
        }
    }

    /**
     * The convenience method to create the file.
     * @param out the file to output the data table.
     * @return The string with error, ot null if file was written successfully.
     */
    public @Nullable String writeCsv(@NonNull File out){
        FileOutputStream fosOut;
        try{
            fosOut = new FileOutputStream(out);
        }
        catch(IOException ex){
            return m_context.getResources().getString(R.string.file_not_writable);
        }
        return writeCsv(fosOut);
    }
    /**
     * Write the table of results to the csv file.
     * @param out the file to write the data on BMI to.
     * @return The string with error, ot null if file was written successfully.
     */
    public @Nullable String writeCsv(@NonNull FileOutputStream out){
        try{
            List<Date> dates = new ArrayList<>(m_dataMap.keySet());
            out.write(m_context.getResources().getString(R.string.date).getBytes());
            out.write(SEPARATOR.getBytes());
            out.write(m_context.getResources().getString(R.string.weight).getBytes());
            out.write(SEPARATOR.getBytes());
            out.write(m_context.getResources().getString(R.string.bmi).getBytes());
            out.write((byte)'\n');
            Collections.sort(dates);
            for(Date dt: dates){
                BMIRecord bmiCurrent = m_dataMap.get(dt);
                if(bmiCurrent!=null) {
                    out.write(SERIALIZATION_DATE_FORMAT.format(bmiCurrent.date).getBytes());
                    out.write(SEPARATOR.getBytes());
                    out.write(Double.toString(bmiCurrent.weight).getBytes());
                    out.write(SEPARATOR.getBytes());
                    out.write(Double.toString(bmiCurrent.BMI).getBytes());
                    out.write((byte) '\n');
                }
            }
        }
        catch(IOException e){
            return m_context.getResources().getString(R.string.file_not_writable);
        }
        finally{
            closeStreamMayBe(out);
        }
        return null;
    }
    /**
     * Close the potentially existing closeable.
     * @param cl the closeable to be closed.
     */
    private void closeStreamMayBe(@Nullable Closeable cl){
        if(cl != null) {
            try {
                cl.close();
            }
            catch(IOException e){
                //Nothing here.
            }
        }
    }

    /**
     * The inner class to store a single BMI record.
     */
    static class BMIRecord{
        private final Date date;
        private final double weight;
        private final double BMI;

        /**
         * Constructor.
         * @param date the date the measurements were taken.
         * @param weight the weight at the date.
         * @param BMI the BMI at the date.
         */
        BMIRecord(Date date, double weight, double BMI){
            this.date = date;
            this.weight = weight;
            this.BMI = BMI;
        }
    }

    /**
     * Return if the data holder is empty.
     * @return true, if the holder is empty.
     */
    public boolean isEmpty(){
        return m_dataMap.isEmpty();
    }
}
