"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _reactNative = require("react-native");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function convertErrors(errs) {
  if (!errs) {
    return null;
  }

  return Array.isArray(errs) ? errs.filter(e => !!e) : [errs];
}

class LegacyAsyncStorage {
  constructor() {
    _defineProperty(this, "_asyncStorageNativeModule", void 0);

    this._asyncStorageNativeModule = _reactNative.NativeModules.RNC_AsyncSQLiteDBStorage || _reactNative.NativeModules.RNCAsyncStorage || _reactNative.NativeModules.PlatformLocalStorage;

    if (!this._asyncStorageNativeModule) {
      throw new Error('[@RNC/AsyncStorage]: NativeModule: AsyncStorage is null.');
    }
  }

  async getSingle(key, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiGet([key], function (errors, result) {
        const value = result && result[0] && result[0][1] || null;
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve(value);
        }
      });
    });
  }

  async setSingle(key, value, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiSet([[key, value]], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async getMany(keys, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiGet(keys, function (errors, result) {
        const value = result.reduce((acc, current) => {
          const key = current[0];
          const val = current[1] || null;
          return _objectSpread({}, acc, {
            [key]: val
          });
        }, {});
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve(value);
        }
      });
    });
  }

  async setMany(values, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      const valuesArray = values.map(entry => {
        return [Object.keys(entry)[0], entry];
      });

      this._asyncStorageNativeModule.multiSet([valuesArray], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async removeSingle(key, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiRemove([key], function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async removeMany(keys, opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.multiRemove(keys, function (errors) {
        const errs = convertErrors(errors);

        if (errs && errs.length) {
          reject(errs[0]);
        } else {
          resolve();
        }
      });
    });
  }

  async getKeys(opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.getAllKeys(function (errors, keys) {
        const err = convertErrors(errors);

        if (err && err.length) {
          reject(err[0]);
        } else {
          resolve(keys);
        }
      });
    });
  }

  async dropStorage(opts) {
    if (opts) {// noop
    }

    return new Promise((resolve, reject) => {
      this._asyncStorageNativeModule.clear(function (error) {
        const err = convertErrors(Array.isArray(error) ? error : [error]);

        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

} // type MyModel = {
//   user: {
//     name: string;
//   };
//   preferences: {
//     hour: boolean | null;
//     hair: string;
//   };
//   isEnabled: boolean;
// };
// async function xxx() {
//   const a = new LegacyAsyncStorage<MyModel>();
//
//   const x = await a.getSingle('preferences');
//
//   x.hour;
//
//   const all = await a.getMany(['user', 'isEnabled']);
//
//   all.user;
//
//   await a.setMany([
//     {user: {name: 'Jerry'}},
//     {isEnabled: false},
//     {
//       preferences: {
//         hour: true,
//         hair: 'streight',
//       },
//     },
//   ]);
// }


exports.default = LegacyAsyncStorage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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