"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _defaults = require("./defaults");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AsyncStorage {
  constructor(storageBackend, asOptions) {
    _defineProperty(this, "_backend", void 0);

    _defineProperty(this, "_config", void 0);

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "error", void 0);

    this._backend = storageBackend;
    this._config = asOptions; // off by default

    this.log = _defaults.noop;
    this.error = _defaults.noop;

    if (this._config.logger) {
      this.log = typeof this._config.logger === 'function' ? this._config.logger : _defaults.simpleLogger;
    }

    if (this._config.errorHandler) {
      this.error = typeof this._config.errorHandler === 'function' ? this._config.errorHandler : _defaults.simpleErrorHandler;
    }
  }

  async get(key, opts = null) {
    let value = null;

    try {
      this.log({
        action: 'read-single',
        key: key
      });
      value = await this._backend.getSingle(key, opts);
    } catch (e) {
      this.error(e);
    }

    return value;
  }

  async set(key, value, opts = null) {
    try {
      this.log({
        action: 'save-single',
        key: key,
        value
      });
      await this._backend.setSingle(key, value, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async getMultiple(keys, opts = null) {
    let values = {};

    try {
      this.log({
        action: 'read-many',
        key: keys
      });
      values = await this._backend.getMany(keys, opts);
    } catch (e) {
      this.error(e);
    }

    return values;
  }

  async setMultiple(keyValues, opts = null) {
    try {
      this.log({
        action: 'save-many',
        value: keyValues
      });
      await this._backend.setMany(keyValues, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async remove(key, opts = null) {
    try {
      this.log({
        action: 'delete-single',
        key: key
      });
      await this._backend.removeSingle(key, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async removeMultiple(keys, opts = null) {
    try {
      this.log({
        action: 'delete-many',
        key: keys
      });
      await this._backend.removeMany(keys, opts);
    } catch (e) {
      this.error(e);
    }
  }

  async getKeys(opts = null) {
    let keys = [];

    try {
      this.log({
        action: 'keys'
      });
      keys = await this._backend.getKeys(opts);
    } catch (e) {
      this.error(e);
    }

    return keys;
  }

  async clearStorage(opts = null) {
    try {
      this.log({
        action: 'drop'
      });
      await this._backend.dropStorage(opts);
    } catch (e) {
      this.error(e);
    }
  } // todo: think how we could provide additional functions through AS, without returning the instance
  // some kind of extension-like functionality


  instance() {
    return this._backend;
  }

}

var _default = AsyncStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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