/*
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the LICENSE
 * file in the root directory of this source tree.
 */
#include "hermes/VM/HeapSnapshot.h"
#include "TestHelpers.h"
#include "gtest/gtest.h"
#include "hermes/VM/CellKind.h"
#include "hermes/VM/GC.h"
#include "hermes/VM/GCPointer-inline.h"
#include "hermes/VM/HermesValue.h"
#include "hermes/VM/SymbolID.h"

#include "llvm/Support/raw_ostream.h"

using namespace hermes::vm;

// Only the main NCGen needs to support snapshots
#ifdef HERMESVM_GC_NONCONTIG_GENERATIONAL

namespace hermes {
namespace unittest {
namespace heapsnapshottest {

// Forward declaration to allow IsGCObject.
struct DummyObject;

} // namespace heapsnapshottest
} // namespace unittest

namespace vm {
template <>
struct IsGCObject<unittest::heapsnapshottest::DummyObject>
    : public std::true_type {};
} // namespace vm

namespace unittest {
namespace heapsnapshottest {

struct DummyObject final : public GCCell {
  static const VTable vt;
  GCPointer<DummyObject> other;
  const uint32_t x;
  const uint32_t y;
  GCHermesValue hvBool;
  GCHermesValue hvDouble;
  GCHermesValue hvUndefined;
  GCHermesValue hvEmpty;
  GCHermesValue hvNative;
  GCHermesValue hvNull;

  DummyObject(GC *gc) : GCCell(gc, &vt), other(), x(1), y(2) {
    hvBool.setNonPtr(HermesValue::encodeBoolValue(true));
    hvDouble.setNonPtr(HermesValue::encodeNumberValue(3.14));
    hvNative.setNonPtr(HermesValue::encodeNativeValue(0xE));
    hvUndefined.setNonPtr(HermesValue::encodeUndefinedValue());
    hvEmpty.setNonPtr(HermesValue::encodeEmptyValue());
    hvNull.setNonPtr(HermesValue::encodeNullValue());
  }

  void setPointer(DummyRuntime &rt, DummyObject *obj) {
    other.set(&rt, obj, &rt.gc);
  }

  static DummyObject *create(DummyRuntime &runtime) {
    return new (runtime.alloc(sizeof(DummyObject)))
        DummyObject(&runtime.getHeap());
  }

  static bool classof(const GCCell *cell) {
    return cell->getKind() == CellKind::UninitializedKind;
  }
};
const VTable DummyObject::vt{CellKind::UninitializedKind, sizeof(DummyObject)};

static void DummyObjectBuildMeta(const GCCell *cell, Metadata::Builder &mb) {
  const auto *self = static_cast<const DummyObject *>(cell);
  mb.addField("HermesBool", &self->hvBool);
  mb.addField("HermesDouble", &self->hvDouble);
  mb.addField("HermesUndefined", &self->hvUndefined);
  mb.addField("HermesEmpty", &self->hvEmpty);
  mb.addField("HermesNative", &self->hvNative);
  mb.addField("HermesNull", &self->hvNull);
  mb.addField("other", &self->other);
}

static MetadataTableForTests getMetadataTable() {
  static const Metadata storage[] = {
      buildMetadata(CellKind::UninitializedKind, DummyObjectBuildMeta)};
  return MetadataTableForTests(storage);
}

TEST(HeapSnapshotTest, SnapshotTest) {
  auto runtime = DummyRuntime::create(
      getMetadataTable(),
      GCConfig::Builder()
          .withInitHeapSize(1024)
          .withMaxHeapSize(1024 * 100)
          .build());
  DummyRuntime &rt = *runtime;
  auto &gc = rt.gc;
  GCScope gcScope(&rt);

  auto dummy = rt.makeHandle(DummyObject::create(rt));
  auto *dummy2 = DummyObject::create(rt);
  dummy->setPointer(rt, dummy2);

  std::string result("");
  llvm::raw_string_ostream str(result);
  gc.collect();
  gc.createSnapshot(str, true);
  str.flush();

  ASSERT_FALSE(result.empty());

  const auto blockSize = dummy->getAllocatedSize();

  std::ostringstream stream;

  stream
      << "{"
      << "\"snapshot\":{"
      << "\"meta\":{"
      << "\"node_fields\":[\"type\",\"name\",\"id\",\"self_size\",\"edge_count\",\"trace_node_id\"],"
      << R"("node_types":[["hidden","array","string","object","code","closure","regexp","number","native","synthetic","concatenated string","sliced string","symbol","bigint"])"
      << ",\"string\",\"number\",\"number\",\"number\",\"number\"],"
      << "\"edge_fields\":[\"type\",\"name_or_index\",\"to_node\"],"
      << "\"edge_types\":["
      << "[\"context\",\"element\",\"property\",\"internal\",\"hidden\",\"shortcut\",\"weak\"],"
      << "\"string_or_number\",\"node\""
      << "],"
      << "\"trace_function_info_fields\":[],\"trace_node_fields\":[],\"sample_fields\":[],\"location_fields\":[]"
      << "},"
      << "\"node_count\":0,\"edge_count\":0,\"trace_function_count\":0"
      << "},"
      << "\"nodes\":[";
  // Synthetic node representing the root of all roots.
  stream << static_cast<int>(HeapSnapshot::NodeType::Synthetic) << ",0,"
         << static_cast<uint64_t>(GC::IDTracker::ReservedObjectID::Root)
         // There's one edge to the Custom root section.
         << ",0,1,0,";
  // Synthetic node for the root section defined by DummyRuntime (which is
  // "Custom").
  stream << static_cast<int>(HeapSnapshot::NodeType::Synthetic) << ",1,"
         << static_cast<uint64_t>(GC::IDTracker::ReservedObjectID::Custom)
         << ",0," <<
      // There's only one root into the system, so say that this node has one
      // edge.
      1 << ",0,";
  // Normal node for the first dummy.
  stream << static_cast<int>(HeapSnapshot::NodeType::Object) << ",2,"
         << rt.getHeap().getObjectID(*dummy) << "," << blockSize << ",1,0,";
  // Normal node for the second dummy, which is only reachable via the first
  // dummy.
  stream << static_cast<int>(HeapSnapshot::NodeType::Object) << ",2,"
         << rt.getHeap().getObjectID(dummy->other.get(&rt)) << "," << blockSize
         << ",0,0";
  // The edges point from the root to the first dummy, and from the first dummy
  // to the second dummy.
  stream
      << "],"
      << "\"edges\":["
      // Edge from super root to custom root section
      << static_cast<int>(HeapSnapshot::EdgeType::Element) << "," << 1 << ","
      << /* Custom root section begins at index 6 in the nodes array */ 6
      << ","
      // Edge from custom root section to dummy.
      << static_cast<int>(HeapSnapshot::EdgeType::Element) << "," << 1 << ","
      << /* The first dummy node starts at index 12 */ 12
      << ","
      // Edge from dummy to dummy2.
      << static_cast<int>(HeapSnapshot::EdgeType::Internal) << ","
      << /*@other*/ 3 << ","
      << /* The second dummy node starts at index 12 */ 18
      // End of edges.
      << "],"
      << "\"trace_function_infos\":[],\"trace_tree\":[],\"samples\":[],\"locations\":[],"
      << R"#("strings":["(GC Roots)","(Custom)","Uninitialized","other"])#"
      << "}";

  std::string expected = stream.str();

  ASSERT_EQ(expected, result);
}

} // namespace heapsnapshottest
} // namespace unittest
} // namespace hermes

#endif
