/*
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the LICENSE
 * file in the root directory of this source tree.
 */
#ifndef HERMES_VM_PROFILER_H
#define HERMES_VM_PROFILER_H

#ifdef HERMESVM_PROFILER_OPCODE
#include <x86intrin.h>

#define INIT_OPCODE_PROFILER      \
  uint64_t startTime = __rdtsc(); \
  unsigned curOpcode = (unsigned)OpCode::Call;

#define RECORD_OPCODE_START_TIME                \
  curOpcode = (unsigned)ip->opCode;             \
  runtime->opcodeExecuteFrequency[curOpcode]++; \
  startTime = __rdtsc();

#define UPDATE_OPCODE_TIME_SPENT \
  runtime->timeSpent[curOpcode] += __rdtsc() - startTime

#else

#define INIT_OPCODE_PROFILER
#define RECORD_OPCODE_START_TIME
#define UPDATE_OPCODE_TIME_SPENT

#endif

#if defined(HERMESVM_PROFILER_EXTERN)
#include "hermes/Utils/Macro10k.h"

// Symbols have the form "JS_[0-9][0-9][0-9][0-9]_xxx...xxx" and the suffix is
// later replaced with the name of the JS function having that ProfilerID.
// Entry 9999 is used as overflow entry if there are >= 10K JS functions.

constexpr int NUM_PROFILER_SYMBOLS_DIGITS = 4;
constexpr int NUM_PROFILER_SYMBOLS = 10e4;
#define PROFILER_SYMBOLS(V) MACRO_10K(V, JS, xxxxxxxxxxxxxxxxxxxxxx)
constexpr int PROFILER_SYMBOL_SUFFIX_LENGTH =
    2 + 2 + 2 + 2 + 2 + 2 + 2 + 2 + 2 + 2 + 2;

namespace hermes {
namespace vm {
class Runtime;

/// Patch the running executable using function metadata from 'runtime'.
void patchProfilerSymbols(Runtime *runtime);

/// Write to \p fileOut mapping between Hermes symbols and JS function names
/// with "<symbol> <JS function>" on each line.
void dumpProfilerSymbolMap(Runtime *runtime, const std::string &fileOut);

} // namespace vm
} // namespace hermes
#endif

#if defined(HERMESVM_PROFILER_JSFUNCTION) || defined(HERMESVM_PROFILER_EXTERN)
#include "hermes/BCGen/HBC/BytecodeDataProvider.h"
#include "hermes/VM/SymbolID.h"

namespace hermes {
namespace vm {

/// ID assigned to each function the profiler encounters.
using ProfilerID = uint32_t;

constexpr ProfilerID NO_PROFILER_ID = UINT32_MAX;

/// Metadata for a function that the profiler has seen.
/// Getting the function offset from functionID is very expensive,
/// thus we store the ID to calculate the offset at symbol dump time.
struct ProfilerFunctionInfo {
  ProfilerID profilerId;
  SymbolID functionName;
  uint32_t functionID;
  std::shared_ptr<hbc::BCProvider> bytecode;
  ProfilerFunctionInfo(
      ProfilerID id,
      SymbolID name,
      uint32_t fid,
      std::shared_ptr<hbc::BCProvider> bc)
      : profilerId(id),
        functionName(name),
        functionID(fid),
        bytecode(std::move(bc)) {}
};

} // namespace vm
} // namespace hermes
#endif

#if defined(HERMESVM_PROFILER_JSFUNCTION)
#include <x86intrin.h>
namespace hermes {
namespace vm {
/// Structure to capture the event of either entering a function or
/// exiting a function. Currently we only cover normal functions,
/// and exclude native and bound functions.
struct ProfilerFunctionEvent {
  ProfilerID functionID;
  uint64_t timeStamp;
  uint64_t opcodeStamp;
  bool isEnter;
  ProfilerFunctionEvent(ProfilerID s, uint64_t t, uint64_t o, bool e)
      : functionID(s), timeStamp(t), opcodeStamp(o), isEnter(e) {}
};

#define INC_OPCODE_COUNT ++runtime->opcodeCount

#define PROFILER_ENTER_FUNCTION(block)                              \
  do {                                                              \
    runtime->maxStackLevel =                                        \
        std::max(runtime->maxStackLevel, runtime->getStackLevel()); \
    auto id = runtime->getProfilerID(block);                        \
    auto o = runtime->opcodeCount;                                  \
    runtime->functionEvents.emplace_back(id, __rdtsc(), o, true);   \
  } while (false)

#define PROFILER_EXIT_FUNCTION(block)                              \
  do {                                                             \
    auto id = runtime->getProfilerID(block);                       \
    auto o = runtime->opcodeCount;                                 \
    runtime->functionEvents.emplace_back(id, __rdtsc(), o, false); \
  } while (false)

} // namespace vm
} // namespace hermes

#else

#define INC_OPCODE_COUNT
#define PROFILER_ENTER_FUNCTION(block)
#define PROFILER_EXIT_FUNCTION(block)

#endif

// Profile frequency and duration of native calls.
#ifdef HERMESVM_PROFILER_NATIVECALL

#if defined(__i386__) || defined(__x86_64__)
#include <x86intrin.h>
#define HERMESVM_RDTSC() __rdtsc()
#else
#define HERMESVM_RDTSC() 0
#endif

#endif // HERMESVM_PROFILER_NATIVECALL

#endif // HERMES_VM_PROFILER_H
