/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.security.wallet;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import oracle.kv.impl.security.PasswordStore;
import oracle.kv.impl.security.PasswordStoreException;
import oracle.kv.impl.security.util.SecurityUtils;
import oracle.security.pki.OracleSecretStore;
import oracle.security.pki.OracleSecretStoreException;
import oracle.security.pki.OracleWallet;

public class WalletStore
implements PasswordStore {
    private final File storeLocation;
    private OracleWallet wallet = null;
    private OracleSecretStore sstore = null;
    private boolean autoLogin = false;
    private boolean modified = false;
    private static final String LOGIN_PREFIX = "oracle.security.client.";
    private static final String LOGIN_CONNECT_PREFIX = "oracle.security.client.connect_string";
    private static final String LOGIN_USER_PREFIX = "oracle.security.client.username";
    private static final String LOGIN_PW_PREFIX = "oracle.security.client.password";
    private static final Object openWalletLock = new Object();

    WalletStore(File storeDirectory) {
        this.storeLocation = storeDirectory;
    }

    @Override
    public boolean create(char[] passphrase) throws IOException {
        this.assertNotInitialized();
        OracleWallet newWallet = new OracleWallet();
        try {
            if (passphrase != null) {
                newWallet.create(passphrase);
            } else {
                newWallet.createSSO();
            }
            newWallet.saveAs(this.storeLocation.getPath());
        }
        catch (IOException ioe) {
            if (WalletStore.exceptionContains(ioe, "PKI-01002")) {
                throw new PasswordStoreException("The specified passphrase is not valid", ioe);
            }
            if (WalletStore.exceptionContains(ioe, "PKI-02009")) {
                throw new PasswordStoreException("Unable to create the wallet directory", ioe);
            }
            if (WalletStore.exceptionContains(ioe, "Permission denied")) {
                throw new PasswordStoreException("Unable to create the wallet file", ioe);
            }
            if (WalletStore.exceptionContains(ioe, "PKI-")) {
                throw new PasswordStoreException("Error creating the wallet", ioe);
            }
            throw ioe;
        }
        this.autoLogin = passphrase == null;
        File walletFile = this.findWalletFile(false);
        if (!SecurityUtils.makeOwnerAccessOnly((File)walletFile)) {
            throw new PasswordStoreException("Unable to set access permissions for file. Correct manually before using the password store");
        }
        try {
            this.sstore = newWallet.getSecretStore();
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("Error retrieving secret store from wallet", osse);
        }
        this.autoLogin = passphrase == null;
        this.wallet = newWallet;
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean open(char[] passphrase) throws IOException {
        this.assertNotInitialized();
        OracleWallet newWallet = new OracleWallet();
        try {
            Object object = openWalletLock;
            synchronized (object) {
                newWallet.open(this.storeLocation.getPath(), passphrase);
            }
        }
        catch (IOException ioe) {
            if (WalletStore.exceptionContains(ioe, "PKI-02002")) {
                throw new PasswordStoreException("Error accessing the wallet.  Check your passphrase", ioe);
            }
            if (WalletStore.exceptionContains(ioe, "PKI-")) {
                throw new PasswordStoreException("Error accessing the wallet.", ioe);
            }
            throw ioe;
        }
        try {
            this.sstore = newWallet.getSecretStore();
        }
        catch (OracleSecretStoreException osse) {
            throw new IOException("Error retrieving secret store from wallet", osse);
        }
        this.autoLogin = passphrase == null;
        this.wallet = newWallet;
        return true;
    }

    @Override
    public Collection<String> getSecretAliases() throws IOException {
        this.assertInitialized();
        HashSet<String> secretAliases = new HashSet<String>();
        try {
            Enumeration e = this.sstore.internalAliases();
            while (e.hasMoreElements()) {
                String alias = (String)e.nextElement();
                if (alias.startsWith(LOGIN_PREFIX)) continue;
                secretAliases.add(alias);
            }
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("Error retrieving secret store from wallet", osse);
        }
        return secretAliases;
    }

    @Override
    public char[] getSecret(String alias) throws IOException {
        this.assertInitialized();
        try {
            return this.sstore.getSecret(alias);
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("error retrieving secret from wallet", osse);
        }
    }

    @Override
    public boolean setSecret(String alias, char[] secret) throws IOException {
        this.assertInitialized();
        try {
            boolean updated = this.sstore.containsAlias(alias);
            this.sstore.setSecret(alias, secret);
            this.modified = true;
            return updated;
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("error modifying secret store", osse);
        }
    }

    @Override
    public boolean deleteSecret(String alias) throws IOException {
        this.assertInitialized();
        try {
            if (this.sstore.containsAlias(alias)) {
                this.sstore.deleteSecret(alias);
                this.modified = true;
                return true;
            }
            return false;
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("error modifying secret store", osse);
        }
    }

    @Override
    public Collection<PasswordStore.LoginId> getLogins() throws IOException {
        this.assertInitialized();
        HashSet<PasswordStore.LoginId> logins = new HashSet<PasswordStore.LoginId>();
        try {
            Enumeration e = this.sstore.internalAliases();
            while (e.hasMoreElements()) {
                String alias = (String)e.nextElement();
                if (!alias.startsWith(LOGIN_CONNECT_PREFIX)) continue;
                String index = alias.substring(LOGIN_CONNECT_PREFIX.length());
                String db = new String(this.sstore.getSecret(alias));
                String userKey = LOGIN_USER_PREFIX + index;
                String user = new String(this.sstore.getSecret(userKey));
                logins.add(new PasswordStore.LoginId(db, user));
            }
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("error retrieving secret store from wallet", osse);
        }
        return logins;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean setLogin(PasswordStore.LoginId login, char[] password) throws IOException {
        this.assertInitialized();
        try {
            Collection<PasswordStore.LoginId> logins = this.getLogins();
            PasswordStore.LoginId existing = null;
            for (PasswordStore.LoginId lid : logins) {
                if (!this.databasesEqual(lid.getDatabase(), login.getDatabase())) continue;
                existing = lid;
                break;
            }
            OutputCapture output = new OutputCapture();
            try {
                if (existing != null) {
                    this.sstore.modifyCredential(login.getDatabase().toCharArray(), login.getUser().toCharArray(), password);
                } else {
                    this.sstore.createCredential(login.getDatabase().toCharArray(), login.getUser().toCharArray(), password);
                }
            }
            finally {
                output.restore();
            }
            this.modified = true;
            return existing != null;
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("error modifying secret store", osse);
        }
    }

    @Override
    public PasswordStore.LoginId getLoginId(String database) throws IOException {
        this.assertInitialized();
        Collection<PasswordStore.LoginId> logins = this.getLogins();
        for (PasswordStore.LoginId lid : logins) {
            if (!this.databasesEqual(lid.getDatabase(), database)) continue;
            return lid;
        }
        return null;
    }

    @Override
    public char[] getLoginSecret(String database) throws IOException {
        this.assertInitialized();
        try {
            Enumeration e = this.sstore.internalAliases();
            while (e.hasMoreElements()) {
                String db;
                String alias = (String)e.nextElement();
                if (!alias.startsWith(LOGIN_CONNECT_PREFIX) || !this.databasesEqual(db = new String(this.sstore.getSecret(alias)), database)) continue;
                String index = alias.substring(LOGIN_CONNECT_PREFIX.length());
                String pwKey = LOGIN_PW_PREFIX + index;
                return this.sstore.getSecret(pwKey);
            }
            return null;
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("Error accessing secret store", osse);
        }
    }

    @Override
    public boolean deleteLogin(String db) {
        this.assertInitialized();
        try {
            OutputCapture output = new OutputCapture();
            try {
                this.sstore.deleteCredential(db.toCharArray());
            }
            finally {
                output.restore();
            }
            this.modified = true;
            return true;
        }
        catch (OracleSecretStoreException oracleSecretStoreException) {
            return false;
        }
    }

    @Override
    public boolean setPassphrase(char[] passphrase) throws IOException {
        this.assertInitialized();
        if (passphrase == null && this.autoLogin) {
            return true;
        }
        File originalWalletFile = this.findWalletFile(false);
        OracleWallet newWallet = new OracleWallet();
        if (this.autoLogin) {
            newWallet.create(passphrase);
        } else if (passphrase == null) {
            newWallet.createSSO();
        } else {
            newWallet.create(passphrase);
            originalWalletFile = null;
        }
        OracleSecretStore newSstore = null;
        try {
            newSstore = newWallet.getSecretStore();
        }
        catch (OracleSecretStoreException osse) {
            throw new IOException("error retrieving secret store from wallet", osse);
        }
        try {
            Enumeration e = this.sstore.internalAliases();
            while (e.hasMoreElements()) {
                String alias = (String)e.nextElement();
                char[] secret = this.sstore.getSecret(alias);
                newSstore.setSecret(alias, secret);
            }
        }
        catch (OracleSecretStoreException osse) {
            throw new IOException("error transferring secrets", osse);
        }
        try {
            newWallet.setSecretStore(newSstore);
        }
        catch (OracleSecretStoreException osse) {
            throw new PasswordStoreException("Error modifying secret store", osse);
        }
        if (originalWalletFile != null) {
            originalWalletFile.delete();
        }
        newWallet.saveAs(this.storeLocation.getPath());
        this.modified = false;
        this.sstore = newSstore;
        this.wallet = newWallet;
        this.autoLogin = passphrase == null;
        File walletFile = this.findWalletFile(false);
        if (!SecurityUtils.makeOwnerAccessOnly((File)walletFile)) {
            throw new PasswordStoreException("Unable to set access permissions for file. Correct manually before using the password store");
        }
        return true;
    }

    @Override
    public void save() throws IOException {
        this.assertInitialized();
        if (this.wallet == null) {
            throw new IllegalStateException("wallet has not been initialized");
        }
        if (this.modified) {
            try {
                this.wallet.setSecretStore(this.sstore);
            }
            catch (OracleSecretStoreException osse) {
                throw new PasswordStoreException("Error saving secret store", osse);
            }
            this.wallet.saveAs(this.storeLocation.getPath());
            this.modified = false;
        }
    }

    @Override
    public void discard() {
        this.wallet = null;
        this.sstore = null;
    }

    @Override
    public boolean requiresPassphrase() throws IOException {
        if (this.wallet != null) {
            return !this.autoLogin;
        }
        File f = this.findWalletFile(true);
        return !f.getName().endsWith(".sso");
    }

    @Override
    public boolean isValidPassphrase(char[] passphrase) {
        return OracleWallet.isValidPassword((char[])passphrase);
    }

    @Override
    public boolean exists() throws IOException {
        return this.findWalletFile(true) != null;
    }

    private File findWalletFile(boolean any) throws IOException {
        if (!this.storeLocation.exists()) {
            return null;
        }
        if (!this.storeLocation.isDirectory()) {
            throw new IOException("Wallet location is not a directory");
        }
        File[] storeFiles = this.storeLocation.listFiles();
        if (storeFiles == null) {
            throw new IOException("Unable to access wallet. Check access permissions.");
        }
        for (File f : storeFiles) {
            if ((!any && !this.autoLogin || !f.getName().endsWith(".sso")) && (!any && this.autoLogin || !f.getName().endsWith(".p12"))) continue;
            return f;
        }
        return null;
    }

    private static boolean exceptionContains(Exception e, String match) {
        String msg = e.getMessage();
        if (msg == null) {
            return false;
        }
        return msg.contains(match);
    }

    private boolean databasesEqual(String db1, String db2) {
        return db1.equals(db2);
    }

    private void assertNotInitialized() {
        if (this.wallet != null) {
            throw new IllegalStateException("wallet already initialized");
        }
    }

    private void assertInitialized() {
        if (this.wallet == null) {
            throw new IllegalStateException("wallet not yet initialized");
        }
    }

    private final class OutputCapture {
        private PrintStream systemOut = null;
        private final ByteArrayOutputStream baosOut = new ByteArrayOutputStream();
        private final PrintStream printOut = new PrintStream(this.baosOut);
        private boolean capturing = false;

        private OutputCapture() {
            this.start();
        }

        private void start() {
            if (!this.capturing) {
                this.systemOut = System.out;
                System.setOut(this.printOut);
                this.capturing = true;
            }
        }

        private void restore() {
            if (this.capturing) {
                this.capturing = false;
                System.setOut(this.systemOut);
            }
        }
    }
}

