/**********************************************************************
 * Copyright (c) 2018 Ericsson
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
package org.eclipse.tracecompass.internal.tmf.core.model.filters;

import java.util.Collection;
import java.util.List;

import org.eclipse.tracecompass.tmf.core.model.CoreFilterProperty;
import org.eclipse.tracecompass.tmf.core.model.filters.SelectionTimeQueryFilter;

import com.google.common.collect.Multimap;

/**
 * Standardized query filter to query time graph data providers for a Collection
 * of entries and filter the data using the given regex.
 *
 * @author Jean-Christian Kouame
 * @since 4.0
 *
 */
public class TimeGraphStateQueryFilter extends SelectionTimeQueryFilter implements IRegexQuery {

    private Multimap<Integer, String> fRegexes;

    /**
     * Constructor
     *
     * @param times
     *            sorted list of times to query.
     * @param items
     *            The unique keys of the selected entries.
     * @param regexes
     *            The regexes use to filter the queried data. It is a multimap
     *            of filter strings by property. The data provider will use the
     *            filter strings to determine whether the property should be
     *            activated or not. See {@link CoreFilterProperty} for supported
     *            properties.
     */
    public TimeGraphStateQueryFilter(List<Long> times, Collection<Long> items, Multimap<Integer, String> regexes) {
        super(times, items);
        fRegexes = regexes;
    }

    /**
     * Constructor
     *
     * @param start
     *            The starting value
     * @param end
     *            The ending value
     * @param n
     *            The number of entries
     * @param items
     *            The unique keys of the selected entries
     * @param regexes
     *            The regexes use to filter the queried data. It is a multimap
     *            of filter strings by property. The data provider will use the
     *            filter strings to determine whether the property should be
     *            activated or not.
     */
    public TimeGraphStateQueryFilter(long start, long end, int n, Collection<Long> items, Multimap<Integer, String> regexes) {
        super(start, end, n, items);
        fRegexes = regexes;
    }

    @Override
    public Multimap<Integer, String> getRegexes() {
        return fRegexes;
    }

}
