import { isKubernetesAssociatedDocument } from '../../languageservice/parser/isKubernetes';
export class LanguageHandlers {
    constructor(connection, languageService, yamlSettings, validationHandler) {
        this.connection = connection;
        this.languageService = languageService;
        this.yamlSettings = yamlSettings;
        this.validationHandler = validationHandler;
    }
    registerHandlers() {
        this.connection.onDocumentLinks((params) => this.documentLinkHandler(params));
        this.connection.onDocumentSymbol((documentSymbolParams) => this.documentSymbolHandler(documentSymbolParams));
        this.connection.onDocumentFormatting((formatParams) => this.formatterHandler(formatParams));
        this.connection.onHover((textDocumentPositionParams) => this.hoverHandler(textDocumentPositionParams));
        this.connection.onCompletion((textDocumentPosition) => this.completionHandler(textDocumentPosition));
        this.connection.onDidChangeWatchedFiles((change) => this.watchedFilesHandler(change));
        this.connection.onFoldingRanges((params) => this.foldingRangeHandler(params));
        this.connection.onCodeAction((params) => this.codeActionHandler(params));
        this.connection.onDocumentOnTypeFormatting((params) => this.formatOnTypeHandler(params));
    }
    documentLinkHandler(params) {
        const document = this.yamlSettings.documents.get(params.textDocument.uri);
        if (!document) {
            return Promise.resolve([]);
        }
        return this.languageService.findLinks(document);
    }
    /**
     * Called when the code outline in an editor needs to be populated
     * Returns a list of symbols that is then shown in the code outline
     */
    documentSymbolHandler(documentSymbolParams) {
        const document = this.yamlSettings.documents.get(documentSymbolParams.textDocument.uri);
        if (!document) {
            return;
        }
        if (this.yamlSettings.hierarchicalDocumentSymbolSupport) {
            return this.languageService.findDocumentSymbols2(document);
        }
        else {
            return this.languageService.findDocumentSymbols(document);
        }
    }
    /**
     * Called when the formatter is invoked
     * Returns the formatted document content using prettier
     */
    formatterHandler(formatParams) {
        const document = this.yamlSettings.documents.get(formatParams.textDocument.uri);
        if (!document) {
            return;
        }
        const customFormatterSettings = Object.assign({ tabWidth: formatParams.options.tabSize }, this.yamlSettings.yamlFormatterSettings);
        return this.languageService.doFormat(document, customFormatterSettings);
    }
    formatOnTypeHandler(params) {
        const document = this.yamlSettings.documents.get(params.textDocument.uri);
        if (!document) {
            return;
        }
        return this.languageService.doDocumentOnTypeFormatting(document, params);
    }
    /**
     * Called when the user hovers with their mouse over a keyword
     * Returns an informational tooltip
     */
    hoverHandler(textDocumentPositionParams) {
        const document = this.yamlSettings.documents.get(textDocumentPositionParams.textDocument.uri);
        if (!document) {
            return Promise.resolve(undefined);
        }
        return this.languageService.doHover(document, textDocumentPositionParams.position);
    }
    /**
     * Called when auto-complete is triggered in an editor
     * Returns a list of valid completion items
     */
    completionHandler(textDocumentPosition) {
        const textDocument = this.yamlSettings.documents.get(textDocumentPosition.textDocument.uri);
        const result = {
            items: [],
            isIncomplete: false,
        };
        if (!textDocument) {
            return Promise.resolve(result);
        }
        return this.languageService.doComplete(textDocument, textDocumentPosition.position, isKubernetesAssociatedDocument(textDocument, this.yamlSettings.specificValidatorPaths));
    }
    /**
     * Called when a monitored file is changed in an editor
     * Re-validates the entire document
     */
    watchedFilesHandler(change) {
        let hasChanges = false;
        change.changes.forEach((c) => {
            if (this.languageService.resetSchema(c.uri)) {
                hasChanges = true;
            }
        });
        if (hasChanges) {
            this.yamlSettings.documents.all().forEach((document) => this.validationHandler.validate(document));
        }
    }
    foldingRangeHandler(params) {
        const textDocument = this.yamlSettings.documents.get(params.textDocument.uri);
        if (!textDocument) {
            return;
        }
        return this.languageService.getFoldingRanges(textDocument, this.yamlSettings.capabilities.textDocument.foldingRange);
    }
    codeActionHandler(params) {
        const textDocument = this.yamlSettings.documents.get(params.textDocument.uri);
        if (!textDocument) {
            return;
        }
        return this.languageService.getCodeAction(textDocument, params);
    }
}
//# sourceMappingURL=languageHandlers.js.map