/**
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * $Id: RDFFactoryImpl.java,v 1.1 2007/03/18 08:10:11 lzhang Exp $
 */
package org.eclipse.eodm.impl;

import java.lang.reflect.Proxy;
import java.net.URISyntaxException;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.eodm.RDFFactory;
import org.eclipse.eodm.exceptions.InvalidLexicalFormException;
import org.eclipse.eodm.rdf.rdfbase.*;
import org.eclipse.eodm.rdf.rdfbase.impl.RDFGraphImpl;
import org.eclipse.eodm.rdf.rdfbase.impl.URIReferenceImpl;
import org.eclipse.eodm.rdf.rdfbase.impl.UniformResourceIdentifierImpl;
import org.eclipse.eodm.rdf.rdfbase.util.*;
import org.eclipse.eodm.rdf.rdfs.*;
import org.eclipse.eodm.rdf.rdfs.util.*;
import org.eclipse.eodm.rdf.rdfweb.*;
import org.eclipse.eodm.rdf.rdfweb.impl.NamespaceImpl;
import org.eclipse.eodm.rdf.rdfweb.util.*;
import org.eclipse.eodm.util.*;


public class RDFFactoryImpl implements RDFFactory{
	
/**
 * Creates an instance of the factory.
 * <!-- begin-user-doc -->
 * <!-- end-user-doc -->
 * 
 * @generated
 */
public RDFFactoryImpl() {
	super();
}

public EObject create(RDFGraph graph,EClass eClass,String uriOrNodeID) throws URISyntaxException
{
	EPackage pkg = eClass.getEPackage();
	
	if ( pkg.equals( RDFBasePackage.eINSTANCE ) && (eClass.getClassifierID() == RDFBasePackage.BLANK_NODE ) )
		return this.createBlankNode(graph, uriOrNodeID) ;
	else
		return this.create(graph, eClass, this.createURIReference(uriOrNodeID));
}
	
public EObject create(RDFGraph graph,EClass eClass,URIReference uriRef)
{
	EPackage pkg = eClass.getEPackage();
	Class jclass = EJClassMapping.eclass2jclass( eClass );
	
	if ( jclass != null && (pkg.equals( RDFBasePackage.eINSTANCE ) || pkg.equals( RDFSPackage.eINSTANCE )))
		return internalCreate(graph, jclass, uriRef);
	else
		throw new IllegalArgumentException("The class '" + eClass.getName() + "' does not belong to RDF(S) package.");

}

public RDFSResource internalCreate( RDFGraph graph, Class jclass )
{
	EClass eclass = EJClassMapping.jclass2eclass( jclass ); 
	if ( eclass == null ) return null ;
	
	InternalCore coreObject = EODMImplFactory.eINSTANCE.createInternalCore();
	
	coreObject.addProxy(RDFBasePackage.eINSTANCE.getBlankNode(), BlankNode.class);
	coreObject.setNodeID(RDFUtility.nodeidGenerator());
	((RDFGraphImpl) graph).getEntityHashMap().put(coreObject.getNodeID(), coreObject);
	
	return (RDFSResource) coreObject.addProxy( eclass, jclass );	
}

protected RDFSResource genericCreate(RDFGraph graph, Class jclass, URIReference type)
{
	RDFSResource product = internalCreate( graph, jclass );
	InternalCore coreObject = (InternalCore) ((EODMInvocationHandler)Proxy.getInvocationHandler(product)).getDelegatedObject();	
	coreObject.addType(graph, type);
	return product;
}

public RDFSResource internalCreate( RDFGraph graph, Class jclass, URIReference uriRef )
{
	EClass eclass = EJClassMapping.jclass2eclass( jclass );
	if ( eclass == null ) return null ;
	
	RDFGraphImpl graphImpl = (RDFGraphImpl) graph;
	String uriRefString = uriRef.getURIString();
	InternalCore coreObject = null;
	RDFSResource rdfsresource = null;
	if (graphImpl.getEntityHashMap().containsKey(uriRefString)) {
		try {
			coreObject = (InternalCore) graphImpl.getEntityHashMap().get(uriRefString);
			rdfsresource = (RDFSResource) coreObject.forceAsType( jclass );
		} catch (Exception e) {
			e.printStackTrace();
		}
	} else {
		coreObject = EODMImplFactory.eINSTANCE.createInternalCore();
		rdfsresource = (RDFSResource) coreObject.addProxy( eclass, jclass );
		coreObject.addProxy( EJClassMapping.jclass2eclass( URIReferenceNode.class ), URIReferenceNode.class );
		rdfsresource.getUriRef().add(uriRef);
		coreObject.setURI(uriRefString);
		graphImpl.getEntityHashMap().put(uriRefString, coreObject);

	}

	return rdfsresource;
}

protected RDFSResource genericCreate( RDFGraph graph, Class jclass, URIReference uriRef, URIReference type)
{
	RDFSResource product = internalCreate( graph, jclass, uriRef );
	InternalCore coreObject = (InternalCore) ((EODMInvocationHandler)Proxy.getInvocationHandler(product)).getDelegatedObject();	
	coreObject.addType(graph, type);
	return product;	
}

/**
 * Creates a new unique '<em>BlankNode</em>'.
 * The nodeID of the new blank node will be different from all blank 
 * nodes created using this factory.
 * 
 * @param graph  the context graph of this blank node.
 * @return a new object of class '<em>BlankNode</em>' with a unique nodeID.
 */
public BlankNode createBlankNode( RDFGraph graph, String nodeid )
{
	RDFGraphImpl graphImpl = (RDFGraphImpl)graph;
	if(graphImpl.getEntityHashMap().containsKey(nodeid))
	{
		try{
			return (BlankNode)((RDFSResource)(graphImpl.getEntityHashMap().get(nodeid))).forceAsType(BlankNode.class);
		}catch(Exception e){e.printStackTrace();}
	}
	else
	{
		InternalCore coreObject= EODMImplFactory.eINSTANCE.createInternalCore();
		BlankNode blanknode = (BlankNode)coreObject.addProxy(RDFBasePackage.eINSTANCE.getBlankNode(),BlankNode.class);
		coreObject.setNodeID(nodeid);
		graphImpl.getEntityHashMap().put(nodeid,coreObject);
		return blanknode;
	}
	return null;
		
}
   
/**
 * Returns a new unique blank node of type RDFSResource.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSResource
 * @return a new unique blank node of type RDFSResource
 */
public RDFSResource createRDFSResource(RDFGraph graph) {
	return genericCreate(graph, RDFSResource.class, RDFUtility.rdfsclass);
}

/**
 * Returns an object of class '<em>RDFSResource</em>'.
 * 
 * @param graph the context graph of this RDFSResource.
 * @param uriRefString the RDF URI Reference of the RDFSResource.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSResource</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSResource createRDFSResource(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSResource(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFSResource</em>'.
 * 
 * @param graph  the context graph of this RDFSResource
 * @param uriRef the RDF URI Reference of the RDFSResource.
 * @return an object of class '<em>RDFSResource</em>'.
 */
public RDFSResource createRDFSResource(RDFGraph graph, URIReference uriRef) {
	return genericCreate(graph, RDFSResource.class, uriRef, RDFUtility.rdfsresource);
}

/**
 * Returns an object of class '<em>RDFSResource</em>'.
 * 
 * @param graph           the context graph of this RDFSResource
 * @param namespace       the namespace of the RDF URI Reference of the RDFSResource
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSResource
 * @return an object of class '<em>RDFSResource</em>'.
 */
public RDFSResource createRDFSResource(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSResource(graph, this.createURIReference(namespace, localNameString));
}
    
/**
 * Returns an object of class '<em>PlainLiteral</em>'.
 * 
 * @param lexicalForm the lexical form of the plain literal. It may be 
 *        <code>null</code> and be set later.
 * @return an object of class '<em>PlainLiteral</em>'. The returned plain literal
 *         does not have a language tag (the value of the language tag is
 *         <code>null</code>) and can be set later.
 */
public PlainLiteral createPlainLiteral( String lexicalForm )
{
	PlainLiteral pl =  RDFBaseFactory.eINSTANCE.createPlainLiteral();
	pl.setLexicalForm(lexicalForm);
	return pl;
}

/**
 * Returns an object of class '<em>PlainLiteral</em>'
 * 
 * @param lexicalForm the lexical form of the plain literal. It may be
 *        <code>null</code> and be set later.
 * @param languageTag the language tag of the plain literal. It may be
 *        <code>null</code> to signify that no language tag is set.
 * @return an object of class '<em>PlainLiteral</em>'. 
 */
public PlainLiteral createPlainLiteral( String lexicalForm, String languageTag )
{
	PlainLiteral pl =  RDFBaseFactory.eINSTANCE.createPlainLiteral();
	pl.setLexicalForm(lexicalForm);
	pl.setLanguage(languageTag);
	return pl;
}

/**
 * Returns an object of class '<em>URIReference</em>'.
 * 
 * @param uriRefString  <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>URIReference</em>'.
 * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
 */
public URIReference createURIReference( String uriRefString )
throws URISyntaxException
{
	URIReference uri = RDFBaseFactory.eINSTANCE.createURIReference();
	((URIReferenceImpl)uri).setUri(this.createUniformResourceIdentifier(uriRefString));
	return uri;
}

/**
 * Returns an object of class '<em>URIReference</em>'. The URI of this
 * URI Reference is the namespace URI appended with the localNameString.
 * 
 * @param namespace the namespace of this URI Reference
 * @param localNameString the fragment identifier of this URI Reference
 * @return an object of class '<em>URIReference</em>'.
 */
public URIReference createURIReference( Namespace namespace, String localNameString )
{
	URIReference uri = RDFBaseFactory.eINSTANCE.createURIReference();
	try{
		((URIReferenceImpl)uri).setNamespace(namespace);
		((URIReferenceImpl)uri).setFragmentIdentifier(this.createLocalName(localNameString));
	}catch(Exception e){e.printStackTrace();}
	return uri;
}

/**
 * Returns an object of class '<em>URIReference</em>'. The URI of this
 * URI Reference is the namespace URI appended with the localName string.
 * 
 * @param namespace the namespace of this URI Reference
 * @param localName the fragment identifier of this URI Reference
 * @return an object of class '<em>URIReference</em>'.
 */
public URIReference createURIReference( Namespace namespace, LocalName localName )
{
	URIReference uri = RDFBaseFactory.eINSTANCE.createURIReference();
	
	try{
		((URIReferenceImpl)uri).setNamespace(namespace);
		uri.setFragmentIdentifier(localName);
	}catch(Exception e){e.printStackTrace();}
	return uri;
}

/**
 * Returns an object of class '<em>UniformResourceIdentifier</em>'.
 * 
 * @param uriString the URI string of this UniformResourceIdentifier.
 * @return an object of class '<em>UniformResourceIdentifier</em>'.
 * @throws URISyntaxException if the uriString is not a valid URI.
 */
public UniformResourceIdentifier createUniformResourceIdentifier( String uriString )
throws URISyntaxException
{
	UniformResourceIdentifier uri = RDFBaseFactory.eINSTANCE.createUniformResourceIdentifier();
	((UniformResourceIdentifierImpl)uri).setName(uriString);
	return uri;
}

/**
 * Returns an object of class '<em>RDFGraph</em>'. 
 * 
 * @param uriRefString  The string must be <code>null</code> or <a href="#uriRefString"> a valid RDF URI Reference</a>.
 *        If it is a valid RDF URI Reference, it serves as the name of the graph
 *        and can not be changed. If it is <code>null</code>, this graph then
 *        does not have a name and can not be given a name later.
 * @return an object of class '<em>RDFGraph</em>'.
 * @throws URISyntaxException if the uriRefString is not <code>null</code> and
 *         is not a valid RDF URI Reference.
 */
public RDFGraph createGraph( String uriRefString )
throws URISyntaxException
{
	return createGraph(this.createURIReference(uriRefString));
}
/**
 * Returns an object of class '<em>RDFGraph</em>'.
 * 
 * @param uriRef the RDF URI Reference of the RDFGraph.
 * @return an object of class '<em>RDFGraph</em>'.
 */
public RDFGraph createGraph( URIReference uriRef )
{
	RDFGraph graph = RDFBaseFactory.eINSTANCE.createGraph();
	graph.setGraphName(uriRef);
	return graph;
}

/**
 * Returns an object of class '<em>RDFGraph</em>'.
 * 
 * @param namespace       the namespace of the RDF URI Reference of the RDFGraph
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFGraph
 * @return an object of class '<em>RDFGraph</em>'.
 */
public RDFGraph createGraph( Namespace namespace, String localNameString )
{
	return createGraph(this.createURIReference( namespace, localNameString));
}

/**
 * Returns an object of class '<em>URIReferenceNode</em>'.
 * 
 * @param graph  the context graph of this node.
 * @param uriRefString the RDF URI Reference of this node. <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>URIReferenceNode</em>'.
 * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
 */
public URIReferenceNode createURIReferenceNode( RDFGraph graph, String uriRefString )
throws URISyntaxException
{
	return createURIReferenceNode( graph, this.createURIReference(uriRefString) );
}

/**
 * Returns an object of class '<em>URIReferenceNode</em>'
 * 
 * @param graph the context graph of this node.
 * @param uriRef the RDF URI Reference of this node.
 * @return an object of class '<em>URIReferenceNode</em>'
 */
public URIReferenceNode createURIReferenceNode( RDFGraph graph, URIReference uriRef )
{
	return (URIReferenceNode) genericCreate(graph, URIReferenceNode.class, uriRef, RDFUtility.rdfsresource );
}

/**
 * Returns an object of class '<em>URIReferenceNode</em>'
 *  
 * @param graph the context graph of this node
 * @param namespace the namespace of the RDF URI Reference of this node
 * @param localNameString the fragment identifier of the RDF URI Reference of this node
 * @return an object of class '<em>URIReferenceNode</em>'
 */
public URIReferenceNode createURIReferenceNode( RDFGraph graph, Namespace namespace, String localNameString )
{
	return createURIReferenceNode(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns an object of class '<em>RDFSLiteral</em>'.
 * 
 * @param lexicalForm the lexical form of the literal, may be <code>null</code> to
 *        be set later.
 * @return an object of class '<em>RDFSLiteral</em>'.
 */
public RDFSLiteral createRDFSLiteral( String lexicalForm ) 
{
	RDFSLiteral literal = RDFBaseFactory.eINSTANCE.createRDFSLiteral();
	literal.setLexicalForm(lexicalForm);
	return literal;
}

/**
 * Returns an object of class '<em>TypedLiteral</em>'.
 * 
 * @param lexicalForm the lexical form of this typed literal
 * @param dataTypeURIRef the RDF URI Reference of the data type of thie literal.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>TypedLiteral</em>'.
 * @throws URISyntaxException if the dataTypeURIRef is not a valid RDF URI Reference.
 * @throws InvalidLexicalFormException if the <code>lexicalForm</code> string is not <code>
 *         null</code> and does not conforms to the data type identified by 
 *         the data type URI Reference.
 */
public TypedLiteral createTypedLiteral( String lexicalForm, String dataTypeURIRef )
throws URISyntaxException, InvalidLexicalFormException
{
	TypedLiteral typedLiteral = RDFBaseFactory.eINSTANCE.createTypedLiteral();
	typedLiteral.setLexicalForm(lexicalForm);
	typedLiteral.setDatatypeURI(this.createURIReference(dataTypeURIRef));
	return typedLiteral;
}

/**
 * Returns an object of class '<em>RDFXMLLiteral</em>'. This literal
 * has an XML content as its lexical form and a URI Reference for identifying
 * the data type: <code>http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral</code>.
 * 
 * @param xmlString the lexical form of this literal in the form of a XML content.
 *        Please see <a href="http://www.w3.org/TR/rdf-concepts/#section-XMLLiteral">
 *        here</a> for the specification on this XML lexical form. It may be
 *        <code>null</code> and changed later.
 * @return an object of class '<em>RDFXMLLiteral</em>'.
 * @throws InvalidLexicalFormException  if the <code>xmlString</code> is not
 *         <code>null</code> and does not conforms to the rdf:XMLLiteral data
 *         type specification.
 */
public RDFXMLLiteral createRDFXMLLiteral( String xmlString )
throws InvalidLexicalFormException
{
	RDFXMLLiteral xmlliteral = RDFBaseFactory.eINSTANCE.createRDFXMLLiteral();
	xmlliteral.setLexicalForm(xmlString);
	return xmlliteral;
}

/**
 * Returns a new unique blank node of type RDFSClass.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSClass
 * @return a new unique blank node of type RDFSClass
 */
public RDFSClass createRDFSClass(RDFGraph graph) {
	return (RDFSClass) genericCreate(graph, RDFSClass.class, RDFUtility.rdfsclass) ;
}

/**
 * Returns an object of class '<em>RDFSClass</em>'.
 * 
 * @param graph the context graph of this RDFSClass.
 * @param uriRefString the RDF URI Reference of the RDFSClass.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSClass</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSClass createRDFSClass(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSClass( graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFSClass</em>'.
 * 
 * @param graph  the context graph of this RDFSClass
 * @param uriRef the RDF URI Reference of the RDFSClass.
 * @return an object of class '<em>RDFSClass</em>'.
 */
public RDFSClass createRDFSClass(RDFGraph graph, URIReference uriRef) {
	return (RDFSClass) genericCreate(graph, RDFSClass.class, uriRef, RDFUtility.rdfsclass) ;
}

/**
 * Returns an object of class '<em>RDFSClass</em>'.
 * 
 * @param graph           the context graph of this RDFSClass
 * @param namespace       the namespace of the RDF URI Reference of the RDFSClass
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSClass
 * @return an object of class '<em>RDFSClass</em>'.
 */
public RDFSClass createRDFSClass(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSClass(graph, this.createURIReference( namespace, localNameString));
}
    
/**
 * Returns a new unique blank node of type RDFSDatatype.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSDatatype
 * @return a new unique blank node of type RDFSDatatype
 */
public RDFSDatatype createRDFSDatatype(RDFGraph graph) {
	return (RDFSDatatype) genericCreate(graph, RDFSDatatype.class, RDFUtility.rdfsdatatype) ;
}

/**
 * Returns an object of class '<em>RDFSDatatype</em>'.
 * 
 * @param graph the context graph of this RDFSDatatype.
 * @param uriRefString the RDF URI Reference of the RDFSDatatype.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSDatatype</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSDatatype createRDFSDatatype(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSDatatype(graph, this.createURIReference( uriRefString ));
}

/**
 * Returns an object of class '<em>RDFSDatatype</em>'.
 * 
 * @param graph  the context graph of this RDFSDatatype
 * @param uriRef the RDF URI Reference of the RDFSDatatype.
 * @return an object of class '<em>RDFSDatatype</em>'.
 */
public RDFSDatatype createRDFSDatatype(RDFGraph graph, URIReference uriRef) {
	return (RDFSDatatype) genericCreate(graph, RDFSDatatype.class, uriRef, RDFUtility.rdfsdatatype) ;
}

/**
 * Returns an object of class '<em>RDFSDatatype</em>'.
 * 
 * @param graph           the context graph of this RDFSDatatype
 * @param namespace       the namespace of the RDF URI Reference of the RDFSDatatype
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSDatatype
 * @return an object of class '<em>RDFSDatatype</em>'.
 */
public RDFSDatatype createRDFSDatatype(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSDatatype(graph, this.createURIReference( namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFList.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFList
 * @return a new unique blank node of type RDFList
 */
public RDFList createRDFList(RDFGraph graph) {
	return (RDFList) genericCreate(graph, RDFList.class, RDFUtility.rdflist) ;
}

/**
 * Returns an object of class '<em>RDFList</em>'.
 * 
 * @param graph the context graph of this RDFList.
 * @param uriRefString the RDF URI Reference of the RDFList.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFList</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFList createRDFList(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFList(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFList</em>'.
 * 
 * @param graph  the context graph of this RDFList
 * @param uriRef the RDF URI Reference of the RDFList.
 * @return an object of class '<em>RDFList</em>'.
 */
public RDFList createRDFList(RDFGraph graph, URIReference uriRef) {
	return (RDFList) genericCreate(graph, RDFList.class, uriRef, RDFUtility.rdflist) ;
}

/**
 * Returns an object of class '<em>RDFList</em>'.
 * 
 * @param graph           the context graph of this RDFList
 * @param namespace       the namespace of the RDF URI Reference of the RDFList
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFList
 * @return an object of class '<em>RDFList</em>'.
 */
public RDFList createRDFList(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFList(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFProperty.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFProperty
 * @return a new unique blank node of type RDFProperty
 */
public RDFProperty createRDFProperty(RDFGraph graph) {
	return (RDFProperty) genericCreate(graph, RDFProperty.class, RDFUtility.rdfproperty) ;
}

/**
 * Returns an object of class '<em>RDFProperty</em>'.
 * 
 * @param graph the context graph of this RDFProperty.
 * @param uriRefString the RDF URI Reference of the RDFProperty.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFProperty</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFProperty createRDFProperty(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFProperty(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFProperty</em>'.
 * 
 * @param graph  the context graph of this RDFProperty
 * @param uriRef the RDF URI Reference of the RDFProperty.
 * @return an object of class '<em>RDFProperty</em>'.
 */
public RDFProperty createRDFProperty(RDFGraph graph, URIReference uriRef) {
	return (RDFProperty) genericCreate(graph, RDFProperty.class, uriRef, RDFUtility.rdfproperty) ;
}

/**
 * Returns an object of class '<em>RDFProperty</em>'.
 * 
 * @param graph           the context graph of this RDFProperty
 * @param namespace       the namespace of the RDF URI Reference of the RDFProperty
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFProperty
 * @return an object of class '<em>RDFProperty</em>'.
 */
public RDFProperty createRDFProperty(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFProperty(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFStatement.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFStatement
 * @return a new unique blank node of type RDFStatement
 */
public RDFStatement createRDFStatement(RDFGraph graph) {
	return (RDFStatement) genericCreate(graph, RDFStatement.class, RDFUtility.rdfstatement) ;
}

/**
 * Returns an object of class '<em>RDFStatement</em>'.
 * 
 * @param graph the context graph of this RDFStatement.
 * @param uriRefString the RDF URI Reference of the RDFStatement.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFStatement</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFStatement createRDFStatement(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFStatement(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFStatement</em>'.
 * 
 * @param graph  the context graph of this RDFStatement
 * @param uriRef the RDF URI Reference of the RDFStatement.
 * @return an object of class '<em>RDFStatement</em>'.
 */
public RDFStatement createRDFStatement(RDFGraph graph, URIReference uriRef) {
	return (RDFStatement) genericCreate(graph, RDFStatement.class, uriRef, RDFUtility.rdfstatement) ;
}

/**
 * Returns an object of class '<em>RDFStatement</em>'.
 * 
 * @param graph           the context graph of this RDFStatement
 * @param namespace       the namespace of the RDF URI Reference of the RDFStatement
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFStatement
 * @return an object of class '<em>RDFStatement</em>'.
 */
public RDFStatement createRDFStatement(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFStatement(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFAlt.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFAlt
 * @return a new unique blank node of type RDFAlt
 */
public RDFAlt createRDFAlt(RDFGraph graph) {
	return (RDFAlt) genericCreate(graph, RDFAlt.class, RDFUtility.rdfalt) ;
}

/**
 * Returns an object of class '<em>RDFAlt</em>'.
 * 
 * @param graph the context graph of this RDFAlt.
 * @param uriRefString the RDF URI Reference of the RDFAlt.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFAlt</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFAlt createRDFAlt(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFAlt(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFAlt</em>'.
 * 
 * @param graph  the context graph of this RDFAlt
 * @param uriRef the RDF URI Reference of the RDFAlt.
 * @return an object of class '<em>RDFAlt</em>'.
 */
public RDFAlt createRDFAlt(RDFGraph graph, URIReference uriRef) {
	return (RDFAlt) genericCreate(graph, RDFAlt.class, uriRef, RDFUtility.rdfalt) ;
}

/**
 * Returns an object of class '<em>RDFAlt</em>'.
 * 
 * @param graph           the context graph of this RDFAlt
 * @param namespace       the namespace of the RDF URI Reference of the RDFAlt
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFAlt
 * @return an object of class '<em>RDFAlt</em>'.
 */
public RDFAlt createRDFAlt(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFAlt(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFBag.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFBag
 * @return a new unique blank node of type RDFBag
 */
public RDFBag createRDFBag(RDFGraph graph) {
	return (RDFBag) genericCreate(graph, RDFBag.class, RDFUtility.rdfbag) ;
}

/**
 * Returns an object of class '<em>RDFBag</em>'.
 * 
 * @param graph the context graph of this RDFBag.
 * @param uriRefString the RDF URI Reference of the RDFBag.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFBag</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFBag createRDFBag(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFBag(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFBag</em>'.
 * 
 * @param graph  the context graph of this RDFBag
 * @param uriRef the RDF URI Reference of the RDFBag.
 * @return an object of class '<em>RDFBag</em>'.
 */
public RDFBag createRDFBag(RDFGraph graph, URIReference uriRef) {
	return (RDFBag) genericCreate(graph, RDFBag.class, uriRef, RDFUtility.rdfbag) ;
}

/**
 * Returns an object of class '<em>RDFBag</em>'.
 * 
 * @param graph           the context graph of this RDFBag
 * @param namespace       the namespace of the RDF URI Reference of the RDFBag
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFBag
 * @return an object of class '<em>RDFBag</em>'.
 */
public RDFBag createRDFBag(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFBag(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFSeq.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSeq
 * @return a new unique blank node of type RDFSeq
 */
public RDFSeq createRDFSeq(RDFGraph graph) {
	return (RDFSeq) genericCreate(graph, RDFSeq.class, RDFUtility.rdfseq) ;
}

/**
 * Returns an object of class '<em>RDFSeq</em>'.
 * 
 * @param graph the context graph of this RDFSeq.
 * @param uriRefString the RDF URI Reference of the RDFSeq.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSeq</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSeq createRDFSeq(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSeq(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFSeq</em>'.
 * 
 * @param graph  the context graph of this RDFSeq
 * @param uriRef the RDF URI Reference of the RDFSeq.
 * @return an object of class '<em>RDFSeq</em>'.
 */
public RDFSeq createRDFSeq(RDFGraph graph, URIReference uriRef) {
	return (RDFSeq) genericCreate(graph, RDFSeq.class, uriRef, RDFUtility.rdfseq) ;
}

/**
 * Returns an object of class '<em>RDFSeq</em>'.
 * 
 * @param graph           the context graph of this RDFSeq
 * @param namespace       the namespace of the RDF URI Reference of the RDFSeq
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSeq
 * @return an object of class '<em>RDFSeq</em>'.
 */
public RDFSeq createRDFSeq(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSeq(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFSContainer.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSContainer
 * @return a new unique blank node of type RDFSContainer
 */
public RDFSContainer createRDFSContainer(RDFGraph graph) {
	return (RDFSContainer) genericCreate(graph, RDFSContainer.class,  RDFUtility.rdfscontainer) ;
}

/**
 * Returns an object of class '<em>RDFSContainer</em>'.
 * 
 * @param graph the context graph of this RDFSContainer.
 * @param uriRefString the RDF URI Reference of the RDFSContainer.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSContainer</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSContainer createRDFSContainer(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSContainer(graph, this.createURIReference(uriRefString));
}

/**
 * Returns an object of class '<em>RDFSContainer</em>'.
 * 
 * @param graph  the context graph of this RDFSContainer
 * @param uriRef the RDF URI Reference of the RDFSContainer.
 * @return an object of class '<em>RDFSContainer</em>'.
 */
public RDFSContainer createRDFSContainer(RDFGraph graph, URIReference uriRef) {
	return (RDFSContainer) genericCreate(graph, RDFSContainer.class, uriRef, RDFUtility.rdfscontainer) ;
}

/**
 * Returns an object of class '<em>RDFSContainer</em>'.
 * 
 * @param graph           the context graph of this RDFSContainer
 * @param namespace       the namespace of the RDF URI Reference of the RDFSContainer
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSContainer
 * @return an object of class '<em>RDFSContainer</em>'.
 */
public RDFSContainer createRDFSContainer(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSContainer(graph, this.createURIReference(namespace, localNameString));
}

/**
 * Returns a new unique blank node of type RDFSContainerMembershipProperty.
 * The resource can not be given a URI Reference later.
 *
 * @param graph the context graph of this RDFSContainerMembershipProperty
 * @return a new unique blank node of type RDFSContainerMembershipProperty
 */
public RDFSContainerMembershipProperty createRDFSContainerMembershipProperty(RDFGraph graph) {
	return (RDFSContainerMembershipProperty) genericCreate(graph, RDFSContainerMembershipProperty.class, RDFUtility.rdfscontainermembershipproperty) ;
}

/**
 * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
 * 
 * @param graph the context graph of this RDFSContainerMembershipProperty.
 * @param uriRefString the RDF URI Reference of the RDFSContainerMembershipProperty.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public RDFSContainerMembershipProperty createRDFSContainerMembershipProperty(RDFGraph graph, String uriRefString)
		throws URISyntaxException {
	return createRDFSContainerMembershipProperty(graph, this.createURIReference( uriRefString ));
}

/**
 * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
 * 
 * @param graph  the context graph of this RDFSContainerMembershipProperty
 * @param uriRef the RDF URI Reference of the RDFSContainerMembershipProperty.
 * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
 */
public RDFSContainerMembershipProperty createRDFSContainerMembershipProperty(RDFGraph graph, URIReference uriRef) {
	return (RDFSContainerMembershipProperty) genericCreate(graph, RDFSContainerMembershipProperty.class, uriRef, RDFUtility.rdfscontainermembershipproperty) ;
}

/**
 * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
 * 
 * @param graph           the context graph of this RDFSContainerMembershipProperty
 * @param namespace       the namespace of the RDF URI Reference of the RDFSContainerMembershipProperty
 * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSContainerMembershipProperty
 * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
 */
public RDFSContainerMembershipProperty createRDFSContainerMembershipProperty(RDFGraph graph, Namespace namespace, String localNameString) {
	return createRDFSContainerMembershipProperty(graph, this.createURIReference( namespace, localNameString));
}

/**
 * Returns an object of class '<em>Document</em>'.
 * 
 * @param uriRefString the RDF URI Reference of the Document.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>Document</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a 
 *         valid RDF URI Reference. 
 */
public Document createDocument( String uriRefString )
throws URISyntaxException
{
	return createDocument(this.createURIReference(uriRefString));
}
 

/**
 * Returns an object of class '<em>Document</em>'.
 * 
 * @param uriRef the RDF URI Reference of the Document.
 * @return an object of class '<em>Document</em>'.
 */
public Document createDocument( URIReference uriRef )
{
	Document document = RDFWebFactory.eINSTANCE.createDocument();
	document.getUriRef().add(uriRef);
	return document;
}

/**
 * Returns an object of class '<em>Document</em>'.
 * 
 * @param namespace       the namespace of the RDF URI Reference of the Document
 * @param localNameString the fragment identifier of the RDF URI Reference of the Document
 * @return an object of class '<em>Document</em>'.
 */
public Document createDocument( Namespace namespace, String localNameString )
{
	return createDocument(this.createURIReference( namespace, localNameString));
}

/**
 * Returns an object of class '<em>LocalName</em>'.
 * 
 * @param name the local name string. The string can not be changed after
 *        the LocalName object is created.
 * @return an object of class '<em>LocalName</em>'.
 */
public LocalName createLocalName( String name )
{
	LocalName localName = RDFWebFactory.eINSTANCE.createLocalName();
	localName.setName(name);
	return localName;
}

/**
 * Returns an object of class '<em>Namespace</em>'.
 * 
 * @param uriRefString  the RDF URI Reference of this namespace which can
 *        not be changed after the namespace object is created.
 *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
 * @return an object of class '<em>Namespace</em>'.
 * @throws URISyntaxException if the <code>uriRefString</code> is not a valid
 *         RDF URI Reference.
 */
public Namespace createNamespace( String uriRefString )
throws URISyntaxException{
	Namespace ns =  RDFWebFactory.eINSTANCE.createNamespace();
	((NamespaceImpl)ns).setNamespaceURIRef( this.createURIReference( uriRefString ));
	return ns;
}

/**
 * Returns an object of class '<em>Namespace</em>'.
 * 
 * @param uriRef the RDF URI Reference of this namespace which can not 
 *        be changed after the namespace object is created.
 * @return an object of class '<em>Namespace</em>'.
 */
public Namespace createNamespace( URIReference uriRef ){
	Namespace ns =  RDFWebFactory.eINSTANCE.createNamespace();
	((NamespaceImpl)ns).setNamespaceURIRef(uriRef);
	return ns;
}

/**
 * Returns an object of class '<em>NamespaceDefinition</em>'.
 * 
 * @param prefix the prefix of the namespace definition. It may be
 *        <code>null</code> and be set later.
 * @param namespace the namespace of this definition. It may be <code>null</code>
 *        and be set later. 
 * @return an object of class '<em>NamespaceDefinition</em>'.
 */
public NamespaceDefinition createNamespaceDefinition( String prefix, Namespace namespace )
{
	NamespaceDefinition nsd = RDFWebFactory.eINSTANCE.createNamespaceDefinition();
	nsd.setNamespace(namespace);
	nsd.setNamespacePrefix(prefix);
	return nsd;
}


}
