/*****************************************************************************
 * Copyright (c) 2022, 2024 CEA LIST.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) <vincent.lorenzo@cea.fr> - Initial API and implementation
 *  Obeo - Improvement on creation checks
 *****************************************************************************/
package org.eclipse.papyrus.sirius.junit.util.diagram;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.sirius.junit.utils.diagram.creation.checker.SemanticAndGraphicalCreationChecker;
import org.eclipse.papyrus.sirius.junit.utils.diagram.creation.graphical.checker.GraphicalOwnerUtils;
import org.eclipse.sirius.diagram.DDiagram;
import org.eclipse.sirius.diagram.DDiagramElement;
import org.eclipse.sirius.diagram.DNode;
import org.eclipse.sirius.diagram.DNodeContainer;
import org.eclipse.sirius.viewpoint.DMappingBased;
import org.eclipse.sirius.viewpoint.DRepresentationElement;
import org.eclipse.uml2.uml.Package;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;

/**
 * Abstract Test for Node Creation
 */
public abstract class AbstractCreateNodeTests extends AbstractSiriusDiagramTests {

	/**
	 * the root of the model
	 */
	protected Package root;

	protected List<?> previousChildren;

	/**
	 * This method set up the test environment
	 */
	@Before
	public void setUp() {
		this.root = this.fixture.getModel();
	}

	/**
	 * Returns the semantic owner of created element.
	 * <p>
	 * Not used by abstraction, can be override freely.
	 * </p>
	 *
	 * @return
	 *         the semantic owner to use for the test
	 */
	protected EObject getSemanticOwner() {
		return null;
	}

	/**
	 * Returns the view container of created view of created element.
	 * <p>
	 * Not used by abstraction, can be override freely.
	 * </p>
	 *
	 * @return
	 *         the "top" node of the graphical container
	 */
	protected EObject getTopGraphicalContainer() {
		return null;
	}

	/**
	 *
	 * @param mappingID
	 *            the mapping used by the subnode
	 * @return
	 *         the subnode of the top graphical Container with the expected mappingID. basically this method is used to find a compartment in the node
	 */
	protected EObject getSubNodeOfGraphicalContainer(final String mappingID) {
		final EObject container = getTopGraphicalContainer();
		if (container instanceof DMappingBased && ((DMappingBased) container).getMapping().getName().equals(mappingID)) {
			return container;
		}
		if (container instanceof DNodeContainer) {
			for (final DDiagramElement diagramElement : ((DNodeContainer) container).getOwnedDiagramElements()) {
				if (mappingID.equals(diagramElement.getMapping().getName())) {
					return diagramElement;
				}
			}
		}
		if(container instanceof DNode) {
			for(final DDiagramElement diagramElement : ((DNode) container).getOwnedBorderedNodes()) {
				if(mappingID.equals(diagramElement.getMapping().getName())) {
					return diagramElement;
				}
			}
		}
		return null;
	}

	/**
	 * This method creates the node and checks that the diagram is unsynchronized
	 *
	 * @param creationToolId
	 *            the ID of the creation tool to used
	 * @param checker
	 *            the checker used to validate the creation of the node
	 * @param graphicalContainer
	 *            the graphical container to use to create the node
	 *
	 * @return the created view
	 */
	protected DRepresentationElement createNode(final String creationToolId, final SemanticAndGraphicalCreationChecker checker, final EObject graphicalContainer) {
		checkSiriusDiagramSynchronization(isSynchronization());

		beforeCreation(graphicalContainer);
		boolean result = applyCreationTool(creationToolId, getDDiagram(), graphicalContainer);
		Assert.assertTrue("The creation of element failed", result); //$NON-NLS-1$
		fixture.flushDisplayEvents();

		// Use the last element in the list if the graphical container already contains elements.
		DRepresentationElement createdView = retrieveCreatedView(graphicalContainer);
		checker.validateRepresentationElement(createdView);

		// undo
		fixture.getEditingDomain().getCommandStack().undo();
		fixture.flushDisplayEvents();
		checker.validateAfterUndo();

		// redo
		fixture.getEditingDomain().getCommandStack().redo();
		fixture.flushDisplayEvents();
		checker.validateAfterRedo();

		return createdView;
	}

	/**
	 * Prepares the state of the test before execution the creation tool.
	 *
	 * @param viewContainer
	 *            container in which the element is created.
	 */
	protected void beforeCreation(final EObject viewContainer) {
		previousChildren = List.copyOf(GraphicalOwnerUtils.getChildren(viewContainer));
	}

	/**
	 * Retrieves the created element representation.
	 *
	 * @param viewContainer
	 *            container in which the element is created.
	 * @return created element
	 */
	protected DRepresentationElement retrieveCreatedView(EObject viewContainer) {
		List<DDiagramElement> children = new ArrayList<>(GraphicalOwnerUtils.getChildren(viewContainer));
		children.removeAll(previousChildren);

		return retrieveFromViews(children);
	}


	/**
	 * Retrieves the created element representation from new views.
	 * <p>
	 * By default, the list must have 1 element.
	 * </p>
	 *
	 * @param children
	 *            list of created views
	 * @return created element
	 */
	protected DRepresentationElement retrieveFromViews(List<DDiagramElement> children) {
		Assert.assertEquals(1, children.size()); // Creation test only support 1 view.
		return children.get(0);
	}

	/**
	 * Applies the node creation tool with the provided parameters.
	 *
	 * @param creationToolId
	 *            the ID of the creation tool to use
	 * @param diagram
	 *            the diagram containing the tool to use
	 * @param graphicalContainer
	 *            the graphical container to use to create the node
	 * @return <code>true</code> if the tool could be applied, <code>false</code> otherwise
	 */
	protected boolean applyCreationTool(String creationToolId, DDiagram diagram, EObject graphicalContainer) {
		return fixture.applyContainerCreationTool(creationToolId, diagram, graphicalContainer);
	}

	/**
	 * This method clean the test environment
	 */
	@After
	public void tearDown() {
		this.root = null;
	}
}
