/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.component;

import java.util.List;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ManualGeneration;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.core.utils.FunctionUtils;
import org.eclipse.papyrus.robotics.core.utils.ParameterUtils;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.functions.FunctionKind;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.CallbackGroups;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.cpp.utils.RosCppTypes;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtend2.lib.StringConcatenation;

@SuppressWarnings("all")
public class CreateMain {
  /**
   * register a ROS 2 components
   */
  public static CharSequence registerComponent(final org.eclipse.uml2.uml.Class component) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("#include \"rclcpp_components/register_node_macro.hpp\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// Register the component with class_loader.");
    _builder.newLine();
    _builder.append("// This acts as a sort of entry point, allowing the component to be discoverable when its library");
    _builder.newLine();
    _builder.append("// is being loaded into a running process.");
    _builder.newLine();
    _builder.append("RCLCPP_COMPONENTS_REGISTER_NODE(");
    String _qualifiedName = component.getQualifiedName();
    _builder.append(_qualifiedName);
    _builder.append(")");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Create the main entry point for a class
   */
  public static void createMain(final org.eclipse.uml2.uml.Class component) {
    org.eclipse.uml2.uml.Package _nearestPackage = component.getNearestPackage();
    String _name = component.getName();
    String _plus = (_name + "_main");
    final org.eclipse.uml2.uml.Class main = _nearestPackage.createOwnedClass(_plus, false);
    component.createDependency(main);
    StereotypeUtil.apply(main, ManualGeneration.class);
    final Include include = StereotypeUtil.<Include>applyApp(main, Include.class);
    include.setBody(CreateMain.createMainCode(component).toString());
  }

  public static CharSequence lambdaStart(final org.eclipse.uml2.uml.Class component) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("[&");
    String _instName = ComponentUtils.getInstName(component);
    _builder.append(_instName);
    _builder.append("](const rclcpp_lifecycle::State&) {");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static CharSequence lambdaEnd() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("return rclcpp_lifecycle::node_interfaces::LifecycleNodeInterface::CallbackReturn::SUCCESS;");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence createMainCode(final org.eclipse.uml2.uml.Class component) {
    StringConcatenation _builder = new StringConcatenation();
    String _instName = ComponentUtils.getInstName(component);
    final String instRef = (_instName + "->");
    _builder.newLineIfNotEmpty();
    final CharSequence onConfigure = CreateMain.onLifecycleEvent(component, instRef, FunctionKind.ON_CONFIGURE);
    _builder.newLineIfNotEmpty();
    final CharSequence onActivate = CreateMain.onActivate(component, instRef);
    _builder.newLineIfNotEmpty();
    final CharSequence onDeActivate = CreateMain.onQuitActive(component, instRef, FunctionKind.ON_DEACTIVATE);
    _builder.newLineIfNotEmpty();
    final CharSequence onShutdown = CreateMain.onQuitActive(component, instRef, FunctionKind.ON_SHUTDOWN);
    _builder.newLineIfNotEmpty();
    final CharSequence onCleanup = CreateMain.onLifecycleEvent(component, instRef, FunctionKind.ON_CLEANUP);
    _builder.newLineIfNotEmpty();
    String _qualifiedName = component.getQualifiedName();
    String _postfix = ActivityUtils.getPostfix(component);
    final String compClassName = (_qualifiedName + _postfix);
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"");
    String _name = component.getNearestPackage().getName();
    _builder.append(_name);
    _builder.append("/");
    String _name_1 = component.getName();
    _builder.append(_name_1);
    String _postfix_1 = ActivityUtils.getPostfix(component);
    _builder.append(_postfix_1);
    _builder.append(".h\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      boolean _hasPeriodicActivities = ActivityUtils.hasPeriodicActivities(component);
      if (_hasPeriodicActivities) {
        _builder.append("using namespace std::chrono_literals;");
        _builder.newLine();
      }
    }
    CreateMain.createTimer(component);
    _builder.newLineIfNotEmpty();
    _builder.append("// declare options");
    _builder.newLine();
    _builder.append("rclcpp::NodeOptions ");
    String _instName_1 = ComponentUtils.getInstName(component);
    _builder.append(_instName_1);
    _builder.append("_options;");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("int main(int argc, char **argv) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rclcpp::init(argc, argv);");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("auto ");
    String _instName_2 = ComponentUtils.getInstName(component);
    _builder.append(_instName_2, "\t");
    _builder.append(" = std::make_shared<");
    _builder.append(compClassName, "\t");
    _builder.append(">(");
    String _instName_3 = ComponentUtils.getInstName(component);
    _builder.append(_instName_3, "\t");
    _builder.append("_options);");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("RCLCPP_INFO(");
    _builder.append(instRef, "\t");
    _builder.append("get_logger(), \"");
    String _name_2 = component.getName();
    _builder.append(_name_2, "\t");
    _builder.append(" has been initialized\");");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      int _length = onConfigure.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("register_on_configure(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _lambdaStart = CreateMain.lambdaStart(component);
        _builder.append(_lambdaStart, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append(onConfigure, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        CharSequence _lambdaEnd = CreateMain.lambdaEnd();
        _builder.append(_lambdaEnd, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(");");
        _builder.newLine();
      }
    }
    {
      int _length_1 = onActivate.length();
      boolean _greaterThan_1 = (_length_1 > 0);
      if (_greaterThan_1) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("register_on_activate(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _lambdaStart_1 = CreateMain.lambdaStart(component);
        _builder.append(_lambdaStart_1, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append(onActivate, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        CharSequence _lambdaEnd_1 = CreateMain.lambdaEnd();
        _builder.append(_lambdaEnd_1, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(");");
        _builder.newLine();
      }
    }
    {
      int _length_2 = onDeActivate.length();
      boolean _greaterThan_2 = (_length_2 > 0);
      if (_greaterThan_2) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("register_on_deactivate(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _lambdaStart_2 = CreateMain.lambdaStart(component);
        _builder.append(_lambdaStart_2, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append(onDeActivate, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        CharSequence _lambdaEnd_2 = CreateMain.lambdaEnd();
        _builder.append(_lambdaEnd_2, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(");");
        _builder.newLine();
      }
    }
    {
      int _length_3 = onShutdown.length();
      boolean _greaterThan_3 = (_length_3 > 0);
      if (_greaterThan_3) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("register_on_shutdown(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _lambdaStart_3 = CreateMain.lambdaStart(component);
        _builder.append(_lambdaStart_3, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append(onShutdown, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        CharSequence _lambdaEnd_3 = CreateMain.lambdaEnd();
        _builder.append(_lambdaEnd_3, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(");");
        _builder.newLine();
      }
    }
    {
      int _length_4 = onCleanup.length();
      boolean _greaterThan_4 = (_length_4 > 0);
      if (_greaterThan_4) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("register_on_cleanup(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _lambdaStart_4 = CreateMain.lambdaStart(component);
        _builder.append(_lambdaStart_4, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        _builder.append(onCleanup, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t\t");
        CharSequence _lambdaEnd_4 = CreateMain.lambdaEnd();
        _builder.append(_lambdaEnd_4, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(");");
        _builder.newLine();
      }
    }
    {
      int _size = ParameterUtils.getAllParameters(component).size();
      boolean _greaterThan_5 = (_size > 0);
      if (_greaterThan_5) {
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("declareParameters();");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append(instRef, "\t");
        _builder.append("initParameterVars();");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rclcpp::executors::MultiThreadedExecutor executor;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("executor.add_node(");
    _builder.append(instRef, "\t");
    _builder.append("get_node_base_interface());");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("executor.spin();");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rclcpp::shutdown();");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static void createTimer(final org.eclipse.uml2.uml.Class component) {
    List<Activity> _activities = ActivityUtils.getActivities(component);
    for (final Activity activity : _activities) {
      {
        final String period = ActivityUtils.getPeriod(activity);
        if ((period != null)) {
          final Type timerBase = RosCppTypes.getType(component, "ros2Library::rclcpp::timer::TimerBase");
          final Type cbgType = RosCppTypes.getType(component, "ros2Library::rclcpp::CallbackGroup");
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("timer_");
          String _name = activity.getBase_Class().getName();
          _builder.append(_name);
          _builder.append("_");
          final Property timer = component.createOwnedAttribute(_builder.toString(), timerBase);
          final Property t_cbg = component.createOwnedAttribute(CallbackGroups.tCallbackGroupName(activity.getBase_Class()), cbgType);
          RosHelpers.useSharedPtr(timer);
          RosHelpers.useSharedPtr(t_cbg);
        }
      }
    }
  }

  /**
   * Create a function to call during the state transition towards a certain
   * state. Calls the function with the passed "kind"
   */
  public static CharSequence onLifecycleEvent(final org.eclipse.uml2.uml.Class component, final String instRef, final FunctionKind kind) {
    StringConcatenation _builder = new StringConcatenation();
    {
      List<Activity> _activities = ActivityUtils.getActivities(component);
      for(final Activity activity : _activities) {
        final List<Behavior> associatedFcts = FunctionUtils.getFunctions(activity, kind);
        _builder.newLineIfNotEmpty();
        {
          int _size = associatedFcts.size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            {
              for(final Behavior associatedFct : associatedFcts) {
                _builder.append(instRef);
                String _name = associatedFct.getName();
                _builder.append(_name);
                _builder.append("();");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    return _builder;
  }

  /**
   * Create a function to call during the state transition towards activation.
   * In particular, call the functions with "kind" ON_ACTIVATE and start periodic timers
   */
  public static CharSequence onActivate(final org.eclipse.uml2.uml.Class component, final String instRef) {
    StringConcatenation _builder = new StringConcatenation();
    {
      List<Activity> _activities = ActivityUtils.getActivities(component);
      for(final Activity activity : _activities) {
        final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
        _builder.newLineIfNotEmpty();
        final String period = ActivityUtils.getPeriod(activity);
        _builder.newLineIfNotEmpty();
        final List<Behavior> activateFcts = FunctionUtils.getFunctions(activity, FunctionKind.ON_ACTIVATE);
        _builder.newLineIfNotEmpty();
        final List<Behavior> periodicFcts = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC);
        _builder.newLineIfNotEmpty();
        {
          int _size = activateFcts.size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            {
              for(final Behavior activateFct : activateFcts) {
                _builder.append(instRef);
                String _name = activateFct.getName();
                _builder.append(_name);
                _builder.append("();");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        {
          if (((period != null) && (periodicFcts.size() > 0))) {
            _builder.append("// periodic execution (");
            _builder.append(period);
            _builder.append(") for ");
            String _name_1 = activityCl.getName();
            _builder.append(_name_1);
            _builder.append(" using a wall timer");
            _builder.newLineIfNotEmpty();
            _builder.append("// create a callback group for the timer");
            _builder.newLine();
            {
              for(final Behavior periodicFct : periodicFcts) {
                _builder.append(instRef);
                _builder.append("timer_");
                String _name_2 = activityCl.getName();
                _builder.append(_name_2);
                _builder.append("_ = ");
                _builder.append(instRef);
                _builder.append("create_wall_timer(");
                _builder.append(period);
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("std::bind(&");
                String _name_3 = component.getNearestPackage().getName();
                _builder.append(_name_3, "\t");
                _builder.append("::");
                String _name_4 = component.getName();
                _builder.append(_name_4, "\t");
                String _postfix = ActivityUtils.getPostfix(component);
                _builder.append(_postfix, "\t");
                _builder.append("::");
                String _name_5 = periodicFct.getName();
                _builder.append(_name_5, "\t");
                _builder.append(", ");
                String _instName = ComponentUtils.getInstName(component);
                _builder.append(_instName, "\t");
                _builder.append("), ");
                _builder.append(instRef, "\t");
                String _tCallbackGroupName = CallbackGroups.tCallbackGroupName(activityCl);
                _builder.append(_tCallbackGroupName, "\t");
                _builder.append(");");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    return _builder;
  }

  /**
   * Create a function to call during the state transition leaving the active state (either shutdown or deactivate).
   * In particular, call the functions with "kind" ON_DEACTIVATE and stop periodic timers
   */
  public static CharSequence onQuitActive(final org.eclipse.uml2.uml.Class component, final String instRef, final FunctionKind kind) {
    StringConcatenation _builder = new StringConcatenation();
    {
      List<Activity> _activities = ActivityUtils.getActivities(component);
      for(final Activity activity : _activities) {
        final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
        _builder.newLineIfNotEmpty();
        final String period = ActivityUtils.getPeriod(activity);
        _builder.newLineIfNotEmpty();
        final List<Behavior> associatedFcts = FunctionUtils.getFunctions(activity, kind);
        _builder.newLineIfNotEmpty();
        final List<Behavior> periodicFcts = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC);
        _builder.newLineIfNotEmpty();
        {
          if ((((period != null) && (periodicFcts.size() > 0)) || (associatedFcts.size() > 0))) {
            {
              for(final Behavior activateFct : associatedFcts) {
                _builder.append(instRef);
                String _name = activateFct.getName();
                _builder.append(_name);
                _builder.append("();");
                _builder.newLineIfNotEmpty();
              }
            }
            {
              if (((period != null) && (periodicFcts.size() > 0))) {
                _builder.append("// stop timer of ");
                String _name_1 = activityCl.getName();
                _builder.append(_name_1);
                _builder.newLineIfNotEmpty();
                _builder.append(instRef);
                _builder.append("timer_");
                String _name_2 = activityCl.getName();
                _builder.append(_name_2);
                _builder.append("_->cancel();");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    return _builder;
  }
}
