/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.component;

import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ApplyProfiles;
import org.eclipse.papyrus.robotics.codegen.common.utils.TopicUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentPort;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.CallbackGroups;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.cpp.utils.RosCppTypes;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Manage constructor creation, including port
 */
@SuppressWarnings("all")
public class Constructor {
  /**
   * Create constructor and (if required) callback methods for handlers
   */
  public static boolean createConstructor(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      final Type lcNodeSC = RosCppTypes.getType(component, "ros2Library::rclcpp_lifecycle::LifecycleNode");
      if ((lcNodeSC instanceof Classifier)) {
        component.createGeneralization(((Classifier)lcNodeSC));
      }
      final Operation op = Constructor.addConstrOp(component);
      _xblockexpression = Constructor.addConstrMethod(component, op);
    }
    return _xblockexpression;
  }

  /**
   * Add an init operation (constructor) that creates the ROS 2 publishers/subscribers/clients/... for ports
   */
  public static Operation addConstrOp(final org.eclipse.uml2.uml.Class component) {
    final Type nodeOptions = RosCppTypes.getType(component, "ros2Library::rclcpp::NodeOptions");
    final Operation init = component.createOwnedOperation(component.getName(), null, null);
    init.createOwnedParameter("options", nodeOptions);
    Create create = StereotypeUtil.<Create>applyApp(init, Create.class);
    if ((create == null)) {
      ApplyProfiles.applyStdProfile(init);
      create = StereotypeUtil.<Create>applyApp(init, Create.class);
    }
    ConstInit constInit = StereotypeUtil.<ConstInit>applyApp(init, ConstInit.class);
    if ((constInit == null)) {
      ApplyProfiles.applyCppProfile(init);
      constInit = StereotypeUtil.<ConstInit>applyApp(init, ConstInit.class);
    }
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("rclcpp_lifecycle::LifecycleNode(\"");
    String _name = component.getName();
    _builder.append(_name);
    _builder.append("\", options)");
    constInit.setInitialisation(_builder.toString());
    return init;
  }

  /**
   * Add a method body to the constructor operation
   */
  public static boolean addConstrMethod(final org.eclipse.uml2.uml.Class component, final Operation constructorOp) {
    boolean _xblockexpression = false;
    {
      Behavior _createOwnedBehavior = component.createOwnedBehavior(component.getName(), UMLPackage.eINSTANCE.getOpaqueBehavior());
      final OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      constructorOp.getMethods().add(ob);
      ob.getLanguages().add("C++");
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          Constructor.createCallbackGroupVar(activity, component);
          _builder.newLineIfNotEmpty();
          CharSequence _createCallbackGroupCode = Constructor.createCallbackGroupCode(activity);
          _builder.append(_createCallbackGroupCode);
          _builder.newLineIfNotEmpty();
        }
      }
      {
        EList<Port> _allPorts = PortUtils.getAllPorts(component);
        for(final Port port : _allPorts) {
          {
            CommunicationPattern _communicationPattern = InteractionUtils.getCommunicationPattern(port);
            boolean _tripleNotEquals = (_communicationPattern != null);
            if (_tripleNotEquals) {
              final CommunicationPattern pattern = InteractionUtils.getCommunicationPattern(port);
              _builder.newLineIfNotEmpty();
              {
                if ((InteractionUtils.isPush(pattern) || InteractionUtils.isPubSub(pattern))) {
                  CharSequence _createPush = Constructor.createPush(port);
                  _builder.append(_createPush);
                  _builder.newLineIfNotEmpty();
                  _builder.newLine();
                } else {
                  boolean _isSend = InteractionUtils.isSend(pattern);
                  if (_isSend) {
                    CharSequence _createSend = Constructor.createSend(port);
                    _builder.append(_createSend);
                    _builder.newLineIfNotEmpty();
                    _builder.newLine();
                  } else {
                    boolean _isQuery = InteractionUtils.isQuery(pattern);
                    if (_isQuery) {
                      CharSequence _createQuery = Constructor.createQuery(port);
                      _builder.append(_createQuery);
                      _builder.newLineIfNotEmpty();
                      _builder.newLine();
                    } else {
                      boolean _isAction = InteractionUtils.isAction(pattern);
                      if (_isAction) {
                        CharSequence _createAction = Constructor.createAction(port);
                        _builder.append(_createAction);
                        _builder.newLineIfNotEmpty();
                        _builder.newLine();
                      } else {
                        boolean _isEvent = InteractionUtils.isEvent(pattern);
                        if (_isEvent) {
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
      _builder.newLine();
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  /**
   * Create a callback_group attribute for each activity
   */
  public static void createCallbackGroupVar(final Activity activity, final org.eclipse.uml2.uml.Class component) {
    final Type cbgType = RosCppTypes.getType(component, "ros2Library::rclcpp::CallbackGroup");
    final Property cbgAttr = component.createOwnedAttribute(CallbackGroups.callbackGroupName(activity.getBase_Class()), cbgType);
    RosHelpers.useSharedPtr(cbgAttr);
  }

  /**
   * Add the code that creates a callback group for each activity and optionally
   * also one for the timer
   */
  public static CharSequence createCallbackGroupCode(final Activity activity) {
    StringConcatenation _builder = new StringConcatenation();
    String _callbackGroupName = CallbackGroups.callbackGroupName(activity.getBase_Class());
    _builder.append(_callbackGroupName);
    _builder.append(" = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);");
    _builder.newLineIfNotEmpty();
    {
      String _period = ActivityUtils.getPeriod(activity);
      boolean _tripleNotEquals = (_period != null);
      if (_tripleNotEquals) {
        String _tCallbackGroupName = CallbackGroups.tCallbackGroupName(activity.getBase_Class());
        _builder.append(_tCallbackGroupName);
        _builder.append(" = create_callback_group(rclcpp::CallbackGroupType::MutuallyExclusive);");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    return _builder;
  }

  /**
   * Create a publisher or subscriber
   */
  public static CharSequence createPush(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("rclcpp::PublisherOptions ");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_options;");
        _builder.newLineIfNotEmpty();
        String _varName_1 = ElementUtils.varName(port);
        _builder.append(_varName_1);
        _builder.append("_options.callback_group = ");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        String _varName_2 = ElementUtils.varName(port);
        _builder.append(_varName_2);
        _builder.append("_pub_ = create_publisher<");
        String _externalName = RosHelpers.externalName(InteractionUtils.getCommObject(port));
        _builder.append(_externalName);
        _builder.append(">(\"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        String _qos = Constructor.qos(port, "1");
        _builder.append(_qos, "\t\t");
        _builder.append(", ");
        String _varName_3 = ElementUtils.varName(port);
        _builder.append(_varName_3, "\t\t");
        _builder.append("_options);");
        _builder.newLineIfNotEmpty();
        _builder.append("// directly activate a publisher");
        _builder.newLine();
        String _varName_4 = ElementUtils.varName(port);
        _builder.append(_varName_4);
        _builder.append("_pub_->on_activate();");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _builder.append("rclcpp::SubscriptionOptions ");
          String _varName_5 = ElementUtils.varName(port);
          _builder.append(_varName_5);
          _builder.append("_options;");
          _builder.newLineIfNotEmpty();
          String _varName_6 = ElementUtils.varName(port);
          _builder.append(_varName_6);
          _builder.append("_options.callback_group = ");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1);
          _builder.append(";");
          _builder.newLineIfNotEmpty();
          final String defaultQoS = "rclcpp::QoS(rclcpp::KeepLast(100)).best_effort()";
          _builder.newLineIfNotEmpty();
          String _varName_7 = ElementUtils.varName(port);
          _builder.append(_varName_7);
          _builder.append("_sub_ = create_subscription<");
          String _externalName_1 = RosHelpers.externalName(InteractionUtils.getCommObject(port));
          _builder.append(_externalName_1);
          _builder.append(">(\"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\",");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          String _qos_1 = Constructor.qos(port, defaultQoS);
          _builder.append(_qos_1, "\t");
          _builder.append(",");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          String _callBackMethodForPush = Callbacks.callBackMethodForPush(port.getClass_(), port);
          _builder.append(_callBackMethodForPush, "\t");
          _builder.append(", ");
          String _varName_8 = ElementUtils.varName(port);
          _builder.append(_varName_8, "\t");
          _builder.append("_options);");
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  /**
   * Create send (publisher and subscriber as well)
   */
  public static CharSequence createSend(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("rclcpp::SubscriptionOptions ");
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_options;");
        _builder.newLineIfNotEmpty();
        String _varName_1 = ElementUtils.varName(port);
        _builder.append(_varName_1);
        _builder.append("_options.callback_group = ");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        final String defaultQoS = "rclcpp::QoS(rclcpp::KeepLast(100)).best_effort()";
        _builder.newLineIfNotEmpty();
        String _varName_2 = ElementUtils.varName(port);
        _builder.append(_varName_2);
        _builder.append("_recv_ = create_subscription<");
        String _externalName = RosHelpers.externalName(InteractionUtils.getCommObject(port));
        _builder.append(_externalName);
        _builder.append(">(\"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _qos = Constructor.qos(port, defaultQoS);
        _builder.append(_qos, "\t");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _callBackMethodForPush = Callbacks.callBackMethodForPush(port.getClass_(), port);
        _builder.append(_callBackMethodForPush, "\t");
        _builder.append(", ");
        String _varName_3 = ElementUtils.varName(port);
        _builder.append(_varName_3, "\t");
        _builder.append("_options);");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _builder.append("rclcpp::PublisherOptions ");
          String _varName_4 = ElementUtils.varName(port);
          _builder.append(_varName_4);
          _builder.append("_options;");
          _builder.newLineIfNotEmpty();
          String _varName_5 = ElementUtils.varName(port);
          _builder.append(_varName_5);
          _builder.append("_options.callback_group = ");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1);
          _builder.append(";");
          _builder.newLineIfNotEmpty();
          String _varName_6 = ElementUtils.varName(port);
          _builder.append(_varName_6);
          _builder.append("_send_ = create_publisher<");
          String _externalName_1 = RosHelpers.externalName(InteractionUtils.getCommObject(port));
          _builder.append(_externalName_1);
          _builder.append(">(\"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\",");
          _builder.newLineIfNotEmpty();
          _builder.append("\t\t");
          String _qos_1 = Constructor.qos(port, "1");
          _builder.append(_qos_1, "\t\t");
          _builder.append(", ");
          String _varName_7 = ElementUtils.varName(port);
          _builder.append(_varName_7, "\t\t");
          _builder.append("_options);");
          _builder.newLineIfNotEmpty();
          _builder.append("// directly activate a publisher");
          _builder.newLine();
          String _varName_8 = ElementUtils.varName(port);
          _builder.append(_varName_8);
          _builder.append("_send_->on_activate();");
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  /**
   * Create a service client or server
   */
  public static CharSequence createQuery(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_srv_ = create_service<");
        String _externalName = RosHelpers.externalName(MessageUtils.getServiceType(port));
        _builder.append(_externalName);
        _builder.append(">(\"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic);
        _builder.append("\", ");
        String _serverCallBackMethodForService = Callbacks.serverCallBackMethodForService(port.getClass_(), port);
        _builder.append(_serverCallBackMethodForService);
        _builder.append(", rmw_qos_profile_default, ");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName);
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      } else {
        int _size_1 = port.getRequireds().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          String _varName_1 = ElementUtils.varName(port);
          _builder.append(_varName_1);
          _builder.append("_client_ = create_client<");
          String _externalName_1 = RosHelpers.externalName(MessageUtils.getServiceType(port));
          _builder.append(_externalName_1);
          _builder.append(">(\"");
          String _topic_1 = TopicUtils.getTopic(port);
          _builder.append(_topic_1);
          _builder.append("\", rmw_qos_profile_default, ");
          String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
          _builder.append(_callbackGroupName_1);
          _builder.append(");");
          _builder.newLineIfNotEmpty();
          Callbacks.clientCallBackMethodForService(port.getClass_(), port);
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  /**
   * Create an action client or server
   */
  public static CharSequence createAction(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    {
      int _size = port.getProvideds().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        String _varName = ElementUtils.varName(port);
        _builder.append(_varName);
        _builder.append("_actsrv_ = rclcpp_action::create_server<");
        String _externalName = RosHelpers.externalName(MessageUtils.getServiceType(port));
        _builder.append(_externalName);
        _builder.append(">(this, ");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\"");
        String _topic = TopicUtils.getTopic(port);
        _builder.append(_topic, "\t");
        _builder.append("\",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _serverCallsbacksForAction = Callbacks.serverCallsbacksForAction(port.getClass_(), port);
        _builder.append(_serverCallsbacksForAction, "\t");
        _builder.append(", rcl_action_server_get_default_options(), ");
        String _callbackGroupName = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      } else {
        String _varName_1 = ElementUtils.varName(port);
        _builder.append(_varName_1);
        _builder.append("_actcli_ = rclcpp_action::create_client<");
        String _externalName_1 = RosHelpers.externalName(MessageUtils.getServiceType(port));
        _builder.append(_externalName_1);
        _builder.append(">(this, ");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\"");
        String _topic_1 = TopicUtils.getTopic(port);
        _builder.append(_topic_1, "\t");
        _builder.append("\", ");
        String _callbackGroupName_1 = CallbackGroups.callbackGroupName(port);
        _builder.append(_callbackGroupName_1, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        Callbacks.clientCallsbacksForAction(port.getClass_(), port);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  /**
   * Create an event client and server
   * TODO - not supported
   */
  public static CharSequence createEvent(final Port port) {
    StringConcatenation _builder = new StringConcatenation();
    return _builder;
  }

  /**
   * Return the QoS associated with a port or defaultQoS, if none is specified
   * 
   * The "KEEP_LAST" history setting tells DDS to store a fixed-size buffer of values before they
   * are sent, to aid with recovery in the event of dropped messages.
   * use best effort to avoid blocking during execution.
   */
  public static String qos(final Port port, final String defaultQoS) {
    final ComponentPort compPort = UMLUtil.<ComponentPort>getStereotypeApplication(port, ComponentPort.class);
    if ((((compPort != null) && (compPort.getQos() != null)) && (compPort.getQos().length() > 0))) {
      return compPort.getQos();
    } else {
      return defaultQoS;
    }
  }
}
