/*****************************************************************************
 * Copyright (c) 2018 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 * 
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen

import java.util.ArrayList
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.papyrus.designer.languages.common.base.file.FileSystemAccessFactory
import org.eclipse.papyrus.designer.languages.common.base.file.ICleanUntouchedTmp
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException
import org.eclipse.papyrus.designer.transformation.core.m2minterfaces.IM2MTrafoCDP
import org.eclipse.papyrus.designer.transformation.core.transformations.ExecuteTransformationChain
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext
import org.eclipse.papyrus.designer.transformation.profile.Transformation.M2MTrafo
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinitionModel
import org.eclipse.papyrus.robotics.profile.robotics.components.SystemComponentArchitectureModel
import org.eclipse.papyrus.robotics.profile.robotics.services.ServiceDefinitionModel
import org.eclipse.papyrus.robotics.ros2.codegen.build.CreateCompCMakeLists
import org.eclipse.papyrus.robotics.ros2.codegen.build.CreateCompPackageXML
import org.eclipse.papyrus.robotics.ros2.codegen.component.ComponentTransformations
import org.eclipse.papyrus.robotics.ros2.codegen.message.CreateMsgPackage
import org.eclipse.papyrus.robotics.ros2.codegen.utils.ApplyProfiles
import org.eclipse.papyrus.robotics.ros2.codegen.utils.MessageUtils
import org.eclipse.papyrus.robotics.ros2.codegen.utils.ProjectTools
import org.eclipse.papyrus.uml.tools.utils.PackageUtil
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil
import org.eclipse.uml2.uml.Package

import static extension org.eclipse.papyrus.robotics.core.utils.InstanceUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.launch.LaunchScript.generateLaunch
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.PackageTools.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.SkillUtils.*
import org.eclipse.papyrus.robotics.ros2.codegen.skillrealization.CreateSkillRelizationPackage
import org.eclipse.papyrus.robotics.ros2.base.EnvironmentUtils

class RosTransformations implements IM2MTrafoCDP {

	CreateMsgPackage msgPkgCreator
	CreateSkillRelizationPackage skrPkgCreator

	/**
	 * Create code for a ROS2 package. The passed is used to
	 * identify the ros2 package (BAD), must be determined
	 */
	def pkgTrafos(Package pkg) {
		val project = ProjectTools.getProject(TransformationContext.initialSourceRoot.projectName)
		if (project !== null) {
			val compDefs = project.compDefs
			val system = project.system
			val fileAccess = FileSystemAccessFactory.create(project)
			CreateCompCMakeLists.generate(fileAccess, pkg, compDefs, system)
			CreateCompPackageXML.generate(fileAccess, pkg, compDefs, system)
			ProjectTools.configureIncludes(project, MessageUtils.calcDependencies(compDefs))
			if (system !== null) {
				// generate launch before the component transformations
				// since the latter remove ports
				fileAccess.generateLaunch(system)
			}
			val ct = new ComponentTransformations(fileAccess, project);
			for (compDef : compDefs) {
				ct.componentTrafo(compDef, msgPkgCreator)
			}
			if (!system.uniqueSkills.nullOrEmpty) {
				skrPkgCreator = new CreateSkillRelizationPackage(system, compDefs);
				skrPkgCreator.createSkillRealizationPkg(pkg)
			}
			for (compDef : compDefs) {
				ct.componentCodegen(compDef, msgPkgCreator)
			}
			val cleanup = fileAccess as ICleanUntouchedTmp
			cleanup.cleanUntouched(project.getFolder("src-gen"), new NullProgressMonitor);
			cleanup.cleanUntouched(project.getFolder("src-skel"), new NullProgressMonitor);

			TransformationContext.current.project = project;
		} else {
			throw new TransformationException(ExecuteTransformationChain.USER_CANCEL);
		}
	}

	/**
	 * Apply transformation, support three cases
	 * 1. User called transformation in a component definition model
	 * 	=> apply transformation in that model (only)
	 * 2. User called transformation in a component definition model
	 * 	=> apply transformation in that model (only)
	 * 3. User called transformation in a system architecture
	 *  => obtain all components within the system
	 */
	override applyTrafo(M2MTrafo trafo, Package rootPkg) throws TransformationException {
		// assure that Common and C++ profiles is applied
		EnvironmentUtils.waitForSetupJob
		ApplyProfiles.applyCommonProfile(rootPkg)
		ApplyProfiles.applyCppProfile(rootPkg)
		msgPkgCreator = new CreateMsgPackage();
		if (StereotypeUtil.isApplied(rootPkg, ServiceDefinitionModel)) {
			msgPkgCreator.createMsgPkg(rootPkg)
		}
		else if (StereotypeUtil.isApplied(rootPkg, ComponentDefinitionModel)) {
			val component = rootPkg.componentFromPkg
			if (component !== null) {
				val pkg = PackageUtil.getRootPackage(component);
				pkgTrafos(pkg)
			}
		} else if (StereotypeUtil.isApplied(rootPkg, SystemComponentArchitectureModel)) {
			val system = rootPkg.system
			if (system !== null) {
				val pkgList = new ArrayList<Package>
				val systemPkg = PackageUtil.getRootPackage(system);
				pkgList.add(systemPkg)

				for (component : system.componentList) {
					val compPkg = PackageUtil.getRootPackage(system)
					if (!pkgList.contains(compPkg)) {
						pkgList.add(compPkg)
					}
				}

				for (pkg : pkgList) {
					pkgTrafos(pkg)
				}
			}
		} else {
			Activator.log.debug(
				String.format("model %s is neither a component definition or a component assembly", rootPkg.name));
		}
 		// CCorePlugin.getIndexManager().setDefaultIndexerId(currentIndexer);
	}
}
