/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.reverse;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Logger;
import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.dom.ast.IASTStandardFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTTranslationUnit;
import org.eclipse.cdt.core.model.IBinary;
import org.eclipse.cdt.core.model.ICContainer;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.core.model.IDeclaration;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IEnumerator;
import org.eclipse.cdt.core.model.IField;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ISourceRoot;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.common.base.StdUriConstants;
import org.eclipse.papyrus.designer.languages.cpp.profile.CppProfileResource;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.CElementChange;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.ChangeMapStore;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.CppChangeObject;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.ModelChangeObject;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.FileWatcher;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.ModelManagement;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.RoundtripCppUtils;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * The main reverser class that will reverse C++ to UML
 */
@SuppressWarnings("all")
public class ReverseCpp2Uml extends ReverseData {
  public enum ReverseMode {
    BATCH,

    INCREMENTAL;
  }

  public enum ConflictResolutionMode {
    FROM_MODEL,

    FROM_CODE,

    UI_INTERACTION;
  }

  private ReverseCpp2Uml.ConflictResolutionMode conflictResolveMode = ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE;

  public static String C_LangID = "C";

  public static String Cpp_LangID = "C++";

  private String langID = "C++";

  private static final String TEMPLATE_PARAMETER_SIGNATURE_NAME = "template_paremeter_signature";

  public static final String REVERSE_FOLDER = "reversed_models";

  public static final String MODEL_POSTFIX = ".uml";

  private static final Logger LOGGER = Logger.getLogger(ReverseCpp2Uml.class.getName());

  private String path;

  private static long timestamp = 0;

  private ReverseCpp2Uml.ReverseMode reverseMode = ReverseCpp2Uml.ReverseMode.BATCH;

  private int pass;

  private List<CppChangeObject> changeList = new ArrayList<CppChangeObject>();

  private static List<ModelChangeObject> modelChangeList = new ArrayList<ModelChangeObject>();

  private static final String projectPrefix = "org.eclipse.papyrus.cppgen";

  public static boolean addModelChange(final ModelChangeObject change) {
    return ReverseCpp2Uml.modelChangeList.add(change);
  }

  public static void clearModelChange() {
    ReverseCpp2Uml.modelChangeList.clear();
  }

  public ReverseCpp2Uml(final ITranslationUnit unit, final IProgressMonitor monitor, final String langID) {
    this(unit.getCProject(), monitor, langID);
    this.unit = unit;
  }

  public ReverseCpp2Uml(final ITranslationUnit unit, final IProgressMonitor monitor) {
    this(unit.getCProject(), monitor, ReverseCpp2Uml.Cpp_LangID);
  }

  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor, final String langID) {
    this(project, monitor, langID, null);
  }

  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor) {
    this(project, monitor, ReverseCpp2Uml.Cpp_LangID, null);
  }

  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor, final String langID, final String path) {
    try {
      this.project = project;
      this.monitor = monitor;
      this.langID = langID;
      this.path = path;
      ReverseData.current = this;
      this.index = CCorePlugin.getIndexManager().getIndex(project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.tuToASTtuMap = _hashMap;
      HashMap<String, Boolean> _hashMap_1 = new HashMap<String, Boolean>();
      this.includesMap = _hashMap_1;
      UniqueEList<ICContainer> _uniqueEList = new UniqueEList<ICContainer>();
      this.containers = _uniqueEList;
      HashMap<ICElement, EObject> _hashMap_2 = new HashMap<ICElement, EObject>();
      this.map = _hashMap_2;
      HashMap<ICElement, Boolean> _hashMap_3 = new HashMap<ICElement, Boolean>();
      this.analyzeMap = _hashMap_3;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void reverse() {
    try {
      String projectName = this.unit.getCProject().getElementName();
      BatchReverseFunctionBody iSync = new BatchReverseFunctionBody(this.unit, projectName);
      iSync.run();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Check whether the passed container is a source folder, in the sense that it
   * contains translation units (directly or indirectly)
   */
  private boolean isSourceFolder(final ICContainer container) {
    try {
      ICElement[] _children = container.getChildren();
      for (final ICElement child : _children) {
        if ((child instanceof ITranslationUnit)) {
          return true;
        } else {
          if ((child instanceof ICContainer)) {
            boolean _isSourceFolder = this.isSourceFolder(((ICContainer) child));
            if (_isSourceFolder) {
              return true;
            }
          }
        }
      }
      return false;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void reverseProject(final boolean reset) {
    this.reverseProject(reset, null);
  }

  public void reverseProject(final boolean reset, final IResource fileOrFolder) {
    try {
      ICContainer[] sourceRoots = null;
      if ((fileOrFolder != null)) {
        ICElement _findElement = this.project.findElement(fileOrFolder.getFullPath());
        sourceRoots = ((ICContainer[])Conversions.unwrapArray(Collections.<ICContainer>singleton(((ICContainer) _findElement)), ICContainer.class));
      } else {
        final Function1<ISourceRoot, Boolean> _function = new Function1<ISourceRoot, Boolean>() {
          @Override
          public Boolean apply(final ISourceRoot it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
          }
        };
        sourceRoots = ((ICContainer[])Conversions.unwrapArray(IterableExtensions.<ISourceRoot>filter(((Iterable<ISourceRoot>)Conversions.doWrapArray(this.project.getSourceRoots())), _function), ICContainer.class));
      }
      final ICContainer[] _converted_sourceRoots = (ICContainer[])sourceRoots;
      int _size = ((List<ICContainer>)Conversions.doWrapArray(_converted_sourceRoots)).size();
      boolean _lessThan = (_size < 1);
      if (_lessThan) {
        throw new Exception("No source folder");
      }
      this.containers.clear();
      final ICContainer[] _converted_sourceRoots_1 = (ICContainer[])sourceRoots;
      final Function1<ICContainer, Boolean> _function_1 = new Function1<ICContainer, Boolean>() {
        @Override
        public Boolean apply(final ICContainer it) {
          return Boolean.valueOf(it.getElementName().contains(ReverseCpp2Uml.projectPrefix));
        }
      };
      ICContainer sourceRootGenerated = IterableExtensions.<ICContainer>head(IterableExtensions.<ICContainer>filter(((Iterable<ICContainer>)Conversions.doWrapArray(_converted_sourceRoots_1)), _function_1));
      if ((sourceRootGenerated != null)) {
        final Function1<ICContainer, Boolean> _function_2 = new Function1<ICContainer, Boolean>() {
          @Override
          public Boolean apply(final ICContainer it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
          }
        };
        Iterables.<ICContainer>addAll(this.containers, IterableExtensions.<ICContainer>filter(Iterables.<ICContainer>filter(((Iterable<?>)Conversions.doWrapArray(sourceRootGenerated.getChildren())), ICContainer.class), _function_2));
      } else {
        CollectionExtensions.<ICContainer>addAll(this.containers, sourceRoots);
      }
      ModelManagement _modelManagement = new ModelManagement();
      this.modelManager = _modelManagement;
      String umlFilePath = null;
      if (((this.path == null) || this.path.equals(""))) {
        IProject _project = this.project.getProject();
        String _elementName = this.project.getElementName();
        String _plus = (_elementName + ReverseCpp2Uml.MODEL_POSTFIX);
        umlFilePath = this.modelManager.getPath(_project, 
          ReverseCpp2Uml.REVERSE_FOLDER, _plus);
      } else {
        umlFilePath = this.path;
      }
      this.modelManager.createOrgetModel(this.project.getElementName(), umlFilePath, (!reset), reset);
      this.models = this.modelManager.getModels();
      for (final Model model : this.models) {
        {
          RoundtripCppUtils.applyProfile(model, CppProfileResource.PROFILE_PATH);
          RoundtripCppUtils.applyProfile(model, StdUriConstants.UML_STD_PROFILE_PATH);
          ReverseUtils.setXmlID(model);
        }
      }
      int count = 0;
      for (final ICContainer container : this.containers) {
        int _count = count;
        int _countTUs = this.countTUs(container);
        count = (_count + _countTUs);
      }
      this.monitor.beginTask("Reverse source files", (count * 2));
      for (this.pass = 1; (this.pass <= 2); this.pass++) {
        {
          this.includesMap.clear();
          for (final ICContainer container_1 : this.containers) {
            this.reverseProject(container_1);
          }
        }
      }
      for (final Model subModel : this.models) {
        subModel.eResource().save(ModelManagement.getDefaultSaveOptions());
      }
      long _timeStamp = IterableExtensions.<Model>last(this.models).eResource().getTimeStamp();
      long _divide = (_timeStamp / 1000);
      ReverseCpp2Uml.timestamp = _divide;
      this.clearRawChangeList();
      this.modelManager.dispose();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public void syncIncrementalProject() {
    try {
      this.reverseMode = ReverseCpp2Uml.ReverseMode.INCREMENTAL;
      final Function1<ISourceRoot, Boolean> _function = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
        }
      };
      Iterable<ISourceRoot> sourceRoots = IterableExtensions.<ISourceRoot>filter(((Iterable<ISourceRoot>)Conversions.doWrapArray(this.project.getSourceRoots())), _function);
      int _size = IterableExtensions.size(sourceRoots);
      boolean _lessThan = (_size < 1);
      if (_lessThan) {
        throw new Exception("No source folder");
      }
      this.containers.clear();
      final Function1<ISourceRoot, Boolean> _function_1 = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(it.getElementName().contains(ReverseCpp2Uml.projectPrefix));
        }
      };
      ISourceRoot sourceRootGenerated = IterableExtensions.<ISourceRoot>head(IterableExtensions.<ISourceRoot>filter(sourceRoots, _function_1));
      if ((sourceRootGenerated != null)) {
        final Function1<ICContainer, Boolean> _function_2 = new Function1<ICContainer, Boolean>() {
          @Override
          public Boolean apply(final ICContainer it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
          }
        };
        Iterables.<ICContainer>addAll(this.containers, IterableExtensions.<ICContainer>filter(Iterables.<ICContainer>filter(((Iterable<?>)Conversions.doWrapArray(sourceRootGenerated.getChildren())), ICContainer.class), _function_2));
      } else {
        Iterables.<ICContainer>addAll(this.containers, sourceRoots);
      }
      ModelManagement _modelManagement = new ModelManagement();
      this.modelManager = _modelManagement;
      String umlFilePath = null;
      if (((this.path == null) || this.path.equals(""))) {
        IProject _project = this.project.getProject();
        String _elementName = this.project.getElementName();
        String _plus = (_elementName + ReverseCpp2Uml.MODEL_POSTFIX);
        umlFilePath = this.modelManager.getPath(_project, 
          ReverseCpp2Uml.REVERSE_FOLDER, _plus);
      } else {
        umlFilePath = this.path;
      }
      this.modelManager.createOrgetModel(this.project.getElementName(), umlFilePath, false, false);
      this.models = this.modelManager.getModels();
      final Consumer<Model> _function_3 = new Consumer<Model>() {
        @Override
        public void accept(final Model it) {
          RoundtripCppUtils.applyProfile(it, CppProfileResource.PROFILE_PATH);
          RoundtripCppUtils.applyProfile(it, StdUriConstants.UML_STD_PROFILE_PATH);
        }
      };
      this.models.forEach(_function_3);
      if (((this.getRawChangeList() == null) || (this.getRawChangeList().size() == 0))) {
        FileWatcher fileWatcher = new FileWatcher(this.project, ReverseCpp2Uml.timestamp);
        List<ITranslationUnit> modifiedItus = fileWatcher.getModifiledTranslationUnits(this.project);
        for (final ITranslationUnit modified : modifiedItus) {
          this.syncTranslationUnit(modified);
        }
        this.modelManager.saveModel(IterableExtensions.<String>toList(Collections.<String>singleton(umlFilePath)));
        long _timeStamp = IterableExtensions.<Model>last(this.models).eResource().getTimeStamp();
        long _divide = (_timeStamp / 1000);
        ReverseCpp2Uml.timestamp = _divide;
      } else {
        this.reverseIncrementalChanges();
        this.modelManager.saveModel(IterableExtensions.<String>toList(Collections.<String>singleton(umlFilePath)));
      }
      this.modelManager.dispose();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Count the number of translation units within a parent, in order
   * to configure the progess monitor
   */
  private int countTUs(final IParent parent) {
    try {
      int count = 0;
      if ((!(parent instanceof ITranslationUnit))) {
        ICElement[] _children = parent.getChildren();
        for (final ICElement child : _children) {
          if ((child instanceof ITranslationUnit)) {
            count++;
          } else {
            if ((child instanceof IBinary)) {
            } else {
              if ((child instanceof IParent)) {
                int _count = count;
                int _countTUs = this.countTUs(((IParent)child));
                count = (_count + _countTUs);
              }
            }
          }
        }
      }
      return count;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * reverse a project
   */
  private void reverseProject(final IParent parent) {
    try {
      ICElement[] _children = parent.getChildren();
      for (final ICElement child : _children) {
        {
          boolean _isCanceled = this.monitor.isCanceled();
          if (_isCanceled) {
            return;
          }
          if ((child instanceof ITranslationUnit)) {
            this.reverseHeader(((ITranslationUnit) child));
            if ((false && (!((ITranslationUnit)child).isHeaderUnit()))) {
              String _elementName = ((ITranslationUnit)child).getElementName();
              String _plus = ("Parsing method implementations in " + _elementName);
              this.monitor.subTask(_plus);
              this.reverseSource(((ITranslationUnit)child));
            }
          } else {
            if ((child instanceof IBinary)) {
            } else {
              if ((child instanceof IParent)) {
                this.reverseProject(((IParent)child));
              }
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Reverse the source code of a translation unit (that is not a header unit)
   */
  private void reverseSource(final ITranslationUnit unit) {
    String _elementName = this.project.getElementName();
    Model _correspondingModel = ReverseUtils.getCorrespondingModel(unit);
    BatchReverseFunctionBody sync = new BatchReverseFunctionBody(unit, _elementName, _correspondingModel);
    try {
      sync.run();
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  public void reverseHeader(final ITranslationUnit headerUnit) {
    try {
      boolean _containsKey = this.includesMap.containsKey(headerUnit.getPath().toString());
      boolean _not = (!_containsKey);
      if (_not) {
        this.includesMap.put(headerUnit.getPath().toString(), Boolean.valueOf(true));
        this.monitor.subTask(String.format("Pass %d: parsing types in %s", Integer.valueOf(this.pass), headerUnit.getElementName()));
        this.monitor.worked(1);
        if ((this.pass == 1)) {
          GetOrCreateP1.getOrCreateClassifiers(headerUnit);
        } else {
          if ((this.pass == 2)) {
            GetOrCreateP2.getOrCreateClassifiers(headerUnit);
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * provide access to instance
   */
  public static ReverseCpp2Uml currentCpp2Uml() {
    return ((ReverseCpp2Uml) ReverseData.current);
  }

  private void syncTranslationUnit(final ITranslationUnit itu) {
    try {
      Iterable<IStructure> structures = Iterables.<IStructure>filter(ReverseUtils.getAllIStructures(itu, false, true, this.project), IStructure.class);
      for (final IStructure structure : structures) {
        {
          Type classifier = GetOrCreateP2.getClassifier(structure);
          if ((classifier instanceof Classifier)) {
            this.syncIStructureToModel(((Classifier)classifier), structure);
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private Object syncIStructureToModel(final Classifier classifier, final IStructure istructure) {
    try {
      Object _xblockexpression = null;
      {
        List<IField> fields = IterableExtensions.<IField>toList(Iterables.<IField>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IField.class));
        List<Property> attributes = IterableExtensions.<Property>toList(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class));
        this.mergeAttributes(classifier, attributes, fields);
        List<IMethodDeclaration> methods = IterableExtensions.<IMethodDeclaration>toList(Iterables.<IMethodDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IMethodDeclaration.class));
        List<Operation> operations = IterableExtensions.<Operation>toList(Iterables.<Operation>filter(classifier.getOwnedElements(), Operation.class));
        this.mergeOperations(classifier, operations, methods);
        List<IDeclaration> declarations = IterableExtensions.<IDeclaration>toList(Iterables.<IDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IDeclaration.class));
        List<Type> nestedTypes = IterableExtensions.<Type>toList(Iterables.<Type>filter(classifier.getOwnedElements(), Type.class));
        _xblockexpression = this.mergeNestedTypes(classifier, nestedTypes, declarations);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private void mergeAttributes(final Classifier parent, final List<Property> attributes, final List<IField> fields) {
    try {
      UniqueEList<IField> foundFieldList = new UniqueEList<IField>();
      UniqueEList<IField> notFoundFieldList = new UniqueEList<IField>();
      UniqueEList<Property> foundAttrList = new UniqueEList<Property>();
      UniqueEList<Property> notFoundAttrList = new UniqueEList<Property>();
      for (final IField field : fields) {
        {
          final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
            @Override
            public Boolean apply(final Property it) {
              return Boolean.valueOf(it.getName().equals(field.getElementName()));
            }
          };
          final Property attrFound = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(attributes, _function));
          if ((attrFound == null)) {
            notFoundFieldList.add(field);
          } else {
            foundAttrList.add(attrFound);
            foundFieldList.add(field);
            final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, attrFound));
              }
            };
            ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_1));
            boolean isModelObjectChanged = (_head != null);
            if (isModelObjectChanged) {
              this.syncAttributeWithMode(parent, attrFound, field, this.conflictResolveMode);
            } else {
              this.syncAttributeWithMode(parent, attrFound, field, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
            }
          }
        }
      }
      for (final Property attr : attributes) {
        boolean _contains = foundAttrList.contains(attr);
        boolean _not = (!_contains);
        if (_not) {
          notFoundAttrList.add(attr);
        }
      }
      final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf((it.eventType == Notification.ADD));
        }
      };
      List<ModelChangeObject> addChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
      UniqueEList<Property> tobeRemovedsInNotFound = new UniqueEList<Property>();
      for (final Property attr_1 : notFoundAttrList) {
        {
          final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, attr_1));
            }
          };
          ModelChangeObject modelChangeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(addChanges, _function_1));
          if ((modelChangeObj != null)) {
            tobeRemovedsInNotFound.add(attr_1);
          } else {
          }
        }
      }
      for (final Property i : tobeRemovedsInNotFound) {
        notFoundAttrList.remove(i);
      }
      UniqueEList<IField> processedFields = new UniqueEList<IField>();
      UniqueEList<IField> remainingFields = new UniqueEList<IField>();
      UniqueEList<Property> remainingAttributes = new UniqueEList<Property>();
      for (final IField notFoundField : notFoundFieldList) {
        boolean _contains_1 = processedFields.contains(notFoundField);
        boolean _not_1 = (!_contains_1);
        if (_not_1) {
          final Type umlType = ReverseUtils.getUMLType(ASTUtils.getDeclSpecifier(notFoundField), notFoundField);
          final String typeName = ReverseUtils.getCppTypeName(notFoundField.getTypeName());
          final Function1<IField, Boolean> _function_1 = new Function1<IField, Boolean>() {
            @Override
            public Boolean apply(final IField it) {
              try {
                return Boolean.valueOf(ReverseUtils.getCppTypeName(it.getTypeName()).equals(typeName));
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            }
          };
          List<IField> sameTypeNameFields = IterableExtensions.<IField>toList(IterableExtensions.<IField>filter(notFoundFieldList, _function_1));
          final Function1<Property, Boolean> _function_2 = new Function1<Property, Boolean>() {
            @Override
            public Boolean apply(final Property it) {
              return Boolean.valueOf(it.getType().getName().equals(umlType.getName()));
            }
          };
          List<Property> sameTypeAttrs = IterableExtensions.<Property>toList(IterableExtensions.<Property>filter(notFoundAttrList, _function_2));
          int i_1 = 0;
          for (i_1 = 0; (i_1 < sameTypeNameFields.size()); i_1++) {
            int _size = sameTypeAttrs.size();
            boolean _greaterEqualsThan = (i_1 >= _size);
            if (_greaterEqualsThan) {
              remainingFields.add(sameTypeNameFields.get(i_1));
            } else {
              final IField tobeProcessedField = sameTypeNameFields.get(i_1);
              final Property tobeProcessedAttr = sameTypeAttrs.get(i_1);
              final Function1<ModelChangeObject, Boolean> _function_3 = new Function1<ModelChangeObject, Boolean>() {
                @Override
                public Boolean apply(final ModelChangeObject it) {
                  return Boolean.valueOf(Objects.equal(it.eObject, tobeProcessedAttr));
                }
              };
              ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_3));
              boolean isModelObjectChanged = (_head != 
                null);
              if (isModelObjectChanged) {
                this.syncAttributeWithMode(parent, tobeProcessedAttr, tobeProcessedField, 
                  ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
              } else {
                this.syncAttributeWithMode(parent, tobeProcessedAttr, tobeProcessedField, 
                  ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
              }
              processedFields.add(tobeProcessedField);
              notFoundAttrList.remove(tobeProcessedAttr);
            }
          }
          final List<Property> _converted_sameTypeAttrs = (List<Property>)sameTypeAttrs;
          int _length = ((Object[])Conversions.unwrapArray(_converted_sameTypeAttrs, Object.class)).length;
          boolean _lessThan = (i_1 < _length);
          if (_lessThan) {
            for (int j = i_1; (j < ((Object[])Conversions.unwrapArray(sameTypeAttrs, Object.class)).length); j++) {
              remainingAttributes.add(sameTypeAttrs.get(j));
            }
          }
        }
      }
      for (final Property notFoundAttr : notFoundAttrList) {
        boolean _contains_2 = remainingAttributes.contains(notFoundAttr);
        boolean _not_2 = (!_contains_2);
        if (_not_2) {
          remainingAttributes.add(notFoundAttr);
        }
      }
      for (final IField remaining : remainingFields) {
        PropertyUtils.createProperty(remaining, parent);
      }
      final Function1<ModelChangeObject, Boolean> _function_3 = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf(((it.eventType == Notification.ADD) || (it.eventType == Notification.SET)));
        }
      };
      List<ModelChangeObject> remainingChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_3));
      for (final Property remaining_1 : remainingAttributes) {
        final Function1<ModelChangeObject, Boolean> _function_4 = new Function1<ModelChangeObject, Boolean>() {
          @Override
          public Boolean apply(final ModelChangeObject it) {
            return Boolean.valueOf(Objects.equal(it.eObject, remaining_1));
          }
        };
        boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<ModelChangeObject>filter(remainingChanges, _function_4));
        if (_isEmpty) {
          if ((parent instanceof org.eclipse.uml2.uml.Class)) {
            EList<Property> _ownedAttributes = ((org.eclipse.uml2.uml.Class) parent).getOwnedAttributes();
            _ownedAttributes.remove(remaining_1);
          } else {
            if ((parent instanceof DataType)) {
              EList<Property> _ownedAttributes_1 = ((DataType) parent).getOwnedAttributes();
              _ownedAttributes_1.remove(remaining_1);
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private Boolean syncAttributeWithMode(final Classifier parent, final Property attribute, final IField field, final ReverseCpp2Uml.ConflictResolutionMode resolveMode) {
    Boolean _switchResult = null;
    if (resolveMode != null) {
      switch (resolveMode) {
        case FROM_MODEL:
          _switchResult = null;
          break;
        case FROM_CODE:
          boolean _xblockexpression = false;
          {
            Type type = ReverseUtils.getUMLType(ASTUtils.getDeclSpecifier(field), field);
            attribute.setType(type);
            attribute.setName(field.getElementName());
            PropertyUtils.updateProperty(field, attribute);
            final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, attribute));
              }
            };
            ModelChangeObject changeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
            boolean _xifexpression = false;
            if ((changeObj != null)) {
              _xifexpression = ReverseCpp2Uml.modelChangeList.remove(changeObj);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression);
          break;
        case UI_INTERACTION:
          _switchResult = null;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }

  private Boolean syncOperationWithMode(final Classifier parent, final Operation op, final IMethodDeclaration method, final ReverseCpp2Uml.ConflictResolutionMode resolveMode) {
    Boolean _switchResult = null;
    if (resolveMode != null) {
      switch (resolveMode) {
        case FROM_MODEL:
          _switchResult = null;
          break;
        case FROM_CODE:
          boolean _xblockexpression = false;
          {
            op.setName(method.getElementName());
            MethodUtils.updateMethod(((org.eclipse.uml2.uml.Class) parent), op, method);
            final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, op));
              }
            };
            ModelChangeObject changeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
            boolean _xifexpression = false;
            if ((changeObj != null)) {
              _xifexpression = ReverseCpp2Uml.modelChangeList.remove(changeObj);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression);
          break;
        case UI_INTERACTION:
          _switchResult = null;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }

  private boolean isSameOperation(final Operation op, final IMethodDeclaration method) {
    boolean ret = true;
    boolean _equals = op.getName().equals(method.getElementName());
    boolean _not = (!_equals);
    if (_not) {
      ret = false;
    } else {
      ret = this.isSameSignature(op, method);
    }
    return ret;
  }

  private boolean isSameSignature(final Operation op, final IMethodDeclaration method) {
    boolean ret = true;
    final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
      @Override
      public Boolean apply(final Parameter it) {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      }
    };
    List<Parameter> params = IterableExtensions.<Parameter>toList(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function));
    int _numberOfParameters = method.getNumberOfParameters();
    int _size = params.size();
    boolean _notEquals = (_numberOfParameters != _size);
    if (_notEquals) {
      ret = false;
    } else {
      for (int i = 0; (i < method.getNumberOfParameters()); i++) {
        boolean _equals = ReverseUtils.getCppTypeName(method.getParameterTypes()[i]).equals(params.get(i).getType().getName());
        boolean _not = (!_equals);
        if (_not) {
          ret = false;
        }
      }
      if ((ret != false)) {
        final Function1<Parameter, Boolean> _function_1 = new Function1<Parameter, Boolean>() {
          @Override
          public Boolean apply(final Parameter it) {
            ParameterDirectionKind _direction = it.getDirection();
            return Boolean.valueOf(Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
          }
        };
        Parameter returnParam = IterableExtensions.<Parameter>head(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function_1));
        if ((returnParam == null)) {
          boolean _equals = ReverseUtils.getCppTypeName(method.getReturnType()).equals("void");
          boolean _not = (!_equals);
          if (_not) {
            ret = false;
          }
        } else {
          if (((returnParam.getType() != null) && (!ReverseUtils.getCppTypeName(method.getReturnType()).equals(returnParam.getType().getName())))) {
            ret = false;
          }
        }
      }
    }
    return ret;
  }

  private boolean isSameMethodDeclaration(final IMethodDeclaration method1, final IMethodDeclaration method2) {
    try {
      boolean result = true;
      result = (result && method1.getSignature().equals(method2.getSignature()));
      result = (result && Objects.equal(ASTUtils.convertVisibility(method1.getVisibility()), ASTUtils.convertVisibility(method2.getVisibility())));
      result = (result && (method1.isStatic() == method2.isStatic()));
      result = (result && (method1.isVirtual() == method2.isVirtual()));
      result = (result && (method1.isVirtual() == method2.isVirtual()));
      result = (result && (method1.isInline() == method2.isInline()));
      result = (result && (method1.isFriend() == method2.isFriend()));
      result = (result && (method1.isVolatile() == method2.isVolatile()));
      result = (result && (method1.isConstructor() == method2.isConstructor()));
      result = (result && (method1.isDestructor() == method2.isDestructor()));
      result = (result && ((method1.getReturnType() != null) && (method2.getReturnType() != null)));
      if (((method1.getReturnType() != null) && (method2.getReturnType() != null))) {
        result = (result && method1.getReturnType().equals(method2.getReturnType()));
      }
      result = (result && (method1.getNumberOfParameters() == method2.getNumberOfParameters()));
      if ((result == false)) {
        return result;
      }
      try {
        IASTStandardFunctionDeclarator declarator1 = ASTUtils.getDeclarator(method1);
        IASTStandardFunctionDeclarator declarator2 = ASTUtils.getDeclarator(method2);
        result = (result && declarator1.getRawSignature().equals(declarator2.getRawSignature()));
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          e.printStackTrace();
          return result;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private int getNumberOfSameParameters(final Operation op, final IMethodDeclaration method) {
    int ret = 0;
    final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
      @Override
      public Boolean apply(final Parameter it) {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      }
    };
    List<Parameter> params = IterableExtensions.<Parameter>toList(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function));
    for (int i = 0; (i < method.getNumberOfParameters()); i++) {
      int _size = params.size();
      boolean _lessEqualsThan = (i <= _size);
      if (_lessEqualsThan) {
        if (((params.get(i).getType() != null) && 
          ReverseUtils.getCppTypeName(method.getParameterTypes()[i]).equals(params.get(i).getType().getName()))) {
          ret++;
        }
      }
    }
    return ret;
  }

  private Operation findMostSameOperation(final List<Operation> ops, final IMethodDeclaration method) {
    boolean _isEmpty = ops.isEmpty();
    if (_isEmpty) {
      return null;
    }
    Operation ret = IterableExtensions.<Operation>head(ops);
    for (final Operation op : ops) {
      boolean _notEquals = (!Objects.equal(ret, op));
      if (_notEquals) {
        int _numberOfSameParameters = this.getNumberOfSameParameters(op, method);
        int _numberOfSameParameters_1 = this.getNumberOfSameParameters(ret, method);
        boolean _greaterThan = (_numberOfSameParameters > _numberOfSameParameters_1);
        if (_greaterThan) {
          ret = op;
        }
      }
    }
    return ret;
  }

  private void mergeOperations(final Classifier parent, final List<Operation> operations, final List<IMethodDeclaration> methods) {
    UniqueEList<IMethodDeclaration> foundMethodList = new UniqueEList<IMethodDeclaration>();
    UniqueEList<IMethodDeclaration> notFoundMethodList = new UniqueEList<IMethodDeclaration>();
    UniqueEList<Operation> foundOperationList = new UniqueEList<Operation>();
    UniqueEList<Operation> notFoundOperationList = new UniqueEList<Operation>();
    for (final IMethodDeclaration method : methods) {
      {
        final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
          @Override
          public Boolean apply(final Operation it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSameOperation(it, method));
          }
        };
        final Operation opFound = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(operations, _function));
        if ((opFound == null)) {
          notFoundMethodList.add(method);
        } else {
          foundOperationList.add(opFound);
          foundMethodList.add(method);
          final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, opFound));
            }
          };
          ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_1));
          boolean isModelObjectChanged = (_head != null);
          if (isModelObjectChanged) {
          } else {
          }
        }
      }
    }
    for (final Operation op : operations) {
      boolean _contains = foundOperationList.contains(op);
      boolean _not = (!_contains);
      if (_not) {
        notFoundOperationList.add(op);
      }
    }
    final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
      @Override
      public Boolean apply(final ModelChangeObject it) {
        return Boolean.valueOf((it.eventType == Notification.ADD));
      }
    };
    List<ModelChangeObject> addChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
    UniqueEList<Operation> tobeRemovedsInNotFound = new UniqueEList<Operation>();
    for (final Operation op_1 : notFoundOperationList) {
      {
        final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
          @Override
          public Boolean apply(final ModelChangeObject it) {
            return Boolean.valueOf(Objects.equal(it.eObject, op_1));
          }
        };
        ModelChangeObject modelChangeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(addChanges, _function_1));
        if ((modelChangeObj != null)) {
          tobeRemovedsInNotFound.add(op_1);
        } else {
        }
      }
    }
    for (final Operation i : tobeRemovedsInNotFound) {
      notFoundOperationList.remove(i);
    }
    UniqueEList<IMethodDeclaration> processedMethods = new UniqueEList<IMethodDeclaration>();
    UniqueEList<IMethodDeclaration> remainingMethods = new UniqueEList<IMethodDeclaration>();
    UniqueEList<Operation> remainingOperations = new UniqueEList<Operation>();
    for (final IMethodDeclaration notFoundMethod : notFoundMethodList) {
      boolean _contains_1 = processedMethods.contains(notFoundMethod);
      boolean _not_1 = (!_contains_1);
      if (_not_1) {
        final String functionName = IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(notFoundMethod.getElementName().split("::"))));
        final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
          @Override
          public Boolean apply(final Operation it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, notFoundMethod));
          }
        };
        final Operation sameSignatureOperation = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(notFoundOperationList, _function_1));
        if ((sameSignatureOperation != null)) {
          final Function1<ModelChangeObject, Boolean> _function_2 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, sameSignatureOperation));
            }
          };
          ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_2));
          boolean isModelObjectChanged = (_head != 
            null);
          if (isModelObjectChanged) {
            this.syncOperationWithMode(parent, sameSignatureOperation, notFoundMethod, 
              ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
          } else {
            this.syncOperationWithMode(parent, sameSignatureOperation, notFoundMethod, 
              ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
          }
          notFoundOperationList.remove(sameSignatureOperation);
          processedMethods.add(notFoundMethod);
        } else {
          final Function1<IMethodDeclaration, Boolean> _function_3 = new Function1<IMethodDeclaration, Boolean>() {
            @Override
            public Boolean apply(final IMethodDeclaration it) {
              return Boolean.valueOf(it.getElementName().equals(functionName));
            }
          };
          final List<IMethodDeclaration> sameNameMethods = IterableExtensions.<IMethodDeclaration>toList(IterableExtensions.<IMethodDeclaration>filter(notFoundMethodList, _function_3));
          final Function1<Operation, Boolean> _function_4 = new Function1<Operation, Boolean>() {
            @Override
            public Boolean apply(final Operation it) {
              return Boolean.valueOf(it.getName().equals(functionName));
            }
          };
          final List<Operation> sameNameOperations = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(notFoundOperationList, _function_4));
          int _size = sameNameOperations.size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            int i_1 = 0;
            for (i_1 = 0; (i_1 < sameNameMethods.size()); i_1++) {
              int _size_1 = sameNameOperations.size();
              boolean _greaterEqualsThan = (i_1 >= _size_1);
              if (_greaterEqualsThan) {
                remainingMethods.add(sameNameMethods.get(i_1));
              } else {
                final IMethodDeclaration tobeProcessedMethod = sameNameMethods.get(i_1);
                final Operation tobeProcessedOp = sameNameOperations.get(i_1);
                final Function1<ModelChangeObject, Boolean> _function_5 = new Function1<ModelChangeObject, Boolean>() {
                  @Override
                  public Boolean apply(final ModelChangeObject it) {
                    return Boolean.valueOf(Objects.equal(it.eObject, tobeProcessedOp));
                  }
                };
                ModelChangeObject _head_1 = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_5));
                boolean isModelObjectChanged_1 = (_head_1 != null);
                if (isModelObjectChanged_1) {
                  this.syncOperationWithMode(parent, tobeProcessedOp, tobeProcessedMethod, 
                    ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
                } else {
                  this.syncOperationWithMode(parent, tobeProcessedOp, tobeProcessedMethod, 
                    ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
                }
                processedMethods.add(tobeProcessedMethod);
                notFoundOperationList.remove(tobeProcessedOp);
              }
            }
            int _length = ((Object[])Conversions.unwrapArray(sameNameOperations, Object.class)).length;
            boolean _lessThan = (i_1 < _length);
            if (_lessThan) {
              for (int j = i_1; (j < ((Object[])Conversions.unwrapArray(sameNameOperations, Object.class)).length); j++) {
                remainingOperations.add(sameNameOperations.get(j));
              }
            }
          } else {
            final Operation mostSameOp = this.findMostSameOperation(notFoundOperationList, notFoundMethod);
            if ((mostSameOp != null)) {
              final Function1<ModelChangeObject, Boolean> _function_5 = new Function1<ModelChangeObject, Boolean>() {
                @Override
                public Boolean apply(final ModelChangeObject it) {
                  return Boolean.valueOf(Objects.equal(it.eObject, mostSameOp));
                }
              };
              ModelChangeObject _head_1 = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_5));
              boolean isModelObjectChanged_1 = (_head_1 != null);
              if (isModelObjectChanged_1) {
                this.syncOperationWithMode(parent, mostSameOp, notFoundMethod, 
                  ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
              } else {
                this.syncOperationWithMode(parent, mostSameOp, notFoundMethod, 
                  ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
              }
              processedMethods.add(notFoundMethod);
              notFoundOperationList.remove(mostSameOp);
            }
          }
        }
      }
    }
    for (final Operation notFoundOp : notFoundOperationList) {
      boolean _contains_2 = remainingOperations.contains(notFoundOp);
      boolean _not_2 = (!_contains_2);
      if (_not_2) {
        remainingOperations.add(notFoundOp);
      }
    }
    for (final IMethodDeclaration remaining : remainingMethods) {
      MethodUtils.createMethod(remaining, ((org.eclipse.uml2.uml.Class) parent));
    }
    final Function1<ModelChangeObject, Boolean> _function_6 = new Function1<ModelChangeObject, Boolean>() {
      @Override
      public Boolean apply(final ModelChangeObject it) {
        return Boolean.valueOf(((it.eventType == Notification.ADD) || (it.eventType == Notification.SET)));
      }
    };
    List<ModelChangeObject> remainingChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_6));
    for (final Operation remaining_1 : remainingOperations) {
      final Function1<ModelChangeObject, Boolean> _function_7 = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf(Objects.equal(it.eObject, remaining_1));
        }
      };
      boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<ModelChangeObject>filter(remainingChanges, _function_7));
      if (_isEmpty) {
        if ((parent instanceof org.eclipse.uml2.uml.Class)) {
          EList<Operation> _ownedOperations = ((org.eclipse.uml2.uml.Class) parent).getOwnedOperations();
          _ownedOperations.remove(remaining_1);
        } else {
          if ((parent instanceof DataType)) {
            EList<Operation> _ownedOperations_1 = ((DataType) parent).getOwnedOperations();
            _ownedOperations_1.remove(remaining_1);
          }
        }
      }
    }
  }

  private Object mergeNestedTypes(final Classifier parent, final List<Type> nestedTypes, final List<IDeclaration> declarations) {
    return null;
  }

  private void reverseIncrementalChanges() {
    this.changeList.clear();
    this.optimizeChangeList();
    for (final CppChangeObject change : this.changeList) {
      final int _switchValue = change.changeKind;
      switch (_switchValue) {
        case IResourceDelta.ADDED:
          this.addToModel(change);
          break;
        case IResourceDelta.CHANGED:
          this.updateToModel(change);
          break;
        case IResourceDelta.REMOVED:
          this.removeFromModel(change);
          break;
      }
    }
    this.changeList.clear();
  }

  private List<CElementChange> getRawChangeList() {
    if ((((ChangeMapStore.changesMap != null) && (ChangeMapStore.changesMap.get(this.project.getElementName()) != null)) && 
      (ChangeMapStore.changesMap.get(this.project.getElementName()).size() > 0))) {
      return ChangeMapStore.changesMap.get(this.project.getElementName());
    }
    return null;
  }

  private void clearRawChangeList() {
    if ((((ChangeMapStore.changesMap != null) && (this.project != null)) && 
      (ChangeMapStore.changesMap.get(this.project.getElementName()) != null))) {
      ChangeMapStore.changesMap.get(this.project.getElementName()).clear();
    }
  }

  private void optimizeChangeList() {
    List<CElementChange> rawChangeList = this.getRawChangeList();
    this.changeList.clear();
    List<CElementChange> processedList = new UniqueEList<CElementChange>();
    List<CElementChange> remainList = new UniqueEList<CElementChange>();
    for (int i = 0; (i < rawChangeList.size()); i++) {
      {
        CElementChange change = rawChangeList.get(i);
        boolean _isInSourceContainers = this.isInSourceContainers(change.getElement());
        if (_isInSourceContainers) {
          boolean _contains = processedList.contains(change);
          boolean _not = (!_contains);
          if (_not) {
            int _changeKind = change.getChangeKind();
            switch (_changeKind) {
              case IResourceDelta.ADDED:
                List<CElementChange> inter = this.findIntermediateEvent(rawChangeList, i, change);
                int _size = inter.size();
                boolean _greaterThan = (_size > 0);
                if (_greaterThan) {
                  processedList.add(change);
                  processedList.addAll(inter);
                }
                break;
              case IResourceDelta.CHANGED:
                List<CElementChange> inter_1 = this.findIntermediateEvent(rawChangeList, i, change);
                int _size_1 = inter_1.size();
                boolean _greaterThan_1 = (_size_1 > 0);
                if (_greaterThan_1) {
                  processedList.addAll(inter_1);
                }
                break;
              case IResourceDelta.REMOVED:
                break;
              default:
                processedList.add(change);
                break;
            }
          }
          boolean _contains_1 = processedList.contains(change);
          boolean _not_1 = (!_contains_1);
          if (_not_1) {
            remainList.add(change);
          }
        }
      }
    }
    final Map<ICElement, List<CppChangeObject>> doubtChangeListMap = new HashMap<ICElement, List<CppChangeObject>>();
    for (int i = 0; (i < remainList.size()); i++) {
      {
        CElementChange change = remainList.get(i);
        CppChangeObject cppChangeObject = null;
        int _changeKind = change.getChangeKind();
        switch (_changeKind) {
          case IResourceDelta.ADDED:
            boolean found = false;
            int j = (i + 1);
            while (((j < remainList.size()) && (!found))) {
              {
                if ((((remainList.get(j).getChangeKind() == IResourceDelta.REMOVED) && 
                  (change.getElement().getElementType() == remainList.get(j).getElement().getElementType())) && 
                  Objects.equal(remainList.get(j).getParent(), change.getParent()))) {
                  ICElement _element = remainList.get(j).getElement();
                  ICElement _element_1 = change.getElement();
                  ITranslationUnit _translationUnitFromElement = ReverseUtils.getTranslationUnitFromElement(change.getElement());
                  ICElement _parent = change.getParent();
                  CppChangeObject _cppChangeObject = new CppChangeObject(_element, _element_1, _translationUnitFromElement, _parent, IResourceDelta.CHANGED);
                  cppChangeObject = _cppChangeObject;
                  List<CppChangeObject> _get = doubtChangeListMap.get(change.getParent());
                  boolean _tripleEquals = (_get == null);
                  if (_tripleEquals) {
                    ICElement _parent_1 = change.getParent();
                    ArrayList<CppChangeObject> _arrayList = new ArrayList<CppChangeObject>();
                    doubtChangeListMap.put(_parent_1, _arrayList);
                  }
                  if ((((change.getElement() instanceof IField) || (change.getElement() instanceof IMethodDeclaration)) || 
                    (change.getElement() instanceof IEnumerator))) {
                    doubtChangeListMap.get(change.getParent()).add(cppChangeObject);
                  }
                  remainList.remove(j);
                  j--;
                  found = true;
                }
                j++;
              }
            }
            if ((!found)) {
              ICElement _element = change.getElement();
              ITranslationUnit _translationUnitFromElement = ReverseUtils.getTranslationUnitFromElement(change.getElement());
              ICElement _parent = change.getParent();
              CppChangeObject _cppChangeObject = new CppChangeObject(null, _element, _translationUnitFromElement, _parent, IResourceDelta.ADDED);
              cppChangeObject = _cppChangeObject;
            }
            break;
          case IResourceDelta.CHANGED:
            ICElement _element_1 = remainList.get(i).getElement();
            ICElement _element_2 = change.getElement();
            ITranslationUnit _translationUnitFromElement_1 = ReverseUtils.getTranslationUnitFromElement(change.getElement());
            ICElement _parent_1 = change.getParent();
            CppChangeObject _cppChangeObject_1 = new CppChangeObject(_element_1, _element_2, _translationUnitFromElement_1, _parent_1, IResourceDelta.CHANGED);
            cppChangeObject = _cppChangeObject_1;
            break;
          case IResourceDelta.REMOVED:
            boolean found_1 = false;
            int j_1 = (i + 1);
            while (((j_1 < remainList.size()) && (!found_1))) {
              {
                if (((remainList.get(j_1).getChangeKind() == IResourceDelta.ADDED) && 
                  (change.getElement().getElementType() == remainList.get(j_1).getElement().getElementType()))) {
                  ICElement _element_3 = change.getElement();
                  ICElement _element_4 = remainList.get(j_1).getElement();
                  ITranslationUnit _translationUnitFromElement_2 = ReverseUtils.getTranslationUnitFromElement(change.getElement());
                  ICElement _parent_2 = change.getParent();
                  CppChangeObject _cppChangeObject_2 = new CppChangeObject(_element_3, _element_4, _translationUnitFromElement_2, _parent_2, IResourceDelta.CHANGED);
                  cppChangeObject = _cppChangeObject_2;
                  List<CppChangeObject> _get = doubtChangeListMap.get(change.getParent());
                  boolean _tripleEquals = (_get == null);
                  if (_tripleEquals) {
                    ICElement _parent_3 = change.getParent();
                    ArrayList<CppChangeObject> _arrayList = new ArrayList<CppChangeObject>();
                    doubtChangeListMap.put(_parent_3, _arrayList);
                  }
                  if ((((change.getElement() instanceof IField) || (change.getElement() instanceof IMethodDeclaration)) || 
                    (change.getElement() instanceof IEnumerator))) {
                    doubtChangeListMap.get(change.getParent()).add(cppChangeObject);
                  }
                  remainList.remove(j_1);
                  j_1--;
                  found_1 = true;
                }
                j_1++;
              }
            }
            if ((!found_1)) {
              ICElement _element_3 = change.getElement();
              ITranslationUnit _translationUnitFromElement_2 = ReverseUtils.getTranslationUnitFromElement(change.getElement());
              ICElement _parent_2 = change.getParent();
              CppChangeObject _cppChangeObject_2 = new CppChangeObject(null, _element_3, _translationUnitFromElement_2, _parent_2, IResourceDelta.REMOVED);
              cppChangeObject = _cppChangeObject_2;
            }
            break;
        }
        if ((cppChangeObject != null)) {
          this.changeList.add(cppChangeObject);
        }
      }
    }
    Set<Map.Entry<ICElement, List<CppChangeObject>>> _entrySet = doubtChangeListMap.entrySet();
    for (final Map.Entry<ICElement, List<CppChangeObject>> doubtMap : _entrySet) {
      int _size = doubtMap.getValue().size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.alignChangedElements(doubtMap.getKey(), doubtMap.getValue());
      }
    }
    this.clearRawChangeList();
  }

  private void alignChangedElements(final ICElement parent, final List<CppChangeObject> changes) {
    try {
      final ArrayList<NamedElement> oldNamedElementsList = new ArrayList<NamedElement>();
      final Map<NamedElement, CppChangeObject> oldNamedElementToChangeMap = new HashMap<NamedElement, CppChangeObject>();
      final Consumer<CppChangeObject> _function = new Consumer<CppChangeObject>() {
        @Override
        public void accept(final CppChangeObject it) {
          NamedElement oldNamedElement = ReverseCpp2Uml.this.getNamedElement(it.parent, it.oldElement);
          oldNamedElementToChangeMap.put(oldNamedElement, it);
          oldNamedElementsList.add(oldNamedElement);
        }
      };
      changes.forEach(_function);
      final EObject namedElementParent = IterableExtensions.<NamedElement>head(oldNamedElementsList).eContainer();
      final ArrayList<NamedElement> alignedOldNamedElements = new ArrayList<NamedElement>();
      final Consumer<NamedElement> _function_1 = new Consumer<NamedElement>() {
        @Override
        public void accept(final NamedElement it) {
          boolean _contains = oldNamedElementsList.contains(it);
          if (_contains) {
            alignedOldNamedElements.add(it);
          }
        }
      };
      Iterables.<NamedElement>filter(namedElementParent.eContents(), NamedElement.class).forEach(_function_1);
      final ArrayList<ICElement> newICElements = new ArrayList<ICElement>();
      if ((parent instanceof IParent)) {
        final Consumer<ICElement> _function_2 = new Consumer<ICElement>() {
          @Override
          public void accept(final ICElement it) {
            final ICElement childInParent = it;
            final Function1<CppChangeObject, Boolean> _function = new Function1<CppChangeObject, Boolean>() {
              @Override
              public Boolean apply(final CppChangeObject it_1) {
                return Boolean.valueOf(Objects.equal(it_1.newElement, childInParent));
              }
            };
            final CppChangeObject founded = IterableExtensions.<CppChangeObject>head(IterableExtensions.<CppChangeObject>filter(changes, _function));
            if ((founded != null)) {
              newICElements.add(founded.newElement);
            }
          }
        };
        ((List<ICElement>)Conversions.doWrapArray(((IParent)parent).getChildren())).forEach(_function_2);
      }
      int _size = alignedOldNamedElements.size();
      int _size_1 = newICElements.size();
      boolean _equals = (_size == _size_1);
      if (_equals) {
        for (int i = 0; (i < alignedOldNamedElements.size()); i++) {
          {
            CppChangeObject change = oldNamedElementToChangeMap.get(alignedOldNamedElements.get(i));
            ICElement _get = newICElements.get(i);
            boolean _notEquals = (!Objects.equal(change.newElement, _get));
            if (_notEquals) {
              boolean sameElementType = ((change.newElement instanceof IField) && (newICElements.get(i) instanceof IField));
              sameElementType = (sameElementType || ((change.newElement instanceof IMethodDeclaration) && 
                (newICElements.get(i) instanceof IMethodDeclaration)));
              sameElementType = (sameElementType || ((change.newElement instanceof IEnumerator) && (newICElements.get(i) instanceof IEnumerator)));
              if (sameElementType) {
                change.newElement = newICElements.get(i);
              }
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private NamedElement getNamedElement(final ICElement parent, final ICElement child) {
    try {
      NamedElement ret = null;
      boolean _matched = false;
      if ((child instanceof IField)) {
        _matched=true;
        if ((parent instanceof IStructure)) {
          IStructure parentStructure = ((IStructure) parent);
          Type classifier = GetOrCreateP2.getClassifier(parentStructure);
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, child.getElementName());
            if (_isExist) {
              final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                @Override
                public Boolean apply(final Property it) {
                  return Boolean.valueOf(it.getName().equals(child.getElementName()));
                }
              };
              Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
              ret = prop;
            }
          }
        }
      }
      if (!_matched) {
        if ((child instanceof IMethodDeclaration)) {
          _matched=true;
          if ((parent instanceof IStructure)) {
            IStructure parentStructure_1 = ((IStructure) parent);
            Type classifier_1 = GetOrCreateP2.getClassifier(parentStructure_1);
            if ((classifier_1 != null)) {
              boolean _isExist_1 = this.isExist(classifier_1, child.getElementName());
              if (_isExist_1) {
                final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(it.getName().equals(child.getElementName()));
                  }
                };
                List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function_1));
                final IMethodDeclaration oldMethod = ((IMethodDeclaration) child);
                final Function1<Operation, Boolean> _function_2 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, oldMethod));
                  }
                };
                Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_2));
                if ((op != null)) {
                  ret = op;
                }
              }
            }
          }
        }
      }
      if (!_matched) {
        if ((child instanceof IEnumerator)) {
          _matched=true;
          IEnumeration cppEnumeration = ((IEnumeration) parent);
          Type enumeration = GetOrCreateP2.getClassifier(cppEnumeration);
          if (((enumeration != null) && (enumeration instanceof Enumeration))) {
            final Function1<EnumerationLiteral, Boolean> _function_3 = new Function1<EnumerationLiteral, Boolean>() {
              @Override
              public Boolean apply(final EnumerationLiteral it) {
                String _name = it.getName();
                String _elementName = child.getElementName();
                return Boolean.valueOf(Objects.equal(_name, _elementName));
              }
            };
            EnumerationLiteral enumerator = IterableExtensions.<EnumerationLiteral>head(IterableExtensions.<EnumerationLiteral>filter(Iterables.<EnumerationLiteral>filter(((Enumeration) enumeration).getOwnedLiterals(), EnumerationLiteral.class), _function_3));
            if ((enumerator != null)) {
              ret = enumerator;
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private boolean isInSourceContainers(final ICElement element) {
    if ((element == null)) {
      return false;
    }
    if (((element instanceof ICContainer) && this.containers.contains(element))) {
      return true;
    }
    return this.isInSourceContainers(element.getParent());
  }

  private List<CElementChange> findIntermediateEvent(final List<CElementChange> list, final int position, final CElementChange change) {
    List<CElementChange> ret = new UniqueEList<CElementChange>();
    for (int i = (position + 1); (i < list.size()); i++) {
      if ((((change.getChangeKind() == IResourceDelta.ADDED) && (list.get(i).getChangeKind() == IResourceDelta.REMOVED)) && 
        this.isSameICElement(change.getElement(), list.get(i).getElement()))) {
        if (((change.getElement() instanceof IMethodDeclaration) && (list.get(i).getElement() instanceof IMethodDeclaration))) {
          ICElement _element = change.getElement();
          final IMethodDeclaration method = ((IMethodDeclaration) _element);
          ICElement _element_1 = list.get(i).getElement();
          boolean _isSameMethodDeclaration = this.isSameMethodDeclaration(method, ((IMethodDeclaration) _element_1));
          if (_isSameMethodDeclaration) {
            ret.add(list.get(i));
          }
        } else {
          ret.add(list.get(i));
        }
      } else {
        if (((change.getChangeKind() == IResourceDelta.CHANGED) && 
          this.isSameICElement(change.getElement(), list.get(i).getElement()))) {
          ret.add(list.get(i));
        }
      }
    }
    final Function1<CElementChange, Boolean> _function = new Function1<CElementChange, Boolean>() {
      @Override
      public Boolean apply(final CElementChange it) {
        int _changeKind = it.getChangeKind();
        return Boolean.valueOf((_changeKind == IResourceDelta.CHANGED));
      }
    };
    List<CElementChange> changed = IterableExtensions.<CElementChange>toList(IterableExtensions.<CElementChange>filter(ret, _function));
    int _size = changed.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      ret.remove(IterableExtensions.<CElementChange>last(changed));
    }
    return ret;
  }

  private boolean isSameICElement(final ICElement e1, final ICElement e2) {
    boolean ret = false;
    if (((e1.getElementName().equals(e2.getElementName()) && (e1.getElementType() == e2.getElementType())) && 
      e1.getParent().getElementName().equals(e2.getParent().getElementName()))) {
      ret = true;
    }
    return ret;
  }

  private boolean isExist(final NamedElement parent, final String childName) {
    final Function1<NamedElement, Boolean> _function = new Function1<NamedElement, Boolean>() {
      @Override
      public Boolean apply(final NamedElement it) {
        return Boolean.valueOf(it.getName().equals(childName));
      }
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<NamedElement>filter(Iterables.<NamedElement>filter(parent.getOwnedElements(), NamedElement.class), _function));
    return (!_isEmpty);
  }

  private Object addToModel(final CppChangeObject change) {
    try {
      Object _switchResult = null;
      final ICElement _switchValue = change.newElement;
      boolean _matched = false;
      if ((change.newElement instanceof IField)) {
        _matched=true;
        final IField field = ((IField) change.newElement);
        ICElement _parent = field.getParent();
        if ((_parent instanceof IStructure)) {
          ICElement _parent_1 = field.getParent();
          IStructure parentStructure = ((IStructure) _parent_1);
          Type classifier = GetOrCreateP2.getClassifier(parentStructure);
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, field.getElementName());
            boolean _not = (!_isExist);
            if (_not) {
              PropertyUtils.createProperty(field, ((Classifier) classifier));
            }
          }
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IMethodDeclaration)) {
          _matched=true;
          Object _xblockexpression = null;
          {
            final IMethodDeclaration method = ((IMethodDeclaration) change.newElement);
            Object _xifexpression = null;
            ICElement _parent_2 = method.getParent();
            if ((_parent_2 instanceof IStructure)) {
              Object _xblockexpression_1 = null;
              {
                ICElement _parent_3 = method.getParent();
                IStructure parentStructure_1 = ((IStructure) _parent_3);
                Type classifier_1 = GetOrCreateP2.getClassifier(parentStructure_1);
                Object _xifexpression_1 = null;
                if ((classifier_1 != null)) {
                  Object _xblockexpression_2 = null;
                  {
                    final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
                      @Override
                      public Boolean apply(final Operation it) {
                        return Boolean.valueOf(it.getName().equals(method.getElementName()));
                      }
                    };
                    List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function));
                    Object _xifexpression_2 = null;
                    final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                      @Override
                      public Boolean apply(final Operation it) {
                        return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, method));
                      }
                    };
                    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<Operation>filter(samenames, _function_1));
                    if (_isEmpty) {
                      _xifexpression_2 = MethodUtils.createMethod(method, ((org.eclipse.uml2.uml.Class) classifier_1));
                    }
                    _xblockexpression_2 = _xifexpression_2;
                  }
                  _xifexpression_1 = _xblockexpression_2;
                }
                _xblockexpression_1 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_1;
            } else {
              Object _xifexpression_1 = null;
              ICElement _parent_3 = method.getParent();
              if ((_parent_3 instanceof ITranslationUnit)) {
                _xifexpression_1 = null;
              }
              _xifexpression = _xifexpression_1;
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.newElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.newElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          Classifier _xblockexpression_1 = null;
          {
            IDeclaration declaration = ((IDeclaration) change.newElement);
            _xblockexpression_1 = GetOrCreateP1.getOrCreateClassifier(declaration);
          }
          _switchResult = _xblockexpression_1;
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IEnumerator)) {
          _matched=true;
          IEnumerator enumerator = ((IEnumerator) change.newElement);
          IPath _path = enumerator.getPath();
          IEnumeration cppEnumeration = ((IEnumeration) _path);
          Type enumeration = GetOrCreateP2.getClassifier(cppEnumeration);
          if ((enumeration instanceof Enumeration)) {
            ReverseUtils.setXmlID(((Enumeration)enumeration).createOwnedLiteral(enumerator.getElementName()));
          }
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof ITranslationUnit)) {
          _matched=true;
          ITranslationUnit itu = ((ITranslationUnit) change.newElement);
          GetOrCreateP1.getOrCreateClassifiers(itu);
        }
      }
      return _switchResult;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private Object updateToModel(final CppChangeObject change) {
    try {
      Object _switchResult = null;
      final ICElement _switchValue = change.newElement;
      boolean _matched = false;
      if ((change.oldElement instanceof IField)) {
        _matched=true;
        final IField field = ((IField) change.newElement);
        ICElement _parent = field.getParent();
        if ((_parent instanceof IStructure)) {
          ICElement _parent_1 = field.getParent();
          IStructure parentStructure = ((IStructure) _parent_1);
          Type classifier = GetOrCreateP2.getClassifier(parentStructure);
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
            if (_isExist) {
              final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                @Override
                public Boolean apply(final Property it) {
                  return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                }
              };
              Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
              if ((prop != null)) {
                PropertyUtils.updateProperty(((IField) change.newElement), prop);
              }
            }
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IMethodDeclaration)) {
          _matched=true;
          Object _xblockexpression = null;
          {
            final IMethodDeclaration method = ((IMethodDeclaration) change.newElement);
            Object _xifexpression = null;
            ICElement _parent_2 = method.getParent();
            if ((_parent_2 instanceof IStructure)) {
              Object _xblockexpression_1 = null;
              {
                ICElement _parent_3 = method.getParent();
                IStructure parentStructure_1 = ((IStructure) _parent_3);
                Type classifier_1 = GetOrCreateP2.getClassifier(parentStructure_1);
                Object _xifexpression_1 = null;
                if ((classifier_1 != null)) {
                  Object _xifexpression_2 = null;
                  boolean _isExist_1 = this.isExist(classifier_1, change.oldElement.getElementName());
                  if (_isExist_1) {
                    Object _xblockexpression_2 = null;
                    {
                      final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                        }
                      };
                      List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function_1));
                      final IMethodDeclaration oldMethod = ((IMethodDeclaration) change.oldElement);
                      final Function1<Operation, Boolean> _function_2 = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, oldMethod));
                        }
                      };
                      Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_2));
                      Object _xifexpression_3 = null;
                      if ((op != null)) {
                        _xifexpression_3 = MethodUtils.updateMethod(((org.eclipse.uml2.uml.Class) classifier_1), op, 
                          ((IMethodDeclaration) change.newElement));
                      }
                      _xblockexpression_2 = _xifexpression_3;
                    }
                    _xifexpression_2 = _xblockexpression_2;
                  }
                  _xifexpression_1 = _xifexpression_2;
                }
                _xblockexpression_1 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_1;
            } else {
              ICElement _parent_3 = method.getParent();
              if ((_parent_3 instanceof ITranslationUnit)) {
                ICElement _parent_4 = method.getParent();
                this.reverseSource(((ITranslationUnit) _parent_4));
              }
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.oldElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.oldElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          IDeclaration declaration = ((IDeclaration) change.newElement);
          Type umlClassifier = GetOrCreateP2.getClassifier(((IDeclaration) change.oldElement));
          if ((umlClassifier != null)) {
            umlClassifier.setName(declaration.getElementName());
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IEnumerator)) {
          _matched=true;
          final IEnumerator cppEnumerator = ((IEnumerator) change.newElement);
          ICElement _parent_2 = cppEnumerator.getParent();
          IEnumeration cppEnumeration = ((IEnumeration) _parent_2);
          Type enumeration = GetOrCreateP2.getClassifier(cppEnumeration);
          if (((enumeration != null) && (enumeration instanceof Enumeration))) {
            final Function1<EnumerationLiteral, Boolean> _function_1 = new Function1<EnumerationLiteral, Boolean>() {
              @Override
              public Boolean apply(final EnumerationLiteral it) {
                String _name = it.getName();
                String _elementName = change.oldElement.getElementName();
                return Boolean.valueOf(Objects.equal(_name, _elementName));
              }
            };
            EnumerationLiteral enumerator = IterableExtensions.<EnumerationLiteral>head(IterableExtensions.<EnumerationLiteral>filter(Iterables.<EnumerationLiteral>filter(((Enumeration) enumeration).getOwnedLiterals(), EnumerationLiteral.class), _function_1));
            if ((enumerator != null)) {
              enumerator.setName(cppEnumerator.getElementName());
            }
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof ITranslationUnit)) {
          _matched=true;
          ITranslationUnit itu = ((ITranslationUnit) change.newElement);
          boolean _isSourceUnit = itu.isSourceUnit();
          if (_isSourceUnit) {
            this.reverseSource(itu);
          } else {
            boolean _isHeaderUnit = itu.isHeaderUnit();
            if (_isHeaderUnit) {
              this.reverseHeader(itu);
            }
          }
        }
      }
      return _switchResult;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private Boolean removeFromModel(final CppChangeObject change) {
    try {
      boolean _switchResult = false;
      final ICElement _switchValue = change.oldElement;
      boolean _matched = false;
      if ((change.oldElement instanceof IField)) {
        _matched=true;
        boolean _xblockexpression = false;
        {
          final IField field = ((IField) change.oldElement);
          boolean _xifexpression = false;
          ICElement _parent = field.getParent();
          if ((_parent instanceof IStructure)) {
            boolean _xblockexpression_1 = false;
            {
              ICElement _parent_1 = field.getParent();
              IStructure parentStructure = ((IStructure) _parent_1);
              Type classifier = GetOrCreateP2.getClassifier(parentStructure);
              boolean _xifexpression_1 = false;
              if ((classifier != null)) {
                boolean _xifexpression_2 = false;
                boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
                if (_isExist) {
                  boolean _xblockexpression_2 = false;
                  {
                    final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                      @Override
                      public Boolean apply(final Property it) {
                        return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                      }
                    };
                    Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
                    boolean _xifexpression_3 = false;
                    if ((prop != null)) {
                      boolean _xifexpression_4 = false;
                      if ((classifier instanceof org.eclipse.uml2.uml.Class)) {
                        EList<Property> _ownedAttributes = ((org.eclipse.uml2.uml.Class)classifier).getOwnedAttributes();
                        _xifexpression_4 = _ownedAttributes.remove(prop);
                      } else {
                        boolean _xifexpression_5 = false;
                        if ((classifier instanceof DataType)) {
                          EList<Property> _ownedAttributes_1 = ((DataType)classifier).getOwnedAttributes();
                          _xifexpression_5 = _ownedAttributes_1.remove(prop);
                        }
                        _xifexpression_4 = _xifexpression_5;
                      }
                      _xifexpression_3 = _xifexpression_4;
                    }
                    _xblockexpression_2 = _xifexpression_3;
                  }
                  _xifexpression_2 = _xblockexpression_2;
                }
                _xifexpression_1 = _xifexpression_2;
              }
              _xblockexpression_1 = _xifexpression_1;
            }
            _xifexpression = _xblockexpression_1;
          }
          _xblockexpression = _xifexpression;
        }
        _switchResult = _xblockexpression;
      }
      if (!_matched) {
        if ((change.oldElement instanceof IMethodDeclaration)) {
          _matched=true;
          boolean _xblockexpression_1 = false;
          {
            final IMethodDeclaration method = ((IMethodDeclaration) change.oldElement);
            boolean _xifexpression = false;
            ICElement _parent = method.getParent();
            if ((_parent instanceof IStructure)) {
              boolean _xblockexpression_2 = false;
              {
                ICElement _parent_1 = method.getParent();
                IStructure parentStructure = ((IStructure) _parent_1);
                Type classifier = GetOrCreateP2.getClassifier(parentStructure);
                boolean _xifexpression_1 = false;
                if ((classifier != null)) {
                  boolean _xifexpression_2 = false;
                  boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
                  if (_isExist) {
                    boolean _xblockexpression_3 = false;
                    {
                      final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(it.getName().equals(method.getElementName()));
                        }
                      };
                      List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier.getOwnedElements(), Operation.class), _function));
                      final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, method));
                        }
                      };
                      Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_1));
                      boolean _xifexpression_3 = false;
                      if ((op != null)) {
                        boolean _xifexpression_4 = false;
                        if ((classifier instanceof org.eclipse.uml2.uml.Class)) {
                          EList<Operation> _ownedOperations = ((org.eclipse.uml2.uml.Class)classifier).getOwnedOperations();
                          _xifexpression_4 = _ownedOperations.remove(op);
                        }
                        _xifexpression_3 = _xifexpression_4;
                      }
                      _xblockexpression_3 = _xifexpression_3;
                    }
                    _xifexpression_2 = _xblockexpression_3;
                  }
                  _xifexpression_1 = _xifexpression_2;
                }
                _xblockexpression_2 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_2;
            }
            _xblockexpression_1 = _xifexpression;
          }
          _switchResult = _xblockexpression_1;
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.oldElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.oldElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          boolean _xblockexpression_2 = false;
          {
            final IDeclaration declaration = ((IDeclaration) change.oldElement);
            boolean _xifexpression = false;
            if ((change.parent instanceof IStructure)) {
              boolean _xblockexpression_3 = false;
              {
                IStructure parentStructure = ((IStructure) change.parent);
                Type parentClassifier = GetOrCreateP2.getClassifier(parentStructure);
                final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
                  @Override
                  public Boolean apply(final Type it) {
                    String _name = it.getName();
                    String _elementName = declaration.getElementName();
                    return Boolean.valueOf(Objects.equal(_name, _elementName));
                  }
                };
                Type childClassifier = IterableExtensions.<Type>head(IterableExtensions.<Type>filter(Iterables.<Type>filter(parentClassifier.getOwnedElements(), Type.class), _function));
                boolean _xifexpression_1 = false;
                if (((childClassifier != null) && (parentClassifier instanceof org.eclipse.uml2.uml.Class))) {
                  EList<Classifier> _nestedClassifiers = ((org.eclipse.uml2.uml.Class) parentClassifier).getNestedClassifiers();
                  _xifexpression_1 = _nestedClassifiers.remove(((Classifier) childClassifier));
                }
                _xblockexpression_3 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_3;
            } else {
              boolean _xblockexpression_4 = false;
              {
                Model containerPackage = ReverseUtils.getCorrespondingModel(declaration.getTranslationUnit());
                final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
                  @Override
                  public Boolean apply(final Type it) {
                    String _name = it.getName();
                    String _elementName = declaration.getElementName();
                    return Boolean.valueOf(Objects.equal(_name, _elementName));
                  }
                };
                Type childClassifier = IterableExtensions.<Type>head(IterableExtensions.<Type>filter(Iterables.<Type>filter(containerPackage.getOwnedElements(), Type.class), _function));
                EList<Type> _ownedTypes = containerPackage.getOwnedTypes();
                _xblockexpression_4 = _ownedTypes.remove(childClassifier);
              }
              _xifexpression = _xblockexpression_4;
            }
            _xblockexpression_2 = _xifexpression;
          }
          _switchResult = _xblockexpression_2;
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IEnumerator)) {
          _matched=true;
          boolean _xblockexpression_3 = false;
          {
            IEnumerator cppEnumerator = ((IEnumerator) change.oldElement);
            IPath _path = cppEnumerator.getPath();
            IEnumeration cppEnumeration = ((IEnumeration) _path);
            Type enumeration = GetOrCreateP2.getClassifier(cppEnumeration);
            boolean _xifexpression = false;
            if (((enumeration != null) && (enumeration instanceof Enumeration))) {
              boolean _xblockexpression_4 = false;
              {
                EnumerationLiteral enumerator = ((Enumeration) enumeration).getOwnedLiteral(change.oldElement.getElementName());
                boolean _xifexpression_1 = false;
                if ((enumerator != null)) {
                  EList<EnumerationLiteral> _ownedLiterals = ((Enumeration) enumeration).getOwnedLiterals();
                  _xifexpression_1 = _ownedLiterals.remove(enumerator);
                }
                _xblockexpression_4 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_4;
            }
            _xblockexpression_3 = _xifexpression;
          }
          _switchResult = _xblockexpression_3;
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof ITranslationUnit)) {
          _matched=true;
          ITranslationUnit itu = ((ITranslationUnit) change.newElement);
          GetOrCreateP1.getOrCreateClassifiers(itu);
        }
      }
      return Boolean.valueOf(_switchResult);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
