package idea

import (
	"bytes"
	"encoding/hex"
	"testing"
)

func TestEncryptDecrypt(t *testing.T) {

	var tests = []struct {
		key    string
		plain  string
		cipher string
	}{
		// From https://www.cosic.esat.kuleuven.be/nessie/testvectors/bc/idea/Idea-128-64.verified.test-vectors
		{"80000000000000000000000000000000", "0000000000000000", "B1F5F7F87901370F"},
		{"40000000000000000000000000000000", "0000000000000000", "B3927DFFB6358626"},
		{"20000000000000000000000000000000", "0000000000000000", "E987E0029FB99785"},
		{"10000000000000000000000000000000", "0000000000000000", "754A03CE08DB7DAA"},
		{"08000000000000000000000000000000", "0000000000000000", "F015F9FB0CFC7E1C"},
		{"04000000000000000000000000000000", "0000000000000000", "69C9FE6007B8FCDF"},
		{"02000000000000000000000000000000", "0000000000000000", "8DA7BC0E63B40DD0"},
		{"01000000000000000000000000000000", "0000000000000000", "2C49BF7DE28C666B"},
		{"00800000000000000000000000000000", "0000000000000000", "9A4717E8F935712B"},
		{"00400000000000000000000000000000", "0000000000000000", "90C77C47804BEDCC"},
		{"00200000000000000000000000000000", "0000000000000000", "156E976EE8194241"},
		{"00100000000000000000000000000000", "0000000000000000", "168A0AD30485FC27"},
		{"00080000000000000000000000000000", "0000000000000000", "3F3B32602C5B4FED"},
		{"00040000000000000000000000000000", "0000000000000000", "7C282EA23CA3B968"},
		{"00020000000000000000000000000000", "0000000000000000", "13EA33701F98CCE5"},
		{"00010000000000000000000000000000", "0000000000000000", "7C92BC91C48F0084"},
		{"00008000000000000000000000000000", "0000000000000000", "95A96731978C1B9A"},
		{"00004000000000000000000000000000", "0000000000000000", "125D24C4E9ABA0EB"},
		{"00002000000000000000000000000000", "0000000000000000", "61ACA85576C2F0CB"},
		{"00001000000000000000000000000000", "0000000000000000", "1885D2A027FCCDC3"},
		{"00000800000000000000000000000000", "0000000000000000", "3E95A27D17961B91"},
		{"00000400000000000000000000000000", "0000000000000000", "1BAAE3AFE3DC73B9"},
		{"00000200000000000000000000000000", "0000000000000000", "CE48F58B50E125E9"},
		{"00000100000000000000000000000000", "0000000000000000", "CE359BDCF1AF56C0"},
		{"00000080000000000000000000000000", "0000000000000000", "80960B2AAFABA050"},
		{"00000040000000000000000000000000", "0000000000000000", "6A475EB5DEE65E05"},
		{"00000020000000000000000000000000", "0000000000000000", "09B03752CA2E76A3"},
		{"00000010000000000000000000000000", "0000000000000000", "8DF2AD1153945DF8"},
		{"00000008000000000000000000000000", "0000000000000000", "38731680D5611515"},
		{"00000004000000000000000000000000", "0000000000000000", "1B96D7962EEE006E"},
		{"00000002000000000000000000000000", "0000000000000000", "B2CF3BE70E2CE320"},
		{"00000001000000000000000000000000", "0000000000000000", "4B9F147724732802"},
		{"00000000800000000000000000000000", "0000000000000000", "398BD9A59E9F5DDB"},
		{"00000000400000000000000000000000", "0000000000000000", "BF70E86B86D9E071"},
		{"00000000200000000000000000000000", "0000000000000000", "E8F3D34C32561B8D"},
		{"00000000100000000000000000000000", "0000000000000000", "564D4C9F9F99EF58"},
		{"00000000080000000000000000000000", "0000000000000000", "3FA5DCB6D71577C9"},
		{"00000000040000000000000000000000", "0000000000000000", "F2470B4C18D8206F"},
		{"00000000020000000000000000000000", "0000000000000000", "C838B0B08145E33D"},
		{"00000000010000000000000000000000", "0000000000000000", "E66B4B804D49FFD4"},
		{"00000000008000000000000000000000", "0000000000000000", "849232D009776E88"},
		{"00000000004000000000000000000000", "0000000000000000", "06114F7642747BF2"},
		{"00000000002000000000000000000000", "0000000000000000", "55E436BF83FCC61F"},
		{"00000000001000000000000000000000", "0000000000000000", "AFC56E209FFE8E5E"},
		{"00000000000800000000000000000000", "0000000000000000", "1FCD5543ED20219F"},
		{"00000000000400000000000000000000", "0000000000000000", "AFD481EEB1DC9798"},
		{"00000000000200000000000000000000", "0000000000000000", "C747245F7C40505E"},
		{"00000000000100000000000000000000", "0000000000000000", "6533869714E1D576"},
		{"00000000000080000000000000000000", "0000000000000000", "AC1D8708AF0A37EE"},
		{"00000000000040000000000000000000", "0000000000000000", "6AE43F39ABEC1634"},
		{"00000000000020000000000000000000", "0000000000000000", "3CA0B0184CE033B0"},
		{"00000000000010000000000000000000", "0000000000000000", "A1FAE8B9764179C9"},
		{"00000000000008000000000000000000", "0000000000000000", "34E7FE59D0ADCEAA"},
		{"00000000000004000000000000000000", "0000000000000000", "3CFB4E3A73CBDFC8"},
		{"00000000000002000000000000000000", "0000000000000000", "E7C34815E4AF6944"},
		{"00000000000001000000000000000000", "0000000000000000", "FD4B9074B638862D"},
		{"00000000000000800000000000000000", "0000000000000000", "C1B9F1684FC2EF4B"},
		{"00000000000000400000000000000000", "0000000000000000", "3C90FA06CEF23D8E"},
		{"00000000000000200000000000000000", "0000000000000000", "E304438D5C8FF203"},
		{"00000000000000100000000000000000", "0000000000000000", "8002EB809EC28F8A"},
		{"00000000000000080000000000000000", "0000000000000000", "4255F0C54F48C1A9"},
		{"00000000000000040000000000000000", "0000000000000000", "AA553A5DEC50E4A4"},
		{"00000000000000020000000000000000", "0000000000000000", "30F1EC7563C810F2"},
		{"00000000000000010000000000000000", "0000000000000000", "B3CDC5EBDA6A4FE9"},
		{"00000000000000008000000000000000", "0000000000000000", "FAE3FA7B8DB08800"},
		{"00000000000000004000000000000000", "0000000000000000", "05207CF4396CC5F4"},
		{"00000000000000002000000000000000", "0000000000000000", "3C6180D320D620B2"},
		{"00000000000000001000000000000000", "0000000000000000", "EEE00F6D006CE0A2"},
		{"00000000000000000800000000000000", "0000000000000000", "B040F041C7831740"},
		{"00000000000000000400000000000000", "0000000000000000", "74203C21EFC23FA0"},
		{"00000000000000000200000000000000", "0000000000000000", "7F901491CC611BA8"},
		{"00000000000000000100000000000000", "0000000000000000", "EDD8C619488123CE"},
		{"00000000000000000080000000000000", "0000000000000000", "FA6603F705135F50"},
		{"00000000000000000040000000000000", "0000000000000000", "64837B43683C68D0"},
		{"00000000000000000020000000000000", "0000000000000000", "DD0A23281E105910"},
		{"00000000000000000010000000000000", "0000000000000000", "FAE505F70760E814"},
		{"00000000000000000008000000000000", "0000000000000000", "2F72C7FB1E856D92"},
		{"00000000000000000004000000000000", "0000000000000000", "40463C6C86D5177A"},
		{"00000000000000000002000000000000", "0000000000000000", "9AD2AB3C260084E7"},
		{"00000000000000000001000000000000", "0000000000000000", "CF960FA9248BC228"},
		{"00000000000000000000800000000000", "0000000000000000", "B5803F82C0633F01"},
		{"00000000000000000000400000000000", "0000000000000000", "62DA6BB42BC888D0"},
		{"00000000000000000000200000000000", "0000000000000000", "B625F883693FF90B"},
		{"00000000000000000000100000000000", "0000000000000000", "5F385CE04727618F"},
		{"00000000000000000000080000000000", "0000000000000000", "6B391EB5C3CF660C"},
		{"00000000000000000000040000000000", "0000000000000000", "2E5B3366EAE2192A"},
		{"00000000000000000000020000000000", "0000000000000000", "587687B9EA5738B9"},
		{"00000000000000000000010000000000", "0000000000000000", "E83C1E95F50B1C45"},
		{"00000000000000000000008000000000", "0000000000000000", "A67559F516391F04"},
		{"00000000000000000000004000000000", "0000000000000000", "2B7012F6AFACC4E7"},
		{"00000000000000000000002000000000", "0000000000000000", "90E9C0B1B067BF0C"},
		{"00000000000000000000001000000000", "0000000000000000", "9E25090B7D4EF24E"},
		{"00000000000000000000000800000000", "0000000000000000", "EF62C1109F374AA8"},
		{"00000000000000000000000400000000", "0000000000000000", "1321DFB1CAF0175A"},
		{"00000000000000000000000200000000", "0000000000000000", "5F0CCFE5EB0F19A8"},
		{"00000000000000000000000100000000", "0000000000000000", "FCC40014010D617C"},
		{"00000000000000000000000080000000", "0000000000000000", "705D780834A498DA"},
		{"00000000000000000000000040000000", "0000000000000000", "9BCA7BF025B38A68"},
		{"00000000000000000000000020000000", "0000000000000000", "5CF67D0181CB01C1"},
		{"00000000000000000000000010000000", "0000000000000000", "ECDE3D81820381C1"},
		{"00000000000000000000000008000000", "0000000000000000", "44071D0142AA8D00"},
		{"00000000000000000000000004000000", "0000000000000000", "76028E8175574680"},
		{"00000000000000000000000002000000", "0000000000000000", "4E21C741CFB4A340"},
		{"00000000000000000000000001000000", "0000000000000000", "8DD9E3A14D02D1A0"},
		{"00000000000000000000000000800000", "0000000000000000", "C781050DC4110220"},
		{"00000000000000000000000000400000", "0000000000000000", "6DFD0287EC4C0110"},
		{"00000000000000000000000000200000", "0000000000000000", "3B8A017EFB61800E"},
		{"00000000000000000000000000100000", "0000000000000000", "A08F7F81FF627FC0"},
		{"00000000000000000000000000080000", "0000000000000000", "00503FC1AFB93FE0"},
		{"00000000000000000000000000040000", "0000000000000000", "4C23F3F71BDA73FA"},
		{"00000000000000000000000000020000", "0000000000000000", "89181EFAA2EBDEFF"},
		{"00000000000000000000000000010000", "0000000000000000", "46D371477F33B152"},
		{"00000000000000000000000000008000", "0000000000000000", "BE67AC7DA294CA7C"},
		{"00000000000000000000000000004000", "0000000000000000", "A02D82CAAA53B63B"},
		{"00000000000000000000000000002000", "0000000000000000", "7B3B1BB98486BABA"},
		{"00000000000000000000000000001000", "0000000000000000", "FB7A48BB85A599FD"},
		{"00000000000000000000000000000800", "0000000000000000", "C7311F4168C070A8"},
		{"00000000000000000000000000000400", "0000000000000000", "96A6BAA8315BF757"},
		{"00000000000000000000000000000200", "0000000000000000", "8C48BE46C7BD823D"},
		{"00000000000000000000000000000100", "0000000000000000", "B409C9F9D1B8F72D"},
		{"00000000000000000000000000000080", "0000000000000000", "3CBB5809405FF067"},
		{"00000000000000000000000000000040", "0000000000000000", "7ADBFA95852405D8"},
		{"00000000000000000000000000000020", "0000000000000000", "B9E879C7461A07E6"},
		{"00000000000000000000000000000010", "0000000000000000", "5E98BE9AA16FD1A8"},
		{"00000000000000000000000000000008", "0000000000000000", "F59D85F90A42B871"},
		{"00000000000000000000000000000004", "0000000000000000", "24853F38C6C7BC3F"},
		{"00000000000000000000000000000002", "0000000000000000", "41518093035E12E9"},
		{"00000000000000000000000000000001", "0000000000000000", "C57ADBDE27BC26CF"},
		{"00000000000000000000000000000000", "8000000000000000", "8001000180008000"},
		{"00000000000000000000000000000000", "4000000000000000", "C00180014000C000"},
		{"00000000000000000000000000000000", "2000000000000000", "6001C00120006000"},
		{"00000000000000000000000000000000", "1000000000000000", "3001600190003000"},
		{"00000000000000000000000000000000", "0800000000000000", "18013001C8009800"},
		{"00000000000000000000000000000000", "0400000000000000", "8C01180164004C00"},
		{"00000000000000000000000000000000", "0200000000000000", "46018C0132002600"},
		{"00000000000000000000000000000000", "0100000000000000", "2301C60199009300"},
		{"00000000000000000000000000000000", "0080000000000000", "9181E3014C80C980"},
		{"00000000000000000000000000000000", "0040000000000000", "C8C1F1812640E4C0"},
		{"00000000000000000000000000000000", "0020000000000000", "E461F8C11320F260"},
		{"00000000000000000000000000000000", "0010000000000000", "F231FC610990F930"},
		{"00000000000000000000000000000000", "0008000000000000", "F919FE3104C8FC98"},
		{"00000000000000000000000000000000", "0004000000000000", "FC8DFF190264FE4C"},
		{"00000000000000000000000000000000", "0002000000000000", "FE47FF8D0132FF26"},
		{"00000000000000000000000000000000", "0001000000000000", "0028FFF5003F0013"},
		{"00000000000000000000000000000000", "0000800000000000", "0001800180008000"},
		{"00000000000000000000000000000000", "0000400000000000", "00014001C0004000"},
		{"00000000000000000000000000000000", "0000200000000000", "8001A0016000A000"},
		{"00000000000000000000000000000000", "0000100000000000", "4001D00130005000"},
		{"00000000000000000000000000000000", "0000080000000000", "2001E8019800A800"},
		{"00000000000000000000000000000000", "0000040000000000", "900174014C005400"},
		{"00000000000000000000000000000000", "0000020000000000", "48013A012600AA00"},
		{"00000000000000000000000000000000", "0000010000000000", "A4011D019300D500"},
		{"00000000000000000000000000000000", "0000008000000000", "D2010E81C980EA80"},
		{"00000000000000000000000000000000", "0000004000000000", "E9010741E4C0F540"},
		{"00000000000000000000000000000000", "0000002000000000", "F48103A1F260FAA0"},
		{"00000000000000000000000000000000", "0000001000000000", "FA4101D1F930FD50"},
		{"00000000000000000000000000000000", "0000000800000000", "FD2100E9FC98FEA8"},
		{"00000000000000000000000000000000", "0000000400000000", "FE910075FE4CFF54"},
		{"00000000000000000000000000000000", "0000000200000000", "FF49003BFF26FFAA"},
		{"00000000000000000000000000000000", "0000000100000000", "0001000000010001"},
		{"00000000000000000000000000000000", "0000000080000000", "0001000180000000"},
		{"00000000000000000000000000000000", "0000000040000000", "00010001C0000000"},
		{"00000000000000000000000000000000", "0000000020000000", "8001000160000000"},
		{"00000000000000000000000000000000", "0000000010000000", "4001000130008000"},
		{"00000000000000000000000000000000", "0000000008000000", "200100019800C000"},
		{"00000000000000000000000000000000", "0000000004000000", "100100014C00E000"},
		{"00000000000000000000000000000000", "0000000002000000", "8801000126007000"},
		{"00000000000000000000000000000000", "0000000001000000", "4401000193003800"},
		{"00000000000000000000000000000000", "0000000000800000", "22010001C9801C00"},
		{"00000000000000000000000000000000", "0000000000400000", "11010001E4C00E00"},
		{"00000000000000000000000000000000", "0000000000200000", "08810001F2600700"},
		{"00000000000000000000000000000000", "0000000000100000", "04410001F9300380"},
		{"00000000000000000000000000000000", "0000000000080000", "02210001FC9801C0"},
		{"00000000000000000000000000000000", "0000000000040000", "01110001FE4C00E0"},
		{"00000000000000000000000000000000", "0000000000020000", "00890001FF260070"},
		{"00000000000000000000000000000000", "0000000000010000", "FFC3FF910055FF9A"},
		{"00000000000000000000000000000000", "0000000000008000", "0001000100008000"},
		{"00000000000000000000000000000000", "0000000000004000", "0001000180004000"},
		{"00000000000000000000000000000000", "0000000000002000", "00018001C0002000"},
		{"00000000000000000000000000000000", "0000000000001000", "8001C001E0009000"},
		{"00000000000000000000000000000000", "0000000000000800", "400160017000C800"},
		{"00000000000000000000000000000000", "0000000000000400", "2001B00138006400"},
		{"00000000000000000000000000000000", "0000000000000200", "900158011C003200"},
		{"00000000000000000000000000000000", "0000000000000100", "C8012C018E009900"},
		{"00000000000000000000000000000000", "0000000000000080", "640116014700CC80"},
		{"00000000000000000000000000000000", "0000000000000040", "32010B012380E640"},
		{"00000000000000000000000000000000", "0000000000000020", "1901058111C0F320"},
		{"00000000000000000000000000000000", "0000000000000010", "0C8102C108E0F990"},
		{"00000000000000000000000000000000", "0000000000000008", "064101610470FCC8"},
		{"00000000000000000000000000000000", "0000000000000004", "032100B10238FE64"},
		{"00000000000000000000000000000000", "0000000000000002", "01910059011CFF32"},
		{"00000000000000000000000000000000", "0000000000000001", "0013FFF500120009"},
		{"00000000000000000000000000000000", "0000000000000000", "0001000100000000"},
		{"01010101010101010101010101010101", "0101010101010101", "E3F8AFF7A3795615"},
		{"02020202020202020202020202020202", "0202020202020202", "93D8C66F869189B9"},
		{"03030303030303030303030303030303", "0303030303030303", "4FDD60E349A14800"},
		{"04040404040404040404040404040404", "0404040404040404", "3D6F7A5EA045E71D"},
		{"05050505050505050505050505050505", "0505050505050505", "13E677DB2AA285D9"},
		{"06060606060606060606060606060606", "0606060606060606", "FFD372957FD5C0D9"},
		{"07070707070707070707070707070707", "0707070707070707", "B970BCAB7DFE96A7"},
		{"08080808080808080808080808080808", "0808080808080808", "283D4E47A1205BD5"},
		{"09090909090909090909090909090909", "0909090909090909", "C060DF27CF6E0A0B"},
		{"0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A", "0A0A0A0A0A0A0A0A", "7D0FF198726E70F1"},
		{"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B", "0B0B0B0B0B0B0B0B", "A419A578F1D35F3E"},
		{"0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C", "0C0C0C0C0C0C0C0C", "AAD7E526E2580A70"},
		{"0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D", "0D0D0D0D0D0D0D0D", "B2FD0FD4BE361D56"},
		{"0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E", "0E0E0E0E0E0E0E0E", "40271EADEA9F08E1"},
		{"0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F", "0F0F0F0F0F0F0F0F", "C0B13DF73B24F9B3"},
		{"10101010101010101010101010101010", "1010101010101010", "EDC563FC23412235"},
		{"11111111111111111111111111111111", "1111111111111111", "F8194B3C1D1A3F25"},
		{"12121212121212121212121212121212", "1212121212121212", "DCCA55E8CF03811B"},
		{"13131313131313131313131313131313", "1313131313131313", "1B06557F2ED07402"},
		{"14141414141414141414141414141414", "1414141414141414", "632C42E8C0165B09"},
		{"15151515151515151515151515151515", "1515151515151515", "323DD1A1C7777F34"},
		{"16161616161616161616161616161616", "1616161616161616", "C8C6A8DB3980216C"},
		{"17171717171717171717171717171717", "1717171717171717", "495D68C77DF651B6"},
		{"18181818181818181818181818181818", "1818181818181818", "6C636E9F4C2D021C"},
		{"19191919191919191919191919191919", "1919191919191919", "F063DD2D7B2DA27D"},
		{"1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A", "1A1A1A1A1A1A1A1A", "14A167F2DB0CCB84"},
		{"1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B", "1B1B1B1B1B1B1B1B", "B83ED64A8E54E188"},
		{"1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C", "1C1C1C1C1C1C1C1C", "B4F70A34ED98D459"},
		{"1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D", "1D1D1D1D1D1D1D1D", "B75C1D5BD1385C23"},
		{"1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E", "1E1E1E1E1E1E1E1E", "B6566547359F7494"},
		{"1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F", "1F1F1F1F1F1F1F1F", "36075B79EBBD781F"},
		{"20202020202020202020202020202020", "2020202020202020", "F61BA4A92716C708"},
		{"21212121212121212121212121212121", "2121212121212121", "B38CEB921E3823B3"},
		{"22222222222222222222222222222222", "2222222222222222", "3882C5E7D5F7E497"},
		{"23232323232323232323232323232323", "2323232323232323", "4503E89E7699A2D2"},
		{"24242424242424242424242424242424", "2424242424242424", "C67C5B150E05C359"},
		{"25252525252525252525252525252525", "2525252525252525", "C778EBF6C5BCF1E4"},
		{"26262626262626262626262626262626", "2626262626262626", "D19271386E913F79"},
		{"27272727272727272727272727272727", "2727272727272727", "702FC41EDD909983"},
		{"28282828282828282828282828282828", "2828282828282828", "04C48C7C5E666014"},
		{"29292929292929292929292929292929", "2929292929292929", "0B4A1ED365CCC041"},
		{"2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A", "2A2A2A2A2A2A2A2A", "2913C1A15A723869"},
		{"2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B", "2B2B2B2B2B2B2B2B", "19358420E3154505"},
		{"2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C", "2C2C2C2C2C2C2C2C", "9E50D09AE5654A29"},
		{"2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D", "2D2D2D2D2D2D2D2D", "0DD76C4E7B8D0767"},
		{"2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E", "2E2E2E2E2E2E2E2E", "43A609F25B31AA36"},
		{"2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F", "2F2F2F2F2F2F2F2F", "D2B772F1B1D8F60E"},
		{"30303030303030303030303030303030", "3030303030303030", "4EE30E9A0DF346B7"},
		{"31313131313131313131313131313131", "3131313131313131", "9D452826980EEAB8"},
		{"32323232323232323232323232323232", "3232323232323232", "8C3ECB2C44C73693"},
		{"33333333333333333333333333333333", "3333333333333333", "6577873EFE130F55"},
		{"34343434343434343434343434343434", "3434343434343434", "1A21753282234FC9"},
		{"35353535353535353535353535353535", "3535353535353535", "56AD2FFA40D78E0A"},
		{"36363636363636363636363636363636", "3636363636363636", "217B6ACAA3719AC7"},
		{"37373737373737373737373737373737", "3737373737373737", "54CF21E389D873EC"},
		{"38383838383838383838383838383838", "3838383838383838", "A8CDE7BC4367512A"},
		{"39393939393939393939393939393939", "3939393939393939", "49F75C2B44652962"},
		{"3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A", "3A3A3A3A3A3A3A3A", "4E779E4691B33241"},
		{"3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B", "3B3B3B3B3B3B3B3B", "96CD991A6034FE54"},
		{"3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C", "3C3C3C3C3C3C3C3C", "1B325A905570B714"},
		{"3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D", "3D3D3D3D3D3D3D3D", "675BE60635825668"},
		{"3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E", "3E3E3E3E3E3E3E3E", "F1F2D1C907270F5B"},
		{"3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F", "3F3F3F3F3F3F3F3F", "D1C024254E589BCE"},
		{"40404040404040404040404040404040", "4040404040404040", "91F5BE3409EAA3D8"},
		{"41414141414141414141414141414141", "4141414141414141", "B3BA47C38151C18B"},
		{"42424242424242424242424242424242", "4242424242424242", "AA9FA0785C870C00"},
		{"43434343434343434343434343434343", "4343434343434343", "331766E3A7605621"},
		{"44444444444444444444444444444444", "4444444444444444", "D6B31F8692EFF6AD"},
		{"45454545454545454545454545454545", "4545454545454545", "4A7744539850DA23"},
		{"46464646464646464646464646464646", "4646464646464646", "76F230AFBD9FB3A1"},
		{"47474747474747474747474747474747", "4747474747474747", "CF335C8F413F87C9"},
		{"48484848484848484848484848484848", "4848484848484848", "BC588B35169D9581"},
		{"49494949494949494949494949494949", "4949494949494949", "21C5AD9B26CA5E87"},
		{"4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A", "4A4A4A4A4A4A4A4A", "759A3CB94586706D"},
		{"4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B", "4B4B4B4B4B4B4B4B", "EA0CF5F73BDC42C7"},
		{"4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C", "4C4C4C4C4C4C4C4C", "A66EC4B0FA5714CE"},
		{"4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D", "4D4D4D4D4D4D4D4D", "4EA3936FC4C7433A"},
		{"4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E", "4E4E4E4E4E4E4E4E", "9D3CCF22E3D48B51"},
		{"4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F", "4F4F4F4F4F4F4F4F", "02208851EB243D8F"},
		{"50505050505050505050505050505050", "5050505050505050", "40907FA634D3C33B"},
		{"51515151515151515151515151515151", "5151515151515151", "90D3C0F4764A6686"},
		{"52525252525252525252525252525252", "5252525252525252", "CC9F85AD1D6DF10D"},
		{"53535353535353535353535353535353", "5353535353535353", "E9A8BEFEDD40E726"},
		{"54545454545454545454545454545454", "5454545454545454", "22B467DA086BF29E"},
		{"55555555555555555555555555555555", "5555555555555555", "2F5D2595487A3B81"},
		{"56565656565656565656565656565656", "5656565656565656", "ADB111A4096683D2"},
		{"57575757575757575757575757575757", "5757575757575757", "85524D410EB428AE"},
		{"58585858585858585858585858585858", "5858585858585858", "E289DC71F9A118C8"},
		{"59595959595959595959595959595959", "5959595959595959", "E97FF43A99D9F716"},
		{"5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A", "5A5A5A5A5A5A5A5A", "BC20B7E259F2F3AA"},
		{"5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B", "5B5B5B5B5B5B5B5B", "AE2806A285E2586C"},
		{"5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C", "5C5C5C5C5C5C5C5C", "779A2AE1ACE570AF"},
		{"5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D", "5D5D5D5D5D5D5D5D", "B4A3C6BFDCFA70F7"},
		{"5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E", "5E5E5E5E5E5E5E5E", "08CD767CC9C0FB8B"},
		{"5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F", "5F5F5F5F5F5F5F5F", "913BB1B547D3B685"},
		{"60606060606060606060606060606060", "6060606060606060", "591609401E61498D"},
		{"61616161616161616161616161616161", "6161616161616161", "F3A9C717A71866E0"},
		{"62626262626262626262626262626262", "6262626262626262", "334D792F2030928C"},
		{"63636363636363636363636363636363", "6363636363636363", "361A22B36A9FC1F4"},
		{"64646464646464646464646464646464", "6464646464646464", "ACF3C17C427A15E3"},
		{"65656565656565656565656565656565", "6565656565656565", "FBAFEF7504A4DDDD"},
		{"66666666666666666666666666666666", "6666666666666666", "C85D4510F4EB4595"},
		{"67676767676767676767676767676767", "6767676767676767", "B9D48C0FAD2B9BA6"},
		{"68686868686868686868686868686868", "6868686868686868", "55D9CF94054DE513"},
		{"69696969696969696969696969696969", "6969696969696969", "786FFD4D4B2CA270"},
		{"6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A", "6A6A6A6A6A6A6A6A", "1D20C40C017B4904"},
		{"6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B", "6B6B6B6B6B6B6B6B", "32ACF7B6774ABCDA"},
		{"6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C", "6C6C6C6C6C6C6C6C", "3181D1AAD37FA973"},
		{"6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D", "6D6D6D6D6D6D6D6D", "C9F9908D5CA14756"},
		{"6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E", "6E6E6E6E6E6E6E6E", "160A9167191C456D"},
		{"6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F", "6F6F6F6F6F6F6F6F", "C78D5E55EB420025"},
		{"70707070707070707070707070707070", "7070707070707070", "4C9B181D763BA5CA"},
		{"71717171717171717171717171717171", "7171717171717171", "4D770F1FAFBF0019"},
		{"72727272727272727272727272727272", "7272727272727272", "06D8DB7EA4005740"},
		{"73737373737373737373737373737373", "7373737373737373", "C2D860A18D35F087"},
		{"74747474747474747474747474747474", "7474747474747474", "8EAA96DEA6C96E48"},
		{"75757575757575757575757575757575", "7575757575757575", "C8C4A259C23B150A"},
		{"76767676767676767676767676767676", "7676767676767676", "A173911B63612FAA"},
		{"77777777777777777777777777777777", "7777777777777777", "D2E486D93304B9B6"},
		{"78787878787878787878787878787878", "7878787878787878", "2A09B1A6AAC15F8B"},
		{"79797979797979797979797979797979", "7979797979797979", "F100C0B06D024164"},
		{"7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A", "7A7A7A7A7A7A7A7A", "D0FDD5472D17DF6B"},
		{"7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B", "7B7B7B7B7B7B7B7B", "8FA5B5BF9189D86B"},
		{"7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C", "7C7C7C7C7C7C7C7C", "EDF27A5306AE1FD0"},
		{"7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D", "7D7D7D7D7D7D7D7D", "CFF5571D3DBB2269"},
		{"7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E", "7E7E7E7E7E7E7E7E", "C75EBEF66F852B22"},
		{"7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F", "7F7F7F7F7F7F7F7F", "F19AC2DFA999A3DD"},
		{"80808080808080808080808080808080", "8080808080808080", "526B1919A9F5085A"},
		{"81818181818181818181818181818181", "8181818181818181", "4C700A84066F8552"},
		{"82828282828282828282828282828282", "8282828282828282", "C80D35576D9EF4B2"},
		{"83838383838383838383838383838383", "8383838383838383", "1E70D258BE1B9335"},
		{"84848484848484848484848484848484", "8484848484848484", "48B6C4698ABF63D4"},
		{"85858585858585858585858585858585", "8585858585858585", "2B2A0CF7447398DD"},
		{"86868686868686868686868686868686", "8686868686868686", "1497261B49A00D15"},
		{"87878787878787878787878787878787", "8787878787878787", "34293936608264F0"},
		{"88888888888888888888888888888888", "8888888888888888", "DE0FB2981307603C"},
		{"89898989898989898989898989898989", "8989898989898989", "53E12A0E076E4E7B"},
		{"8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A", "8A8A8A8A8A8A8A8A", "FA1524CBFDA6B5CD"},
		{"8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B", "8B8B8B8B8B8B8B8B", "B3B5DAEA96ACF66F"},
		{"8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C", "8C8C8C8C8C8C8C8C", "87BE37BABA89A892"},
		{"8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D", "8D8D8D8D8D8D8D8D", "B68394FE788167C5"},
		{"8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E", "8E8E8E8E8E8E8E8E", "99D305C9934F4CDC"},
		{"8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F", "8F8F8F8F8F8F8F8F", "9149B572B9491FF7"},
		{"90909090909090909090909090909090", "9090909090909090", "753B35B791741626"},
		{"91919191919191919191919191919191", "9191919191919191", "4DD72550F1C5454A"},
		{"92929292929292929292929292929292", "9292929292929292", "D753560558AC0AA7"},
		{"93939393939393939393939393939393", "9393939393939393", "606FF1D92F089562"},
		{"94949494949494949494949494949494", "9494949494949494", "92459BD1A161D30D"},
		{"95959595959595959595959595959595", "9595959595959595", "6351C0143049BEFF"},
		{"96969696969696969696969696969696", "9696969696969696", "FE161083DA8DA3B4"},
		{"97979797979797979797979797979797", "9797979797979797", "09E78EEFE5663072"},
		{"98989898989898989898989898989898", "9898989898989898", "FE7EDEF1C82F47AE"},
		{"99999999999999999999999999999999", "9999999999999999", "E0DEFDEA63EDE361"},
		{"9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A", "9A9A9A9A9A9A9A9A", "D9DCA3371D6BC5F9"},
		{"9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B", "9B9B9B9B9B9B9B9B", "60888B004AD881FB"},
		{"9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C", "9C9C9C9C9C9C9C9C", "A38D5ED8AA81AE55"},
		{"9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D", "9D9D9D9D9D9D9D9D", "339D10588A064416"},
		{"9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E", "9E9E9E9E9E9E9E9E", "899224DDCFD99A36"},
		{"9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F", "9F9F9F9F9F9F9F9F", "590377175A85AF06"},
		{"A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0", "A0A0A0A0A0A0A0A0", "D2CFEB2C8E3FFA42"},
		{"A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1", "A1A1A1A1A1A1A1A1", "B9AC86C539B64764"},
		{"A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2", "A2A2A2A2A2A2A2A2", "5671F045487CF2CA"},
		{"A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3", "A3A3A3A3A3A3A3A3", "E1F4E8F6B5C9E251"},
		{"A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4", "A4A4A4A4A4A4A4A4", "ACFC1F6E15C26417"},
		{"A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5", "A5A5A5A5A5A5A5A5", "CF3CFD0FCC41060F"},
		{"A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6", "A6A6A6A6A6A6A6A6", "77CD999CDAA028C8"},
		{"A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7", "A7A7A7A7A7A7A7A7", "C87793AC3EB54812"},
		{"A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8", "A8A8A8A8A8A8A8A8", "4B64309F64A4C30E"},
		{"A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9", "A9A9A9A9A9A9A9A9", "69EF3527C32DE353"},
		{"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", "AAAAAAAAAAAAAAAA", "0B3C3AA30E09B216"},
		{"ABABABABABABABABABABABABABABABAB", "ABABABABABABABAB", "2747CE4AE850B89E"},
		{"ACACACACACACACACACACACACACACACAC", "ACACACACACACACAC", "2BF63DCD10EA41A2"},
		{"ADADADADADADADADADADADADADADADAD", "ADADADADADADADAD", "D11EAE8BDD2A31DF"},
		{"AEAEAEAEAEAEAEAEAEAEAEAEAEAEAEAE", "AEAEAEAEAEAEAEAE", "DB8341FDE6963165"},
		{"AFAFAFAFAFAFAFAFAFAFAFAFAFAFAFAF", "AFAFAFAFAFAFAFAF", "42E985FBBA5E26F0"},
		{"B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0", "B0B0B0B0B0B0B0B0", "C1FA020940DF634B"},
		{"B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1", "B1B1B1B1B1B1B1B1", "DA7CFBC1CB8D45F7"},
		{"B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2", "B2B2B2B2B2B2B2B2", "83B3CD2CB214516C"},
		{"B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3", "B3B3B3B3B3B3B3B3", "26D5100CB4EBB6A0"},
		{"B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4", "B4B4B4B4B4B4B4B4", "047AE61C30FA9A5C"},
		{"B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5", "B5B5B5B5B5B5B5B5", "066FC02236598051"},
		{"B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6", "B6B6B6B6B6B6B6B6", "440CCB69E640302B"},
		{"B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7", "B7B7B7B7B7B7B7B7", "56A57125B8516A75"},
		{"B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8", "B8B8B8B8B8B8B8B8", "A653F6A73B3E6B5E"},
		{"B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9", "B9B9B9B9B9B9B9B9", "4B5384F3ADA500F1"},
		{"BABABABABABABABABABABABABABABABA", "BABABABABABABABA", "54E7A8FEDA7C0DF0"},
		{"BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB", "BBBBBBBBBBBBBBBB", "29D79357B5726EFC"},
		{"BCBCBCBCBCBCBCBCBCBCBCBCBCBCBCBC", "BCBCBCBCBCBCBCBC", "93057429D244C88B"},
		{"BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD", "BDBDBDBDBDBDBDBD", "B36F0E7B57A5574B"},
		{"BEBEBEBEBEBEBEBEBEBEBEBEBEBEBEBE", "BEBEBEBEBEBEBEBE", "18430FD744EBE618"},
		{"BFBFBFBFBFBFBFBFBFBFBFBFBFBFBFBF", "BFBFBFBFBFBFBFBF", "3736F16946F6DFD2"},
		{"C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0", "C0C0C0C0C0C0C0C0", "272AAA5C14BC3AA6"},
		{"C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1", "C1C1C1C1C1C1C1C1", "BA96DAD61640FEB6"},
		{"C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2", "C2C2C2C2C2C2C2C2", "274D4ACBE4ED5E65"},
		{"C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3", "C3C3C3C3C3C3C3C3", "8028A6C19BFB4585"},
		{"C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4", "C4C4C4C4C4C4C4C4", "392123DB00472530"},
		{"C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5", "C5C5C5C5C5C5C5C5", "E2CB0F8C53085E40"},
		{"C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6", "C6C6C6C6C6C6C6C6", "1043137A9370CC3E"},
		{"C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7", "C7C7C7C7C7C7C7C7", "36EF8AF8599F9304"},
		{"C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8", "C8C8C8C8C8C8C8C8", "EC8549DFF44C09C1"},
		{"C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9", "C9C9C9C9C9C9C9C9", "78A1D8E55D8990DF"},
		{"CACACACACACACACACACACACACACACACA", "CACACACACACACACA", "2E0F4A961FB95229"},
		{"CBCBCBCBCBCBCBCBCBCBCBCBCBCBCBCB", "CBCBCBCBCBCBCBCB", "2EA46D330EA675D4"},
		{"CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", "CCCCCCCCCCCCCCCC", "AB6E3B2C903B0719"},
		{"CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD", "CDCDCDCDCDCDCDCD", "4627BECC1050E51C"},
		{"CECECECECECECECECECECECECECECECE", "CECECECECECECECE", "A1CDE06D6DEF1598"},
		{"CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF", "CFCFCFCFCFCFCFCF", "EDC2E88F4ABF46F5"},
		{"D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0", "D0D0D0D0D0D0D0D0", "AB0921F002272D9A"},
		{"D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1", "D1D1D1D1D1D1D1D1", "EE89ABBA0F5FC976"},
		{"D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2", "D2D2D2D2D2D2D2D2", "1DDC4FD0C5F087BE"},
		{"D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3", "D3D3D3D3D3D3D3D3", "721DB8CA349FE6B8"},
		{"D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4", "D4D4D4D4D4D4D4D4", "C7A120CB6DA70346"},
		{"D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5", "D5D5D5D5D5D5D5D5", "75F7C7005EA47839"},
		{"D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6", "D6D6D6D6D6D6D6D6", "933BF0248F4B22E3"},
		{"D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7", "D7D7D7D7D7D7D7D7", "F07105D5EEFFD3EC"},
		{"D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8", "D8D8D8D8D8D8D8D8", "77C484A7F07AF3BB"},
		{"D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9", "D9D9D9D9D9D9D9D9", "BE63CEBE5679A85C"},
		{"DADADADADADADADADADADADADADADADA", "DADADADADADADADA", "E65D8B23E23C4E6D"},
		{"DBDBDBDBDBDBDBDBDBDBDBDBDBDBDBDB", "DBDBDBDBDBDBDBDB", "4B0D4AD2EBC84EFF"},
		{"DCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDC", "DCDCDCDCDCDCDCDC", "EBD6A238089D647A"},
		{"DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD", "DDDDDDDDDDDDDDDD", "EA689855ABC54BE9"},
		{"DEDEDEDEDEDEDEDEDEDEDEDEDEDEDEDE", "DEDEDEDEDEDEDEDE", "238296946AAE639C"},
		{"DFDFDFDFDFDFDFDFDFDFDFDFDFDFDFDF", "DFDFDFDFDFDFDFDF", "085514DED80D6D11"},
		{"E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0", "E0E0E0E0E0E0E0E0", "4C9FAF546BB09F7D"},
		{"E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1", "E1E1E1E1E1E1E1E1", "C28E6EE5BF3FA2E1"},
		{"E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2", "E2E2E2E2E2E2E2E2", "FBB2548BC3DCFB4C"},
		{"E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3", "E3E3E3E3E3E3E3E3", "56B7AA03853E734F"},
		{"E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4", "E4E4E4E4E4E4E4E4", "70B0F380ADEA55A3"},
		{"E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5", "E5E5E5E5E5E5E5E5", "15E7182235A91BE9"},
		{"E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6", "E6E6E6E6E6E6E6E6", "642960CD5A9C0326"},
		{"E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7", "E7E7E7E7E7E7E7E7", "0EF506AB845D4183"},
		{"E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8", "E8E8E8E8E8E8E8E8", "CF1E13F825C72D69"},
		{"E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9", "E9E9E9E9E9E9E9E9", "95C30EF3C983EE5A"},
		{"EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA", "EAEAEAEAEAEAEAEA", "450DA77D932EA463"},
		{"EBEBEBEBEBEBEBEBEBEBEBEBEBEBEBEB", "EBEBEBEBEBEBEBEB", "541F51337C3C01C6"},
		{"ECECECECECECECECECECECECECECECEC", "ECECECECECECECEC", "0A22773B4A2E642F"},
		{"EDEDEDEDEDEDEDEDEDEDEDEDEDEDEDED", "EDEDEDEDEDEDEDED", "8E693227B920CDF5"},
		{"EEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE", "EEEEEEEEEEEEEEEE", "3FE734B14D289987"},
		{"EFEFEFEFEFEFEFEFEFEFEFEFEFEFEFEF", "EFEFEFEFEFEFEFEF", "189508CEFB37C933"},
		{"F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0", "F0F0F0F0F0F0F0F0", "1AAFF239FEFA4F4F"},
		{"F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1", "F1F1F1F1F1F1F1F1", "5475CB058EEC4990"},
		{"F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2", "F2F2F2F2F2F2F2F2", "E9E66F99002B96C7"},
		{"F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3", "F3F3F3F3F3F3F3F3", "C67076D06A3C184C"},
		{"F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4", "F4F4F4F4F4F4F4F4", "3705AEE3D95EF7D6"},
		{"F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5", "F5F5F5F5F5F5F5F5", "9E8B984DD82C4061"},
		{"F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6", "F6F6F6F6F6F6F6F6", "4738C2BE9CDD7A95"},
		{"F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7", "F7F7F7F7F7F7F7F7", "8E13C368F53E55AF"},
		{"F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8", "F8F8F8F8F8F8F8F8", "69269091D1F6D10F"},
		{"F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9", "F9F9F9F9F9F9F9F9", "D039CDC8F806D0DB"},
		{"FAFAFAFAFAFAFAFAFAFAFAFAFAFAFAFA", "FAFAFAFAFAFAFAFA", "52009461E4F37F47"},
		{"FBFBFBFBFBFBFBFBFBFBFBFBFBFBFBFB", "FBFBFBFBFBFBFBFB", "08B02B3F76889032"},
		{"FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC", "FCFCFCFCFCFCFCFC", "77E9D9405D35F1CC"},
		{"FDFDFDFDFDFDFDFDFDFDFDFDFDFDFDFD", "FDFDFDFDFDFDFDFD", "1A10E3F318AEDA9A"},
		{"FEFEFEFEFEFEFEFEFEFEFEFEFEFEFEFE", "FEFEFEFEFEFEFEFE", "DC6D99012DC01E0B"},
		{"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF", "FFFFFFFFFFFFFFFF", "CD1AB2C1211041FB"},
		{"000102030405060708090A0B0C0D0E0F", "0011223344556677", "F526AB9A62C0D258"},
		{"2BD6459F82C5B300952C49104881FF48", "EA024714AD5C4D84", "C8FB51D3516627A8"},
		{"80000000000000000000000000000000", "78071EE87F0130E8", "0000000000000000"},
		{"40000000000000000000000000000000", "98AA167965B52792", "0000000000000000"},
		{"20000000000000000000000000000000", "20BC104058293C2D", "0000000000000000"},
		{"10000000000000000000000000000000", "6050001FFA1A0219", "0000000000000000"},
		{"08000000000000000000000000000000", "51D07EAB030C7BA3", "0000000000000000"},
		{"04000000000000000000000000000000", "5C2DB4E17368C7B4", "0000000000000000"},
		{"02000000000000000000000000000000", "702AD34DB37C6E58", "0000000000000000"},
		{"01000000000000000000000000000000", "82997FEDEB622DAC", "0000000000000000"},
		{"00800000000000000000000000000000", "00F29B90EF196F2E", "0000000000000000"},
		{"00400000000000000000000000000000", "50156B3387228984", "0000000000000000"},
		{"00200000000000000000000000000000", "0F41E41CF1E31023", "0000000000000000"},
		{"00100000000000000000000000000000", "424F1393C36C787D", "0000000000000000"},
		{"00080000000000000000000000000000", "AEC2F7DF4625D267", "0000000000000000"},
		{"00040000000000000000000000000000", "1DCF880D88A40CC0", "0000000000000000"},
		{"00020000000000000000000000000000", "7B4711086141DDDF", "0000000000000000"},
		{"00010000000000000000000000000000", "B505D373FBF27C39", "0000000000000000"},
		{"00008000000000000000000000000000", "14D47C44835EEB99", "0000000000000000"},
		{"00004000000000000000000000000000", "E5A7E81CECC9B474", "0000000000000000"},
		{"00002000000000000000000000000000", "3A11C283DA6F60C9", "0000000000000000"},
		{"00001000000000000000000000000000", "71EAE2049E1887EE", "0000000000000000"},
		{"00000800000000000000000000000000", "9B437A81E33694D5", "0000000000000000"},
		{"00000400000000000000000000000000", "27A1BA81E070A4EB", "0000000000000000"},
		{"00000200000000000000000000000000", "E9173EF408C4DDB7", "0000000000000000"},
		{"00000100000000000000000000000000", "01834E54CE64F9CE", "0000000000000000"},
		{"00000080000000000000000000000000", "7BB7655098FA86AD", "0000000000000000"},
		{"00000040000000000000000000000000", "BC42A5F19507F0D9", "0000000000000000"},
		{"00000020000000000000000000000000", "72473C47FE650F94", "0000000000000000"},
		{"00000010000000000000000000000000", "415D572E8B0C411A", "0000000000000000"},
		{"00000008000000000000000000000000", "937FC76683BA171A", "0000000000000000"},
		{"00000004000000000000000000000000", "412689C8C42CE82A", "0000000000000000"},
		{"00000002000000000000000000000000", "668590908BC18834", "0000000000000000"},
		{"00000001000000000000000000000000", "ACDFB787978B4DC2", "0000000000000000"},
		{"00000000800000000000000000000000", "CEA444C8CE44C2C2", "0000000000000000"},
		{"00000000400000000000000000000000", "F8F9A74F773AF827", "0000000000000000"},
		{"00000000200000000000000000000000", "FF3000AF4087BF1F", "0000000000000000"},
		{"00000000100000000000000000000000", "4F1D2F78E00ED078", "0000000000000000"},
		{"00000000080000000000000000000000", "020C060173F57DF3", "0000000000000000"},
		{"00000000040000000000000000000000", "C11DA87BD6174883", "0000000000000000"},
		{"00000000020000000000000000000000", "D73CCB416EC3923A", "0000000000000000"},
		{"00000000010000000000000000000000", "1D8A3C51E599BDC0", "0000000000000000"},
		{"00000000008000000000000000000000", "4F78A0B0420F1F5E", "0000000000000000"},
		{"00000000004000000000000000000000", "9F0F377C3EBC7823", "0000000000000000"},
		{"00000000002000000000000000000000", "CFD6AFAB61ED7408", "0000000000000000"},
		{"00000000001000000000000000000000", "F764E0629C53720D", "0000000000000000"},
		{"00000000000800000000000000000000", "F596DF8B7E63100C", "0000000000000000"},
		{"00000000000400000000000000000000", "CEA11289498D2F95", "0000000000000000"},
		{"00000000000200000000000000000000", "8569B4195F87B731", "0000000000000000"},
		{"00000000000100000000000000000000", "2F39C8595258C9A1", "0000000000000000"},
		{"00000000000080000000000000000000", "B0BB70B7E277E269", "0000000000000000"},
		{"00000000000040000000000000000000", "9FC141F1A64190C1", "0000000000000000"},
		{"00000000000020000000000000000000", "C8D5BF3D72650600", "0000000000000000"},
		{"00000000000010000000000000000000", "30FAE75F0CCE0174", "0000000000000000"},
		{"00000000000008000000000000000000", "102260B390107E64", "0000000000000000"},
		{"00000000000004000000000000000000", "90D3477061F8FB40", "0000000000000000"},
		{"00000000000002000000000000000000", "FD4FBB0AD84300A0", "0000000000000000"},
		{"00000000000001000000000000000000", "B07FFD863EDC3B04", "0000000000000000"},
		{"00000000000000800000000000000000", "9429E1C4642F1250", "0000000000000000"},
		{"00000000000000400000000000000000", "9DA0130BEFF2EE23", "0000000000000000"},
		{"00000000000000200000000000000000", "3CFC0000F7FE3198", "0000000000000000"},
		{"00000000000000100000000000000000", "DC2BBBBB956C9720", "0000000000000000"},
		{"00000000000000080000000000000000", "5CF6CDB2E90CC73A", "0000000000000000"},
		{"00000000000000040000000000000000", "460640F5067CA894", "0000000000000000"},
		{"00000000000000020000000000000000", "9AC3397BB3C0E08D", "0000000000000000"},
		{"00000000000000010000000000000000", "5DFF62FF39E86E59", "0000000000000000"},
		{"00000000000000008000000000000000", "D03652EFAEBDE7C4", "0000000000000000"},
		{"00000000000000004000000000000000", "C975419434DC59E9", "0000000000000000"},
		{"00000000000000002000000000000000", "08C2F2A853E65B62", "0000000000000000"},
		{"00000000000000001000000000000000", "D06933570ADF1770", "0000000000000000"},
		{"00000000000000000800000000000000", "982DAE603BD972B6", "0000000000000000"},
		{"00000000000000000400000000000000", "82D1C2079912E6AB", "0000000000000000"},
		{"00000000000000000200000000000000", "5BD0A6C11A41C763", "0000000000000000"},
		{"00000000000000000100000000000000", "6CD4381A523716F0", "0000000000000000"},
		{"00000000000000000080000000000000", "6A9EF2F77DE21D8E", "0000000000000000"},
		{"00000000000000000040000000000000", "0553F2DA029CF102", "0000000000000000"},
		{"00000000000000000020000000000000", "2CE768E22ED768B8", "0000000000000000"},
		{"00000000000000000010000000000000", "F085E604E26C0AB4", "0000000000000000"},
		{"00000000000000000008000000000000", "966B11EF02176A6E", "0000000000000000"},
		{"00000000000000000004000000000000", "E1BD49B0C0534CEE", "0000000000000000"},
		{"00000000000000000002000000000000", "EB208CCAC317A132", "0000000000000000"},
		{"00000000000000000001000000000000", "03797C90BF07C736", "0000000000000000"},
		{"00000000000000000000800000000000", "9640966A738A706C", "0000000000000000"},
		{"00000000000000000000400000000000", "CE989DD00307AD8B", "0000000000000000"},
		{"00000000000000000000200000000000", "DAA2C4BE43D75BB7", "0000000000000000"},
		{"00000000000000000000100000000000", "91B53F1BDF3AFBD6", "0000000000000000"},
		{"00000000000000000000080000000000", "90740A948E9E3B93", "0000000000000000"},
		{"00000000000000000000040000000000", "67169672C0CAF670", "0000000000000000"},
		{"00000000000000000000020000000000", "2F123F48482C5F12", "0000000000000000"},
		{"00000000000000000000010000000000", "0ADAD627F50721FB", "0000000000000000"},
		{"00000000000000000000008000000000", "DB6E9E3B7612560A", "0000000000000000"},
		{"00000000000000000000004000000000", "D721D05AEFCAA18E", "0000000000000000"},
		{"00000000000000000000002000000000", "B4A5E438241D67A6", "0000000000000000"},
		{"00000000000000000000001000000000", "7B44667D1E796531", "0000000000000000"},
		{"00000000000000000000000800000000", "E698BE39AEA13C79", "0000000000000000"},
		{"00000000000000000000000400000000", "FAA7517DBF38AB35", "0000000000000000"},
		{"00000000000000000000000200000000", "106EEC176ECE4259", "0000000000000000"},
		{"00000000000000000000000100000000", "8FA2D68B88D85E2B", "0000000000000000"},
		{"00000000000000000000000080000000", "E3C1ED0EFC4F6F38", "0000000000000000"},
		{"00000000000000000000000040000000", "7C4D85F6B673184C", "0000000000000000"},
		{"00000000000000000000000020000000", "75AD06E940CB6ECB", "0000000000000000"},
		{"00000000000000000000000010000000", "F6D58769DC0F7D8B", "0000000000000000"},
		{"00000000000000000000000008000000", "36B001BDD4918EEF", "0000000000000000"},
		{"00000000000000000000000004000000", "89F000DDDA93B90F", "0000000000000000"},
		{"00000000000000000000000002000000", "6AF0006DC391471F", "0000000000000000"},
		{"00000000000000000000000001000000", "6EE08035B72BF227", "0000000000000000"},
		{"00000000000000000000000000800000", "DE18C019DDB10DCB", "0000000000000000"},
		{"00000000000000000000000000400000", "0579E00B945ED0B2", "0000000000000000"},
		{"00000000000000000000000000200000", "E2D5F6E83FBBE35D", "0000000000000000"},
		{"00000000000000000000000000100000", "CBE37B75A281F1AE", "0000000000000000"},
		{"00000000000000000000000000080000", "F0903DB58BEFF8CF", "0000000000000000"},
		{"00000000000000000000000000040000", "71519CE10AB17A10", "0000000000000000"},
		{"00000000000000000000000000020000", "5C694E7137F93D08", "0000000000000000"},
		{"00000000000000000000000000010000", "ADB5A739EE1D1E84", "0000000000000000"},
		{"00000000000000000000000000008000", "6CDBD39D32CE0F42", "0000000000000000"},
		{"00000000000000000000000000004000", "364AE9D1995D07C1", "0000000000000000"},
		{"00000000000000000000000000002000", "9B3774E9CCB083E0", "0000000000000000"},
		{"00000000000000000000000000001000", "CC004681FE204800", "0000000000000000"},
		{"00000000000000000000000000000800", "25012041F810E200", "0000000000000000"},
		{"00000000000000000000000000000400", "CF81092125081100", "0000000000000000"},
		{"00000000000000000000000000000200", "E4C1019111840880", "0000000000000000"},
		{"00000000000000000000000000000100", "E76105C915C20C40", "0000000000000000"},
		{"00000000000000000000000000000080", "F4BB07E11A0B0E1A", "0000000000000000"},
		{"00000000000000000000000000000040", "F75986F389F08110", "0000000000000000"},
		{"00000000000000000000000000000020", "F9A7466C4600406A", "0000000000000000"},
		{"00000000000000000000000000000010", "F9EF2643265C205B", "0000000000000000"},
		{"00000000000000000000000000000008", "C1DF2DDB127CDFE1", "0000000000000000"},
		{"00000000000000000000000000000004", "8A1495E062113BE9", "0000000000000000"},
		{"00000000000000000000000000000002", "8A2AD21859D1E3E1", "0000000000000000"},
		{"00000000000000000000000000000001", "F61D37EC1C099DE5", "0000000000000000"},
		{"00000000000000000000000000000000", "8001000180008000", "8000000000000000"},
		{"00000000000000000000000000000000", "C00180014000C000", "4000000000000000"},
		{"00000000000000000000000000000000", "6001C00120006000", "2000000000000000"},
		{"00000000000000000000000000000000", "3001600190003000", "1000000000000000"},
		{"00000000000000000000000000000000", "18013001C8009800", "0800000000000000"},
		{"00000000000000000000000000000000", "8C01180164004C00", "0400000000000000"},
		{"00000000000000000000000000000000", "46018C0132002600", "0200000000000000"},
		{"00000000000000000000000000000000", "2301C60199009300", "0100000000000000"},
		{"00000000000000000000000000000000", "9181E3014C80C980", "0080000000000000"},
		{"00000000000000000000000000000000", "C8C1F1812640E4C0", "0040000000000000"},
		{"00000000000000000000000000000000", "E461F8C11320F260", "0020000000000000"},
		{"00000000000000000000000000000000", "F231FC610990F930", "0010000000000000"},
		{"00000000000000000000000000000000", "F919FE3104C8FC98", "0008000000000000"},
		{"00000000000000000000000000000000", "FC8DFF190264FE4C", "0004000000000000"},
		{"00000000000000000000000000000000", "FE47FF8D0132FF26", "0002000000000000"},
		{"00000000000000000000000000000000", "0028FFF5003F0013", "0001000000000000"},
		{"00000000000000000000000000000000", "0001800180008000", "0000800000000000"},
		{"00000000000000000000000000000000", "00014001C0004000", "0000400000000000"},
		{"00000000000000000000000000000000", "8001A0016000A000", "0000200000000000"},
		{"00000000000000000000000000000000", "4001D00130005000", "0000100000000000"},
		{"00000000000000000000000000000000", "2001E8019800A800", "0000080000000000"},
		{"00000000000000000000000000000000", "900174014C005400", "0000040000000000"},
		{"00000000000000000000000000000000", "48013A012600AA00", "0000020000000000"},
		{"00000000000000000000000000000000", "A4011D019300D500", "0000010000000000"},
		{"00000000000000000000000000000000", "D2010E81C980EA80", "0000008000000000"},
		{"00000000000000000000000000000000", "E9010741E4C0F540", "0000004000000000"},
		{"00000000000000000000000000000000", "F48103A1F260FAA0", "0000002000000000"},
		{"00000000000000000000000000000000", "FA4101D1F930FD50", "0000001000000000"},
		{"00000000000000000000000000000000", "FD2100E9FC98FEA8", "0000000800000000"},
		{"00000000000000000000000000000000", "FE910075FE4CFF54", "0000000400000000"},
		{"00000000000000000000000000000000", "FF49003BFF26FFAA", "0000000200000000"},
		{"00000000000000000000000000000000", "0001000000010001", "0000000100000000"},
		{"00000000000000000000000000000000", "0001000180000000", "0000000080000000"},
		{"00000000000000000000000000000000", "00010001C0000000", "0000000040000000"},
		{"00000000000000000000000000000000", "8001000160000000", "0000000020000000"},
		{"00000000000000000000000000000000", "4001000130008000", "0000000010000000"},
		{"00000000000000000000000000000000", "200100019800C000", "0000000008000000"},
		{"00000000000000000000000000000000", "100100014C00E000", "0000000004000000"},
		{"00000000000000000000000000000000", "8801000126007000", "0000000002000000"},
		{"00000000000000000000000000000000", "4401000193003800", "0000000001000000"},
		{"00000000000000000000000000000000", "22010001C9801C00", "0000000000800000"},
		{"00000000000000000000000000000000", "11010001E4C00E00", "0000000000400000"},
		{"00000000000000000000000000000000", "08810001F2600700", "0000000000200000"},
		{"00000000000000000000000000000000", "04410001F9300380", "0000000000100000"},
		{"00000000000000000000000000000000", "02210001FC9801C0", "0000000000080000"},
		{"00000000000000000000000000000000", "01110001FE4C00E0", "0000000000040000"},
		{"00000000000000000000000000000000", "00890001FF260070", "0000000000020000"},
		{"00000000000000000000000000000000", "FFC3FF910055FF9A", "0000000000010000"},
		{"00000000000000000000000000000000", "0001000100008000", "0000000000008000"},
		{"00000000000000000000000000000000", "0001000180004000", "0000000000004000"},
		{"00000000000000000000000000000000", "00018001C0002000", "0000000000002000"},
		{"00000000000000000000000000000000", "8001C001E0009000", "0000000000001000"},
		{"00000000000000000000000000000000", "400160017000C800", "0000000000000800"},
		{"00000000000000000000000000000000", "2001B00138006400", "0000000000000400"},
		{"00000000000000000000000000000000", "900158011C003200", "0000000000000200"},
		{"00000000000000000000000000000000", "C8012C018E009900", "0000000000000100"},
		{"00000000000000000000000000000000", "640116014700CC80", "0000000000000080"},
		{"00000000000000000000000000000000", "32010B012380E640", "0000000000000040"},
		{"00000000000000000000000000000000", "1901058111C0F320", "0000000000000020"},
		{"00000000000000000000000000000000", "0C8102C108E0F990", "0000000000000010"},
		{"00000000000000000000000000000000", "064101610470FCC8", "0000000000000008"},
		{"00000000000000000000000000000000", "032100B10238FE64", "0000000000000004"},
		{"00000000000000000000000000000000", "01910059011CFF32", "0000000000000002"},
		{"00000000000000000000000000000000", "0013FFF500120009", "0000000000000001"},
		{"00000000000000000000000000000000", "0001000100000000", "0000000000000000"},
		{"01010101010101010101010101010101", "6D33179CE8B3C1FA", "0101010101010101"},
		{"02020202020202020202020202020202", "DF163E38AD71A489", "0202020202020202"},
		{"03030303030303030303030303030303", "A74AB61FE64CCD14", "0303030303030303"},
		{"04040404040404040404040404040404", "DECD1268CE7B3CD5", "0404040404040404"},
		{"05050505050505050505050505050505", "E2F73C827CC38E67", "0505050505050505"},
		{"06060606060606060606060606060606", "FA93BE07E3B0E09C", "0606060606060606"},
		{"07070707070707070707070707070707", "AB246AC5AADE1B14", "0707070707070707"},
		{"08080808080808080808080808080808", "AC0C17067C0DA182", "0808080808080808"},
		{"09090909090909090909090909090909", "EE0EA8A773BE26C6", "0909090909090909"},
		{"0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A", "95BF40E823FB39C1", "0A0A0A0A0A0A0A0A"},
		{"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B", "EB316AA4993C68C6", "0B0B0B0B0B0B0B0B"},
		{"0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C", "E2C7D821AC4AD86C", "0C0C0C0C0C0C0C0C"},
		{"0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D", "7514A54BA6CFA933", "0D0D0D0D0D0D0D0D"},
		{"0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E", "2E5A8279A756DEF4", "0E0E0E0E0E0E0E0E"},
		{"0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F", "F124018C83415FF3", "0F0F0F0F0F0F0F0F"},
		{"10101010101010101010101010101010", "21D3F5BC44E9E1F1", "1010101010101010"},
		{"11111111111111111111111111111111", "3A1D3B4DB127C8B7", "1111111111111111"},
		{"12121212121212121212121212121212", "2497189242CDB152", "1212121212121212"},
		{"13131313131313131313131313131313", "DFC779D8613ECF5D", "1313131313131313"},
		{"14141414141414141414141414141414", "FFE36A1199F71AD9", "1414141414141414"},
		{"15151515151515151515151515151515", "E34F99D4B42F4DDB", "1515151515151515"},
		{"16161616161616161616161616161616", "E7A3CE5041458E86", "1616161616161616"},
		{"17171717171717171717171717171717", "607EE74A70676DAB", "1717171717171717"},
		{"18181818181818181818181818181818", "E89C1BD91C5CE63B", "1818181818181818"},
		{"19191919191919191919191919191919", "B679DF3338C7CEE9", "1919191919191919"},
		{"1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A", "B66BC9876BA3A6B8", "1A1A1A1A1A1A1A1A"},
		{"1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B", "EA695916687862A9", "1B1B1B1B1B1B1B1B"},
		{"1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C", "C4F9DA6928BFDB2C", "1C1C1C1C1C1C1C1C"},
		{"1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D", "4D068B78C0B1F45B", "1D1D1D1D1D1D1D1D"},
		{"1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E", "EB2D9BDB2F32A328", "1E1E1E1E1E1E1E1E"},
		{"1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F", "ECE75E6730910ACF", "1F1F1F1F1F1F1F1F"},
		{"20202020202020202020202020202020", "6242F5CC44A48D6C", "2020202020202020"},
		{"21212121212121212121212121212121", "95669A6794A3C260", "2121212121212121"},
		{"22222222222222222222222222222222", "400C910B1DEC3244", "2222222222222222"},
		{"23232323232323232323232323232323", "82954E32CD66F748", "2323232323232323"},
		{"24242424242424242424242424242424", "A28B98468B24105A", "2424242424242424"},
		{"25252525252525252525252525252525", "B88321E1A912C1CF", "2525252525252525"},
		{"26262626262626262626262626262626", "10EF9242C256D127", "2626262626262626"},
		{"27272727272727272727272727272727", "CC1BBE51581A159F", "2727272727272727"},
		{"28282828282828282828282828282828", "B5D7B562C27F8F2F", "2828282828282828"},
		{"29292929292929292929292929292929", "1893A5DF78EE02B1", "2929292929292929"},
		{"2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A", "26924C8140545B22", "2A2A2A2A2A2A2A2A"},
		{"2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B", "C88B1D305B712CF0", "2B2B2B2B2B2B2B2B"},
		{"2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C", "145BCEAE2F8B9332", "2C2C2C2C2C2C2C2C"},
		{"2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D", "EA6E631C0E7A0F76", "2D2D2D2D2D2D2D2D"},
		{"2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E", "4872CAF8FCD1D626", "2E2E2E2E2E2E2E2E"},
		{"2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F", "60FAA2607A7A174E", "2F2F2F2F2F2F2F2F"},
		{"30303030303030303030303030303030", "B1C48C467EC93D5A", "3030303030303030"},
		{"31313131313131313131313131313131", "C1779459BCD715C3", "3131313131313131"},
		{"32323232323232323232323232323232", "3890E76816A035D4", "3232323232323232"},
		{"33333333333333333333333333333333", "4575D36364CCF9D8", "3333333333333333"},
		{"34343434343434343434343434343434", "27EE09EF96B86533", "3434343434343434"},
		{"35353535353535353535353535353535", "50FFFE9A739CCFE0", "3535353535353535"},
		{"36363636363636363636363636363636", "C1D850AA9A08BFA7", "3636363636363636"},
		{"37373737373737373737373737373737", "EF6F54302E12184F", "3737373737373737"},
		{"38383838383838383838383838383838", "E3F3871178E7CC4F", "3838383838383838"},
		{"39393939393939393939393939393939", "0220D644B0143BDE", "3939393939393939"},
		{"3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A", "577A16E8964202BC", "3A3A3A3A3A3A3A3A"},
		{"3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B", "3E02E11AE0609C78", "3B3B3B3B3B3B3B3B"},
		{"3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C", "D92F00BB598101B5", "3C3C3C3C3C3C3C3C"},
		{"3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D", "C3A6754BFF90516D", "3D3D3D3D3D3D3D3D"},
		{"3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E", "07F539C02822590D", "3E3E3E3E3E3E3E3E"},
		{"3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F", "8896E8479A2EB33F", "3F3F3F3F3F3F3F3F"},
		{"40404040404040404040404040404040", "36D4CF3A566970AF", "4040404040404040"},
		{"41414141414141414141414141414141", "897D2CF7994B2470", "4141414141414141"},
		{"42424242424242424242424242424242", "F601A9632C3C62B7", "4242424242424242"},
		{"43434343434343434343434343434343", "BACBDD51B31EE859", "4343434343434343"},
		{"44444444444444444444444444444444", "F7638F197696DE7C", "4444444444444444"},
		{"45454545454545454545454545454545", "5B19AF843DE1A00D", "4545454545454545"},
		{"46464646464646464646464646464646", "133043708039BA9C", "4646464646464646"},
		{"47474747474747474747474747474747", "9639B2731CF6BD65", "4747474747474747"},
		{"48484848484848484848484848484848", "F2FDE6ECAD971C0A", "4848484848484848"},
		{"49494949494949494949494949494949", "988FD9549FAD5CA2", "4949494949494949"},
		{"4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A", "7460EC4F2FF98F4B", "4A4A4A4A4A4A4A4A"},
		{"4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B", "4202E7A8B8C82F53", "4B4B4B4B4B4B4B4B"},
		{"4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C", "7B9C3715642AA3AF", "4C4C4C4C4C4C4C4C"},
		{"4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D", "CF96E84CDCF75056", "4D4D4D4D4D4D4D4D"},
		{"4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E", "1DB2D04ADDC599E6", "4E4E4E4E4E4E4E4E"},
		{"4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F", "A4CFFD289BF0988E", "4F4F4F4F4F4F4F4F"},
		{"50505050505050505050505050505050", "93BA810A625138EE", "5050505050505050"},
		{"51515151515151515151515151515151", "5795C3C244BE0984", "5151515151515151"},
		{"52525252525252525252525252525252", "B255918917D30DB6", "5252525252525252"},
		{"53535353535353535353535353535353", "DBD63E3F426BF141", "5353535353535353"},
		{"54545454545454545454545454545454", "1F6C4D99DCC8863D", "5454545454545454"},
		{"55555555555555555555555555555555", "6074C411B8651B08", "5555555555555555"},
		{"56565656565656565656565656565656", "338E59304F374C81", "5656565656565656"},
		{"57575757575757575757575757575757", "D669692FC58844F4", "5757575757575757"},
		{"58585858585858585858585858585858", "F401BA8CD09CBDFB", "5858585858585858"},
		{"59595959595959595959595959595959", "EEECA9B832961D1C", "5959595959595959"},
		{"5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A", "7196F6C2933D0048", "5A5A5A5A5A5A5A5A"},
		{"5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B", "A159602CF7BA36F4", "5B5B5B5B5B5B5B5B"},
		{"5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C", "2F45050AF47066B6", "5C5C5C5C5C5C5C5C"},
		{"5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D", "55C8274CF9E5122F", "5D5D5D5D5D5D5D5D"},
		{"5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E", "4393C3BEF64A0962", "5E5E5E5E5E5E5E5E"},
		{"5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F", "87CDFDA728278056", "5F5F5F5F5F5F5F5F"},
		{"60606060606060606060606060606060", "C419303C1BC6078B", "6060606060606060"},
		{"61616161616161616161616161616161", "14189D710244EC0F", "6161616161616161"},
		{"62626262626262626262626262626262", "E283182D89DC2ABC", "6262626262626262"},
		{"63636363636363636363636363636363", "5F7C9EF49BE17A09", "6363636363636363"},
		{"64646464646464646464646464646464", "ABF61E3F6354537E", "6464646464646464"},
		{"65656565656565656565656565656565", "3116CF29793667EE", "6565656565656565"},
		{"66666666666666666666666666666666", "C9248B00868D8651", "6666666666666666"},
		{"67676767676767676767676767676767", "FA85B0D5EB81E00E", "6767676767676767"},
		{"68686868686868686868686868686868", "ABD32134D248DE95", "6868686868686868"},
		{"69696969696969696969696969696969", "CE4C7A6696E7045F", "6969696969696969"},
		{"6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A", "BF217A062A3F2752", "6A6A6A6A6A6A6A6A"},
		{"6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B", "1EE92826247B879F", "6B6B6B6B6B6B6B6B"},
		{"6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C", "534051CFAE0E474B", "6C6C6C6C6C6C6C6C"},
		{"6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D", "FE541578DFA1EF21", "6D6D6D6D6D6D6D6D"},
		{"6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E", "A6DA49EB274AA034", "6E6E6E6E6E6E6E6E"},
		{"6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F", "E91200E1BEC746D8", "6F6F6F6F6F6F6F6F"},
		{"70707070707070707070707070707070", "12242D95C7B2E1AA", "7070707070707070"},
		{"71717171717171717171717171717171", "11112654F080A009", "7171717171717171"},
		{"72727272727272727272727272727272", "B6B4AF43C3C1BD5A", "7272727272727272"},
		{"73737373737373737373737373737373", "7FC1DA4D8A29927A", "7373737373737373"},
		{"74747474747474747474747474747474", "4D1424A6E2F99777", "7474747474747474"},
		{"75757575757575757575757575757575", "C6DC9F097BC971E7", "7575757575757575"},
		{"76767676767676767676767676767676", "B88002C84F836EA6", "7676767676767676"},
		{"77777777777777777777777777777777", "12C6774BE0C5E93A", "7777777777777777"},
		{"78787878787878787878787878787878", "D1A2580870C505DF", "7878787878787878"},
		{"79797979797979797979797979797979", "65B79538071AF9D8", "7979797979797979"},
		{"7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A", "0481B20F32B63B37", "7A7A7A7A7A7A7A7A"},
		{"7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B", "02ACD30476C23A0C", "7B7B7B7B7B7B7B7B"},
		{"7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C", "2C4CEDBF6A2C54AD", "7C7C7C7C7C7C7C7C"},
		{"7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D", "50F38D16D6A0AE03", "7D7D7D7D7D7D7D7D"},
		{"7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E", "BDF7F828449AA693", "7E7E7E7E7E7E7E7E"},
		{"7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F", "A66CF7F4B4C3103D", "7F7F7F7F7F7F7F7F"},
		{"80808080808080808080808080808080", "1736C7434D3A1A54", "8080808080808080"},
		{"81818181818181818181818181818181", "78C56A5C99E3885F", "8181818181818181"},
		{"82828282828282828282828282828282", "24B2420569DC9513", "8282828282828282"},
		{"83838383838383838383838383838383", "2E46220B77AB7A4D", "8383838383838383"},
		{"84848484848484848484848484848484", "B60DFA18180164EE", "8484848484848484"},
		{"85858585858585858585858585858585", "1751553EF8A1C49D", "8585858585858585"},
		{"86868686868686868686868686868686", "41F68E0ACBDE794D", "8686868686868686"},
		{"87878787878787878787878787878787", "0A8AECE3AE569CCC", "8787878787878787"},
		{"88888888888888888888888888888888", "4B29633EC0D32ED7", "8888888888888888"},
		{"89898989898989898989898989898989", "BC1EB08BD3EEBFFF", "8989898989898989"},
		{"8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A", "14FE00047A151C2B", "8A8A8A8A8A8A8A8A"},
		{"8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B", "7F0EC436C9C4633B", "8B8B8B8B8B8B8B8B"},
		{"8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C", "7DB08FD2710CC441", "8C8C8C8C8C8C8C8C"},
		{"8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D", "2AA6DB5A727691A3", "8D8D8D8D8D8D8D8D"},
		{"8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E", "CF2F3F532FAA99BF", "8E8E8E8E8E8E8E8E"},
		{"8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F", "E5D56F002C4FB53F", "8F8F8F8F8F8F8F8F"},
		{"90909090909090909090909090909090", "E1DD1375465F9D16", "9090909090909090"},
		{"91919191919191919191919191919191", "BECB1FF1A78450A7", "9191919191919191"},
		{"92929292929292929292929292929292", "0185E9A9135E639D", "9292929292929292"},
		{"93939393939393939393939393939393", "A3626C4517B4CAD0", "9393939393939393"},
		{"94949494949494949494949494949494", "0A629CADE4897292", "9494949494949494"},
		{"95959595959595959595959595959595", "F725DE012AB68DD9", "9595959595959595"},
		{"96969696969696969696969696969696", "0E89BE3D5B3B866B", "9696969696969696"},
		{"97979797979797979797979797979797", "5B746609C56BA450", "9797979797979797"},
		{"98989898989898989898989898989898", "F2DD9359B1FF1598", "9898989898989898"},
		{"99999999999999999999999999999999", "78C169C1CF547F0D", "9999999999999999"},
		{"9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A", "55C13E79CDF4A6C8", "9A9A9A9A9A9A9A9A"},
		{"9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B", "3C7847ECDBD22E7F", "9B9B9B9B9B9B9B9B"},
		{"9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C", "2EB78E05CCBEB44D", "9C9C9C9C9C9C9C9C"},
		{"9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D", "8D9B8A6C18615855", "9D9D9D9D9D9D9D9D"},
		{"9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E", "C5CC7E174C80ED46", "9E9E9E9E9E9E9E9E"},
		{"9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F", "1DB1528F10686802", "9F9F9F9F9F9F9F9F"},
		{"A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0A0", "FE504481BDCCAA9B", "A0A0A0A0A0A0A0A0"},
		{"A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1", "ED3D73069A19297C", "A1A1A1A1A1A1A1A1"},
		{"A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2", "7CDFE35AEF5A1E7C", "A2A2A2A2A2A2A2A2"},
		{"A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3A3", "2DDCD253556E0858", "A3A3A3A3A3A3A3A3"},
		{"A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4A4", "A7BDA86B1F764E3A", "A4A4A4A4A4A4A4A4"},
		{"A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5", "F2E0BF5370908BC4", "A5A5A5A5A5A5A5A5"},
		{"A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6A6", "B55DA7709795D136", "A6A6A6A6A6A6A6A6"},
		{"A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7A7", "41B77E59B3ECFD12", "A7A7A7A7A7A7A7A7"},
		{"A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8A8", "108580BC7A99BE27", "A8A8A8A8A8A8A8A8"},
		{"A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9A9", "5FC7AEDC2F0714F0", "A9A9A9A9A9A9A9A9"},
		{"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", "24AACB435BA4778B", "AAAAAAAAAAAAAAAA"},
		{"ABABABABABABABABABABABABABABABAB", "5A4C4870F25A207F", "ABABABABABABABAB"},
		{"ACACACACACACACACACACACACACACACAC", "607D4E32E138BB42", "ACACACACACACACAC"},
		{"ADADADADADADADADADADADADADADADAD", "7F937C3392173535", "ADADADADADADADAD"},
		{"AEAEAEAEAEAEAEAEAEAEAEAEAEAEAEAE", "FBB59D88D9C7E032", "AEAEAEAEAEAEAEAE"},
		{"AFAFAFAFAFAFAFAFAFAFAFAFAFAFAFAF", "E762CC43752B56CA", "AFAFAFAFAFAFAFAF"},
		{"B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0", "9C319C17F979D8A7", "B0B0B0B0B0B0B0B0"},
		{"B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1", "7118249DF312E1EA", "B1B1B1B1B1B1B1B1"},
		{"B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2B2", "341359D2B974161A", "B2B2B2B2B2B2B2B2"},
		{"B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3B3", "4F50E094D64B9A71", "B3B3B3B3B3B3B3B3"},
		{"B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4", "F471458A304D4AD4", "B4B4B4B4B4B4B4B4"},
		{"B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5", "7D95A537EF4FD20D", "B5B5B5B5B5B5B5B5"},
		{"B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6B6", "2B19B8D18E9E9D9E", "B6B6B6B6B6B6B6B6"},
		{"B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7B7", "90B1D81ED1930DCA", "B7B7B7B7B7B7B7B7"},
		{"B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8B8", "0D70077A42A0F3FC", "B8B8B8B8B8B8B8B8"},
		{"B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9B9", "33BE0D577E657AAA", "B9B9B9B9B9B9B9B9"},
		{"BABABABABABABABABABABABABABABABA", "8642A8FA64B33D74", "BABABABABABABABA"},
		{"BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB", "ECEFAA828E05250C", "BBBBBBBBBBBBBBBB"},
		{"BCBCBCBCBCBCBCBCBCBCBCBCBCBCBCBC", "BFEACC224C09B041", "BCBCBCBCBCBCBCBC"},
		{"BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD", "25CE35BBBC4CE5B6", "BDBDBDBDBDBDBDBD"},
		{"BEBEBEBEBEBEBEBEBEBEBEBEBEBEBEBE", "FCF3D5FA7F3C103D", "BEBEBEBEBEBEBEBE"},
		{"BFBFBFBFBFBFBFBFBFBFBFBFBFBFBFBF", "7E8C714B5539D948", "BFBFBFBFBFBFBFBF"},
		{"C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0", "5AE793705E429DCF", "C0C0C0C0C0C0C0C0"},
		{"C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1C1", "403ACA4C4D536916", "C1C1C1C1C1C1C1C1"},
		{"C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2C2", "DE7B750CF725CD86", "C2C2C2C2C2C2C2C2"},
		{"C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3C3", "E01699BFA8B8D615", "C3C3C3C3C3C3C3C3"},
		{"C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4C4", "05EA04208311AA5B", "C4C4C4C4C4C4C4C4"},
		{"C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5C5", "E41742FEBAD80931", "C5C5C5C5C5C5C5C5"},
		{"C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6", "3F2C325AC3F15A0F", "C6C6C6C6C6C6C6C6"},
		{"C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7C7", "E82E2BFF7127942A", "C7C7C7C7C7C7C7C7"},
		{"C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8C8", "0CCCD6D4EE2F2132", "C8C8C8C8C8C8C8C8"},
		{"C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9C9", "BEBA48342116C6B8", "C9C9C9C9C9C9C9C9"},
		{"CACACACACACACACACACACACACACACACA", "295608C4CD38451B", "CACACACACACACACA"},
		{"CBCBCBCBCBCBCBCBCBCBCBCBCBCBCBCB", "3D058D6AA17E5F82", "CBCBCBCBCBCBCBCB"},
		{"CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC", "E143A92DBA6EC493", "CCCCCCCCCCCCCCCC"},
		{"CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD", "238E6C5A341BD450", "CDCDCDCDCDCDCDCD"},
		{"CECECECECECECECECECECECECECECECE", "8443582E6810A7ED", "CECECECECECECECE"},
		{"CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF", "E5DF9D8A645559C0", "CFCFCFCFCFCFCFCF"},
		{"D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0D0", "512620AE09ACB3BA", "D0D0D0D0D0D0D0D0"},
		{"D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1", "378C3A04D3A09F9A", "D1D1D1D1D1D1D1D1"},
		{"D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2", "D7DEDE3AF9410258", "D2D2D2D2D2D2D2D2"},
		{"D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3D3", "A7E0F6A26D81B598", "D3D3D3D3D3D3D3D3"},
		{"D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4D4", "33FC82EE6BB350F4", "D4D4D4D4D4D4D4D4"},
		{"D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5D5", "3F663D6FB864212A", "D5D5D5D5D5D5D5D5"},
		{"D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6D6", "99C853C822F2C451", "D6D6D6D6D6D6D6D6"},
		{"D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7D7", "4AD4BE6D0D2F57D8", "D7D7D7D7D7D7D7D7"},
		{"D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8", "D6D60732A807D08B", "D8D8D8D8D8D8D8D8"},
		{"D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9D9", "E6A6BE8437EC6983", "D9D9D9D9D9D9D9D9"},
		{"DADADADADADADADADADADADADADADADA", "472A2DEAD18618F6", "DADADADADADADADA"},
		{"DBDBDBDBDBDBDBDBDBDBDBDBDBDBDBDB", "4567D7128C786B31", "DBDBDBDBDBDBDBDB"},
		{"DCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDC", "6954437E083CB8DD", "DCDCDCDCDCDCDCDC"},
		{"DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD", "BB0056DE857988CE", "DDDDDDDDDDDDDDDD"},
		{"DEDEDEDEDEDEDEDEDEDEDEDEDEDEDEDE", "D2058501E91225A5", "DEDEDEDEDEDEDEDE"},
		{"DFDFDFDFDFDFDFDFDFDFDFDFDFDFDFDF", "E35E6B186283F441", "DFDFDFDFDFDFDFDF"},
		{"E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0E0", "242588C5F782A93F", "E0E0E0E0E0E0E0E0"},
		{"E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1", "7095A50C1E99E008", "E1E1E1E1E1E1E1E1"},
		{"E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2E2", "9933DF5A78A355A8", "E2E2E2E2E2E2E2E2"},
		{"E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3E3", "D1D945E51221B893", "E3E3E3E3E3E3E3E3"},
		{"E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4E4", "915D915B79B927ED", "E4E4E4E4E4E4E4E4"},
		{"E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5E5", "E0C9897A94D0F2C4", "E5E5E5E5E5E5E5E5"},
		{"E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6E6", "911E376BF4497437", "E6E6E6E6E6E6E6E6"},
		{"E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7", "B82FAE4092690A95", "E7E7E7E7E7E7E7E7"},
		{"E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8E8", "E6ACA65966B4AC3F", "E8E8E8E8E8E8E8E8"},
		{"E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9E9", "DFA504DD9CCF1E78", "E9E9E9E9E9E9E9E9"},
		{"EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA", "DF00C0F695286D74", "EAEAEAEAEAEAEAEA"},
		{"EBEBEBEBEBEBEBEBEBEBEBEBEBEBEBEB", "A3BA29F79071549E", "EBEBEBEBEBEBEBEB"},
		{"ECECECECECECECECECECECECECECECEC", "BB46376E1CF4EC6A", "ECECECECECECECEC"},
		{"EDEDEDEDEDEDEDEDEDEDEDEDEDEDEDED", "1195A6DE7052280E", "EDEDEDEDEDEDEDED"},
		{"EEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE", "415E756EFDCB7A4A", "EEEEEEEEEEEEEEEE"},
		{"EFEFEFEFEFEFEFEFEFEFEFEFEFEFEFEF", "56A88197AE5924BD", "EFEFEFEFEFEFEFEF"},
		{"F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0F0", "89951D70270C9B17", "F0F0F0F0F0F0F0F0"},
		{"F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1F1", "2A775028D170B29C", "F1F1F1F1F1F1F1F1"},
		{"F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2F2", "9A4CF8B7C599EF4A", "F2F2F2F2F2F2F2F2"},
		{"F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3", "F9966866D43965CD", "F3F3F3F3F3F3F3F3"},
		{"F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4F4", "80E41CCC7641E959", "F4F4F4F4F4F4F4F4"},
		{"F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5F5", "0B7C702996413B45", "F5F5F5F5F5F5F5F5"},
		{"F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6F6", "EB69BC3BF5B3FEA5", "F6F6F6F6F6F6F6F6"},
		{"F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7F7", "36BF2856E58435EA", "F7F7F7F7F7F7F7F7"},
		{"F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8F8", "1CD03F0A40A7B20C", "F8F8F8F8F8F8F8F8"},
		{"F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9F9", "F56AF83339036916", "F9F9F9F9F9F9F9F9"},
		{"FAFAFAFAFAFAFAFAFAFAFAFAFAFAFAFA", "3114B9E8F15EA604", "FAFAFAFAFAFAFAFA"},
		{"FBFBFBFBFBFBFBFBFBFBFBFBFBFBFBFB", "55BC49104267BD78", "FBFBFBFBFBFBFBFB"},
		{"FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC", "1A578AB6029071AE", "FCFCFCFCFCFCFCFC"},
		{"FDFDFDFDFDFDFDFDFDFDFDFDFDFDFDFD", "1C2292FA2BDF8F40", "FDFDFDFDFDFDFDFD"},
		{"FEFEFEFEFEFEFEFEFEFEFEFEFEFEFEFE", "C7E3F87FFE503CC3", "FEFEFEFEFEFEFEFE"},
		{"FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF", "28886D814399E782", "FFFFFFFFFFFFFFFF"},
		{"000102030405060708090A0B0C0D0E0F", "DB2D4A92AA68273F", "0011223344556677"},
		{"2BD6459F82C5B300952C49104881FF48", "F129A6601EF62A47", "EA024714AD5C4D84"},
	}

	for _, tt := range tests {
		k, _ := hex.DecodeString(tt.key)
		p, _ := hex.DecodeString(tt.plain)
		c, _ := hex.DecodeString(tt.cipher)

		var dst [8]byte

		cipher, _ := NewCipher(k)

		cipher.Encrypt(dst[:], p)

		if !bytes.Equal(dst[:], c) {
			t.Fatalf("encrypt(%x) with key %x failed: got %x wanted %x\n", p, k, dst, c)
		}

		var plain [8]byte

		cipher.Decrypt(plain[:], dst[:])

		if !bytes.Equal(plain[:], p) {
			t.Fatalf("decrypt(%x) with key %x  failed: got %x wanted %x\n", dst, k, plain, p)
		}
	}
}
