% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getopt.R
\name{getopt}
\alias{getopt}
\alias{getopt-package}
\title{C-like getopt behavior}
\usage{
getopt(
  spec = NULL,
  opt = NULL,
  command = get_Rscript_filename(),
  usage = FALSE,
  debug = FALSE
)
}
\arguments{
\item{spec}{The getopt specification, or spec of what options are considered
valid.  The specification must be either a 4-5 column matrix, or a
character vector coercible into a 4 column matrix using
\code{matrix(x,ncol=4,byrow=TRUE)} command.  The matrix/vector
contains:

Column 1: the \emph{long flag} name.  A multi-character string.

Column 2: \emph{short flag} alias of Column 1.  A single-character
string.

Column 3: \emph{Argument} mask of the \emph{flag}.  An integer.
Possible values: 0=no argument, 1=required argument, 2=optional argument.

Column 4: Data type to which the \emph{flag}'s argument shall be cast using
\code{\link[=storage.mode]{storage.mode()}}.  A multi-character string.  This only considered
for same-row Column 3 values of 1,2.  Possible values: logical,
integer, double, complex, character.
If numeric is encountered then it will be converted to double.

Column 5 (optional): A brief description of the purpose of the option.

The terms \emph{option}, \emph{flag}, \emph{long flag}, \emph{short flag},
and \emph{argument} have very specific meanings in the context of this
document.  Read the \dQuote{Description} section for definitions.}

\item{opt}{This defaults to the return value of \code{commandArgs(TRUE)} unless
\code{argv} is in the global environment in which case it uses that instead
(this is for compatibility with \code{littler}).

If R was invoked directly via the \code{R} command, this corresponds to all
arguments passed to R after the \code{--args} flag.

If R was invoked via the \code{Rscript} command, this corresponds to all
arguments after the name of the R script file.

Read about \code{\link[=commandArgs]{commandArgs()}} and \link{Rscript} to learn more.}

\item{command}{The string to use in the usage message as the name of the
script.  See argument \emph{usage}.}

\item{usage}{If \code{TRUE}, argument \code{opt} will be ignored and a usage
statement (character string) will be generated and returned from \code{spec}.}

\item{debug}{This is used internally to debug the \code{getopt()} function itself.}
}
\description{
\code{getopt} is primarily intended to be used with \code{Rscript}.  It
facilitates writing \verb{#!} shebang scripts that accept short and long
flags/options.  It can also be used from \code{R} directly, but is probably less
useful in this context.
}
\details{
\code{\link[=getopt]{getopt()}} returns a list data structure containing names of the
flags that were present in the character vector passed in under
the \code{opt} argument.  Each value of the list is coerced to the
data type specified according to the value of the \code{spec} argument.  See
below for details.

Notes on naming convention:
\enumerate{
\item An \emph{option} is one of the shell-split input strings.
\item A \emph{flag} is a type of \emph{option}.  a \emph{flag} can be defined as
having no \emph{argument} (defined below), a required \emph{argument}, or an
optional \emph{argument}.
\item An \emph{argument} is a type of \emph{option}, and is the value associated
with a flag.
\item A \emph{long flag} is a type of \emph{flag}, and begins with the string
\verb{--}.  If the \emph{long flag} has an associated \emph{argument}, it may be
delimited from the \emph{long flag} by either a trailing \code{=}, or may be
the subsequent \emph{option}.
\item A \emph{short flag} is a type of \emph{flag}, and begins with the string
\code{-}.  If a \emph{short flag} has an associated \emph{argument}, it is the
subsequent \emph{option}.  \emph{short flags} may be bundled together,
sharing a single leading \code{-}, but only the final \emph{short flag} is able
to have a corresponding \emph{argument}.
}

Many users wonder whether they should use the \code{getopt} package, \code{optparse} package,
or \code{argparse} package.
Here is some of the major differences:

Features available in \code{getopt} unavailable in \code{optparse}
\enumerate{
\item As well as allowing one to specify options that take either
no argument or a required argument like \code{optparse},
\code{getopt} also allows one to specify option with an optional argument.
}

Some features implemented in \code{optparse} package unavailable in \code{getopt}
\enumerate{
\item Limited support for capturing positional arguments after the optional arguments
when \code{positional_arguments} set to \code{TRUE} in \code{optparse::parse_args()}
\item Automatic generation of an help option and printing of help text when encounters an \code{-h}
\item Option to specify default arguments for options as well the
variable name to store option values
}

There is also new package \code{argparse} introduced in 2012 which contains
all the features of both getopt and optparse but which has a dependency on
Python 2.7 or 3.2+.

Some Features unlikely to be implemented in \code{getopt}:
\enumerate{
\item Support for multiple, identical flags, e.g. for \verb{-m 3 -v 5 -v}, the
trailing \code{-v} overrides the preceding \verb{-v 5}, result is \code{v=TRUE} (or equivalent
typecast).
\item Support for multi-valued flags, e.g. \verb{--libpath=/usr/local/lib --libpath=/tmp/foo}.
\item Support for lists, e.g. \verb{--define os=linux --define os=redhat} would
set \code{result$os$linux=TRUE} and \code{result$os$redhat=TRUE}.
\item Support for incremental, argument-less flags, e.g. \verb{/path/to/script -vvv} should set \code{v=3}.
\item Support partial-but-unique string match on options, e.g. \code{--verb} and
\code{--verbose} both match long flag \code{--verbose}.
\item No support for mixing in positional arguments or extra arguments that
don't match any options.  For example, you can't do \verb{my.R --arg1 1 foo bar baz} and recover \code{foo}, \code{bar}, \code{baz} as a list.  Likewise for \verb{my.R foo --arg1 1 bar baz}.
}
}
\examples{
#!/path/to/Rscript
library('getopt')
# get options, using the spec as defined by the enclosed list.
# we read the options from the default: commandArgs(TRUE).
spec = matrix(c(
  'verbose', 'v', 2, "integer",
  'help'   , 'h', 0, "logical",
  'count'  , 'c', 1, "integer",
  'mean'   , 'm', 1, "double",
  'sd'     , 's', 1, "double"
), byrow=TRUE, ncol=4)
opt = getopt(spec)

# if help was asked for print a friendly message
# and exit with a non-zero error code
if (!is.null(opt$help)) {
  cat(getopt(spec, usage = TRUE))
  q(status = 1)
}

# set some reasonable defaults for the options that are needed,
# but were not specified.
if (is.null(opt$mean)) opt$mean <- 0
if (is.null(opt$sd)) opt$sd <- 1
if (is.null(opt$count)) opt$count <- 10
if (is.null(opt$verbose)) opt$verbose <- FALSE

# print some progress messages to stderr, if requested.
if (opt$verbose) write("writing...", stderr())

# do some operation based on user input.
cat(rnorm(opt$count, mean = opt$mean, sd = opt$sd), sep = "\n")

# signal success and exit.
# q(status=0)
}
\author{
Allen Day
}
\keyword{data}
