/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/* 
// 
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (P192r1 precomputed)
// 
// 
*/
#include "owncp.h"
#include "pcpgfpecstuff.h"


#define OPERAND_BITSIZE (192)
#define LEN_P192        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* P192 affine point */
typedef struct{
   BNU_CHUNK_T X[LEN_P192];
   BNU_CHUNK_T Y[LEN_P192];
} P192_POINT_AFFINE;

extern const __ALIGN64 P192_POINT_AFFINE ec_p192r1_precomputed[28][64];


#if defined ( _IPP_DATA )

#if !defined(_DISABLE_ECP_192R1_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

const __ALIGN64 P192_POINT_AFFINE ec_p192r1_precomputed[28][64] = {
/* digit=0 base_pwr=2^0 */
{
   LL(0x332fa108,0x0d8cb30c),LL(0x76d12909,0x8a4bd3f7),LL(0xf3d218f7,0x954cc8f9), LL(0x1e422289,0x7b12a337),LL(0x8966f05e,0xde22b524),LL(0x6aeda84d,0x6a293d83),
   LL(0x91fae7b3,0x04a5cf09),LL(0xa7838baa,0xd7fb036c),LL(0x3e00e321,0xae53f3bb), LL(0xf5bc34a6,0x369ef507),LL(0x09d7bb36,0x7d5170c4),LL(0xad59278a,0x241e55c9),
   LL(0x230c0249,0x56b3fd7f),LL(0x42bebbf4,0x3386809f),LL(0x770c581a,0x53b5ad45), LL(0x7fb74b1e,0x6be17b49),LL(0x50d91fbd,0x16445c48),LL(0x43dc19c0,0x228f18e2),
   LL(0x9a01a832,0xda417f6b),LL(0xd148d209,0x8a42d66e),LL(0x60c3ed50,0xe544900a), LL(0x0fc71a62,0xc79d34b6),LL(0x8733ef53,0xe640e166),LL(0x69803121,0xc1084635),
   LL(0x1d8490a8,0x69bca784),LL(0x4d92a6ad,0xa7c43520),LL(0x7012b11d,0x4ec31c34), LL(0x843e401c,0x6261825c),LL(0x6a65a2da,0x102b7a93),LL(0x2d96f43a,0xdf00083f),
   LL(0xd6f1d9ee,0xae494b37),LL(0x81585220,0x4708a709),LL(0xed8612f6,0x3c3a183d), LL(0xda5571cc,0x8d3ab905),LL(0xf6131ada,0x8c11bd0f),LL(0xe556b1ea,0x83f240b8),
   LL(0xe5efca73,0xeb8ad91f),LL(0x1650b841,0x4c83fc44),LL(0xa63ebb45,0xeea07d43), LL(0xcc6ab309,0x2c8262c8),LL(0xca1e7327,0x44a67081),LL(0x66147372,0x6fef6d88),
   LL(0x5264d6fe,0x7afb70b5),LL(0x1e79609b,0x8f728a49),LL(0xeae172bd,0x441912c0), LL(0x8ec468d4,0x96163a6d),LL(0x07b3dc14,0xbd51c5c7),LL(0x707ae554,0xca743b38),
   LL(0x449ce446,0x9f27848b),LL(0xed6f60e4,0xedc613b6),LL(0x33ef2855,0xd028dc5c), LL(0x97566f06,0x32bbfe7f),LL(0x529bebe5,0xaee52dfc),LL(0xdbd304ac,0x7dfad9da),
   LL(0xcb9cdc1c,0x301800ef),LL(0xa4c0645f,0x01c6df1b),LL(0xd2c1c6da,0x7c2b2dca), LL(0x91015dd9,0x92fe0217),LL(0x43f5e36d,0x94ec3870),LL(0xfae24bbd,0x3400d9bd),
   LL(0xf19ed4f9,0x44a299cc),LL(0x3ec18b4b,0x5ce790e3),LL(0x3898e8a1,0x34de50ac), LL(0x52948777,0xa2641e2f),LL(0x538eff35,0xa5037393),LL(0x38e46331,0xf1b6bb32),
   LL(0xa53425b9,0xa02e1119),LL(0x9dfe0d6b,0x6a2f4991),LL(0x360f54c1,0xda626cb7), LL(0x39cdd976,0xc224f8ac),LL(0x21725fbe,0x5b23f190),LL(0x164b3680,0xf305458e),
   LL(0x0f8904a9,0x153f3b78),LL(0x14a75349,0x44a7bd96),LL(0xd85d4a3f,0x4093735f), LL(0x3f7db015,0xed1d5ed0),LL(0x1fff90b3,0x174975f7),LL(0xdcd47496,0x9838d562),
   LL(0x24fc4afc,0xf299b94a),LL(0x1cc29901,0x4bb0fffe),LL(0x3eb21741,0x6cd077ba), LL(0x95c82991,0x6e97b4f9),LL(0x8209cb7d,0x848b6921),LL(0xa082e4c9,0x34d09e8b),
   LL(0xb0cf681b,0xc373e084),LL(0x6610b602,0x7da92737),LL(0xf098041b,0x46cadc98), LL(0x6cdec872,0xd3e0adf8),LL(0xa7842f28,0x50d43591),LL(0xcdf46778,0xa35efdc3),
   LL(0x7944ad9c,0x90c57eb8),LL(0x1e7a0999,0xb9c5cf4b),LL(0xee3159ba,0xe0315bd7), LL(0x06a19eaf,0x17e47405),LL(0x4be877f9,0x5abccc61),LL(0x22195568,0x427706fb),
   LL(0xe7056d4e,0xcfdba265),LL(0xa9b98e89,0xc9d1189c),LL(0xa4a88eff,0x3e1cd309), LL(0xd040ec58,0x8b2429c3),LL(0x7859d5e6,0x6967afd6),LL(0xa5fbf616,0xcdf1095b),
   LL(0xfe4697b7,0x78060e67),LL(0x2648e5f8,0x1ed0242c),LL(0x4f235455,0x687ef0c6), LL(0x80964fe2,0x95a99ec5),LL(0xa71449df,0xe59296c6),LL(0xe31fca8c,0x6e0faa87),
   LL(0x417692ee,0x37b23549),LL(0x7a4db977,0x3fafb8db),LL(0x5d550c5c,0xc85fef61), LL(0xc591dfcc,0x88f70525),LL(0xadbba7d1,0x20d82c9e),LL(0x3bc105c7,0x34bdc150),
   LL(0x312e2b25,0xa96781ce),LL(0xc28a98fb,0x21d19734),LL(0xb32fbb94,0x33d91d89), LL(0xf1a39faa,0x4b423b1e),LL(0xd6d73b09,0xd6b91a11),LL(0x5cf7ee7b,0x1d5bbc7d),
   LL(0xc7b622a0,0x09a14d89),LL(0x04ac07cd,0x719fb611),LL(0x485b1f75,0x487f1ad3), LL(0xf779459c,0x6604b185),LL(0x79ffa446,0x633d924a),LL(0xe2ed4e42,0x0a48ac49),
   LL(0x97a2ac99,0x8f8141df),LL(0x559a156a,0x2f85aea4),LL(0x0de172c5,0x045569d1), LL(0x70195a60,0xd31e98ed),LL(0x31901571,0x639f8252),LL(0x017861e5,0xddfb3017),
   LL(0x8d76bd59,0x791db75f),LL(0xa5c667f8,0x664088fb),LL(0x5a1f032a,0x7cc0335c), LL(0xbf031b03,0xabb460b3),LL(0xc8f559b0,0xcb61b85e),LL(0xfed307dc,0x282335ba),
   LL(0xe9b47661,0x398b31e5),LL(0x4785fbe8,0x17a770ad),LL(0x2bf4567e,0xdb5e0ce3), LL(0xc82fa654,0xd8db9c86),LL(0x1b760f9c,0xdc7dbdfc),LL(0x65df38f7,0x7dc26c5a),
   LL(0xbfd973a2,0x422b5a19),LL(0x7de885a1,0xe637c08a),LL(0x28af843c,0x493d27a7), LL(0x27d0f709,0x0635af16),LL(0xa46b8241,0xfaa0731d),LL(0xde8513de,0x30152936),
   LL(0xc2c6a98b,0xe5b1f5c5),LL(0xdd1f1a85,0x9999e0ab),LL(0xf09f0de9,0x0f017d6e), LL(0xffad11c5,0x2cd84536),LL(0x97ac8fd1,0xc150b469),LL(0xf4a156d0,0x7275b4b6),
   LL(0xdea3626f,0xeaa0614e),LL(0x51b201c7,0xb0778899),LL(0xbc24cf40,0xe1e17ccf), LL(0x6f6a4f09,0x5b9a2356),LL(0x81b35be5,0x6b43dd89),LL(0x8eb6864b,0x7134a900),
   LL(0x6256a868,0x1fbfbc8d),LL(0x3b84fa71,0xc3c97795),LL(0x75676360,0xaf1330f2), LL(0x5ab6bd0c,0xaa33c244),LL(0xbd5e407e,0xece4da7e),LL(0x2f438421,0x8e4be95e),
   LL(0x267c3b3b,0xb550886c),LL(0x072ada2a,0xe4ef0b6a),LL(0x16cf9337,0xefb23bc9), LL(0x035d6cdd,0xf4945a35),LL(0x24d8682c,0x7c2fbc75),LL(0xb35b5932,0xb4b6f148),
   LL(0x1704593e,0x5ea32695),LL(0x487113d8,0xcb897688),LL(0x48e6fb62,0x9a382e50), LL(0x59da6a81,0x845eeeee),LL(0x72ab659e,0x427e258f),LL(0x0186211c,0x8d145e39),
   LL(0xc1a83d71,0xa5549752),LL(0x07c519da,0xe9b2c58f),LL(0x286a27e2,0x6f5e6aee), LL(0x7cfc6dd4,0x99cece8d),LL(0xbbc24c84,0x18c9b5f7),LL(0xd5a88338,0x7e5875ca),
   LL(0xde0dd9bb,0x655f1b75),LL(0x2ef38753,0x19df3f60),LL(0xeceb06b1,0x7a25bb17), LL(0xce909ba8,0x1359f3db),LL(0x65603d61,0xaa2802b7),LL(0xe147b10c,0xf7070a53),
   LL(0x4fbffaeb,0x41095e96),LL(0x8b3c6012,0x422337e2),LL(0x563eef5d,0xb731e12b), LL(0xc07cc789,0x7e950bdd),LL(0x9d9cc3aa,0xf97c2753),LL(0x5a7f7345,0x7af21a9a),
   LL(0xe3a0c0e2,0x7e91f2d6),LL(0x54a1cd29,0xa286afb7),LL(0xa6160c29,0x8c0ddde8), LL(0x8c5398c8,0xb03e8586),LL(0xfe6503e1,0x4db7075d),LL(0x7ec6bed6,0x999eec3e),
   LL(0x16036842,0xb35bcfb0),LL(0xecedc764,0x036d725d),LL(0x2c6f8668,0xcbcb0161), LL(0x1b04bc1a,0x0348f719),LL(0xe1e99a42,0x0b2d84f8),LL(0x9c1ec6e6,0xf098e97a),
   LL(0x3f5a8497,0x17f2106b),LL(0xf56d582b,0xc4c8ac85),LL(0x514207b2,0xde518ab8), LL(0x2e30b7c4,0x33018834),LL(0x643786e1,0xf867de48),LL(0x735735ab,0x72d64984),
   LL(0x2775c0c5,0x9d83fd01),LL(0x065da96c,0x8658cc5a),LL(0x604210f1,0x466cf509), LL(0xe8fc750b,0xdc4d8237),LL(0x1477a111,0xb13663ff),LL(0x4252c38b,0x6a0ef58f),
   LL(0xe5bcf759,0x717f1389),LL(0xa8ecb8d4,0x2146493b),LL(0x3e3c9d15,0xd51427c9), LL(0x77c1c32f,0x7ec8afc4),LL(0xf63856de,0xfa0a577e),LL(0x32750be7,0x95a6c860),
   LL(0xcaa33960,0x49b1bb22),LL(0xe647c5a4,0x45871855),LL(0x167d6801,0x6d71e970), LL(0xa2764f1f,0xbbeae386),LL(0x1284c39a,0xc7df73b8),LL(0xe8a8b423,0x3ddc63be),
   LL(0x6c2484c5,0x4cf0be08),LL(0xbff601e9,0x59028533),LL(0xcc241c87,0x94453c36), LL(0x0b18c6d5,0xb6e2065d),LL(0xfb4927d4,0xafde2953),LL(0x51875c38,0x69d1f3bc),
   LL(0x6709fe0c,0xc4bdbeb2),LL(0x2aa46589,0x86c2f225),LL(0x1c6d96c6,0x6e215f98), LL(0x1900c952,0x3b102255),LL(0x0b476bea,0x09b6f447),LL(0x1643c833,0x88877356),
   LL(0xe1f4c47f,0x897e15e4),LL(0x3deb58b0,0x9055b4df),LL(0x5d73f210,0xb189fd7b), LL(0x344e98cb,0xabbc3bfa),LL(0x17fe8809,0x355345a8),LL(0xb8a63413,0x848ea408),
   LL(0x0ee8588b,0x111d94b7),LL(0x1cae2230,0x5f7ba2ed),LL(0x8b428b68,0xd1119eb2), LL(0xc5901d8a,0x5ac72be9),LL(0x1a7e213d,0xd11c39be),LL(0xae1574db,0x5eba5216),
   LL(0x78e7a47d,0x2c6b4874),LL(0x03871f7c,0x5cbfeb97),LL(0x8dfcd4af,0xdb875c76), LL(0x6202ffc6,0x922461a3),LL(0xf1463db1,0x084e44fe),LL(0xa6b3ba89,0x55a890d4),
   LL(0x0d359302,0xbbb38087),LL(0x3bf7212f,0xe0c552b5),LL(0xce70eda4,0x6873aa54), LL(0xf0dcd6fe,0x42eb0476),LL(0x202c6fba,0xe354ae37),LL(0xbb182481,0xd2eeb9a4),
   LL(0xac565505,0xfb13f470),LL(0xe0d9811e,0x8e6be184),LL(0xb7fb0ea8,0xc727bd2b), LL(0x2a49df43,0xe0cc92d4),LL(0x6c43c82b,0x551ef99f),LL(0x449a2247,0x50d3acd2),
   LL(0x8959dc53,0x2e246f36),LL(0x3f449349,0xc6882bd2),LL(0x52d56c22,0xfbcda062), LL(0x037c14b7,0xaf1525bc),LL(0xf51db774,0x79f2c364),LL(0x397c0db5,0x51d5156e),
   LL(0x88ea2408,0x15fdfc20),LL(0xc31c09fc,0x39ad6859),LL(0xa7479cce,0x23bf58d5), LL(0x6df4616c,0x200420e7),LL(0x0bb71847,0x6516f75d),LL(0x60976e77,0x478ec1a4),
   LL(0x2d623962,0x38f56704),LL(0xb8e018b4,0x1d5927f2),LL(0xb7159417,0x60f41495), LL(0xe71a9c08,0xa531058e),LL(0xc4427fd5,0x038edfcf),LL(0x16bcd9b9,0xf40f0c3f),
   LL(0x4e8b0660,0x5c3875f8),LL(0xba508d58,0x4835397b),LL(0xa986f017,0x28f855c2), LL(0x947b88a2,0xab86055f),LL(0x4038096a,0xf6aab3c8),LL(0xc66df877,0x5babb42e),
   LL(0x218251e9,0xbc40bae0),LL(0xff92ffab,0x86a6f175),LL(0x15c92a68,0xc538fbe8), LL(0xce8e05fa,0x94b037e5),LL(0x38df007b,0x1e23b8c7),LL(0x6dfc21b0,0xcf60919b),
   LL(0xeaab0281,0x88c58a61),LL(0xfd515dfa,0xbdef49bd),LL(0x7b29a823,0xafb4049d), LL(0x51fb148d,0x803b4141),LL(0xefcde52e,0xa607366b),LL(0x756dd392,0xdc243912),
   LL(0x07b07222,0x6c66ac7c),LL(0xabbb5a7e,0x7a2251bf),LL(0xea506766,0x6898eec1), LL(0x0c8100c3,0x8ee773e9),LL(0xb5ac1e10,0xe267812a),LL(0x45ea2616,0x51d3e17e),
   LL(0xc5d7bcee,0xec435b31),LL(0x692565f6,0x45068195),LL(0xee8a64da,0xabbe69a5), LL(0xf3259fd4,0x10b23048),LL(0x6cf98be2,0x0df0afcf),LL(0xce0764fa,0xf317669c),
   LL(0xab0c2496,0xe0cad6d0),LL(0x3e58a13e,0xad9d4a4b),LL(0xcb2f20c6,0xb4009b9f), LL(0xfbd04c92,0xf586adaa),LL(0x65e4a2c2,0x27f06af1),LL(0xe6ac820a,0x5efc658f),
   LL(0x24c1a379,0x14a80bf6),LL(0xab134b06,0x0879a1fe),LL(0xe46fa8bc,0x6425c07b), LL(0xf328660c,0x626b781e),LL(0xa6f4006e,0x32c537fd),LL(0xa5cee8a3,0x3de55b30),
   LL(0x45111281,0xd072005c),LL(0xc6dd923a,0xc38687f7),LL(0xe7e55c30,0xc5493c3b), LL(0x1e7bcaf3,0xc801a0dd),LL(0x66af003d,0x7d896378),LL(0x5c5fc040,0x88c92d2e),
   LL(0xb5ac249b,0xd7ac3494),LL(0xd031be50,0x646f3e8c),LL(0x925d3ebe,0x6a6d2984), LL(0x0e7ed329,0x4696646d),LL(0x1822a37a,0xe4167a29),LL(0x3f6245b2,0x9419d1d2),
   LL(0x489578d4,0x752de436),LL(0x6c0f251c,0x0ade8b1c),LL(0x59ad5609,0x29b34ffc), LL(0x857c870b,0xd4af9bf6),LL(0xe6b0527e,0xb779fab3),LL(0x70ef261f,0x20970dfe),
   LL(0x392b4efc,0x2bcf9ad9),LL(0x8a796449,0x24ce4be0),LL(0xdc2b03ad,0x099a3cad), LL(0x223fe0fc,0x35749d6b),LL(0x8c2e618e,0xe9a66616),LL(0x3910e994,0xc7df5779),
   LL(0x14729183,0x4af80891),LL(0x2650cc8e,0x44458190),LL(0x587ad69a,0x738c0bd4), LL(0xefc4b5e7,0xdab8713a),LL(0x902363c0,0x7d6d6ae3),LL(0x45acff6b,0xad593898),
   LL(0xedc9ecd6,0x4c3723e7),LL(0x9731c5bf,0xc8fa6afb),LL(0xcc9bf5cb,0x162d7a77), LL(0xd28f51e8,0x623c7ce2),LL(0x45d36aad,0xb68fedf5),LL(0x2e95d0ea,0xc840c399),
   LL(0xa5542c7d,0xcc1674d5),LL(0x1fb770dd,0xc8ebdbdd),LL(0x0a3b16ce,0x7d70a1ed), LL(0x685db534,0xa74f8148),LL(0x37c0b157,0xa0eab0ad),LL(0xa3971e22,0xcd8063a7),
   LL(0x794542c1,0x9bded711),LL(0x3360a470,0xa56154fa),LL(0x79e1003c,0xf0bdb57b), LL(0x36c1f48f,0x48e694c5),LL(0x0eb1a568,0x3b71c023),LL(0x6090a9d3,0x4f574953),
},
/* digit=1 base_pwr=2^7 */
{
   LL(0xb3da4d9d,0x9cb08eb5),LL(0x8c573da9,0x95707fee),LL(0xe85fccd9,0xa5ad07fb), LL(0x769af379,0x5d3b50b3),LL(0xb9761267,0x79e01d7e),LL(0xee770876,0xdd24ac5b),
   LL(0x492d1c7c,0x7808129c),LL(0x6b626f03,0x97ad42c7),LL(0x5d974059,0xbf9a52c8), LL(0xeaa619c8,0xa888802d),LL(0xbbe19555,0x171450c4),LL(0xb980305f,0x173483c8),
   LL(0xa8a831fc,0xa5fa35ab),LL(0xf7ffad71,0xe4f05b10),LL(0x3f8d1cc3,0xc1ed613b), LL(0x6c60e44a,0x9d0f2a96),LL(0xfc64d797,0x8ece4c89),LL(0x8a315b0e,0x6fbdadb4),
   LL(0xc352becd,0x824863da),LL(0xa2c14e2f,0xd5d43090),LL(0x46bc9bb9,0x85272175), LL(0xaa9a25da,0x75eb8695),LL(0x85c7c93c,0x4a05c961),LL(0xa9043492,0xcb5861df),
   LL(0x62da9c26,0x0a04a3c6),LL(0x6e6542fd,0x9907614b),LL(0x8e9ada48,0x39c0b3b7), LL(0xe51bb6c4,0xf7dffbcc),LL(0xe566dea8,0x0316edb7),LL(0xb0ac507f,0x8b1a59cd),
   LL(0xd1581b68,0xbfa7fc59),LL(0xdd5c0d6d,0xdd9906b2),LL(0xfadb3203,0x50570aa0), LL(0xb207b0c6,0x026b8114),LL(0xcf4b11ff,0xf3829145),LL(0xc3abfb69,0x97eef9fe),
   LL(0x82a8e3aa,0x3ced44cc),LL(0x8cfe1e20,0x991f72d1),LL(0xeed3e6cb,0xce242a8b), LL(0xf6a15c38,0x70905e2f),LL(0x1a2a9a53,0x0d5db763),LL(0xa414c46c,0xe60a8854),
   LL(0xaf1c73a9,0x0b6b1174),LL(0xc0678153,0xaef24601),LL(0x63cc9eb3,0x5dac22ed), LL(0xa0f27800,0x80606754),LL(0xf3dc2a70,0xe3154503),LL(0xec01fbfa,0x6ba23ce4),
   LL(0xdc5f2b20,0x53cf654b),LL(0x3e7d94fb,0x4203ff82),LL(0x43547670,0x1f39db20), LL(0xf6716d31,0xbf42ccca),LL(0xa8ef535f,0x7be941f0),LL(0xaaac51de,0x890245d9),
   LL(0x2c853305,0x3aba1480),LL(0x43e3e158,0xd4554887),LL(0x135360e9,0xcbae9038), LL(0x6e23c4f0,0xe8d3054b),LL(0xe66d94c6,0x19bd813e),LL(0xafcd895d,0xf4397ee0),
   LL(0xc23cf227,0x5526335b),LL(0x6e23ef80,0x1053b920),LL(0x06c75842,0x656efa8a), LL(0x6397c380,0x7b7d1a0a),LL(0xe8cbe845,0xfea5cd22),LL(0x47a7e673,0x620c0df1),
   LL(0x7aaf742d,0x8d07f60f),LL(0xeda0b762,0x74469872),LL(0xe237de09,0x3f90d096), LL(0xf4d8b55e,0x679535d3),LL(0xbc04ae92,0xa463084f),LL(0xf1202dff,0x2a0d536d),
   LL(0x3ac85f57,0x27649c81),LL(0x235fa2cf,0x8b41bf96),LL(0x9df79933,0xe866618f), LL(0xadc59139,0x6d476615),LL(0xd404e889,0x09b1345f),LL(0x0f7eb70f,0x6bd51082),
   LL(0xdc3366d9,0x63fe2f64),LL(0x7732bce9,0xf48f309a),LL(0x6f637459,0xbcd3fb82), LL(0xa9255eaf,0x6d9b5c4b),LL(0x2e1fdd05,0x01ff5ae4),LL(0x72e33bef,0x66e45b33),
   LL(0xe080d6b8,0x1d521fcf),LL(0x14f56cf4,0x4e5d581b),LL(0x9a50ece4,0xea6d7579), LL(0xe1c8a0af,0x2d34a6d6),LL(0x28e5f588,0xd459901c),LL(0xf52fe058,0x8d41ab2b),
   LL(0x436cba75,0x780fd8f6),LL(0x09f798f7,0x7216f9a2),LL(0xa3720506,0x72c9e7a9), LL(0xfa919d28,0x72cbdd01),LL(0x053500b6,0x045474ba),LL(0xaf3af74d,0x6630329d),
   LL(0x5fa22efe,0x548b87c5),LL(0x62b464fe,0xcf449871),LL(0x7637a7dc,0x1eb2ccd0), LL(0x693b51ae,0x6ae5b6b1),LL(0x0a635765,0xcd15e6cc),LL(0x32160012,0xb8139da2),
   LL(0x88eff9b5,0xf4477fb6),LL(0x9b8264aa,0x1abe49f0),LL(0x6f59e67b,0x2ed2f447), LL(0x598889ff,0x71460c4d),LL(0x94f3f05f,0x04b1e2ed),LL(0xf04b29b7,0x9b1b5eef),
   LL(0x0a36430d,0x2de30220),LL(0x00e9adf9,0xe1df5595),LL(0x1d56ee5c,0x4642ab6f), LL(0x321cd0e6,0x32675e76),LL(0xbd822a36,0x37eaa735),LL(0xff9b1023,0xb807c5a0),
   LL(0xb77035f4,0xc1637964),LL(0x2e67810c,0x627deaa6),LL(0xe68721d1,0x11c73b51), LL(0xddd6a178,0x00b6c78f),LL(0x43326826,0x63414a2f),LL(0x9370cb95,0x3f3157a5),
   LL(0x28cac9b8,0xa6f2bedb),LL(0xdb9143d1,0xa8d660f1),LL(0xff740aca,0x28161efc), LL(0x81b9607f,0x6a8c99b0),LL(0x3379b6fd,0x6f6991fb),LL(0x54b4b97f,0x3463ae20),
   LL(0x5901b54f,0x85e79db9),LL(0x55a0c83e,0xffbd105b),LL(0xef647c2f,0x3b871868), LL(0x18cedd8a,0x5aa96847),LL(0xb6ac9ecc,0xabd9b3d6),LL(0x93bddbd9,0x707f61d7),
   LL(0xfcf1d3f7,0xa3e82462),LL(0x0b854fa4,0x619933c3),LL(0xd12e3bed,0xdceb86e0), LL(0x5ac77287,0x8e500715),LL(0x65cf6840,0xe5cb2c78),LL(0xc874ed06,0x6a2fcd3a),
   LL(0x75c74d72,0xa9af4af2),LL(0x7df05ffc,0x0f341f15),LL(0x31713e68,0x13bae06e), LL(0xd8e1e3c0,0xbeb2661f),LL(0x9d5b5ff4,0xffe49ef1),LL(0x61d5cae6,0x04eac206),
   LL(0xedd6a166,0x2f4fee08),LL(0x2eb29647,0x646cc729),LL(0x4f0f6cd5,0xb5a3c6bb), LL(0x2779dead,0xd1a9c06b),LL(0x4aed0996,0x066e3452),LL(0xbdc98d70,0xcbab5149),
   LL(0x88f89e9b,0xf4185c2e),LL(0xf9985e9e,0xc8e5dd64),LL(0xbb24492e,0xb8d5b8d3), LL(0x0fab80d7,0xad79a272),LL(0x5b3133e3,0xb766095b),LL(0x8b2b7fda,0xc83eaf38),
   LL(0x4448648e,0xedc5a97c),LL(0x6ae43bdd,0x2658d407),LL(0x48a8e439,0x766de9f3), LL(0x127fb314,0xca58b583),LL(0xb49667b5,0x65246221),LL(0x99398ac5,0x5b180e85),
   LL(0xcdf0d01f,0x75910da1),LL(0xae61cd5f,0xb0fbade7),LL(0x17e45ebd,0x0e94c843), LL(0x72944c2a,0xe13b1cad),LL(0xa8546c01,0x80e5d5cc),LL(0xd98df1d8,0x8cb5077e),
   LL(0x8e8cdd4c,0xc73d42b7),LL(0xf107a930,0x3a4ade51),LL(0x0da1e241,0x775acdae), LL(0xe97e3d45,0x1919504b),LL(0xb09e94e4,0x15d0774a),LL(0x4d58aaf7,0x3bd3c7f9),
   LL(0x7998dd8f,0x07eb2872),LL(0x2cf2179a,0xe9813bff),LL(0x06921a45,0xfa9171da), LL(0xcf58180f,0xb8c04797),LL(0x8721e9b4,0x31db6313),LL(0x91b2c60c,0x381c63f7),
   LL(0x66013a01,0x11b0d72b),LL(0x6c92f9e0,0xaf00d7aa),LL(0x47dcc4fb,0x2a4a91be), LL(0x0fa92641,0xb049b8d4),LL(0x448bd3a0,0x069abbac),LL(0xe74c7a8c,0xca288262),
   LL(0x385633ba,0xdffd6286),LL(0x19ddfcd5,0xdad4d394),LL(0x4bc0ef1f,0xe90ccfaa), LL(0xc1fc6f55,0x016bf2e9),LL(0xab3c2edb,0x879b641f),LL(0x89695f01,0xd5a122fc),
   LL(0xc302a272,0x12a1065f),LL(0x782cef51,0x368397a3),LL(0xbc4ad296,0xebac9fb3), LL(0xe0227d62,0x0caffee7),LL(0x88e685f2,0x63679dcd),LL(0x6aff1b35,0x85ced1a6),
   LL(0xa3155a1c,0x8b4e0205),LL(0x8310b45e,0x53afa029),LL(0xce35c0c9,0xa7571202), LL(0x06f892ac,0xa0b7a35a),LL(0x796af8cf,0xe4fbab54),LL(0x9d086c19,0x14ac6bfc),
   LL(0xc6267432,0x2e6b041b),LL(0xd1a15019,0x60267755),LL(0x50c3b7c6,0x7d6c36c2), LL(0x9bbe459a,0xf25437d5),LL(0x23468209,0xe1eff41c),LL(0x046c931a,0xe95608fd),
   LL(0x3a8012a2,0xd410363c),LL(0x62137413,0x2c5a7c08),LL(0x711e77de,0xaeca7f42), LL(0x08ce5358,0x492fb4fb),LL(0x48f0de03,0x715abe73),LL(0x5f625701,0x726faa61),
   LL(0x739b9455,0x2262f752),LL(0x511abdd2,0x8cd71649),LL(0xdeb366db,0x6765aca3), LL(0x3e0a005c,0x686587e3),LL(0x33f1710e,0x029b3605),LL(0x828cc559,0xf7b50e02),
   LL(0x66442cbd,0xe50f5b8e),LL(0x84763a5c,0xf27cc25d),LL(0x3120d35d,0xa92103eb), LL(0xfcef0800,0x91c0542f),LL(0x2d72f1ce,0x8bd12bb5),LL(0xf8086d1d,0xe97d290f),
   LL(0x4ef89505,0xab4b24f7),LL(0xfe72dd92,0x2bc14e79),LL(0xc60af192,0x94226552), LL(0xfd45454c,0x474c2b8d),LL(0xc7753a83,0xac99b5d3),LL(0xc8f1d3ae,0x6946a61d),
   LL(0x0c6a72ee,0xcb9c16c5),LL(0x64f9d8b2,0x96e95912),LL(0xdaf45f48,0x7c59cb24), LL(0x79b6515b,0xe520df3f),LL(0xded63e3c,0xb79c6ab5),LL(0xec685103,0x49767609),
   LL(0x6d892dae,0xbae68558),LL(0x3e48a96b,0xf07e8b59),LL(0x49e4d777,0xa7115219), LL(0x56e5e52c,0x724d7d29),LL(0x6fe58f07,0xb2483de9),LL(0x67836a1d,0xb28ffc89),
   LL(0x507bc006,0xaebcc451),LL(0xce8569b0,0x27b4e0db),LL(0x02bf59f6,0x579ae452), LL(0x5cff7495,0xc53b052a),LL(0x4cc9201b,0xc816b206),LL(0x540dd63b,0xd3c7350a),
   LL(0x26022a40,0xd9df51ca),LL(0x92be30ea,0xc1842bc2),LL(0xfa161a61,0xcdd5cee3), LL(0x554031f4,0xdaefdb75),LL(0x4935fd8a,0xd28afc65),LL(0x0a32f4c9,0x439d6d68),
   LL(0xeb1dc395,0xbe55112c),LL(0x531a3e8c,0x90def44c),LL(0xf77a7b59,0xf5b9ef12), LL(0x4c8ce124,0xc3ec1edf),LL(0x1c16391a,0x4edffe21),LL(0xe9ada0cf,0x40b8ec6f),
   LL(0xae68790a,0xc586619e),LL(0x0e0ba4ca,0x5bbe744a),LL(0x70ee1ec0,0x21a82b73), LL(0x7b9850f1,0x82066320),LL(0xee9bcc2c,0x2a7536c0),LL(0x9d661f43,0x9632c632),
   LL(0x1724c849,0x6d3b65ad),LL(0x8cfb77ce,0x3bef3b0e),LL(0x71fb5452,0x670e88e5), LL(0x1035a015,0x82d57c5f),LL(0x23736115,0x340c145b),LL(0xb1c2dc13,0xf1a704ea),
   LL(0x0ed1952e,0xd3aa33da),LL(0x9a32b9c9,0x459e7596),LL(0x15d05a80,0x15dcd976), LL(0x75895b93,0x85eea023),LL(0xc2fcd91d,0x8daa32c3),LL(0xba5669ce,0x2290ba90),
   LL(0x66e7e5af,0x85ac1742),LL(0x05bff18e,0x9ba521e0),LL(0x556a9133,0x9f081540), LL(0x929cd03f,0xb913fd04),LL(0xf46a514d,0x3065b31b),LL(0xe351ba04,0xe2fe761d),
   LL(0x139c5933,0xaf1e2230),LL(0xc129bd42,0x0c6c8096),LL(0xf4cf79c9,0x56f406b5), LL(0x4e86f542,0x772cb3d0),LL(0x843d10bd,0xbe007c1b),LL(0xe8e6693c,0xd3a201f5),
   LL(0xcfee362e,0x5a6556de),LL(0xa9264918,0x00dafb80),LL(0xc2ab4fcd,0x608dfd33), LL(0x091a15a3,0x29946cea),LL(0xfcbef4a0,0xb8caca39),LL(0x27c3eb56,0xb0c39fce),
   LL(0x21b66fdb,0x9fad053d),LL(0x74192d67,0xb529c9b6),LL(0xf21d3c4f,0x34c31800), LL(0x33cb3cc5,0x80797948),LL(0xb20d1f9b,0x18c66091),LL(0xe49755a6,0xc9fe30a8),
   LL(0x4beb71b8,0x90ae4d45),LL(0x4083ee91,0x2fc24099),LL(0xab8d6bb8,0xbcb9c603), LL(0xbca4c1b4,0x72d50046),LL(0x1a6be188,0xbc5fbf1c),LL(0x3da03c7d,0x4bc93b91),
   LL(0x3931092b,0xcbc53ee8),LL(0x9af003b5,0x455bb703),LL(0x9ac0e7d7,0x8da6de21), LL(0x9ccbcb20,0x6a4900cb),LL(0x73963c8e,0x3146af37),LL(0xd86a3d10,0xfc9469ab),
   LL(0x8078ecda,0xcd49df84),LL(0x45a4ff92,0x9caf52e4),LL(0x5a8fc8e7,0x4e8bd802), LL(0xdfeff51d,0x4339ea5f),LL(0xc4433bd8,0x25169385),LL(0xebe52ca1,0xdb3a7ae3),
   LL(0x09e9bb4b,0x8cb174c9),LL(0x224cf62c,0x44657124),LL(0xe1f6dbe1,0x70c991d3), LL(0xb10cd328,0x561771b8),LL(0xb5242c51,0x4e08ed68),LL(0x9e7d3968,0xd3eb6498),
   LL(0xbef053e4,0xaef71aa7),LL(0x49740ae4,0x3c3c52b5),LL(0x8eca22b9,0x5b6f510e), LL(0xf2645282,0xf1b861ae),LL(0xe96d07f6,0x670312de),LL(0x0c04dcfb,0x413388fe),
   LL(0x7734d6a7,0x7b924e76),LL(0xa86fc636,0x3616aaaa),LL(0xb9d584ca,0xd600f983), LL(0x67c54321,0x6324b5b2),LL(0x843f1742,0xee50247c),LL(0xde7f93af,0xf92e7c7a),
   LL(0x0aab4e71,0x1e668209),LL(0x236d6f67,0x76b4b879),LL(0x1fe85adc,0x8358a8ae), LL(0x550e1974,0x7567b0d6),LL(0xbd316399,0x3d193081),LL(0x92a0d80d,0x383ef613),
   LL(0x26ad5111,0x04a8ee34),LL(0x2efe5eeb,0x56e28325),LL(0xcd888cec,0x88e3f877), LL(0x0de0fb50,0x9ce73666),LL(0x7be9e17e,0xc84bc9e9),LL(0xbc67f215,0xf3a496a6),
   LL(0x41f7e964,0x548c525b),LL(0xafc8296f,0x18b03c78),LL(0xc171d822,0x14e4c1df), LL(0x03b39f89,0x090e2c4e),LL(0xe157c072,0xaf0feb62),LL(0x77227f20,0xe10a4331),
   LL(0xc72c5d07,0x40314b18),LL(0x1407f165,0xb2b01e30),LL(0x512cd1ec,0x4ad000d1), LL(0xcf5ce913,0xd7310420),LL(0x865fb1f8,0xa38701df),LL(0xb1db3c80,0xac4c2674),
   LL(0x36bf878b,0x07697637),LL(0x6ddd10b4,0x662c148d),LL(0x8d97a563,0x75b55463), LL(0x7f1198b4,0x34bf8bd9),LL(0x643001f6,0x31155240),LL(0x0d3f4f13,0xda176596),
   LL(0xcdcf7cab,0x44c61a3b),LL(0xecb257b3,0xcda79f8f),LL(0x8a15332d,0x7f701080), LL(0x4472224a,0x6f48e031),LL(0x31fafaff,0x3e89e0e6),LL(0xbfc3f192,0x1f066674),
   LL(0xa2df8c34,0x132f7d0e),LL(0xbc28bb95,0x66d020ad),LL(0xa0c4325a,0xb942cb64), LL(0x07a82d3a,0x4922c971),LL(0xad5d63e9,0x22ce594b),LL(0xc2e553d3,0xdf369ada),
},
/* digit=2 base_pwr=2^14 */
{
   LL(0xa96c5146,0x773311a3),LL(0xafc15c15,0x8ff8dce1),LL(0x3e382d13,0x59b9e79c), LL(0x3e0a552a,0x0ed1f12e),LL(0xce4662af,0x487503ec),LL(0x7000683e,0x624bc8ab),
   LL(0x58cb54ac,0x59ffacde),LL(0x492dc39c,0xef64164f),LL(0x337cd1ec,0x16b13c76), LL(0x47ca44a2,0x1e2033a7),LL(0x193743a8,0x8477b54e),LL(0x8e016823,0x557a7155),
   LL(0x5da83913,0x96d2f426),LL(0xc427cf5a,0x41062046),LL(0x1dd15a14,0x8016f670), LL(0x78840ddb,0xacd78288),LL(0x7feae4af,0x21529372),LL(0xc77acb08,0xdb65ffee),
   LL(0x9f1f177a,0x23d068f6),LL(0xbec8525d,0x508a8bed),LL(0xabb44bb0,0xa9de104e), LL(0xaef83c7c,0xb1ad4ce8),LL(0x9b842889,0xf045e04d),LL(0xa931cb25,0xca1ba541),
   LL(0x87b11b0a,0x377c8c86),LL(0x70d0248a,0x4eb952ff),LL(0xec2537f4,0xbde06ebe), LL(0x48f400cf,0xfc17552b),LL(0x84cd18c3,0x73e1b8a5),LL(0x4e54b85c,0xa3a8b445),
   LL(0xa64a281a,0xa9a806ba),LL(0x7b3bc32a,0xbc16fb01),LL(0x8ab7aede,0x8d2a8d5a), LL(0x2d7db806,0x7db641a7),LL(0x2d8c945e,0x8afe3e84),LL(0x1aefc1de,0x3e700391),
   LL(0xf8271efc,0xd9e4e55a),LL(0xecac958d,0xd50a0d2a),LL(0x3293a393,0xc58b1b30), LL(0xca65ac88,0x3dfed217),LL(0x13dbff4e,0xe308b3d8),LL(0x7a70bcfa,0x185024cb),
   LL(0x73d7e444,0xd7ca2cc9),LL(0x5d9cfe00,0x2ab7d94e),LL(0xd1333221,0x24a41faa), LL(0x6b27127b,0x9ce62b32),LL(0x20aca054,0xf66d0897),LL(0x079bb62d,0x41f3f259),
   LL(0xc11087c3,0xd74686ed),LL(0x99d4d141,0xd199b1e3),LL(0xdb7b97dc,0x2e3a551d), LL(0x8910fa3f,0xb59f58e6),LL(0x7ecfc3ef,0xdef98365),LL(0x349b65ff,0x29ec545a),
   LL(0x0598f0ef,0x7fc08cb5),LL(0x4400c739,0xa5208904),LL(0x4c9f98b7,0xb02f25b3), LL(0x901fc002,0x4d9901cf),LL(0x49babf27,0xdde7d571),LL(0xf2d65888,0x13b3e92c),
   LL(0xb6dc4b6d,0xd55436e9),LL(0xc04a2f37,0x746c5407),LL(0x4485d943,0x84d53bb7), LL(0x16b241f8,0x0ba98680),LL(0xf8feecce,0xbdd3b788),LL(0x87dd6420,0x1801aa7c),
   LL(0xcd59a045,0x047bdddb),LL(0xb8464ecc,0x0b74793b),LL(0x37cce6df,0x95661e95), LL(0xfc3170e4,0x20d1535b),LL(0x68c8b253,0x14546482),LL(0xc42e9c3d,0xbae5cef0),
   LL(0xe87e6d86,0x0201464f),LL(0xbd67f280,0x1bab0e73),LL(0x64600ebe,0x69b188cb), LL(0x5f9c98da,0x3641dea5),LL(0x9aa645be,0x94d2a2cd),LL(0xbd545f72,0x1ef74b42),
   LL(0x914d779c,0x19faba59),LL(0x27ba5f19,0x90007928),LL(0xe1ee88ff,0x46f12e72), LL(0x726a44cc,0xa9e0848d),LL(0x285108c9,0x13d4e943),LL(0xdf77e165,0xb5721271),
   LL(0x492ca661,0xa444123a),LL(0x8e0769ae,0x9ff4a82c),LL(0x718014db,0xc6786b05), LL(0x19f577ac,0x702027a5),LL(0x56e9b5ef,0x349ca959),LL(0x4f6e4b5a,0x1f746299),
   LL(0x4f0efa59,0xb23ca413),LL(0x15a3b35c,0x003f4a5b),LL(0xe4a4d655,0xa0fe8a5a), LL(0x9e5467af,0x044479ec),LL(0x6d4b3c89,0x5abb290c),LL(0x1c2f5364,0xfd845d3c),
   LL(0x6837a39b,0x21cdab37),LL(0xf18d4f99,0x7037f403),LL(0x6621328c,0x964d48a7), LL(0x930c4f4e,0xd70553ec),LL(0x5274352b,0xca35a6fd),LL(0x6199ce61,0xff584ff2),
   LL(0x479ebc39,0x60a7a4ce),LL(0xaa703602,0xaf3c24ce),LL(0x89689136,0xfe79ffc9), LL(0xc7ede59c,0xaab98ecc),LL(0xbf5659bd,0x933f24a2),LL(0x1d702fc6,0x68fd0d06),
   LL(0xedbb3147,0xbb51d576),LL(0x5d358026,0x038e2a2c),LL(0x45ed9fed,0x10fef0db), LL(0x55be7ba9,0x78b92751),LL(0x69bd1f87,0xae493919),LL(0x186699c2,0x632e8ffd),
   LL(0x9085fbe9,0xb0b4b604),LL(0x808be4f0,0xf93f9bf1),LL(0x6d8bf009,0x2b58f977), LL(0xd6188f6c,0x020fe5ea),LL(0xd25805ca,0x85d4a860),LL(0x07bb3697,0x7547f724),
   LL(0xe0b4ea32,0xb225b950),LL(0x33f7d58d,0xbc6e0982),LL(0xcb7418f4,0xb81e161f), LL(0xc0020282,0xfa1572b5),LL(0x8ab14fe1,0x3fababab),LL(0x08b2a18e,0x3a033779),
   LL(0x6495bacb,0xe816e0fb),LL(0x5d8888be,0x238bd498),LL(0xe3df2fc2,0x7ab0b8ca), LL(0x978404d8,0x88ee40ec),LL(0x26a463b7,0xbc3d1a07),LL(0x65d21535,0xa4c92dcd),
   LL(0x2928f5f1,0x819bd554),LL(0xbf6255d1,0x82906c78),LL(0x406326b1,0xc710bf1a), LL(0x09f4cd28,0x3a9bbfcb),LL(0xded76f80,0x83e41584),LL(0xbbe53844,0xca1c61fe),
   LL(0xae17eaf5,0x80c38892),LL(0x61b16da4,0x3670cf14),LL(0xa795a9c9,0x9f9ddd50), LL(0x1255afff,0x6f1841d7),LL(0xb69e10fc,0xc733cd3f),LL(0xdaf931ce,0x515ff527),
   LL(0xcc8a9a7a,0x14969bd6),LL(0x2c1d2761,0x579c2fb7),LL(0xf09d2a18,0xb0bfb370), LL(0x073b22bd,0x0baf361d),LL(0x1a52d79f,0xb5e11353),LL(0x22d425c2,0x9d475b0d),
   LL(0xecdc5b6e,0x24d4fea9),LL(0x1731ad86,0x8b3af04e),LL(0x7dfe43b4,0x0a1c0e04), LL(0xaf2af7bb,0x5cf225bc),LL(0x69ef4bed,0xb0885ab2),LL(0x830333e4,0x90999bc9),
   LL(0x985d6058,0xa8c2e9f0),LL(0xd1b5a3f0,0x8e9e9162),LL(0x3e407711,0x1055feff), LL(0xaaba085f,0x5a33934f),LL(0x23b30a1b,0xf961622f),LL(0x6dafb1dc,0xb1a2ee28),
   LL(0xa448ee27,0xe49caf6b),LL(0x4560f7a7,0xb8552bfb),LL(0x93f90be8,0xb68fd6ca), LL(0x1f2499d1,0xbb85cfca),LL(0x093c01a1,0x1a51059e),LL(0xd231b79e,0xdf98052f),
   LL(0xc076155c,0xf696a525),LL(0xf93ce35a,0x3a519224),LL(0x2e706fd0,0x2771d43b), LL(0x51a451c7,0xde78cd08),LL(0xae569dba,0xc9a60017),LL(0x22a0b1b7,0xa3cafd39),
   LL(0x422b409c,0x4afddf96),LL(0x75f5eb64,0x6313a926),LL(0x626ac381,0xf9420ee7), LL(0xe9858ee9,0xfbe23e3f),LL(0x3c678a64,0x658013be),LL(0xf2ad2c6c,0xbbcfddc5),
   LL(0xd05334aa,0xf9303aca),LL(0xd7626b98,0xad3e45cf),LL(0x8ca3a1f8,0x56c81f5d), LL(0x4aab3342,0xe892a0b4),LL(0xa1db86fc,0xf8a3f244),LL(0x97566251,0xdee7ed3e),
   LL(0x5efced68,0x0c65a22c),LL(0x9ce653ac,0xfa181866),LL(0x1e590014,0xa46d7d86), LL(0xc79656dc,0x69d234da),LL(0x00d2b7a7,0x4dbb31d4),LL(0x3452ca10,0xc9636e5d),
   LL(0x0778d7f8,0x5850e572),LL(0x31b9202b,0x4323e55b),LL(0x5e8d8006,0xaea618dd), LL(0x999bf27e,0xe0f4408d),LL(0x8fab0cc5,0x4bd06762),LL(0x5a75c171,0xddb647f2),
   LL(0x452f35a5,0x6ce09fdf),LL(0x91938a04,0xe6f5d9dd),LL(0x4cab01a5,0x79821593), LL(0xdc0ff9f5,0x054bf7a2),LL(0x613b5386,0xdb52d927),LL(0xf35b7179,0x661fe231),
   LL(0xa035b792,0x8a4adcd0),LL(0x2790ce5c,0x854bed6f),LL(0xac3fa847,0x64b0eee2), LL(0xfeb32887,0x7b752cc2),LL(0x09fcfa29,0x5e34ee86),LL(0xac0cd2e1,0xd06594aa),
   LL(0x10995206,0x6151e607),LL(0x93bb033b,0x93532d80),LL(0x7a33bd69,0x0beac8c5), LL(0xfb8f97c6,0x6f85d5ec),LL(0x0517b3eb,0x661303e8),LL(0xbc3f62c8,0xe17cc0cf),
   LL(0x4f4d78fb,0x279803d5),LL(0x66c81c4e,0xca63bac0),LL(0xc854f59a,0x6ec15cd1), LL(0x0f159ecf,0xb345c1f6),LL(0x3763e854,0x84807c78),LL(0x8cc6374a,0x691abd9b),
   LL(0xe4f37f9a,0x77c0e93d),LL(0x611d93dc,0xc60578c7),LL(0x1aff93a6,0x7c98f95b), LL(0x5bb698ce,0x534f1e2d),LL(0xd8bad550,0x0cdc34b6),LL(0x16afd591,0x597794f0),
   LL(0x571c8a4e,0xb795a174),LL(0xe651a52f,0x15ce6b83),LL(0xf49edec9,0x9394c28a), LL(0xf091dfa7,0x0f74e17e),LL(0x65c44142,0xcc9e8180),LL(0xa4df670a,0xc33d77e9),
   LL(0xc1633318,0x44536321),LL(0xc0b9deff,0xefb782b9),LL(0xc6b86a19,0x24fda05e), LL(0xf868535b,0xba601337),LL(0x91c21fc3,0xd3d126a5),LL(0x0663ffb0,0x10b1b7b1),
   LL(0x7fa1153d,0x8162598c),LL(0x688e7c55,0xc3c97288),LL(0x89c8fc1b,0xb0c2872d), LL(0xa859461e,0xfbc2a074),LL(0x82f7d203,0x91591a5e),LL(0xa909de2c,0xe278dac7),
   LL(0x7d4e49c7,0xad689f7f),LL(0x940bf215,0x12072de0),LL(0xde694ba0,0x7ec61fdc), LL(0xfed04942,0x4a4ee787),LL(0x0f665159,0x0497d6c5),LL(0x94136075,0xe9e08707),
   LL(0x19850a32,0x391c4103),LL(0x50d61027,0x0ad15740),LL(0xd883b3e7,0x9ca3e6c7), LL(0xf0d758fb,0x76891ec0),LL(0x04d6f53d,0x89ae897b),LL(0xcffefb93,0x77e9b63a),
   LL(0x216ee99a,0xf201c95e),LL(0x4fb9bed8,0x65648551),LL(0xda5d4d62,0xda9c97f4), LL(0x6aab7149,0xfae41f2e),LL(0x56aa67d1,0xfb649545),LL(0xc024449c,0x493a5061),
   LL(0x4df4c709,0xd0aa84eb),LL(0x06a93789,0x69c9246f),LL(0x917b88eb,0xe19b7340), LL(0x9fd99349,0x7f7cfe6c),LL(0xa8ebc191,0x6c82e590),LL(0xade7110e,0x7c50c13d),
   LL(0x50ffdce6,0x17155bf9),LL(0x5281bb39,0xab2bb8dd),LL(0xc4ba3181,0xe73086fe), LL(0xa736d737,0x41d9f1f7),LL(0x3df7fbbd,0x831b8099),LL(0x3b6ffde2,0x566467da),
   LL(0x36bb3e58,0x6cb9c1b9),LL(0xb8a84147,0x450324ab),LL(0x8bedffc5,0x99535f56), LL(0xa7e14db3,0xb0a718ef),LL(0x304dcfa7,0x65a62f9f),LL(0xd623bb6e,0xfb2f289a),
   LL(0x9d546bd5,0x0888c3e7),LL(0x772582a9,0x1a97b3cb),LL(0x7645a4c8,0x50937134), LL(0x93ea1519,0x70d77533),LL(0x079ff6e5,0x5837b53c),LL(0x45bcc030,0x8f5c8745),
   LL(0x1370e46d,0x28549138),LL(0x0bcf90ad,0xf0c3a434),LL(0x639db090,0x4d42664a), LL(0x0158f802,0x345c315a),LL(0x0463e159,0xef82afb0),LL(0xd3e83b12,0xfcf2c419),
   LL(0x9e3fc0b3,0xe04ea5a2),LL(0x37fd1788,0x98b3901d),LL(0x2d603830,0xc6326a28), LL(0xbe5e9c25,0x6e6c5c6b),LL(0x0fc60fb7,0xac753036),LL(0x12b1dad0,0xc4cf690c),
   LL(0x46768f3f,0x273e19a9),LL(0x4414d65a,0x93aa3754),LL(0x5fabe262,0x4ecff706), LL(0x7ff1403d,0x6d9cfcee),LL(0x33284115,0xf6e4192e),LL(0x5e218033,0x80aa8768),
   LL(0x485363f3,0x0067ee42),LL(0xfab8b185,0x5fcd61f5),LL(0x8e63d384,0xe3c345c8), LL(0x23ff551f,0x80847073),LL(0xe339aa9e,0x6a804b17),LL(0x6bb64f96,0xb9142088),
   LL(0x7679428f,0x855916bb),LL(0x9c4cf460,0xbab9e9be),LL(0x55dad941,0x0c3bdf26), LL(0xfedb73be,0x6b8e8c3a),LL(0x0355a4c2,0x07fdbfa0),LL(0x5e3c1b3d,0x8e9137e5),
   LL(0x57927722,0xb4afd653),LL(0x5f997d66,0xd48cbfc8),LL(0x7719d80d,0x5125ce07), LL(0x654d2350,0x19d47bce),LL(0x4d7a3060,0xfc671097),LL(0x15d775f7,0xddad1eae),
   LL(0xc6a285a3,0x9e8ea16e),LL(0xc535adfb,0xc2f0c22d),LL(0x217f73ce,0x3fc464b9), LL(0x73a8ee71,0xc543babb),LL(0x0da708c9,0xce410235),LL(0xc2f48649,0x99849442),
   LL(0xc83b501e,0x6b420f3d),LL(0x642f062d,0xe08c411c),LL(0xf03d2cf5,0xc79fd495), LL(0x07236a83,0x2184fa72),LL(0xc9a0ea3b,0x31b0e4f4),LL(0x5b82a3b3,0xd7795d4b),
   LL(0x3fe60842,0xb2520494),LL(0x45bdeed9,0x751336f1),LL(0x9718fa41,0x1b66bb31), LL(0xda9a21c4,0x5a50a96b),LL(0x615dcdec,0x94cc1fb5),LL(0xe39c3066,0x556b641f),
   LL(0x15ea6075,0x69d00e4c),LL(0x83005a21,0x3e707e72),LL(0x767ca1f8,0x84e2a469), LL(0x3f8c4c84,0x94b82739),LL(0x81a2db60,0xe00dd19b),LL(0xd7cabd28,0x16f30fec),
   LL(0x6305314b,0xb08563f9),LL(0x7ac22e44,0x73d8fb6a),LL(0x55919896,0x9b969f2f), LL(0xa3460685,0xcc9dd3fa),LL(0x1226dbee,0x377b5d9e),LL(0xaaa83ccc,0x0b4bbd51),
   LL(0x61f910de,0x929c87b0),LL(0x5f2bdd34,0x0595e87f),LL(0xb587735f,0xa5b54a4e), LL(0x6ac0195c,0xa3bbb78b),LL(0x1d44758a,0xe6b38b87),LL(0xac01f679,0xfd575f0c),
   LL(0xb21a6382,0x223f54eb),LL(0x672c43f7,0xf6f79b9d),LL(0xb8b258f5,0xcf8054d3), LL(0xa119b37d,0x88ec08ee),LL(0xafda5f1f,0x4b0fd837),LL(0xad4546ad,0xc0dde1b4),
   LL(0xe673eb95,0xbf7186d6),LL(0xc9d6a6a8,0xa324e752),LL(0xc117d914,0x5ebf8238), LL(0xad6715dc,0x9c33641e),LL(0x1d597adb,0x99439b58),LL(0x9c6a2ef6,0x83ba580e),
   LL(0xb17b3816,0xbf87295c),LL(0xee1db90d,0x05c753ce),LL(0x2554bda2,0xa31eccaa), LL(0x81176cf2,0x66691105),LL(0xb4e7bb4c,0x67eb9bec),LL(0xc294982a,0x6443f703),
   LL(0xd0ab699f,0x86c4e0ba),LL(0x748ed429,0xce3d72f1),LL(0x53e24aed,0x0ed09a39), LL(0x4db6e8ed,0x0ac3cda5),LL(0xeed93b68,0x26b6ee99),LL(0x2d53c5f0,0x3707c0fb),
},
/* digit=3 base_pwr=2^21 */
{
   LL(0xc3bba334,0x549606d0),LL(0x013ca242,0x6dcb5519),LL(0xdcf78255,0x784d65cc), LL(0xdbd76716,0x73d0de0c),LL(0x1010fb15,0x2bc13bf7),LL(0xe8d77e91,0x21d2c3cc),
   LL(0xa396d29f,0xf8efa6ac),LL(0x767470ec,0x33ec678b),LL(0x7241f5fc,0x23f69526), LL(0xd2a4a560,0xd0338dfe),LL(0x9d30be61,0xce5baa02),LL(0x4db7b6e5,0xb39ee0e3),
   LL(0x4268d7ee,0x93adf307),LL(0x2c9a12bf,0xfcbae0f0),LL(0x534822a7,0xb76859b8), LL(0xf961bd84,0xab38334c),LL(0xd09e4e94,0xb17e54bd),LL(0x43d3a64a,0x0fe2b812),
   LL(0x9577e8d9,0xe17445a1),LL(0xd2d34121,0x330f6634),LL(0xa1aa8076,0xcea03741), LL(0xfcfafd3e,0xfddda6ff),LL(0xe05b9369,0x440df5db),LL(0xbb0d7ca7,0x51fbac8e),
   LL(0xc9b4ce1e,0x320accbd),LL(0x99292bb2,0x3798c1f3),LL(0xa402db93,0x8371a2e6), LL(0x65d07e1c,0x716af227),LL(0x23851a3e,0xd06eaa2f),LL(0x61f48045,0xc42ee506),
   LL(0xc40da094,0x538f7dc5),LL(0x54a44890,0x4c651fc4),LL(0xee59f9d2,0x80876d6c), LL(0x66c8ac59,0xad6aff6e),LL(0x257e1b91,0x32b4e3ba),LL(0xec717e05,0x957bedc7),
   LL(0x7459bbfe,0xe45216bb),LL(0x2ef69f90,0x3dac0bfe),LL(0x6161cb8b,0x2cc6a2b9), LL(0xf6e7e3b3,0x472e91fd),LL(0x304e8482,0x4d2e40f0),LL(0xaf1042aa,0xf4afc219),
   LL(0xd64583ec,0xffc53726),LL(0xeef5bce9,0x6b462381),LL(0x24829cb3,0x753a4bef), LL(0x404af02f,0x976d6897),LL(0x448a7822,0x93313219),LL(0x399e5d55,0x79449ea9),
   LL(0xc5ecabd1,0x3574a11c),LL(0x34e48369,0x11ee8559),LL(0x915873b6,0x7fa30b5d), LL(0x2ae5b9b4,0xe1a71608),LL(0x2e04edad,0x57296722),LL(0x30d27fbf,0x84cdfbff),
   LL(0xbe3bd4b2,0x0c29de57),LL(0x2898b5b6,0xfdc6a64c),LL(0xc8bb55be,0x5604fc5b), LL(0x2e98dac3,0x1e6f616c),LL(0x8e9baefb,0x494a5143),LL(0x5501fe87,0x251c23c9),
   LL(0x6cdda2c2,0xebcbef56),LL(0xe3ee721e,0x85b1e422),LL(0x9dcdd732,0x32bcca8f), LL(0x72410bf8,0x182e8ce7),LL(0x36f1dca3,0xaa7f4322),LL(0xdcd402ab,0x893d6417),
   LL(0x43cd1278,0x87f47e84),LL(0xe1831ddd,0xc8b878ac),LL(0x093ba446,0x4d0a3597), LL(0x40080bf9,0xafc47c89),LL(0x3832f14c,0x3157bb67),LL(0xb8eb8baf,0x2abb24b3),
   LL(0x063399bf,0x30752625),LL(0x2979063f,0x47abe9c9),LL(0x55165854,0x01c9def8), LL(0x6fb4555d,0x0bab44d6),LL(0x567eb3a7,0x8a53411f),LL(0x726f9e4e,0xb8aab552),
   LL(0x2440a85b,0xad782e43),LL(0x2fbb1ce6,0xfa5a74b7),LL(0x1ae20947,0xc49e442b), LL(0x772117aa,0xd3fca276),LL(0x6115a3ed,0x303be2dc),LL(0x1a6f32b8,0x3b808eb6),
   LL(0x71faca97,0x673ae260),LL(0x086041a0,0x32f3b688),LL(0x1367ca73,0x26acc3be), LL(0x8c66b759,0x48400913),LL(0x9b73c0e4,0xc88e5241),LL(0xd44b961b,0x17a8a61d),
   LL(0xe56aeb1a,0x0a3a7387),LL(0x461fdd3f,0xa4010610),LL(0x016484bb,0x601e6822), LL(0x04bbe3e8,0xde403502),LL(0x9f071625,0x6a1b8c7d),LL(0xdd3f60b3,0xd1e9028a),
   LL(0x1496f098,0x7b626130),LL(0xbf161daa,0x62dffbe1),LL(0x9c44fcc7,0x401ddc78), LL(0xb9872e73,0xc81c3a33),LL(0xc751f26b,0x0e48c326),LL(0xa0829b6c,0xffaeec0e),
   LL(0x77a5b096,0xf5530fa7),LL(0x19734245,0xb008b72c),LL(0x848b3785,0x7462bc45), LL(0x6cf9e5f2,0x29fa43a4),LL(0x9854d78b,0xd8b27b11),LL(0x9097f3d4,0x6f8fe5a5),
   LL(0x30c97aa5,0x63ec5b71),LL(0xc6488713,0xfa8462af),LL(0xd7b335fa,0x51f2d273), LL(0xcd075cb1,0xf2891fbd),LL(0xd007c58e,0xffe2576c),LL(0xd8bc9492,0xaeec8e91),
   LL(0x6f905b81,0x305b6908),LL(0xbf19b3bd,0x5d023c2e),LL(0x22ae8649,0x9c503cbe), LL(0x25255b41,0xde8be78f),LL(0xe809567c,0xb1e4b2de),LL(0xb09e4638,0x986a2fd5),
   LL(0x213fb6cf,0x0b36dcd5),LL(0x834d7c84,0x5f340cf2),LL(0x42696c00,0xf835cd54), LL(0x71e8739d,0x7f02199e),LL(0xb4f38aed,0x0663d540),LL(0x1f7c9176,0xed21b27e),
   LL(0xbc7e8681,0x1810377d),LL(0x1545ee0d,0x242d18eb),LL(0x45281613,0x3360ba32), LL(0x808ebc75,0xb14e805f),LL(0xad26b94c,0x965c6c59),LL(0xebf7f5e2,0x24e3cd91),
   LL(0xc0623d66,0x493d0777),LL(0x05131c2a,0xafe840f2),LL(0xfe23fc98,0xb282512d), LL(0x3735071c,0x47342277),LL(0x105f0570,0xfe0900ce),LL(0xad56ac6e,0x7f80e17f),
   LL(0x7da80215,0x5509a924),LL(0xe2039096,0x0d1b8304),LL(0x32330f7c,0xb8d360d9), LL(0x866bf185,0x8b6b538a),LL(0x60f48919,0x65565d06),LL(0x4a88485d,0x457d3e36),
   LL(0xb4d530a4,0x21ffdbfe),LL(0x134babd4,0xb68a01ac),LL(0x3c545d8d,0xeac3a42a), LL(0x525bd6e8,0x2ec4920e),LL(0x11415be9,0x91d90569),LL(0xc859b9a9,0x27f2bf78),
   LL(0x3a492c73,0xdef766d6),LL(0xbd647210,0xd395535f),LL(0x7a9398e6,0x2cf03d25), LL(0xa10ca7f8,0x00d0d0ec),LL(0x05dde357,0x384e9692),LL(0x66d09f76,0x86dfa64d),
   LL(0x97c72efb,0x949c71bd),LL(0x3942aaad,0xeaebb176),LL(0x6b83fb91,0x2998f398), LL(0xccd511b6,0xa3fa1c7e),LL(0x8636cb8b,0x30452ba9),LL(0x0c63c213,0xfd01b34f),
   LL(0x654f4b1f,0x161634df),LL(0xba6b7612,0x786d3d9b),LL(0xa3f145c5,0x7b731aff), LL(0x81bf5f5d,0x318ec0fd),LL(0x28ec1847,0x5aac3946),LL(0xef3f2a56,0xb5b34960),
   LL(0x8a25550d,0xb8603afb),LL(0xca58e8f8,0x7d76ca1f),LL(0x52edd227,0xb124c5a5), LL(0x749a0bb0,0x4ff6045a),LL(0x166a72cb,0x9d1c4729),LL(0x18c9dd50,0xa1165c3d),
   LL(0x32ec01e6,0xaf916e4f),LL(0x74270c7a,0x9b18b797),LL(0x2bde0960,0xc31c12bd), LL(0xe3a52afc,0x6fdbedeb),LL(0x99dcd3e1,0x2fcf26c9),LL(0x2ec4570a,0x15d634a7),
   LL(0x02e98f6e,0xe27fb84e),LL(0x016e8348,0x19815362),LL(0x5ee51a3d,0x72e0c390), LL(0x36959aff,0x5f3ea99c),LL(0xe880a6fb,0xec5b0f2d),LL(0xac887faf,0xaecf25fd),
   LL(0xca7c485f,0xd0640e47),LL(0x8c826885,0xa0eae2dd),LL(0xf0750194,0xfcd0e169), LL(0xe05f5115,0x04c965ed),LL(0xd7ba9d0d,0xe9482a7a),LL(0xd261b83f,0xf2660cd3),
   LL(0xf53ab896,0x722e7324),LL(0x83e9fccc,0xcabeeaad),LL(0x0886fae1,0x332ff124), LL(0xfa23970c,0x0e8c2e7d),LL(0x433ab568,0x26f580ee),LL(0xcdcf0e3e,0x7cdcc1f4),
   LL(0x513d4695,0x83ac29d5),LL(0x9101ca09,0x525837bc),LL(0xe8e10f27,0xecd93292), LL(0xeea1edfd,0x312f9d84),LL(0x83f862a3,0xdf9354ea),LL(0x15909df4,0x0f9f120a),
   LL(0x5c61ed01,0x16fa2ab0),LL(0x816f7ce0,0xeab42f96),LL(0xf1cbe36b,0x1aca9b66), LL(0x1763d0c8,0x7fb3d70e),LL(0x7c64a8fc,0x5fb8567b),LL(0x45fb302c,0x08b686bd),
   LL(0x6270b30e,0x0124d430),LL(0xcf5b5235,0x566f1fd2),LL(0xbee07063,0x923da430), LL(0xe2c6736c,0x2ecebb06),LL(0xad6d6ee4,0x3a3fdafe),LL(0x82bd6945,0x16522430),
   LL(0xd6d11f78,0x0d162ae9),LL(0x69047eb7,0xddb375c5),LL(0x926e0966,0xd125f66b), LL(0x100fb84c,0x1cfafcac),LL(0x40c50ecf,0x1f4c7fd4),LL(0x499513b2,0xbd8ac3ec),
   LL(0x5feaf69f,0x937412eb),LL(0x641cace4,0x760d3378),LL(0x203e5367,0x9bcec28e), LL(0x45fa3774,0x457f3a0d),LL(0xdf85b5f1,0xec51fd4f),LL(0x37d174ba,0x47390235),
   LL(0x0a893990,0x7df7dee7),LL(0x8f08f682,0xa525af4c),LL(0xfae95f9b,0x9edbb45a), LL(0xcdc01949,0x58f3b29d),LL(0x773a1cde,0x9afdfd03),LL(0x8b16ab71,0xd6077f9e),
   LL(0x13611c0d,0xc96eabbc),LL(0x7581f50a,0x8af0d512),LL(0x93c963cd,0xf75a7835), LL(0x9060a883,0xb7487030),LL(0x17ceb72c,0xfc03c9b1),LL(0x79773be4,0x64773851),
   LL(0xd416e391,0x8f73e056),LL(0xb0289136,0xf9c29801),LL(0x1ebcb0de,0x85fd38b0), LL(0xf0af703b,0x9928f60a),LL(0xbbd1143a,0xd77a236c),LL(0x2b99b61f,0x120472d4),
   LL(0x5f4a504c,0x0e448282),LL(0x5cc9b39f,0x1fca16bf),LL(0xdf0166de,0x4e47e87a), LL(0x76644e8f,0xf11ccae7),LL(0x4b20ddcf,0x84113220),LL(0xe274c42f,0xae2e75b4),
   LL(0xf97bdda6,0x604a226f),LL(0xadbee2bb,0xadd35dc6),LL(0x986e403e,0x8b7db9ea), LL(0x6ad6318f,0x9007482c),LL(0x891688d9,0x22b19d05),LL(0x4d848a95,0xe974ab42),
   LL(0x88cc05d4,0x339df4aa),LL(0x5855f83a,0x14e21ec8),LL(0xd790d568,0xddcd7ff2), LL(0xcf09873b,0x5c8a6e40),LL(0x134b7a3c,0xb76d74a0),LL(0x07549e84,0xd05b80d6),
   LL(0xc5f26048,0x845e4b79),LL(0xeac9a450,0xcb338886),LL(0x9b807f9a,0x78f027d1), LL(0xb02ce220,0xf81fb903),LL(0xcd541a4a,0x2bc100d8),LL(0xb3d3cbfd,0xf10cf19f),
   LL(0xb198d5f3,0x74645ba6),LL(0xcebf9671,0x9fbabd70),LL(0x5d7a8816,0x57cf26bc), LL(0xc18901bd,0x415d9e95),LL(0x08448312,0xa71715a6),LL(0xe80f10b3,0xbb1d05fa),
   LL(0x488ffac2,0x6065bcb2),LL(0x21393e2d,0x02526899),LL(0x4f000282,0x259f0d34), LL(0xf12f0afa,0x10a4333c),LL(0x62a1dba3,0xaa0b50d7),LL(0x8cf13d7e,0xe09542dd),
   LL(0xc2ba51fa,0xf15ba587),LL(0xcdcc4e52,0x72f2bc15),LL(0xe3ca3356,0xd1c3673d), LL(0x3028612e,0xabbfd1f0),LL(0x9def1d98,0x4ab9792d),LL(0x4815b9f7,0x02e11fbf),
   LL(0xf804ca53,0x24940965),LL(0xd69ecbb9,0x5073162c),LL(0xdccb4f88,0xe078c3a3), LL(0x1f34a5f8,0x06f13d89),LL(0xfca14e1e,0xa848dd8e),LL(0x8c4c3fac,0xb989d0a4),
   LL(0x3f31ee3d,0x670aa037),LL(0xc8999458,0xf72dfbba),LL(0x4698301e,0x36e3a8b0), LL(0x1a579aef,0x26b32ad4),LL(0x98b2b956,0xdff0c7ec),LL(0x77953721,0xa732a2c4),
   LL(0x436265d5,0x0ff83c66),LL(0x06547c6f,0xfa5166e1),LL(0x5bfd456d,0x03018748), LL(0xf306cdd5,0x1b4ec2b6),LL(0x4b56f91f,0xe064b766),LL(0xa5805ee3,0xfdeb932d),
   LL(0xc49208e8,0xa583f200),LL(0x369d91c7,0x34a07e82),LL(0xa89b1fd5,0x3fe43f6f), LL(0xb439d920,0x76aecf08),LL(0x4547896e,0xf76a9539),LL(0x3019148f,0x3c69e2e4),
   LL(0x41fc2ca3,0x79e74959),LL(0x6a5af93e,0x09a99e14),LL(0x8293aacb,0x6610c0ad), LL(0x84dff17a,0xc56e9614),LL(0x0ff4cf52,0x0a4f0290),LL(0x35902cdb,0x50fd4a21),
   LL(0x7297ff6b,0xc526cdd2),LL(0x47459c97,0x2c8409af),LL(0xb7537567,0xbd3097f3), LL(0xb47675e9,0x5b29ab79),LL(0xfd4e344a,0xded7e0b2),LL(0x0eb1e2c4,0x40324c92),
   LL(0x9c381ce8,0x617df28b),LL(0x4475066c,0x6771e091),LL(0x82a57c2e,0x7ffb1259), LL(0x90dbf30a,0x6997a10e),LL(0x4603a795,0xa412e99c),LL(0x6da0de96,0xfb49f3ca),
   LL(0x24c43f70,0x28fb7939),LL(0x9b346573,0xa7d31444),LL(0x20545584,0xdf8f9b6b), LL(0x8a9b1477,0x54464b03),LL(0xad8c0fa5,0x623ad546),LL(0xae0601f7,0x5cf44ed1),
   LL(0x0f36e763,0xbdc4a6e7),LL(0xd3fb2441,0xaa5058c6),LL(0xbd11e44f,0x7603b5a1), LL(0x6c72527d,0x65a896f3),LL(0xd59c82f0,0x214ffbf2),LL(0x0ec375a1,0xf886b405),
   LL(0x6933ecb2,0x12e89a15),LL(0x2c1f2f43,0x3ed0ad89),LL(0xe0d5c6de,0xcafb442c), LL(0xe8cd1af4,0xb06cf2fd),LL(0x9c4265bc,0xb06e7452),LL(0xa73a0639,0x6af8b9d8),
   LL(0x513e4297,0xc18ced3e),LL(0x7fab97a1,0xb4f2f3ca),LL(0x79cc3e70,0x16a64cee), LL(0x64e9eb0e,0x1f3006b4),LL(0xfedf5a33,0xf997d410),LL(0xffe83ec6,0x7fed67a8),
   LL(0x3d30aa08,0x2452d81d),LL(0x1cd85802,0xcac330c3),LL(0x79a76f43,0x9a89abac), LL(0x17c27a7a,0xa965909b),LL(0xa62a311a,0xa1534cd2),LL(0x71a72a64,0xe4fe0317),
   LL(0x261133ef,0x2d7c475e),LL(0xff4f2dd3,0x37952ead),LL(0xe360fb3a,0x10b0a5a5), LL(0xea3f4d28,0xb88323f0),LL(0xebeff23e,0xf1488cef),LL(0x8d265c93,0xa3b9477f),
   LL(0xb76e90d9,0x57b098a1),LL(0x0449bf9c,0x988fbb42),LL(0x062f911b,0x95642d7d), LL(0x61b08116,0x788996fd),LL(0x09d205af,0x5b02272e),LL(0xb3d8f31e,0xdea85f72),
   LL(0x09b907f5,0xdcc15267),LL(0xf7971f9b,0x784511d8),LL(0x20119354,0xe309478a), LL(0x4ce572f8,0x4d5e16a3),LL(0x2c62f607,0x7eb6eeed),LL(0x57f7d26b,0x6bcb2bff),
   LL(0xd75c445c,0x2559d692),LL(0xb7443959,0x49d40028),LL(0x55e49cd0,0x5ee0d518), LL(0x33bb1b6d,0x0cade75f),LL(0x32bf7888,0xd8bac375),LL(0xa0531d90,0xa47d4b3b),
},
/* digit=4 base_pwr=2^28 */
{
   LL(0x58d5549f,0x881d4415),LL(0x7328bce3,0xed748e21),LL(0x2f1f8481,0x56f0d441), LL(0x021c9c2c,0x74ff865e),LL(0x93500f82,0xbaa63b47),LL(0xc2e3b807,0x0c40fade),
   LL(0x9568fad3,0x6f93d120),LL(0x7ff0a77f,0x291875f3),LL(0xdd42f359,0x0608b32f), LL(0x644f9985,0xfee522d9),LL(0x05996cc8,0x4beb2bfd),LL(0x769aa367,0x5a77212a),
   LL(0x504a6f07,0x96957fc9),LL(0x6acad37b,0xfd75c98d),LL(0x548f17da,0x2cc5e7b2), LL(0x7bdfaedc,0x039b865d),LL(0x1970dc2b,0x1ed21afd),LL(0xb4169fbc,0x93b309be),
   LL(0xc373ba13,0xca0eaf9d),LL(0x798e41ad,0xe25f5f11),LL(0x1eca3ea4,0x5af4b3c7), LL(0xa23b9c8a,0x47bc86f1),LL(0xfae9d7b2,0xe9768f07),LL(0xed3e806e,0xf94f89f5),
   LL(0xe4f24c73,0x43f87b32),LL(0x643fb20c,0x41c6cddb),LL(0x602b32c9,0x7093785c), LL(0xa13dc1f1,0xdf9d7247),LL(0xd42502e0,0x3a967995),LL(0x4df43de6,0xa0e28ce3),
   LL(0xf53d962a,0xf3011fff),LL(0x1767f31f,0x9ef38fb2),LL(0x5f883ec2,0xa11413c6), LL(0x4c344917,0x84383633),LL(0x3f5cae22,0xca855976),LL(0x15b2fddf,0x929a1484),
   LL(0xccfc0e1d,0x58946718),LL(0x809ffae5,0x651d4a6d),LL(0x87a653a0,0x6eeb8892), LL(0x6881e966,0x2c4c3c55),LL(0x6dfb6d35,0xbdb6d87e),LL(0x92113e28,0x7cf86719),
   LL(0x6203f1ec,0x6451f3e3),LL(0xab59d68f,0x40681d08),LL(0x66af6fee,0xd0bf9608), LL(0xc1e4b6a6,0x5df24a1f),LL(0x5e6a069f,0x64c1b0f2),LL(0x63fbf5bd,0x124e2748),
   LL(0xedf146a2,0xe6cbf0bf),LL(0x085c71de,0x85e47e89),LL(0x603e3ba3,0x39330eae), LL(0x6841460f,0xbcd9f912),LL(0x55ffa24e,0x4212f2ba),LL(0x3fb12ad2,0xadf49ddb),
   LL(0xadb7dfa6,0xfb02103e),LL(0xbabeeea2,0x4dce5edb),LL(0x622cfe41,0xe4e76c1e), LL(0x5d2cde0b,0x318e8d34),LL(0x4448a054,0x5d3e168a),LL(0xa23e5457,0x444ff058),
   LL(0xa8083fb8,0x06020b4b),LL(0x73e1d9fe,0xc6d07af1),LL(0x137f9872,0x2056f42e), LL(0x523b9b18,0x5ee31758),LL(0x39042106,0x58771cbe),LL(0xa0e5b696,0x114531cf),
   LL(0x8a9552c6,0xa4eedc90),LL(0x6eb5fedf,0x23be4e8c),LL(0x5ad0744a,0xe5f6ada1), LL(0x4ca7d6f4,0x373a4a9e),LL(0x57cea5a6,0xb5a5a64d),LL(0x7f48cb04,0x3a65f463),
   LL(0xc83aaf37,0x73506b7f),LL(0xd4bce539,0x9785f792),LL(0x7df31e2d,0xebbfa1b8), LL(0xd5a2ea75,0x3b66c6b3),LL(0xf2196549,0x32ac5d33),LL(0xf8082490,0x89d1486d),
   LL(0xf9fe7377,0xc0c2ec19),LL(0x39415cfb,0x07911ff1),LL(0xa1c91f71,0x1b54a66d), LL(0xd4f59851,0x29646a9e),LL(0xe209ab6d,0x0a7fcf62),LL(0xfca0de15,0x6ea260fe),
   LL(0x5fcc220a,0xd026ef4d),LL(0xd9364e42,0x7303fde9),LL(0xd35f5fad,0xcf2f621c), LL(0xd439a9f6,0x6e346476),LL(0xccf2de6e,0xb108eab2),LL(0x6d3db6c6,0xca46cb58),
   LL(0x1de59604,0xc89fccf4),LL(0x59f8e37a,0x72ee2825),LL(0xd52aa0eb,0x9423de94), LL(0x1110f4b8,0xb6837166),LL(0x5e0ca5e7,0x00b22047),LL(0x7f72b3f8,0x79573c27),
   LL(0xe67b0c29,0x6dfbc44f),LL(0x05177565,0xa0a8e62f),LL(0xbf5b1bbb,0x7f494f35), LL(0xd472f2c7,0xcc727439),LL(0x48ea2e18,0xcafc4607),LL(0xc5b2f1e5,0x25499b27),
   LL(0x86c1e086,0x83fb7e04),LL(0xafbb1111,0x19eeb012),LL(0xa0c8df49,0x620d5c9f), LL(0xd8adc0b6,0x85368dc4),LL(0x0c2f7efe,0x174dcd3c),LL(0x52fe89c3,0x71d50509),
   LL(0xbbeb2341,0x03ec18cc),LL(0xc49153aa,0xf386e999),LL(0xd01e004b,0x58b7a90b), LL(0x3bf5753d,0x66713f74),LL(0x23ce4a5f,0x96fa5445),LL(0xa7a2e6a6,0x06b17741),
   LL(0x1fa5345b,0x93e18885),LL(0x2b10748b,0xc3faca64),LL(0xdf987dbc,0xc1b82b06), LL(0x205e25f4,0x8f3aa4d9),LL(0xb7dc364e,0x31bdcd03),LL(0xb91f4dfe,0x5a21cbcc),
   LL(0xbc5b1ae6,0xc9f8dd3e),LL(0x09390a2d,0xfd14096e),LL(0xd6fbd5f0,0xcf2461e3), LL(0x02a263e4,0xaa775c22),LL(0x0fbb901d,0xc7829c8b),LL(0x1840557a,0x338e09b6),
   LL(0x1c6d5d1d,0x6aa505da),LL(0x5646e930,0xf7906394),LL(0xa52fc317,0xfc0035dd), LL(0xb185221a,0x9fefcec6),LL(0x49de64e9,0x692267cc),LL(0xc316de61,0x2d105157),
   LL(0xd4e3b784,0x57ac5de5),LL(0x13ab394f,0xac9c4222),LL(0x027f1648,0x5aa27972), LL(0xc83390e1,0x7a4b5ae2),LL(0xa1a47d58,0xba9ff075),LL(0x4a3d0cd8,0xf9ad2400),
   LL(0xede14e49,0x70ac2786),LL(0xbdf4b5ad,0xd79e0890),LL(0x01edd96f,0x7b6dd833), LL(0x565fd682,0x1b4f533d),LL(0x474f71f9,0x9222c9a2),LL(0x83ebda25,0xa56581a9),
   LL(0x25ee5573,0x7050f5eb),LL(0xdb88f4cd,0x4758c876),LL(0xc852b937,0x9abc5336), LL(0xa5b25a0b,0x88d32963),LL(0xf471412c,0x3e117bb5),LL(0x081986d4,0xf4fe7f16),
   LL(0x976cf433,0x1a1189da),LL(0x7202a031,0x8d95553c),LL(0x844752a4,0x59d33ccf), LL(0xb361b89b,0x8145a133),LL(0x43423d20,0xf78e0d37),LL(0x61a44de5,0x522e94dd),
   LL(0x8e540c0b,0x7c2237f8),LL(0xe30c14a2,0x89d8e57c),LL(0xbf6aa5e6,0x456a042a), LL(0xf3216fe6,0x22ab0e28),LL(0x5cb9a52f,0x54763652),LL(0x0b10595d,0x5ee60169),
   LL(0x3800818b,0x06cc9e76),LL(0x7f5020ad,0x426cc669),LL(0x18577c5f,0x53ff7b9e), LL(0x30789f9e,0xbecc9a3d),LL(0xac798e8a,0x685f1d84),LL(0xcf85daa7,0x3c0745ce),
   LL(0xdfd3615b,0xd516bc63),LL(0x9ad06650,0x36e8a2de),LL(0x9a930d91,0x71f2ade0), LL(0x88d6e03e,0xe237231a),LL(0x80cd199b,0x8af22a7b),LL(0x68638e0e,0x999ac96b),
   LL(0x3b855f82,0x92421da5),LL(0x229553e1,0xdf39524a),LL(0x389ff1eb,0x388c4741), LL(0x49806f26,0xb73b3db2),LL(0x116a75da,0x0066ae14),LL(0x7d37c100,0x224f589b),
   LL(0x4defbdd3,0xc60c22e1),LL(0x5449bd64,0x6d574468),LL(0x46ea4c3b,0xee1e887e), LL(0xc608f1e7,0x16b1d73e),LL(0x87301c1a,0x30a39cb8),LL(0x76725f2b,0x5e7728f2),
   LL(0xbeda23d1,0xd3a7aed2),LL(0xaf09fa81,0x2d4adeac),LL(0xf998500d,0xcc705a59), LL(0x97b62945,0x2ec67209),LL(0x2156a3f4,0xfefbf4e8),LL(0x0a5fb347,0x20684eab),
   LL(0x785d1eb6,0xeb66a59a),LL(0x82fd2480,0x98279fc5),LL(0x1a00554b,0x6b259c82), LL(0x913e2492,0xa46b9ebf),LL(0xea3691cc,0xc3524af6),LL(0x6746a57e,0x00ec234a),
   LL(0x71718339,0x33a6b51e),LL(0x206c1075,0xbffbd973),LL(0x0461a56a,0xd7489944), LL(0x5ee41c8e,0xb4440567),LL(0x7b86fe5a,0x2bf38abf),LL(0x8cd45ef5,0xf70c63a1),
   LL(0x845aa2a3,0x80c95a21),LL(0xd508e765,0x4882eaac),LL(0xcb5bb20c,0xc5f7b8fe), LL(0x104b3369,0x96426128),LL(0xd131d49d,0x23598393),LL(0x27af509a,0x059d6d87),
   LL(0x330dc205,0xecb4bd8d),LL(0x793b3824,0x3934dce1),LL(0xf505c154,0x2b8cedbe), LL(0x7a873a69,0xba92295b),LL(0x386ee396,0x8449398b),LL(0x4717b4b5,0x72584bae),
   LL(0x73765fd1,0xa18a909b),LL(0x7db9952c,0x2494a304),LL(0xe91c490d,0x2747b58d), LL(0xb6d10153,0x1178cdf0),LL(0xbf4c5889,0x35235125),LL(0x90f823d5,0xdf63fa73),
   LL(0x4066c1a7,0x38748fb7),LL(0xf5cf90a5,0xfbfbb8e2),LL(0x9b83fd71,0xa828018b), LL(0xb34ce140,0x3de7f633),LL(0xd54824f7,0xd760a3e5),LL(0x94e2e012,0xea872fd7),
   LL(0x479fbc3f,0xd2aa4fdf),LL(0x1979cfcb,0x11a9a799),LL(0x03e344fe,0x01733a8f), LL(0xb5b3cc49,0x0c61b034),LL(0x10c7d2cf,0x27483deb),LL(0x1884d1f1,0x6dc92d21),
   LL(0xdce73a42,0xebf1943b),LL(0xa62405be,0x91a018b4),LL(0x5684989d,0x844d759b), LL(0x4678912e,0xbdc609b4),LL(0xd12582c1,0x4b151529),LL(0x7f8e1390,0x7ac0cfc4),
   LL(0x987acd67,0x5245ca01),LL(0xf023cfd4,0xd4688d7c),LL(0x4b96af1c,0x799e7c06), LL(0x0a9e14c2,0xa96c4571),LL(0x2ded73d6,0xc2dc67eb),LL(0x2783e2b5,0x43588841),
   LL(0xeef15ae5,0xa3b115c6),LL(0x87b86741,0xf305a27e),LL(0x4711a4a3,0xa5fdc849), LL(0x7b6ef648,0xeb58f0f1),LL(0x6f3ea084,0xdccddcbe),LL(0xa1b862a7,0x6f7470fd),
   LL(0xd88e9f3d,0x141947dd),LL(0xfbb2d23f,0x1b677675),LL(0x16da518b,0x12a4c8c1), LL(0x032c0bc7,0x1cccf6b7),LL(0x20ed64ed,0x5b7cfe57),LL(0x5269cf39,0x7d427f2c),
   LL(0x8d53b039,0x7faea5da),LL(0xacb5b11a,0xf19f5f61),LL(0xfe27af9c,0xfd679e18), LL(0xc0e391aa,0xbe29ec0c),LL(0x8cef3cd7,0x42a815d2),LL(0x42c2564f,0x3dd43747),
   LL(0x6370d284,0xbaa0ccdf),LL(0xeb70e670,0xb68dae3c),LL(0x9917aadb,0x5e25db51), LL(0x57c8020e,0x27888369),LL(0x0cd601a7,0x4ca76fa7),LL(0x4509eeb6,0x46794e0a),
   LL(0x4538b5b5,0x9f21619a),LL(0xd6754ebf,0x6c1f3f7b),LL(0xb7b2f926,0xa2a0e2e7), LL(0x42728394,0xc5747d30),LL(0x0a0328e3,0x2662cd33),LL(0x3bfbcc49,0x89c445cf),
   LL(0x96bba2f7,0xce07c166),LL(0x93dfae9a,0x6828eab6),LL(0x4036abaf,0xa907dcce), LL(0x6cd5853d,0x61e23034),LL(0x331dc277,0x541d6f4e),LL(0x82796b5a,0xdaa10e8c),
   LL(0xfac61d43,0x339d6949),LL(0x75b34910,0xedc8bc2f),LL(0x1ee6f866,0x4c686fe9), LL(0x5f02368e,0x9bc462bf),LL(0x638abf60,0xc3902774),LL(0xb7ed5413,0xc403b179),
   LL(0x664e0b41,0x35bf0ed9),LL(0x889af4e3,0xc3ad5755),LL(0x5153a58d,0x14291828), LL(0x64798a76,0xb3d9c453),LL(0x827c237a,0x71e6ce15),LL(0x41e52b67,0xbbcce06b),
   LL(0x1a05cd67,0x523df124),LL(0xf104c7c7,0x969598b8),LL(0x7f7b1fa6,0xc16d7bc1), LL(0xfb1a2aa4,0x935b18e9),LL(0x4f99ebf4,0xba7ff2a2),LL(0x862cab62,0x0613fa7b),
   LL(0xd8d26e75,0x02f5cfb1),LL(0x98906c57,0x5e0d4363),LL(0x1032e39f,0xca80873c), LL(0x204c0680,0x2fa5b881),LL(0x7c178dd3,0xca227906),LL(0xdf2e321b,0xba33bb48),
   LL(0xd9a13914,0x4cbde5bb),LL(0x621b5b3c,0xaa89d2d4),LL(0x75821515,0x66fd1db4), LL(0x66154f11,0x3e616070),LL(0xed85f2f5,0x2dde1732),LL(0x528e1b57,0xfa1fe78f),
   LL(0x600e571d,0xb51f5c95),LL(0x15059c58,0x13f0e99e),LL(0x72e8f443,0xe2a1a34e), LL(0x2285a044,0x5c484292),LL(0x908d9855,0xac4b6bdd),LL(0x8cffdb0a,0x4b4ac5ae),
   LL(0xd691153c,0xf8f74a29),LL(0xd21533bb,0xc0cb0e05),LL(0x86067d1d,0x52ceb52e), LL(0x9cd93bda,0xeb021228),LL(0xd7c98d3b,0x27d7ef30),LL(0x78fae7b9,0x7c18f1bf),
   LL(0x5ef7655c,0x1b3562c7),LL(0xb552fd07,0x62ce6adb),LL(0xb9c6217d,0xac6f179d), LL(0x7fb8394e,0x3b329c45),LL(0x9b340753,0x418d2ac5),LL(0x68c5c9d3,0x5adca639),
   LL(0x15b71702,0xbfd6f204),LL(0x569ed998,0x292bfd32),LL(0x003d0141,0x5b5d5b09), LL(0xefcd36d1,0xd512f4d8),LL(0x16c87812,0xc81f1c34),LL(0x1777dcaa,0xa54cef3e),
   LL(0x76585d99,0x53830298),LL(0x30d1c285,0xe2d41a7b),LL(0xad60f2ff,0xb6eabff3), LL(0x3d361b12,0x073639c4),LL(0x16f921f3,0x10e605bf),LL(0x3394e5eb,0xff665c90),
   LL(0x359bf1b9,0xadae33b1),LL(0x61554e45,0x183beb7e),LL(0x559e3f90,0x28d50fcb), LL(0x4bc0b8cd,0xa9a26299),LL(0xb403463a,0x25b24e6a),LL(0xf076ebd6,0x715055a6),
   LL(0xd381a2e0,0xa6b7fe33),LL(0x6b248193,0xc2aacec9),LL(0xc9d6ba70,0x1f617084), LL(0xcd06c9a1,0xce945304),LL(0x1b4c3d5b,0xb957b98b),LL(0xfa6b8f04,0x50c61d1f),
   LL(0xdaa47b46,0xbd69e6d9),LL(0x7a9479bb,0xc12cbd8a),LL(0xa8b92f63,0xf07cbcd9), LL(0x27d2f85e,0x4998f584),LL(0xbf43ba36,0x0ab2e076),LL(0xbad0d0a2,0xbed73de0),
   LL(0xad129e1b,0x3e02902b),LL(0xa008c72c,0xd2e0bd57),LL(0x30e61c2b,0xe88a075a), LL(0x5d9402a8,0x136e1b7f),LL(0x7bc0a653,0xaa06f30f),LL(0x068e6e0b,0x51b52693),
   LL(0x695404da,0x0bb86a49),LL(0xb1059de5,0x339cbbca),LL(0xdfc1bfa7,0xd45db64b), LL(0xa98831bc,0xad07e52d),LL(0x51a7a71a,0x5ef54a92),LL(0xee02c742,0x7fda785f),
   LL(0x6686a65a,0x05b7848e),LL(0x6f478ff9,0xc12a7893),LL(0x2d137586,0xf1f54c31), LL(0xd50ea873,0xfe8799f4),LL(0x5fabdd98,0x62d996af),LL(0x5181afce,0xf0fe39fd),
   LL(0xef010cef,0x646b35f9),LL(0x7d54d120,0xa63852dd),LL(0xadec6e34,0x3e5c14f4), LL(0xc2f4b7e6,0x030270aa),LL(0xec560a92,0xdedf4875),LL(0x9da64c53,0xd6d97f65),
},
/* digit=5 base_pwr=2^35 */
{
   LL(0x1386b66e,0x18754788),LL(0x9a2eb0d8,0x783dc08a),LL(0xf074e5e1,0xfe2682d1), LL(0xcbf4f5b8,0xb6ab7696),LL(0x6da5b0aa,0x382401e6),LL(0x0f93d45d,0xfd5a6375),
   LL(0x6fefd17e,0x51d517b8),LL(0x18fba339,0x35e97b40),LL(0xcf221c7b,0xbdc8467c), LL(0x9eaee11c,0xba6035ff),LL(0x18128100,0xb417857d),LL(0x4f40acec,0x383235c3),
   LL(0x4197bc61,0xdcfc9f38),LL(0xfe9c1104,0xeb61eb5e),LL(0x5486c84c,0x7ba40c29), LL(0x1377a769,0x2f6a8ebc),LL(0x914c369a,0xe4e70f07),LL(0x57afc5f6,0x8d954007),
   LL(0x0fa7e4d2,0x56b010d6),LL(0x1d9544ef,0x91cd7d72),LL(0x48b3fb02,0xe2c51636), LL(0x683588bd,0xa2440769),LL(0x98239b52,0x7054bb40),LL(0x3bb5090c,0x2019db99),
   LL(0xf043605e,0x6da23d19),LL(0x69d16792,0x4f2d4a4f),LL(0x6d5127a3,0x5061d4c8), LL(0x71c846bd,0xa0ef332c),LL(0xca929e1f,0x7948a241),LL(0x0baf2f1a,0x4152aec5),
   LL(0xdc72a138,0xf15dd5cb),LL(0x91a32dbf,0x6d2d2597),LL(0xfd5852dc,0x7f46cffd), LL(0xd817364b,0xda270e8e),LL(0xa1df99e0,0x97e5e85f),LL(0x5a90c0ae,0xc77a3a0a),
   LL(0x19a3eb23,0x1636651c),LL(0x12294385,0x34a5e78a),LL(0xec25d3d6,0x71cac725), LL(0xe5aa03cc,0x98956393),LL(0xf1d2e766,0xcda62f63),LL(0xf7c046ed,0xb8c8909f),
   LL(0x1578b888,0xb68a6a3a),LL(0x30e1dbfd,0x01e39894),LL(0x71dafb3f,0x03ca0487), LL(0xf8241fab,0x7690a577),LL(0xe64d2bb7,0xac71ef0b),LL(0x3eae3938,0x4fd69e13),
   LL(0xa2e02fc2,0x7a04ff5d),LL(0x75b01d9c,0x11114bbf),LL(0xe08f4299,0x1233f026), LL(0xf491d93a,0xd888bb1c),LL(0x07c1f414,0x7c6fc026),LL(0x97d8d7df,0x753a2913),
   LL(0xea21ec86,0x0fe566b4),LL(0x75ddae93,0x743f5133),LL(0xee43927e,0x0173fdd9), LL(0x8b3a99d1,0xe9491f6e),LL(0x90e9d60e,0x6af11d34),LL(0x373fafeb,0x7ca50de3),
   LL(0xb53b609f,0x3a3eaa6d),LL(0x1cf1fa53,0xcafff213),LL(0x2cc998b7,0xb64116ef), LL(0x8eac5d3a,0x63148b28),LL(0x1dc633b7,0xff5f7662),LL(0xc94b635d,0xf174dcd1),
   LL(0x013644fb,0xdf0f0994),LL(0x526fa9f2,0x77369779),LL(0x4197eb1b,0x41ef1970), LL(0x22cb653d,0x7bf4bf00),LL(0xb7dd509f,0x94f0f226),LL(0xbe77682f,0x5fd7dd2b),
   LL(0xf658d2b2,0x80435f56),LL(0x5f8cc779,0x1e864847),LL(0x8b56aaf0,0xdd4cb7e9), LL(0x72c3024f,0x458c8bb7),LL(0x8cffdca0,0xa5c186c5),LL(0x377f96ba,0xc0022799),
   LL(0x1f7d9343,0x7e4d4404),LL(0x299cd36e,0xa496646a),LL(0x600decf3,0x06372825), LL(0x08bf05fa,0xe19b609d),LL(0x1b7e153a,0xfe6a7182),LL(0x9808f5c7,0xfe191f76),
   LL(0x2a8c6b66,0x2c13eeb6),LL(0x0fcafcfc,0xcd3cc6d3),LL(0x236f08ff,0xb85a615d), LL(0x14e4e83c,0x92187d5e),LL(0x21979d04,0x5fdaaa79),LL(0xe519147a,0xed2d2008),
   LL(0x02fe620c,0xd8083896),LL(0xde8984fb,0x277803e8),LL(0x7058d34c,0x2698ed9f), LL(0xbce1dcc3,0x00b9c0d3),LL(0xbc2c1c85,0x356b310e),LL(0x67cacabb,0x41e412dd),
   LL(0x2790dc4a,0x396267ca),LL(0xd5bfe8c6,0x5d4593a3),LL(0x335a7f8d,0x5fddd3d4), LL(0xab0ac92c,0x92f002c0),LL(0x6226c031,0x590a93b2),LL(0xdfe775be,0xb3c10c49),
   LL(0x85889144,0x0209bd99),LL(0xb9b83e2d,0x77df1ced),LL(0x4cca6a2d,0xbf9f5870), LL(0x35978ee6,0x2c4f310e),LL(0xeae8acff,0x98debae5),LL(0x778c9461,0xf2f40fc1),
   LL(0x9d363814,0x4c6c9e7e),LL(0x7b9d3d53,0x52fe5e06),LL(0x9cecbe7a,0x19bd3915), LL(0x272c9325,0xcbce0a07),LL(0x6a7c9eaa,0x48336f5c),LL(0x77a17578,0xd234b427),
   LL(0x3ca7e0eb,0xb22492a2),LL(0xc5a32881,0x91ba7704),LL(0x00d1425b,0x5a2a7f1e), LL(0xbc09eeae,0x6b15d61f),LL(0x86b7bff1,0x2e8f822c),LL(0x32c4dd9e,0x56357b66),
   LL(0x51e3d663,0x74a1f2c6),LL(0xa42a3633,0xf42faf2f),LL(0xf6e193fb,0x1b1be8b6), LL(0x4200b0ee,0xb647f284),LL(0x02d4935e,0x0950b424),LL(0x4deee646,0xdbdd7c52),
   LL(0x77c3a95a,0x15c8ddba),LL(0x6ba71f9c,0x901a11bd),LL(0x440741f5,0x54fc905b), LL(0xf1358db4,0x4b6b0fc5),LL(0x6ed93612,0x3c4c39a3),LL(0x395cb400,0xa764c9a8),
   LL(0x2b54f47d,0x089cc68e),LL(0x4fa19e45,0xd6f94042),LL(0x62803448,0x52875bc1), LL(0xdf571d9b,0x1f43a201),LL(0xff9a0750,0x9e8645cd),LL(0x61a710d0,0xa177f4bb),
   LL(0xddae695a,0xd55cc687),LL(0xe7471217,0x90d2c914),LL(0x917d18a0,0x884f4e6d), LL(0x744d11d9,0xd2871c7d),LL(0x02972782,0xef60b7d8),LL(0x2531f9a0,0x491aa401),
   LL(0x8f0f3404,0xf881970a),LL(0xe6a7678b,0xb681fd66),LL(0x45c5eec8,0x673a8192), LL(0x216460f1,0x39e65db1),LL(0xee6d42eb,0x28cd720e),LL(0x08badd82,0xcf155154),
   LL(0xf5c7d936,0x35ab0302),LL(0x7eee29e7,0x0a6d5f94),LL(0x7ed08644,0x03232350), LL(0x6d2a8356,0x505064a6),LL(0x979af3f3,0x6e9937d1),LL(0x0758cb3d,0xec148fcc),
   LL(0xa047cb43,0x46a50ee9),LL(0x2c02de4c,0x6c6b569f),LL(0x3d95c934,0xcd21154d), LL(0xd55720da,0x3a14afa6),LL(0x5512e601,0x6ceee63b),LL(0x4c47b641,0xafe6bbff),
   LL(0xf19e974d,0x717ac089),LL(0xf71eadf4,0xfb0cebe8),LL(0xe85c30e2,0x503535ba), LL(0x072130ee,0x05e57133),LL(0xa8037e5f,0xf33e81ed),LL(0x3f0c555a,0x4b33d96d),
   LL(0xd71f6adf,0x47ff2d79),LL(0x7a985d02,0x88a9de0d),LL(0xb2dc0cc5,0xc9cc2822), LL(0xfbc2972b,0x38052b28),LL(0xbc8465ba,0xdaecc9da),LL(0x7ff5a249,0xaaffa0d1),
   LL(0x60361946,0x4f326618),LL(0xae1695c7,0xe7eaffff),LL(0x1311036b,0xa7aed10b), LL(0x9eb2dfa7,0x6eb81e71),LL(0xf680fea3,0xeda95d34),LL(0x83630d7a,0xd2ab66c5),
   LL(0xb891fc22,0xa7433cf1),LL(0x1111e0db,0xdbea5480),LL(0x658341c1,0x823a8595), LL(0xe2d8917b,0x7a171afa),LL(0xe713e015,0x79e4154d),LL(0x3ebc430e,0xbd07aa2c),
   LL(0x0523f8b0,0x091a19d2),LL(0x0950ab93,0x9fbf9c8c),LL(0xfbb731ea,0xd8377e50), LL(0x18b26813,0xe3b8b5bd),LL(0x1bc1257d,0x78a16f16),LL(0xcfd383ea,0x837106a0),
   LL(0x11c7579c,0x01e9427c),LL(0x1580ef3e,0x24504560),LL(0x0b447ba0,0xd7408db5), LL(0x0e1cfc39,0x6749f5e6),LL(0xdfe158ad,0xd26d7dbf),LL(0xe7b17909,0x66d731fa),
   LL(0xea86e123,0x46d3addc),LL(0xe46c37d8,0x3b52f72b),LL(0x8036a8e6,0xda02c7ee), LL(0xfeeaba0d,0x9f141b52),LL(0x0049be34,0x7a66cbac),LL(0xb9de2e84,0xc5b5d9d0),
   LL(0x6d15b711,0x190e0e24),LL(0xc7ca01a7,0x4eec20f0),LL(0x7803ca5d,0x12ff14dd), LL(0xd59fb3a4,0x84991c92),LL(0xc64b50b6,0x86a00ae1),LL(0x175bbf50,0x6bf38341),
   LL(0x5310953f,0x3cd33d82),LL(0xef7027f4,0xcf3d011e),LL(0x3eee08ca,0x10779dfa), LL(0x31f432ec,0x2c3228c4),LL(0x7f319ca8,0x94c51072),LL(0x3cfe5d7c,0x2ac3a2fe),
   LL(0x4acc7cc4,0xe2ccc5e9),LL(0xff58443f,0xfedc0da8),LL(0x9567fea7,0x776014d0), LL(0xa2b90e62,0xbca6d7b2),LL(0x164be21a,0x11148f7f),LL(0xfec4a5f1,0xebbdf677),
   LL(0xca5a2ef0,0x4bd9a436),LL(0xfd095c81,0x320744ae),LL(0xc307a816,0x3da008cf), LL(0x2f6801e7,0xc5cc3ff9),LL(0x4407869e,0xf0ba2f46),LL(0x213338b4,0x6eed1a15),
   LL(0x274dd179,0x7fa61ef4),LL(0xa631cf50,0x6e6453fd),LL(0xb86291a1,0xaf1017f5), LL(0x9cf7ace4,0x757564ec),LL(0x2535cbaa,0x0fa1ec1c),LL(0xbc9b95bd,0xc03cec64),
   LL(0x7686669e,0xfbcb2d77),LL(0x6cdcb0f1,0xf9bed26f),LL(0xb5a8f081,0x37aff505), LL(0x6053e4cc,0x1f258d7c),LL(0x45bc699e,0x86384c5f),LL(0x25a2f865,0xe0025c0a),
   LL(0xb24a8eb1,0xea42c8e1),LL(0xdc2dd956,0xf670d8ea),LL(0x035bfb42,0x71d4c32b), LL(0xe7824aa9,0x213fce6d),LL(0x4b22f111,0x9b34d5b0),LL(0x3a14d9cd,0xa299994c),
   LL(0x7ba4a4bd,0x975d551f),LL(0x32d65fb4,0x862bc216),LL(0x151762af,0xfaaf16ad), LL(0xcd1fe1b3,0x931eac72),LL(0x00f9e6e2,0xcd730742),LL(0x85ccbcf2,0x581b13c3),
   LL(0x559cb856,0x013ed0f3),LL(0xd22cf5f3,0x5c1af1b5),LL(0x3badc3ca,0x11b231ec), LL(0x68ce4eac,0xc22972b9),LL(0x1a2f55b5,0xcb92e129),LL(0x038e230a,0xa61289eb),
   LL(0x309e8cec,0x625ac394),LL(0x49b13bac,0x99dab1a7),LL(0xdf48a436,0x77940cbf), LL(0x78881636,0x963cb651),LL(0x7541e4c5,0x889ac28e),LL(0x5407fbf2,0x39918bd5),
   LL(0x69209c9c,0x63d26894),LL(0x255dc3aa,0xb6b122cb),LL(0x932b582c,0x12866e42), LL(0xc373f8ad,0x0b0a7dd0),LL(0x1fe0f36b,0x2fed9f9a),LL(0x2345d1f2,0x6dbbf47f),
   LL(0xdefc4c20,0xdf69b818),LL(0x2e3eb2cb,0xf31ff9a8),LL(0x0be0bfa6,0x946578a6), LL(0xe8b31f47,0xbecddbf2),LL(0xa45fadaa,0x86e3594c),LL(0x999c547a,0x56d2d2e3),
   LL(0x3b5a1df0,0xcfe909ee),LL(0x94fd4014,0x4b62c64a),LL(0x258a8f81,0x7d6e3ae9), LL(0x5804ee7d,0xfc15381b),LL(0x71710702,0x3b444911),LL(0x5f18965e,0x46497ae3),
   LL(0x33dd9555,0xfbb99cfe),LL(0xfcbeecde,0xc6e14198),LL(0x51da2e20,0x8d4e83b3), LL(0x75a0719b,0x2bc54a6b),LL(0x56b03328,0x1aba20d3),LL(0xede40907,0xd84a7db6),
   LL(0xeae9291d,0x6f52a996),LL(0x33d0f1a3,0x507c3701),LL(0xafebeab2,0xb7809c3c), LL(0x80b978c3,0x301a5ed5),LL(0xd10f0edd,0x1ba9c42c),LL(0x150d6504,0x0e9f9f93),
   LL(0xee90dfeb,0x56ff221e),LL(0xc6815c60,0x279a7940),LL(0x1b0208a3,0xa119572a), LL(0xd2710835,0x5bb04e27),LL(0x67414bf2,0x61d8d945),LL(0x6aec6a5c,0x440a1d39),
   LL(0x1ecc000e,0x07e60a25),LL(0xe6efdc85,0x2325aab6),LL(0xf4872ceb,0x29438ae3), LL(0xe8d10241,0xeea2a3f2),LL(0x548a7556,0xd2ff366b),LL(0x2a6e27bf,0x85e8ada2),
   LL(0x11a6e295,0xb10332e7),LL(0x609a424a,0x44c9ba0e),LL(0x986e93f0,0x08f222e9), LL(0x9dcb7c9b,0xeb8bea94),LL(0x1d5aa8a0,0x36e44e93),LL(0xa3e0dccc,0xe0deccc6),
   LL(0xc22e314b,0x92be41b1),LL(0xaabbe524,0x388494bc),LL(0xbde2517d,0xaaf52dfe), LL(0x186fedfc,0x4ce61d88),LL(0x300b7bc2,0x9f29e844),LL(0xedab8ca6,0x03a2221d),
   LL(0xda0cae4d,0x7c3f687b),LL(0xe7da757e,0x84dc6167),LL(0x85d25286,0xc0c08f69), LL(0xb4889367,0x17f7921d),LL(0xf81fddec,0x223c9181),LL(0xa6a05bf7,0x3b72e938),
   LL(0x60326fa4,0x0ff61ea1),LL(0x97c51293,0xa54cc228),LL(0xf976ab17,0x7d25531f), LL(0xa8f84662,0x1e8c3f37),LL(0x007ac2f1,0x4d8c3063),LL(0x4598d053,0x80c8de31),
   LL(0xe4198b95,0x3ed83a96),LL(0x90ef6fa9,0x4435dc6a),LL(0x9ab2a2d6,0x47791ebf), LL(0x1564b80a,0x3e0c8b07),LL(0xa0ec1ad9,0xb4079d88),LL(0xee9bfef8,0x6fc566af),
   LL(0x7f1a5c21,0xbfc3ee41),LL(0x2727a00a,0xe3694ec6),LL(0xa4087b70,0x3ff4d181), LL(0x37de763e,0xda656043),LL(0x8a50ec66,0xc2da4f75),LL(0xb94a5c19,0x096c88ff),
   LL(0x3121ec7a,0x05e200b8),LL(0xf23f234f,0xc5f282e5),LL(0x94aebbc2,0x0d95e5bc), LL(0x6922306e,0x389ff8bd),LL(0x0c458f95,0x949a4362),LL(0xa8a67759,0x2ed544e9),
   LL(0xd643944f,0x46d3e81e),LL(0xa5cfb8de,0x36211db8),LL(0x95b4f959,0x8a5ee9eb), LL(0xcd19ebcd,0xc3d6941b),LL(0x61306b29,0x4bc41c81),LL(0xeac1ab81,0x0093af39),
   LL(0xe6622883,0x301a7d6e),LL(0x270ac148,0x5c49e719),LL(0x4bbec99b,0xff3296f0), LL(0x80061376,0x880c775d),LL(0x322403c9,0x587357fe),LL(0xc24a4307,0x4db2f9fb),
   LL(0x76f5e74e,0x3a8b93a4),LL(0xaa1b2dd8,0x5e9e5474),LL(0x5ad377ae,0x43c84363), LL(0x8b7ef6de,0x3ef3ceea),LL(0x7be9f363,0xa3457c60),LL(0x6c4f30eb,0x8a0c2c94),
   LL(0x3eef5158,0xc50fa97a),LL(0x86d6a054,0x52c0d8a2),LL(0x3be5c327,0xebdf472b), LL(0x0852c07e,0xc34b020b),LL(0x42fa2f1c,0x5b2b3552),LL(0x3a4f7817,0xbff68be9),
   LL(0x959b0d24,0x2cf2220a),LL(0x56dd4a59,0xeb164f48),LL(0x9d8fff97,0xd9927903), LL(0x233eaa77,0x0ddf5e1b),LL(0x7969f9e5,0xf32d29d8),LL(0x5562cc50,0x26ee53ae),
   LL(0xf53c8cf7,0xcbc8f03f),LL(0xc7ff11a2,0xb6321218),LL(0xb9346795,0xb37d632a), LL(0xaf7e83a5,0xbc24e219),LL(0xe6dbd3da,0x95b4863a),LL(0xcfb3f25e,0xc3f00af4),
},
/* digit=6 base_pwr=2^42 */
{
   LL(0x097b6df3,0x807a7671),LL(0x0fb780e0,0x41faf5e2),LL(0x157c3ef1,0xdaa12450), LL(0xd7cb0c23,0x31d72356),LL(0xb251a877,0xd5d2c185),LL(0x92fa894a,0xdbdaee99),
   LL(0x98704ce6,0x85e18802),LL(0x306d7501,0x25a349dc),LL(0x03de69b2,0x6c5ba821), LL(0xc403903d,0x9fd2d533),LL(0xbf5698ac,0xe3b96ca6),LL(0xcc20c102,0x7d5e41c1),
   LL(0xcc888fdc,0x6d434dd0),LL(0xed221a04,0x47e731b4),LL(0xa107a3c9,0xda0d85b5), LL(0xebf163d0,0x489780a7),LL(0x9892bf9b,0x1ddd62aa),LL(0x04e0db7d,0x7f988bcb),
   LL(0xe492a9fc,0xb55c1496),LL(0x597a9eb3,0xd0cd80f2),LL(0x9f1d8391,0xeaf34b51), LL(0x785286c4,0x81e7ca50),LL(0x6c0c5297,0x5b63c071),LL(0xb1b52561,0xa09ca533),
   LL(0x19fafaf2,0x082de97f),LL(0x25cc03ed,0x92de16ec),LL(0x0fabb7cf,0x415b7e1c), LL(0x721659a0,0xb121761c),LL(0x4a3f2f1d,0x2c646d3f),LL(0x13cef73d,0x6c441f7c),
   LL(0x26cc7cca,0x7d4281a3),LL(0xe302e412,0xf9d8fdf7),LL(0xed1531b3,0x9b29a767), LL(0x8daae632,0x9713865b),LL(0x51561f0a,0x5c4d8193),LL(0x74daf2a8,0x786d917a),
   LL(0x08c24f78,0xf4b82a3f),LL(0xe3565350,0x82c57c65),LL(0xc56bf74b,0x92213214), LL(0x066173e2,0x51cf2f2a),LL(0x22931b9e,0xb67e7f28),LL(0x0e448ae2,0x87983d74),
   LL(0xcd93498b,0xa08d39d8),LL(0x81fb94c6,0x98c24fae),LL(0xe2ec50b6,0x2c1f6da0), LL(0x703366fa,0x96b60285),LL(0x6ce66461,0xa46d2ed9),LL(0xc7bef52c,0xaf14665e),
   LL(0x614dfb0d,0x03487de0),LL(0x950342d5,0xae8d0e16),LL(0x77195a73,0x4f2ab2f7), LL(0x9757d4ae,0x0cfdcfec),LL(0xe42519f0,0x20cca44c),LL(0x4b2eb661,0x3a973637),
   LL(0xf0c80fdb,0x5ed30224),LL(0x9d39980f,0xb74201d0),LL(0xc2a6b759,0x705dd1b2), LL(0x080941d1,0xd6535dc8),LL(0xd279a85b,0x0d532b08),LL(0xa6c95800,0x55279407),
   LL(0xed2f99c0,0xa5e5dccd),LL(0x26335c8f,0x1d51b65a),LL(0x08067e36,0x930c04c4), LL(0x0583c23c,0xb14d764c),LL(0xe5be4e67,0xe9f1f8b8),LL(0xd3b9a111,0x2b684b5a),
   LL(0xce971c7a,0x9420fabb),LL(0xc95e25ab,0x3ade0eab),LL(0xa490d8c4,0x350f2e80), LL(0x3f6eb3a6,0xa76804d4),LL(0xe63fd72e,0xd2ce7028),LL(0x85be7c39,0xb27222e2),
   LL(0xfccf002e,0xe372c921),LL(0x514dc3ea,0x96a39294),LL(0xdad77509,0x4e6aeb46), LL(0xd37b54ed,0x9a5799a8),LL(0xfc0e43fe,0x64a87cee),LL(0x753b6b4a,0x7f05b1aa),
   LL(0x549af75f,0x807a2862),LL(0xd359bf07,0x1c8b5f7e),LL(0x95d01de7,0xf52d3291), LL(0x5ffe6146,0x7a461465),LL(0xe83b1659,0x224a8202),LL(0x470e8470,0x01236b76),
   LL(0x61865ca5,0x6208d875),LL(0x9571dfd5,0xa1a42854),LL(0x2d2d7940,0x6d5edb28), LL(0x6ff2b93c,0x8af66e50),LL(0x1baad60c,0xb6eb8e7d),LL(0x4a16a63e,0xbe114d2c),
   LL(0x388a91ae,0xbb783662),LL(0xc9cc2382,0x34e00a1b),LL(0x2aa28a74,0x66d0d031), LL(0x6fbf826c,0x5c0cc1a7),LL(0xc1ac7477,0x78033af5),LL(0x9a21f372,0x1d7e2839),
   LL(0x5af2db66,0x036516b8),LL(0xab38cfc3,0xd3fec155),LL(0x416d0d54,0xa6e7d1b4), LL(0xb397c8ca,0x9d762998),LL(0x4467324b,0x73e4c3a2),LL(0xbf7ada2d,0x9f7dd282),
   LL(0xc9c10ad9,0x3f452c2a),LL(0xb2609a4d,0x7a827181),LL(0xe31734f3,0xfb87d393), LL(0x2bbf4efe,0xdafffdda),LL(0x9837686b,0x53193aba),LL(0xd61a044e,0x17401c18),
   LL(0xf7fe6834,0x8561d6d5),LL(0x040422c5,0x20eee370),LL(0xc924342e,0xa16cd1c2), LL(0x0aba73e6,0xc017dbd6),LL(0x9e7b1683,0x209b1479),LL(0xf14b41af,0x4a639bbb),
   LL(0x0be4ccf0,0x59308c11),LL(0x49cc5fa5,0xa69616fd),LL(0x35696487,0xed6dfed1), LL(0x1657ce60,0xdef27e77),LL(0xab4a8d76,0x346c3382),LL(0x98a008a6,0x54c10f0e),
   LL(0x9d792671,0xbda7a3ff),LL(0xae4fafb1,0xab2ba801),LL(0x79f9c043,0xe27406e3), LL(0x14bcd570,0xefff1790),LL(0x80a00e1c,0xae57b991),LL(0x7b38c603,0x63b01e35),
   LL(0x388ceb24,0x7f1b2eef),LL(0xadc25ce7,0xbf81721a),LL(0x6ae36f35,0x1fd0c684), LL(0xfa611602,0xdae9c8db),LL(0x3f40a562,0x9d7b2264),LL(0x9e4cec86,0x64de9a7e),
   LL(0x396229d2,0x413e342b),LL(0x0afa76eb,0x97291d8c),LL(0x790ad9ca,0x18f63384), LL(0x307ad8f6,0x90deccfd),LL(0xcc49b61a,0xefb21d4b),LL(0x3d724666,0xcda971d5),
   LL(0xb10df2cb,0xe1a0e3dc),LL(0xe9334feb,0xe5509cd4),LL(0x38f21828,0x4dae1419), LL(0xe9309400,0x932c62f1),LL(0x289fd6f2,0x14acca95),LL(0x4e1325f1,0x92934423),
   LL(0x0edb12da,0xbc9bd40a),LL(0x3cd6bd8d,0x78f3256c),LL(0x644bdf08,0xb8fe8191), LL(0x314f2c73,0x3ba94523),LL(0xf3d8e905,0x33ddf165),LL(0xdd9721f4,0x603ecb74),
   LL(0x04e15cb9,0xa1a8d802),LL(0xe4eef1ba,0xb3bf09c2),LL(0xf7ea8116,0xfcaecf4f), LL(0xe68eb8dd,0x72f4dc39),LL(0x761271cf,0xeab7abaa),LL(0x51b8259a,0xfe99e9d8),
   LL(0x625f396b,0x6325bcfc),LL(0x7ad03440,0x36560d57),LL(0xaa639416,0x76246bab), LL(0x8fd064b6,0x381de8c9),LL(0x95537e9e,0xe4c3f8ac),LL(0x6e56f0e3,0x0dd1b6fa),
   LL(0xf065498d,0x96a10f94),LL(0x0038c2dc,0xf88a1867),LL(0xc58b18bc,0x0de21830), LL(0x9a5da75f,0xc86e1dd4),LL(0xb092c397,0x05e00843),LL(0x3154c419,0x2839c7bd),
   LL(0x18ef22bc,0xf6457fe0),LL(0x1df42e96,0xbb096001),LL(0xdef286c9,0x1a337c91), LL(0x4580b8e7,0xd4bb225d),LL(0xa2ce3586,0x6bba9b93),LL(0xf2d150c2,0x9d51106c),
   LL(0x943953ef,0x0a3f8284),LL(0xe9c8b50e,0xa6f7e5d6),LL(0x1bfa459d,0xe75546eb), LL(0x34529d4a,0x3846abb5),LL(0xe1da3b07,0xbbbd1f98),LL(0xfc1e8935,0x05a92469),
   LL(0x6e52f511,0x3edb02c2),LL(0xe6cea9d0,0x0306e54f),LL(0x5fe7cc33,0x1944a8aa), LL(0x53fd6127,0x3da98252),LL(0xce628fe6,0x9d5fa436),LL(0xb3f9ba33,0xe7ccb20b),
   LL(0x2b5eabc9,0x22d21323),LL(0x8ddb58d2,0xfa85732a),LL(0xc0a4b7a7,0xf80ab9d0), LL(0xefe1b0aa,0xe505f00a),LL(0xd93f46b0,0x1388c1d5),LL(0x94416a57,0xf44495f8),
   LL(0xd1b9d175,0x3e5e07cd),LL(0x28229715,0x713fb4a2),LL(0x750b4e8c,0x6a1dc0df), LL(0xb1837ba8,0x2b9fe39c),LL(0x8a4e51e1,0xad0b8e77),LL(0xed0e4aae,0x823a02e0),
   LL(0x5f16b838,0x8a9de11c),LL(0x95820340,0xfda67f12),LL(0xe5b3813a,0xaf424ffa), LL(0x1961cfba,0x8eccd1ab),LL(0xf3022e67,0xacea8ccc),LL(0x1522dd90,0x799ce2cb),
   LL(0x85c6d5b4,0xad3a9564),LL(0x11a4286d,0xbf57a898),LL(0x86df6b27,0xeb453d87), LL(0x4baef190,0x676b9e41),LL(0xa947a521,0xf94b92ee),LL(0x71257aaf,0xdc1c303e),
   LL(0xe1f3b4df,0x84384afe),LL(0xc6f889c1,0xb9f7ae71),LL(0x0dc943bb,0x2f227f6c), LL(0x994e3369,0xd7a6ccfd),LL(0x4042431f,0xf4480749),LL(0x54ff110a,0x4abda3ff),
   LL(0x9882e341,0xd50de25e),LL(0xed0228e4,0x7e68284a),LL(0x794de327,0xcb751d4a), LL(0x99ac62a1,0xb3cb0de3),LL(0x0001de50,0x870815cf),LL(0x82c9f5fd,0x7957d4ee),
   LL(0xbda3fcde,0x8760035a),LL(0x5734a5bc,0xf131439d),LL(0x1de171b9,0xbef8fa6e), LL(0xb076e2ab,0x35d32607),LL(0x8bfa83eb,0xb132de84),LL(0x1b272008,0x7ca46b88),
   LL(0x4437ac91,0xbc0e5b55),LL(0x7381e979,0xd7cb1a53),LL(0xd154616b,0x906abe2a), LL(0x4c1b6f06,0xebc02151),LL(0xf2e0afa1,0x6ede04a9),LL(0xa50c46b3,0x7f666548),
   LL(0x4abfde1a,0xda143ed3),LL(0x5fb5c163,0x968f5f5c),LL(0xc56b0f29,0xe9641696), LL(0x40766adb,0xcaade44d),LL(0x41a6905e,0xfac69f2b),LL(0x96dd9b54,0x8a2a6278),
   LL(0xead8c9c2,0x3a498bf3),LL(0x38527bb3,0x3f5fdc4f),LL(0x5a898416,0x78259bc3), LL(0xd2bc2a01,0xd7f94979),LL(0x058e4b40,0xb28701a4),LL(0xddf1c2fa,0x9908452f),
   LL(0xd85e5adf,0x30be2fcb),LL(0x492081e6,0x740f18cf),LL(0xd2f40d30,0x4d4090f9), LL(0xf3be491b,0x3770d2ac),LL(0xd47a45ef,0xad71d7b8),LL(0x85b820f4,0x0b2cd763),
   LL(0xa83085f6,0x1d1f6f65),LL(0xf7b897d7,0x4643738e),LL(0x209fede2,0xa9ae5e94), LL(0x929a2b44,0xf63bafe4),LL(0xef7d17e3,0xeb605e21),LL(0x526ff05c,0xa858b5cb),
   LL(0x3d3a82b9,0xd365dd2e),LL(0xcc2cfd70,0xfeccda5c),LL(0x344ed0f5,0x36ffafe6), LL(0x04ae99a2,0x934a88f7),LL(0xc20491ef,0x0af5b488),LL(0x7a66a1c8,0xcda2acb7),
   LL(0xdd5e1098,0x2b25a1ea),LL(0x909058c4,0x345012e3),LL(0xb5670aae,0x965a49e2), LL(0x1766fa93,0xc31d4480),LL(0xd3941b2a,0xd6f8bac6),LL(0x6eb3985e,0x46cd2776),
   LL(0xbf004633,0x4c45b908),LL(0xdec0c1bf,0xe6b9457e),LL(0x78c0aa4f,0x2c6f62d8), LL(0xf80fdbd0,0x32bd6508),LL(0x97f86a7a,0xaa4f54c5),LL(0x2eee225e,0xc3186238),
   LL(0xca5d7fb0,0xc11da036),LL(0x79e6d224,0x2bc98024),LL(0x779b9f68,0xed3b8e2b), LL(0x2c18b348,0xac7614f9),LL(0x36c756d6,0xc2072c63),LL(0x35e4340e,0xa6c5e629),
   LL(0x8d338dff,0x09c64919),LL(0x8d7f5e34,0x347f9f9d),LL(0x8044621a,0xd3012dad), LL(0x27327013,0x4520210b),LL(0xe20f7a66,0x9f76824c),LL(0x7b70408e,0xa136be56),
   LL(0x455a2af9,0x4b1f9940),LL(0x48945af9,0x1385b4c9),LL(0xac68fc09,0x3da47592), LL(0x5f05a04d,0x533995d0),LL(0xd7c5146e,0xc20d45f0),LL(0x22d2ae20,0x351ffb64),
   LL(0x7ab5d0f9,0xb5eb2af1),LL(0x7762a3a5,0xc5cc2409),LL(0x6b374f6a,0xc65f2281), LL(0x0f46e6ce,0x440d2fe0),LL(0xb868814b,0xde15331e),LL(0xee22f041,0xd1d2f6c0),
   LL(0xee1b9004,0x9ff4734b),LL(0x2e67a085,0x062dc3a3),LL(0xde8f3162,0xb7ea5b38), LL(0x824c31e7,0x19ce04cd),LL(0xbc61fd9e,0x4252601c),LL(0xf45482d3,0x95c7e121),
   LL(0x31057137,0xde58e0a9),LL(0x0cac0d1b,0x30c96d91),LL(0x9104641a,0x0daab9d7), LL(0x1fd0668e,0x9ea156fc),LL(0x977683f4,0x0f6b404b),LL(0xa30e9a24,0x7966c84d),
   LL(0xbff9caf0,0xe79728e1),LL(0x90e27f5d,0x82d66ec6),LL(0xbc650306,0x5ea57c74), LL(0xb2a82b96,0xcbe5e4ba),LL(0xfdfca395,0x24199d5f),LL(0x06991511,0x0ccbd94e),
   LL(0x7876e2ff,0x4c91e7e0),LL(0x5d650fe2,0xba74af49),LL(0x5c7b9759,0x6f6df70a), LL(0xed511d0e,0xa5be2df5),LL(0x0c9af38f,0x0f540f6d),LL(0xb785f2e4,0x646f1fef),
   LL(0xcb1eb7ef,0xf07cd946),LL(0xdd475a7a,0x41582424),LL(0xeccd0cd0,0xecabb646), LL(0x20672fdc,0xb08c9043),LL(0xd4fabfa8,0x0a0e852e),LL(0x08465cac,0x2df86c08),
   LL(0xaddeb802,0x4a9d718e),LL(0x19257426,0x442b8a24),LL(0xf292c043,0xe1bf4df6), LL(0xf420119a,0xd846f55a),LL(0xb81b55b6,0x883b5be3),LL(0x7fa04e04,0xd233b497),
   LL(0xb25f824c,0xa72a9e9b),LL(0x148d4350,0xcd9fa1a4),LL(0xe649b6b6,0xfe7be423), LL(0x1f055a16,0x54ff5456),LL(0x99348b7f,0xc73dc1b8),LL(0xb58ae286,0x072f9ae0),
   LL(0xa2f354be,0x17fbb428),LL(0x8e54a9f4,0xf07c8cc9),LL(0x26d6dff2,0xc5dce60a), LL(0xfb7068ab,0x73ba83b1),LL(0xd6ccfb08,0x7f786cdc),LL(0x298a0fd6,0x57ea8ba3),
   LL(0x6c93cbfc,0xb4637d1a),LL(0xbdb72d64,0x3b43839c),LL(0xa8ecb34f,0x80874787), LL(0xd5e642b2,0x57d9b566),LL(0x7a9c42a5,0x923772c0),LL(0xff8b1b47,0x24629516),
   LL(0x07ceeae9,0x120c441a),LL(0x07740927,0xe82c0626),LL(0x91dda975,0x82b56342), LL(0xe8904b5f,0x5ccadd6d),LL(0x72da2998,0x55dc6fed),LL(0x5595c1fd,0xa6d506c2),
   LL(0xeab8eea0,0x1a6a3b0f),LL(0xcafabd92,0xdf46f04a),LL(0x297dbb29,0xc5e2814d), LL(0x71d1e312,0x39d8e04c),LL(0x265b4075,0xc7017a13),LL(0xf78fbd26,0x4c21493b),
   LL(0x0b9f158c,0x5c28c603),LL(0x9a07acaf,0x33de6098),LL(0x0e9c4c73,0x7b77cebd), LL(0x498db066,0x62e1c44c),LL(0xe28af701,0x57f684b7),LL(0x1f8b8dfd,0xac841f0c),
   LL(0x6bc2d75f,0x9ca86d3a),LL(0xe7876500,0xab63e1ca),LL(0x436712c4,0xab42a0ee), LL(0x1d001327,0x8aead0f1),LL(0xf0a70715,0xf563e655),LL(0x6f079f79,0x58f26dd9),
   LL(0x2922bad4,0x2b795683),LL(0x83b5370a,0x16a63631),LL(0x2638ee32,0x530fec6c), LL(0x44d0b441,0x23b918a6),LL(0x4b00c9b9,0xf1ebe8e2),LL(0x5e410b3b,0x77681a00),
},
/* digit=7 base_pwr=2^49 */
{
   LL(0x1db6b479,0x22b1c723),LL(0x39e84714,0xd62f06f1),LL(0xe23256f7,0xdee9ec7e), LL(0x95968194,0xe889db11),LL(0xa9e6d7d7,0x11d66cd0),LL(0x62416021,0x09b2418b),
   LL(0x6adf5658,0x0be70575),LL(0x73b1138c,0x17f6b55a),LL(0x2a9fc1a8,0xce3bcb88), LL(0x01211935,0x4edfa8ef),LL(0x9b82e1b6,0x8972dc1c),LL(0x3e9d7b44,0x84866f69),
   LL(0xcee3c64a,0x3f2da340),LL(0xf63dc83e,0xfcedb4bf),LL(0x56ff2a25,0xdedd2aac), LL(0x250762ff,0x9b5af4ff),LL(0xcbf6e216,0xd67cbdc0),LL(0xae6521ed,0x1b147e2c),
   LL(0xcfcb0905,0x5315d6ee),LL(0x1a685aaa,0x269200d9),LL(0xc1f724bd,0x96f1c3a3), LL(0x3025b425,0x3a231812),LL(0x39d51119,0xff7672e6),LL(0x85a58c8f,0x1280679d),
   LL(0x1ee4c1e9,0xe004f1c3),LL(0x6695e2c7,0x51f5f62b),LL(0xad1a015a,0x4c8db9dd), LL(0x0175ea4c,0x141a6846),LL(0xdb97814c,0xf8e3d91e),LL(0xf678fa80,0xdb30a211),
   LL(0x0a4bf8e8,0x6e08e548),LL(0x95d4a1f9,0xd1e1533e),LL(0x74eb6623,0x01c6b2d2), LL(0x584b99ba,0x9e3dcf2f),LL(0x44a872c9,0x1211df9b),LL(0x13cf855a,0x44d0a919),
   LL(0x7a71a32a,0x689a8dc1),LL(0x0969c604,0x6862a375),LL(0xbe398e22,0xd324e9b9), LL(0xfe84fe60,0x2175c004),LL(0xa817cdb4,0xa607ef1f),LL(0x721cb883,0xcd9d9495),
   LL(0x3a36ef0d,0x4f8ebef8),LL(0xf4ef55c3,0x5c910c5a),LL(0x2a959873,0x8ce26809), LL(0x376d6a96,0xc2132558),LL(0x026083f4,0xa6c1cfb2),LL(0x0d455653,0x46c5dbd9),
   LL(0x8ba48c45,0x48969551),LL(0xc643319e,0xba14588e),LL(0x6164409f,0x79cc5f79), LL(0x7cd52144,0xbbcfa5ac),LL(0x01be08de,0x291192cf),LL(0x1b5535ee,0xe09fcd9b),
   LL(0x8b222a3d,0x4e761342),LL(0x9ee6a22d,0x874f88e9),LL(0xdbb503a4,0xfb29bb24), LL(0x27a88472,0x1e5923d8),LL(0x27cc978b,0x968ac6a8),LL(0xb31d7b22,0x8cd1a928),
   LL(0x0643e7f0,0xc14f22a2),LL(0x97a5e590,0xa920ca01),LL(0x2c3fef97,0x4e4e7c4b), LL(0x8e7baa94,0x0200b159),LL(0x16b1757c,0x24b9c68b),LL(0x6ca9d309,0x51ff6499),
   LL(0x96b7c630,0x7b132fca),LL(0x63d2264b,0x8df911e7),LL(0x164dbc12,0xff71dd08), LL(0xcd036c72,0x2f7b92af),LL(0x24d891cc,0xf1336105),LL(0x66b47dc3,0xe614aee7),
   LL(0x12ead645,0x1c53214c),LL(0x3c7f65a7,0x32aa24b9),LL(0x118a5225,0xfd7b67e9), LL(0xdad33487,0xf58a2515),LL(0x3a563a0b,0xa9ce4387),LL(0xd92455a8,0xad5dc427),
   LL(0xaca6b802,0xfa852455),LL(0x26bfb5b4,0x448d6d2d),LL(0x0accc4e4,0xa76981e9), LL(0xfbedc1b3,0x2c43f05a),LL(0xdfa0cdbc,0x684e84d3),LL(0xb07a9af9,0x60fa3028),
   LL(0x46cbf3c2,0x55adc2d2),LL(0x0ab9aba5,0x5ddaea63),LL(0xf06d91eb,0x457318d3), LL(0x3face92b,0x210c0bda),LL(0xc8cca904,0xcb6be0b9),LL(0xac932c9e,0xcc059269),
   LL(0x3b076836,0x35440607),LL(0xcfbc594d,0x6e139ab7),LL(0x0f818173,0x24c24eb1), LL(0x4c3d9a9f,0x26b87e2a),LL(0x1fcc4242,0x03671da7),LL(0x643cbca5,0xd08f6b5e),
   LL(0x69cce513,0x3b1962bf),LL(0xbbceb26d,0xe0091d0c),LL(0xf374c0a2,0xda304365), LL(0xa099ddf8,0x27a29bfc),LL(0x969c677a,0x5f53ba53),LL(0xf0627255,0x77b8d2a7),
   LL(0x3819f9dd,0x221375dd),LL(0xe08b0ca2,0x04a971e7),LL(0xd46c54ac,0x5f7345a7), LL(0x93d0f966,0x8f01f13d),LL(0x6ef17236,0xf5cf276c),LL(0x0dd038f0,0xca0bba4c),
   LL(0x525bd2b0,0x86ab6164),LL(0xa0a0d81f,0x0fddac9e),LL(0xe3827450,0xc5322927), LL(0x7081e340,0x5965a1a9),LL(0x427b0701,0x46c3aef7),LL(0x6692b94a,0x98a9fa7c),
   LL(0xa495e99e,0xd2932e7f),LL(0xd82f6584,0xdeb528fd),LL(0xbe305394,0x93b3c1c1), LL(0x7d8622a3,0xa4cb19b6),LL(0xb6d5eb26,0x7b46a163),LL(0x9239f879,0x247e327b),
   LL(0x57a9df34,0xa102051f),LL(0x1aaf0cff,0xe1fbc51e),LL(0x9fd0f53a,0xe533029e), LL(0x75a0f971,0xda45f13c),LL(0xfe0d2922,0xadee768d),LL(0xfc9bca7d,0xdcf7b7bb),
   LL(0x39f2a51e,0xa84b4adc),LL(0xa4c4231a,0x9c257e5f),LL(0xe6835319,0x5dd04181), LL(0x276cf633,0xb2ea7ce6),LL(0xd0c18fe5,0xf245702e),LL(0xb338e5a9,0x994cccff),
   LL(0x33e5c5c0,0x6ba4e93a),LL(0x258e8029,0xa7b7d1f8),LL(0xad202aff,0x304ea934), LL(0x54bb2c25,0x7b28d6b0),LL(0x50b1a4e9,0x624bdf85),LL(0x7ddc1ccd,0x2c1079d6),
   LL(0x7bc06924,0xd757678b),LL(0x4a9230a8,0x761f6494),LL(0xa38af7ae,0x4a74e816), LL(0x55556207,0x5cb7e1ad),LL(0xf5417b00,0x51a88664),LL(0x78c7935e,0x48886cf1),
   LL(0xc5ae666d,0x112a047b),LL(0xe0d72464,0xe3e8915c),LL(0x88f740ce,0x18fd8fe6), LL(0x67d49be0,0xa5843ed0),LL(0x91b85641,0x44616e3f),LL(0x40e3c66c,0xfc394c04),
   LL(0xce689251,0x4d6ed19e),LL(0xf28dfceb,0x81078128),LL(0xc82c2dd5,0x71bade6c), LL(0xc7bf5e3a,0x20cfbae4),LL(0xd1983380,0x0d1c23fb),LL(0xfb5cb3bc,0xd56434b8),
   LL(0x496c7720,0xdbdd0273),LL(0xd7c99031,0x20b010b3),LL(0x06d61176,0xf5ff610c), LL(0xd3631b15,0xa0b6ea34),LL(0xe950e355,0x1528b1e0),LL(0xf0c84a6d,0x6e27fab8),
   LL(0xf55bab95,0x8e66ed20),LL(0xbc6b300a,0x63e73fb5),LL(0xab652058,0x753ebe56), LL(0x46319ffa,0x0d97e216),LL(0xfbb99b70,0xf605457b),LL(0x8829b2aa,0x03a28a89),
   LL(0x0073b797,0x3eac149e),LL(0x9bfeb2fd,0x95792d96),LL(0x7dc85ebc,0x0ccf0041), LL(0xecc61042,0x449bf3dc),LL(0x986664d6,0xaa4fce55),LL(0xa8c146ef,0xbc5ee8ae),
   LL(0x1ff1dcbf,0xd904f9b2),LL(0x5d8a1197,0x35a18d0d),LL(0x6e3d4745,0x3341db2d), LL(0x4b050939,0x6b1eec19),LL(0xd70a0a15,0x9c1fe3f4),LL(0xcdfdd83e,0x654fc4dd),
   LL(0xe9ba38d1,0x4ebcf740),LL(0x7de3b4a7,0xb42141e4),LL(0x58b093ca,0x09454a65), LL(0xa475ef3c,0xa87bd8d2),LL(0xcd2de97b,0xa1e89c23),LL(0xcb134c1e,0xe3000fa9),
   LL(0xe03d89fc,0x56e13148),LL(0x560317fc,0x737b874d),LL(0xaee37ff7,0x28ce8556), LL(0x600002bd,0x21a8f876),LL(0xd20a7373,0x11a46449),LL(0x849772fb,0x280be2ee),
   LL(0x44d8341d,0x81aa4216),LL(0xcfc93e1e,0x413bbc51),LL(0x5cbb96ed,0xc61e7fbd), LL(0x348b93bf,0x183ffd78),LL(0x1a6cac3d,0xbc237f9a),LL(0x075f2c01,0xcbe09c38),
   LL(0x05a97f5f,0x337c5514),LL(0xf04dfeee,0xc2e77233),LL(0x5bb4ca71,0x400f4852), LL(0x8553221b,0x9626c894),LL(0xb57b0765,0x11aa8bb3),LL(0xa12d3f73,0x543ce8fa),
   LL(0x8faba6a0,0xf77adb46),LL(0x435376cf,0x1174ee49),LL(0xedd2bd00,0x19430a06), LL(0xe4cbbaef,0xbfdb0a6f),LL(0xfe5c89b0,0x3b520354),LL(0xbb7ab998,0x8acd5ed9),
   LL(0x28bc63d9,0x4358e216),LL(0x6d84054f,0x1d99f606),LL(0x923248e3,0x8e24aa2e), LL(0x2d2082d7,0x004093da),LL(0x0c19a153,0x180e72ac),LL(0xa4539eb7,0xb4d8b7b3),
   LL(0xab583068,0x850a27b8),LL(0xeacb954e,0x85790f5d),LL(0x533b4d63,0x49567edd), LL(0x2c83d758,0xeedc0e79),LL(0xfcced827,0xe007c1d7),LL(0x2000bb00,0x68232c77),
   LL(0x2ab39a53,0xdce3a7d4),LL(0xe408976d,0x5401ad0e),LL(0xb9ffd974,0xfcd5a5e8), LL(0x06214587,0x6adae145),LL(0x3b0e57df,0x15d783fc),LL(0x35f73e2f,0x8d4a1061),
   LL(0xb2e9c8be,0xf6b94404),LL(0xe122287a,0x57269f9d),LL(0x7f0b3575,0x3f9849e2), LL(0x691d4ea8,0x1d85bc98),LL(0xc811fe44,0xbf34f840),LL(0x6fdeeca9,0x188d1f53),
   LL(0xda6d3dcf,0x78294002),LL(0xaddf5b48,0x3f86f8c4),LL(0x6f8761ea,0xca9c8dbb), LL(0xb56d54cf,0x6d4be379),LL(0x3d32e3d0,0x9504874e),LL(0x1b4d470d,0xe0959574),
   LL(0x60c89192,0x03618e26),LL(0xec22e3da,0xa067fbde),LL(0x202d430d,0x5656229a), LL(0xc7703466,0x1f782bd8),LL(0xa0026a97,0x3b278178),LL(0xf1c08768,0xc86a636a),
   LL(0x754cbc65,0x175ae3ec),LL(0x4f765d06,0xc6f3677d),LL(0xa11dc7c6,0x699a40ca), LL(0xc2c20678,0xf85e4e41),LL(0x3ee230c1,0xc4fd583c),LL(0xf604d6e1,0xfbf97e3e),
   LL(0xc6e43d77,0xc2a7b60e),LL(0x5e09cfba,0x5c44a7ab),LL(0x20323a01,0x31279294), LL(0xd83e9122,0xbef510c2),LL(0x910635c8,0xce89cd72),LL(0x6525654f,0x40c3123c),
   LL(0xabb9fa3a,0xe5efa39c),LL(0xd54d0553,0x0dd062d8),LL(0x5f654ea6,0x2447ebea), LL(0xb205ee7f,0xc6b30d78),LL(0x6b327d54,0xac800865),LL(0x2d494627,0x3564831f),
   LL(0xbf996e35,0xe5612ddb),LL(0xb2c8a71e,0x82b679c1),LL(0xb8522766,0xadfb7763), LL(0x6edb7f6b,0x31f681bc),LL(0x6c1396b8,0x0ac77865),LL(0xc16391a7,0x4fec5edc),
   LL(0x6a8ba5e0,0x79be8744),LL(0x76a3f5dc,0x9cad48d3),LL(0xd3a4a677,0x3b4ddd6a), LL(0x2427a393,0x53d9dd5a),LL(0x99061179,0xe3992757),LL(0xc5ab35de,0x9a76a0ad),
   LL(0x270ecf31,0x18e8dbf5),LL(0x3f3da910,0x6723152b),LL(0xe687db02,0x74b53e35), LL(0x89fccc8e,0x72dd231b),LL(0x6ca0a007,0xdf98fe28),LL(0x9b8fc127,0x5c469a08),
   LL(0xf7c47e53,0x3d45d6b3),LL(0x2bca9e62,0x830867c6),LL(0xcef81d38,0x84276d68), LL(0xcc30ac2b,0x60d227cc),LL(0xaa747e61,0x514c1992),LL(0x4ed869e9,0x90083817),
   LL(0xf15ff76b,0x9c199d37),LL(0x4a4fc2fd,0x0e5ab91a),LL(0x25ed5ca0,0x142d9700), LL(0xc7398261,0x417bd5e9),LL(0xca5133ba,0x074f5397),LL(0xbbf69817,0xda80ec1f),
   LL(0x29f549a1,0x66cc5bce),LL(0xd6af775a,0xbe183389),LL(0x7b066b44,0x492f66bf), LL(0x431fc807,0x0f1397b4),LL(0xe83dd557,0xb6f675ba),LL(0x62489a2a,0x972d9038),
   LL(0x8eff888d,0x71c228f7),LL(0x75586fd7,0xcd59c5d1),LL(0xdd3143e7,0xa966bb14), LL(0xbf353a1f,0xa7f87d54),LL(0xf484d5f8,0xf9cbceff),LL(0xb68ff698,0x698dae86),
   LL(0x6f3235f2,0x66286731),LL(0x1579c24d,0xcba86933),LL(0xba144ac8,0x8864f364), LL(0x3ffd19b1,0xaad3e749),LL(0x6fdf894e,0x148d7a44),LL(0xb448c223,0x3aefa6dc),
   LL(0x84e0548b,0xc6c90b42),LL(0xa6072bb0,0x9234d1be),LL(0xafb14b2a,0x3b28ca69), LL(0xcfbcc4d3,0x80eda096),LL(0xac4275fd,0x8bf29e73),LL(0xa9e0838e,0x10655c00),
   LL(0x8eff3cbe,0x75e78f0c),LL(0xb0a9ad9e,0x2d549d6a),LL(0x50d83c74,0xd9179f56), LL(0x45988bd0,0xdf61dfe9),LL(0xe8f87cbb,0xb76286ea),LL(0x033c1c00,0xa64bca3d),
   LL(0x664f67a7,0xd82e91bd),LL(0x19c561de,0x7329573d),LL(0xce6bab53,0x5faf92d6), LL(0xcc978c65,0x715cc536),LL(0xe1e82686,0x843e802e),LL(0xbb3c2514,0xd7a82374),
   LL(0x46e3c384,0x7dec5bcc),LL(0xb5c8409d,0x874ece30),LL(0x0fb8869d,0xbc0f588d), LL(0x65d88ad2,0x11c4d95d),LL(0x349f7fc1,0xef6a5587),LL(0x5d3c89c7,0x8ceb5e7b),
   LL(0xfdbac03d,0x8326fccd),LL(0xc0e3814f,0x8592d5ac),LL(0x92780f4e,0x4c4f9181), LL(0xcbd77d47,0xdfa95b95),LL(0x116d8d69,0x5c695658),LL(0x45d71633,0x3b9e5958),
   LL(0x1f933a54,0xf15d931e),LL(0x572c2915,0x16a8e689),LL(0x6078e8b9,0xe37672fe), LL(0xb7252a68,0x37af9fd5),LL(0x6b8ac148,0x9e17b01d),LL(0x942a5a17,0x5ad4c09c),
   LL(0xef904c15,0xcf9f44da),LL(0x1b8bd3f2,0xfd9b0973),LL(0x1a01c966,0xa2209dbc), LL(0x4d678467,0x0878c83b),LL(0x41418cd8,0xdfcce3e9),LL(0x24b1f3e9,0x10ed020b),
   LL(0x6eb0a25e,0x1d05d913),LL(0x6ba1b2f1,0xd8dfea69),LL(0x4570c9ce,0x224f88a7), LL(0xb8c5a0d1,0x82087d08),LL(0xcdf8086e,0x2d689252),LL(0x33330715,0x7f5cbba0),
   LL(0x143708d2,0x793030ec),LL(0x3aa671de,0xe1e44a09),LL(0x6d0c6b60,0xf8d14831), LL(0x4c6b4751,0x43814c65),LL(0x07c35ce6,0x23531a9b),LL(0x30f97153,0xec6a9878),
   LL(0xa300f8ce,0xd0ced276),LL(0x1ef83f9a,0x7eb2f7db),LL(0x3ff788a7,0x8bdc98f4), LL(0x29e0e1c2,0x8c5f9af2),LL(0x7ac14e4e,0xb6cd8700),LL(0x073f9d19,0x54909c4d),
   LL(0x3fb6efd9,0x9481d096),LL(0xb50cc9b5,0x04a156f3),LL(0x8baffff7,0x4b8e541d), LL(0xdb644174,0xe7f5e90c),LL(0x530c3696,0x754d867a),LL(0xf47978bb,0xfb20bb76),
   LL(0xd5e0c70e,0x16986cc1),LL(0x52e7e6c7,0x42ce3058),LL(0xa618254e,0xf31ea341), LL(0xae1ae829,0xaf2001d4),LL(0xd97c4da7,0x3f93eca5),LL(0x8b5c3229,0x01dd7a1c),
},
/* digit=8 base_pwr=2^56 */
{
   LL(0x0c6a351e,0x6c78a1f4),LL(0x807a4e21,0x5192b2cf),LL(0x72479dc8,0x531cf1e5), LL(0x15863b65,0x78f3e85f),LL(0x6f917ccf,0x008524f6),LL(0xd0dccb94,0x531192d9),
   LL(0xc490a7f0,0xf7d5ce6e),LL(0x06cc1312,0x646319a8),LL(0xbc37af6c,0x01d71545), LL(0x7f32c096,0x92077f6a),LL(0x02223fa6,0x3c4192cf),LL(0x0b59de1f,0x72b36fe0),
   LL(0xee192ea3,0x6cc9d6a8),LL(0x4008c57e,0x951ca552),LL(0x0ccb74a8,0x0906ec42), LL(0x09e39599,0x80c4c111),LL(0xa5246b67,0x4a79b1c3),LL(0x5ead0c0f,0xff7242ea),
   LL(0xf2d97c8d,0x9138c2cd),LL(0x2650a9b4,0x3caecb47),LL(0xf8760ddf,0x23a73447), LL(0xc1d98546,0x1ebf0619),LL(0xcc7f0b0a,0xd9f11b6e),LL(0xde8d96f1,0x16ab2c05),
   LL(0xeffca98d,0x0dd3b147),LL(0xe1abc2a9,0x10c4fdcc),LL(0xf74f036e,0xc4b7b199), LL(0x7301d513,0x6db9d5ac),LL(0x774b8020,0x154acf22),LL(0xa60f3784,0x0030085f),
   LL(0x859de975,0x6b73749b),LL(0x67883183,0xc97b9ead),LL(0xbfb919b1,0xaaae5948), LL(0xf5c8912d,0x8150189a),LL(0x67fe0faf,0x6b988a5c),LL(0xf5839146,0x21d1f6d9),
   LL(0xa3635200,0xaf8c2c49),LL(0x1d377a92,0x0400e2f7),LL(0xfa0d421c,0xaa0ec06c), LL(0x0b0d997b,0x6e3c3b69),LL(0x6756ee3e,0x7d3832dd),LL(0xe78e3204,0x30d62b54),
   LL(0x5174cd2b,0xcb5e1dcb),LL(0x18f6e4b5,0x8984cb7b),LL(0x72a12150,0x363d0f1f), LL(0xcc62f17b,0xb5c36272),LL(0x3a2f964b,0x4a992731),LL(0x4e8f1ce1,0xbca57238),
   LL(0xff1a436b,0xdfcf7d1d),LL(0xee098e33,0xf6028d75),LL(0x85091828,0xf98fd92a), LL(0x8cbf1897,0x535e81c3),LL(0x100e48e4,0x4b9286c9),LL(0xa517213e,0x6dcb2c57),
   LL(0xb451f187,0x1bb72786),LL(0xac01e718,0xbfa69395),LL(0x9acebb3e,0x2c1852ea), LL(0xddabfc41,0x5eae9e20),LL(0x1bd561f4,0x661e7254),LL(0x270a6fa5,0x71f20425),
   LL(0x961036e2,0x15abdd47),LL(0x35b92111,0xe3edb318),LL(0xfeae7403,0x6a5f093b), LL(0x1595973a,0x4bca351b),LL(0x3a5ff8b1,0x10b5c69d),LL(0x826996cd,0x61699ca4),
   LL(0x78bdb771,0xef90144f),LL(0xa6fa7f1d,0xc15f7dab),LL(0xb45483c8,0x367a4d90), LL(0x770b4b1a,0x3524f21b),LL(0x0446fddb,0x31111556),LL(0xf304737e,0x77b6be1c),
   LL(0xabde9561,0x37c8901e),LL(0x122cf2de,0x607d5c3b),LL(0x8479673d,0x31a6f869), LL(0xf1df773f,0x1bc908c7),LL(0x6e7f32aa,0xb13f1008),LL(0xd9fc2d4c,0xd0b64b4e),
   LL(0xf338ea23,0x4c0c5170),LL(0x49fbff41,0xb0639fbf),LL(0x6cdb87d1,0x0c71578a), LL(0x5d00e5a8,0x6a0fac6e),LL(0xfb624d14,0xf923d1e7),LL(0x65ec474b,0xe2af600c),
   LL(0x0afa00ec,0x96a0e907),LL(0xe93d5449,0xa58ae31f),LL(0x1e050306,0xf0090575), LL(0xce3c1478,0x3570e2bc),LL(0xd154329a,0x6c779b82),LL(0x8a7caddf,0x2e65670c),
   LL(0x428ffdee,0x7850b688),LL(0x1e3103a7,0x5630065c),LL(0xdc08408c,0xd774279e), LL(0x6eaf0bcb,0x85499490),LL(0x72f72173,0xb1954318),LL(0x4ac8220e,0x2e5e8eda),
   LL(0xe86f4f00,0xbd47e5b5),LL(0x8338414d,0x312a0a57),LL(0x5dc0dd30,0x8832fc40), LL(0x8be0076e,0x5b2ec051),LL(0x45348887,0x3064cc96),LL(0x61596442,0x9d0e00fe),
   LL(0x747e8db9,0xa49b588c),LL(0x1c9851ef,0x91bd49ac),LL(0x21c7c36b,0x547ad2e7), LL(0xcf0a07e7,0x91725ea2),LL(0xcbbcb82e,0xf7fc5911),LL(0x5646ba9c,0xd5f7714f),
   LL(0x8edccbb0,0x0f86777c),LL(0x7068d3ce,0x4df433f8),LL(0x77731ed1,0xfd2d02a0), LL(0x1f323ab3,0x9ce12310),LL(0xabf47059,0xab127e12),LL(0x4257eed4,0xe37b4198),
   LL(0x61b7a7bc,0x96bff478),LL(0x24a24d11,0xf70c91dc),LL(0x2ed5f8ef,0xf25c49f7), LL(0x8b8cd0bb,0x66103231),LL(0x337a16a3,0x9af494be),LL(0xbcdf06c9,0x8c8b1723),
   LL(0xb8d9c7fd,0x5370d17d),LL(0x1320d963,0xa9848001),LL(0x3ab21366,0x4b08ede7), LL(0x8ee6ae31,0x3640d969),LL(0x7deca729,0x782297a2),LL(0x8cf1b471,0xb99c1450),
   LL(0xc55e2971,0x3a206eb5),LL(0xe3b5b1a3,0xe65b0a7a),LL(0x7f789c24,0x611ca3ef), LL(0xaee050b4,0xffbed20f),LL(0xf7561895,0x81e86366),LL(0xc2fa8046,0x64671984),
   LL(0x0c3c9013,0xade34ec5),LL(0x5eb41025,0x50bd68e1),LL(0x0d4894dc,0x78808518), LL(0x7c29a657,0x7ed39b6b),LL(0x241517d5,0xbafdbdd0),LL(0x188218bf,0xd6f50092),
   LL(0xf03ef732,0x81177e12),LL(0x56a3507b,0x1574fe49),LL(0xab053718,0x6e960496), LL(0xc3a21218,0xbcd97662),LL(0x3bf70829,0x0cf68dea),LL(0x25b800e2,0x32d30c54),
   LL(0xc9428b22,0x95452160),LL(0x72a22c82,0xc2622e6b),LL(0x5b27cb42,0xb6731ec6), LL(0x4fce33d5,0x0f8a1fc2),LL(0x6407716e,0xa9838902),LL(0xfa98bc54,0x5e553dc0),
   LL(0xcaa69408,0x59683c61),LL(0x6598df18,0x82bc08db),LL(0x6634b238,0x154e5197), LL(0x43fc4c77,0x70ad310c),LL(0x835272f8,0xf9144e70),LL(0x9b7d0d58,0xf41beaab),
   LL(0x7a860980,0x7aef7397),LL(0x7175ed79,0xab751037),LL(0x797c12f0,0x7dd8fb29), LL(0xebd2f077,0xa63e2440),LL(0xce865a7d,0x645fe5ce),LL(0x317de834,0x5e57db53),
   LL(0x2bb8134c,0xc28c2ef9),LL(0x520d82a8,0x3cfa28dc),LL(0x7103b732,0xc31217da), LL(0xfef50290,0x9daf07ed),LL(0xb5fbffea,0x5b6cc3a2),LL(0x710a59db,0x98d2a6f3),
   LL(0x9174bf48,0x44b0ff44),LL(0xeef16e79,0xb5562ba6),LL(0xea9adffa,0x129bbad4), LL(0x8ccd2a64,0xc93aa7fc),LL(0x2ce754e9,0x0c252964),LL(0x88a4559c,0xbaf7f806),
   LL(0x548d0e57,0x924da3ef),LL(0xbe13e6c9,0xde8c8973),LL(0xd73979cb,0xe5dc0c62), LL(0xf5d1e78f,0xc2f92566),LL(0x95a8fdc6,0x3152a7cc),LL(0xdc528326,0x6601c017),
   LL(0xd38d9fea,0x1235bcd4),LL(0xd899201d,0xe4d00d99),LL(0x2709bee4,0xf0a9da65), LL(0xf33d4d62,0x69266ab1),LL(0x7a7c073b,0xc5c102a1),LL(0x1fddcaa7,0x42e277b8),
   LL(0x176de7d5,0xa3b38048),LL(0x3ae77e40,0xcc01f3c4),LL(0x1b8310bc,0xec1be1f1), LL(0xb0b6554b,0xbdc976c3),LL(0x9c4bcd13,0xd86117f4),LL(0xd41ce646,0x9feb9193),
   LL(0x2120685b,0x7d1547dd),LL(0x610034f4,0xb3b1ec97),LL(0x947b29d6,0xe98a9207), LL(0xdb48b264,0x3dfa24b2),LL(0xa5afaebc,0x6c8e33db),LL(0x0db735ad,0xfcad214a),
   LL(0x3a1e3514,0xbc366d47),LL(0xd584ab3c,0x67228d9c),LL(0x90009ee1,0xd8abe402), LL(0x68d15f9b,0xb06fcb6e),LL(0x3cbc8586,0x9f4fab36),LL(0xac1ca214,0xeb6fc7d3),
   LL(0x212701d8,0x644ffbc6),LL(0x3e7f5a64,0x0dd77816),LL(0xbba949bd,0x4182d12c), LL(0x9dd68a5b,0x566f707c),LL(0x3374e3d4,0xc32b41f7),LL(0x33a887f2,0xe25506ae),
   LL(0x7d747270,0x6809dda9),LL(0x34b9c161,0xe0ff644d),LL(0x5735d1ce,0xedce5894), LL(0x3c299049,0x9a14955a),LL(0xe770315b,0xdf5340ba),LL(0x09ab6581,0x21d8b57d),
   LL(0xa3a84876,0x6e2ebe98),LL(0x33705d95,0xb25b877f),LL(0x17be7330,0xfb32cded), LL(0xb3db892e,0x91cd62d6),LL(0x782eca61,0xca99ae59),LL(0xf953f71e,0xe927b38e),
   LL(0xfdefd2fc,0x932d6c0e),LL(0xd45bc513,0xc02f4213),LL(0x97a9f940,0xb9554eff), LL(0xe0f84e1a,0x4e5d7ebe),LL(0x2f4302e3,0x77035f14),LL(0x9555b2f5,0x45bcd681),
   LL(0xb757b6a7,0x55313f4d),LL(0xf62039e3,0xc1b3ca66),LL(0xa68faba3,0xbc76d877), LL(0x5fc39ebe,0x8addb47a),LL(0x2205e6df,0x52e4f8b8),LL(0xe999baf6,0xa4963df4),
   LL(0x01e9090d,0x43ec78a5),LL(0x670ed5c0,0xada7c41a),LL(0xca0ccd63,0xf441fc60), LL(0x9f8a35c9,0x2b6b14cb),LL(0xd67b0c7a,0xea90296f),LL(0x475bfd04,0x9ec66ada),
   LL(0xae9c5b50,0x785fcd1c),LL(0x43f8004f,0x1793f480),LL(0xff4b7c93,0x4ac0e11d), LL(0xbdbd07ff,0xfe1f7829),LL(0xb36c82d1,0x928d4b94),LL(0xedb0863e,0x9afbd51b),
   LL(0x35ef7543,0x45b8e6c7),LL(0xefcbf15e,0x04bedfec),LL(0xd26c073f,0xe5d2b221), LL(0x9eeedb4d,0xbb74aed4),LL(0x8329872b,0xbdeee909),LL(0xd5027bee,0xde0ef4fd),
   LL(0xfdec460d,0x4373d235),LL(0x5367ccc3,0x889db5d3),LL(0x6ba366f2,0x75b7adbd), LL(0x59e47152,0xda4f82c6),LL(0x74645dce,0x2f5d2404),LL(0x72715896,0x73bc8042),
   LL(0xc2538c33,0xd97aa542),LL(0xc4176743,0xb0d4f42f),LL(0xd9117f2c,0x4e46a104), LL(0xa1dbb89c,0x749b728c),LL(0xbb2d1a5c,0xc91c80f8),LL(0xc41a94ef,0x3668d74b),
   LL(0xc88dbe72,0x64a248da),LL(0x8c9c23c7,0x90d78a4c),LL(0xa56cbc8b,0xd49e40ab), LL(0x305322ae,0x764a4a3b),LL(0xd2f6d2ca,0xf13655d0),LL(0x0ad04588,0x5e206b78),
   LL(0xa4e2f277,0xb295d1f4),LL(0x2b0f59d4,0x1d49ae57),LL(0xfc85f875,0x77ccdf3f), LL(0x40f7816d,0x98ca5c5e),LL(0x2cb0558a,0xea990a23),LL(0x7a723586,0x76ef8645),
   LL(0xd4b23830,0x1618d79d),LL(0x0f9647c2,0x0a1387ea),LL(0x166ce29c,0xd2ba4d03), LL(0xacf33156,0xcc88e78f),LL(0xcaf5f200,0x57da063f),LL(0xae63a300,0xa5cbc1c3),
   LL(0x017ded2b,0x8abdda43),LL(0x02c5993c,0xabbe3891),LL(0xfd7ffd34,0x52df96de), LL(0xd7e89e8c,0x6c4d980c),LL(0xbd659dc7,0x8e3c41be),LL(0xfe1fe1f8,0x7f6f7496),
   LL(0xa4a51425,0xecae5b7b),LL(0xdd9d83af,0x81e2c971),LL(0xf7b85342,0x39b1d2a3), LL(0xfa8bf8c0,0xc45ad2c1),LL(0x7f18d82c,0xf495b15f),LL(0x9ccdc3b1,0x8b260d45),
   LL(0x1d241682,0x5a2e6fa8),LL(0xa49fd352,0x313e6fc8),LL(0xa7ddd234,0x75139843), LL(0xd3d7609e,0x6549aefc),LL(0x8f6fd0cd,0x1b477e03),LL(0x70f7a7f4,0x6140a881),
   LL(0x6718b3b6,0xaf1106f1),LL(0x44eaa8c1,0x767e6522),LL(0xf735f07a,0x527f2eed), LL(0x14ea415a,0x45ad0cfb),LL(0xabde5ef6,0xc09636ac),LL(0x9fcee205,0x88d7fa8d),
   LL(0x95543299,0x004861d5),LL(0xa61e24e5,0xd594bc3c),LL(0xb3b7cffa,0x2bcafd63), LL(0x5669358b,0xcb375694),LL(0xcb56b897,0xc5c022e4),LL(0x12de2824,0xb2b30bdf),
   LL(0x2cfdf68c,0xce6b8b13),LL(0x910076c2,0x942efa8e),LL(0x4719c7c3,0x661fc5fb), LL(0x82ebe6b8,0xcfdf4b8d),LL(0x3bb3c18e,0xdeee7446),LL(0xd641180b,0x00ed3964),
   LL(0xcb3382da,0x1a1ea271),LL(0x30bf39fb,0x0256f492),LL(0xaf66aa74,0xc817dad0), LL(0xf968fb2d,0x2b653957),LL(0xf0fbf6ca,0x5d0b99f4),LL(0x80454299,0xe59af736),
   LL(0xc2755f73,0xec4992e6),LL(0x4ca484fc,0xdfc3c447),LL(0x7217c264,0xa1f98659), LL(0x6549d8ea,0x43da1644),LL(0xee1b8b43,0x5a967704),LL(0xd404bff3,0x2a8a337d),
   LL(0xbe4b6d8b,0x94c51eb5),LL(0xc07e4713,0x8fbc9eb8),LL(0xd390eddc,0xa19d59bb), LL(0xe0be24bf,0x0ccb795a),LL(0x340f70c0,0x84f1de5a),LL(0x582c2b9b,0xa699c480),
   LL(0x7414a8d3,0x5eb36220),LL(0xe5364537,0xf828a75c),LL(0xc830d477,0x1082ff06), LL(0x037d83d0,0xd1528c4d),LL(0x45baf298,0x2b40c250),LL(0x35ee8eae,0xe2e4f82d),
   LL(0x12e4391e,0x6d36ed91),LL(0x4f322b3c,0x1ea89876),LL(0xfbc19d19,0x60200183), LL(0xda185be5,0xea5a5069),LL(0x248f835b,0x0e2393fc),LL(0x15ee6f20,0x61d051fa),
   LL(0xe1b5b6fb,0xf886470a),LL(0x60b4fdb1,0xda09259d),LL(0xf88455b3,0x088e9c86), LL(0xbc2676c1,0x52c2860a),LL(0xd1a102f0,0x6d6fa953),LL(0x7cc062e0,0x9299dce6),
   LL(0xf1a2b5ca,0x812c18ab),LL(0x21bc03a1,0x9b74e2b7),LL(0xc81ccc62,0xbca14295), LL(0x05d9bf48,0xfb6c986d),LL(0xb169ef26,0xc284d240),LL(0x901172ad,0x6292fc73),
   LL(0x5ef17d8f,0xeddaa5d3),LL(0x7c406cd1,0x15267457),LL(0x9531e2d8,0x2e303a5e), LL(0x75ca396a,0xf7bf1c5f),LL(0xc12acc0d,0xb10e432e),LL(0xeac37f33,0x2ffed1db),
   LL(0x8e561f72,0x1fef1ecd),LL(0xdd1089ac,0x38fe7b1b),LL(0x3bc9bed2,0x2f24cac2), LL(0x8beb4a49,0x0d99d3cd),LL(0x9f925b5f,0x7dd5cfb6),LL(0xfd9b7744,0xa23894f0),
   LL(0x14747d9d,0x1f353554),LL(0x75836ce2,0x31bcdf39),LL(0xf2ace220,0xea43017e), LL(0x0246a810,0x55ea2317),LL(0x4ad388ad,0x14671291),LL(0x6a13372a,0x93633c21),
   LL(0x3a2d7c85,0xcb969eff),LL(0x73683dda,0xe59df301),LL(0xd6450189,0x07e6d627), LL(0x5bb295f6,0xdddae32b),LL(0xe18b49b4,0x6746310c),LL(0x9b220769,0x828865f2),
},
/* digit=9 base_pwr=2^63 */
{
   LL(0x06290b5b,0x76334d9f),LL(0x1c02aaf4,0x096fa00d),LL(0xb1c345b9,0xb29c7abc), LL(0xdb7dfee2,0x91b1e9d2),LL(0x28848019,0x0c63c40d),LL(0x3e9d93a9,0xa131323f),
   LL(0xf13eb9c2,0x1935ea95),LL(0x2c20a03b,0x37b07e4b),LL(0x2897bd62,0x7ccebf49), LL(0xae088f35,0x671df286),LL(0x1a339948,0x9b826c19),LL(0x5b176e1c,0x2e503224),
   LL(0x6ba550e2,0x674ed6b8),LL(0x1cda326c,0xccbad8d2),LL(0x99aa3448,0xd1542f83), LL(0xbf457736,0xcf8eee5d),LL(0xcebef35d,0x40007023),LL(0x3c3cacd6,0x4f6c8f17),
   LL(0x4971b9d2,0x5794be45),LL(0x22f3503e,0x64a2a44e),LL(0x337993cc,0x62069df9), LL(0x8cf5aa65,0xb5a8be91),LL(0x4266a4f2,0xdcf77c1c),LL(0x479a02fc,0xbbe137f3),
   LL(0xf8b77a15,0xbd930e7c),LL(0x565819b0,0x1f604624),LL(0xc4a43c29,0x9a01433c), LL(0x8a108d0f,0x8cb9e675),LL(0x86e761d4,0xa4cf1e24),LL(0x30951662,0xf54b8c4e),
   LL(0xa45636a1,0x5b66a405),LL(0xaaad9d66,0xaaddb23d),LL(0xb2f86dd3,0x896cbfcb), LL(0xdb9ca210,0x01d8e6d4),LL(0x6cca4804,0x6623a1bd),LL(0x51bacaa2,0xf657c1fc),
   LL(0x3447e4e1,0x94172f67),LL(0x9c87c415,0x54f322cc),LL(0x9e98c65c,0xdc8ef479), LL(0x45d91721,0xd3538e9b),LL(0x1506e6bd,0x6291aa36),LL(0x13f2dbec,0xf61e47e5),
   LL(0x1a8522ff,0x95dc6196),LL(0xb2a0c2fd,0x7800e4de),LL(0x493d2023,0xb2d8b5c9), LL(0xb3d564a2,0x2c595ab6),LL(0x4e856f46,0xee5106ea),LL(0x0d824ff7,0xdf6ac56c),
   LL(0x8d5303f8,0x0cc840ed),LL(0x019b0602,0xb756ad36),LL(0x723c0494,0xe56dfdab), LL(0xd268efe1,0x8ba759dd),LL(0xc3466388,0x6c72c1b7),LL(0x85cf6c47,0x7a5f5d09),
   LL(0x4fe757ec,0xdbab4d4a),LL(0xdb01add0,0xfb7fb787),LL(0x6922f7b1,0x521bd604), LL(0x6b994d8e,0x4c317222),LL(0x9e46ab73,0xa7a17e9e),LL(0x20326632,0xa39d84d6),
   LL(0xa2841ffc,0xa302b6c0),LL(0x4e85da4c,0x4809d2c3),LL(0x16eeee34,0xf8bea692), LL(0xfda40073,0x049d0306),LL(0xbffc68be,0xe8956671),LL(0x21e27827,0xf5ae3b98),
   LL(0xe56f2481,0xcf5a16d7),LL(0xcb81a7ba,0x3def2594),LL(0x032f3df9,0xe07beed8), LL(0xc369a737,0xab17091a),LL(0x438b3620,0x0956e828),LL(0x10f465fc,0xab6ecf14),
   LL(0x6b2649d8,0x6e182bdb),LL(0xb09ecfad,0xc7601742),LL(0xeb6286a3,0xb43cc8c7), LL(0xb948e18a,0x89b48a00),LL(0x42ed648d,0xe337b779),LL(0xe4fc127d,0x1da0987e),
   LL(0x88ae8b76,0x45aa621b),LL(0x3162f9f7,0xb127ad6f),LL(0xc8f2cb42,0xeb56c7f7), LL(0xcddb2b13,0x6b7993cd),LL(0xfba0d637,0x6f2f2d51),LL(0xb16542d8,0xcabe79c9),
   LL(0xbde530d9,0x6753d960),LL(0x9b8986a8,0x77b91029),LL(0x9cefc1c4,0x5af09d8a), LL(0xa78c0b08,0xb396dcb1),LL(0xb5928a9c,0x44dc48ef),LL(0x2f48a964,0x9fafb33c),
   LL(0x408cae9b,0xa2a98dd1),LL(0x61c6bec6,0xb3ff192c),LL(0x3ead88da,0x607b0c64), LL(0x63184290,0x0039f8e6),LL(0x6dd9836f,0x0b9f09db),LL(0xc51a47fa,0x5d9d8f22),
   LL(0x166920d2,0xe9cf521f),LL(0xbec39b63,0xaf5b1d19),LL(0xee934b22,0x3099fced), LL(0x959116f1,0xef6f72af),LL(0xacea8ba9,0xf710ef28),LL(0x13990b15,0x588d2bbc),
   LL(0x86049369,0xcbe2755c),LL(0x46e988e6,0x738787b9),LL(0x4a2e3319,0x520ae3e4), LL(0x932e0896,0xf7e43ece),LL(0x2d3989f8,0x906f0e67),LL(0x9e24fec1,0xc4ce28e2),
   LL(0x9519aaac,0xd4391f09),LL(0x1c2d3eb4,0x923a951d),LL(0xeaa8d831,0xbba9e206), LL(0x3b0369ed,0x86a8f819),LL(0x84f45ee3,0xf08f5db8),LL(0xc7fd4306,0xde9d97b7),
   LL(0x82162e5e,0x4abae764),LL(0x820c91df,0xda046c14),LL(0xcac95307,0x9be5686d), LL(0x1926ce21,0x8d90109e),LL(0xa11ddecb,0x346eafc4),LL(0xd72ecc75,0x3ce252dd),
   LL(0x3d30e9b1,0xb559793c),LL(0x21faa664,0xcf651f10),LL(0x0af936f8,0x58827716), LL(0x3390af36,0x2e460004),LL(0x2e93e7e8,0x355c08cc),LL(0xda2c6727,0x84eb137e),
   LL(0x2625e961,0x9e901d5b),LL(0x499b3624,0x53244f50),LL(0x3c07fdd0,0xf7b5e924), LL(0xc8b4fccb,0x5e1b6db8),LL(0xd9a4ede5,0x4d6122ee),LL(0xf719582d,0x96df7864),
   LL(0xc9be27d1,0xeafcb131),LL(0x985a8e07,0x86458dd3),LL(0x63ff40d4,0x3e039563), LL(0xd1407667,0x2541540f),LL(0xd9795e3f,0x8b27f2d9),LL(0xd60d0743,0x5dbbd4b3),
   LL(0x605be24c,0x2b496309),LL(0x496bfcf4,0x3afbea78),LL(0x46ed79bc,0xfe6069c6), LL(0xad7bb67a,0x3cbeb331),LL(0x37463aa4,0xeedcaf16),LL(0x7c12f87f,0xc41a07ca),
   LL(0x2fccd260,0xbd41b63e),LL(0xc658b3a5,0x3bb5676c),LL(0x15c9ccb5,0x84c591dd), LL(0xa5fcc6ca,0xc5573715),LL(0x9fcba543,0xf1800487),LL(0x63da6fef,0xfb887cbf),
   LL(0x9f37dc0e,0xfd99db21),LL(0x5783e53d,0x85bbbefe),LL(0x09d5d09d,0xa10360ba), LL(0x3cc09837,0x12e9679e),LL(0x793efd44,0x81a4b2dd),LL(0x8153dba4,0x0c93ed4d),
   LL(0x2883764f,0xb7b5a250),LL(0x75b781bc,0xd3ca90ea),LL(0xc5198673,0x8da40a3f), LL(0x237103fa,0x434744ad),LL(0xcbd65a76,0x16d7466c),LL(0x9d0ce6ec,0x97f5ff82),
   LL(0xc3b3f666,0x0134d746),LL(0x7c3910eb,0x62a3254f),LL(0x6906613f,0x989510d0), LL(0xe75bfea7,0xa63aead6),LL(0xf6f48b5d,0x2c7bb201),LL(0x0cf11cdd,0x17c485c4),
   LL(0x01346a36,0x44072775),LL(0xeccf5986,0xd81f9e8b),LL(0xffca3195,0xcedbaa7d), LL(0x5744e89d,0xeea2a618),LL(0xad358476,0x65f3b598),LL(0x49cae477,0x31ef0e2a),
   LL(0x82f6a471,0x0f618b7c),LL(0x437cc826,0xac49eab7),LL(0x7209d33e,0x6ac9e2b8), LL(0xa161856e,0x14857b21),LL(0x638a1099,0x2c515ae8),LL(0xe8f5e86f,0xd34ba334),
   LL(0xdf15afb4,0x96dc51a8),LL(0x5b39ee73,0x0db5cc4d),LL(0x59face85,0xb166ae3d), LL(0xb3c9bf88,0x94ee7aca),LL(0x34ba27bc,0x0724cb81),LL(0x8aa869b4,0xac1a10a4),
   LL(0x621e0951,0xe85de117),LL(0xf75855ac,0x7b464098),LL(0x53d1abe6,0x06b33c28), LL(0xb538c2cf,0x095bdc74),LL(0xd859366c,0xa64b61e0),LL(0xe9922d24,0xa5c92136),
   LL(0xc221fe54,0xa1285a95),LL(0x7964a5fc,0x50c3e10e),LL(0x497b1e90,0xdbaa4092), LL(0xa9408c0f,0x3f4e7d97),LL(0x92b70f2b,0x1e83c2f5),LL(0x6911dac9,0x2ed8ab9c),
   LL(0x73b888d5,0x0bc16a46),LL(0x8a011451,0x4305504a),LL(0x5de19cb5,0xea5d6a02), LL(0xa7166a08,0x8571e2f0),LL(0x6ea97470,0x9edcb0ba),LL(0xd6fc7b1d,0x1ed74e9b),
   LL(0x48e4b5fe,0x960652fe),LL(0xd0e6f19c,0x391d4162),LL(0xb59dd2ef,0x8c6e00ef), LL(0x95fd71ce,0x0cc08f13),LL(0x8fedafc2,0x6d81789b),LL(0xe498c253,0x3ec56b1a),
   LL(0xb7edb5da,0xb9bf335a),LL(0x693e2fe3,0x1a3b9636),LL(0x94c320a4,0xb12ae440), LL(0x35ef6698,0xf7d1986e),LL(0x273f0a93,0xc1db1a91),LL(0xadbd81a4,0x72fd29a3),
   LL(0x1dcb4e55,0x57a9af55),LL(0xa46743c4,0xa430b677),LL(0x4459d987,0x2e2d09e0), LL(0xc1831698,0x58bc26b8),LL(0x8e8a86f9,0x02cdfd4d),LL(0x86532393,0x81824a71),
   LL(0xfdf7070a,0x4b7d4311),LL(0x46b2d2cd,0x790ff04e),LL(0xa669dca9,0x5f194b6e), LL(0x6ee0a73c,0x8d27f077),LL(0x3f03f66f,0x7db8300e),LL(0x52542218,0x894792c2),
   LL(0x320a391f,0x709e970a),LL(0xd301afda,0xffc55682),LL(0x187a421b,0x8c7c2ac7), LL(0x05c2a76b,0xb9413568),LL(0x50d693ea,0x4ce64d45),LL(0xf8118a62,0x133a1d85),
   LL(0x1634f569,0xb78ee791),LL(0x0985837d,0xc8fdd735),LL(0x42337dc2,0xb40ab147), LL(0x2ece8062,0x58ee37a9),LL(0xb4fb1ae1,0x1b80e3de),LL(0xddf42642,0x55797958),
   LL(0x3a4a5d2b,0x4ced5be1),LL(0xf46469b5,0x537cc02f),LL(0x9140aadc,0x369faf66), LL(0xbba54c56,0x91646050),LL(0xf1627bac,0xb6ae4660),LL(0x03bf4ace,0xd39a1183),
   LL(0x5ad40b74,0x9a746086),LL(0x09825b65,0x8d116e80),LL(0x4c9e508b,0x66642780), LL(0xca3af3bd,0x10c1db38),LL(0x67df4e51,0x91ca8866),LL(0xadf6517e,0x3b4ebf22),
   LL(0x69754168,0x906ece9b),LL(0x3842db37,0x78a4999a),LL(0x45118f97,0xa54934b5), LL(0xd6600715,0x17c76b7f),LL(0xfef1a9a9,0xec8d0be6),LL(0xdaf01223,0x80902814),
   LL(0xe7382aa9,0x8ffad587),LL(0x64489fbc,0xff48de9c),LL(0xc8d3ef62,0xe539dabd), LL(0x6e44245a,0x037fa21e),LL(0x6ffc89e7,0x44aeb6c6),LL(0xc24d18dd,0x903826c9),
   LL(0xfa798920,0x7e6202ad),LL(0x97dc220c,0xa69a704f),LL(0x27b68255,0xd7e7cafe), LL(0x52b2a274,0xa6aa0342),LL(0xe49cef76,0x72ecec87),LL(0x82afed60,0xee4bbb4a),
   LL(0x7c8213f0,0xb7bfa6d1),LL(0xe2e5eb2d,0x55e77d8a),LL(0xecb57d00,0xde5c925a), LL(0x91bfc135,0xc67d7eea),LL(0x14df6915,0x6915f948),LL(0x8e5e081a,0xbc1fffee),
   LL(0x4f0a6b86,0x1ca04a6c),LL(0x07a69d26,0x3f736642),LL(0x8eb91e1c,0xbcf425f5), LL(0xf7a230eb,0x0ee5e1c6),LL(0x11f1574d,0xafdd21cd),LL(0x31d6dc6d,0xb5ef9bf3),
   LL(0x7cc6bcf3,0x6774ed5f),LL(0x0a75d23e,0x6feadf8b),LL(0x81ec8c8a,0x5898828e), LL(0x414cd64b,0x3e9103bb),LL(0x1542efe4,0x703f0fb2),LL(0x7388d6c8,0x3df2c7c6),
   LL(0x60a7e99b,0x02e73081),LL(0x7986406b,0x1b3db98b),LL(0x3fff6009,0x9959bac0), LL(0x75e1d2fa,0x80625a36),LL(0xa198da2b,0x95492b2e),LL(0x59742464,0x81018f71),
   LL(0x3cdf8db3,0xc1fc725f),LL(0x59326d48,0xd58228f1),LL(0x1d74eaaa,0x4c0a442e), LL(0x1b886d60,0xa814c005),LL(0xfaa58d26,0x9756df79),LL(0x176bed0c,0xc17f3346),
   LL(0x7ec82b2a,0x874840bd),LL(0x1e1e2156,0x8b253112),LL(0xd1a58bca,0x8264f8ad), LL(0x23a4105c,0xc2a71227),LL(0xf0882599,0x8add8a3d),LL(0xd491e94b,0xfadbbc8e),
   LL(0x0f86b5cf,0x8d48becb),LL(0xdd83a12b,0x700c6428),LL(0x87164a6a,0x9175e238), LL(0x300636b7,0xdd10d85e),LL(0x080c01ea,0xec0b0e20),LL(0x18c937fc,0x5b27d26f),
   LL(0xaba54a99,0xe45deb1a),LL(0x88f0f0e9,0xea11e26a),LL(0x8d1c5c7c,0x3f29d26a), LL(0x261af059,0xaad1697c),LL(0x8bc6950e,0x65b29644),LL(0x3940ae67,0x33e4b7c4),
   LL(0x18f510ad,0xe7e52f31),LL(0x4aba84d1,0xecf901f6),LL(0x1d0d87ae,0xe6d9492c), LL(0xf4371e3c,0x21861ebd),LL(0xac94b579,0xd7f34636),LL(0xe45e7910,0x72e22e1d),
   LL(0x0d44e5c0,0xe0a30cd5),LL(0x672259aa,0x2d74d417),LL(0x48769d5d,0x208c07a6), LL(0x3ab04b13,0x0182f535),LL(0xd1493acf,0xa88ddf33),LL(0xe65f0e82,0xd692f755),
   LL(0x200e00e2,0x57480257),LL(0x80f4a13a,0xd24d38be),LL(0x8b6535bd,0xe2d4702a), LL(0x6d6d0cd9,0x597abaed),LL(0x1a32e8b7,0x2eaf61f2),LL(0x06a3acc5,0x48a99a4d),
   LL(0x39489527,0x25304144),LL(0x832761c7,0xb36d9b74),LL(0xaa187256,0xe33bc0a7), LL(0xa72fb33d,0x4d4e2c96),LL(0x7ed273fe,0x1dba3ef7),LL(0x577de798,0x2b9978d8),
   LL(0x54644ced,0x467e81b5),LL(0xc2a02440,0xd4ef3e1b),LL(0x15b3a380,0x0df99eec), LL(0x250b43d1,0x549afec6),LL(0x1598f520,0x0c4fc7e2),LL(0x67422c36,0x9f8e42d7),
   LL(0x6c0213e3,0x3a3886aa),LL(0x8b5b752a,0xc12533ce),LL(0x814f4729,0xbe235906), LL(0xbffc94e3,0xdf45bc94),LL(0xf2a001db,0xccdad604),LL(0xb8cdcec3,0x11ddcaa6),
   LL(0x25ddf146,0x7eee967e),LL(0x0bf6e4d3,0x9a2d1d73),LL(0xabd48b07,0x57a43f05), LL(0x32b3a8a7,0x77d9ad80),LL(0x55872fbd,0xfbab22cf),LL(0xd02e7501,0x2100eede),
   LL(0x25faa7bf,0xf4f4c899),LL(0xaa9fc84a,0x235a5736),LL(0xfae4fdb9,0x37647c20), LL(0xa26ae472,0xfc3f8ca6),LL(0xa2f519e3,0x01c6ea11),LL(0xc2fc610f,0x1ce14f58),
   LL(0xb767a1fd,0xb6f923fc),LL(0x07e07c0c,0x31a760f3),LL(0x6a7085a9,0xe913fcfb), LL(0x40442930,0x86adb2bb),LL(0xe41a1b14,0xc8c20759),LL(0xd8d1924a,0x399d5ad6),
   LL(0x76ac6874,0x6f0508e8),LL(0xafba3d82,0xd96bd7cd),LL(0x56cc2c07,0x336b813e), LL(0xb0c7906c,0x0fca9b73),LL(0x232e056d,0xb85518fe),LL(0xd1c21ece,0xa97bdc68),
   LL(0xfbc66efc,0xf3327dfd),LL(0x34a1c591,0xa401731e),LL(0x3371b4b7,0x8c1ca43a), LL(0x526198b2,0x835393dc),LL(0x686e9c85,0xa8d2d86f),LL(0x7a0c3d07,0xa4491de7),
},
/* digit=10 base_pwr=2^70 */
{
   LL(0x309b2b98,0x790e100d),LL(0x50a4f09b,0xaf54cb0b),LL(0x956ec705,0xbfd5af9f), LL(0x3bab82f5,0x4692a125),LL(0xec637260,0xac0be759),LL(0x580441bf,0xaf835bc0),
   LL(0x513c3f3c,0x80ab94a2),LL(0x77e64a9c,0x39c245d0),LL(0xa7543bd5,0xc3ba4002), LL(0x7eacb224,0x6658e2bd),LL(0x1f419c54,0x7e005bba),LL(0x58b62644,0xd54bc0f7),
   LL(0x9194cd13,0x5c1c461d),LL(0x95f13f43,0x843427bf),LL(0xba86794a,0x6844d36f), LL(0x393e443b,0x90e5c334),LL(0x9f72ba01,0x8cec77f9),LL(0x145d81fc,0x28ef5fba),
   LL(0xdd9dc934,0xa6b331c2),LL(0xde2f138b,0x32a15ddf),LL(0x2b7697a8,0x490117ee), LL(0xb3b1e935,0x5eaf7f20),LL(0x9b807876,0x4b750ed0),LL(0x864be6f5,0x98f1673d),
   LL(0x9a5d2cee,0x4a36329f),LL(0x373aae9d,0xd6288e69),LL(0xd918ee97,0xbf2fd4fb), LL(0x870b1cdd,0x020ef080),LL(0x22a9b3e7,0x5578e9ae),LL(0xeb392f0b,0x8887e7e2),
   LL(0xf5b8a656,0x533c438d),LL(0xbc6d32ae,0x98531dda),LL(0x1af7184c,0xc8e1927f), LL(0x393cf3c4,0xb9f8c2c8),LL(0x73ac1607,0x37cffce8),LL(0x3312e5ff,0x76d8c894),
   LL(0x2b57f886,0x9a855c17),LL(0x3f649b7d,0xa334a009),LL(0x6e284b9c,0x90468704), LL(0x008bb313,0x633d15c5),LL(0x0ec3ca93,0x384f430f),LL(0x83b412c8,0x109e0fec),
   LL(0x1a53d7a9,0x53582049),LL(0xf0099876,0xa35696c9),LL(0xfab32bde,0x34991882), LL(0x9092df43,0xa6e48215),LL(0x0f2f3443,0xf674d02f),LL(0x4d862a69,0xeeb91808),
   LL(0x8ece35fc,0xdc9c1776),LL(0x9ab87051,0x7e6e4720),LL(0x7fd83120,0xf6d82143), LL(0xa0bcd27e,0x7ee28542),LL(0xd6a86383,0x5e5835d3),LL(0x3ddae47d,0x1b6caf5f),
   LL(0x48ce8d7a,0x66cc8b26),LL(0x5f4e54c1,0xb8f25b70),LL(0x8abade9b,0x9c222105), LL(0xe91600e2,0x6a5880be),LL(0x55c01418,0xd8214454),LL(0x3e0670f2,0x1510a747),
   LL(0xefd35bbf,0xe70de601),LL(0x01575c49,0x23fea84f),LL(0x04ef42e0,0xc9dc4bb5), LL(0xb139d368,0x01d78685),LL(0x4c1d31f0,0x6acc3ed7),LL(0xbeac2fcd,0x7ebf6809),
   LL(0x8b17cc56,0xb221b461),LL(0x6c918f42,0x3a388a6c),LL(0x61ec2c4a,0x43b22feb), LL(0xdd8892c4,0x1f31f72b),LL(0x7aae5ed6,0xff9e63db),LL(0xc5aa19b7,0xcf33f8e0),
   LL(0xb6d01e70,0xb2b290ae),LL(0x70c988ca,0x32377051),LL(0x5c47c52d,0xb49ee722), LL(0x73ed265f,0xa41919e1),LL(0xf4708b44,0x41428dd0),LL(0xda4fedfa,0x15d760a1),
   LL(0xe8537c4b,0xfed502b6),LL(0xfc7dde03,0x8bab770b),LL(0x7d1f76d0,0x05c46da4), LL(0xc0db9267,0xc8dcbdf0),LL(0xcf6393d4,0x01cc7b6a),LL(0xeaf6dc69,0xab93ff59),
   LL(0x9da53ff6,0xc026d3db),LL(0xd6c5a028,0x6b07911d),LL(0x9f01eec9,0xf2e88f7a), LL(0x35fb3be7,0x30ac879a),LL(0x9c8e073c,0x92871e9a),LL(0x44eb456d,0xc18518f1),
   LL(0xe32b5a2f,0x2ca049f6),LL(0x3c2dc1f8,0x164d195b),LL(0x0eafa9d1,0xd1364e77), LL(0x59bac901,0x87a85a1f),LL(0x31b28912,0x040599eb),LL(0x73989f3d,0xe5138d57),
   LL(0x59e18159,0xda271207),LL(0x87bdce2f,0x62d2e07e),LL(0x9c9da191,0x00bc160e), LL(0x9f9f3edd,0xc2aa6fab),LL(0x3e0cc568,0xd0bd4459),LL(0x6d2f3dbc,0x80621f6a),
   LL(0x2608caf0,0xc02d8317),LL(0x3b8d5f88,0xb2b7cf8e),LL(0x97aa5235,0xe268620c), LL(0x53373262,0xc5d88092),LL(0xcbdc45c7,0xb80c631a),LL(0x3758fbc8,0xa82fa7f0),
   LL(0x61946f84,0x71ca7112),LL(0x26f634dc,0xdbb91c74),LL(0x48b33bf4,0xc61feaa2), LL(0x0f0b622c,0x8c1f4e50),LL(0x4a976276,0x54a1b037),LL(0x9ac35cd3,0x46fdaca0),
   LL(0x62a4a464,0xa78d6498),LL(0xbe50e07c,0x8185cc99),LL(0x046a15b3,0xe0df6a84), LL(0xb1747017,0x3fbd2fe4),LL(0xd8a28d37,0x40bd174e),LL(0x3ba62fe3,0xeb67a57c),
   LL(0x9cd4dbf3,0xc674a040),LL(0x8ef4b42f,0x342823c9),LL(0x82a815c3,0x51bb59b7), LL(0x56fc0c39,0x4b239627),LL(0xc5f5d825,0x28ec23b4),LL(0x6c8e1e27,0xf7c1b3d8),
   LL(0x27ec4b30,0x9635ffa2),LL(0x100ea34e,0xc42039eb),LL(0x90076fbc,0xd912642e), LL(0x1f77514a,0xaafcaa31),LL(0x0fb82d1f,0x79d17151),LL(0xb2f1317a,0x7c0989e3),
   LL(0x89f462e7,0xb4998e80),LL(0xe6fbd086,0xb36cee42),LL(0x2313e077,0xc8121a21), LL(0x2726f97c,0x5989f410),LL(0xcd4d494f,0x0186977b),LL(0x5a5b30ab,0xc3fb6b78),
   LL(0x424d3124,0xf9bb1829),LL(0x22c5cfed,0x4def4592),LL(0x116b8380,0x635090f0), LL(0xedc582ce,0xdc36aaf6),LL(0xcb88543f,0x8c97c600),LL(0x79d7d839,0xea4cedf6),
   LL(0x3a7fe603,0x8e22dd53),LL(0xb6e13252,0x4f6646e7),LL(0xcc10cc0b,0xecb71522), LL(0x8fe0bd6f,0xb14b31d5),LL(0x7a6c5efb,0x2fe52f94),LL(0x1d6b6945,0x46144410),
   LL(0x41fe4d44,0xd9cc6cd5),LL(0x3e28dc8d,0x358e33f3),LL(0x521ff50b,0x1a26ddf8), LL(0xa3eab01f,0x3ee649e8),LL(0x03050e22,0x98968099),LL(0x12ed1dd8,0x49721509),
   LL(0x1e67197b,0x384a1493),LL(0x56c98a4b,0x0d7943cf),LL(0xaf2f7e74,0x23d774ac), LL(0x79355a55,0x2c21ca0b),LL(0x1fad2d85,0xf5e43dea),LL(0xa8d0b6bd,0x57ab8f61),
   LL(0xd56cc2a3,0x0e518afc),LL(0xe2dc0d52,0xc6a83b40),LL(0x809ea39f,0x6f5059aa), LL(0x2721bc4d,0x94390dc2),LL(0xfb429338,0x879b35f1),LL(0x14546ddd,0x3891b090),
   LL(0xc56f4571,0x7b43040e),LL(0x5fa40c73,0x0b5a0144),LL(0x274f5c74,0xbe5cf035), LL(0x9af86d27,0x9e879540),LL(0xd3c0d8b2,0x77e49fb8),LL(0xc770f9db,0x61fb43b6),
   LL(0x92022253,0x4ad13b39),LL(0x71a9d129,0x8396b4e5),LL(0xdce6f688,0x8182898f), LL(0x9bab196f,0xfc0794cd),LL(0x7d4e5580,0x4100be42),LL(0x73474464,0x1417be29),
   LL(0x0b909777,0x50d8f53e),LL(0xb81656d9,0x9e5bcdce),LL(0x68e77a3f,0xd3e75c90), LL(0x22b5d762,0x3e315195),LL(0x1511374c,0x1fe3bea5),LL(0x39e43804,0x7dcf6bb6),
   LL(0x98b4883d,0xaa84b4ab),LL(0x5003afb8,0xfb1203a4),LL(0xf113ef60,0x49b549bd), LL(0x421a26db,0xb8a534eb),LL(0x4176bcfa,0xae3abef5),LL(0xc9fd2e50,0x65ad7ddc),
   LL(0xc9927e31,0x32999234),LL(0xc5f17855,0xcbc9be39),LL(0x91ce524e,0x4189b3bb), LL(0xedfa4e44,0xc72add92),LL(0xfa0ae77e,0x59dfa5aa),LL(0xd3b54994,0x1bc2e1a2),
   LL(0x743258e5,0xc7638300),LL(0x4e7252c9,0x508d7943),LL(0x39297969,0x882f689a), LL(0x5a87da5a,0x87d951d3),LL(0x24116564,0x21c31a28),LL(0xbf2be0ff,0x76451670),
   LL(0x500e0aee,0xeb561121),LL(0x89b20709,0x1d9b8fb4),LL(0xc9f555ae,0x7ba3f437), LL(0x276c8c91,0xc08e6586),LL(0x6d0f76b9,0x7daa15eb),LL(0x3c2bff70,0x1769cb69),
   LL(0x242a43b4,0xbd62fc3b),LL(0xe4ceb019,0x853d3ce1),LL(0x267cdfd1,0x81780739), LL(0x9cd6475c,0x60ecb687),LL(0x48a209f2,0xf3e199ed),LL(0x02dae3a1,0x3cf6b9f3),
   LL(0xe7cec18a,0xb0e37780),LL(0x98f8a592,0xaa6eb4e7),LL(0x2d0d06e7,0x0427150e), LL(0x911cd620,0xef4a40dc),LL(0x6e271737,0x71162ad1),LL(0x700d088c,0x82437eca),
   LL(0xf1cc972e,0xee1967ba),LL(0xd4eddaa0,0x920e2ab5),LL(0x985d7040,0x5c5ee245), LL(0x3950b6ed,0xb0b435d4),LL(0x364257b2,0x3d814c4e),LL(0xec06a200,0x5baa5fcd),
   LL(0x79c5f683,0xea62de1d),LL(0xf682ff65,0x02d1dab0),LL(0xad34060a,0x001e1565), LL(0xbc38c15a,0x09c411dd),LL(0xda6f1853,0x5f688345),LL(0xdd138b16,0x3a60f9df),
   LL(0x0bcf4074,0xb21d2687),LL(0x66ea60fa,0xcb9d6c7f),LL(0x86febc79,0x33b4a213), LL(0x9bb71bf2,0x6c5e209c),LL(0x63580404,0x53f871bf),LL(0x0c55bce3,0x4c4a3634),
   LL(0x50a5c1d5,0xd05e1b75),LL(0xeda43a63,0xac2eb90d),LL(0xff9c26a8,0x66eba3d3), LL(0x3c837bb1,0xe4f6c13e),LL(0x6e9b77ce,0xee6c1ed4),LL(0xf727fe74,0x7107e1fe),
   LL(0x88d7d553,0xbd0548b8),LL(0x4afe5186,0x78876c17),LL(0xa7285f30,0x621d5e10), LL(0x5a304147,0x051e0777),LL(0x334a494d,0x5390f3bd),LL(0x4579e150,0xbebd4f2a),
   LL(0xc9672b55,0x0aa436fe),LL(0xaf1550c6,0x52a75c26),LL(0x3858b864,0xd0cbe739), LL(0x0d008d67,0x8f10eef9),LL(0x72b9fc05,0x8f75f104),LL(0x364e6b20,0x143ef589),
   LL(0x533b4c98,0x2be2a8f3),LL(0xc7ae49a0,0x982b2540),LL(0xefebfa00,0x546e27b9), LL(0x900dd378,0xb5d294cc),LL(0x3658dbf4,0x3be2150d),LL(0xb34a20e7,0x02874ed2),
   LL(0x40cc25ec,0x478a065c),LL(0xada695ef,0xbeaf90cb),LL(0x862215ea,0x4aae42ff), LL(0x165174a2,0x381fc293),LL(0x3d459c99,0xe86a1fe7),LL(0x401dd67b,0xe890d568),
   LL(0x82e7aada,0xb46876e1),LL(0xd804209c,0xecca3439),LL(0x1d006b77,0x2c44843b), LL(0x8dd030ac,0x61576007),LL(0xfcb53cac,0x43b0a3df),LL(0xdebeb1f9,0xc5ca915d),
   LL(0x0ca9fabd,0x261ec1c6),LL(0xcd7cb986,0xef1e6df1),LL(0x2bbbbc23,0xa827cc65), LL(0xcbd255f5,0xbe5fb899),LL(0x512312f5,0x909568d8),LL(0xa6726a82,0x681dd29a),
   LL(0x488ad19a,0x0de34649),LL(0xa836d120,0xaedbbf66),LL(0xc3d2bb79,0x3db458a5), LL(0xf1b23abb,0xa516213a),LL(0x1fb13cf4,0x57cb0ebe),LL(0x2a71fb5b,0xb43b7de1),
   LL(0x5a77c99c,0x4136b8b1),LL(0x8ea5e01e,0x7e568b1a),LL(0x63a9ae35,0x218b1234), LL(0xe124bab5,0x31875171),LL(0xccc750f2,0x8b5a214c),LL(0x085b0bf6,0xcda64152),
   LL(0x65aa7c4a,0x823a1144),LL(0xd237a179,0x887834cc),LL(0x79b0e325,0xa2884253), LL(0x71245d03,0x05200952),LL(0xefa5fe27,0x6b7e9f1d),LL(0xd8acec2b,0xda6faffe),
   LL(0xf6bd56f2,0x8136bac4),LL(0x996005c4,0x13e6d8cc),LL(0xa4d43df3,0xd7ea43e1), LL(0x2f377e2d,0xfc227ad7),LL(0xa8878d26,0x90549e9b),LL(0x4aa1f76a,0xb578b012),
   LL(0x4a188fe4,0x42ca87e7),LL(0x5d8e390f,0x4ba3638d),LL(0x9476562a,0xb3568c45), LL(0x0c0a6cc4,0x0c426680),LL(0x3e63129f,0x64115513),LL(0xcec9a6eb,0x711c89d3),
   LL(0xd14dddc5,0x923db3a9),LL(0x7dac544b,0xd001ba98),LL(0x692ca6d5,0xc5b16136), LL(0xcdf3aa4c,0x105a9b51),LL(0x52023455,0x909991a3),LL(0xe71eda94,0x8699a77a),
   LL(0x20abff43,0xde5a86c2),LL(0x5e0fe305,0x51a3ccb0),LL(0x9ea73b83,0xb76c1d2b), LL(0xf9a975a1,0x9ecfed6c),LL(0xfb0c036d,0x9c0332e0),LL(0x6889bbce,0x4e7d344a),
   LL(0xeff8ef6a,0x4f5b02ba),LL(0x165b7828,0xca84ea74),LL(0x6fd1b432,0xc50328c5), LL(0x4166147f,0x56d815b8),LL(0x3da09b08,0xabfb8a90),LL(0x2d02b8d2,0x6b35b6b2),
   LL(0x398c7dbd,0xbd664dac),LL(0xd96e1ba7,0xf61eb365),LL(0x65e6672a,0x4063219c), LL(0x8c7ea7dc,0x4bacb326),LL(0x40eb1485,0xcb7c53dc),LL(0xc8724e95,0x75993332),
   LL(0xb8af3e69,0x65ace7ac),LL(0x92a2146c,0xc1b257bf),LL(0xcd5a446f,0x5901f7b5), LL(0x8c135291,0xfd4954c0),LL(0x7287802d,0xf9489d56),LL(0x79ca70bb,0xe671da6c),
   LL(0x7342afb2,0xcf1f68a7),LL(0xd2662b38,0xb7c95d16),LL(0xb90fd0c0,0xfb1d7741), LL(0x836a9bb4,0xfd7f66be),LL(0x3f666fd7,0x9196236b),LL(0xf2666854,0x97d5dc11),
   LL(0x13be0faf,0x53b96191),LL(0x4aad75e2,0x8984caab),LL(0x8c638642,0x5f016b43), LL(0xfe4962b3,0xbe9ab98b),LL(0xf41d7325,0xce94bc4e),LL(0x2727019c,0xa8761364),
   LL(0xae07f4a8,0xbb80b445),LL(0x8294ccdf,0xe5881e44),LL(0xad112cda,0x3a4c8e39), LL(0x35d8c72f,0xa5340d8e),LL(0x48955a80,0x47142cce),LL(0x86d4ad66,0xb5ddcf06),
   LL(0x2ba33bbf,0xdf4d64b8),LL(0x38311d97,0xb562f723),LL(0x3ae0b587,0x9f7f5f29), LL(0xc4c06982,0x7233a026),LL(0x72ac8ffe,0xe33867b4),LL(0x508eade6,0xf85acdb1),
   LL(0x67a2a49a,0x8261c3a7),LL(0x9775813d,0xcb58ffc5),LL(0x932bb91b,0x06c28465), LL(0xb5e56211,0xd07de2e2),LL(0x8f5bfd9a,0x6e0b5a38),LL(0x6f9d74ef,0x1cccc9f7),
   LL(0xe3a75abb,0x48de8e6e),LL(0x39374610,0x0ebaaa1e),LL(0x2106e8ab,0x003aaedd), LL(0x92e2817c,0x6ae46c78),LL(0xfbd7d633,0xc52e217b),LL(0xc280871c,0x804476ea),
   LL(0x8f4ad647,0x1399bf87),LL(0x2f0e2288,0x81692c0a),LL(0x4c8fbcb7,0x4eb07838), LL(0xe3874f86,0xdb2f862c),LL(0x89f62e6a,0x9e0ee74a),LL(0xb8e8633f,0xa0fef663),
},
/* digit=11 base_pwr=2^77 */
{
   LL(0x596b6e25,0x3628c4e5),LL(0xe3c48c5e,0xc4406956),LL(0x2e8ae234,0xe7c4cb00), LL(0x2ed43c07,0xa11d2ed3),LL(0x77b2799d,0xc33b975f),LL(0xcaefccd6,0x3715cc2d),
   LL(0x29a801e4,0x98e5a9fd),LL(0xd84bf0a4,0xfa4d3abb),LL(0x2c426bed,0x8f4dc6f8), LL(0x1b83f02d,0x9decebac),LL(0x31afcc78,0x27563177),LL(0x28657c91,0x25efdee0),
   LL(0x10bed657,0x44f7577b),LL(0x05089a71,0xd2686801),LL(0x687d749e,0x44e84ecc), LL(0x2ffe47f3,0xeba9b10c),LL(0x977d2767,0xed6470da),LL(0x70a6ec4e,0x7568cede),
   LL(0x5fdc4b62,0xb3dfb691),LL(0x55109b96,0xd851f390),LL(0x24e13631,0x55c4dac4), LL(0xb02dcba6,0x12c87918),LL(0xb6a6220f,0x082a25d8),LL(0xcc90ccd7,0xf8abc990),
   LL(0xb4ca91b2,0xef95af0a),LL(0xa99a156c,0x693d8362),LL(0x4cc32c97,0x8b1f5532), LL(0xf16e87af,0xff44233f),LL(0x09622892,0x6fffb3de),LL(0x676ed99f,0x9c976523),
   LL(0x19ea3118,0x486075b2),LL(0xe8b84a78,0x288a821c),LL(0x603792cd,0x53ce6139), LL(0xcd02301a,0xcc3dfb91),LL(0xafc2d623,0xd6745a56),LL(0x2bdb691f,0x55aa143c),
   LL(0x38ddf5e1,0xb104e58a),LL(0x2b17bd48,0x504256db),LL(0x1af21432,0x9e5bdc70), LL(0x364f820d,0x478233b6),LL(0x6c7d0127,0x150c2f89),LL(0xd8d44844,0xdf664b62),
   LL(0x15f1c173,0x5dd16958),LL(0xc46a9115,0x3312d6a3),LL(0xfe869bbc,0xc9527a68), LL(0xf6cbcbb4,0x6e5effcd),LL(0x362c0478,0x35bd9c75),LL(0x55bb0376,0x02094198),
   LL(0x86c1d92c,0x20c0085c),LL(0xb4122bba,0xbd007e1e),LL(0x25befafd,0xd7c99c19), LL(0xa3ef65e4,0xf07b6cd4),LL(0xcfed174f,0x670355d0),LL(0xd9b990a8,0xff36e219),
   LL(0xdc2a6498,0xdc527d73),LL(0x6e1a780d,0x08934ec0),LL(0x70e3a7c9,0xc2577255), LL(0xc59481bd,0x3e9b3c1c),LL(0x33ca9c3c,0x148a477f),LL(0xdf421578,0x5ac5a917),
   LL(0x0d689c44,0xecd077c7),LL(0xb91f2750,0xb9946fa4),LL(0xf06d9aee,0x6a48a5de), LL(0x09b44876,0x90d6c760),LL(0x8d0df7fc,0x4768b79a),LL(0x528d023b,0x894c061b),
   LL(0xeb1f23e5,0xc155c9fc),LL(0x87fb0d4d,0x30da65cc),LL(0xfe29aaf9,0x39ddc6f1), LL(0x9b7d2157,0xe1559681),LL(0x62eec022,0xe62ee793),LL(0xa973795b,0x6841ff58),
   LL(0x8c5f6a6f,0x5041ffe5),LL(0xa1e49e9e,0xb047d430),LL(0xa1870cf1,0x55c67f67), LL(0xfdc4137f,0xa52cc630),LL(0xeeacaf6a,0x708f8900),LL(0xf29a770c,0x4489b888),
   LL(0x427f459d,0xa9893015),LL(0x3af11c58,0x29c3a358),LL(0x3159a1a0,0x9fe19047), LL(0x16e111ce,0x3590c6ca),LL(0xb144f41e,0xbb21fd6b),LL(0x48ff3048,0x668bc4a8),
   LL(0xcd902ca6,0xf80a9619),LL(0xf2ba49cb,0x75608dc6),LL(0x0d7ead99,0x1ac32280), LL(0x7b3c121e,0x0f8cc5ca),LL(0x541e43fb,0x70c044bb),LL(0x6838ded3,0xfb10ee49),
   LL(0x3fe39fff,0x75b96345),LL(0xf45a96d7,0xb4aeb917),LL(0x8932cc8f,0xcc29a1ee), LL(0x682514c2,0xd1317dff),LL(0xa9ef556d,0x5cecce58),LL(0x80d7dfc4,0x51b5fe83),
   LL(0x496e1c06,0xef91106b),LL(0xf293c76e,0x28e0bd07),LL(0x504a0bfc,0x42b5d427), LL(0x507248f9,0xe4090019),LL(0x391b523a,0xf41922a2),LL(0xa9b23a0a,0xf8bf6bbc),
   LL(0x03c42d14,0x4538e84f),LL(0x125c5ecc,0xc3304e31),LL(0x0528582e,0xdb87c51f), LL(0x63617411,0xd5eb0dd8),LL(0xbab74fe4,0xf4502fb9),LL(0xbda4957c,0xce6a62a7),
   LL(0xcc524a7e,0xa34e113a),LL(0x0e326637,0x256f2404),LL(0xe9251c73,0xeb005bad), LL(0xd5be4e06,0x3ac0298f),LL(0x97f2b09f,0xb76eb4f5),LL(0x71ef690a,0x93965d52),
   LL(0xc620384d,0x1050467a),LL(0x466e292b,0x8b677cf7),LL(0xe11cabe6,0xf2ac4850), LL(0xb63594f1,0xcec62350),LL(0xe4d91805,0x9afedaa5),LL(0x27dff452,0xd7455a8e),
   LL(0x1f700bc7,0x19af1010),LL(0xc8359c70,0x5cd5ad5d),LL(0x70f71fce,0xc8be23e9), LL(0x467d70f8,0x80764bb7),LL(0xf6206a40,0x62069652),LL(0xdd25262a,0x17473085),
   LL(0x1ba99063,0x32fe5625),LL(0xdb47d5bd,0xac325742),LL(0x30faac68,0x1621ec3c), LL(0x1c66729b,0x1956c250),LL(0x073aef1f,0xb76bcafe),LL(0x5159e850,0xeea7292a),
   LL(0x090f7e37,0xc1a04582),LL(0x3f65d4c2,0x8a13821a),LL(0x10d83d97,0xce32c6c5), LL(0x26738cf6,0xedabac6c),LL(0x1200113a,0xf2615b30),LL(0x4c0415f6,0x7cee46f2),
   LL(0x6d603672,0xabe36235),LL(0x4bf4eb2c,0x5f4f1ae8),LL(0xa26e6c0e,0x9a8e44c8), LL(0x167b41d3,0x0944d401),LL(0x76f5c7d2,0xb60f0fd0),LL(0x0f6051df,0x5efdc9a4),
   LL(0xd2fd3d8c,0x456e8957),LL(0x5374a59a,0x22dbddfd),LL(0x2a3a89ab,0xbac78cef), LL(0xc10c1d79,0x9c3ba82a),LL(0xa3dd19d1,0x1f935fcc),LL(0x1919803f,0xeb08bb8f),
   LL(0x6500b9f4,0xddb31f74),LL(0xe0649288,0xaf383638),LL(0x861c8b10,0xe0dbe97a), LL(0x76552cfe,0xca7ec939),LL(0xfe33f78b,0x7aec0efb),LL(0x11123c97,0xf4548120),
   LL(0x53e3f6a8,0x900bdaa4),LL(0x03a79d16,0xbe717d84),LL(0x5f55a064,0x3c99d359), LL(0x7553c946,0xdfd860a0),LL(0xf4133310,0xb4480bb2),LL(0x4f946025,0x6401d83d),
   LL(0x1f75cad5,0xa03d7c76),LL(0xf4042451,0x97da3fcd),LL(0xcbdbfbe2,0x6a1b945d), LL(0x7a8cce9d,0x516a95b8),LL(0x30905cc3,0xb16e3a1e),LL(0xfcfc0d25,0x6adefe9e),
   LL(0xae7626a8,0x3d86d527),LL(0xfeb4e78c,0xdecc9c13),LL(0xf94a1fd5,0x3ba67e62), LL(0x8e7a588c,0xf5ac7572),LL(0x0fd2201e,0x2b741d8f),LL(0x20073536,0x3be1b453),
   LL(0xcba034d7,0xe2b80fd1),LL(0xb0031b65,0x6edad8d5),LL(0x927a36ec,0x3aee1cb4), LL(0xe39b9efd,0xbb88d751),LL(0xc92d3641,0x0129b42f),LL(0xd09b53ac,0x4b434faa),
   LL(0xdde24bc8,0x15b64975),LL(0xb1d4c6ee,0x481df832),LL(0xb358244d,0xae6cc85d), LL(0x6e6afac6,0xd54043d7),LL(0xc89eec23,0x50af2efc),LL(0xb2d29f99,0xf5fab1cc),
   LL(0xb6f646fa,0x41e0d32a),LL(0x6faa2688,0xec8093eb),LL(0x42550962,0x991ad76f), LL(0x07681a7f,0xcd41aff9),LL(0x0304701d,0x1316c11d),LL(0x133a0046,0xcce7ab79),
   LL(0x79249a06,0x34db09fa),LL(0x48f2c060,0xf3356247),LL(0xfc942369,0xe8272db6), LL(0x9849b02d,0x2b844d9c),LL(0x5af3f78c,0x8cc29c86),LL(0x79d64275,0x0bf13430),
   LL(0x5c5ea1b6,0xa299919b),LL(0xd6ef8e0e,0xb6184c83),LL(0xf2084644,0x3628b342), LL(0xe68eb3e3,0x017b62a9),LL(0x6f608feb,0x6d5b1e74),LL(0x8218965a,0x1b45f2e2),
   LL(0x3529b0e2,0x76707d69),LL(0x60a070db,0x64fb0f14),LL(0x83396a82,0x42737c50), LL(0x3b5d56c9,0x74698c2d),LL(0x284fca9f,0xa8fceed2),LL(0xe3d44208,0x30c3e857),
   LL(0xe3bfd480,0x78614fa4),LL(0xa0e38787,0x51ce5be0),LL(0x4d57abb0,0xa92d75fa), LL(0x9fad5db2,0x8d4d97f2),LL(0x1ed2bf25,0x701a80b0),LL(0x3a2db6d9,0x8bac6dfd),
   LL(0x3bb9e4df,0x092ecae1),LL(0x1a72623d,0xb57cfcce),LL(0xdac4d79c,0x38943831), LL(0x1340b17f,0x7bf10d57),LL(0x89300f7c,0xac6ec65d),LL(0xf71b46d5,0x1d4109cb),
   LL(0xcb8aeb3f,0xa00a19e7),LL(0x18b1c766,0x850b5102),LL(0x6318ebe3,0x3ec2566d), LL(0x42b9bbaf,0x8e70f71f),LL(0xb35f50b3,0x777d9ed5),LL(0xccf46394,0x332280ce),
   LL(0x87ceef93,0xf49974f8),LL(0x371f9494,0x06bef26e),LL(0xe9ac3207,0xa5e9736d), LL(0x9df3a98e,0x83769455),LL(0x8dd1f6f9,0x1ccdbfca),LL(0xe399f893,0x8b554d16),
   LL(0xefc22c30,0x36752ea8),LL(0xff7e7724,0x104614bf),LL(0x3a74e7a5,0x306ea93d), LL(0x1c1b929a,0xae54fe5c),LL(0x289c4ec7,0xd8ed29e0),LL(0x105ca5d9,0x6153b2a0),
   LL(0x29494bd7,0x5e854f89),LL(0x04a28ba0,0x85a970a5),LL(0x394d6453,0x9422f2ac), LL(0x4df93f6b,0x8e084e65),LL(0x9cbe0ba0,0x3963ac8e),LL(0x84ce0da3,0xd8aec4c5),
   LL(0x9cb36f40,0xd3d12931),LL(0x36f44ee4,0x5868644a),LL(0x92d4f69a,0xa2106e73), LL(0xa33cf53e,0x2987838c),LL(0xe1a0a530,0x8085e7c7),LL(0x906c8b5a,0xfe92589b),
   LL(0x0056310a,0x67d14723),LL(0x9e76d3b8,0xbb0a63ad),LL(0x299a1026,0xe8012f4b), LL(0x9d1ee6ac,0x620ecb4b),LL(0xd16c0542,0xd7f0c277),LL(0xe42209e3,0x8dc7db24),
   LL(0xb0f70b40,0xbc708816),LL(0x979dc8d3,0xefe99e05),LL(0xc18443d4,0x80932dc0), LL(0xd18a619c,0xea8bf51a),LL(0x9468aa4b,0xbc18f6ca),LL(0xc3414f98,0xa296c07d),
   LL(0xe4242d98,0x6cf05251),LL(0x656af78b,0x3e220300),LL(0x23a061af,0x0fdbbc02), LL(0xe31e3724,0xcd04e892),LL(0x629b6426,0xf5c0c669),LL(0xf4bf0309,0x9499d7ec),
   LL(0x685aa7b3,0x0e8c0c01),LL(0xd6abde69,0xf9179568),LL(0x8273d4cf,0x9bab3bd4), LL(0x90b3b1df,0x21b5831c),LL(0xfe3fe362,0xa86800a5),LL(0x244e7750,0x1b1f0560),
   LL(0x6ec31a03,0x45e0b747),LL(0xe1072284,0xe2ebe7fd),LL(0xec603fd9,0x9f5c1060), LL(0xf78eec39,0xab1556b3),LL(0x9ad1144b,0xbc3da215),LL(0x3ac47d57,0x84785cf3),
   LL(0x49ffcfa9,0x9c91a508),LL(0x609ea085,0xc6f3b880),LL(0x673b6c64,0x9883274f), LL(0x8302014d,0x3b40d18c),LL(0x599dff26,0x9c68a876),LL(0x1d6601fc,0x689e6aea),
   LL(0xac0742df,0x80d1dbe3),LL(0x08f7ebbd,0xba614de7),LL(0x9d2a62b1,0x69a683fb), LL(0xc964cf44,0x2c1a9700),LL(0x6bdd35c1,0xeb51e86a),LL(0x02fff771,0xcea21c66),
   LL(0x42d0fe7d,0x89b1dfdd),LL(0x85376650,0xa2a98bb3),LL(0x94d35fef,0x4f94ad00), LL(0xbd9991f1,0xe483a5b9),LL(0xce3c5861,0x8a2f4bb3),LL(0x9b125141,0x0fa548d3),
   LL(0x914022da,0xec7c9526),LL(0x44ce5265,0x21974559),LL(0x11bbfb98,0xfdc0f3b4), LL(0x077f573e,0x59c23e06),LL(0x4c97db66,0xf09dfcc0),LL(0x56abb779,0x537a65cc),
   LL(0x08a4b325,0x11440278),LL(0x0980ea50,0xf848026a),LL(0x676accd3,0x4c3ebe77), LL(0xb9eaeece,0x560720d4),LL(0x0940b1b7,0x78f3fbdb),LL(0x3d9d9db6,0xed41f518),
   LL(0xd0762b40,0x7098160c),LL(0xd83f532c,0xb70a2692),LL(0xa861e3ae,0x4d388dfa), LL(0xb6eb9422,0x4c663b91),LL(0x867c33b3,0x16423348),LL(0xa8a6c969,0x291f41c6),
   LL(0xd5028ec8,0xd9ad7bad),LL(0x02886a38,0xdaaa8d81),LL(0x2a6977bc,0xa1f8c88c), LL(0xad4366c0,0xad549474),LL(0xae44ac0b,0x36c0cffa),LL(0xeff81937,0x98fbbad5),
   LL(0x23a08ab0,0xbe19d7e3),LL(0xf3339f11,0xc5d656d6),LL(0x1c1ecd0d,0xf11fa3f3), LL(0x3cadfca3,0xe59f4653),LL(0xaeb0abcb,0x6901c263),LL(0x46ba47b6,0xe409428d),
   LL(0x3e3c4443,0xec64fc99),LL(0xefabc7a2,0x906e0f24),LL(0xcb0e7955,0xd3d83a34), LL(0x2801f179,0x932e43c1),LL(0xecfcc1d0,0xa02e934b),LL(0xea7f5e30,0xfe881c78),
   LL(0x96e7b595,0x2be7b0b5),LL(0xc93de911,0x87aed5b7),LL(0xf98e09ee,0xe0574143), LL(0x84396d55,0x7dca05ee),LL(0xbe0f509b,0xf643767c),LL(0xc4d026c7,0x3e216685),
   LL(0x4bf02e63,0xbd80154b),LL(0x9eb6f151,0x195227a0),LL(0x3ea46c97,0x07e13a17), LL(0x2363d5e9,0x9dcd5c80),LL(0x48c4488f,0x511f8482),LL(0x6cad16d3,0x7c333567),
   LL(0x46373252,0x2b07ac5a),LL(0xa0decd5f,0x90999421),LL(0xb9e96cce,0x96747330), LL(0x5632ceab,0x82d5a2c1),LL(0x636dddf2,0x73828eca),LL(0xf65213f2,0xd6797c5d),
   LL(0x64911f54,0x20daab28),LL(0xc600c4de,0x01e50c2a),LL(0x63487592,0x2111ded1), LL(0x8f857962,0xcd1073ae),LL(0xb7ae56ad,0xb4caf0a9),LL(0xd86137f2,0x416f957f),
   LL(0x11b1bc91,0x8f563687),LL(0xaa0794bf,0x4e72baac),LL(0x91c1bc51,0x932208a9), LL(0x2ae20dc1,0x5e2bb84b),LL(0xea1e042f,0x598ceb61),LL(0x5004d259,0xd635344b),
   LL(0x1a52940d,0xed0255c2),LL(0xe03eca1f,0x4150996f),LL(0x23f5b4d1,0x3aacce4e), LL(0xed0cf35f,0xcb550cd0),LL(0xa494561e,0xea36f4b7),LL(0xebef6e80,0x41cb8c5f),
   LL(0x879159f8,0xb73d9294),LL(0x7ca6da8e,0xc9eb2492),LL(0x1844cdd2,0xba2e5973), LL(0x7f9e35dc,0x544261cc),LL(0xea359cba,0xd5e6aff7),LL(0x24c0084a,0x066e78f8),
   LL(0xeedeaac4,0xd7604eb4),LL(0xa5c24ee9,0xaf1638b2),LL(0xfa90be8f,0xe4a2346d), LL(0x3004fb1e,0xbd43097a),LL(0xb073fbcc,0x442aafb2),LL(0xfdc31cd4,0x290829b1),
},
/* digit=12 base_pwr=2^84 */
{
   LL(0x514a7898,0xe8d780ad),LL(0x4494b63e,0x328ab404),LL(0x8744b64b,0xd2f0e876), LL(0x04b2c586,0xbd70e64c),LL(0xf8a4ba4f,0x5658ddee),LL(0x6ac85e37,0x3219d4c7),
   LL(0xa9c333d1,0xfc159a9c),LL(0x40607a08,0x02ba3b00),LL(0xbab9d4d1,0x6912063f), LL(0xdfb380c0,0x206a4fc7),LL(0x9a973b24,0xdaf80877),LL(0xb68651e7,0x95fd7ff4),
   LL(0xe8792caa,0x188e1e12),LL(0x02bc1f2b,0x5fa2294a),LL(0x69b0d8b0,0xe3949d79), LL(0x216824ea,0x7d86c141),LL(0x8b1e5097,0x7f37c42b),LL(0x92789c7b,0x08d30e6e),
   LL(0xa1ef9f43,0x07cfa020),LL(0x930de093,0xc17f6f2a),LL(0x241ddea8,0x9589ec4c), LL(0xf3c510a8,0xfd97e3b2),LL(0x182c452d,0xbce604aa),LL(0x40027066,0xf22902d5),
   LL(0x3293d199,0xc5644219),LL(0xf219caa6,0x393795af),LL(0xd5c761c5,0x7db08017), LL(0x3853454c,0x8d7e1f84),LL(0x29caaa33,0xe327130c),LL(0xb25301f0,0x5f09f900),
   LL(0x4e8c95fc,0xf87caee2),LL(0xd0ce4d35,0x631480fd),LL(0x79b8b374,0x4449a646), LL(0xd413863e,0x1eed07c7),LL(0x013cba81,0x0f3da431),LL(0xbc56baf3,0x8cc5bf37),
   LL(0x70315569,0x15bb751f),LL(0x676c5956,0xdff9fa34),LL(0x9e87acdf,0x0f196a3c), LL(0xf121946f,0xab7ac330),LL(0xab0ad7f7,0x5e9ba277),LL(0xd0c2d9c0,0x4951ba09),
   LL(0x9ed1500e,0x22bf13fb),LL(0x59f5da53,0x67273921),LL(0xfe704a34,0xe840d89a), LL(0xda4b1fa3,0xb4d02cc7),LL(0x2895a3f9,0xba32df4e),LL(0xdea82804,0x6986cfc6),
   LL(0x5da22871,0xe56de760),LL(0xa8a67254,0xe13cb688),LL(0xcdcd1ae1,0x3834e31a), LL(0xf18fd6d9,0x9c68e6a3),LL(0xfc21c435,0x695c7ff0),LL(0xe7b0f948,0x99254013),
   LL(0xebb3db6a,0x229fd0ae),LL(0x22db0b74,0xae94f977),LL(0xae623467,0xa05744c8), LL(0x8556c819,0x3bbcfbe2),LL(0x60bdc20c,0x39d79df9),LL(0x59b1b053,0xdc97b133),
   LL(0xd4d6ee50,0x9f703270),LL(0xa184f3ed,0xd444c71d),LL(0x978e07a9,0x7918b4b1), LL(0xbc8d8792,0x970363af),LL(0x02b0f92e,0x17f5dc50),LL(0xc8c384c4,0x2554094f),
   LL(0x2ccd5235,0x9637b2a2),LL(0x27e2ac09,0x75dba5f0),LL(0xbae90634,0xda15c76b), LL(0xe38c9c8e,0xa8374267),LL(0x2c3af338,0x062f336b),LL(0x66c7b787,0x2fa98651),
   LL(0x0837c3ac,0xc2a0d764),LL(0x62680588,0x43ba0369),LL(0x5c8c094c,0x8ec53f20), LL(0x5e289f3c,0x0ff5b7bb),LL(0xa5268547,0x8785c02f),LL(0xf5cd3301,0x1f17344d),
   LL(0x6269e364,0x8b71131f),LL(0xe5772b51,0xe8bda70e),LL(0x9ec4dbc0,0x76ca9cf6), LL(0x6b091b64,0xe6f8b934),LL(0x8b563cf7,0xda109f46),LL(0x4b3bb11d,0xb109f7e0),
   LL(0x0beddcf4,0xf97b7930),LL(0x9c3c946b,0x342f1b05),LL(0x0204a23f,0x4a73cecf), LL(0xe3adbd26,0x15f75cd1),LL(0x41c9110c,0xf09a81fb),LL(0xf5d8cda5,0x88dc3f03),
   LL(0x8c96a49d,0x960481f1),LL(0x464ab9fc,0xc6b88ef5),LL(0x8849c44e,0x67e3f698), LL(0x5956b294,0x32068517),LL(0xd22cadb5,0x3705b95a),LL(0x836f6d0e,0x567b470d),
   LL(0xcd97cefe,0x441ff140),LL(0x54db07ef,0x659b7aa0),LL(0xfe435aa3,0x8784a63a), LL(0xf04a616d,0x1d970140),LL(0x53af3b98,0xe29b21bb),LL(0x6bb3c12a,0xe6d0f2b8),
   LL(0x368f5fc6,0x02132f5d),LL(0x66fdcb81,0x49d6a524),LL(0x9c2be80a,0x05aa3ab5), LL(0xc2500560,0xb7b2f0f3),LL(0x37df204b,0x6359ab2c),LL(0x6d32b4ab,0x25a10b66),
   LL(0xf6591fb3,0x03360968),LL(0xc52b20c7,0x16a7fad4),LL(0xb2062758,0x4140a2e8), LL(0xd82f2254,0x905dee52),LL(0x53363565,0x29405a93),LL(0x416c034d,0x1ba1ab88),
   LL(0xddd5cfd7,0x8bc8af26),LL(0xc77da82b,0x658c496b),LL(0x1827a062,0xd77384a1), LL(0xfce01201,0xee17f994),LL(0x2d3b276d,0x46605636),LL(0xff5d6088,0x02a31f1b),
   LL(0x9e7536a0,0xdb49e057),LL(0x99c35008,0xbcc58d33),LL(0xb687e5e3,0x1f8bc7f4), LL(0x72027d06,0x8638a410),LL(0x2049ced9,0x0c6a070f),LL(0x69ef4623,0xd2729607),
   LL(0x3355ca14,0x32aef267),LL(0x848620d7,0x8bbdfbda),LL(0xb0bcc374,0x65eccd58), LL(0x0370119b,0x15730aad),LL(0x349eba95,0xc7d99e64),LL(0x27105839,0xd30d076f),
   LL(0x9f258a16,0xcccc32e2),LL(0x535d3667,0x1dacc394),LL(0x81f623e7,0x3f955c62), LL(0x38cb46b2,0xfe599a1f),LL(0x831c4abc,0x88fa6156),LL(0x1708497b,0xb9242cc8),
   LL(0xffd874ce,0x9bf9b706),LL(0xa0f974f9,0xf632747c),LL(0xd17df676,0x3f1263b7), LL(0x44df1a8a,0xd7c0eb1a),LL(0x382c181a,0xaaac2337),LL(0xef3d9b5d,0x8cf66156),
   LL(0x5bd8a32b,0x1d1d11dc),LL(0x8663b96a,0x57cdb2bf),LL(0xd7aca4c0,0x0983005a), LL(0x50c2f986,0xb3cffa36),LL(0x65f29ae6,0x069eb437),LL(0xec49900e,0xd896f1aa),
   LL(0x455dcf4f,0x7152f4c1),LL(0x238cda30,0xa6b6e460),LL(0x2b553cad,0xb4b732b3), LL(0xe87524d0,0xa6d73263),LL(0x378032e6,0x6d48d14c),LL(0x0366d951,0xde0def0d),
   LL(0x258e8fa1,0xf3647476),LL(0xd3d8e3c9,0x11f581d6),LL(0x3da389c4,0xf95d4219), LL(0xdd4c30f4,0xfe37e7d7),LL(0xcf2e5808,0x9a47035d),LL(0xbcccbd4e,0xe167952c),
   LL(0xf19a3807,0x26816e58),LL(0x34036c34,0xbcdfe39a),LL(0x559a4898,0xa9d96a3c), LL(0x5922653b,0x2be83f20),LL(0xdb63d585,0x31e57750),LL(0x90d8cdb6,0xe25a5789),
   LL(0x07016f2e,0x7598794f),LL(0x5b52794b,0x8631b406),LL(0x0473326c,0x4a7a3e13), LL(0x38098ff0,0xfa8445dc),LL(0x224ae6d5,0xe0829d6c),LL(0xa34dbb8b,0xb4ecb6ec),
   LL(0x41a5e88e,0x505e3c0d),LL(0x3428e632,0xedcfa76d),LL(0x7378d468,0x1a6892e9), LL(0x3387bfd7,0xcf8f2e12),LL(0xcfedce94,0x75c21720),LL(0x64252690,0x92bd9a2d),
   LL(0x7db48f12,0xb3dc859a),LL(0xa7e11cf8,0xf26b3a7f),LL(0x4b8bcf8c,0x38994648), LL(0xf70596d4,0xbe5e497b),LL(0x475a1d46,0xb0e98337),LL(0xb6229496,0x15d374ea),
   LL(0xaffa1419,0x10c53832),LL(0xca139934,0x41d4f8c7),LL(0x2df85ac8,0x39e8b454), LL(0x835eedf8,0x8c98e7d9),LL(0xa2d40a9b,0xd931771c),LL(0x7f3156e2,0xf172ea3d),
   LL(0x85791984,0x551aa33e),LL(0x5a936719,0xe7900736),LL(0x706f62f6,0x6cf471c1), LL(0x35562192,0xee2f4edb),LL(0x23457cb9,0xa3b54c39),LL(0xbfa40e0a,0x893cc476),
   LL(0x84db691f,0x9fdf4d5a),LL(0x1b4ae453,0x9c0411b9),LL(0xa116f967,0x49d4f2d8), LL(0x0b67725e,0x76a1f188),LL(0xc88bd7ed,0xe7d9ffcd),LL(0x68698380,0xc0d8b68d),
   LL(0x689cd83d,0xa6a6cba8),LL(0x9cff9c33,0x2f3e63e2),LL(0x86212397,0xf4698015), LL(0x84afdbb8,0xbdda0fc7),LL(0x582808e9,0x03350c9b),LL(0xb37b0173,0x2682ff4a),
   LL(0xbc9fa923,0x8bd44537),LL(0x62a20c38,0x68ce391a),LL(0xa248616b,0x22585962), LL(0xe3f811fb,0x5dd9a480),LL(0x8909e98b,0xea0859e8),LL(0xa790c444,0x5f7e1b54),
   LL(0x62155fc6,0xb10a93d1),LL(0x0700ee23,0x32320e65),LL(0xc17d480b,0x9eb37705), LL(0xae0b68e3,0x21f11ede),LL(0xda53ea55,0x999286c3),LL(0xa359fe14,0x97dee225),
   LL(0x1f39aaa7,0xa59acbcd),LL(0xaaf5a212,0x5386d1a7),LL(0xcbc7628e,0xbda27b0b), LL(0x8a7e5aa7,0x6bd51dca),LL(0x59bbf399,0xb7dbe2e2),LL(0x81cc4d35,0xc15c94bf),
   LL(0xaeb9c83c,0x600a3d7b),LL(0x8659b786,0x2b13cbc2),LL(0x5cc8f7f6,0xeac2aed7), LL(0x6bf18dea,0xfc7f4bb5),LL(0x11ea3f69,0xc2d16cb7),LL(0x341cbeea,0x69784d06),
   LL(0xbe212759,0x5b823c92),LL(0x0f3cd127,0xa65cc417),LL(0xe67e7aeb,0xd7ba38a5), LL(0xe796850c,0x4e572182),LL(0xf9fc752f,0x36463b09),LL(0x1b698241,0x26827970),
   LL(0x7c17865d,0x4cb9ca41),LL(0x7e6cfd97,0x214fe7d9),LL(0x4f587903,0x61d96b63), LL(0xec411bc5,0xb0101b58),LL(0x77f687cf,0x36692f2c),LL(0xcc9bfb45,0xe54d3eac),
   LL(0x99723bab,0x121bfab7),LL(0x0884ebfb,0x893ab325),LL(0x6502d3f7,0x0e6788ad), LL(0xb7a108f2,0xb0513c5b),LL(0x4b2eef9c,0xa4f73e16),LL(0x6a483c20,0x342415c6),
   LL(0xe752defe,0x39a27b3f),LL(0x9126ae15,0x0d86022f),LL(0x68883326,0xd8ba4a8e), LL(0x3b100c9a,0x715f52c8),LL(0x4addc384,0x336e496a),LL(0x8a6f3e95,0xec21f02c),
   LL(0x7769257d,0x2894dfef),LL(0x5ac4bb61,0x2fd58cea),LL(0x708fac5f,0x3cf6d8b6), LL(0xe836e867,0xa76d99fc),LL(0x42a369eb,0x0d863adf),LL(0x15d821ff,0x82c333e2),
   LL(0x4042c21a,0x9143b876),LL(0xc737f640,0x0c242403),LL(0xc719ca5a,0x7c397a93), LL(0x022b09e8,0xcf7897df),LL(0x12178c00,0x5c92e3dc),LL(0x4bd00abc,0x4fb2687d),
   LL(0x911220ce,0x5750f3ec),LL(0xfb86ec9f,0x58bb4fbf),LL(0x94d00517,0xe33fcc7d), LL(0x5d9eefa0,0xa09ae695),LL(0x9756d5a4,0x3a331936),LL(0xf5c16f2c,0xc6da863b),
   LL(0x07532fc8,0x05c5e926),LL(0x9d0d96bb,0xfe0c58e9),LL(0xcff62c16,0x98056d33), LL(0x346a4ec2,0x77cbe60b),LL(0xf43fd4c2,0x40989b21),LL(0xfd0645ec,0x1a3c47ee),
   LL(0x5f51dc39,0x91089174),LL(0xb5a23122,0x7cbbe6a0),LL(0x8c2818e7,0x83b5abb0), LL(0xb9e654d4,0x24081e1a),LL(0x5ba06cb3,0x63528af7),LL(0x69d7dcc6,0xf570710c),
   LL(0xb34b5890,0x583afeef),LL(0xc813cc91,0x6095edb9),LL(0xd3c329f1,0x166a5e48), LL(0x33293a13,0xee878e89),LL(0xeb1ad806,0xe6c0325e),LL(0x210c80b5,0x8add3663),
   LL(0x3811f849,0x6b9517b7),LL(0x9d8bad91,0xd0aa5e75),LL(0x3160bda4,0xf70a8172), LL(0x916ce0d5,0x24b52ea7),LL(0xac05f7b8,0x13da7006),LL(0xbd773df2,0x1213f1bf),
   LL(0x72d96f4d,0x22b2b776),LL(0x98ecfb1e,0xb0c964b2),LL(0xba7dd824,0xed9d0828), LL(0x869c0f63,0x6faddff3),LL(0xf8b13355,0x2b409d2c),LL(0x31237c9a,0x53431320),
   LL(0x72a9170c,0xde41027d),LL(0x3a82f137,0x40f7b012),LL(0x099b7d4a,0x343a36ac), LL(0x2606178e,0xbe82d933),LL(0x19c93f1e,0x741f76ea),LL(0x27fd9465,0xd13d492f),
   LL(0xf2518396,0x9b09dc67),LL(0x14669921,0xa79dc92b),LL(0xbea82992,0x0eb149ce), LL(0x13c60cbc,0x62ff22de),LL(0x6141a946,0x610724f4),LL(0x0793021b,0xf57b976b),
   LL(0xc699a47f,0x0e516c18),LL(0x5d6f8d8f,0xd719adfb),LL(0xbdb43f6e,0xf067a17e), LL(0x0b490085,0xce5eb760),LL(0xcfebbb1d,0x15113e6a),LL(0xee6d2951,0x2194fd0a),
   LL(0xd1abb57b,0x0bddf14e),LL(0x3416ad68,0xe0e768c2),LL(0xc4534e1b,0x0326b07d), LL(0x3da9b4be,0x796b6192),LL(0x474dbec9,0x85976f8b),LL(0x64001998,0xe1b892ef),
   LL(0x3ddc43d1,0xb387d9c2),LL(0xdcc58653,0xd879f295),LL(0x316adb79,0x680e4335), LL(0x771da848,0x6ebecfdd),LL(0x98e7ae92,0x21e44da1),LL(0x2a3e21ba,0xf7db9238),
   LL(0x6d480c11,0x7be805ba),LL(0x5ed87195,0x657824d5),LL(0x152a427d,0xad9bb855), LL(0xd479e14d,0x54fa5b2b),LL(0x967d66b3,0x393ff9fd),LL(0x1f8cde27,0x2e20e269),
   LL(0xced8a78d,0xbc607c2a),LL(0xa8eed88d,0x191f87df),LL(0x5861efa0,0xae1d6b52), LL(0xcafa53fe,0x57c60dc3),LL(0x67c35922,0xb27b8932),LL(0x9b29125d,0x0fa6246f),
   LL(0xb808e2df,0x50a069c2),LL(0x9a88a445,0x84e557c0),LL(0x38d713c5,0x27e6460d), LL(0x3c86b058,0x7b410b71),LL(0xb1b113ec,0x93fa53b8),LL(0x7441167f,0xc634cba5),
   LL(0x282c12ce,0x1e671bc2),LL(0xa604f3aa,0x39b311ce),LL(0x78a148f0,0x98bd47d2), LL(0x371a0da8,0x5e27388b),LL(0x0fd9e315,0xfa513fbd),LL(0x34376073,0x611e38e3),
   LL(0xb58100d4,0xd5ebd78f),LL(0x6e9f40fc,0x274fae3f),LL(0x8942e656,0xc1a74338), LL(0x46604337,0x6c53a23e),LL(0x660df2a9,0x2ed93bd4),LL(0xbf37f8e3,0xdfa53649),
   LL(0xa0632b8b,0x02511435),LL(0x3a69fc4e,0xdf7abcb9),LL(0xd9e52727,0x6717e277), LL(0x929a903a,0x990834f2),LL(0x80c4abbb,0x8b89a584),LL(0x94e5ac4c,0xd57fcb53),
   LL(0x6acebef8,0x7293af18),LL(0xdc7aa8d3,0x12646628),LL(0x409608e9,0x89165145), LL(0x01ea6183,0x9086b84f),LL(0x78ce39d6,0xc508268d),LL(0x6ff226ec,0x1d30f560),
   LL(0xdd6833a4,0x49bf690d),LL(0xa575c955,0xf290aff6),LL(0x7bfea4f5,0x0e0b673b), LL(0x8e59e02b,0x3fa210fa),LL(0xbbfb09d6,0xc1cecbf6),LL(0x670a5aa4,0xa76eef60),
},
/* digit=13 base_pwr=2^91 */
{
   LL(0x01c06ca9,0xa1498bce),LL(0x7a08bab0,0x21c01611),LL(0xca17bd74,0x2c85a0b7), LL(0x37d0f77a,0x7b903162),LL(0x6e81e65b,0xd14898cd),LL(0x218cd115,0xe0ed1f2f),
   LL(0x1731d3e6,0x78c6c92d),LL(0x7d3b6343,0xee373594),LL(0x614f0ed3,0xc8e90901), LL(0x680df68d,0x9dd54b4b),LL(0x983f94c5,0xd0c3cba6),LL(0xe9dd9140,0x80983823),
   LL(0x9da2d6f5,0x3de0421a),LL(0xe135d7df,0x933d779e),LL(0x45fbc83f,0xdcc10777), LL(0x552833c4,0x437d64cc),LL(0x581bf745,0x616eb481),LL(0xaf9d7fdf,0xc3d7614b),
   LL(0x4b010379,0x72673639),LL(0xb40b8ccd,0x784d95cd),LL(0x32191a11,0x3647d089), LL(0x83ad425e,0x1383335b),LL(0x5ed3ff39,0x3c5b0250),LL(0xa814df0f,0xf1f78c9e),
   LL(0xd81332fb,0x48ee4921),LL(0x75c4a231,0x62f08af2),LL(0x52b5815b,0x81cee592), LL(0x47ff2530,0x5ee938f3),LL(0xbb73d595,0x9a58ec1e),LL(0x29b94b64,0x473405f4),
   LL(0x0f89bcf3,0x423677b2),LL(0x3746d402,0xaafec85e),LL(0x8cec15af,0xb61ce3f8), LL(0x5ba02dd3,0x7618d63f),LL(0xc28a6515,0x4f774238),LL(0xe74d6f15,0xf6a3b5c5),
   LL(0x0a141a66,0x695b4444),LL(0x09e7251e,0x9b502847),LL(0xa2883803,0x52285abb), LL(0x5f2f7fb4,0x71858bb4),LL(0xa5943828,0xbf5eef5c),LL(0x81bbe54b,0xb29d0c74),
   LL(0xaa092e25,0x83536ebb),LL(0xb4e5d213,0x4768e469),LL(0x8289c98b,0xb540b589), LL(0x080c0531,0x350a6e40),LL(0x590b38c3,0x94d12807),LL(0x40fe5f65,0x83badade),
   LL(0xe86caac6,0xd77e33c7),LL(0xb8ead427,0xd1935c95),LL(0x5183cc83,0xde18e630), LL(0x261521a1,0xbdcd00f1),LL(0xc82809bb,0xbd18e1d9),LL(0xf960fc2b,0xe142711a),
   LL(0x1da8931c,0xdfe9e3bf),LL(0x6e1829f7,0xeb943b45),LL(0xe6eeee44,0xa1c817bb), LL(0x4fdd6559,0x0f852c30),LL(0xa27f2549,0xd9ad9b4d),LL(0xfb111f65,0x266fb914),
   LL(0x917fc712,0xa20f882d),LL(0xf7566719,0xd05d7f13),LL(0x12c11721,0xd5c196b9), LL(0xc566b104,0x3eba0c8a),LL(0xecae1039,0xe301bd90),LL(0xfb24c348,0xd64eb499),
   LL(0xb4738f00,0xa0c38991),LL(0xe4dd35c2,0x63bc012b),LL(0xf72495e7,0x980e89b0), LL(0x68ad6dd8,0xb530252c),LL(0x2642e8d1,0x35d6d8ef),LL(0xc235ef4f,0x1309691e),
   LL(0x7262cd15,0x64a61d20),LL(0x04da445b,0xf2916951),LL(0x4ed50bd4,0x3beb5815), LL(0xe0204220,0x35638863),LL(0x69f21c08,0x8929e2ab),LL(0x14756735,0x5f273190),
   LL(0x819c11ec,0x47befaa3),LL(0x90deb435,0x3bcfa2d2),LL(0x19569f95,0x4caca5ba), LL(0xd48453ad,0xbc78fd0f),LL(0xfa2100f2,0x7e4c3646),LL(0xf96c19da,0xc75a1a82),
   LL(0xed646507,0x6e45f6c3),LL(0x86558b67,0xae0601e1),LL(0x3ccec830,0x20634c8e), LL(0x7c87c192,0xc3fea47e),LL(0xb66a402d,0xf14aba2e),LL(0x5bb3f7e1,0x8dcecf35),
   LL(0x9751dc6a,0x33396804),LL(0x1c107cd8,0xc56b5879),LL(0x5b41235b,0xd8a41e90), LL(0x35971626,0xa956148b),LL(0x79168709,0x185bfe69),LL(0xeba02351,0x32829b76),
   LL(0x39f2bf71,0x28f979a3),LL(0x53316bad,0x212c0931),LL(0xc1f8d9f1,0xf02710c6), LL(0xbb871267,0x6e6cbf36),LL(0x9c425ef5,0x11c79382),LL(0xb94f8b01,0x6ce18819),
   LL(0xbd5bf1f3,0xd9a44d46),LL(0xda2c3d1d,0x662f70a1),LL(0xb13563ca,0x43d5374b), LL(0xcbcce692,0x8f4ad04c),LL(0x0c72e84b,0x3151ee4d),LL(0x5accc2ed,0x50c80454),
   LL(0x74697919,0xdce044f6),LL(0xedbef994,0xfce0b586),LL(0x8b92c529,0x0d47e307), LL(0x6ac46081,0x64904019),LL(0xad80c124,0x6f52cad2),LL(0x996d42ee,0x02f7e44d),
   LL(0x6aa2e142,0x257eae2c),LL(0xba53beb1,0x747a167a),LL(0xa8f5eba2,0x3b3ac5e3), LL(0xb6fa720b,0x6c6ee79a),LL(0x3a7bd99a,0xf6dc5ff5),LL(0x1a36d3b5,0x2447970d),
   LL(0x7b6c6e86,0xd717e889),LL(0xc64d98d4,0xb808d76c),LL(0x7f8c0bca,0x9a06f942), LL(0xb253f5b4,0x58b8b04b),LL(0x0c103d51,0x82d27325),LL(0x1810ceba,0x7dd6d9a6),
   LL(0xa27bfff3,0x0a1f1bb3),LL(0x3f63695b,0x3ba841c4),LL(0x4dba6a84,0x0e4c6ef4), LL(0xc6d82a09,0xe283f321),LL(0x78f4a01f,0xd8bcb661),LL(0x504f48b4,0xf527ac07),
   LL(0x750dfc07,0xade1c74f),LL(0xa0611072,0xb8acc3bd),LL(0xad13e407,0x3886a58c), LL(0x0011bec7,0xdf6dd4c5),LL(0x350b7b68,0x01d96ff3),LL(0x76a879ea,0x2be73b43),
   LL(0xca38d7cf,0xc1eca6c4),LL(0xe4217860,0xed1d926b),LL(0x0c476828,0xffc2433d), LL(0xf0c18d59,0xee65a719),LL(0x0ca1dca8,0x6a00e59f),LL(0x10811b54,0xc137d01e),
   LL(0x66442ac3,0x74f38545),LL(0x1ecdd1f4,0x4947a37b),LL(0x5407e0c6,0xdaf73a23), LL(0x14140508,0x1b6175f0),LL(0x7b23a861,0x628c409f),LL(0x60c7c2f4,0x2550fdb1),
   LL(0xd23a25b8,0x0e026ec7),LL(0x2b9ed7bc,0x435dbf9e),LL(0x027cd584,0x16c87162), LL(0xd7f56db4,0xa3ab5d25),LL(0x7f477034,0x8c9fa8ef),LL(0xac46696a,0x2eb3909a),
   LL(0x7bc6b191,0x6f288477),LL(0x72cf52e6,0x7857879c),LL(0x4438a334,0x9ccb9b67), LL(0x0110d13e,0x016f2b86),LL(0xa623a4ce,0x7355fff7),LL(0x88c1b0b9,0x1dd555dd),
   LL(0x543e3d58,0xecc48b76),LL(0x63049aab,0x251ea3ca),LL(0x1753f7aa,0xd091864f), LL(0xd65af237,0x0fd2d1af),LL(0x0ac895b0,0x70403da1),LL(0xa89e9386,0x6c3bb93a),
   LL(0x48e9bf38,0x4d1df04d),LL(0x36a1b91c,0x4d05792d),LL(0x6405251a,0xd07633cc), LL(0x3ef7d87d,0x0077621d),LL(0x0c74697c,0xb35da825),LL(0x2698f465,0x521ee502),
   LL(0xf4454069,0x2f0af2e4),LL(0x0799d8f7,0x1c6595ad),LL(0x57d14169,0x8cc1b16b), LL(0x660a073c,0x6ef9a078),LL(0xcad9cb79,0x4d04db63),LL(0x2d73e47f,0xbd1843d6),
   LL(0x9838b1cc,0xc34b6a45),LL(0xe4d57b01,0x37705ff3),LL(0x8dde6264,0x7b15a361), LL(0x7bf8c818,0x22c4dd55),LL(0xfccc811a,0x404bd481),LL(0x601ccc8a,0x6f6b37a1),
   LL(0xd8ecaa77,0x033ee5b1),LL(0x59bdb383,0x4a6fbaa6),LL(0x991bd043,0x98d656cd), LL(0xc754b213,0x489d8614),LL(0xf6bcf12a,0xc57d54b4),LL(0x1023de85,0xd8b7ed83),
   LL(0xdc84066f,0xcc88e0d0),LL(0x112d5d04,0x563deb7a),LL(0xe6715a55,0xcf71acb5), LL(0x29268b7f,0xf49a247f),LL(0x5b36fd67,0x8e53cf14),LL(0x06da22fa,0xa4093a91),
   LL(0x2dfb4f20,0xfbe6fb96),LL(0xd6946271,0xadb8875c),LL(0xc95a344b,0x50528b99), LL(0xd2627469,0xd7442d08),LL(0x47e943cf,0xb8b049e1),LL(0x65cd148f,0xa47b0b3a),
   LL(0x50579b69,0x1cd28588),LL(0xbb9f9f10,0x38acf2f6),LL(0x0b35f238,0x67eaec1e), LL(0x00a3119a,0x72fc9c42),LL(0x39f88c39,0xa1cfc76e),LL(0x14c7fda2,0x67b702d8),
   LL(0xeeafce5a,0xe782ecf6),LL(0x5eccfbde,0x16b562b4),LL(0x2a70e93b,0x23a8eaf5), LL(0x64855ed4,0x0fb734f6),LL(0x2bd5f2b8,0xde37f9b5),LL(0x959a5858,0x3d01215d),
   LL(0x667d4abf,0x5a4a7d0c),LL(0x68d13116,0xf4f05c66),LL(0xdb565813,0x09cc67cd), LL(0xb34d0f55,0x53d0e496),LL(0x5a179043,0xa4f11437),LL(0x39b26f7e,0xf4ed66b3),
   LL(0xbf723512,0xc670ce6a),LL(0xb17e4ba3,0x5bbf65a7),LL(0x16e38386,0xb5ddc6ff), LL(0xb0dd0439,0x9c28d121),LL(0x66dfee98,0x06d89034),LL(0xf7d33983,0x547138ac),
   LL(0x92a04c6b,0x8503bf04),LL(0x008a5092,0x8761e5ef),LL(0x13902f4b,0xfa9bfa8d), LL(0xd7c9b5cb,0xbc37af37),LL(0x9f7a27b8,0xcfb05e37),LL(0x1d5b124f,0x81418e4d),
   LL(0x03353c76,0x0d57da43),LL(0xdd7feecd,0x3f36caaf),LL(0xdb56957c,0xdcaadf69), LL(0xb652acff,0x2c36792e),LL(0x4213d519,0xc5d03a97),LL(0x98734fe1,0x11c6aa92),
   LL(0xf714efdd,0xeca42e87),LL(0x9655a0c3,0x9d65e8e3),LL(0xe741a619,0x5be55b09), LL(0x1066c180,0x53b47016),LL(0x805e1ebf,0xe9ddc1b3),LL(0x79a748ef,0xc1057ad9),
   LL(0x66d0174b,0xb2d62622),LL(0x37c05be2,0x10a0512c),LL(0x63e47f0e,0xc2eeab52), LL(0x6f08ceef,0xce6e28b5),LL(0x2f1a5a13,0x7716afb7),LL(0xc55ab74e,0x83942c61),
   LL(0x6539037b,0xcef92b51),LL(0xdc615b4b,0x04a4f1e8),LL(0x49e546ee,0x0d56014a), LL(0x94d44546,0xbc1437a1),LL(0x147375c2,0xfb1e3aca),LL(0xdfc71c1d,0xb0f2abf0),
   LL(0x44424b7c,0xb5679c9d),LL(0x36c89cc1,0xde54d53d),LL(0xd9a702ed,0x23252ab5), LL(0xf775d90f,0xdccb09e3),LL(0x15cac902,0x28e652e9),LL(0x08ca0b18,0x193bd66a),
   LL(0x7abafda3,0xb81c2e5e),LL(0xbe90ac49,0x34a1192a),LL(0x9d1512b5,0x5d6945d3), LL(0x9cb6c182,0x0184a059),LL(0x868f5d65,0x329ea8fd),LL(0x66410f60,0x03783cc2),
   LL(0x1f3d38ad,0x8524ab85),LL(0x3ad4fe62,0x58a7ccec),LL(0x37ece281,0x6a00a442), LL(0x16a1a2ac,0xfa3d6fc3),LL(0xb727b625,0xf6fc8af9),LL(0x4c3b3361,0x42699573),
   LL(0x22414f66,0x2f55c436),LL(0x512968a1,0x8869eec6),LL(0xd98ecda1,0x925a1ae7), LL(0x9fb914fe,0xf112e9e3),LL(0x60e41857,0x16496689),LL(0xc8f300d8,0x1f93ed91),
   LL(0xc8208043,0xb3e2c2fc),LL(0x7f1acebb,0x19f61c61),LL(0xc1e96d31,0x1ff2f9dd), LL(0xf44de6f3,0xd956fdfb),LL(0x0685ff3a,0x101656c8),LL(0xb98bc38c,0xe725cebe),
   LL(0x354b318c,0x77e2edcd),LL(0xe1580079,0x9ac1cfbd),LL(0xedff64a0,0xc3c197e8), LL(0xaa113966,0xa2da7af6),LL(0x0aa54ff6,0x8f9df560),LL(0x88555266,0x5aaaa100),
   LL(0x4885a4fa,0x01e3a2fe),LL(0xc469f05d,0x4aea2d91),LL(0xb3ea443c,0xf5947f39), LL(0x34d7b0fc,0x6f22b634),LL(0x93042129,0xc843cc3e),LL(0x2f14f438,0xf884bf04),
   LL(0xa710c7fe,0xa0a19c29),LL(0x3057ad61,0x7fabfd39),LL(0xfe6ff084,0x94e9125d), LL(0x100a8cf8,0xd4fdf49a),LL(0x55eac3de,0x1839418e),LL(0x1aa6a2ce,0x59253c55),
   LL(0xee768081,0xe01ff1f4),LL(0x144d7219,0x083342a6),LL(0x51fe8705,0x34cfa650), LL(0x75c5938a,0x499516d1),LL(0xf2c8f347,0x6350e5fe),LL(0x73f4eb28,0xe030f2a7),
   LL(0x035b04cd,0xcd5808db),LL(0xc6f54f17,0x93d6ac3f),LL(0xbae00379,0x56d72b91), LL(0xc7448e15,0x211f276f),LL(0xaecaad32,0x979d0736),LL(0xe64e8a11,0xe581cfa6),
   LL(0x2904cb1e,0x94ac596c),LL(0x626952e7,0x53e1e42d),LL(0x775ae6d8,0xcc989b89), LL(0x44343d68,0x0fb5c605),LL(0x0f175322,0x87608356),LL(0x6db60843,0x182f1dd6),
   LL(0xe8867c7c,0x0ad3c4a8),LL(0x103cb06f,0x5479684c),LL(0x14756b01,0xe2961491), LL(0x9dfc5ad6,0x5207d54f),LL(0x3c865a45,0x184260a4),LL(0xbea2f905,0xd1158cb4),
   LL(0xc78b0c7f,0x9de51643),LL(0x31357d76,0x1a3b3589),LL(0x60f16852,0xc2175848), LL(0x4d90d2d7,0x5eb827a7),LL(0x79abe6d4,0xe83be898),LL(0xb0974ddc,0x7c03935b),
   LL(0xd54dae93,0x7ae1a72b),LL(0x0144c969,0xd4c8e333),LL(0xac7744c4,0xb3b92e3c), LL(0x751865f4,0x81a6f8e6),LL(0xb9b51064,0x11e0f64a),LL(0x8cfce4fc,0xedbcfd08),
   LL(0xfe811795,0xe33545de),LL(0x33edeb4a,0x663a0fc4),LL(0x23e2eaeb,0x0fb9d61f), LL(0x99b52602,0xf136b9ba),LL(0x766a8136,0x6a15e5ae),LL(0x05333f8f,0x5a1a79b6),
   LL(0x5a85d6be,0xda1cb3bb),LL(0x835d8007,0x28e30bff),LL(0xef4cb0ee,0x6a8c3d85), LL(0xd6781054,0xe8517460),LL(0x48b0c592,0xa0234065),LL(0x4d7b7aa5,0x5cf1f874),
   LL(0x9e511ada,0x007bf10f),LL(0x8687e7c5,0xc5edcfb1),LL(0x047a613c,0x507ca4c5), LL(0x8f117e83,0x3c8707d9),LL(0xe75c9755,0xda6e38e5),LL(0x6dbd6711,0xb3ffd623),
   LL(0xb9e2ce9a,0x8530669d),LL(0x020d181a,0xffc85696),LL(0x02c1a8c3,0xfa2f2979), LL(0xee4a1a98,0x989146d8),LL(0x76ac6289,0xc3f98bc9),LL(0x76d28437,0x1d491a8a),
   LL(0x3fb6cb28,0xf9d04b0c),LL(0xbc28bce9,0xf246e5c6),LL(0x48fd1173,0xd3d455ed), LL(0xe3e9d40a,0x00ea713a),LL(0xf5af8e29,0x4d4c9262),LL(0x9e95cf17,0xd5e49303),
   LL(0x685350ef,0x9f5fe067),LL(0x788ffe11,0xe22b2f70),LL(0xe6ff086f,0x74d4e6e8), LL(0xa937a949,0xef631c8e),LL(0xd122a378,0x707bb8ff),LL(0xff54817b,0xc1ee7dda),
   LL(0xfebc7fc2,0x328f6f5a),LL(0xd933fdc0,0x8333e284),LL(0x51ee741b,0x9c135c87), LL(0x90923a1a,0x856760be),LL(0x44a46bc0,0xbe7905c7),LL(0x77415a22,0xbdc7248e),
},
/* digit=14 base_pwr=2^98 */
{
   LL(0xd646c05b,0xd1893bfc),LL(0xa963fee4,0x4e61d76b),LL(0xbdcc95da,0x71adc08a), LL(0xdf5decf2,0x42f481e9),LL(0x0872ca53,0xbdfbba8b),LL(0x21ece5ce,0x0eabd6ac),
   LL(0x45d4fc03,0xb2367fe3),LL(0x75cbdc62,0xb115bdad),LL(0xef8b952a,0x07522180), LL(0xfd23c45d,0xc0448864),LL(0x87eb202e,0x8c068193),LL(0x313c2e61,0x7fe87505),
   LL(0xf2ede850,0xbfa39552),LL(0x93cab795,0xa40a845d),LL(0x09e97bd9,0x56ca4f4e), LL(0xccbdb84b,0x207a1e71),LL(0x1893a956,0xd0ce1e11),LL(0x84c34b10,0x7a22a433),
   LL(0xe404586f,0x48becb6a),LL(0xf5d6372a,0xa136232d),LL(0x87ba83a5,0x438af2bd), LL(0xf4e459ff,0x37075ede),LL(0x582d1ce8,0xaa42aae7),LL(0x0daeb82b,0x973dba33),
   LL(0xc40861a3,0x48bc98ae),LL(0xefd9a196,0xc100f037),LL(0x052eb352,0xda4029f6), LL(0xef34169d,0xd61b4933),LL(0xa357cc0b,0xcedde9cc),LL(0x4d006fcc,0x8983c01e),
   LL(0x92f2a1c1,0x31ad5f19),LL(0x924bcb08,0x7c7a6319),LL(0xd0c36056,0xcbee249b), LL(0x5ac4f6eb,0x63135518),LL(0x37f8cd7c,0x8b2aaaed),LL(0x0b9bcd79,0xe71c855a),
   LL(0x5828ecb9,0x1f12da90),LL(0xbed745a9,0xbd094b68),LL(0x59a321b0,0x27d4a84e), LL(0x71683549,0xd05c2695),LL(0xc9c9f2ae,0x56e4dcf2),LL(0x199abda7,0xdfff6393),
   LL(0x576fff20,0xc28ae07c),LL(0xb9024805,0x88ed10b6),LL(0x851ed43d,0x81944c6a), LL(0x158549cd,0x31192f87),LL(0xff08eac3,0x36dce6e4),LL(0x9ffacd59,0x7837d3e2),
   LL(0xc8329917,0xe820d2e4),LL(0xed56083c,0xc343156c),LL(0x67c118a9,0x60116d83), LL(0x699583a8,0xbf2c1598),LL(0xc03c3370,0x77da3db5),LL(0x88522e50,0x35b0e741),
   LL(0x3e4cf4e3,0xcb945a8b),LL(0x45fbb1c3,0x1ac37aba),LL(0x7e016a40,0x05b3d362), LL(0x93c0799e,0x175cb9f3),LL(0x9102d347,0xad1068c4),LL(0x8c6b55d4,0x01200565),
   LL(0x1fcfd55e,0x6396a275),LL(0xc76eb5a2,0x26c5c201),LL(0xc49cc617,0x9eafc039), LL(0x62fdc2b8,0xecf4f6fd),LL(0x73f3babc,0x795af8b3),LL(0xb456ddc8,0x9d6380f0),
   LL(0x00481dbf,0x8435b4c8),LL(0x586fec04,0xd68ed151),LL(0xbad5a7ec,0x904b3a7d), LL(0x755a599b,0x5ac0c68f),LL(0xeed07240,0x8202bd5e),LL(0xc2111eb1,0x50108f7a),
   LL(0x64114206,0x7966f061),LL(0x08252097,0xc7e5a500),LL(0x4a27d84f,0x1efdcfb5), LL(0x992fb0b4,0x05327d63),LL(0x109f651f,0x33c1800c),LL(0x4858bb75,0xe90ccb5c),
   LL(0xdfd79c9e,0x3f05f6b9),LL(0x94808d4d,0xb0db9df5),LL(0xda56ea30,0x170d0bfa), LL(0x46ea52b5,0xfc08aa5f),LL(0xd7247824,0xf5eb81e2),LL(0x26357344,0xab23fe73),
   LL(0xb37efb0c,0x8d9e1ced),LL(0xd7b33644,0x3b126789),LL(0xa25195f0,0xee7f591f), LL(0xe6380d90,0x5572f635),LL(0x6381422f,0x570f7d77),LL(0xe69759c5,0x44378d7d),
   LL(0xa08babdc,0x1b30b2aa),LL(0x3cc52896,0x9ad9a870),LL(0xd7cb91be,0xd1afe3d1), LL(0x4a5535b4,0x1f838c05),LL(0xcfab1e86,0x86b14aaa),LL(0x6f77d53e,0xe64af8a4),
   LL(0xebdc38c4,0x2028f1f2),LL(0xf31ac53e,0x534fc79a),LL(0x1a0e531f,0x987d5d41), LL(0x76fd27ac,0xf9d7d8c3),LL(0x40a77dbd,0x930da5d5),LL(0x57ef7f4f,0x2c9fae32),
   LL(0xdb6f852a,0x7232dd8f),LL(0xddea21cd,0xcacd16a9),LL(0x21f8ec9e,0xf04c4671), LL(0x4688eec7,0xfec90470),LL(0xba277998,0xd49d9514),LL(0xb4cf1ab4,0xbec829aa),
   LL(0xe594415f,0x6fb54104),LL(0x2c3a2383,0xcd39d8af),LL(0xe44a8388,0xd50673e0), LL(0xf14f72d0,0x55137ed9),LL(0x1abb3c7b,0x4475a031),LL(0x246ece39,0x73e9fa03),
   LL(0xee6222b2,0x7017567a),LL(0x03448302,0x603051f2),LL(0x5aaa55f0,0x8e7a299a), LL(0xc61d2cb5,0x6a43ef05),LL(0xd1923efe,0xc780996b),LL(0x29e3847b,0x71c3dd7d),
   LL(0xfdc2519e,0x34bff2b3),LL(0x4e5b30b6,0x55e13340),LL(0xdf3b31e6,0xf0159d7d), LL(0xffa85b11,0xe041606a),LL(0x437094a8,0xc9438445),LL(0x8b1d800e,0x13df929b),
   LL(0xdf21b623,0x0e2925cb),LL(0x9010bc70,0x24051cf5),LL(0x891c2a68,0xc2176b95), LL(0x01e795a1,0x991bf219),LL(0x30ef7d3d,0xc069d0c2),LL(0x1bc05ca1,0x6021da66),
   LL(0x1e52bf74,0xfcf4cf04),LL(0xd74c9731,0xd7cb6e14),LL(0x7e857c02,0x7628ac80), LL(0x126c5b91,0x0cc391d5),LL(0x7e2d15bd,0x731062b4),LL(0xabdfd049,0x1e87cea9),
   LL(0x363f0203,0x10006c7d),LL(0xf63971b4,0x51ea9151),LL(0x80734bc5,0xd951b96d), LL(0x89a7c10f,0x1d164750),LL(0xcae9c4ae,0x488652d2),LL(0x185669d2,0xb4ca6a36),
   LL(0xbb8f19e9,0x6e84424a),LL(0xc9d19687,0x8b9aa75e),LL(0x0b88d08b,0xc120a35c), LL(0x2a8cf8ec,0xdeeb61da),LL(0x6bebce59,0xfcc8ccf8),LL(0x8de4133a,0x679e7f58),
   LL(0xe3c5a444,0x0f3eb897),LL(0x5751461e,0xce575c28),LL(0xdc20e60f,0xe8d1e5b3), LL(0x7d07536e,0x9dc103d7),LL(0x2f685999,0x86a2a40a),LL(0x47c6f8fe,0x15ae9ff4),
   LL(0xaa9b291a,0x885babc0),LL(0xfedd2942,0xe3f51080),LL(0xb5d39214,0x304ccff4), LL(0x0a3b3582,0xf107db2f),LL(0xea86a325,0xe2fa620c),LL(0x6b20d46f,0xa0b718ec),
   LL(0x72969183,0x9d14762b),LL(0xfce848f5,0x6b2baf0b),LL(0x968f6133,0xe7660ec7), LL(0x8299819f,0xc6ee65e4),LL(0xef1b9b29,0xb57828af),LL(0xa50176ea,0x5b1c99f3),
   LL(0x62f3aae5,0xc35057fa),LL(0x46f3c966,0xa5437e70),LL(0x8864f29f,0x6bc5d22a), LL(0x86323fd9,0x143c464b),LL(0xd1570c61,0x30852173),LL(0xb5b6f3d4,0xc188ba63),
   LL(0xc8406bdc,0x8b26d419),LL(0x3fafc9be,0xc0c95def),LL(0x5d5eb753,0xcd58ca48), LL(0x57e8a8a8,0xb937d969),LL(0x8d184ad0,0x8c18d5f4),LL(0xf5485ef3,0xbd764f21),
   LL(0xecb39885,0xc5483510),LL(0x705df629,0xf0122752),LL(0x93d33824,0xe43d6cca), LL(0x49923c5e,0xe00a9514),LL(0xd8080e76,0x6993aecc),LL(0x633b9fba,0x4dc6d79e),
   LL(0xefcd68d8,0xa33bbb48),LL(0xe68f5543,0xacaebe65),LL(0xa37a2bbd,0x7383a726), LL(0x770555c6,0x5d3e8c9c),LL(0x9b90f1ad,0x86e220c9),LL(0xab26ba0b,0xfc32885e),
   LL(0x89e7a1a1,0x35837dd0),LL(0x08889b69,0xb8c46a1e),LL(0x1957ef47,0xbf3d0f2f), LL(0xf0a25602,0x7b89c2b9),LL(0x0656bcd8,0xab5068da),LL(0xf4298e4b,0x0c5c3a38),
   LL(0xcaf9bcee,0xd20c9cc5),LL(0xbd6c6fd4,0x783987bd),LL(0x93ba43c5,0x696e2d85), LL(0x391285c2,0x356cbc57),LL(0xb533c31e,0x536b15a5),LL(0xc4c4ba41,0xc375dc50),
   LL(0x405fe9b5,0xb961c694),LL(0xd5283673,0xb3983e9c),LL(0x96e95f05,0xd455be9e), LL(0xc75e8014,0xb38412e7),LL(0x96860cdb,0x1033ac0e),LL(0x00b0a103,0x303881b4),
   LL(0xb3b3e63d,0xd1fc3bf7),LL(0xda50174a,0x35ab8665),LL(0x01a402c8,0x2944d50f), LL(0x4d26ddfc,0xba75bddb),LL(0x80efbf29,0x530839a3),LL(0xb93462eb,0xbf8adca3),
   LL(0x84d08e74,0xd534a5d0),LL(0x4efdf525,0x436eb254),LL(0xdca89d2b,0xc653af06), LL(0x41a37013,0x0895e017),LL(0x91294447,0xb9f0b192),LL(0xb7f407e1,0xfd2cb9b0),
   LL(0xe6cc78dc,0xc5323fdc),LL(0x2a74ed72,0x9c863011),LL(0x1aa90d73,0x86d391c8), LL(0x308de318,0xae9c2f27),LL(0xcc2d2cad,0x133f4d65),LL(0x42a244ba,0x0bfd5ed6),
   LL(0xa7843758,0xb5b22ff6),LL(0xd93d7ed0,0xbdd3456d),LL(0xe0432d41,0x4bb279a7), LL(0x14acb55c,0x375da883),LL(0xad01ebbf,0xf9493a0b),LL(0xf0543ccf,0x5063d083),
   LL(0xac19b11f,0x5c97aefb),LL(0x820ab407,0x253042d0),LL(0x550d30a7,0xf6b2204e), LL(0xa702ecbd,0xa091b635),LL(0xb34417cf,0xbd011942),LL(0xd3f92344,0xcb969929),
   LL(0xeca70aa5,0x46c47688),LL(0x383cfca9,0x057cd161),LL(0x7ea5b03d,0xe538b089), LL(0xf15b8704,0x0f3fc549),LL(0xc4be9475,0x0bdcf648),LL(0x40851b7d,0x31fb0690),
   LL(0x7b90be33,0x8dcd148f),LL(0xb5e956ad,0x3720de48),LL(0x2a303f7c,0xa0ce39c4), LL(0x89d0fdb7,0x054206d6),LL(0xb76fa220,0xe9991e88),LL(0xbac4c9ee,0xa0eb60ae),
   LL(0x01065ebf,0x9feaf043),LL(0x23e6a889,0x001f921d),LL(0x526d4d12,0xbe2a6e6a), LL(0x762aaed0,0xae7aa8bd),LL(0xde1e43cb,0xa8aa4f13),LL(0x626356ac,0xea279354),
   LL(0xc65bd80c,0x54727769),LL(0xf135559b,0x0f18555c),LL(0xc8a92ba2,0x3ef9e205), LL(0xf4e4b870,0x32f97cc9),LL(0x9fdfc295,0x764e6c3b),LL(0xa0be0176,0x97f95ca0),
   LL(0x3547707f,0xa25af6ad),LL(0xae94e108,0xc46e1845),LL(0x3a29996f,0x481c7fd4), LL(0x1aac0d60,0x4e9a38c3),LL(0x2d811b3d,0x40172f5e),LL(0x58ba5cdb,0x69750750),
   LL(0xd402cb8c,0x29404a7b),LL(0xc4a0deb6,0x8698ab1c),LL(0xf10f49fa,0xb0336bae), LL(0x4d34533b,0x27cbe364),LL(0x53d57740,0xb22e812c),LL(0x90f4aa01,0x82b80434),
   LL(0x479f2e14,0x1b20606f),LL(0xb817e38d,0x5fa6e4bb),LL(0xcedd7aac,0x860fdf9c), LL(0xb63a00f6,0x66fa3783),LL(0x73c45ad2,0x2719710a),LL(0x64de62a2,0xa3db3619),
   LL(0xf04aa960,0xb2e191c9),LL(0xf57411c2,0xd419a1dd),LL(0xa4a57d93,0xd860a8ea), LL(0xc1f433bb,0x5001230c),LL(0x8ec92a7f,0xf28ce31b),LL(0x9a4d25a6,0x8c582d34),
   LL(0x96219ee4,0xf2826086),LL(0x4b4f0c24,0x0c21e1e2),LL(0x387213d6,0xd1804821), LL(0x1cce49ca,0xfa342c74),LL(0x4341ef6f,0x880ea569),LL(0xe968aa8c,0xa6fdc07f),
   LL(0x91a41ded,0x293a6180),LL(0xe5d87356,0x51412d3b),LL(0xf0b2d509,0x925d4c9e), LL(0xd251fb90,0x80c77ead),LL(0x90e53a2d,0x1d6e3cd3),LL(0xf9a2ab15,0x6e11e4d0),
   LL(0x1d009ea4,0x3535dc7b),LL(0x75c6c6f0,0xb6b48635),LL(0x72e3c50c,0x12ac6480), LL(0x6464430a,0x6d71a720),LL(0xe1dfdda2,0xf553de43),LL(0xcc6c345d,0xa1ee9a4e),
   LL(0x2ac5a8c8,0x46f7bd04),LL(0x063f2739,0x62c11b7e),LL(0x69a36c58,0xaa7522c7), LL(0x06a4df65,0x6a10fbe0),LL(0xdf06bc54,0xd5f96d18),LL(0x38151768,0x545f76b0),
   LL(0x712eb633,0xe9dd6bff),LL(0xf6972761,0x6c1e035a),LL(0x7a4d81a8,0x060bfcaf), LL(0xd2653839,0xa262cba7),LL(0x95a4c747,0x561af30d),LL(0xe33e339a,0xb9a3b8ac),
   LL(0x5f371d0d,0x2fc5342b),LL(0xefbb2027,0xf9654b68),LL(0xf7520a2a,0x14820afb), LL(0x43d0ea24,0xb24c0b99),LL(0xc37df573,0xa1a8898f),LL(0x7bed1103,0xa1c07bb8),
   LL(0xc4d993a3,0x664a8ae7),LL(0xfe44732b,0x1c2e8847),LL(0xb2426ed1,0xdb1e5a22), LL(0x7a45d008,0x2bdd7cb9),LL(0xf3d91e11,0x59477a34),LL(0x4cd742a4,0xf53dcb02),
   LL(0x6c709f3b,0xa28f6d2b),LL(0x9ac0b1d9,0x479b31b7),LL(0xd568d9ff,0xe8237a0f), LL(0x0580975a,0x6895a188),LL(0x7b234cf3,0x450b5fe5),LL(0x70629bf6,0x1b4e0f89),
   LL(0xe5bcec01,0x455ac64e),LL(0xc6bf296c,0xaf051300),LL(0x2306584f,0x119cb4c6), LL(0x7c0dee75,0x3242a0e6),LL(0xae1fd52a,0xd09fce7c),LL(0x2d5cf9ca,0x53608dc2),
   LL(0xc8506617,0xaf6edd22),LL(0x3241e286,0x5ed21b88),LL(0x81282c66,0xf27d3439), LL(0x308d6ca8,0xbd393fac),LL(0x9032cca3,0x5155e229),LL(0x56a04a1c,0x29acd1b2),
   LL(0xf78b6573,0x95c60d2b),LL(0xed576414,0xe53bb875),LL(0x452f8871,0x389200b3), LL(0x5af9dd53,0x69a42e38),LL(0xbf0327d1,0x01e3b18e),LL(0xba8d03df,0x985d811f),
   LL(0x559c7bda,0x0cfef9cf),LL(0x5bb4cd8c,0x5096e369),LL(0xc8011afa,0x8d16d0e2), LL(0xf25b0ff3,0xa816a8be),LL(0x2f161a22,0x7edbd05e),LL(0x8ba1f83c,0x1ae570e5),
   LL(0x690bc4ad,0x7c0084c0),LL(0xad2fd216,0xe0a5451c),LL(0xb130c417,0xb4926a84), LL(0x9309e50a,0x39e4ec36),LL(0x5c1bc674,0x094fb1e0),LL(0x67878735,0x19572f99),
   LL(0xd011ed00,0x8f62726f),LL(0x0793cd2b,0x430d8c6c),LL(0xb8f172e8,0x26226dc9), LL(0x11935136,0x550ff9fa),LL(0x8c9798d3,0xb415bbd9),LL(0x83b3b2c1,0xb136921d),
   LL(0x669e0918,0x940bce84),LL(0x298448d1,0xca050170),LL(0xa8e4042d,0x1fa3041a), LL(0x1e2abce5,0x0f73a3aa),LL(0x448c4fcf,0xe5196a8a),LL(0x2774a5b1,0xfe25d513),
   LL(0x7192ab9c,0x6799b1b3),LL(0x7f02ea02,0xfd1cd774),LL(0x01b1e71f,0x1b4e5cd1), LL(0x13c4b9f9,0xb09c08b2),LL(0x87870c13,0x6be2f294),LL(0x3c2d5610,0xb74d7045),
},
/* digit=15 base_pwr=2^105 */
{
   LL(0x7c1f0b12,0x950114b6),LL(0xd9406636,0x4862f5b4),LL(0xe79e2ad1,0x4b94f772), LL(0x2162024c,0x283df03b),LL(0x1eae9865,0x730f9f5b),LL(0x1260cbf5,0x70b3f304),
   LL(0xe16c6f09,0x850addcf),LL(0x5592d8d7,0x36a07642),LL(0x74af4f32,0xcdbf2690), LL(0xf354ed72,0xf452488f),LL(0xa99eb244,0x8ceb8c62),LL(0x92a47252,0x471774a0),
   LL(0xf938b030,0xc7285307),LL(0x51cc80fd,0x81689ad0),LL(0x483aeb1a,0xbb8e3d60), LL(0x8286229d,0x6dec5453),LL(0x0cc83f74,0x3edd27c2),LL(0x152e37a2,0x1cd6ed5f),
   LL(0xd377bce5,0x537244ab),LL(0x92bbe196,0x333760b5),LL(0xd7b6c795,0x373f1920), LL(0xb99b6824,0xd28c3c25),LL(0x501ea35f,0x14e96723),LL(0x8bebdafd,0xb3c3f757),
   LL(0xd82a0a63,0x4a680526),LL(0x0558bb75,0xeeb46a35),LL(0x7589e42c,0xc27b9513), LL(0x59267a86,0xaf5a87ec),LL(0x5c4b67d0,0xbbf1e057),LL(0x09598a27,0xe37e471c),
   LL(0x2a71504c,0x894af9cd),LL(0xd1e8512f,0x38e0d48f),LL(0xfc2cdbf4,0x96db568c), LL(0xeac2f293,0x911f7b8d),LL(0x5c965527,0x5821420d),LL(0x078559b0,0x832e957e),
   LL(0xfbf84029,0xff114368),LL(0xd21a72b0,0xd8dd9e2d),LL(0x057c15f6,0x4ba812a4), LL(0x64fa748e,0x976cf8b5),LL(0x73812963,0x1801ab26),LL(0x394d2cf3,0x4a748d37),
   LL(0xb8ffb76d,0x6b23479d),LL(0x898a63b9,0x401bf60b),LL(0xdc64023e,0xee58a7ae), LL(0x6c70a3b4,0xd288cadf),LL(0x5cf80831,0xc1da3aea),LL(0x5ee941bf,0x941b411d),
   LL(0x1029b199,0xc4cccd83),LL(0x6d56502f,0x683863df),LL(0x43f9e9f5,0xe362783b), LL(0xa34a9ef2,0xb224d946),LL(0x72b438e0,0xccabcf49),LL(0xc4af3f0f,0x78d92dda),
   LL(0xceee3620,0x123b5621),LL(0x90e8a27d,0x513add4e),LL(0xe45a02ac,0xaf127aef), LL(0xc0d7ba62,0x21a78e17),LL(0xf98d0ce5,0xcf09e816),LL(0xae64e8c5,0xc28cab36),
   LL(0xef7a2cc1,0x0912071a),LL(0x3b451d17,0xd3f2bc6c),LL(0x35ea478f,0xf3796f91), LL(0xc9c756e8,0x9b5e64d7),LL(0x2d58e1bc,0x7ed6f7fe),LL(0xeae6b6de,0x503d9af2),
   LL(0x5e314295,0x57cc1308),LL(0x89230510,0x74c77119),LL(0x9968ac6f,0xa314891f), LL(0xf033b995,0x8dd4df22),LL(0xe0d71497,0x7ffe0a44),LL(0xac968d35,0x5c5f54f3),
   LL(0x406b0bca,0x246a971a),LL(0xd805a357,0x90eedbcb),LL(0x5c690ec1,0xa6dbc124), LL(0x7bc65cbe,0xbe369f3d),LL(0x96065417,0xe0ff616a),LL(0x337f567d,0x3654f681),
   LL(0x9bf8c833,0xffa30743),LL(0xafe911c4,0x6bd11b77),LL(0x6762a958,0x30eee614), LL(0xa9e63f35,0xfba9fbc0),LL(0x332389bf,0x70edc028),LL(0xb9798769,0xb5d35ca9),
   LL(0xb8c620c1,0xcde63d0f),LL(0x06942335,0xcdeaad60),LL(0xbdef1579,0xefb74b79), LL(0xd8b128a9,0x13dd5771),LL(0x56343492,0x00e9fe16),LL(0xd6baa45a,0xf4ffba1d),
   LL(0x19d575fd,0xfdb1ece3),LL(0x61ad3345,0x231c53ac),LL(0x523e013e,0x3f3df394), LL(0x71c3b97d,0x254a291f),LL(0x396eb0c1,0x645a1aea),LL(0x0282f254,0x680b79e1),
   LL(0xf3ef6b21,0xcededfad),LL(0x577a8e6e,0xa57437f5),LL(0xf5f051f3,0x413575b4), LL(0x4d1ca0ed,0xd88fe7f3),LL(0xdcceffdf,0x1849d754),LL(0x141422af,0x40e7733f),
   LL(0x00721b87,0xcd5a690a),LL(0x0b2801ae,0xf0e84ac9),LL(0x17c73f20,0xaf0765ff), LL(0x33595f95,0x976d23d8),LL(0x72c8f457,0xdece9d39),LL(0x6604f618,0x3dddbfa2),
   LL(0x2bb19559,0xef6c219c),LL(0xd57790b8,0xfbf7ac31),LL(0xc8f6c246,0x676751ad), LL(0x4c3b7ff6,0x0f4384d0),LL(0x39ee5f11,0x040a4f5a),LL(0x698f2f7f,0x30c58138),
   LL(0x0d1a6b11,0x097f5ccf),LL(0xf9b7701b,0xf6cdd1f6),LL(0x4786ad28,0xb965d0ce), LL(0x70cd6a22,0xadd506d0),LL(0xd6299b81,0x9c4667a5),LL(0x1c032341,0x64056720),
   LL(0xeb125185,0xfd01d07b),LL(0x48235596,0xbec6636b),LL(0x293de94e,0x9017825d), LL(0xe499b69d,0x888e08c6),LL(0x3b6a7ced,0x4f4a6117),LL(0x58f8d42c,0xd6d2a6fc),
   LL(0x79062ff2,0x66a2ad0a),LL(0x9bc01a36,0xf187ed18),LL(0xb3f3d332,0x4fce4f5a), LL(0x38533148,0x903af8c9),LL(0xd058ab8e,0x34ff5da6),LL(0x6b2b6b4d,0x8fc66e30),
   LL(0xd4eb310c,0x7e0843bb),LL(0xbefbb992,0xcd48672b),LL(0x75c258ee,0x839c6b0b), LL(0xd70a705e,0x11733171),LL(0x15042807,0x92fb3004),LL(0x9ca0a17a,0xe091def9),
   LL(0xaa917028,0x48c33a95),LL(0xb87bb667,0xdef22091),LL(0x106fa5c7,0xb6223052), LL(0x6934d6e3,0x12491351),LL(0xbce2aa51,0x55c0b4d5),LL(0xb70985ca,0x734ec89c),
   LL(0xbc8fd35d,0x000e0cd7),LL(0xb50502f2,0x05b488af),LL(0x1252793c,0xb56958a5), LL(0x8a310520,0x4e8c9cc1),LL(0x7f1db7bd,0x0a6e45d6),LL(0x5ecdeb31,0x72d3eb5a),
   LL(0x85f7d20f,0x1ac14fe9),LL(0xb430d3cc,0x0e020f68),LL(0x409668ed,0x273e515f), LL(0x78e0031b,0xbf622f1b),LL(0x9bf7fcd7,0x5aa6dd5e),LL(0xb6b977f5,0x30573193),
   LL(0x275d217e,0x7c4023a7),LL(0xcca3f6f1,0xac6a6b67),LL(0xf0fcc9b9,0x176bacc3), LL(0xb0858d73,0xb280416f),LL(0x6fec46e4,0xe48e3c6d),LL(0xf3235cfd,0x9aeb3a09),
   LL(0xa486a8eb,0x33c068c6),LL(0x5def6704,0xc60c21cb),LL(0xbefdd067,0xff4ac34f), LL(0xe174348b,0x4f1427e6),LL(0x9f27fd60,0xc80afd96),LL(0x35955ccc,0x1799ed29),
   LL(0xddebd485,0xdbf71628),LL(0x4b8207f2,0x13897ccd),LL(0xfd8365e9,0xe12c2822), LL(0xb003566c,0x705fe026),LL(0xf12863e8,0x8a97247d),LL(0x1748fc9e,0x698a3477),
   LL(0x0669c4d8,0x90917d8d),LL(0x80746165,0xc65e4aa5),LL(0x12c0f44c,0x96763d21), LL(0x97318a9b,0x00a7db8f),LL(0x5fd1a0e9,0x29079037),LL(0xa9b6ec4f,0x2f4f1c47),
   LL(0x0bd63292,0xd861004e),LL(0xca40a1da,0x0176d94c),LL(0x59fb9a7d,0x5e91abe8), LL(0xf6d4f1ff,0xa881a8f5),LL(0xa24ce9a9,0xded77256),LL(0xc492f4f1,0xacf4db67),
   LL(0x391ca168,0x7146928e),LL(0x11e6bfe4,0x11981f6f),LL(0x32f997ad,0xfce0f814), LL(0xe85cd202,0x7cf18ed9),LL(0x03c315e4,0xa94f2b6d),LL(0xb5f8a4d2,0xe2a93315),
   LL(0x2fab5a21,0xcd259208),LL(0x928dcc30,0x3d24f816),LL(0x9d66a41a,0x779e1c36), LL(0xc2e41af4,0xa8248120),LL(0x06fc8b0e,0x243a6bbb),LL(0xed8d12b1,0xb9de9d41),
   LL(0x30f64b0e,0x23ff17d9),LL(0x8f083ea7,0x1eab5984),LL(0x554ae817,0x9965f2c0), LL(0xf4d8956a,0xb0a399b6),LL(0x0ebf134d,0x315694aa),LL(0x7fde4077,0x53567f64),
   LL(0x5c71762b,0xfdfeb029),LL(0x1eb1144d,0x968d0bd7),LL(0x1f08b2a6,0x171297f4), LL(0x897a2666,0x7944101d),LL(0xda77b1a1,0xf4936746),LL(0x44547953,0x0e752c75),
   LL(0x492b2fa0,0x69d5b44f),LL(0x5d483119,0xda5f6651),LL(0xac522988,0x40814f2b), LL(0x4e1f60df,0x344aeee8),LL(0x5ff6f053,0x2ddba5e0),LL(0xa4008621,0x4453d9ee),
   LL(0x179e2967,0x37ab4aec),LL(0xba87af44,0xe91479f7),LL(0x0b55c02f,0x4b292d1e), LL(0xb69ad1a8,0x9deeddd2),LL(0xe91e913c,0xb610d598),LL(0x912b8e9d,0xa6b91331),
   LL(0x20cdff63,0xaa819620),LL(0xaa6d0ed4,0x8d70fbba),LL(0xcb318c6a,0x5c951edc), LL(0x57b1fd8f,0xa4cc6573),LL(0xe00bbceb,0xd7a23636),LL(0x8d179500,0x33bb3513),
   LL(0xe0168751,0x86da6dfe),LL(0x66604b83,0x940ba0c2),LL(0x22e9e186,0x15a8aa02), LL(0x65a6f19d,0xe5b39c58),LL(0x32f441b2,0xf591dc15),LL(0x9d3a2de0,0xb85cad59),
   LL(0xfb422575,0x35abc7e2),LL(0x4b8acd9c,0x517b6718),LL(0x2ee6263b,0x3e303015), LL(0x7393f470,0x51c8c440),LL(0x974d3dad,0x032cce2d),LL(0x4c104799,0x68e57ee5),
   LL(0xc0514589,0xb95807db),LL(0x0402a3fb,0x09a79c01),LL(0xd3fdd054,0x92bf681e), LL(0x393adbfc,0x9f2f00f4),LL(0x1ca244f6,0x13789212),LL(0x00d3a9db,0xe2f45a02),
   LL(0x207a53c9,0x29ea3e07),LL(0x163549d5,0x61213e9a),LL(0x4656d770,0xe0b025d0), LL(0xbd0d85da,0x050f7501),LL(0xe3b43784,0xe95027f4),LL(0xf1a6b6b5,0xca3907cc),
   LL(0x4a7a00be,0x670062e7),LL(0xace7ec67,0x9ea4759e),LL(0xc1d92037,0xf5d29365), LL(0x9c39e2de,0x6d421c87),LL(0x1593c52d,0xb3e3d2c7),LL(0xcd74d083,0x6733eb50),
   LL(0x5ad1d924,0x30029e9b),LL(0x84925a40,0xc15c63d1),LL(0x17e5f0bb,0xb2c098b9), LL(0xda6f02f1,0x2646d78b),LL(0xb8ae8095,0x1f8b74a3),LL(0xb03283d5,0x2bfdcbee),
   LL(0xb7d7ef7f,0xc78a8e01),LL(0x574b90f6,0x7e0a26ab),LL(0x26861a59,0x9be1c8b4), LL(0xc787cf5d,0x5dddc986),LL(0x217b8936,0x77e06114),LL(0x94d79885,0x0b58deb0),
   LL(0x6ef1f554,0xb89d079b),LL(0xf1deef2a,0x87ee6d78),LL(0x7a7b9098,0x8985bc42), LL(0x165957c3,0x0843395d),LL(0xccf6daf8,0x11293e7b),LL(0xfce740f4,0xb2527ca8),
   LL(0x12fbd080,0x54f19e66),LL(0x4aed0e22,0x82f0835d),LL(0xb56935f6,0x5f73c822), LL(0x5b0f4a59,0x4b255bef),LL(0x6b23634c,0xc2542ee4),LL(0x858eb55c,0x41a4fe31),
   LL(0x70d71c90,0x5c19755b),LL(0xdee1b0d6,0xb15b8f0d),LL(0xeab4b7ef,0x07e40808), LL(0x6faa90c6,0x9f4f2665),LL(0x9fcb51fb,0x9e72096a),LL(0x541dbffb,0x1660e2f9),
   LL(0x9aff8d55,0xfb1cf7e0),LL(0x7a1815da,0x7276df1a),LL(0x913d43ca,0x604cd6bf), LL(0xe0227dec,0x4f016034),LL(0x0e990a9e,0xb70eb020),LL(0x83b0d458,0x9db0e5f0),
   LL(0x4bda5c36,0x137d2ba6),LL(0xc9d84ed7,0xf2ad5444),LL(0xb4878d48,0x37ba4976), LL(0x0c6237c0,0xe3518c97),LL(0xb5e9c50f,0x32e71c0a),LL(0xa325250d,0xfa2c1d25),
   LL(0x80be6431,0xfa8cf7f2),LL(0xe5a2035e,0x74e8f698),LL(0x03eff799,0xa032a4e4), LL(0xc53ac08f,0xbdfd91ab),LL(0xebf217bc,0xe17b20eb),LL(0xdf84edd2,0x40de3e73),
   LL(0xdf81c5d3,0x4110f517),LL(0x53f20019,0xbefb623d),LL(0xf346f4d3,0x67061890), LL(0x26ff57dc,0x096120d3),LL(0xccb068a2,0x00b49cf6),LL(0xf00d1e3e,0x613293d4),
   LL(0x467e8604,0x248b65d5),LL(0x3ab4a1b3,0xcd983257),LL(0x1e96833c,0x1dc552b2), LL(0x89592574,0x851b4255),LL(0x1da99f23,0x62d85452),LL(0x401ba0d8,0x4cdbe71f),
   LL(0x9760901c,0xdd6660af),LL(0xb8ad97c4,0x503a92d0),LL(0x6fbd51d5,0xdd313e4e), LL(0x6e919a43,0x441e5529),LL(0x65e9a71d,0x8613c8d8),LL(0x3d2336b4,0xc0c8b396),
   LL(0x7b97021e,0x811cc3a9),LL(0xf50aa806,0x6d1a1d14),LL(0xf7f8f8f8,0x85a4588e), LL(0xe67b610a,0x23ba0ee5),LL(0xe3f78d9b,0x77df65eb),LL(0x99a02de1,0x0295f1aa),
   LL(0xae21de63,0xd3284a43),LL(0x6e23bdc6,0xa976e9a5),LL(0xa757b390,0xcc8cfb98), LL(0xa4697fde,0x5e6a1ae0),LL(0x5aafbacc,0x5c62485d),LL(0x27e4f7d3,0x804597be),
   LL(0x53bc971a,0xea91a442),LL(0x436f90ba,0x41484d9b),LL(0xd87d03a9,0x21b57dab), LL(0x8f1888f0,0x935c940c),LL(0x4c82401f,0xe1c2a3eb),LL(0x7847c8a6,0xf42fb0f8),
   LL(0xb4347b62,0xa649770f),LL(0xb29590ef,0x3bf77343),LL(0x6a205e84,0x92b6e5a1), LL(0x63cac357,0x8c973844),LL(0x7e8ab06a,0x058f0f22),LL(0x58d62ed7,0xaddca6e4),
   LL(0x1160ca95,0xf4b969c6),LL(0x0d73f647,0x60e6b905),LL(0x9b98ce7e,0xd8413d40), LL(0x9ec67398,0xebd6924f),LL(0x628424b1,0xe4c95b85),LL(0xc7f2b86c,0x157759fb),
   LL(0x195680a9,0xd6f1e165),LL(0x965d62c7,0xd433c1dd),LL(0xd3dd177d,0x0890fc5d), LL(0x8c44dbc2,0x3e0f43d4),LL(0x9cb07f9d,0xaa0c14ed),LL(0x6ff8b939,0x6d7feea2),
   LL(0xc17b0751,0xa1c5b89f),LL(0x46164d7b,0xf47724f4),LL(0x60ec3b26,0x9b1dd929), LL(0xcb50b27b,0x6c3c835c),LL(0xf86980fd,0x259fde39),LL(0xde92e229,0x84e73227),
   LL(0xd05a1ec4,0x1f1e9817),LL(0xe6e48ece,0x2692c83d),LL(0xf4903352,0xa2c5bd53), LL(0xaf3eb235,0xa1f9c3ff),LL(0x71158432,0x52c141c2),LL(0x16d971ea,0xeddd3c85),
   LL(0x3724189a,0x7554e3b9),LL(0x2e63e62b,0x5eea6e3b),LL(0xe23434b0,0xe28ad863), LL(0x5f932bc7,0x6016f1e8),LL(0x7d76d1fd,0x1240c223),LL(0x5124123a,0x2ce14c4f),
   LL(0x6d50fbd1,0x6c89d0a8),LL(0x31779189,0x4a38b313),LL(0x8d8124b8,0xa1718ddc), LL(0x0e02b2d0,0x58f48b51),LL(0x6114c7e3,0xd38936a6),LL(0x1b70ec0a,0xb5e924e8),
},
/* digit=16 base_pwr=2^112 */
{
   LL(0x1f47a973,0xcf447109),LL(0x627fd943,0x24c191c4),LL(0x32c6da19,0x4f5f134d), LL(0x9d08667d,0x1b92e5f9),LL(0x1d9684b0,0xe6b9a4e8),LL(0x553bcb2a,0xd60fdeba),
   LL(0x46cc2ad3,0x840f76a1),LL(0x6235d7a8,0x63b85026),LL(0xb2ef5ad3,0x5616a569), LL(0xc3f03ea7,0xa056663a),LL(0x38ce234b,0xd93ddef0),LL(0x57a8650d,0x0f729e40),
   LL(0x360595c1,0x1b437776),LL(0x080dc311,0x6aa3a75d),LL(0x11147932,0xd1288177), LL(0x8a241c89,0x8c953ffc),LL(0x7de54cc4,0x7de8a558),LL(0xa5a3ed5a,0xa8dc2b2f),
   LL(0x084ee344,0xf6b53652),LL(0x4048d28e,0xb51a92cb),LL(0xd688ff26,0x58fb44be), LL(0x9247a804,0x136a7d51),LL(0x3ef5f22c,0xf2176fba),LL(0xa999af5d,0xb4a457be),
   LL(0xf23f79c8,0xb4cf845c),LL(0x1ad85366,0x8233b3b2),LL(0x78e68d8c,0x4efef3d6), LL(0x22c7941c,0x17b1e851),LL(0x761dde7d,0x40a27661),LL(0xcbd06300,0xac29e1a1),
   LL(0x0ced7008,0xff48c690),LL(0xebfce074,0xbed07115),LL(0x27aa6e6f,0x4ac10aa5), LL(0x5e777e33,0x7652ebb3),LL(0x4dd30e6c,0x255c16ab),LL(0x9159219a,0xc0e1b073),
   LL(0xec32e62a,0x6d2fc306),LL(0xe70096e4,0x426dd67b),LL(0x70d9383d,0x994877cc), LL(0xd011ca1f,0x8bc75d3a),LL(0xaa1362ae,0xcb18fa82),LL(0x2c292316,0x54fb1473),
   LL(0x8c720bd1,0x45aa7854),LL(0xc5257a6e,0xceba0e5d),LL(0x796e4320,0x0e924974), LL(0x866fc34b,0xf2ce13c7),LL(0x6a58601d,0xbe8606f3),LL(0xbf56d38c,0xa25d6005),
   LL(0xbdbaa117,0x9432504c),LL(0xba422130,0x4dfc9c38),LL(0x500ef320,0x32d66ac1), LL(0x0679d51e,0x1a38a920),LL(0x93a9c6f2,0x3e2a3c53),LL(0x0a59702f,0xe0b71451),
   LL(0x4cefcb03,0x04e4087a),LL(0x4ab1d062,0x68686886),LL(0xdd24b69f,0x67054513), LL(0xa89e5edb,0x21d5ff58),LL(0xd8e0f856,0x9412a213),LL(0x3faa0ffa,0x1a05bd39),
   LL(0x14033a58,0xee2aec92),LL(0x5161890c,0x38e267ef),LL(0x8c51e740,0x2f0ae797), LL(0xea57871c,0x6d2143ac),LL(0xc5475c3e,0xef3ce79b),LL(0x1fa55fd9,0xdd9cfa6a),
   LL(0x412965e9,0x6ae0ef90),LL(0xe9188107,0x2430c1f0),LL(0x42b98992,0x9682e74d), LL(0xb638ccb1,0x3094b342),LL(0x16948a40,0x6aa863ef),LL(0x60f1cca5,0x2ef984e5),
   LL(0x694a80f4,0x4980d35b),LL(0x2c3d3326,0x4dbbe36d),LL(0x1c899dfb,0x1c719190), LL(0xe9938032,0x96febca8),LL(0xa63775ac,0x7b17130d),LL(0xd12cfb2c,0x17e68ea4),
   LL(0xfde8a210,0xf111a32f),LL(0x83b56b17,0xc3d61a7e),LL(0x22b86813,0x48125662), LL(0x410c959f,0x1ddf2f99),LL(0x68591a75,0x886a1893),LL(0xa21da760,0x92dc4bbf),
   LL(0xb9b71c82,0xa2d0b7f2),LL(0xe59c5616,0x206fb318),LL(0xa9f1f448,0x01112ace), LL(0xd40341f9,0xbc6ee063),LL(0x90590590,0xe2c50a8d),LL(0x795c4ef5,0x3aa25a83),
   LL(0xf5016294,0x582102f1),LL(0x8ecaf2bb,0x6c6a8cc5),LL(0x9e733710,0x7887a218), LL(0x82da339d,0x18ea2c41),LL(0xef8def66,0x428afd13),LL(0x5f923e0a,0x171ec875),
   LL(0x98756704,0xd9233e20),LL(0x86a8b3bb,0x5b3a223a),LL(0x216d0c55,0x844ef9e9), LL(0xb7f4d4b6,0xb8a1a350),LL(0x64bb191f,0xf70cb37c),LL(0xefb8ae25,0x7c42a6c0),
   LL(0xc8074a8b,0xf789197b),LL(0xbd20de1c,0xf7984f1b),LL(0x50eb451d,0xc0b1f821), LL(0xa88538d0,0xeb36fa99),LL(0xaa22b727,0x87c78bb1),LL(0x3644f1eb,0xfe697599),
   LL(0x6fb08720,0x775050a3),LL(0x7c92825b,0x6547216d),LL(0x4a525878,0xe4c237f8), LL(0x63d4f3fd,0x584a8f77),LL(0x80b593cd,0xfefbf7fb),LL(0x89778016,0x8f52fc2a),
   LL(0x275beb93,0xdf1d7c82),LL(0x7512f5a7,0x4e3e1e22),LL(0x833674bc,0x0a8c3404), LL(0xabc4edcb,0x92c482d9),LL(0x9fc05328,0x90d5620d),LL(0x72e82254,0x49facc26),
   LL(0x2ecb1fc1,0xc5acdaaf),LL(0x6e74dd57,0x2a6d3302),LL(0xda6dfd6a,0x7ac86ef8), LL(0x4074218a,0x24edf1a2),LL(0xf9162dd2,0x9010034e),LL(0xae5e58fb,0x035ed18c),
   LL(0x55f3fe0b,0xa961e6d8),LL(0x46a86b10,0xcd5156e2),LL(0xc2988880,0xc4fc85c9), LL(0x5e00c084,0x68c21329),LL(0x9af42336,0x505ffd96),LL(0x2d4346e2,0x2d3d12ff),
   LL(0x81aaf463,0x19a0b708),LL(0xe4d6eec0,0xc956aa02),LL(0x897c0465,0xa8dba669), LL(0xbee2b8a9,0x5c8a3e27),LL(0x80c032bf,0x8ae8552d),LL(0x482aaf27,0xe7526fc1),
   LL(0xc12240f9,0x4767cb3f),LL(0xfe66cca2,0xa31fe2bb),LL(0x0c03119a,0x15f6f333), LL(0x9283ee27,0x67a68e9a),LL(0x53e4d988,0xe2d7d349),LL(0x0150cf1a,0xaf919ee4),
   LL(0x4bc9e09b,0xcd0b84c0),LL(0x8f7286bb,0x0e18d19f),LL(0x5aa9039c,0xf51f3b13), LL(0xb25604be,0xe0bf876a),LL(0x60e8118a,0xaf35db67),LL(0x6dca47f8,0xd6f2ebd8),
   LL(0xe4942689,0xdb226347),LL(0xd061e06d,0xda1e15d6),LL(0x3d043dec,0x2a99ad74), LL(0x7a5cac1c,0x78a33189),LL(0xd8bfd02e,0xb3b6c52b),LL(0x7aaec4f4,0xcae776f5),
   LL(0xe13ee1b0,0x00bd80ca),LL(0x5570567c,0x678837f3),LL(0xe22c402b,0x379e4e13), LL(0xb68e8230,0x7f51fdac),LL(0x5d23aa66,0xc76acd65),LL(0x2caaa8e7,0x4b8a0d89),
   LL(0xb2c47709,0xeeb913e9),LL(0x01b76a21,0xc2683d5a),LL(0xec8c8ce7,0x58e53311), LL(0x0654cd8a,0x0a179454),LL(0x5999ac6c,0x011152f5),LL(0xc8b4fa8f,0x15eb6f48),
   LL(0xfdb6b880,0xe5363d68),LL(0x3ad457bf,0xef83e507),LL(0x6fd08edd,0x63874010), LL(0x85975c5f,0x64b77762),LL(0xbed19eac,0x11cf13e2),LL(0x19f07470,0x823d8c2e),
   LL(0xdc4e8957,0xd091f448),LL(0xcf0950b8,0x9825fdfb),LL(0x0d1e5204,0x1e297e4b), LL(0x499cbea9,0xab53b2a8),LL(0x4917c840,0xca4a1f80),LL(0x2a785cab,0xe6506492),
   LL(0x0f900918,0x9c532f97),LL(0xcb00118b,0xfc209bf8),LL(0x6290d47a,0xdd4a28aa), LL(0xcb4c67cc,0x8a7aa0f1),LL(0x3a3a7acc,0x1d7cc682),LL(0x504f910f,0x085b0335),
   LL(0xc3978c89,0x2a3e44bc),LL(0x3724d81a,0xbbc69641),LL(0x8e42053e,0x8cd0bcf6), LL(0xebd9205c,0x9a84e877),LL(0x8bbf687b,0x6c153f89),LL(0xdee7732a,0x730b14c7),
   LL(0x3e54212a,0x71f21e01),LL(0x2044a858,0x2fce6291),LL(0xd5b7e3b0,0xb3b630f4), LL(0x59a7edb1,0x42ceac09),LL(0xbddf2781,0x6a40528a),LL(0x8954daf0,0x68b67f7b),
   LL(0xd4cc9c00,0x6f07450a),LL(0x243d9b91,0xac4f418f),LL(0xb3930836,0x862c4d7c), LL(0xb829804c,0xad9509d4),LL(0x35d6a88a,0x5448355a),LL(0xa02a8977,0xb1fd6cf6),
   LL(0x12fda311,0x926dc0b6),LL(0xa713c6b6,0x3858aa1e),LL(0xb59d713b,0x32532f12), LL(0x0f81bc84,0xe0a3bd86),LL(0x34b57212,0x28bfc374),LL(0x59e6b4f4,0xc1d92564),
   LL(0xd1adc5f0,0x749961eb),LL(0x6897fc6f,0x070b3de4),LL(0xb71a0ade,0x0b95a35d), LL(0x4c6cf54f,0x993a0e3f),LL(0x07030f4a,0x57dd5c90),LL(0x12298384,0x191f3d38),
   LL(0xfe98a51d,0xd30028ba),LL(0xf0f77aef,0x536fceb1),LL(0x15f7d4d9,0x58a8d8cd), LL(0xafa42baa,0x498dd178),LL(0x8ad8d1db,0x107ae753),LL(0xf4a6e0d7,0x79957dae),
   LL(0x098aa8a4,0xdefb4a06),LL(0x5a536615,0xd45f01be),LL(0x2175624e,0xdfb9ab8f), LL(0x432b8122,0xb7f76e8a),LL(0x66ba6ee3,0x838ee280),LL(0x275d1fcf,0xe6911bea),
   LL(0xe82bfc51,0x5ddd5dc2),LL(0x126dd7dc,0x36994ad7),LL(0x520c3a43,0xa3be5780), LL(0x6b94a716,0x093dec86),LL(0x9ba657b4,0x5189af95),LL(0x2c605098,0x188fb2ee),
   LL(0xb76392f7,0x14886c31),LL(0xdecb5423,0x304dd2c9),LL(0x0214664e,0x853a2344), LL(0xca815581,0xa150c793),LL(0x5edee2f8,0x1334dabe),LL(0x58d437d1,0xfa263569),
   LL(0x31b3f896,0x348804e5),LL(0x4bc1df4d,0xf9778c68),LL(0x5ba2cff5,0x9174fb72), LL(0x3156fc57,0xdae23f11),LL(0x0eaf5c36,0x146fe941),LL(0x6954d2fa,0xa756199e),
   LL(0xfc879267,0xd875fdac),LL(0xeac024c1,0x504843c3),LL(0x9abdbb10,0x2880c735), LL(0x3b9acd44,0x45f8969d),LL(0x6261a810,0x86744967),LL(0xda56c33d,0x2889aa10),
   LL(0xcca4b465,0x4cd3e586),LL(0x0438f6dc,0xb9858004),LL(0xff4ba237,0x5ecb0167), LL(0x2de8c238,0xbf2dfbac),LL(0x86067a4e,0xd1a1f341),LL(0x514110d5,0x6e001905),
   LL(0x32515872,0xcc0b6188),LL(0x9db67e8d,0x8a5a4765),LL(0x4032cd76,0xfc5572bd), LL(0x4ced4c98,0xa8148877),LL(0x7e29acf7,0x3a3f4af1),LL(0x4878a56e,0x19e7ba4b),
   LL(0x6ffe2a99,0x7d54fa14),LL(0x602fe247,0xd9f414c7),LL(0x284f46df,0x31abd712), LL(0xe6a9ef58,0xf55a66ce),LL(0xb2d9463d,0x8d76dacb),LL(0xc14ace92,0xac7aebc3),
   LL(0x7fc941bf,0xa7b46744),LL(0x3f70d808,0x0c420029),LL(0x177bab8e,0x3da5d17e), LL(0xb3a67013,0x3ce9a56e),LL(0x34c0d947,0x4fc18745),LL(0x4571d7d0,0x1295a40c),
   LL(0x59a9c46e,0x55635c3d),LL(0xbfe9682c,0x425f4119),LL(0x47824565,0xd6486529), LL(0x8a96f0bc,0x741c7a34),LL(0x382a3617,0x9cb36316),LL(0x7e0bb8c3,0xe8704d9b),
   LL(0x261438fc,0xba5fc6e5),LL(0x304dce0b,0xd455c9e0),LL(0x5d776157,0x2c5cd9cc), LL(0x58000464,0x2ad5cc3e),LL(0x9efd5253,0x88b007d4),LL(0x031f40e8,0xc5024b19),
   LL(0x89f85ced,0x43e291bc),LL(0x2565e514,0x704c3d56),LL(0x6f19bf54,0x401941da), LL(0xc42b17b9,0x407ffe45),LL(0xee6116a1,0x68753698),LL(0x0f1e7e3c,0xf69c5def),
   LL(0x59d50565,0x7e95369e),LL(0xc3f575f3,0x089278bf),LL(0xd7d4ed14,0xfbf02dc1), LL(0xc063b9b3,0xe98848a5),LL(0xb3d74962,0x49058917),LL(0x937ac3e7,0xbaee0b60),
   LL(0x9517cd36,0xd3f4711d),LL(0x5ab34ba6,0x72055716),LL(0x8054f1b6,0x55781094), LL(0x854d305f,0xe3ad29bc),LL(0xa21b9037,0xc6f2e79f),LL(0x0478e475,0xa01b493a),
   LL(0xf078fc36,0x80124211),LL(0x203d5414,0x29e04253),LL(0x1a40dfb6,0x53737763), LL(0x19d3137d,0x698334c3),LL(0x22b7e891,0x02c4658d),LL(0x285de1ab,0xe33063e7),
   LL(0xe76c0ac6,0x56eb9f2e),LL(0x90aef3e0,0x3898b635),LL(0x3e409c12,0x016a7234), LL(0x32cd68c5,0x488d72fe),LL(0x809dd2bd,0xc9cfcd28),LL(0x0b99c1d2,0xec284a02),
   LL(0xaf16bcb5,0x7cc64eaf),LL(0xdb4ffe42,0x6a629feb),LL(0x2d775fbd,0xdf4a9495), LL(0x41ee0c37,0xde89a2db),LL(0x2b89d6f5,0x4bcb9b55),LL(0xaa29912c,0xabeefb50),
   LL(0xde0b2010,0x026c18f7),LL(0x43b3d1d2,0x39495f8b),LL(0x30288eb8,0x63546dd7), LL(0x68b69732,0x0a49e9b2),LL(0x0ffd4a14,0x9ef54b7e),LL(0xd1fd4c67,0x834add8b),
   LL(0x95444724,0xaf5a35ac),LL(0x98374f56,0xa4f56e13),LL(0xbad358c4,0x779136c5), LL(0x0b77d7db,0xda98e86d),LL(0xd83f4464,0x664c97a2),LL(0xa95543ed,0x9e4c6820),
   LL(0xa63fde03,0x4189250d),LL(0x252242a2,0xfdf0c2da),LL(0x04072ae9,0xc46d8769), LL(0x6679d6e8,0xac1026c9),LL(0x30b3ab9e,0x2932cf08),LL(0x52f5c3ef,0x543f55db),
   LL(0x14b85fd4,0xbea36af7),LL(0x5bcce101,0x32771ea0),LL(0x798ec0fd,0xd9a59a8c), LL(0x4d437c35,0x165e1bfe),LL(0xedfb6d49,0xb55f1e8b),LL(0xec21ec21,0x362e3826),
   LL(0xd4c60ba2,0x3aebfb11),LL(0xc7e3f014,0x77426fe0),LL(0x8a27fdc3,0xe457c3e5), LL(0x5e5bc8a5,0xb301854c),LL(0xd258898f,0xe6b8c9d9),LL(0x415129b5,0x3a7ead26),
   LL(0x1e2c52bd,0xe3760794),LL(0x36b33409,0x383b64a3),LL(0xe476721f,0x3e25c3ed), LL(0x636f7572,0x209893c6),LL(0xe5f8134f,0x7e26bf15),LL(0x155ae70b,0x27e0a220),
   LL(0x9e9f2836,0x8f6b7ae3),LL(0xf9826937,0xf5eb2b96),LL(0x27c80d30,0x6b213e6c), LL(0x5320af31,0x52d85c63),LL(0x719df6d3,0x468df649),LL(0x2744251b,0x69403a14),
   LL(0xd9a2d948,0x28b1b9c5),LL(0x8e0c0b46,0x171331b5),LL(0xc359284b,0x84749e1b), LL(0x2e820b7b,0x184eb606),LL(0x740d7291,0x4f45113f),LL(0xaacb8d25,0xd1743386),
   LL(0xf464caa0,0x003c4b6d),LL(0x6280078a,0xe1e5b15f),LL(0xbfdf8ef2,0x2c0ace42), LL(0xbdcc429b,0x695d68df),LL(0xc9b7c777,0xb9a74231),LL(0xe2ee83bc,0x72976a32),
   LL(0x649ab654,0x72625a05),LL(0x40672830,0x4c950283),LL(0x6265499f,0xaa52b9e0), LL(0x9e60c50f,0x82c299fa),LL(0x697cef66,0x32ed0262),LL(0xf4c1412d,0xa5fad068),
},
/* digit=17 base_pwr=2^119 */
{
   LL(0xa11d1043,0x5db5f2ac),LL(0x3144df6d,0xb1054ec2),LL(0x484fcf47,0x4d5655f8), LL(0x9dbb1826,0x0650cfdb),LL(0x21f5feda,0x27dd6048),LL(0xd7202fb5,0x0ac68e0b),
   LL(0x1af46fd9,0xc42442cd),LL(0xfcf7cac7,0xb01213f0),LL(0x7eee2308,0x36e25d39), LL(0x17b3a506,0xcbf7331b),LL(0xcbb09169,0x9cb57ff6),LL(0x2928d485,0x819296b4),
   LL(0x8f4e94b8,0x22111a83),LL(0xb5d0778a,0x808a8a94),LL(0xe721738a,0x6dc353a0), LL(0x622c3cad,0xfbc21338),LL(0x2a4f4740,0x33ad7aa1),LL(0x7bd23daf,0xc3ce2ea5),
   LL(0x17755e9d,0x5214c0df),LL(0x22952389,0xd0989763),LL(0x2069bb1e,0x51accd0b), LL(0xe9cdd0be,0xe8ff9789),LL(0xbb212bd4,0x7e57a08a),LL(0x31edf24e,0x6b3e08ed),
   LL(0x6be00a20,0xf20543a1),LL(0x190d57cb,0xb8bcaa66),LL(0xef2b91a2,0xdd7c09be), LL(0x5592a7fb,0xd0e3d7f0),LL(0xa4e5380b,0x71dd3f1f),LL(0x7ad60960,0xe1a6c123),
   LL(0xd3ca68f9,0x57bc69bd),LL(0x52100876,0x4ada2eed),LL(0x8c3bb3a7,0x2e966e5b), LL(0xe177995d,0x507cdf48),LL(0x62dec6cf,0xa34769db),LL(0x615eb17f,0xd9cd4150),
   LL(0x7de68ea6,0x0135519a),LL(0xe8a86804,0xe966ae0c),LL(0x4d390e2d,0x63738258), LL(0x91e8e57c,0xfd018375),LL(0xd49915f4,0x31e5d307),LL(0xd1e1542b,0x173b719a),
   LL(0xfc46d6d5,0xb492fccc),LL(0x46eddcd3,0x973958f7),LL(0xb0f21fba,0x7222fff6), LL(0x41f0f218,0x8eb1db20),LL(0x9d5451aa,0x7353ea3b),LL(0xf054ffc5,0xf2cd3afc),
   LL(0x25b37d1b,0x148268ee),LL(0x6c58898f,0x9d597241),LL(0x5ef964a3,0x516c6871), LL(0x73756f60,0x777d95dd),LL(0xce04cb87,0x79b9b5e6),LL(0x371ab2f1,0xdbe50dca),
   LL(0xb9c32a0e,0xc12a3617),LL(0xa6d8dedd,0xd368a553),LL(0xef101236,0xdf472f4d), LL(0xa749514e,0xd6bc5900),LL(0xc2a203dd,0x7531976c),LL(0x983e67ab,0xa5bbf91a),
   LL(0x5ee73780,0xad4ada0e),LL(0x41fe693b,0xdf363734),LL(0x1a88c54c,0x996727c5), LL(0x6f0cd51a,0x635e6cd5),LL(0x4e75f89b,0x93d7e6cd),LL(0x2326e236,0x761b08f9),
   LL(0x27a21294,0x719c6065),LL(0x0f25f450,0x5fd2859d),LL(0xd52b85df,0xd10bf835), LL(0x40de93c0,0xa6fd2831),LL(0xf7c3a0d4,0x89c1bd7d),LL(0xefebe6be,0x75fed863),
   LL(0x17c97de1,0x921b88ad),LL(0x2cd59e91,0xb48dc81c),LL(0x73f3504e,0xda03b882), LL(0x5f2ee761,0x39ff606f),LL(0x5ad4f4de,0x608772a2),LL(0xc72d7dbc,0x2b4d5732),
   LL(0x17a8bdb6,0xd5d5f3d4),LL(0xde7a418e,0xcdadbfcd),LL(0x73f2b0da,0x2fbe10bc), LL(0xdd2b933a,0xcc436783),LL(0x120a26f4,0x7d464c7e),LL(0x3295a8b8,0x2e25a06a),
   LL(0x4edfcc33,0xce566ba6),LL(0xb883585e,0x72ced5ee),LL(0xe257f582,0xc7a1597f), LL(0x42dd036e,0x469e4698),LL(0xcf916dd4,0xf534e335),LL(0x5409b755,0x2a1a0ac7),
   LL(0x9c7ba23a,0x2ee09586),LL(0x2d9a1fbb,0xd9dae4aa),LL(0x0701d2cc,0xad876506), LL(0x35d64ab2,0xb9eaf2f9),LL(0xa2ca2cf4,0xc8a57b67),LL(0x0938f8b2,0x7e81d43e),
   LL(0xf1f4c06c,0x517ac75d),LL(0xd2fb833a,0xebd83a3b),LL(0xd0ceba1a,0xe14412b0), LL(0xc203df53,0x14bfd5f7),LL(0xa3cc4222,0x8eddee82),LL(0xe0413bc6,0x987370db),
   LL(0x2c70f714,0xc3aa5190),LL(0x1e87511c,0x35550f45),LL(0x207b35cc,0xacdb541a), LL(0xddc40e1e,0xd645313b),LL(0x5d7745fb,0xfadf08df),LL(0x63f5d14b,0x34c4a2da),
   LL(0x74f5662b,0x3bf9a89c),LL(0xc3293614,0x26fa24f5),LL(0x93645819,0x2232f2bc), LL(0xb642a3d2,0xae2493be),LL(0x4354eadc,0x67888dfb),LL(0xd5a343ce,0x7917ebd9),
   LL(0x22e18ac9,0x83f1a578),LL(0x96227250,0x754ac8ad),LL(0x02f84593,0x8720c7b5), LL(0x6077017a,0xbe8034ce),LL(0x125f0aba,0x0e38bcc9),LL(0xd7c001b7,0x491175d9),
   LL(0xdc6d9d65,0xdc830479),LL(0x8402b7f8,0xd961ba98),LL(0x85d31cef,0x9a870f49), LL(0x747a1a30,0x3f08d97d),LL(0x5e845976,0xf5cd0560),LL(0x39c404c9,0x34b2e606),
   LL(0x479f9f28,0xc6be3e89),LL(0x8639cfd5,0x277e7391),LL(0x6eac029f,0x90018430), LL(0xccc296ef,0xdeebdb98),LL(0x6c909542,0x225d9267),LL(0x37a527ab,0xcf151ac4),
   LL(0xa65460ad,0x40e22878),LL(0xd388bcc5,0x0a345fb2),LL(0x01c72449,0xfdbb8d02), LL(0xb73e0d19,0xeef0f1e3),LL(0xbbfdd87f,0x4ec39f63),LL(0x5ef57d3b,0x79501995),
   LL(0x60873c20,0xd6dbf138),LL(0x6715b03a,0xeee2c9aa),LL(0xa7482c51,0x2f5e5ce2), LL(0xc3dd1233,0x927e5fb2),LL(0xd64c61c6,0x72dde264),LL(0xcd32e0b4,0x605f8de1),
   LL(0x5b9091af,0x8ca120c2),LL(0x0f0fb8bb,0x6f7fa55e),LL(0x87264d00,0x78e97110), LL(0xd0f9cca4,0x63be0590),LL(0xe863c20c,0x4825ad2d),LL(0x12449414,0x4c34ba2d),
   LL(0x21fd0243,0x0256c27d),LL(0xa2406f04,0xbe85c36f),LL(0xb457d240,0xa30e427f), LL(0x3b4a0eb7,0x4bc8ea0f),LL(0xbfb3b1d4,0xf84a988b),LL(0x20a37ec6,0x1e9d2665),
   LL(0xb27e7722,0x07d6c0ef),LL(0x6729edd8,0xb81d03e2),LL(0xecf969a0,0x75b9e7d4), LL(0x72ad86dd,0xef9c8251),LL(0x9292331b,0xe609a51d),LL(0x35e1a81c,0x0df83add),
   LL(0xc5478838,0x73e31c76),LL(0x44828fc8,0x1a131a98),LL(0xb5e70dd2,0x14e09dce), LL(0x4a72dd31,0xb38126a3),LL(0x5b0e772f,0x3d06723b),LL(0xb7d1198d,0xca600717),
   LL(0x57b91e05,0x047cb901),LL(0x7e21d7a5,0x1946f1b6),LL(0xccec26a7,0x042e3eca), LL(0xfd416a33,0x0d9f9119),LL(0x49019bf8,0xdf57814c),LL(0x6527b664,0xe01c04b8),
   LL(0x359b599d,0xc0fb1444),LL(0xbd4e90c3,0x09858bf6),LL(0xe4adda27,0x1ad5019b), LL(0xe2069275,0xa8229db9),LL(0x2e9a9c96,0x4af49c04),LL(0x32909ac0,0x2eba27df),
   LL(0xba29ac20,0x44771234),LL(0xa8f7d70a,0x91271936),LL(0x35e3439b,0x1d2553dd), LL(0x66de658c,0xa35e4915),LL(0x5505ca57,0x958c5d53),LL(0xefac11c3,0x5806c454),
   LL(0x68947de6,0x480af451),LL(0x20ed7c76,0x1970a51c),LL(0xfd273a2b,0x13be8cf8), LL(0xde3fb216,0xdaa99fc9),LL(0x007e796d,0x9e1ecd60),LL(0x1145bac9,0x0d61162b),
   LL(0x1a7240fd,0xd324f3e5),LL(0x02dead6f,0xb652ceb5),LL(0x945e4927,0x9d2efe09), LL(0x09d3fe5a,0x505409cd),LL(0xd0b1d40f,0x4068d39b),LL(0x3d2add11,0xb169ceb8),
   LL(0xf87110f6,0x95209c85),LL(0x9d8c8473,0xce5e933a),LL(0x2dac6c5c,0x50e9c87d), LL(0x6c95d63f,0x5ad288c5),LL(0xc2e80fdb,0x9801bde5),LL(0xcf50ed4c,0x6d577a05),
   LL(0x5c341a0f,0xb2952c0b),LL(0x08fc2f1a,0x5296d67b),LL(0x595be428,0xb671036e), LL(0xf1974091,0xca815a34),LL(0xd7822f17,0xe4a2f88a),LL(0xaeba3ed7,0xdfc557f7),
   LL(0xf1f5523e,0x78c30cae),LL(0xa828197b,0x467ad201),LL(0x41532182,0xccf347b4), LL(0x84ef4bc9,0x67c4ff37),LL(0x98cade6e,0xebeb3d71),LL(0xa59ab992,0x979dba66),
   LL(0x0a363201,0x5ff9bcb7),LL(0xcc8b691a,0xb6995231),LL(0x62797899,0xc9aa7830), LL(0xcf5ece20,0x57d1c888),LL(0x963c0086,0x688d3d70),LL(0xc85a248d,0x91091fbe),
   LL(0xe39ec208,0x4d6e7182),LL(0x8d5cf62a,0xda235ace),LL(0xde7dbbdb,0x33f70b00), LL(0xe66a3b86,0x60f841c4),LL(0x42d7e874,0xbf8660af),LL(0x80f2c57c,0x073c5f84),
   LL(0xbc4f1599,0xd80d4f36),LL(0x185811a7,0xfd5e23b6),LL(0x7ac38f06,0xc8b0fa81), LL(0x43e96490,0x652d79b3),LL(0x73d98971,0xe222aa1d),LL(0xd121248f,0x4b6278dc),
   LL(0x3c14b2d0,0xce20c25e),LL(0x98193e2e,0xd1379c97),LL(0x6d804130,0x21102482), LL(0xf20fb7a3,0x577f8229),LL(0xb4f08660,0xdef46260),LL(0xe4dffb3e,0x8bb54fac),
   LL(0xc0ecadfb,0xbc45d692),LL(0xd3435c47,0x2d92602f),LL(0xa5dfe7a5,0x29e826c0), LL(0x42a0c989,0x9272dc04),LL(0xe3b3a440,0xc1eddc74),LL(0x65741581,0x741fa074),
   LL(0x75dbc772,0x0e2a8f97),LL(0xcc5e88b1,0xed5c6a60),LL(0x715808ac,0x099753c4), LL(0x9cd4d1fa,0x515d4892),LL(0x78393560,0xc4e5c9fe),LL(0x59206312,0x7ce9ac16),
   LL(0xba61c10c,0x05e139c7),LL(0x56285997,0x83c2806d),LL(0x80f93027,0xf308c086), LL(0x3d95ff97,0x192570cb),LL(0xb69f20a2,0xd7677bc9),LL(0x9a9416a8,0xfea2554b),
   LL(0x2ac3e62c,0xe26cdbf8),LL(0xb719cfa9,0x99cfefda),LL(0x7c7c2f30,0x67fc003a), LL(0x529cd8a8,0xfb8eb991),LL(0x8902bddd,0xe63830d5),LL(0x87d9a82d,0x8d0dd890),
   LL(0xcdd4c46f,0x239f32f2),LL(0x4de15fb1,0xf9517e3f),LL(0xbb8c38ff,0xc8183a19), LL(0x4f41bc08,0xefe4c3ef),LL(0x36ec7fd4,0x900d74fe),LL(0xfbf56005,0x49b24c64),
   LL(0x2aae4376,0x8c1c4bb0),LL(0xf9c1a936,0xe9eadc2b),LL(0x9b62791b,0xea437c5a), LL(0xafe6482d,0xb43a409d),LL(0x5ae0e629,0x405189df),LL(0xb6b6e06a,0xb60dbb63),
   LL(0x122f06d7,0x015b080a),LL(0xe52d69ac,0x8fe18e03),LL(0xbe34ac7f,0x54894766), LL(0xd5131f42,0x60d01c43),LL(0x24b2ae7b,0x179b14a6),LL(0x6e7c925c,0x2478209d),
   LL(0x7667ec3c,0x037b49bb),LL(0x0ddcd693,0xb4751c25),LL(0xe82b8e88,0xa9933bfb), LL(0x741344e2,0x14f812fc),LL(0x25e24d12,0x7253361b),LL(0x071df26d,0xaba4f18b),
   LL(0x05bcc791,0xfe1f3f2f),LL(0x33608ce5,0xe3ae1232),LL(0x672aa21b,0x3325659b), LL(0x9a01ebde,0x6d572c9a),LL(0xa10425ad,0x733c2067),LL(0xf7e7b67a,0x80f1da8b),
   LL(0x2e21e1c3,0xdcba0c7d),LL(0x79eb4307,0x08883499),LL(0x27cd2b91,0x5a5ee0f8), LL(0x13a4c6d0,0xcc6494e2),LL(0xb3270d14,0xdc1c75e3),LL(0x66eb04f8,0x64e17db8),
   LL(0x3e232e2c,0x1b8e2672),LL(0x0e99fc0c,0x1553fed2),LL(0x3c3d73af,0xe4e6608a), LL(0x3ed25820,0x8dc28c2a),LL(0xcadc377d,0x5f375146),LL(0x4b113c00,0x09e1c08d),
   LL(0x2d096c20,0x567cec1c),LL(0x9c346dd2,0x0b3fb247),LL(0x79a71044,0x1e4c1e43), LL(0xd86defcc,0x4980b895),LL(0x7175491a,0x9ee53061),LL(0x979cdece,0x28b5bb15),
   LL(0x16b49afc,0x36cb6dd6),LL(0x9372557b,0x498a1aef),LL(0xfa192344,0x4e51a28e), LL(0x6cdd93bc,0xf5a4373e),LL(0xec6a751e,0x54a73a75),LL(0x841194aa,0x9a563884),
   LL(0x91ad93f9,0xa95a8a06),LL(0x540e0c19,0xb0681b8a),LL(0x4f815aec,0xb0cb9c4d), LL(0x9b9c744a,0x5acf61c1),LL(0xfa7f4dc9,0x72c1126c),LL(0xcc2a7aa2,0xa5afea44),
   LL(0xf614aace,0x5a9fbecf),LL(0x794afb09,0x2e940ee6),LL(0x0ecaf20e,0xcc5a62de), LL(0x7a41866c,0x57634449),LL(0x4335ac0c,0x365bb619),LL(0x537165fb,0xe8c5699a),
   LL(0x0db61f1e,0x61966988),LL(0xf31a42d9,0x9a86bd4a),LL(0x93826e9b,0x712f2cdb), LL(0x17d5867b,0xd8edee16),LL(0x8710c50b,0x1e25628f),LL(0x7daab4cc,0x4f0c8f12),
   LL(0xd3fea70e,0x0ec1ef9f),LL(0x6591a1bc,0xdff00f61),LL(0x126ace6d,0xb680207f), LL(0xc5b9722e,0xfaa3fd69),LL(0x0b3e7f42,0x4a46ce3d),LL(0x49ba8836,0xdb76aec5),
   LL(0x77738737,0xeaa46cad),LL(0xe1b12f15,0x5c0f2ebd),LL(0x0ca12f78,0x1b07bf63), LL(0x8dd854dc,0xbe19584f),LL(0x11c1c918,0x9ac8971a),LL(0xb3600069,0x72f4b00d),
   LL(0x43f232a4,0x8c0c1e19),LL(0x6a116ed0,0xbd774022),LL(0xbebaeb71,0xee2ead3a), LL(0x861ab743,0xf224296d),LL(0x650a381e,0x37b4d979),LL(0x8d64e6f5,0x7a8da726),
   LL(0xd9e956fe,0x528c8c28),LL(0xa1f5f87c,0x1bb92006),LL(0xff91e588,0xf2e662c7), LL(0xbb5cfe01,0x36ad72ae),LL(0x19e772d5,0xf489b95d),LL(0x8b6ddd88,0x77675dbf),
   LL(0xf10d59a9,0xa40a5023),LL(0x250fb5de,0x15965ab7),LL(0xb539afcc,0x27808277), LL(0x8cab9918,0xf7f7ee31),LL(0x6c0353bf,0xd5b05608),LL(0x5255c6a1,0x20a150e8),
   LL(0x192fc198,0xdc6a57e6),LL(0xc3c7ed17,0x3506fc9d),LL(0xa4a271bc,0x916219f6), LL(0x964b3816,0x03f8b160),LL(0xf1a09190,0x454bae16),LL(0x6fab73b1,0x6d316b41),
   LL(0x5c22fc9b,0x768669f6),LL(0xe0219ce7,0xc2bbb0c5),LL(0xaf224185,0x77d99a44), LL(0xcead6ae8,0x1bdcaaa5),LL(0x80bcad3d,0xa0015109),LL(0xac464ccb,0xde74800f),
   LL(0x7c5f58b6,0x419fcc04),LL(0x423d4004,0xa28d071d),LL(0x30014ed7,0x2b51b7b8), LL(0x1e910824,0x51d2e74a),LL(0x049aba24,0x46142c53),LL(0x1c4c9bfc,0xeee290c9),
},
/* digit=18 base_pwr=2^126 */
{
   LL(0x3d7eba56,0x1432c61d),LL(0x9f3bcbe3,0x3e5748be),LL(0x70d4b4cf,0xd060396a), LL(0xd56cbe10,0x5cf4dd3a),LL(0x8c26eccd,0x6e120e45),LL(0xd44782b8,0x5a349466),
   LL(0xece48d34,0x0aa69704),LL(0x72c3e801,0x5ae78e58),LL(0xdbbf3257,0x8cae7cdf), LL(0xad51d342,0x7f33e06d),LL(0x8c54f76e,0x842b3c9b),LL(0xd9738bf1,0x27439730),
   LL(0x8d92d2ec,0x606dcdf3),LL(0x47bce9e6,0x80661389),LL(0x20424ac9,0xf9e37e1c), LL(0x1d4308c4,0xb4271df8),LL(0x2524d693,0x32991fcb),LL(0x3eb1eb87,0x1b3e3739),
   LL(0xb01d8c67,0x1366f4ea),LL(0x12c079e1,0xfc0e2aaa),LL(0xc73eb944,0xde8828da), LL(0x4fb7452a,0x1be389c3),LL(0x3ab20ed9,0x1b47de61),LL(0xe88a74b5,0x0a0e4dd5),
   LL(0x9f3a0e52,0x42086dea),LL(0xdea9b529,0x2b27362c),LL(0x0f6b324a,0xa601b101), LL(0x8d8999b3,0x50deb5db),LL(0x0f97d636,0xb5146466),LL(0x7452ad66,0xe400edf5),
   LL(0xf40e2c2a,0xcea8e0f5),LL(0x5c7a6851,0xd7a70dcf),LL(0x3c3eef95,0xb07f27bf), LL(0x6b31486d,0xebeaedec),LL(0xae6fd833,0xa6218332),LL(0x4c2f458c,0x19ce8eaa),
   LL(0xb1bfa9ec,0xfaae66f3),LL(0x2742714b,0xc7b516dd),LL(0x8f55fcc6,0x6bd999fa), LL(0x1b07a49e,0xbdf9e3dc),LL(0xdb922eb3,0x8c8fe856),LL(0xe3b05757,0xf0e446dc),
   LL(0xcab82ecb,0x12bfbae5),LL(0xe279d4d5,0xae24cbb6),LL(0xaaa064ed,0x2ae08e15), LL(0x053194f4,0xe4f4d9f8),LL(0x4fb349c2,0x66627f60),LL(0x7d673df8,0xa57270c4),
   LL(0x8de75f76,0x79c8e28f),LL(0xb5103e06,0xc46d7632),LL(0xb99c0669,0xee23de7a), LL(0x112100cb,0xb6a05082),LL(0x559c6fe6,0xd3aaf2ba),LL(0x310d9cda,0x1fa98099),
   LL(0xd68c0cdd,0x66b15c32),LL(0x316bb565,0xc1a8dcd1),LL(0x66388e62,0xfe124230), LL(0x6aa0caf8,0x87421a82),LL(0x45ae8aba,0x163ed8a0),LL(0xf8869a95,0x799ecc06),
   LL(0xb1f08b18,0x9b5f811f),LL(0x8a8e6e51,0x21df77e4),LL(0x41906e3b,0x0ece8f82), LL(0xb358dbd7,0xcc8f1a7f),LL(0x348f81d2,0x7eae1659),LL(0xba2fc496,0xaeb9bf86),
   LL(0xe8ee3475,0x1d24ece4),LL(0x0333898d,0x3a9629c6),LL(0xaa4b4d26,0x51fe2f63), LL(0x45450f91,0x65713078),LL(0xed53b1a6,0x9fe570de),LL(0xccfc903e,0x49d5947a),
   LL(0xd12646d4,0xd1a4ee1f),LL(0x108a021d,0x8be0f550),LL(0x0c9a397c,0xadca94e2), LL(0x7a116b52,0x443e40d2),LL(0x4a7e4f8c,0x8922a749),LL(0x11b9faaf,0x13dd25d4),
   LL(0x677337c6,0x23169fd1),LL(0xf4275e2f,0xe8b155ff),LL(0xaa432608,0x199fd451), LL(0xd6a8a707,0x43fae1e4),LL(0x22598f5c,0x30b74dae),LL(0x4412a37e,0x3231c5fb),
   LL(0xa6ef8dc3,0x242076a6),LL(0x3531d9ba,0xa91fa776),LL(0x15395cc0,0x77f95454), LL(0xc333f3fd,0x2d82f42d),LL(0xd57091bf,0x0ebcdf5c),LL(0x04a7020c,0xced34ce1),
   LL(0x8a8c3087,0x996010a0),LL(0x8731577a,0x2e13009d),LL(0xfb82ff62,0x4a13fc99), LL(0x54ffd5e2,0x03887d55),LL(0x72a2d703,0x81a63ca5),LL(0x61a70095,0xffd0fd49),
   LL(0xa9ab8d73,0x08cb1949),LL(0x30fcdafe,0xbcd463e1),LL(0xcb2946e3,0x1bd9c440), LL(0x7b685a62,0x67df0c20),LL(0xe6a0b599,0xc85033b3),LL(0xdad9fc5e,0xc26cb4c0),
   LL(0xd189ee56,0x2915b334),LL(0x7df4385e,0xec153690),LL(0x74e10903,0x78c1233a), LL(0x99e6424b,0x1aad36db),LL(0x8e6c1b2a,0xf0985d3c),LL(0x94d7d170,0x2378ac4b),
   LL(0x4f4a6728,0x2b339c66),LL(0x682d8695,0xb36bf9a0),LL(0xea6efc7f,0xe5077906), LL(0x8c8272d1,0xbe19ad2b),LL(0x80ba118b,0x2273dd26),LL(0xbc2ebef4,0x1afaebc8),
   LL(0xd967a41e,0xebfd1e2b),LL(0xc3fded13,0xf89bd7eb),LL(0x5e968c2e,0xfa2e6197), LL(0xd2842061,0x75e0b500),LL(0x81e69daa,0x484219aa),LL(0xe5989ba9,0x3a9fc283),
   LL(0x827deaa4,0xc54392cb),LL(0xf39f1cc6,0x09b5364b),LL(0xd2711eb8,0xccce7212), LL(0x9b8acd72,0xd4aab549),LL(0x0cb6dec2,0x030af88b),LL(0xc5188526,0x9852b073),
   LL(0x8119cabd,0x0cd88f26),LL(0x92f36a37,0x09ad060f),LL(0xea4ffb39,0x60e5c4a0), LL(0x44aa9c46,0x29530c7c),LL(0x98565223,0xbc1d4e89),LL(0xd46de9ba,0x1aa44a30),
   LL(0x2ac56598,0x33cdcbc5),LL(0xdb40e83e,0xe16fab9e),LL(0xa7fb4b01,0x2e90d12b), LL(0xfe18a04f,0x80c85443),LL(0xf59d5ddb,0x27fae391),LL(0xa121c824,0xd75f7ffa),
   LL(0x21d0d7f1,0xacf5993c),LL(0x04b7c788,0x159623b7),LL(0x35cea4b4,0xc631a623), LL(0x36f64979,0xc75060d1),LL(0xde81f49b,0x93d67ee5),LL(0xb1c080dd,0xc8b7ddec),
   LL(0x88fc80b3,0x86602cee),LL(0xa52e2b85,0x8d378408),LL(0x7b43597c,0xb66f2489), LL(0x34ead5db,0x8fa5fe75),LL(0x872b559f,0xa91b7b3d),LL(0x969b74d2,0x34007f49),
   LL(0x31a991bc,0x7ac027f0),LL(0x231f0bd6,0xce3e5a83),LL(0xa724434a,0x6059c8c4), LL(0xbf199050,0x3949f4fc),LL(0x50f4c59e,0xc8d1ce42),LL(0x2c2e840e,0x406e976e),
   LL(0xff33a74e,0x70c69777),LL(0xbce579fb,0x0caa3c59),LL(0xe3cc09ca,0xcba746d0), LL(0x7493a1a1,0xe544dacb),LL(0x74ebeba1,0x84d251ac),LL(0x7b5a26cb,0x24c351ad),
   LL(0x438babd3,0x882cde8d),LL(0x4a0c3725,0xc88a8ff9),LL(0xa80a190f,0xee17ee44), LL(0xb11442f3,0x50337a67),LL(0x7f7eed2a,0xc6803f55),LL(0x36750d2d,0xee2e0425),
   LL(0xe5eccbeb,0x9614df9b),LL(0x6d8a00a6,0xa42fa89b),LL(0x5cb36d70,0x90d5f6eb), LL(0x06790958,0xcc2994e9),LL(0xbda37f56,0x4c4820a0),LL(0x109395c2,0x4ac16bf5),
   LL(0x4d5e93c7,0x57f12bfa),LL(0xae11e7bf,0x6f57c3c7),LL(0x805ee169,0xe0c53938), LL(0x27da73c9,0xb1ea61d1),LL(0xf1ca8fcf,0x812387b6),LL(0x9aaad2cb,0x812bbf9b),
   LL(0x673f3249,0x103ab65e),LL(0xf91c5bac,0xbccd32cd),LL(0xb7502b99,0x2cd17117), LL(0xc6ee643f,0xacbc6b98),LL(0x7e3a9007,0xe45beed2),LL(0x39540d2d,0xd6e7cbb3),
   LL(0x735d7a7e,0xf5ba275a),LL(0x939bb080,0x7d56e5bb),LL(0xa316d8f3,0x632ae0df), LL(0x44834265,0xb16f0da3),LL(0xc0e774a9,0x2ac41378),LL(0x963763b2,0x2d1368a7),
   LL(0x2910cb9b,0xbb909572),LL(0x7ec8764f,0x9b97d7d3),LL(0x0a1f7187,0x45ed9127), LL(0x91607114,0x4911eae6),LL(0xbec9af64,0x5a3283f3),LL(0x24181e7a,0xfad12d4a),
   LL(0x304aa2f6,0x7130416a),LL(0xa93ea2a4,0x3d8ec32a),LL(0xb42283de,0xf746be9c), LL(0xfaf3861d,0x122fff14),LL(0xfc5c58f9,0x85c1a762),LL(0xa1051cf0,0x096434ec),
   LL(0x7f5300f5,0x991bef56),LL(0xdb18ccbb,0x8a974cb7),LL(0xd51e5e8d,0x95e6464d), LL(0xb24414cc,0x7891a137),LL(0x78ca6881,0x614852b1),LL(0x11512b7e,0x8ce344f9),
   LL(0x14180a17,0x225c5475),LL(0xe0402e95,0x20f7198f),LL(0x7dc14a78,0x834328aa), LL(0xf00864aa,0xf9f3f9d9),LL(0x5c929f4e,0xc14ac9fd),LL(0x67caed5a,0x565edd57),
   LL(0xa3116bfd,0xc3febfae),LL(0x25b0091a,0xd28b9297),LL(0x43d43e9d,0x71940761), LL(0xa6743cde,0x0b0dcd52),LL(0x85916ef5,0x0d9c9307),LL(0xed0ea307,0x16d7fa15),
   LL(0xd874f8fb,0xcd73c3b4),LL(0xdd4872db,0x89f049a0),LL(0xd6c574e0,0xa3c1e344), LL(0x1501c2b5,0x42aa48a1),LL(0x4379b187,0x08c07386),LL(0xdb94daef,0x651b8ea0),
   LL(0x2a46e40e,0xb29008a4),LL(0xafe5ce10,0xf8cfacfa),LL(0x1f36cf21,0xc20de58c), LL(0x77586b4e,0x1667beb5),LL(0xdb697901,0x549fbdad),LL(0xbec7838a,0x05d84e16),
   LL(0x86573220,0xca40cc00),LL(0xce72bdb3,0x2b43ccde),LL(0xf187d609,0xd8d285d3), LL(0xcbb90bdf,0x975d5583),LL(0xdb3ce7c8,0xd8e1edb1),LL(0x1be6a0d8,0xf24a6f87),
   LL(0xc5db734e,0x72d204bf),LL(0x1ead06dd,0x4b794609),LL(0xd899503a,0xab7e900a), LL(0x808698fa,0x682a67a7),LL(0x81d2be81,0x8d5f7a3f),LL(0x6db17983,0x5ac7f636),
   LL(0xf5d8bc59,0x58e88381),LL(0xbc6677a0,0x0c14b7eb),LL(0x75f0beea,0x26d7fb70), LL(0x3c789c7d,0xf71d6c28),LL(0x31a64d6a,0x0e1c3ef5),LL(0x3a025a05,0x930f15cb),
   LL(0xf84e9a66,0xa848b469),LL(0x0a6acdfb,0x8b3620ef),LL(0xc0f3c653,0x803dc9e4), LL(0x70d4aaad,0x87b9452d),LL(0x2e5341df,0xd2039635),LL(0xcee3e654,0x1518ec72),
   LL(0x75e30599,0xd89bb62e),LL(0x358eb463,0xffb8be40),LL(0xac2b9c1d,0xcc7b8c90), LL(0xbe967747,0x14076fcd),LL(0x9e9118cc,0x4c590a0d),LL(0x7fdedab7,0x255af971),
   LL(0x0bf36a0e,0x7cb0703b),LL(0x2f6e1ad6,0x9d4f60d3),LL(0x80851e9d,0x0ddfa57d), LL(0x320a950b,0x0d7e34d2),LL(0x3cf9b367,0x344c00bb),LL(0x7f5fb665,0xbf5c2104),
   LL(0x5c85310d,0x01355f1f),LL(0x430840fb,0xa17a28ca),LL(0xbddb8e53,0x1573b14f), LL(0x1be8bb25,0xb6e9f88e),LL(0x68f882ed,0xf8283ea2),LL(0x9f8a6c36,0x97b1f85c),
   LL(0xf96965eb,0x16d8df1e),LL(0xc2b9fd6d,0x347159dd),LL(0xc973dc04,0x5a8ff74e), LL(0xe385ad6d,0x6b89c04a),LL(0x76c7bedd,0x1604bc1a),LL(0x09914658,0xc1e216de),
   LL(0xe5d7f3b0,0xe285c4b9),LL(0xde0ec465,0x340ea40f),LL(0x51eaa759,0xddc7e286), LL(0x2a2118e1,0x3d1f99a9),LL(0x99fa6ada,0xf8dbccb1),LL(0x2e831534,0xdcb7d5d1),
   LL(0xa33e53ea,0x90c5048d),LL(0xc03b6f58,0xe68e4543),LL(0x0585ccce,0x2c9860a2), LL(0x15f92a46,0x948b7875),LL(0xa62a0f0f,0x913f01ea),LL(0xa35eeb9b,0x06623f96),
   LL(0x9c77ae5f,0x403740c1),LL(0xd975126c,0x25df8030),LL(0xaf1bd02a,0x762b3d5f), LL(0x7a17ca9e,0xa9ef424f),LL(0x5bbdefcb,0xf0e425ee),LL(0x6ed00a8b,0xafa22cf9),
   LL(0xfdcc0852,0xdcb9848b),LL(0xe4e8b45d,0x22851be9),LL(0x195c527a,0x25543398), LL(0xc4ffcb5c,0x39a1709a),LL(0xc84717ea,0x9d6bc1dc),LL(0x3080227e,0xc5a9436a),
   LL(0xafcc481e,0xfcc2dbcb),LL(0xe8bdacd0,0x936af760),LL(0xa208d23c,0x37762746), LL(0xcf4cfb3d,0x7c32898d),LL(0xbe8d86f7,0x4c7103db),LL(0x29137736,0x0e8e29e9),
   LL(0x5fb1f4df,0x4dddcd9d),LL(0xf8e40af7,0xcee0b177),LL(0xd5ed5e8c,0x2d5eed04), LL(0x4436fde2,0x8c57cd9f),LL(0x59431816,0x6d9ae88a),LL(0xfcd05844,0x410d80be),
   LL(0x4f2e214e,0x72cb7d76),LL(0x50149bdc,0x84aecdf4),LL(0x29a10599,0x55c54af4), LL(0xff356cf8,0x15b4a13c),LL(0x94c1cdb7,0xe251ef48),LL(0x3c294aa2,0x3d868b7d),
   LL(0x38ac42ad,0x61562246),LL(0x9857cdde,0x35bea87c),LL(0x3bff3a31,0xe288cd86), LL(0xfeff112b,0x6de40901),LL(0xfb86b4fa,0x4c3ff10e),LL(0xa2760141,0xefb2ebbb),
   LL(0xb651d24d,0xfe675b81),LL(0x1bf57d70,0x58decfa4),LL(0x3e407d11,0x6d577314), LL(0xc363e64b,0x99100a53),LL(0xd5b05ad2,0x3700486d),LL(0x3d16a4f6,0xd725f452),
   LL(0x4369d7d0,0x7987446d),LL(0x082a8d9a,0x6f73c9d4),LL(0x90c45db8,0x50f2643b), LL(0xa8b8a26e,0xc0547960),LL(0x0419c5f9,0xcf0cc729),LL(0xa88c242d,0x36ec0c68),
   LL(0xab61869c,0xc39acd0f),LL(0x52d501d6,0xb3676312),LL(0xe963c20d,0xcce446cb), LL(0xd5427311,0x171ef8f9),LL(0x0a43e236,0xd2250f20),LL(0xbf664fa3,0x3734c3a6),
   LL(0x3a865bc1,0x4debd2e7),LL(0xbffefe51,0x48d3b6e2),LL(0x39246490,0x63d793f1), LL(0x07251f37,0xf2e568cb),LL(0xb86ac8ba,0x0543ce7c),LL(0xf0233d6d,0x34c94bb8),
   LL(0x6b22f5d4,0x825d20cc),LL(0x87e8b87c,0x3008f0d6),LL(0x6fe24ecc,0x1f179a44), LL(0x3732b8f4,0x45b86fc1),LL(0x5bf33be2,0x280c34e0),LL(0xacc6b35b,0x3656ee7c),
   LL(0x3de8e3cf,0xef5aea9a),LL(0x13295f9a,0xdaf53c2f),LL(0x37faa7d5,0xef4cb920), LL(0x7b389099,0x7cc42259),LL(0xf4bbc5ab,0xa229e4b0),LL(0x369283c9,0x75a6dd11),
   LL(0x208de900,0x2472c7bd),LL(0xfe199477,0xa707b3a7),LL(0xe84304aa,0x4eeabcfd), LL(0x69fb0f07,0x8a125707),LL(0xcff32ccb,0x8146ef85),LL(0x6874ae86,0x924e6627),
   LL(0x5baf5149,0x7ebfdd60),LL(0x6fd26802,0x060d10ff),LL(0x433383ae,0xed4cab01), LL(0x6ae53b2f,0xf6529a62),LL(0x17680c05,0xfb786ea5),LL(0x6a368e73,0xe22d7ea8),
   LL(0xd5b48338,0x52e49947),LL(0x0b58cffa,0x70575502),LL(0x3097c699,0x0f568fcf), LL(0x2ebb3f51,0x3a78100a),LL(0x6ebf4f52,0x8ca651ed),LL(0xaf034c4a,0xc21be721),
},
/* digit=19 base_pwr=2^133 */
{
   LL(0xb439db86,0x318e0141),LL(0x6ba0c36d,0x24dccf9d),LL(0x278f654b,0xe30ecdee), LL(0xcc96cfd2,0x3b53ec8b),LL(0xffdef7f9,0xb15365ab),LL(0xb77c2bdb,0x2886538d),
   LL(0xdd6dd374,0x16beccc0),LL(0xe0de912b,0xab022ba4),LL(0x9fa6c113,0x0fa1e066), LL(0x5602f255,0xa62f3440),LL(0xde5080c3,0xf1744743),LL(0x959de74a,0x5dc10f57),
   LL(0x30bcd508,0x3fea4a00),LL(0xd7784a23,0xc4b99502),LL(0x22777f72,0xc2aae8e2), LL(0xb738b26a,0x0998acd7),LL(0x284a4687,0x01c21758),LL(0x60aa1544,0x86331b2c),
   LL(0x044902c4,0xb138ec30),LL(0xbbc5aa92,0xedbb10a5),LL(0x8cf968b8,0x78dab5f0), LL(0x5582adf1,0xc522aa4d),LL(0x757f6d77,0x27b2cad3),LL(0x8496e06e,0xd99cc418),
   LL(0x14b10f6b,0x353a03a9),LL(0xd618e82c,0x0e0360ad),LL(0x5c8ba6c1,0xe95b9c21), LL(0xa2aca406,0x420fcef8),LL(0x3d0af393,0x511e7327),LL(0x4d7c2d87,0x240d0932),
   LL(0x0431f61e,0x72bc67e3),LL(0x35cbaced,0xe0b8c3f5),LL(0x811b966d,0xec526634), LL(0xedbca0fb,0x994e956d),LL(0xec733f7b,0x1d1d456a),LL(0x10d8156a,0x481983f5),
   LL(0x25b3e1e1,0x2d4c36da),LL(0x63bd55c6,0xf81e5e71),LL(0x1d80e9a3,0x8ebaa53f), LL(0x5d2b6568,0xe162ca96),LL(0x58264463,0xae3e0e1d),LL(0xca65ecfa,0x4d50c694),
   LL(0x2b0d1ddf,0x9dbc08f7),LL(0xabf627f7,0xe9e345da),LL(0xa7badd66,0x110cebe6), LL(0x48702def,0x8055893e),LL(0x987a98ba,0x04470e1b),LL(0x1d380312,0xc27475da),
   LL(0x9fea7b29,0x8a275184),LL(0xcaf14cdd,0xdc9a9a94),LL(0x5aa56285,0x6c8859b5), LL(0x242a5eee,0x9eede9fa),LL(0x0b4d4c44,0xb738b0f2),LL(0xda57c6cc,0xcd8174f8),
   LL(0x3cb5df38,0x32b55a3c),LL(0x0cbf8de1,0x2bf1c097),LL(0x9b7812a9,0x1f27c2f2), LL(0xbb4241cd,0x41cfb699),LL(0xb8cefac0,0xf25fb9ad),LL(0x4b462fe6,0x0e8ea882),
   LL(0xb2fa7ed3,0x6d1fe0cd),LL(0xb805f909,0xa5afec06),LL(0x50929ff7,0x7114c77f), LL(0x41e73d2a,0xb70a0449),LL(0x412afd6a,0x17353c86),LL(0xe777054a,0x4b1ab341),
   LL(0xb6b4736a,0x205f30a3),LL(0xec96f97b,0xdc57c0dc),LL(0x50892f3d,0x6026e016), LL(0x9872ee46,0x5587a900),LL(0xea0f599b,0xfa2e40d1),LL(0x5698483e,0xd0dfe28e),
   LL(0xc978c548,0x96ea419d),LL(0xd41fdcca,0xc78106e5),LL(0x2c3d871f,0x1fcd9bd6), LL(0x8f18504e,0xde3a3b13),LL(0x147e11bb,0x41af768d),LL(0xbea88154,0x57db2d46),
   LL(0x359605ca,0xa03b50ed),LL(0x8d2e6f9e,0x102d624a),LL(0x021fbaee,0x75846c7d), LL(0xfde66141,0x7ce93a17),LL(0xcb75846b,0xff36566b),LL(0xdfe8687b,0x13f15dbc),
   LL(0xb6900145,0xa1762cdc),LL(0xd04f6298,0xc52758d8),LL(0x05fb29ed,0x2388b649), LL(0x952caba9,0xf559602d),LL(0x612d83d0,0x7a57aa1c),LL(0x798f6741,0xb0a70b52),
   LL(0x718b8cfc,0xd977e424),LL(0x9724fb0c,0x0bd2e7ad),LL(0xb4b77d96,0x62055643), LL(0xf3b93b52,0xabe7d1ae),LL(0x29f1f8a5,0x2573a99e),LL(0xf1ae95ae,0x9d861485),
   LL(0xf37551bf,0xff5d9174),LL(0xeed4fbcc,0x2120b50e),LL(0x61f2e529,0xe442a255), LL(0x2f0b3221,0x1991c2ac),LL(0x598ea6d0,0x39e9fba8),LL(0x38f4004e,0x3e215ecb),
   LL(0x7112f48d,0xab373c03),LL(0x0cbf0146,0xb36e09a1),LL(0xf8d04ac7,0x750d400d), LL(0x1a257ba4,0x7a522a7e),LL(0xa6fe3d3c,0x19d94e30),LL(0xab24c6d5,0x8f563b6d),
   LL(0xce3f76db,0x4e8f7ae4),LL(0x951e6883,0xf041cd61),LL(0x677e20c7,0xd5346ef4), LL(0x4f2fa361,0x2ecde00a),LL(0x24e52b5a,0xda39ea83),LL(0xa4037195,0xf13a1d6a),
   LL(0x8122da8e,0xc4ef50c0),LL(0x1e4dfe54,0x44245ba2),LL(0x549eb736,0x51a3afe6), LL(0x149e53e2,0xb389a041),LL(0x019f7f33,0x59e1806e),LL(0x207e8b03,0xf0bf62ca),
   LL(0xf023e16c,0x2afa74f0),LL(0xe45110c3,0xc0e3985f),LL(0xef4dc058,0x30d3c844), LL(0x0932e063,0x74a8fe78),LL(0x8d089afd,0x1694b37f),LL(0xc02cfea5,0x688b8d05),
   LL(0x00b00b60,0x82d73a13),LL(0xec1be174,0x9914b5f8),LL(0x260cca25,0x4b3aa37e), LL(0xdb9eedcc,0x80edd7e4),LL(0xfac065fd,0xd734f788),LL(0x95ca6a6d,0x3bad348e),
   LL(0x394ac74a,0xcc754ddb),LL(0xfdf2392c,0x499071de),LL(0xbcd3b3a9,0x64980287), LL(0x8ac174ca,0xf9ad94b1),LL(0x48adbfcc,0x7d8487b5),LL(0xeaa22b97,0x3b94b5a7),
   LL(0xd830a7f8,0xdd6a5e09),LL(0x0a707d6b,0xe88ab4ee),LL(0x6f282e77,0x65064114), LL(0x05630c1d,0x732b3f5f),LL(0xa32278e4,0xedc7f608),LL(0x40aa906b,0xf9cc5a48),
   LL(0x84cb7f2e,0x22cc8910),LL(0x492f6b49,0xe72c3a7e),LL(0xd9172542,0xcdfb7abb), LL(0x5379490b,0x332ca313),LL(0xc019dc08,0x74e34e7e),LL(0xffac012c,0xae01f427),
   LL(0x6589755c,0x28a5ee38),LL(0xf5484971,0xc3c09b28),LL(0xebdc2e2d,0x98c22f2b), LL(0xb739eaf1,0xf34f7446),LL(0x40de5322,0x71c9f8f5),LL(0x8cf40eb6,0x94ace574),
   LL(0x660ffcd8,0x14a05e05),LL(0x384c2a8e,0x13f0de7d),LL(0x192dd045,0xa52ddb4d), LL(0xea4aaf10,0xf4d36e86),LL(0xe77d2f45,0xfd9db24d),LL(0x904e7b10,0x55f85a8b),
   LL(0x260ee499,0x315e7fcb),LL(0x15cbbecf,0xa3bb055e),LL(0xfda09d93,0x2634a003), LL(0x71c4d7c5,0x5b3c4fe2),LL(0xc9380c8d,0x6805f34b),LL(0xc8a539f9,0x587eeb71),
   LL(0x51635be4,0xdd9571b7),LL(0x9e7aabb0,0x321a0e22),LL(0x55781706,0x8b2b23f5), LL(0xaea3c730,0x7cca33ac),LL(0xf95cb480,0x3f6401d1),LL(0x26a55858,0xa5ef93a5),
   LL(0xc8b7c736,0x1eaf9f4c),LL(0xeb5b2b70,0x1dcb1722),LL(0x65aab50b,0x0d948725), LL(0x15e7c8ef,0x193c8d06),LL(0x6c391c08,0xf7745835),LL(0x897265d5,0xcbd15810),
   LL(0xb54eb895,0x205436fe),LL(0x059b35b6,0x4300c84f),LL(0x9d231cce,0xab267da3), LL(0x20fa381d,0x25239fea),LL(0x004d10be,0x5781e9de),LL(0x14641496,0x36c78b47),
   LL(0x99963302,0x0c2bd51b),LL(0x6afdcb5d,0x1958d45e),LL(0x25a00883,0xaa86d186), LL(0xe705abde,0x14852c1f),LL(0xc056e6f3,0x352304ed),LL(0x83c10ff8,0xd05eedf0),
   LL(0x702067ff,0xd0cd6dc3),LL(0xd3d111ac,0x3ee83612),LL(0x5ef57788,0xad67877e), LL(0xff6b4f08,0xda249b82),LL(0xd696f4f4,0x9dbbe132),LL(0x51804942,0xfe5ba023),
   LL(0x060154d9,0xdc19d20b),LL(0x522b27ce,0x2c6233ef),LL(0x20fde2af,0x8b235bec), LL(0x3bb960c1,0x2adcfe01),LL(0x603b8fb9,0xc79aa84e),LL(0x6b460288,0xe204af13),
   LL(0x21d24338,0xe9c83cec),LL(0x644ee8c2,0x768ea11b),LL(0x94fcc664,0x59088ada), LL(0x9374c03c,0x900c7bb8),LL(0x43cb2818,0xad48000f),LL(0x9145786e,0xa2041cca),
   LL(0x02582993,0x1765e1a6),LL(0x34c86e0f,0xe74f0372),LL(0x2bfc967b,0x5ba6c193), LL(0x154fde15,0xa09688b9),LL(0xd02973ed,0xcc002cb5),LL(0x92f64238,0x01037f55),
   LL(0x8597bf52,0xcef12f0f),LL(0x323d2edc,0xd1bbb357),LL(0x6cac9c11,0x06254d2f), LL(0xa913fbcd,0x13b6a932),LL(0xa97bee41,0x9865d3ba),LL(0x224479fb,0xab2a703b),
   LL(0xbe515d1a,0xfd4472bf),LL(0x1dc983d4,0x7ca21984),LL(0x2a2db967,0x1f62c4c0), LL(0xcf71a671,0x4bcade52),LL(0xdfadf5eb,0x50fdda39),LL(0x650fac39,0xb020e598),
   LL(0x6d401575,0x05292bde),LL(0x1448bd42,0x0c3c2b31),LL(0x526536c1,0xa76d46cd), LL(0x2a839d2c,0xaa0c74ff),LL(0x62f94ad3,0x7283bfb0),LL(0xe710c108,0x063159c2),
   LL(0x11840343,0xbf96687c),LL(0xd26b7f84,0x36468056),LL(0x362582a6,0x6ab84c43), LL(0x7d43997a,0x3bb7a3cb),LL(0x19bc5d68,0xfafa8874),LL(0x042ff6c3,0x73dd38cb),
   LL(0xe5d2eb17,0xab12eac0),LL(0x07bac547,0x4b8a41cb),LL(0xffa6df36,0xe9bfd303), LL(0x3c9a6423,0x6bb4517b),LL(0x9ca20f86,0xd393d1f0),LL(0x2231716f,0xb8e17b6c),
   LL(0x35417747,0x4a3a2eeb),LL(0xc8c6b7c4,0x9b638e84),LL(0x16cbe067,0xc1215698), LL(0x322fa78f,0xfe96edae),LL(0x35c0584c,0x7b161bc3),LL(0xb32db702,0x86d59110),
   LL(0xdc2fd5f6,0x232b8b24),LL(0x3c7926e0,0x082d2045),LL(0xb2c6285b,0xa73041dc), LL(0x8dcded40,0x60172712),LL(0x04a70ed1,0xc62c9174),LL(0x962a378d,0x6fb37121),
   LL(0x823c3d77,0xb6f17b32),LL(0x9f7e5c97,0x5c45e3a8),LL(0xc3396a05,0x2daa3c77), LL(0xb063c7fa,0xbb2c63d7),LL(0x2ce90d3a,0x7a1c8227),LL(0xe4b3c32a,0x109b4b69),
   LL(0x35f82f23,0xfdf6efba),LL(0x35381294,0x805d2980),LL(0x35aa14e0,0x894b62fa), LL(0x83d15a96,0x9b0d470a),LL(0x562cbafe,0x16a073f7),LL(0xdd734859,0x7851278e),
   LL(0xd678f3a7,0xd0bf1f91),LL(0x5c227a14,0xe3d6d2e9),LL(0xf043aca3,0xd262c755), LL(0xee561894,0x60988c89),LL(0x73ca3bc8,0x64ec6b35),LL(0xc42baa7e,0x69a377b3),
   LL(0xc96115ea,0x71695d26),LL(0x5b21f1d3,0x72271453),LL(0x39bafd21,0x8697c31b), LL(0x90197919,0xbd705026),LL(0x24ac9116,0x3bd494e4),LL(0xdfc15b14,0xc80bb868),
   LL(0xf71afdd8,0x279ba623),LL(0x033c4ae8,0x9f980a73),LL(0x470c4517,0x21606d51), LL(0xe11ef4db,0x6d179aa7),LL(0xf1ca7162,0x869eac72),LL(0x86deb5b6,0xbc4a973d),
   LL(0x200897e5,0x162b8299),LL(0xfccd9352,0xd6d516d0),LL(0xabe7feae,0x77eb73ea), LL(0xc3484609,0x8c10355c),LL(0x3426fefb,0xeea4f0b5),LL(0x1520bff2,0x140ba2d5),
   LL(0xa781a4b6,0x8e02c736),LL(0xc1a1dc16,0x42c46ee6),LL(0x40321197,0xcae4633e), LL(0x98f07389,0xf04c64f2),LL(0x037c647a,0x6ee9ba3c),LL(0x48084ba5,0x4b003962),
   LL(0xe74b1cd4,0x858e1b6d),LL(0x32cb7b0f,0x1d3abad5),LL(0xed92d1b4,0x5b519e77), LL(0xc30f4257,0x1c92c6d0),LL(0x4987bf78,0x5f383d3b),LL(0x4a7f80cc,0xd1c583d1),
   LL(0xb56713fe,0x93af26d2),LL(0xa2ab05fd,0xc1226000),LL(0x7254b6e9,0x841945df), LL(0x5061c175,0xc5109d82),LL(0x00ff256d,0x62d1e3dd),LL(0xf0de138f,0x843b00ee),
   LL(0xf374c5cf,0xf615bb55),LL(0x48c4c8a3,0xf0ad394d),LL(0x4d11c95a,0xdc7346c8), LL(0x4c5c8014,0x458a607e),LL(0xa5494d82,0xfe616aeb),LL(0xec4b9fb5,0xb0b346d1),
   LL(0x06273f56,0x69c25150),LL(0x0c65034e,0x67589cbf),LL(0x93836375,0xd9a2c967), LL(0xa738e250,0x65e7cf03),LL(0xf4d22339,0x96e92577),LL(0x81400fed,0x4bfc9f81),
   LL(0xbe5747bd,0x6e69c9c5),LL(0x2c6c18e6,0xe0472ac1),LL(0x083b884c,0xba3a6bdf), LL(0x52e63728,0x68985297),LL(0xeec09362,0x953f2dd9),LL(0xf975e840,0x3f3a2694),
   LL(0x3edfaffd,0x77325f02),LL(0xff7d543b,0xc74420f8),LL(0xf9f15b53,0xb3944991), LL(0x0aac0254,0x86959454),LL(0x3b374e03,0x6f43d3e6),LL(0xb30e1624,0x844012c1),
   LL(0x65028b0a,0xd064ea02),LL(0x3b4d2ee7,0x806c09ba),LL(0x7fe4f230,0x8a1f8901), LL(0xeda1696a,0xbd498bb0),LL(0x3bc968a5,0xe51465d8),LL(0x2bd1dae9,0xd28bdafe),
   LL(0xf62ea381,0x65982b48),LL(0xcd73058f,0x4807eaa8),LL(0x240c2f9b,0x345875b6), LL(0x65a17c9e,0x778bc20a),LL(0x23054c23,0x6fe407b8),LL(0x75ba53f6,0xfae57b0f),
   LL(0x78d45156,0x32fbbcb8),LL(0x71e3fd03,0xaefa78d7),LL(0x0f74090b,0xd7c46ea4), LL(0xcc57cee3,0x89b357ff),LL(0x5840243b,0x9e5cedb2),LL(0x0d0e8ac0,0x98a48d58),
   LL(0x610ce375,0xcff03f53),LL(0xc9bfbd54,0x7b3c5728),LL(0xcd327836,0xeb868866), LL(0xae712c4e,0x79819aee),LL(0xd7f04c0b,0x0b11d0e4),LL(0xf8a5a59a,0x95bd124d),
   LL(0xcfda2cc6,0x924e7e79),LL(0x212e1d27,0x3ec19199),LL(0xe70982f2,0x596b8e05), LL(0x08cd6128,0xd664a1a5),LL(0xa30c9229,0x374d9700),LL(0xdea4ef83,0x44226abe),
   LL(0x49136355,0x9b6659c1),LL(0xc404424c,0x138972a0),LL(0x38bb5edb,0x75c55b59), LL(0x16af7574,0xcf1bc9c8),LL(0x3386b6af,0xaad6e8c6),LL(0x76fe803f,0x5d136ae2),
   LL(0x9d03f245,0x3912bca6),LL(0xc3a1fb1b,0xe6ff0efa),LL(0xbb5db166,0x7876e5c1), LL(0xf0b460b5,0x23c904b1),LL(0xe69528b2,0xee8e2bf6),LL(0x5191c468,0x2e8d3640),
   LL(0x06516227,0x980138d8),LL(0x861c3973,0x2b71e24d),LL(0x5ca35d8d,0x2580dd9c), LL(0x5c9c05ae,0xac4df0a9),LL(0x3c68c54a,0x63916a87),LL(0xf536c869,0x3a6ffdbe),
},
/* digit=20 base_pwr=2^140 */
{
   LL(0xe9cb9264,0x1012e415),LL(0x14ab08a4,0x4f91e85d),LL(0x35df1c1d,0x61431287), LL(0xe79281d7,0xbbd3824f),LL(0xaf54a4fc,0x99d4e30c),LL(0x642e1333,0xc4948701),
   LL(0xbae8e3bd,0xe93eda3e),LL(0xf46b38aa,0x4ef30da4),LL(0x4151dbfa,0xd6cab5b2), LL(0xcf5c0161,0x04fd2b14),LL(0x80166092,0x7c8a0389),LL(0xd94e32be,0x47096659),
   LL(0x6156b873,0xf7feeab3),LL(0x5be66206,0x4a2f2b83),LL(0xdcfc9038,0xdf3277f1), LL(0xf1d66fb4,0x9b170776),LL(0x076dcabf,0x25ebdd78),LL(0x38b46ab5,0xa7c6ee6f),
   LL(0x79f1a500,0x4cfc4785),LL(0xaf098654,0x3ba69756),LL(0xc21ecc8e,0x41491597), LL(0x0c8f0d93,0x4982c841),LL(0x2b324495,0x03c63423),LL(0x697188c1,0x7c3952ab),
   LL(0x56b9ce4d,0xe770f45f),LL(0xa9a47842,0x07148130),LL(0xced2b9b0,0xbc38458c), LL(0x047d7534,0xbe545be0),LL(0x1c6ff074,0x6768b949),LL(0xde3dbd8d,0x1dca8c16),
   LL(0xcc3bcbc4,0x0636cefc),LL(0x5159e643,0xc92b9ff8),LL(0x91bb0476,0x00a152ac), LL(0x42dea60d,0x3fc87a4d),LL(0x64d6fc10,0x6803f0e3),LL(0xf9f370c8,0x99409032),
   LL(0x5826fd73,0xc6616960),LL(0x5517ee73,0xf1955831),LL(0x51c70ff4,0x88b7cec6), LL(0x366d0443,0xb0888a9e),LL(0x3af869e2,0xa9181fd0),LL(0x0a248328,0x26076aef),
   LL(0xbaed8f78,0xad53d93a),LL(0xd71de257,0xa0b9be84),LL(0x603a33a2,0x4630f086), LL(0xb911d04a,0x5b44638b),LL(0x6e3fa849,0x02150fde),LL(0x4184d866,0x8421d2e5),
   LL(0xe5897526,0x52557cb2),LL(0x502c34ec,0xdfe0933e),LL(0x72eb3dba,0x4c1d921f), LL(0xc524a1ed,0x1989729a),LL(0x18b3826f,0xc702ac61),LL(0x0d5cd8f7,0xaa34f109),
   LL(0xfa609590,0xe4ef5cf9),LL(0x6f3d8102,0x215b5177),LL(0xccb2d3ca,0xb1886914), LL(0x65884534,0xcda19104),LL(0x870769b4,0x2cf8baf3),LL(0x7339dc8e,0x22f2d15b),
   LL(0x21a810cf,0x35c9c17a),LL(0xfb6bdd26,0x24cd74b4),LL(0x1a7c7f5a,0x6786a13c), LL(0x817609c4,0xec7e4bb7),LL(0x256f4592,0x6de1e630),LL(0x8326a507,0x923f4ca6),
   LL(0xb5bfac37,0xbe78c4a0),LL(0x197cb928,0xedbb3fc8),LL(0x8b563167,0x477ac821), LL(0xd224f360,0x70cb3ee4),LL(0x53d0e1fe,0xe1b338d6),LL(0xf10e26d1,0x7a5c3f25),
   LL(0x745f0869,0xb0c7846e),LL(0xc667523a,0x50becb24),LL(0x14521b55,0xc6acff6b), LL(0x0c4a3c6a,0x70ed855c),LL(0x8e159fe3,0x1ec2b9d5),LL(0x74b5c29a,0x06249cb5),
   LL(0x2dd180d8,0x0f3b2f65),LL(0x7b859a81,0x56200850),LL(0x4b18d95c,0x9d0f51df), LL(0x6f0b29af,0xe82ce6d3),LL(0x2a4310d4,0x75a0fcf0),LL(0xc7bed80c,0xdeebf877),
   LL(0x64d180d0,0xcf0d6afa),LL(0xa6434a9a,0xb8f7c1c9),LL(0x8cfb2948,0x6cdb576a), LL(0x863bb933,0xed57f57a),LL(0x4df11d72,0xc7f06882),LL(0x4c2b933c,0xe29a2fa1),
   LL(0xa17d0463,0xbba79104),LL(0x5a81b9f9,0xd7cbed07),LL(0x4e242aa9,0x20e07b74), LL(0xbb9e790d,0xb9d12856),LL(0x47c00f37,0x45bdbce1),LL(0x898b8efb,0xa0ed2b72),
   LL(0xb466dfb9,0xe8318ed6),LL(0x94a4baab,0x9faa662b),LL(0x051206c6,0x01503060), LL(0x3c19ae55,0xa2ebeefc),LL(0x84d2a9c1,0x210ae455),LL(0x82e43956,0x562df0b5),
   LL(0x17080cf3,0x89be703d),LL(0xf1ca3642,0xe83c11cb),LL(0x13271110,0xe61d20e1), LL(0xd21d05b8,0xd631b1d8),LL(0x6452ed8c,0x620fc76f),LL(0xe58a415b,0xcc6794e6),
   LL(0x15a94078,0x0a5c8a7e),LL(0x86688f1d,0xd4226082),LL(0xa85bc83e,0x8ede9371), LL(0x0df393c7,0xecd1f5e2),LL(0xd2fc7359,0xc32847c6),LL(0x08190efe,0x1e9fd4bc),
   LL(0xeea5827a,0x83dd7278),LL(0xcbc88ba5,0x1ddaadd7),LL(0x218e4199,0x588e4903), LL(0xc8310baf,0xb685d516),LL(0x74bb3deb,0xc9aa64ca),LL(0x931bdaa3,0x0cb7a0ea),
   LL(0xcdf5ecc3,0xc1ecf641),LL(0xf8c7946c,0xdeac2eff),LL(0x02ca92e6,0x73c8e009), LL(0x7361770a,0x19bc9700),LL(0xacc0e88b,0x455770d3),LL(0x7ddafde7,0xf7cd145c),
   LL(0x3d17e97a,0xa33b896c),LL(0x30a84be8,0x238a70a9),LL(0x044f68ec,0xb629bdd9), LL(0x5614b150,0x6d180693),LL(0x7fde6cd3,0x54ba0757),LL(0x4980c20b,0x19edd13b),
   LL(0x1073d079,0x4b1584e9),LL(0xa794787f,0x6e60302e),LL(0x8088839b,0xfeb8358c), LL(0x1a701ad3,0x86f51cbf),LL(0xd07f6815,0x61b9c974),LL(0x653f1425,0x00a0d546),
   LL(0x11894381,0xf75ced35),LL(0x319c7001,0x49e325d8),LL(0xa67066ad,0xd20b2435), LL(0x4c608028,0xdd23ebc4),LL(0xba2205ea,0x51d78928),LL(0xfb1fb5cc,0x737b9cb1),
   LL(0xc03586f9,0xbbb8ad72),LL(0xd27fea7c,0x402a9fff),LL(0x5dda8a10,0x831e331a), LL(0x1eb0456e,0x6d5a2c5e),LL(0x9f7b81ed,0x65290756),LL(0xbec1e59d,0x976b7327),
   LL(0x419baf59,0xea1a80a5),LL(0x57278748,0xbc517958),LL(0x32e85407,0x29245035), LL(0xc0d421ac,0x812b0397),LL(0xd7a01665,0xa9f235bf),LL(0x24652cd8,0x9d164236),
   LL(0x7f2c6c70,0xbf6ae2aa),LL(0xee9e0e91,0xa021d9a9),LL(0xf1766121,0x409464f1), LL(0x4a73446a,0xe180eb42),LL(0xc5f86d28,0xe7e1ed8b),LL(0x903583e2,0x2e4a2f9a),
   LL(0xc5e91397,0xba644637),LL(0xd74dad5e,0xfc63ab71),LL(0x824f5cb0,0x1baa70ad), LL(0x72783f79,0x5f89e95b),LL(0x3813fcb5,0x45bb2a10),LL(0xf3165835,0xab43fa7f),
   LL(0x14c22638,0xb7c86dac),LL(0xdae71747,0x3656ca1f),LL(0xc534e20e,0x85da5ae4), LL(0x49153a00,0x3aa49d12),LL(0xbd68b553,0x34857651),LL(0x47329cb6,0xe4fb4c67),
   LL(0xd2659f96,0xf91390b8),LL(0xae3ada10,0x507dcc75),LL(0xea42c05c,0xec3156e1), LL(0x28627de0,0xa851f6db),LL(0x3f5bd276,0x10f1771e),LL(0x29bb086a,0xda6fb38d),
   LL(0x72b97a47,0x9bdec367),LL(0x657aba3c,0x033db4d1),LL(0x0a26ba3d,0x6d7dcdad), LL(0xb13b21c6,0x93253eae),LL(0xf3e1e435,0xf49e4546),LL(0xfc1d9651,0xda2f3b67),
   LL(0x98a48009,0xf35a51b4),LL(0x74249dd0,0x6d03d805),LL(0xcb813640,0x3a2ea471), LL(0x8fb0ba2e,0xf40ccd7e),LL(0x967a4bda,0xe1511909),LL(0x9c401b43,0x92f26a91),
   LL(0x9b338eaa,0xc7019ac8),LL(0x1ff7bf17,0x9d084573),LL(0x565ad4f0,0x64807759), LL(0x0a7bdf41,0x1c42c251),LL(0x9e01efd3,0x63caad5e),LL(0xc624e5ba,0x55785f80),
   LL(0xee0bde78,0x7db8ae10),LL(0xb62c8387,0x0e6313ac),LL(0xcd166c86,0xd4ec5c51), LL(0x39448747,0x1a9c939d),LL(0x731d066a,0x3a75230d),LL(0x73962926,0x91bf93ae),
   LL(0x61e7a7be,0x1485af16),LL(0x71c2640c,0x112d3d53),LL(0xcbbb3b9b,0x277a24c5), LL(0xfa95ddae,0xa17a1980),LL(0x7a40ef8e,0x4ae12d96),LL(0xfb1c3fd7,0x339601a8),
   LL(0x74e8c96d,0x05fc1df3),LL(0x8ad8de59,0x004fd161),LL(0x747f9809,0x0ab94cbb), LL(0x341b687f,0xc689e96d),LL(0x03d6ec78,0xc3ed2540),LL(0xc677b947,0x58414ee5),
   LL(0x79d557f9,0xd4b38421),LL(0xb80fd2ca,0xe5f1e680),LL(0x5b2a650d,0x403029fa), LL(0x6ea522b1,0xbc0c8825),LL(0xa186a767,0xcaa2d945),LL(0x254875d3,0xb85ecf45),
   LL(0xf6031493,0x098fe2ee),LL(0x95f3b2dd,0xd8a4c0c1),LL(0xf565fbfa,0x84d7151f), LL(0x0d46e2ab,0xab92d636),LL(0x7fcbe589,0x822ee95c),LL(0xd061506c,0x855320c5),
   LL(0xd8c4c89b,0x82458f67),LL(0x4a55190e,0xb7f499f0),LL(0x2be3b154,0x8f90b016), LL(0x13d9c5d2,0xf943056e),LL(0xdf50c757,0x7de98cd6),LL(0xc1785279,0x97d69ee8),
   LL(0x3fe0c0ab,0x140bc9a7),LL(0x0452da51,0xbf26b0db),LL(0xdad75d1d,0x31658423), LL(0xeb0c6035,0x0f75fa69),LL(0xe800bb3f,0x4cbb16f7),LL(0x66429e7b,0x4892f0a7),
   LL(0x4c6c9c18,0x218c66a2),LL(0x08cad414,0x97355790),LL(0xb15e573a,0x3b5b1298), LL(0x0e6834ba,0x7aca61bf),LL(0x3f5ac90a,0x025883d3),LL(0x9dff4949,0x6462025c),
   LL(0x701e063f,0x6370b52c),LL(0xfcbed6d5,0x575b74fa),LL(0x54464ddc,0x87196ca2), LL(0x6aded252,0xabf56ac6),LL(0xe859df43,0x195e68db),LL(0xd3b0c609,0xd7d0b724),
   LL(0xc7d916d9,0x68fee7bc),LL(0xd2261a1d,0x7c342ab5),LL(0x2d528b45,0x6dc620c3), LL(0x0ae80fdf,0x74e628a7),LL(0xc87865c1,0xb6024eaa),LL(0xb87c19a5,0xcfbcc11e),
   LL(0x7fc82526,0x16e10271),LL(0x4e53a7eb,0x1894ff31),LL(0xd6fd2213,0x483dd45d), LL(0xe86573ca,0xf832f47b),LL(0x93e56334,0x3fba3845),LL(0xedd7db47,0x88af40db),
   LL(0x48c963ee,0x478c4ff6),LL(0x41711eb6,0xe3c63604),LL(0x40ebd5fb,0xdc23721b), LL(0x38c5e8a3,0x73b730cb),LL(0xa14de57f,0x379e6a4b),LL(0xc789f412,0xed816ee8),
   LL(0x0ce5eda0,0x22553add),LL(0xae4119b5,0x2cced448),LL(0xaa185ab7,0x2a352ea6), LL(0x29a2bdee,0x01bd04b4),LL(0xe26e2a0c,0xf76ade92),LL(0xd94aaa8d,0xda8577d6),
   LL(0x07d56f07,0xc7f17e2f),LL(0x7d87d713,0xc5e3f9d2),LL(0x20f51bc9,0x1497d098), LL(0xed2a6960,0xf4d19f47),LL(0xc38a9534,0x1c57c5ec),LL(0x18c80df7,0xdbffd9a1),
   LL(0x5f168647,0xf0cc9844),LL(0x1593d662,0x2e63ed9f),LL(0x0ac8146a,0xf5fa425d), LL(0xce76be64,0x67f11299),LL(0xf7727692,0xd42902cf),LL(0xf1f0d4a5,0x06ce330a),
   LL(0x799a13c8,0xfd425657),LL(0xcf29d9a8,0x2c8a973c),LL(0x1e78973e,0x40ec7ae0), LL(0x78699406,0x4d9856a5),LL(0x08dfdad4,0x7247ba62),LL(0xa468c80b,0xc9427c34),
   LL(0x3aeaaf13,0xe79cd000),LL(0x9ec3462c,0x6018f62b),LL(0x2ace3a5a,0x1192c3d9), LL(0x64a64ae3,0x715b684d),LL(0x34bf6e70,0x9d318356),LL(0xfdc656ed,0xa23a9958),
   LL(0x77357c1f,0xe6acb14b),LL(0xe9438466,0x554e9c51),LL(0x8ba9a286,0x97a5abd9), LL(0xd3eed3bb,0xfc1cd0de),LL(0x700f6eee,0xf311fa4a),LL(0x8a99cd1d,0x4dd6d73d),
   LL(0x6b41a2bb,0x43f68039),LL(0x083ab1a3,0xba3ca572),LL(0x1b173556,0xea841d92), LL(0x928c1fd4,0xb23978ab),LL(0xb985008a,0xd6d7c468),LL(0x2248ceb2,0xa6d40f4e),
   LL(0xb9ad7efa,0xc9627a3a),LL(0x6759753d,0xbebc0a32),LL(0x7cf0a0fb,0xa3bacd1a), LL(0xdb7cc4c0,0x4ac922fd),LL(0xd7afe1c2,0xfc69f5fa),LL(0x7afcfb11,0x6dde5c15),
   LL(0xbbd42505,0x9df0ef70),LL(0x464d44cf,0xa4274df4),LL(0x57e809eb,0x8113b3de), LL(0xc10dcf7e,0xaeab98c6),LL(0x1e6f2975,0x58a34894),LL(0x8f9ccf40,0x9cb5ea9d),
   LL(0xb579d154,0x1c4eac89),LL(0x4b706046,0xafbc0bf0),LL(0x7f6fc44a,0x5ecdfe60), LL(0x0030d612,0x3d9c58e8),LL(0x7811ca4f,0x3d9b1e58),LL(0x78e80d2a,0xd3ace80a),
   LL(0x48e70c10,0x5a2c98a4),LL(0x1f1028cf,0xa72ad42e),LL(0x2fe8c440,0x30739ff7), LL(0x5fe64d92,0xe83f9edf),LL(0x2fd3764b,0x7cb62281),LL(0x51ab555f,0x9f114c85),
   LL(0xba081442,0x3a838e28),LL(0x27db39bd,0x71ea2048),LL(0xaf2896ba,0x3cdb99dc), LL(0x962c484b,0x00d6fcb4),LL(0xcd7377d4,0xe72a7bcf),LL(0x5164a634,0x3b6505a3),
   LL(0x8bddb5d0,0xaec6f498),LL(0x7f02675d,0x92b09fb7),LL(0xa83cefd0,0x79345be5), LL(0x1f828e29,0xea0f3427),LL(0x7cadddfa,0x1ea0786e),LL(0xd159e1b8,0x75ca82d5),
   LL(0x62c0533a,0xaaaf5a5c),LL(0xe510bc9f,0xa96e494b),LL(0xf2debee9,0x46b1aedc), LL(0xf98911db,0xedd51f34),LL(0xf9056ce1,0x83a0eab8),LL(0x7eae269d,0xa541abff),
   LL(0xd5d26026,0xb68b8dce),LL(0x28bf2023,0x6ceb559f),LL(0x74bc843f,0x7dd76130), LL(0x35f97a24,0x79c99ea9),LL(0x7b478eaf,0xcc786699),LL(0xc10dc8c8,0x7788ee3c),
   LL(0xbcba6716,0x88ad66c4),LL(0x95300687,0x2bdf72c8),LL(0xb8fa1b0f,0xdd8629e6), LL(0xe183a10a,0x93d5a8e4),LL(0xae5d5956,0xe3e2d85f),LL(0xddd1b643,0x8b623a95),
   LL(0x8cc90e24,0xfaf59c8f),LL(0x7f1c7245,0xb7173e32),LL(0x9baad443,0x568e30bf), LL(0x718cbe51,0x7eb7c7cb),LL(0xc1f0f949,0x2a3a214d),LL(0x96a84ad4,0x21a777be),
   LL(0xb2081c31,0xcdb78081),LL(0x7a9d57b5,0x2bfa8d65),LL(0xb750969c,0x9db52d99), LL(0x645d4012,0x6ebcd855),LL(0x86245e59,0xbcd2fcd1),LL(0x812b3b4b,0x8fff22a6),
   LL(0x6ffb620b,0x1130ffec),LL(0x22eb38ff,0x279544f9),LL(0x940699a7,0xaeb474fd), LL(0x4c85071c,0x33329bf0),LL(0xaf3d6bd2,0x56b019ec),LL(0x8b3cf686,0xc4aaaf7c),
},
/* digit=21 base_pwr=2^147 */
{
   LL(0xac2ec7bf,0x8824e17f),LL(0x9bc85814,0x131057c0),LL(0xdbe08716,0x7e43698b), LL(0xab93ba96,0x42ad5ab0),LL(0x7c518be6,0x1ecf5fda),LL(0x673d6c18,0x1a02e6f8),
   LL(0xb3503cee,0xbf0f89da),LL(0x331d9e6b,0x9169467b),LL(0x3b1900fc,0xe3aa5241), LL(0x3f13a573,0xae45a71e),LL(0x1940c9af,0xec133c7c),LL(0x55c2087c,0x0409942c),
   LL(0x24e5717c,0xbac3bea8),LL(0x9b83ae02,0xebbc708e),LL(0x7e9ea1f2,0xfa57a0ad), LL(0x704d566a,0x107a38cd),LL(0x85e59024,0x18e24faf),LL(0x0f84a27a,0xec3157c2),
   LL(0x58b3bb0c,0x17889e93),LL(0xb4031a98,0xbaa2495a),LL(0xb10778db,0x5d3a3a63), LL(0x8f7c139d,0x2fd443d8),LL(0x5ea5ad03,0x83c71634),LL(0xce562cfd,0xaba81b6a),
   LL(0xd98102aa,0x73e43632),LL(0xc1c41614,0x86932d35),LL(0xeec41de3,0x1c66a427), LL(0x72427be4,0xef9616dc),LL(0xd48ea6d2,0x206de291),LL(0x8160c0e5,0x755f3ab3),
   LL(0x376fc02d,0x23668837),LL(0xde1f643f,0xc4608c43),LL(0x05775439,0x6f04d8d7), LL(0x0f5452a2,0xb32fa6bc),LL(0x3b09e219,0x6a200395),LL(0xfb0a3adb,0xf58e4d28),
   LL(0x5f9f2818,0x56285629),LL(0x594adc3c,0x19ba097b),LL(0x0d5212fe,0x6a73f84f), LL(0xb0680c11,0x5106510c),LL(0x13904ea1,0x4d287906),LL(0x8ed4bb25,0x9e4b6db5),
   LL(0x3efd07b3,0x5c58a701),LL(0x8a62c9c4,0x5569ea8b),LL(0xe3eb4a21,0xbabcab23), LL(0x2ed936e5,0x9f05b68c),LL(0x25d25b7a,0x5b8fcbe4),LL(0x40745f94,0x5104d275),
   LL(0xe96ebb1c,0xbf9a348a),LL(0xadcee8f6,0xde39269a),LL(0x1891395c,0x92012c94), LL(0x6584db0f,0x3b71f247),LL(0x8f7e3a23,0x9fc90016),LL(0xbe7d1077,0xb5f10362),
   LL(0x137efed8,0xf06fe20d),LL(0x096f2e2c,0xf037f990),LL(0x416e7f9e,0x7e828e25), LL(0xd17229cc,0xb9066cdf),LL(0x74e2f3cf,0xe1b5d7b4),LL(0xcbc63300,0xa21dc884),
   LL(0x73213695,0x09291e46),LL(0x7d591bb2,0x599398e7),LL(0x8693f17f,0xbc316181), LL(0x323b2584,0x29e970f3),LL(0x8253ea0d,0x07db9ebe),LL(0xaf8d9499,0x62d8e321),
   LL(0xce2b6bb7,0x468f741b),LL(0x423f1cec,0xb8703663),LL(0xa1385cee,0x045ddc8d), LL(0x9b1abc32,0xfb61bc91),LL(0x8b0a1f38,0xdc221abc),LL(0x5f8c7e7c,0x174b6f77),
   LL(0x8aedb4c2,0xf975efc4),LL(0x150dc348,0xb7126f5b),LL(0x8f316668,0x1826a434), LL(0x47b0020f,0x89a0a561),LL(0xddf906f1,0x7c814d16),LL(0x5cbe397c,0xae729976),
   LL(0xac73b7b2,0x73507bd6),LL(0xd77d1311,0xdc109e64),LL(0xd07e0b84,0x40745a92), LL(0xeae6ea71,0x80a25d92),LL(0xdba1caf3,0x7d16cdaf),LL(0x178d162b,0x532678a1),
   LL(0x89ce3929,0xf04f0f69),LL(0xfccffe8c,0x623a881a),LL(0xc3cc5b57,0xe35d518d), LL(0xd7cbfd70,0xcddb9b25),LL(0x517679db,0xcec3736a),LL(0x3964a183,0xff07c63c),
   LL(0x1ee7af47,0xb6f1e8e5),LL(0x774114f6,0xdae8bedd),LL(0xe9b23727,0x7e40fd87), LL(0x4e0b09b4,0x677a0144),LL(0xecabbeea,0x37b376fb),LL(0x05a3a145,0x3d536296),
   LL(0xf07be04d,0x0b60e86c),LL(0xb0f7efb6,0x72210de8),LL(0x44de2248,0xe661f7e3), LL(0x5e1a2e6a,0xa50972cd),LL(0xf3c16459,0x8ea7ea6c),LL(0x3d5cc6fa,0xce1eac26),
   LL(0x3fb24e2e,0x12057921),LL(0xe703d891,0x2c6418a6),LL(0x85a84ce2,0xba1fdc36), LL(0xd7969f76,0x15c19ded),LL(0xce8a554f,0x0ef1f54f),LL(0x9dae18ed,0x45215796),
   LL(0xea05050a,0xe4fb6484),LL(0x249b636e,0xf0ac280e),LL(0xa3b0a746,0x622dc53a), LL(0xc2c4b6e2,0xcbd0132e),LL(0x608cf966,0x71f38ed9),LL(0x47ac6ed1,0x733cdc1c),
   LL(0xedbd5f2c,0xeec38461),LL(0xb053878a,0x02367abe),LL(0x366f0019,0xbc7279d2), LL(0x3917f0f2,0x383b04a6),LL(0xd170c6d6,0x7a5ac16b),LL(0x6e32c8f9,0x4f3c4a0e),
   LL(0x0bd61a8f,0xcc2c1858),LL(0x34f527e4,0x87c8a389),LL(0xf23dc622,0x618a666d), LL(0x364108ca,0x40fecc5b),LL(0x98a0bf79,0x29742c4f),LL(0x272071ba,0x37d45180),
   LL(0x685f2d3f,0x1625e4f4),LL(0x14c1f55c,0x3470dfe8),LL(0x44d91a5a,0x969f3ad2), LL(0x9f30454d,0x6a5be95e),LL(0x7281af65,0xed104a4e),LL(0x36d5ad8c,0xb23076ad),
   LL(0x5eb808b3,0x4ed6c763),LL(0x60e8d7fb,0x70f86e24),LL(0xde84f30f,0x2b18c950), LL(0xfaf0ac4d,0x435aaafe),LL(0x3276078f,0x6f3e4e4e),LL(0xb8977a67,0xf3b9b7dc),
   LL(0x3bcf4e15,0x28ef1bcf),LL(0x358303c9,0xbe939b53),LL(0x21a319d2,0x204114a7), LL(0xaa81d7a8,0x259ed8fa),LL(0x18fe7097,0x7d9a642b),LL(0x184765b8,0x9acf2e2a),
   LL(0x9149e0ef,0x77900481),LL(0x8dfdc1e8,0x0ae5deb5),LL(0xd6f33c8e,0x23b0cfeb), LL(0xa26da632,0x9a6ca40b),LL(0xc537ce62,0x34b9c619),LL(0x5a4ec3f7,0xd37f4428),
   LL(0xcbf87088,0x16d03a12),LL(0xb1c521c9,0x872be585),LL(0x1e8e4fc6,0xff44095d), LL(0x5c32815d,0xa7e6bc74),LL(0x8d3db724,0xdf3b021b),LL(0x101b1403,0x6b363726),
   LL(0xbe2d31d8,0x4cceb610),LL(0xa171cad9,0x4b4a33ca),LL(0x37cd3a7d,0x8e335bfb), LL(0x5185da89,0xa4922968),LL(0x5914cb63,0x45f929dd),LL(0xa3aa3f6a,0x040366eb),
   LL(0x2d373ab6,0x2ae211b1),LL(0x671cfd45,0x81bd0544),LL(0xcb962cdb,0x0c2e8457), LL(0x64ce8ebd,0x58fa7e29),LL(0xcdb6f3cd,0xfdc40b43),LL(0x1c69bac1,0x17d10e47),
   LL(0x86421e7c,0x292c5ae4),LL(0x6758a6b1,0x32584cff),LL(0xaf7a9c3f,0x0288b16a), LL(0xeba67e24,0x03aadb8c),LL(0xc87df2d4,0x0f3e777b),LL(0xbb91d9ea,0x6ba2d690),
   LL(0x404ff38f,0x4a5352c0),LL(0x5340d952,0x0f04982f),LL(0x2798c5c2,0xcfa86bde), LL(0x94dd1186,0x1ab8b19e),LL(0xccfd8af2,0x83ce68ee),LL(0x06cb2c1b,0xb6f99034),
   LL(0x6ecd5c6a,0xadfb667e),LL(0x59a0fb16,0xb5401239),LL(0x5b9e22cb,0xe1100936), LL(0x21d0430d,0x34d54f0a),LL(0xa6197440,0xcb34b278),LL(0xc927d3c9,0x1c07cf85),
   LL(0x31e24e54,0xa386ba5c),LL(0xa48e552b,0xdfa72d1c),LL(0x075794d0,0xcb7594cc), LL(0x0b3dc6ce,0xa52aceb3),LL(0xc6d190de,0x16c80954),LL(0x203a1303,0x7cd54e57),
   LL(0xd5313dd9,0x72983edf),LL(0x4bf35d2e,0x7557de74),LL(0xfde38dd0,0x29719af0), LL(0x6838b644,0xe3ce6ea4),LL(0x39486a01,0x70976ef8),LL(0xb7d8d84d,0x7b16d6b6),
   LL(0xc80b2253,0x44cffea0),LL(0x53e978f5,0x39b12203),LL(0x76e654ca,0x6efe5c6c), LL(0xf5ecd362,0x9ce55175),LL(0x32904094,0x945bc283),LL(0x419b4c47,0x16be694e),
   LL(0xb25ff2d8,0xc6d695b8),LL(0xfbc1c346,0x95848769),LL(0xf78df5b6,0x4cd68253), LL(0x07a395a6,0x366208a9),LL(0xb300e26e,0x6a100464),LL(0x2c4c9774,0xcd75f9ef),
   LL(0x58587dee,0xa6dde508),LL(0xaba27af9,0x1fb5756e),LL(0x2435fdfc,0x886588cb), LL(0xafa3a40b,0x8836939e),LL(0x23550a04,0x8e4ea6e4),LL(0x13129070,0x1f60a305),
   LL(0xafd87664,0xcd87217f),LL(0x62eaab37,0x33999750),LL(0x63e77715,0xc5c71386), LL(0xadc2dc58,0x8f7ec765),LL(0x06ec10e5,0x0166d81b),LL(0x4bab2eec,0xd978b136),
   LL(0xabc9d990,0x518d8792),LL(0x848bc38a,0xb5e9214e),LL(0x890c38ca,0xbc62bb10), LL(0xa5d74a39,0x24395eb5),LL(0xe81ca0ef,0xbaacb032),LL(0xf968482f,0xaa7d10d1),
   LL(0x3c8d1134,0x55fb3866),LL(0xc866fb41,0x2f0c11d9),LL(0x60d9da04,0x470f0d0c), LL(0x30199fda,0x2ce76abb),LL(0xf1754859,0x68239b6e),LL(0x36c8db28,0xabcbfa61),
   LL(0x15fd8cdd,0x59bc69c7),LL(0x7dc71686,0x8c35acbe),LL(0xdf56f4b5,0xa864fdf4), LL(0x73cb6096,0x7b019a3f),LL(0x6d0212a9,0x9bec6ec0),LL(0xd403dbf3,0x96f43d54),
   LL(0x25f69dd1,0x1f8ef8de),LL(0xd4c811f4,0x01ea5842),LL(0x7bd1206b,0x6b00713a), LL(0xc77c3c64,0xcc267e59),LL(0xe68a4f20,0x788a8c0d),LL(0xf4b3c37a,0x968e777b),
   LL(0xf39053e8,0xef230e41),LL(0x7839732e,0x3acdd9f9),LL(0x815c2a4c,0xa80f263e), LL(0x544f6fe5,0x20c61e68),LL(0x2c146750,0xd0a8228e),LL(0x4770e4ae,0x30857c49),
   LL(0x42a4aa18,0xcba4107a),LL(0x68f36209,0xe72e3f7e),LL(0xf0a19740,0x42a02b75), LL(0x9946daf5,0xd4783068),LL(0x862f2de7,0xc01f64b0),LL(0x6209f44c,0xf00cfcca),
   LL(0x7beb13fe,0x69d1c1f7),LL(0xfa01681f,0xf547c381),LL(0xfe808e67,0xe7fbdc0a), LL(0xc9b56b2e,0xa0fe9ead),LL(0xb2682ce9,0x1c53a580),LL(0x556a1034,0xe6cae9a2),
   LL(0x7ff5d034,0xf4497140),LL(0xd0b40f9d,0x95b5f6f9),LL(0xc68600e2,0xec27b942), LL(0xdbb02501,0xd8f6e734),LL(0xf3ced49b,0x1e62a674),LL(0xd9620b28,0xd64cb872),
   LL(0x418878cc,0x4506163a),LL(0xa5575e5a,0x96d1754d),LL(0x9a25f013,0xbf853bbe), LL(0x03c3aa89,0xfd8c5194),LL(0x14a549d4,0x0eb1954d),LL(0x0dbe140e,0xa88b972e),
   LL(0x476facb7,0x5b45ac78),LL(0xcb7dbe77,0x8dc145b4),LL(0x4e47ee7a,0x0dd76bfc), LL(0xc1569361,0x06f554cf),LL(0xf665649a,0xbd841afa),LL(0x6e9773f4,0xe27421d7),
   LL(0xe76c5baf,0x65ee7fbc),LL(0xb8dfe92d,0x6ddd2b14),LL(0xda22cc76,0xb9f4b2b4), LL(0xd9c02353,0x36ffd8da),LL(0xb62c010e,0xc23f7f57),LL(0x3308d355,0x00c51af6),
   LL(0xa8fb84f5,0x7cdc4353),LL(0xbd3b9efa,0x431a25d9),LL(0xe921fd8b,0xe357451e), LL(0xa7d0d22c,0x3607231e),LL(0x7ebf0d0b,0xd8d0370f),LL(0xc5ac6f7c,0xfdbb3c4f),
   LL(0xc826f1d9,0xfaa73e4f),LL(0x3e954cb4,0x95f6b20c),LL(0xc5935901,0x56f618ef), LL(0xc1141c9f,0x3a55c846),LL(0x27edf041,0xbc5bd1dc),LL(0x0e373045,0x51933533),
   LL(0xb5f4dca2,0xacf42b35),LL(0x7eac22dd,0x0eedba21),LL(0x75502f43,0x2945fad0), LL(0x04c50653,0x94995ef4),LL(0x373f4e93,0x7992d051),LL(0x1b1c94b7,0x8a7c3436),
   LL(0xaa607973,0xe636537b),LL(0x4e2f86bb,0x1c845ea2),LL(0x81f6999a,0x99f38921), LL(0xfed9173e,0xc1716d20),LL(0x1e801b36,0x792b9ffc),LL(0xca8f7a6b,0x06765f7d),
   LL(0x79b19761,0xc6771baa),LL(0x29433d11,0x80d67617),LL(0x6b423d37,0x196fe0e0), LL(0x47bec8f6,0xe752e3bd),LL(0x52fbbe1c,0xd63ec3fc),LL(0x2ebe6309,0x30d727b9),
   LL(0x1d17b14e,0xd9f486d8),LL(0x6761b05a,0x47c47d53),LL(0x7041158f,0xdf1f5e86), LL(0xac78e6bd,0x626756b3),LL(0xd777224d,0x01aedca6),LL(0x4a443a1f,0x025d3c43),
   LL(0x5634214b,0xa12d740e),LL(0x375af247,0x4bc99161),LL(0xbe3e2882,0x346bf9f0), LL(0x4211297b,0xdad14a72),LL(0x1c532542,0x6554266b),LL(0x15d18e75,0x030daf54),
   LL(0x51583335,0x7cdc99eb),LL(0x4e821ee0,0xd2239bce),LL(0xb41ded13,0x8974b1a0), LL(0xfe676f40,0x17dcd329),LL(0x9d22a9f9,0xc3253117),LL(0x36c57e47,0xdd5a6256),
   LL(0xeb7827f5,0x04fcaeb7),LL(0x97baafab,0xcbba55a8),LL(0xd29a034b,0xccd2a2c8), LL(0x34045d21,0x37a19e21),LL(0x20ebeb66,0x1319ad06),LL(0xaa2041fe,0xfed9eef7),
   LL(0x5ef2c6ce,0xa3fa82ba),LL(0x60524b3f,0x764801ce),LL(0x714ec445,0xdcc5c27d), LL(0x885e5fc2,0x5cdbaa30),LL(0xc779e89b,0x0aa6c9a8),LL(0xb797f8a8,0x9db3b836),
   LL(0xa3ddb5c8,0x8c8b5a06),LL(0xc17aa963,0x1904c79d),LL(0x4a40b438,0xd6e2cd45), LL(0xa689e7d5,0x90b1ed27),LL(0x3f834964,0x350a4374),LL(0x1d9e5301,0x4c0cbe62),
   LL(0x193559c9,0x9959ff0e),LL(0x3a77311c,0x0a6908c4),LL(0x714d000c,0x93d61072), LL(0xe471b9f7,0xc054c8c1),LL(0xfb602066,0x78534fd5),LL(0x614fcf16,0x9a5f7b8b),
   LL(0xc459f1a8,0xd1d42f2a),LL(0xb574b235,0x9403a4ae),LL(0x42ef8a7f,0x1335a414), LL(0x6a4e1b79,0x4eb9a067),LL(0xd7a319db,0x90dd6eca),LL(0x8cfa5579,0xd54bf9e1),
   LL(0x72331044,0x063b4a7e),LL(0xc4e13d1e,0xb82357d4),LL(0x38e9b37d,0xfb4981b7), LL(0xee950c17,0x0c64d898),LL(0xa553cc9d,0x16ac1d7c),LL(0x6e7d9643,0x27624002),
   LL(0x62dc7931,0x70e5badb),LL(0x3d4149b7,0x92048bb3),LL(0x626d709c,0xa45d15d5), LL(0xe83bc3f4,0xc1e9c751),LL(0x07a26fe9,0x35bb7bd8),LL(0xb8fee30d,0x2713c9e1),
   LL(0xc4895195,0xb26a41dd),LL(0xbec62d17,0x3695b237),LL(0x5bd95196,0x272d55e9), LL(0x89b5e05f,0xcb659d58),LL(0xa5687e17,0xcba46e22),LL(0x623cf306,0x2ece876a),
},
/* digit=22 base_pwr=2^154 */
{
   LL(0xce363c03,0xd16b25a3),LL(0x69b561cd,0xe21621f4),LL(0xabf845bf,0x03a8e393), LL(0x45c545c4,0xf68a5989),LL(0x492c1276,0x0d669f1b),LL(0x239a2436,0x762cdc78),
   LL(0x4e33515c,0x32e5a4ad),LL(0x107349fc,0xfaf120c4),LL(0x42b588b2,0x18f529a5), LL(0xad41f013,0x53afbf6a),LL(0xfce98199,0x7e107d23),LL(0x49fc022a,0xb3e6ca6e),
   LL(0xf43c63d0,0x2f0fa15f),LL(0x3c5f1062,0x2ee7972b),LL(0x1251d981,0x217bc709), LL(0x4f44f67b,0x869cc500),LL(0x6e74d49a,0x410ce8d8),LL(0x295b72ee,0xa2668d2e),
   LL(0xc2e1ab9c,0x377d1c96),LL(0xf02f335e,0xc4d44912),LL(0x17483b67,0x91a1b453), LL(0x0a35ed35,0xd87bfb05),LL(0x06388074,0xda51aa7f),LL(0x3bc985d2,0xe8eded68),
   LL(0x62093566,0x53a65438),LL(0xd40d4ada,0x1a184bde),LL(0xb1ce835f,0xcf398a06), LL(0x2e3db6b2,0x3e0368f2),LL(0x768efcf7,0x5b1e672f),LL(0xce1191fc,0xcde23025),
   LL(0x1965b94c,0x7d18a84c),LL(0x42798e4d,0xa92514b7),LL(0xffa944d6,0x6df7a966), LL(0xc415a2b6,0x7c9cf75d),LL(0x0048e5ed,0xac0c7c55),LL(0xe02f7dae,0xd5c826e6),
   LL(0xf728251a,0xee934c6f),LL(0xbd48f9f8,0xb7ec512a),LL(0x2ec83189,0xc5f704a8), LL(0xed82fce0,0x3c184301),LL(0x378c8f4d,0x8cbc0d6f),LL(0xa97c2ab3,0xc0ff9d1b),
   LL(0xd18c264e,0xa6c6c290),LL(0x768df841,0xe5061160),LL(0x519df852,0x7913aaad), LL(0xf67253ef,0x61958563),LL(0xd45ffadd,0x36c367f5),LL(0x9e16d8cd,0xf1b45f01),
   LL(0xec831ea0,0x3d1cecd3),LL(0xfa15690f,0x2bde5c72),LL(0x38b353ee,0xba0efaa9), LL(0xe8d4b6ee,0x15f17a68),LL(0x954159dd,0x9699e58a),LL(0xe8f17b7a,0x819578f2),
   LL(0x61ee7443,0xc7eeaf29),LL(0x1b16a9dd,0xfa9c2747),LL(0x45e6c8f4,0xf723daa8), LL(0x80e39749,0x2c263cb4),LL(0xe1796ef8,0x69b29a11),LL(0x41cf69fe,0x71b39ec0),
   LL(0xc93aacf4,0x277c0653),LL(0x90aa54e4,0xe7cedf79),LL(0xfbb3e9e4,0x5f3dacd8), LL(0xd0335cd7,0xebf8b3d2),LL(0x54b9f51a,0x320526dc),LL(0xda07d715,0x670feaee),
   LL(0xc815957e,0xed31303c),LL(0x9dc47dbd,0xe3fd01ce),LL(0x7796d887,0x69884abe), LL(0x91801969,0x5eeac021),LL(0x1fede637,0x3f7f3b0a),LL(0xb577b6b1,0x94177a5d),
   LL(0xeab50a36,0xffccab96),LL(0xc61d85f9,0x0d130c6e),LL(0x7da399fb,0xef804dc2), LL(0xa18f07c8,0x24faa196),LL(0x04fcf0bf,0x64bb949f),LL(0xd5aeb209,0xb1fcfc87),
   LL(0x32651bcd,0xf00f320d),LL(0x06c32d6c,0x9df50f13),LL(0x5f7039a8,0xe5521172), LL(0x637e1a26,0x1ea8cc96),LL(0xe1edd4f0,0x2b163456),LL(0x0a1c1b35,0xf5bf574c),
   LL(0xe6a6c9d7,0xd11076c0),LL(0x892f314b,0xc2f5ad9a),LL(0x5737733a,0x1ce795fb), LL(0x7ff3d547,0x2ea960a2),LL(0x1b76089a,0x44189130),LL(0xa3a6506c,0x6ef7149a),
   LL(0x1032f79c,0x5903c244),LL(0xdcc0e2de,0xf21d72a3),LL(0x1762ce11,0x9ba803fb), LL(0xca5ae539,0x76e31b74),LL(0x9fd4f343,0x4e502c66),LL(0xd2eaa2ab,0xb6a9a1fc),
   LL(0x073efd81,0x5888451f),LL(0x9927bf88,0x398711b5),LL(0x018be818,0xbb190a8b), LL(0x79e61240,0xc797d10a),LL(0x3beaa53f,0x8696dba6),LL(0xcfb7766d,0xb542db1a),
   LL(0xd903933d,0x6f4e1530),LL(0xcd7c2795,0x12359176),LL(0x5a5fca32,0x755f299a), LL(0xc4a38b20,0xd72339f0),LL(0x842195c0,0x375cc0c0),LL(0x30d23f6c,0xbc55da2b),
   LL(0x9aac817d,0xce0ba5f7),LL(0xa1a45388,0x3b669115),LL(0xf409cf17,0x1dedbc8b), LL(0x3ffa552b,0x31a461a0),LL(0x382ba937,0x86758b16),LL(0x795dc73f,0x4a479210),
   LL(0x48aba8e6,0x83c44b08),LL(0x62517119,0x17c26d0a),LL(0xd01e1f9d,0x5b018ac9), LL(0xe9efcfe5,0x4397fd63),LL(0xfe0829a3,0xca651042),LL(0x13eb60a0,0x46cee0c3),
   LL(0x8600caea,0x27529486),LL(0x35717139,0xd38810c6),LL(0x41461bde,0x63178e7e), LL(0x045c484c,0x227b3172),LL(0x97348e92,0xbcfee10d),LL(0xa51d3833,0x3fde1f78),
   LL(0xf10b4b60,0x50b3debe),LL(0xa5ad595f,0x7901e0b9),LL(0x47e58c1f,0x87df2160), LL(0xfa209201,0xad4fd6ef),LL(0xa55d0556,0x9335bbb9),LL(0x00fee680,0xe8d815f3),
   LL(0xd99bdf5c,0x3e3ecfe1),LL(0x9bc2df7b,0xcf690949),LL(0x102d61ba,0x7e6008b1), LL(0xe00329c7,0xc1283828),LL(0xeb7b35fd,0x5b105f4e),LL(0x3dcb3985,0x9509ed61),
   LL(0x81c63288,0x045a1445),LL(0x76cc9ac1,0xa4e7b140),LL(0x48acdd5e,0xa2f07d10), LL(0xa2b81e15,0x90c45e5f),LL(0x308acd63,0xa933ffa1),LL(0xd7e46f9d,0x17365897),
   LL(0x466f886f,0x8852a2bf),LL(0x4d14f3d8,0x4ace368b),LL(0xe440d441,0x8d94e2a7), LL(0x72866999,0xfcb3c915),LL(0x1b47c102,0x9760bf19),LL(0xe720c285,0x6611674e),
   LL(0x11d915d4,0xdcb98a2d),LL(0x6beac9be,0xf990fa0f),LL(0x6dd7048a,0x681bfe50), LL(0x689f0ae5,0xb41ece47),LL(0x44ff0641,0xe6fdbd83),LL(0x0e9753dd,0x42e0fbcc),
   LL(0x3ffa710c,0xc7877d3d),LL(0x9919eb3c,0x7f0c21c6),LL(0x290c9014,0xc85786a5), LL(0x5689cb11,0x093aae39),LL(0xb5d01c9f,0xfb777943),LL(0x8cebd9c1,0x319aaf33),
   LL(0x1a2fd277,0x902b0342),LL(0x9af2dbda,0x0912afff),LL(0x48589a78,0x15e0d236), LL(0x1fa3327f,0xf8b35812),LL(0xec713d1d,0x8990816e),LL(0x8e343f2d,0x5e1a860b),
   LL(0x3614d67e,0xf61322f7),LL(0x29935fae,0x4d6b23b9),LL(0xf4a58207,0x5c8bdcc6), LL(0x9265fa5c,0x750d615a),LL(0x40ca78a9,0x2022ff1f),LL(0x19732ba7,0xd9585db6),
   LL(0xf29d3bbc,0x694abdf0),LL(0x24a568a9,0x03a757fe),LL(0xeb85bcf9,0x69e64a5f), LL(0x3a6d438e,0x11c2527c),LL(0xe22585b6,0x4aa39cb8),LL(0x20f51f14,0x696f234b),
   LL(0xcd374897,0x796f3388),LL(0x747f55ca,0x9cae05cb),LL(0xc3244db7,0x0178bf8f), LL(0xcad92264,0x2cf6b0b2),LL(0xd9851a5c,0xa5eaffed),LL(0xe5613e1a,0x677ef509),
   LL(0xb8a7827b,0xd774ccd5),LL(0x30c2fdb5,0x1a9035d8),LL(0x634655f8,0xeed95b80), LL(0x56b0b0bc,0xc2b700c0),LL(0xc9f0aa76,0x37259c87),LL(0x4c9ae4a6,0x7f6b6b1a),
   LL(0xf72a765b,0x23bfe902),LL(0x838d2dcf,0xf220ca7d),LL(0xfa8c5fe9,0x6aa1a258), LL(0xdd1fbec2,0x80a3ac57),LL(0xbfd1c7fa,0xa9d70cd6),LL(0x997ee6ac,0x6c5953bc),
   LL(0x6a40f84b,0x3b3747a2),LL(0xb83105a1,0xfd1a969b),LL(0xbbcb761d,0x3c5c4b12), LL(0xda6bebae,0x7c7f4e65),LL(0xa14f1495,0x7a4fdbf2),LL(0x49b37a93,0x7f59536b),
   LL(0x3240d78b,0x0c50a5f1),LL(0xb32a0cce,0xb51ad026),LL(0x53101002,0x4ccf9cdb), LL(0x0073f0a9,0x2fa22f3b),LL(0xbd1d7831,0x3a7fe366),LL(0x740322c6,0x5f31b6ce),
   LL(0x013aca29,0x5cbc9486),LL(0xe542ffae,0x03e60edc),LL(0x8457faa5,0x80539a8f), LL(0x6355b3b2,0xeba394b1),LL(0x2249fcd3,0x0bd1b305),LL(0x4e3f192b,0x5454e867),
   LL(0x4416b978,0x78e9316d),LL(0x81c4173a,0xf2a15f86),LL(0xb6731547,0xa6971d20), LL(0x98dbfdb5,0x2c5b69c5),LL(0xfeac68cc,0xf7b1130c),LL(0x0e84299e,0xb1cbec33),
   LL(0xc44d578e,0xb354773a),LL(0x712087f3,0xcc55ac98),LL(0xba9554a8,0xd9b75e7a), LL(0xe80e95c0,0x93eb6a79),LL(0x99fb7e95,0x667e67eb),LL(0x14b32858,0x105a22cd),
   LL(0x6f70ee86,0x0ded3bd7),LL(0xc1a43b13,0x7e0e5a6c),LL(0xc41e1afa,0x893c3ede), LL(0xeaed250c,0x21e88f45),LL(0x93d3ef9c,0x2cfc36f3),LL(0x7da31990,0x5b4e1c71),
   LL(0xc4126c1f,0x1b61be56),LL(0x5abc20ba,0x274763f8),LL(0xed1afb78,0xb49d2c4d), LL(0x3df52e62,0x1bdeb4ea),LL(0xfff32461,0x8dcdd1d1),LL(0xbb1ca9df,0xe3e523f7),
   LL(0x809ca4e9,0x4672ae1d),LL(0xdde4d528,0x58d55485),LL(0x8b74e2f5,0x19428043), LL(0x203314f7,0x1745d62a),LL(0xd4c37bee,0x6157bba4),LL(0x7dd72c9f,0x49805c53),
   LL(0xa37d788c,0xca78b3b0),LL(0xf475a121,0x17a44e3f),LL(0x1ad2fff6,0xe5c1c959), LL(0x70139e2b,0xd0312c1e),LL(0x8b27ca8c,0x6724a7cc),LL(0xd962035e,0xaf28caf8),
   LL(0x9570d848,0xbcc2ea7f),LL(0x7f265e1e,0x05fc9759),LL(0x244ce79b,0xb4954372), LL(0x084d1046,0x164e75ef),LL(0x89adc16e,0x4defb358),LL(0xfa4dc623,0x550e503a),
   LL(0x5564a17f,0x370b2b55),LL(0x55a566cf,0xe074512c),LL(0x78ab0d7a,0x110463ca), LL(0x5d2460f6,0xefd6c296),LL(0x9c33b5d1,0x9abe3d42),LL(0x08c3f981,0x4e016175),
   LL(0xc21b36dc,0xbf03ca84),LL(0x0b8c07f1,0xf34003bf),LL(0x63eb0039,0xc537b09e), LL(0xb82887e2,0xcacbf643),LL(0xbd3bdbf4,0x9d053615),LL(0x61dda14a,0x5c36ffc2),
   LL(0x528c2064,0xe1664301),LL(0xbcae3cc1,0x2e16357e),LL(0x52311eed,0x025d8360), LL(0x2f0a83b4,0x76f15854),LL(0x06dc8ff7,0x10d87d9e),LL(0xf989d477,0xaded98a6),
   LL(0xa52127c0,0x3ed6c171),LL(0xa69dee80,0x0847d132),LL(0xd8445a15,0xb2e7e29f), LL(0x71f974e1,0x156e4663),LL(0xe5bd0273,0xd7512685),LL(0xc73100df,0x6f515794),
   LL(0xc2ab5306,0x951131c1),LL(0x6c73f4f5,0x89255bf9),LL(0xe69b21c9,0x6ac84582), LL(0x0ab9dce0,0x018b49af),LL(0x7954a835,0x27e5b722),LL(0xb1229414,0x865a648f),
   LL(0xc8dd070b,0x388cdc70),LL(0xf52522ad,0x75642aeb),LL(0xcc49bc29,0xe25ec387), LL(0x41ee6cb0,0x214f62c1),LL(0x041d74c3,0xc5ee3195),LL(0x449dbc9b,0x04d43b7f),
   LL(0xc7e8f6b2,0xb3a64235),LL(0x323a93a3,0xeeb2edcd),LL(0x4938cf75,0x67e22e11), LL(0xa914b04d,0xd64b0693),LL(0x88b818df,0x688fa54c),LL(0x591d0c01,0xd481be4f),
   LL(0xcf3d8ba4,0xaecd7d60),LL(0xa97c0b6b,0xc7a415c8),LL(0xeebbc7e4,0x3839dd69), LL(0xe2ff8cc5,0x0691d0ee),LL(0xe92d7a4a,0x0c253257),LL(0x36ca20eb,0x740e2dd7),
   LL(0x2a1939f7,0x5d07cf91),LL(0x5cf3ecae,0x3f6c26a8),LL(0xcb547d04,0x92001c06), LL(0xdf774ad1,0x4175ff52),LL(0x6b5f9c2c,0x7abd12ff),LL(0x4d6d8453,0x1658d69d),
   LL(0x40729414,0xcd926583),LL(0x8ebcd09c,0x8a3d6ad5),LL(0x95dc70c7,0x83f72907), LL(0xb5be9edc,0x200031c9),LL(0xd05c8986,0x472b56e6),LL(0x07e0faca,0x9899422b),
   LL(0xf88b02ea,0xd7bf0be1),LL(0x9bffc69c,0xb622e02a),LL(0xbc6df9ca,0x1e5b8cb4), LL(0x350dfef3,0x048f54ec),LL(0x0a312a13,0xe1f864f8),LL(0xd6ca4ce3,0xd4bfba4d),
   LL(0xb0c052cf,0xc3ff21f8),LL(0xfdfba872,0x19f1a2b6),LL(0x5b694edc,0xd6f10827), LL(0x68a8a596,0x6097aff6),LL(0x954b1904,0x518e1cd2),LL(0x7c51ccd6,0xfd80ae29),
   LL(0x1a7e9066,0xbc7a7b59),LL(0x23db5c12,0xcce91b75),LL(0xa3c0a935,0xba7bf0ae), LL(0xda23595c,0x653f64a6),LL(0x08f45389,0x2ae29a0f),LL(0x88589fca,0x724f46cd),
   LL(0x3372c3ee,0x45b34ad7),LL(0x985555fa,0x0f609319),LL(0x9f3dcb8f,0x7fd63b9f), LL(0x0bd28eba,0x13a518c5),LL(0xf5706481,0x57cffe62),LL(0xd032721f,0x11718aca),
   LL(0x9dd7bfaa,0x32ef81d0),LL(0x7adeeac3,0x3ed2d217),LL(0x6157a51e,0xac2de8bf), LL(0x4d16615b,0x7a7ca0e7),LL(0x217f0fad,0x5bdd6a51),LL(0x099a123e,0xae2064e5),
   LL(0x6f3b0a4c,0xd8eb2d51),LL(0x59883368,0x556551c4),LL(0x5f48ad2f,0x6c8195a1), LL(0x62de8d83,0x0a88379b),LL(0xd0b6ecc9,0xaaa811b2),LL(0x83c9d046,0xba49bf0f),
   LL(0xe6eed978,0x11f53192),LL(0x41560616,0x0cdd4a81),LL(0x55689932,0x106a49de), LL(0xa791e389,0x79fdb67e),LL(0xe28a67d9,0xba7af653),LL(0xc7aa61e7,0x44db5d83),
   LL(0xfee81242,0xe86f2877),LL(0xe05b4568,0x1ebfa60c),LL(0x66c639bb,0x6131750f), LL(0x3da39228,0x2f9b4000),LL(0xb25c8ded,0x8e37e3c0),LL(0xaa3c48d1,0x8c3858ea),
   LL(0x37de8469,0x98bf7c76),LL(0x9d051f00,0xd78e0afb),LL(0xc71691e0,0x4553e903), LL(0x539040a0,0x807d903d),LL(0xf73fcb62,0x30fdeee9),LL(0xcc1519f0,0x9c01a657),
   LL(0x0b7f4996,0xc5c73b26),LL(0x50750843,0x2ca19471),LL(0x5a92f717,0xe4d8d372), LL(0x4abe9028,0x832ebb12),LL(0xe09a7f03,0x7f0123dd),LL(0xe8c1d6a0,0xa8456dbc),
   LL(0x28302bb6,0xc8639947),LL(0x037bebfc,0x1fc1410a),LL(0x9f0b40a6,0x7432946a), LL(0xe06af4e8,0xff9565e1),LL(0x8b9882d9,0x02d05ead),LL(0x13e179cd,0x6e5ca86c),
},
/* digit=23 base_pwr=2^161 */
{
   LL(0x4584bdf6,0x5fd3ebe9),LL(0x847693d1,0xd8a4299d),LL(0x40b5d01a,0xc9c0c5d6), LL(0x3afa9eda,0x70b51263),LL(0x805fb55b,0xde94920e),LL(0xb5a46cc6,0xf0c432e1),
   LL(0xf6dcaedc,0x0ce0859d),LL(0x3a3f7708,0xa55b2e15),LL(0xc0a8b787,0x200ce3fc), LL(0x3c7bddae,0x9e0f9782),LL(0x0b81aa87,0xe9d43e19),LL(0x4666b8f8,0x9c47c1bb),
   LL(0x228ed42e,0xa9c19cef),LL(0xc0249ead,0x22c955e1),LL(0x264c2cb1,0x40d5d287), LL(0xa4070d06,0x420e5723),LL(0xc0604717,0xc2a3bf15),LL(0x4a963f1c,0x7ab3da61),
   LL(0xda80e5fc,0x739b4f37),LL(0xcf587883,0xb45a6fe6),LL(0xcadd521f,0xe4a9b791), LL(0x57562a1c,0x27e1b300),LL(0x2fd0f712,0xb04af2a5),LL(0x93409099,0x12cd55e1),
   LL(0xad7c2bed,0xe518dc8f),LL(0x65eedead,0x52acef20),LL(0xb2efa9d0,0x4227e093), LL(0x99c38a81,0x8b7ca42d),LL(0x4c509cec,0x30c1a73e),LL(0x51635a28,0x3dc9300c),
   LL(0x46be452b,0x92d5ea3d),LL(0x215ba744,0x40841923),LL(0x9930df86,0xd8335b49), LL(0x92c4868b,0x12da53b2),LL(0x60a4383f,0x2400365b),LL(0xe190b71c,0x646771ed),
   LL(0xf14687de,0x30e142c5),LL(0x9403fa68,0xad298203),LL(0x2c2dd3e0,0x3e98cfd3), LL(0x5bd48575,0xe2426602),LL(0xc9d41416,0xc7a5a2b7),LL(0x4ade8bbf,0xdcaa3d31),
   LL(0x5f6b0fff,0xb8792b08),LL(0xfac1431d,0x17e00f0f),LL(0x479b79be,0xcb5f6f9f), LL(0x32c49e8c,0xbfecbd08),LL(0x654100b0,0x6a6be7f1),LL(0x0890c3b8,0x2e545747),
   LL(0xb07447fd,0xa559c912),LL(0x56728d8c,0xb33a2b44),LL(0xf526bd48,0xa027ee4c), LL(0xc58f6af2,0x12d8001f),LL(0x374fa755,0x79e6db7c),LL(0xf25a1455,0x2cbc98cc),
   LL(0xd10a0b59,0x597f4f1e),LL(0x84bc0000,0xe166c79f),LL(0xf74bc520,0x5ae7890a), LL(0x8660d37d,0x3303f3aa),LL(0x405c0955,0x2d0ab6f5),LL(0x1f6fea86,0xe98a1550),
   LL(0xb1fec6f3,0x58dac0ba),LL(0x0018597d,0xc24ea4df),LL(0xebe0dd78,0x9ec12302), LL(0x2bec23bd,0xfa749d3c),LL(0xd47a29a6,0x91a607e5),LL(0x273be752,0xb0e81d66),
   LL(0xb3a95f3e,0x6975c26e),LL(0x1d7290dd,0xcb4ee68d),LL(0x4c6ba170,0x8120bbe9), LL(0x1a407679,0xf428065b),LL(0xc12ea429,0x92b64255),LL(0x56d99374,0xc94c96d3),
   LL(0x0103f0e7,0x1bd6a47d),LL(0x1affc26a,0xce0f3821),LL(0x1b702d5d,0x284426b0), LL(0xc65050e4,0xe5ba26be),LL(0x77a391dd,0x29e14db3),LL(0x8ee16508,0xb917678f),
   LL(0x0ddf3f74,0x18c4fe3a),LL(0xd8e5db90,0x477e39cb),LL(0x56614560,0xaf43028b), LL(0x8c492bee,0xffdf3688),LL(0xac548919,0x595987ef),LL(0x74dc2b98,0x4d0d250a),
   LL(0x3c574a92,0x25fa0aa2),LL(0xfaa1a418,0x89be8ba2),LL(0xb69739a1,0x1b002c0d), LL(0xe3f5b3a7,0xa6b854ef),LL(0x1204504d,0xcaa17499),LL(0xa58cedd3,0xad836998),
   LL(0xc4d5d262,0x3cddb768),LL(0x4251792d,0x0a0801ad),LL(0x0d3897eb,0xd704ec49), LL(0x9c91baee,0x0e13d4f4),LL(0x82831726,0x090f2c69),LL(0x76422a7f,0x829b8edd),
   LL(0x49398523,0xe83170b2),LL(0x1b62a533,0x253100b2),LL(0x5e36efc0,0xc18431d0), LL(0xc4c34823,0x26325714),LL(0x628d7b24,0x74c3cfd0),LL(0x9d919045,0xa961e996),
   LL(0x41d95f75,0xe3a642ca),LL(0xf0a8d5c1,0x4c53ea80),LL(0xa9d3eedc,0x07463073), LL(0xc4e66d00,0xa1439f9e),LL(0xb0694462,0x451a25c4),LL(0x8dfe8fe6,0x713bf4e8),
   LL(0xc9e24326,0xae9ef904),LL(0xe8bfd9ba,0xe039837f),LL(0x59511a8e,0xaca967be), LL(0x7acac091,0xd2386b22),LL(0xe9d27517,0x63bdb595),LL(0xce41bbd3,0x51f95292),
   LL(0x9ec42924,0xee0ee680),LL(0xb77fdef0,0x4af753cb),LL(0x7cb4d8a9,0x15078aba), LL(0xb4177c3e,0xe19806b9),LL(0x88f090af,0x3e1d4fc5),LL(0xeeb90099,0xc7b2cc43),
   LL(0xae6a6f66,0x24b3ffd7),LL(0x308cc05f,0xab2000ed),LL(0x107b362e,0xd89d6e75), LL(0x1fa91ce6,0x835736db),LL(0x390c487c,0x6429da58),LL(0x0d312fc5,0xbaed5289),
   LL(0x4a018455,0x0feb44af),LL(0xad578332,0x1af273ce),LL(0x5f625cc3,0x31349adb), LL(0x2d2f3877,0x3848e62e),LL(0x1bfe8dd6,0x620372c5),LL(0x16c9a24e,0xbcc4d459),
   LL(0x0a45de42,0x40c8630c),LL(0x30cc053a,0x74488edb),LL(0x119b7194,0x9cdb7fad), LL(0xd629e05d,0x02121283),LL(0xfa4bbe6d,0x727585f8),LL(0xea1f9677,0xc2a6e653),
   LL(0xbc14220c,0x4656ea75),LL(0x00b087d6,0xc70a76ac),LL(0x3d83a9ec,0x037ac35f), LL(0xaac1649a,0xf8f1e014),LL(0x4c5f804f,0xaa6de597),LL(0x7d523732,0x8cd45ec3),
   LL(0xa04c2d63,0x4540fa7b),LL(0x9eb857a9,0x544d7dec),LL(0xe8f40c8a,0x8ada66ad), LL(0x4598d9e3,0xb8c4fa18),LL(0x516e5110,0xd3ef8a28),LL(0xd56843a4,0xb761d4d5),
   LL(0x91f59c5b,0x0aed399e),LL(0x43f5de7f,0xe4868fd3),LL(0xe58c53b6,0x01b524d0), LL(0x712c183e,0x77219f8f),LL(0x036dd938,0x6371e9d4),LL(0xa8cc4985,0xd4de675a),
   LL(0xc894130a,0x7da252c3),LL(0x85384ea0,0x9e4041cc),LL(0xeba418c6,0xd0a69993), LL(0xaf043892,0x9a9ebb66),LL(0xc703f8bc,0x9baa32db),LL(0x0bf6fcfd,0xc250e1c4),
   LL(0xf56a38f1,0x5913cd6a),LL(0x61ecfa22,0x1e16a4f0),LL(0xb5c22ada,0xeb036c2e), LL(0x95892d2d,0xca1a39c5),LL(0xd31c9856,0x4688a1cc),LL(0x29e69ad8,0x3a6ea062),
   LL(0xda5d2bab,0x283f3d5c),LL(0xa82a537d,0xc4f0d483),LL(0x81e8516a,0x953f037a), LL(0x5a8f3330,0x6bf902cd),LL(0x9216e842,0xda96e730),LL(0xd1f1d62f,0xa8ca4ed6),
   LL(0x50bbcd76,0xd9521c88),LL(0x5533084d,0x3d8ab30a),LL(0x40a22bb5,0x4328e72f), LL(0x680fccc1,0x6594e079),LL(0xbd6afc46,0xc578705e),LL(0x724db349,0x677a7e9e),
   LL(0x77b5d2f3,0x376f9327),LL(0xc9b18afe,0x87bd1401),LL(0x7395cc2c,0x8f06b3fb), LL(0x0f11a8a7,0x5d2d7a7c),LL(0xa3a0d64e,0x84cc49c1),LL(0xf720e6f2,0x0b5a5743),
   LL(0x80df0a37,0xaeedece7),LL(0x9fa00fed,0x78b407d3),LL(0xfb5d3331,0x8ada177e), LL(0x8af3669a,0xa526660c),LL(0x26f37f61,0x95b36f2f),LL(0x1e1206f2,0x21aeaae2),
   LL(0xade1a30f,0x0fbc3955),LL(0x70682b66,0x27c9b059),LL(0x2bfa453b,0x2fce5ba7), LL(0x1316e4e1,0xc90c4a00),LL(0x761c39fd,0xd6355b7b),LL(0x2532c4da,0xf0e4ff59),
   LL(0xd62425cc,0xa9d571ca),LL(0x842bd560,0xdce97426),LL(0xc461be1b,0xbb38f10b), LL(0x822d623f,0x3d2249f7),LL(0xe1eeb405,0x202ec2c7),LL(0xb64d9541,0x4527855d),
   LL(0xc433bf19,0x6fb57a38),LL(0x2a004581,0xa01483cc),LL(0xf1fb8cac,0x7cb982cf), LL(0x7ea92be5,0x8e24b22a),LL(0xc526028a,0x44dc6883),LL(0xc3c10ecd,0x3ccba012),
   LL(0x6e378a16,0xfe5d98dd),LL(0xd7a9f035,0x959df6be),LL(0x856a9b08,0x9123863f), LL(0x4a4861a1,0x37a49207),LL(0xe726e927,0xf3abe482),LL(0xe950eead,0x80ca1765),
   LL(0xf21dac1d,0xa899447b),LL(0x62bb8a55,0xca34599d),LL(0x599442ee,0x3ace3361), LL(0x9b2bdec8,0x19fb35d6),LL(0x1ba85c82,0x60e552f0),LL(0x0ecd12b9,0xe293d542),
   LL(0xa2c1ef2f,0x6effd23e),LL(0x114661d2,0x5cd387d8),LL(0xdffe2dc7,0x8e8c7a48), LL(0x16c4945a,0xa2c03c10),LL(0x9644389a,0x9d7fd950),LL(0x30888d62,0x64455c93),
   LL(0x910d7674,0xfda175a4),LL(0x233d1789,0xa7147ea4),LL(0x9013a09d,0xf101e4a2), LL(0x07139a87,0x67af7675),LL(0xf5941b5c,0x94837a17),LL(0x2afba5b9,0x8d2a3dc4),
   LL(0xbfe0067a,0xe2c65840),LL(0xe666562e,0xd0b29ae1),LL(0x3f69d5ff,0x994f3344), LL(0xb38b5225,0x4895550c),LL(0x67443db8,0x317f6133),LL(0x6a3efbb9,0xe78c5ffd),
   LL(0xf20ec153,0xb98ff8ef),LL(0x54026d0e,0xbb891684),LL(0xe7d12956,0x54ec1dc1), LL(0x8a5739e8,0x9c22e187),LL(0x2ada1917,0x1afc3eb0),LL(0xc839e43b,0xff8822a3),
   LL(0x7425abda,0x9d4a9d18),LL(0x7bd3fdc1,0xf8e9fbc5),LL(0x9ebefe1b,0x697f7987), LL(0x1e6ed20f,0xa9e72448),LL(0xcdf74ccf,0xa345725d),LL(0xbe698fa7,0x7896df8e),
   LL(0x3ba72507,0xa0c0c8b7),LL(0x0fa2a2a3,0x65954bbd),LL(0x16257ac3,0xa5e65d96), LL(0xbba0d229,0x3818228d),LL(0x021796d5,0xb8be11d8),LL(0xb97dc1f3,0xe6c3332a),
   LL(0xdfda1f51,0xb405e455),LL(0xcbe599bf,0xa82537f5),LL(0x9f755190,0xb60bf1e8), LL(0x32a7e242,0x38f7bb06),LL(0x98acbf9b,0x3212ccc5),LL(0xe7ff0a94,0xeebc9ac0),
   LL(0xd6c9a5f3,0xbb8969ed),LL(0x5bd66149,0x12b8c0a4),LL(0xd0b87b2f,0xb58f5e63), LL(0xd834d568,0x3567a0dd),LL(0x19662011,0x9a569d08),LL(0xf86f6443,0x089e7840),
   LL(0x632d7db6,0xbdbee0e9),LL(0xc3b20b5a,0x6d9f02c0),LL(0x998a8d90,0x1050c608), LL(0x2f06ab1d,0xd8d0282e),LL(0xab0aa37e,0x4c54a44d),LL(0xd082d2ae,0x6e1b0aa1),
   LL(0x25ecff7f,0xfeabb634),LL(0xd1a47294,0x97ef1829),LL(0x45d079b6,0x478931f1), LL(0x90a34a1a,0x0c0a2fa3),LL(0x25caf0d5,0x152a763f),LL(0x0e40b542,0x35d8af73),
   LL(0x2007d582,0xc49fbdb0),LL(0x26d43be1,0xbba36d92),LL(0xd8e1f054,0x4e8feb86), LL(0xdf679100,0x2d7a4eb1),LL(0xa594368f,0xd99cc183),LL(0x7a3bbb39,0x5953cee1),
   LL(0x8b7ac847,0x9db6811e),LL(0x2196bcd8,0x611682a4),LL(0xab14055b,0x5d9d1f5a), LL(0xec03a248,0xde8b16e7),LL(0xd7f2292b,0x8fdd20f4),LL(0x55aedc44,0x0ab41c2a),
   LL(0x4c2182b6,0xdcca73f6),LL(0x236096b4,0x8fb4b87f),LL(0xd5653730,0xce326270), LL(0x76943c44,0x5c9e40b8),LL(0x0ab1c0c4,0xde4cd5b3),LL(0x3940c729,0xf6ed23bd),
   LL(0x77a7afa6,0x5c07d98d),LL(0x5fb74010,0x3b469a44),LL(0xd7860cfb,0xfd595617), LL(0x59e07d3b,0x98976412),LL(0xba3430ef,0x90454a37),LL(0x0a3e1885,0x46a2a6dd),
   LL(0xfb32d876,0x25adf7fa),LL(0x774e5344,0x256577cf),LL(0x1805d5e4,0xfd44d54d), LL(0x8fb4e6b6,0x92207fea),LL(0x11c5d5d5,0x4a220d0a),LL(0xf8e91bdc,0x3be12723),
   LL(0xc245b8f3,0x2ca7da35),LL(0xa88aee56,0x2870eaac),LL(0x46354fd1,0x550b013d), LL(0x79eff102,0x42ddcbc1),LL(0x111f871c,0x66a3ed1f),LL(0x3e002a53,0xbddf6a08),
   LL(0x89d6175b,0xdadeac86),LL(0xdedeed02,0xedba73cd),LL(0x5fae1ea3,0xf56ed9fa), LL(0x46fb11d2,0xd27a3c2e),LL(0x8111c477,0xe8196e09),LL(0x4511b448,0xff123cde),
   LL(0xe26a279a,0x0f843c1c),LL(0xa3576554,0xc5e0d5bb),LL(0x41361ba5,0x23b9fb1b), LL(0xe963cc9e,0x23b72bfd),LL(0xb847fc14,0xdce7e767),LL(0xf14fcef9,0x162fc75f),
   LL(0x5495b1c0,0x835fb910),LL(0x9d67f026,0x4bf8073b),LL(0x8899fab2,0xbf47587e), LL(0x82303583,0x5d3d606d),LL(0x3016c2cb,0x8dc9e515),LL(0x70ad2ecd,0x1941fb0d),
   LL(0xaaf9bf1a,0x52d3382e),LL(0xd97fd623,0x82ca0b62),LL(0x421adaf5,0xd415fb55), LL(0x5bae728f,0x9e737a64),LL(0xa226fda8,0x8bb1f7ec),LL(0xc8fc211a,0xbbe52052),
   LL(0x98b18dd9,0x111f6855),LL(0xfc1a8284,0x07963287),LL(0x049bc41f,0xadd96aeb), LL(0xa4b9431a,0xf9cdc9cf),LL(0x3186f178,0xed472952),LL(0x9e6b099c,0x7c483bff),
   LL(0xecf00f66,0x0f259f0b),LL(0xdc4d7da1,0x42adeebe),LL(0x8347831b,0xa84be35a), LL(0x1cb75e95,0xfcc87599),LL(0xaf570741,0x4d352430),LL(0x560a12e1,0x3985430b),
   LL(0x269c3158,0x0d101d88),LL(0x62a2e7c9,0xf4f4fd03),LL(0x5bd4e966,0x2868c3a9), LL(0x3f2fe7d2,0x28560f6f),LL(0x9f97e5c5,0x33e9acc4),LL(0xaa6fdb21,0x13e585da),
   LL(0xa9d10acd,0x8c93261c),LL(0x39ee8556,0xaa0c4ecf),LL(0xa1ae594e,0x24174c20), LL(0xe22539a2,0x3741dd7f),LL(0x592618e2,0x2d22e571),LL(0xdf3458f1,0x2f7bfd39),
   LL(0xd69e123f,0x0a50b744),LL(0x16a94796,0x64461277),LL(0xe18a1d4d,0xd71b262f), LL(0x5e180b7b,0xd9040b30),LL(0x47d6ce9d,0x26f0f403),LL(0x67a15aa3,0x78d16700),
   LL(0x31529faa,0xa97d1af6),LL(0x2ce2270e,0x6598f797),LL(0xb178d3c3,0xfeb8f446), LL(0x16f5a2ab,0xa967e2c7),LL(0x04b96d78,0x205fdcb6),LL(0xe986b7e7,0x0e580f2a),
   LL(0x2dd71da3,0x5b769229),LL(0x88f598fb,0xfaf8b016),LL(0x143492ae,0x8f8d4b70), LL(0x96a9c03f,0xa7f5e617),LL(0xa0c34682,0x6bd67a8e),LL(0xa12c32cc,0x7b459945),
},
/* digit=24 base_pwr=2^168 */
{
   LL(0x4fee65e2,0x00b598c5),LL(0xf18f37b1,0xf85f1a2c),LL(0xe3b72548,0x8659b393), LL(0x696e8d4b,0x3f89080e),LL(0x219f661f,0x4b0794d1),LL(0xa6fda57a,0x95783cf3),
   LL(0xacf31bb7,0xd47dbc28),LL(0x44076b4a,0x228ba361),LL(0xeff372cb,0xc98f0d65), LL(0xed79eb73,0x73179523),LL(0x9c75c7fb,0xc172348a),LL(0xfd4dc8f8,0x56a7598b),
   LL(0x611e59a4,0x59648973),LL(0xa9f595e0,0x2151777b),LL(0xf5551b53,0x4c47c966), LL(0x3972a3d3,0x021253a3),LL(0xb4befe84,0x90bc0c61),LL(0x77515e8a,0x16d84bd2),
   LL(0x94f40d96,0x5c350e2e),LL(0xa5864e3f,0xdfcbcae9),LL(0xfb5d92bb,0x9a2e4e30), LL(0x44281abe,0xd9bde421),LL(0xea539b37,0xa7f9d3a7),LL(0x27b92e31,0x61c3ae50),
   LL(0xf2aa70a9,0x2244e258),LL(0x35be44a9,0x73df3f53),LL(0xa6238039,0xdcefe8bb), LL(0xe47497a3,0x3a811b6d),LL(0x25465973,0x620c900a),LL(0x4aa632c7,0xe81d4109),
   LL(0x81a533c4,0x49894327),LL(0x9c521884,0xb510e537),LL(0xd2db22d8,0x8e091dec), LL(0x2112921b,0xe40883cb),LL(0x3023be06,0x77a182a5),LL(0x344bcdce,0x9406af56),
   LL(0x5f5b1245,0xb0c7d9e5),LL(0x01cd2020,0xf2cae30e),LL(0x4f24138a,0x300d3991), LL(0xe5eb6668,0xb1a3ab1f),LL(0x1468e662,0x1c21f927),LL(0xe4627789,0x2cd78fd3),
   LL(0xa2475808,0x6bb937ce),LL(0x89fbe7fa,0x190dea20),LL(0xa4a879ab,0x2b19299e), LL(0x02e58d85,0xd08bb8b3),LL(0x44d8e850,0x5f3c807f),LL(0xa807b92b,0x7163052b),
   LL(0xf139c859,0x33e3fbb7),LL(0xb61d1651,0xae851c86),LL(0x711ebfa9,0xd19bb9d2), LL(0xab556ef8,0x04eb2dec),LL(0x26e75498,0xa90032a2),LL(0xad236dde,0x1b6e1eac),
   LL(0x72b36422,0x45e0d154),LL(0x440553f6,0x505c7e2a),LL(0xcfaf2624,0x51c21ee7), LL(0xeead1c76,0x28cfa503),LL(0x21322f7a,0xe7e08256),LL(0xca8f7baf,0xb50012c8),
   LL(0x0e8b3583,0x3ec7b6f2),LL(0x40191db4,0x75562e4b),LL(0xd97200b3,0x2de64fe8), LL(0x67766784,0x9035e032),LL(0x00818616,0x5f64b295),LL(0x9d38c296,0xd62bff91),
   LL(0x20e7da86,0xc1c38d7c),LL(0xaa79a592,0x29d1701d),LL(0xcb4964c4,0x43853290), LL(0x558d0744,0x157c8a28),LL(0xd845fdae,0x9359aa13),LL(0x0c16e633,0x6938504e),
   LL(0x303aaf9c,0x7551d181),LL(0xa6ee9450,0x4f75790a),LL(0x5a5ee5ae,0x30cda47c), LL(0xcaeeda67,0x5cecbe96),LL(0xb6acf1ec,0xbb304fb4),LL(0x23d4e0e9,0xa8d438de),
   LL(0x55bbf072,0xe3c174cd),LL(0x096526d5,0x4c192612),LL(0xa3622362,0x72b1886e), LL(0x38ad3224,0xf9ff34ac),LL(0xde248da9,0x3c0e6f03),LL(0x904019ba,0xaed9b48d),
   LL(0xeb4c2fe6,0x8b931dfc),LL(0x6fb1fd3f,0xb59385ae),LL(0xcdbf8b2c,0x34d15d09), LL(0x501dfe27,0x354fb5bd),LL(0x8678f10a,0x3244c239),LL(0xdefc3322,0x27fb966c),
   LL(0xd835d9d2,0xa7bfa23b),LL(0x3d77d0c6,0xe2e0879f),LL(0x7198caba,0xcd307f9f), LL(0x8ef6dc8a,0xaaf47ac7),LL(0xe089b5a0,0x79d488a1),LL(0xd3898ccd,0x9c5910d1),
   LL(0x99fedbbc,0xe1a69186),LL(0x8c8fedb7,0x1cb035cb),LL(0xb456f31c,0x2cd27f26), LL(0xde4d4827,0x5fe6041c),LL(0xc126f7bf,0x3566eaa3),LL(0x4361f83f,0xafba18a2),
   LL(0x5b63f6d1,0x79cf0a94),LL(0xd989b598,0x00ce6a85),LL(0x03f0b2c6,0x0f8d928b), LL(0x018ff12d,0x4a4299b9),LL(0x5f379fa7,0x59ab07e4),LL(0x5381194b,0x5e5deef5),
   LL(0x3c4ab2c9,0x6e7f659c),LL(0x65654e5a,0x8a446f62),LL(0x6327f61a,0xc3cccd42), LL(0xd14cb9b1,0xc9e4ca54),LL(0x6bde8567,0xeee5364e),LL(0xdd7a0868,0xac8150fc),
   LL(0x5f134595,0xa719652f),LL(0xae508424,0xfcfec81c),LL(0x11ba658e,0x9e16ae3f), LL(0x60511e27,0x88494b81),LL(0x66979b9a,0x495d153e),LL(0x49fad3d1,0x55611c7a),
   LL(0x8e3ffac8,0xfc730512),LL(0x83da2d11,0x1b6bf5a3),LL(0xac075bc5,0x99558261), LL(0x7587269d,0xd44f89c7),LL(0xdee8b114,0x05413990),LL(0xe309d884,0x35258a3c),
   LL(0x6a76d0d3,0xba07c28f),LL(0x5067e16c,0x18a52a08),LL(0xc175e5e2,0x6ae1d870), LL(0x9b314f1f,0x5404c2f8),LL(0xb85b02be,0x22c45028),LL(0x69fe8fb5,0xbfa91557),
   LL(0xd5e20541,0xf420471f),LL(0x93b4597e,0xb7cd0563),LL(0x9e8fced4,0xcd762134), LL(0x667246ed,0x93f1a293),LL(0xcc37aa97,0xa67f1015),LL(0xccc76489,0x54ecfc1d),
   LL(0xffed38d3,0x2867d280),LL(0x339ff836,0x34e2e617),LL(0x843d9ee5,0x90c7a0a0), LL(0x0bc5ad9d,0x38be6f0d),LL(0x61784c24,0x5be128b1),LL(0xf5c92d27,0x6d1d7024),
   LL(0xf28e8e61,0xb93c9e42),LL(0x2e7b7f4b,0x1cb53b6a),LL(0xe4e0fef9,0xe1f605e9), LL(0x1f88f7d1,0x069d28f6),LL(0x22c71a8d,0x96265f16),LL(0xdab828ef,0xac157d26),
   LL(0x341a67b9,0x98b3a5ed),LL(0x3f3639df,0x9aaa605c),LL(0x8b92ce1e,0x59297d2f), LL(0x36504280,0xb9812805),LL(0xa4416a3d,0xe5ae2be0),LL(0x41919442,0xb31daa16),
   LL(0x0feb1f18,0x1634ffbc),LL(0xd33d0a9d,0xa51dcbe0),LL(0x767a41f1,0x0e03bb87), LL(0x148212a0,0x01041281),LL(0x0e55fb63,0x07f9ce13),LL(0x9bbfabe9,0x9dcfb9b9),
   LL(0x4eb575e2,0x3ac5febc),LL(0xc5aedbca,0x8aaae188),LL(0xd5870b65,0xc14938bb), LL(0x37600276,0x05a4c6b7),LL(0x6227e53f,0xa3b7f153),LL(0x4703e577,0x3b37d9f1),
   LL(0x6aea82a4,0x7354429d),LL(0xd061ccd4,0xa8b693b4),LL(0x52d0b458,0x3120ff37), LL(0x92f80ea7,0x254dbf7f),LL(0x89430f4b,0xd229a339),LL(0x94de0c1e,0x97b253ff),
   LL(0xe4b60a58,0x6556cb19),LL(0xf3b79531,0xaf91e209),LL(0x5bd7c1e7,0x4600c789), LL(0x4100417e,0xe92bfc4e),LL(0xb6d8a9d9,0x772261d9),LL(0xa9a6916b,0x2e573e06),
   LL(0x98b95e3a,0x05709266),LL(0x1dbd0d3c,0xef67fdbf),LL(0x7552cee7,0x950e2fa7), LL(0xf4ae00d3,0x03b57561),LL(0x46ba7e71,0xf070c70a),LL(0xe39a20da,0x196843b1),
   LL(0x60091a66,0x4213bea2),LL(0xa0e25486,0xbed0b79e),LL(0xe5c60f95,0xd2bc9dfb), LL(0x3ce65338,0x7aa159d5),LL(0xc8ec70b6,0xef1f0d85),LL(0x28d23ccb,0xb47100d7),
   LL(0x2f8c540a,0x7c602da6),LL(0x854edfb6,0x9e3193eb),LL(0xa18a05e6,0xd1b97d97), LL(0xc430fcd8,0xff9bdf8d),LL(0x3b7fb6c4,0xdfde4b67),LL(0xa97725fd,0x3d8b2185),
   LL(0x0cd2fdea,0x20f59fe1),LL(0x5923a8af,0xfe8a4dc0),LL(0xfa9e9992,0xf9d94a99), LL(0xd35b2020,0xb9c441eb),LL(0x07668b98,0xcce38d28),LL(0x2bf90eab,0x582f54bd),
   LL(0xab0e3f41,0x9f1d9413),LL(0x3a412ee4,0x5e8c119b),LL(0xf030ad58,0x8879b87f), LL(0xde74321c,0xe48dc015),LL(0xf6cee695,0x522fb0ae),LL(0xb8d5054e,0x40162c77),
   LL(0xc129d829,0x2f8ae88f),LL(0x3f3ab771,0x999f091f),LL(0xffb12ab6,0xa7e1a5ad), LL(0x6c51306e,0xbf7d078b),LL(0x3307b832,0xb084c854),LL(0x64450a20,0x46824c00),
   LL(0xff4b278e,0xafad7324),LL(0xb780a046,0xf2aa88cf),LL(0x03143e85,0x429e746d), LL(0x91dc0962,0x5727ab2b),LL(0x67771a60,0x489517bd),LL(0xd28b48a1,0x2d01a677),
   LL(0x68e73dfd,0xcecba270),LL(0xcdb43f75,0x2988378a),LL(0x8a6f1487,0x617e4cc8), LL(0x77958a86,0x7b3bbea4),LL(0x00baf816,0x097fa14d),LL(0xf3dd7875,0xaad04d0d),
   LL(0x6ac04cd8,0x03d57abe),LL(0xc1971431,0xe65c48d6),LL(0x57dbcf69,0xc762049d), LL(0x408042bd,0x967f3d11),LL(0x0ff3e3b8,0x8b9583ed),LL(0x11593e70,0xafcb3c34),
   LL(0xc3f9b58c,0x41df50a9),LL(0x55f9cc35,0x259d92b8),LL(0x2f2195a7,0x670ff896), LL(0x242e6aed,0x1a7c5825),LL(0x0b8b7fa6,0x04cc2dba),LL(0x0f24c794,0x20eb7a8c),
   LL(0xab8c47fd,0xbb5b87fa),LL(0xe425d990,0x24a897b6),LL(0x680c630a,0x86cc5725), LL(0x31fb1789,0x98095baf),LL(0xe7c13217,0xc3655f2e),LL(0xf43a1aa1,0x014736b8),
   LL(0x52deee88,0x62bcaf35),LL(0x5b973eba,0xb396c4e7),LL(0xbda679ab,0x51c1903d), LL(0xd5c3a5aa,0xae30078a),LL(0x40876de9,0x8b17d5cd),LL(0x86473f13,0x150e6256),
   LL(0x09da0150,0x3afdb04f),LL(0xa6c1fbb6,0x944dc9f4),LL(0xd6d37139,0xf978d7e1), LL(0xd7688302,0x294df22d),LL(0x050fc353,0xcb4a2989),LL(0x4d693ea6,0xb7355cc1),
   LL(0x705e6c1d,0x603b9990),LL(0x6e20c02e,0xbc22cb07),LL(0xda4a714d,0xcf26fedd), LL(0xc5f7dd2d,0x499a5020),LL(0x870ff472,0xd5f6d465),LL(0xd7242e9f,0x64105bcd),
   LL(0xdfc15966,0xe8651d8a),LL(0x3f36c08c,0x65f62567),LL(0xd7cc1614,0x7e1b118a), LL(0x5f8c1b51,0xe547ba77),LL(0x40fd1c4a,0xbe829b43),LL(0x7393994c,0x27ef6fa1),
   LL(0xf06e1517,0x47b06a9b),LL(0x9538a94f,0xfc4162b2),LL(0x8787dbb2,0xda0b09f1), LL(0x69ed3599,0xaf40f743),LL(0x20668861,0xd0539dd8),LL(0x4d25d347,0xd97dd381),
   LL(0x8ed5efa1,0xf37b0062),LL(0xf0d1e9a1,0x33b8ee0e),LL(0x4e88bddd,0xfc5ed264), LL(0xebc6518d,0x9c8a10bf),LL(0xed07b158,0x3bacc4d1),LL(0x38679a57,0xf3a21ff7),
   LL(0x28d869ca,0xb696e37a),LL(0x5e14048e,0x1c92d21f),LL(0x09011d46,0xbd612912), LL(0x344de336,0x4253b283),LL(0xb1b9160e,0xc5500601),LL(0xe5dcbf35,0x6a09d20c),
   LL(0x94adc78d,0x0d0de35b),LL(0x0dc1d07c,0x77e19853),LL(0x0bdd2c5b,0x34b0dc75), LL(0x753ad557,0xfb35d357),LL(0x63471eb9,0x5d25872a),LL(0xc1012690,0x0deb74f6),
   LL(0xaaab12f1,0x8af62a30),LL(0x1013c549,0xb94f6b3e),LL(0xe29c422c,0xc66fd2e8), LL(0x5ac31b52,0xf14b58ae),LL(0x222c930c,0x341a40c6),LL(0x79810099,0x97a887df),
   LL(0x4d81c717,0xd365310a),LL(0x01f352f5,0x03f4d8f5),LL(0x7eea6a1f,0xf252f189), LL(0xf8638d81,0xb159cdf9),LL(0xe3f109b7,0x984ed9c3),LL(0x782d6bf4,0xef2542fb),
   LL(0x587be148,0xd0c19f88),LL(0x86172fd1,0x607cef08),LL(0x75ee2bbe,0x7d46f8eb), LL(0xf39fe2f1,0x0d243305),LL(0x362a46ff,0x595aba8f),LL(0xe19fa4d4,0x5a9e9116),
   LL(0x9d07c0e7,0x7c3998c0),LL(0x3ab74d57,0x5dab40f7),LL(0xc6ea1153,0xb0deb5d1), LL(0xdba8086e,0x2f3a9092),LL(0x9dd0d434,0xdf0db996),LL(0x30e52883,0x63af92c7),
   LL(0xfb8379d7,0xce592f6b),LL(0x57237edd,0xf182db14),LL(0xa33ff0fb,0x435ce079), LL(0x16e22f7b,0xf67443b4),LL(0xd01a4364,0x1a603d76),LL(0x0d3849b6,0xf37e32ca),
   LL(0xe6dac621,0x2456397b),LL(0x1585b538,0xa98d7499),LL(0xd2fb419a,0x249b8fe2), LL(0x2d722c8e,0x1971929c),LL(0xeb0dfb04,0xb835ed94),LL(0x7786927d,0xe0ec2ab9),
   LL(0x96808ab1,0xda420127),LL(0xae1677d6,0x9d1d0cfe),LL(0x4381a97e,0x641b4256), LL(0xcf4f8245,0x57a2072c),LL(0x24192acf,0xc038aab8),LL(0x6ee05997,0xf708b6d9),
   LL(0x82628553,0xd0f52e74),LL(0x96add683,0xd6dfc4ae),LL(0xe70500d4,0x5197189e), LL(0x82ee5eac,0x28fe6d10),LL(0x24145176,0x6c314538),LL(0x7a0ce5ad,0x0b507a03),
   LL(0xbf1b56ed,0x28341dba),LL(0x2bd1775a,0xa704c00d),LL(0xcd969de8,0xe8c68f7f), LL(0xad40ac8e,0x64fd4b91),LL(0xfbb5f7b8,0xbc8f2d47),LL(0xb0338932,0x43d6bbe3),
   LL(0x81bc04b1,0x72af7125),LL(0x4e2f9274,0x88d535a1),LL(0x3974e880,0xf0578277), LL(0x15b341a6,0x1539a26b),LL(0xb3e593cb,0x603717a3),LL(0x7d61dcd7,0x477f833b),
   LL(0x032e59e7,0xee7a6462),LL(0x3532e58d,0x70ac18e5),LL(0x0e36ad7b,0x9382dda0), LL(0xf911a5a7,0xf4c500c9),LL(0x800f8273,0x6d46aa73),LL(0x3e10963e,0x670e4f56),
   LL(0x29429437,0x4164dbd4),LL(0x41281f62,0xd9358c9b),LL(0xcbf5eb0a,0x176d1b60), LL(0x2ccdd253,0x3a70e691),LL(0xa25d63e9,0xc53385f2),LL(0x2388d4cf,0xecfe446a),
   LL(0xbc93d305,0x5fe1afcf),LL(0xf307c07f,0x59c6b4d8),LL(0xec492ace,0xa805d89d), LL(0x7c31b553,0x572efc64),LL(0xd5bf85fd,0xb15afb14),LL(0xfa431d83,0xb8a6a24c),
   LL(0x3057e6e1,0x031dd1e6),LL(0x919f1a50,0xac205cef),LL(0x6bf4b8fe,0x3236d945), LL(0xd11eb95a,0x3cdc048e),LL(0x2b4d6ce2,0xaaa916ac),LL(0x7d395898,0x381aca94),
   LL(0x9ec42177,0xc5146299),LL(0x2d66626c,0x6b043df3),LL(0x82ea3769,0x9ed7174d), LL(0x9da03903,0xf8f18beb),LL(0xcd642f4c,0x72c11f7a),LL(0xd4de2f90,0x4bdad457),
},
/* digit=25 base_pwr=2^175 */
{
   LL(0x58d3510e,0x5641060f),LL(0x364db85d,0x8e313a7f),LL(0x5cafd58c,0x203122d9), LL(0xccb848a1,0xf79b8cdf),LL(0x67e06429,0xda00c247),LL(0x50dbd8ef,0xe8ddf520),
   LL(0xd06db4a0,0x13b4d6c6),LL(0x1c320a37,0x9ca098dd),LL(0x4660ce85,0x23635473), LL(0x70e85a9d,0xcb8bdc7f),LL(0x92b6fa4b,0xd96a49de),LL(0x6f48c32b,0xd22ee1cc),
   LL(0x526a7a48,0xbaf0570d),LL(0xfd4d36d1,0x9436cae8),LL(0x24fac988,0x44427e92), LL(0xb39cb050,0x935deddc),LL(0x95237984,0x2db73da9),LL(0xe0403a31,0x6d812614),
   LL(0xa4968444,0xf1cb1e84),LL(0x15156d72,0x8624f8a4),LL(0x66276ea6,0x83d9f12a), LL(0x3b28aaa8,0xd15883d2),LL(0x06aba758,0x4043aea5),LL(0x19ad8608,0x3b9cffbe),
   LL(0x71c764ac,0x19b195dd),LL(0x16c89a6f,0xb10cdd8e),LL(0xdee07ce5,0x51f783dc), LL(0xbf7febaa,0x05b98ceb),LL(0x0026e40f,0xcbdf20ec),LL(0x8ea46069,0xf4ff428d),
   LL(0x19829b4e,0x204f3d40),LL(0xfa6cc5ed,0xd37d49f5),LL(0x07a5325c,0x55bfb8db), LL(0x7c3cffdf,0x349c9305),LL(0x44e85a40,0x15868ca4),LL(0xe664b649,0xab792855),
   LL(0xa8931c2a,0x1488def0),LL(0x93e47cf2,0x20b6b035),LL(0xc9c0e09a,0xfd90bf99), LL(0x9e4cf905,0x28fe0815),LL(0x48dbf990,0xf397729a),LL(0xb09e4e66,0x82000550),
   LL(0x152fdc18,0x7c7b2c74),LL(0x81529a8b,0x2e09d47c),LL(0xdffec896,0x116d1ef9), LL(0x9ea4077e,0x448891e4),LL(0x9566eb00,0xadad99e4),LL(0x9df75199,0x7d6c3d02),
   LL(0x1d7fffc5,0xbc3f75a2),LL(0x741fd001,0x47e4ce19),LL(0x45b7d08c,0x24659725), LL(0xc74dcf3a,0x1a4b43cd),LL(0xdc7dd457,0xa1e3c71d),LL(0xe49194dc,0x37aeb7d2),
   LL(0xbe8af7ed,0xf6372965),LL(0x2e04049f,0x4e36da0e),LL(0x34f4f1d8,0xcc4e4c0f), LL(0x387c3ce4,0x185a80da),LL(0xf81efae2,0x999c4880),LL(0xa0cc97d1,0xd2002692),
   LL(0x2a469d56,0x97b45c6f),LL(0x4932fa9a,0x0c918aab),LL(0x43aff643,0xc2a2d43e), LL(0x33806c7d,0x685eab5b),LL(0x69645ab9,0xa44c2171),LL(0xa8df46a4,0x128ed8da),
   LL(0x650a4592,0x561ca2b2),LL(0x3ca61b0b,0x01673084),LL(0x680d8c8f,0x79ebf7d1), LL(0xc216ae2a,0x33cd5c09),LL(0x260a7797,0xdc44dbeb),LL(0xe5886569,0x80046982),
   LL(0x01f3457c,0x6ffe44e9),LL(0xcbd42d4f,0x9f2fb631),LL(0x77426e19,0x372de1ab), LL(0xb5b9cfe8,0xe73bea79),LL(0x105e1632,0xd2af87d0),LL(0x91def158,0x562e0ebb),
   LL(0xe8eadc48,0xb3051141),LL(0x214796b8,0xe9797985),LL(0xb0cf5916,0xc7174b9f), LL(0x7c17697a,0xccdaf130),LL(0xe6a3f8cb,0x2ff86ffa),LL(0xe0e4d911,0x655d7ff4),
   LL(0x7f8a7546,0xeb590d5a),LL(0xd84598ae,0x2e33657c),LL(0x1cf22291,0x6865f925), LL(0x80d54d83,0xf33b0db7),LL(0xd095724d,0xd2045fa3),LL(0x37a729ca,0x2a65650a),
   LL(0x9b7d8eb7,0xe5a6dff7),LL(0xdd67daf1,0x104734b8),LL(0x503bfa5d,0x49050eee), LL(0x49376195,0xa3422472),LL(0xd85aa682,0x9d58a5f9),LL(0x27b209d1,0xef14a582),
   LL(0x10c74322,0x72142664),LL(0x161a9f7f,0xed4f37db),LL(0x5f7f95dc,0x0f64b4a6), LL(0x3907231a,0x66c0fe9a),LL(0xc7664836,0xeeadbbaa),LL(0x7907bf1b,0xece73bcd),
   LL(0x9c17a6d5,0x16aded71),LL(0x08ca25bb,0x2a92a505),LL(0xe1061e03,0x180b69be), LL(0xd61b7fbe,0xad3ee8dc),LL(0xe5e8b274,0xf8d14495),LL(0x3bb4fd37,0xa5269565),
   LL(0xfb62dc03,0x170bf89b),LL(0x18b9c79a,0x1fe9859a),LL(0x839e677b,0x7605d9e1), LL(0xaa1b4663,0x6d265c97),LL(0x48202008,0xb34d878c),LL(0xa03d375a,0x2b696896),
   LL(0x166a71cd,0x789e3b3d),LL(0x10983ef5,0xa9f4f3ca),LL(0xdb908c7d,0xf634e8df), LL(0x8192d652,0x0375b69c),LL(0x7e4b6e1c,0x2f5cbad4),LL(0x45be1a16,0x4518ff99),
   LL(0x408d39c0,0xb664989d),LL(0x2e4280b7,0xa947849c),LL(0xece6ea89,0xbad0f997), LL(0x3a7571ed,0x434fd6c9),LL(0xcf47df20,0x46da8ae4),LL(0x638076c4,0x7708eff6),
   LL(0xc63c8ac2,0x17e1b4e7),LL(0x36830de3,0x0fff916b),LL(0x55092352,0x2fe8eb88), LL(0xd0af6ad5,0x8b72540b),LL(0xf26f7df4,0xf41a415f),LL(0xb3259e07,0xb7be4c73),
   LL(0xb3b6ce27,0x4d668949),LL(0xf8705df2,0x332b0818),LL(0x2b168b67,0x11736e3f), LL(0x96ab51a4,0xb7a67c69),LL(0x57e547e7,0x00855b88),LL(0x17aac6bc,0xe9ff07b6),
   LL(0x2efd47f8,0xeebd36d2),LL(0x288c8f9e,0x4fe7bc31),LL(0x034e796a,0xe82f1e28), LL(0xf0e8922a,0xd25c0279),LL(0xf0f8dcb2,0x9ec4ded4),LL(0x860e5a78,0xa66ac462),
   LL(0xb8a729d3,0xef72f5c5),LL(0x13b9c03c,0xb6915595),LL(0x2ed45b90,0xa4e66b65), LL(0x0845bf8a,0x763c3414),LL(0xac93c4fb,0x13ac686d),LL(0x2095313b,0x923efc3c),
   LL(0x3349c905,0xac570164),LL(0x5ca04e5a,0xc85ebcd6),LL(0x6a5d9dc0,0x616a35c7), LL(0x68d2818e,0xfbce6895),LL(0x59769d61,0xd85a342b),LL(0x025ba76e,0x0aa5703c),
   LL(0xe391b459,0x205537fb),LL(0x5d410558,0x8d2a3d0b),LL(0x64a2d47d,0x71444a23), LL(0x822b0c30,0x0c86c2f6),LL(0x64c2f548,0x8b276842),LL(0x94dbb24c,0x51a73b54),
   LL(0x29e9c4f7,0xf0ead209),LL(0xe1401652,0x5eddd090),LL(0x190c1898,0x6d6b0600), LL(0xef639412,0x762be19e),LL(0xe78dd0d2,0xe322a3e9),LL(0x824c6d20,0xde53b7e9),
   LL(0xc90500e8,0xc6b2dd77),LL(0x5a53da9e,0x3a399dc1),LL(0xd09ab46b,0x0969d406), LL(0x249ac549,0x526ec7ed),LL(0x1a09dad3,0x4ae8309f),LL(0xb0c3ad12,0x143165c0),
   LL(0x610d6c08,0xd1d7d354),LL(0xd1dbf837,0x12d3da0b),LL(0x01a98f82,0x73fb7218), LL(0xa8666766,0x71fe7d63),LL(0xdc316634,0x74ea9c07),LL(0x55aa5452,0xad9f5798),
   LL(0x92ab7a01,0xeb52fc54),LL(0xfa5e4eea,0x6ec23c07),LL(0x01a397ea,0xa9cb3e84), LL(0x69945c66,0xebcac5af),LL(0x3f194ceb,0xdf57e59f),LL(0xa296c88e,0xb5ec451f),
   LL(0x794f2a13,0x9cf79425),LL(0x310e1c2d,0xff079313),LL(0x5317d195,0x4d8c0f20), LL(0xf670d56c,0x0d854923),LL(0x31cfadcf,0x7564177d),LL(0x0845bd19,0x67d58b2f),
   LL(0x6ce7d0f0,0xed2fd628),LL(0xa2ff8523,0x0a338e2e),LL(0x6968e5ed,0x4cd9123a), LL(0xf2f06ab4,0x14edcc8f),LL(0x6e52fb45,0x23c39e14),LL(0x41a9c448,0x529a89d3),
   LL(0xfa8c503f,0x436e8ebc),LL(0xc7ec91f5,0x2d692350),LL(0x2f6a42f3,0xe847216b), LL(0xa68613e7,0xe12297b3),LL(0x10dbc984,0x16e343fb),LL(0x16d59123,0x7dbec088),
   LL(0xd64ff51c,0xf3e6abe1),LL(0xd3afe2c2,0x87accfb2),LL(0x4dba0813,0x57b5f905), LL(0xb3b8c3fe,0x17038f08),LL(0x247ea7e7,0xfe435876),LL(0xa6d8a7a0,0x61415630),
   LL(0x72efb994,0x685e474d),LL(0x495c513e,0x05b09ce3),LL(0x36da77db,0xf03f489e), LL(0xd1920e81,0x5bce5842),LL(0x4ba12592,0xc3fe249f),LL(0x0a5eed91,0x9671b08a),
   LL(0xaac51e01,0x387fcc9a),LL(0x9c3e2419,0xf8c9ecb3),LL(0x754e0ecb,0x1017d74c), LL(0x9a76367c,0xb1528d0c),LL(0xa2e5c47a,0x4dcd49c1),LL(0x3751eba3,0x7b635160),
   LL(0xae011f19,0x3879777a),LL(0xf839016e,0x6a63aafe),LL(0xe3944a17,0x199a76e4), LL(0xb399afa0,0xa0455c72),LL(0xf9479065,0x95450e12),LL(0x82b89c67,0x90479fcb),
   LL(0xe392cb40,0xdc585a30),LL(0xafe2f1cd,0xbd161799),LL(0xaf671659,0xf0794ecb), LL(0x3e558726,0x7144b4e9),LL(0x0e8fb051,0x61aa80d1),LL(0x0a97d49b,0xd2a41b0b),
   LL(0x0a74e744,0x10f7c811),LL(0x96ab89c0,0xdd94a330),LL(0x53525096,0xb617d078), LL(0xcfdfa40c,0x2e5b46f0),LL(0xcedc53d0,0x6edea8ab),LL(0x6399837f,0xaee682f0),
   LL(0xf2adc050,0x398c7fd2),LL(0xc7638b0a,0x1030cdae),LL(0x8867f1d6,0x1cd2be4b), LL(0x1ba68d19,0x2f97a204),LL(0x18cd2ec7,0x47f3b28a),LL(0x8e77c014,0xf164c987),
   LL(0xc33772e4,0x8489b718),LL(0x2f4128d3,0x08ac1911),LL(0xda0369a0,0x8370f1f5), LL(0xf1d75833,0x50ff1409),LL(0x0f83a8e7,0xa314c13c),LL(0x5a766258,0x776de80d),
   LL(0x77ba6858,0xffe94ccf),LL(0xcc2a35c8,0x1b87cbeb),LL(0x57a1fa64,0xd5528288), LL(0x5937abfc,0x2e32d797),LL(0x284ca1f0,0x89a76fe3),LL(0x482441bc,0xdbeb8eda),
   LL(0x397926fd,0x6bd5d9e9),LL(0xf5fbd488,0x3e97744f),LL(0x4ff64738,0x020f694d), LL(0x968aa895,0x34b3915d),LL(0xb88572cd,0x60b03cd4),LL(0x67dcf49f,0x0cd44b01),
   LL(0xac9b8b5f,0xc6f1228c),LL(0x7822905f,0x5f49fb8e),LL(0x950e82e9,0xa61a8520), LL(0x6f44f547,0xec78c658),LL(0xaa640c3e,0x86d54b78),LL(0x54c16691,0xf6aab7a4),
   LL(0x44fbab19,0x0cae07b3),LL(0xcc4d821d,0x61e61d3c),LL(0x5beae87e,0x5c2ff6ed), LL(0x9ba67712,0x719d81cf),LL(0xcb1e1329,0x8fb7e1c8),LL(0x32f7e9b7,0x82a0507a),
   LL(0x0d8af7c0,0x94bf5956),LL(0x3dde66aa,0xeb94e708),LL(0xcc491968,0x2924bca4), LL(0x1bae682d,0x348c6486),LL(0x0da0931b,0x5490288c),LL(0xa923db17,0x79cae045),
   LL(0x4ed8cc93,0x99ae0e06),LL(0x1f211af5,0xe0dca803),LL(0x54413c07,0x3e0da321), LL(0xd2e5771c,0x11bf91ee),LL(0x5d9efe6d,0xde57fde7),LL(0x2bae2eca,0x140cd249),
   LL(0xebe25fb3,0x8c25772c),LL(0x5ae86876,0x9ebd7151),LL(0x227c8fc6,0x9e01abfa), LL(0xe8279d68,0x10316800),LL(0xa07da963,0x394a6fff),LL(0xc8b9c7e0,0xd8316e12),
   LL(0x4866b100,0xe34a3346),LL(0x64640c51,0xbe82e03e),LL(0x8c26b45a,0x2228e85c), LL(0xaaec11b1,0x4f53739a),LL(0xed773bc3,0xedcebe7c),LL(0xfcef0ce2,0xfee917ad),
   LL(0x44c7c595,0xdf6dca1f),LL(0x374d0d04,0x421ac2e7),LL(0x9e3ed6ef,0x842f0ded), LL(0xe9dcc612,0x0e20b04a),LL(0x2040e59d,0xbe403a6f),LL(0x8611c050,0x0ed4eb43),
   LL(0x559e8de2,0x8479f4bf),LL(0x851f7b34,0xb12ebeb5),LL(0x3a3d71ea,0xc34bfd18), LL(0x24e9273c,0x56d9d399),LL(0xb9f63b13,0x7f404c7f),LL(0xe50afb2d,0x9b679748),
   LL(0x09f98b35,0x9be55951),LL(0xff869a57,0xde925893),LL(0x21a9054e,0x387f817a), LL(0x70ee95b2,0x1e674786),LL(0x36b7d89e,0x628421b4),LL(0xe95fc254,0x5c66493f),
   LL(0x3f27517a,0x68f7f079),LL(0xa2634bb9,0x44e0f99f),LL(0xc49ed6bb,0xff40ec9d), LL(0xd4c15f22,0x308b3e76),LL(0xddaa3039,0xb3f77d17),LL(0x7ea641c5,0x3dc87e6e),
   LL(0x004a3af7,0x8d01afc4),LL(0x5cb71d7c,0x118e5970),LL(0x48ba3691,0x4facd336), LL(0xa796a7d3,0x30530fed),LL(0xcc7913f8,0x437ad3b9),LL(0x8adebd12,0xe0c151c6),
   LL(0x2e553c5f,0xf37ea5cf),LL(0x45ebc28e,0xd35512e6),LL(0x05331492,0xd4f2b6b1), LL(0xaba58423,0x0ab0b63f),LL(0xd8791541,0x0c6d3499),LL(0x81abb7ed,0xc911fda1),
   LL(0x53f134aa,0x4316e48a),LL(0x492e43e8,0xc26f2068),LL(0x126d373f,0xaea01397), LL(0x37d4baed,0xdfb016dc),LL(0x68550352,0x70fd4cd9),LL(0x90c02756,0x82a6c302),
   LL(0x66cf543d,0xbbfbd662),LL(0x9f372f9c,0x9198ae05),LL(0x0d3c6392,0x548bc0b0), LL(0x71100572,0x5e227e8c),LL(0x189496d9,0xbd10e2ef),LL(0xb6a6798f,0x66b7ddd2),
   LL(0x1d9b4627,0xdbfa89d2),LL(0xe9722687,0xa4144f05),LL(0x62a9f0b3,0xa03a6658), LL(0x09a6ee11,0x11e24e34),LL(0x610d808b,0x02a2f96d),LL(0xb68d12eb,0x3434da69),
   LL(0x420733a0,0x224f3f9b),LL(0x99244104,0xa3f7e5c2),LL(0x9dd4dd08,0x2018ee4f), LL(0x230f84ef,0xb993d248),LL(0x69d385df,0xbb1c8f5b),LL(0xddffd0f2,0x9242e20a),
   LL(0x09654b6a,0x6d842a62),LL(0xfe652314,0x2f79695e),LL(0x193c89e2,0x0c65ea9f), LL(0x6b79b085,0x009cf4f9),LL(0x2268da91,0x390256cc),LL(0xb23a9a13,0x9a5602a1),
   LL(0xc087ed5f,0x7e6b4f66),LL(0xa1c63549,0x14a8e4d1),LL(0xfed4b006,0x01ca8c96), LL(0x6933bc42,0x1f5e13f7),LL(0xcc727c9e,0xd642cd3f),LL(0xa4a52672,0x60a81922),
   LL(0xcb37c6b1,0x21d4016a),LL(0x6aaea4e2,0xbf9c3ad6),LL(0x4d227e29,0x9bd42e82), LL(0x66fe8ded,0x7b43dfb3),LL(0x4f6d7cab,0xccc64f49),LL(0x709cf722,0x9fb84f64),
   LL(0x99a34b60,0x657bba39),LL(0x5bf3ff4c,0x54c86454),LL(0x6a3fc2c9,0x91cab991), LL(0x9d937181,0xad3baab9),LL(0xa318a96e,0x9b39648e),LL(0xd5a8dd54,0x416fe984),
},
/* digit=26 base_pwr=2^182 */
{
   LL(0x10e89d46,0xa5d2d3fc),LL(0x267decf4,0x426ccb6f),LL(0xf65161a7,0xc430a435), LL(0x45ff08ed,0xf32e4fd9),LL(0x52fce424,0x29bd88e4),LL(0x7df49592,0xb58dca75),
   LL(0x2de63b4f,0x08b2eb9d),LL(0xc5a70a53,0x2dd60214),LL(0x183b9f0a,0x8bf59ae3), LL(0xa4d75e6a,0x127db136),LL(0x2dd19a85,0x70a4c22d),LL(0x6bb9e133,0xa04ca68b),
   LL(0x1e0ac0de,0x7c1f8f54),LL(0xcc70b75d,0x33fa9eda),LL(0x1661ad09,0x190f2578), LL(0xc6af0599,0xc8d29939),LL(0xd76f1d21,0x8f0f6073),LL(0x28d0432c,0xda9f9fbc),
   LL(0x9c65f79d,0xa73ce4f4),LL(0xcb12b4d2,0x81728494),LL(0x34a30e52,0xb3709952), LL(0xcc5be142,0x34e61905),LL(0x0d1b9183,0x0237c5ce),LL(0x51d9f58a,0xcd502078),
   LL(0xb77a9580,0x5935c009),LL(0x60590762,0x109f262c),LL(0xa9bef637,0x40cd7b02), LL(0xe1e887cb,0xbcf02172),LL(0x22fc3d18,0x23ab267e),LL(0x2b110de6,0xbb9632d8),
   LL(0x1e71e027,0x939849be),LL(0x0916d6dc,0x04493ee8),LL(0x7a4c25ff,0x2456a4fa), LL(0x80e2c3af,0xbd7a130d),LL(0x19ebceb4,0x63b50b49),LL(0xca679a32,0xcf3af9f6),
   LL(0x568806a1,0x8bf5ebcd),LL(0xd6906912,0x9d9e980d),LL(0x2224ffdf,0xbc01567e), LL(0x9b382f20,0xec45fad8),LL(0x2ae9b081,0xc81ce8e7),LL(0x98932797,0xe492d96d),
   LL(0x29a231ea,0x5d2a1162),LL(0x59e196df,0xa3a8d097),LL(0x7f4b3fb1,0xe049a755), LL(0xc6316d29,0x5a3a8033),LL(0x927eae96,0xcfa30505),LL(0xbe935448,0x08b8a76c),
   LL(0xedc1da3d,0xae5b906d),LL(0xe511be7d,0x7e4c4f50),LL(0xbf36de87,0x8d50de21), LL(0x0ed128b9,0xa3e529b5),LL(0x39d260df,0x902d56ab),LL(0x5e3d15b3,0xaafa263e),
   LL(0x0e6cecf4,0x201392c1),LL(0x49f3f177,0xf3e51d03),LL(0xcb103c3c,0xa778f941), LL(0x7666e871,0x663a85f9),LL(0x67c98f13,0xa46b0c85),LL(0x3d7ea941,0x3419ecca),
   LL(0x0cd251b9,0x6e9c319c),LL(0x740ebef4,0xe03a5e58),LL(0xeb45dea1,0xef0db2a0), LL(0xe6e4e097,0x12e7626f),LL(0x19914347,0x27cfc38c),LL(0x6046e4f4,0x92b17248),
   LL(0xfd260bad,0xad9a809e),LL(0x84cf1f3c,0x0faa61f5),LL(0x6afd71d3,0x7b60a72c), LL(0xd1fd427b,0x67f98a21),LL(0x45809f86,0x55e94be4),LL(0x8e08a8fc,0x55f38c03),
   LL(0x78ac4a59,0xe1bd82eb),LL(0x80599bf6,0xc195517f),LL(0xcba34b90,0xd57d73bd), LL(0x68e80682,0x70b22ccf),LL(0xfbe46433,0x9ef082ba),LL(0x4e1de964,0x0e3b2665),
   LL(0x629523a0,0x40c1bd3d),LL(0xb3c7fa46,0x55c4d1e4),LL(0xc3ac04c5,0xe731d1dd), LL(0xb98ea38f,0xee1fa82d),LL(0xc3da8420,0x2a53d82c),LL(0xf22acfea,0x368c36c9),
   LL(0x29b755d0,0x0266ab97),LL(0xcd211255,0xf06c3dd8),LL(0xda4b92c3,0x815c935c), LL(0x7c35910f,0xc8b3d452),LL(0xe14a590c,0x0f3aed26),LL(0x8e1ce1c0,0xf6ecbf3a),
   LL(0x574f2fbb,0x552b5d7a),LL(0x8de8d85f,0xfa3986c3),LL(0x9614d372,0xaa53bc82), LL(0xc58f40f9,0x1118e769),LL(0xe3a00d02,0xa39c3653),LL(0x92267b25,0xb0de6989),
   LL(0xc7b64858,0xa306b7c8),LL(0xb64b2f3b,0xf22489d0),LL(0x277347b5,0xb4d82d56), LL(0xb3c23db8,0xc6120600),LL(0xeb1d672f,0xdd104bc6),LL(0x91821d86,0xf46c3e19),
   LL(0x182aa649,0xbd1809b8),LL(0x9def0d8a,0xa14bcb07),LL(0x150bc2b2,0xbc355f52), LL(0x303a6966,0x9b74419e),LL(0xeb2b3cb5,0xf922f6bf),LL(0x18f7ca08,0x5d7aa60c),
   LL(0x0be5968b,0x42346530),LL(0x2a740a40,0xe0a37f25),LL(0xe665427d,0x03bd64c1), LL(0xe4074164,0xe3911c26),LL(0x7960eb4e,0xc9770b1c),LL(0x5c67feb2,0xbe470d6d),
   LL(0x3894d144,0x3b30ad55),LL(0x16768125,0x51da7959),LL(0xbed864b3,0xe8cfebda), LL(0x833b7cb3,0x49a98925),LL(0x6f174ec4,0x830c736e),LL(0xcda2e7e0,0xf9efa80a),
   LL(0x376e70ba,0x01f86bbf),LL(0x0eddf0bb,0x686b6cb6),LL(0x64d59f33,0xdc9eec10), LL(0xa46828be,0xdb7037cf),LL(0xd2c7ca5a,0xceeb8c28),LL(0xe6196bda,0xe40cbf09),
   LL(0x8bd42f71,0x0ac2d045),LL(0xb62d83fe,0xd4215f7e),LL(0x03acf7bd,0xaab77a58), LL(0xc2eeb2e0,0x7e1aa3e8),LL(0xd86c3480,0x16dbe858),LL(0x8a7fb0b3,0x65a386cd),
   LL(0x8894f88b,0xc426e3e1),LL(0x4593936f,0xa748da73),LL(0xca40633d,0x9e640d2a), LL(0x02ac44dc,0x84fd2be4),LL(0xd437cf7f,0x0471f362),LL(0x7306c123,0x8f4155e7),
   LL(0x21dd0f82,0xc9871691),LL(0x28d633a3,0x2bd96ea5),LL(0xfc2ba2af,0x1906ae24), LL(0x1ee92586,0x2b4efac7),LL(0xf7f07861,0x4b97cbfa),LL(0xf7281f7e,0x4dbf54e2),
   LL(0x25b5b11c,0xc39fffbf),LL(0x98a6b660,0x41973093),LL(0x0f7b07f8,0xd02dcfda), LL(0xfacfc8d6,0x42bd5635),LL(0xfa119a0d,0x49c754a0),LL(0x7a67703b,0xf3900a31),
   LL(0xb7648cd6,0xffe76894),LL(0x973e597b,0xa790612b),LL(0xd6d78341,0x5a817ff0), LL(0xc9960802,0xe4a6c1a8),LL(0x6884bf58,0xa296fc0a),LL(0xcaee199c,0x8387842f),
   LL(0x78a21751,0x505a5fbc),LL(0x04338ed3,0x8d557dd4),LL(0xf8316933,0xc6daa6eb), LL(0x9fbe57c9,0xce8b52fc),LL(0x82ffca53,0x93375f08),LL(0xe48a4172,0xd2e09822),
   LL(0x2d8b1f67,0x59d6d11a),LL(0xf984b355,0x3df9851e),LL(0xeef602c2,0xfad1f4e1), LL(0x1df4b332,0x6c07fd2b),LL(0x0a339f9f,0x1672e0ae),LL(0x3e2e73f1,0xa7aac7c0),
   LL(0x53de8d30,0xd015e91f),LL(0xdeee9e90,0x9cef3987),LL(0x9b54ea0f,0xcae1fbee), LL(0x0b2e4bfc,0x5541f0dd),LL(0x00c1a228,0x15f9dffc),LL(0xd9dfb7f7,0xeed05a2b),
   LL(0xf8bcb7fb,0xbeb350fb),LL(0x3fb74891,0xa6a2efe1),LL(0x4216f4a5,0x1cfaf6a0), LL(0xccd45bce,0xd268b24b),LL(0xe3db6dfa,0x5c85e954),LL(0xf04e8161,0x127ce38c),
   LL(0xfc6c17ef,0x749016b3),LL(0xc849c80f,0x840e079d),LL(0x55aebfa6,0xef3493b4), LL(0xbdddb794,0x5645586b),LL(0x0a0da6a8,0x1e1ddb6d),LL(0x33397277,0xbd31053b),
   LL(0x257efe71,0x64c4bcd5),LL(0xf7abc98c,0xd35b6fad),LL(0x49479a50,0x3b0df461), LL(0x0c01f68e,0x7aa9ea28),LL(0xfc247d85,0xa37c8d40),LL(0x82d5d062,0x4214db8f),
   LL(0xd4665035,0x615f976f),LL(0xb2fd9161,0x39f27e8e),LL(0xce638414,0xa1ae7fee), LL(0xbee083df,0x0717218c),LL(0x80a73a7f,0x66d2bccf),LL(0x77555a78,0xc1cfe44c),
   LL(0x151f9185,0xab6467ac),LL(0x25664569,0x6df53a18),LL(0x466dd4b8,0xa54fbfd4), LL(0x8b3c8dc6,0x6f063338),LL(0x41623458,0x64adde6f),LL(0x112ad869,0x08056bf1),
   LL(0x3434a31e,0xfdd32b7a),LL(0x04cdeecc,0x5728361d),LL(0x3a7b8da1,0x2aa81490), LL(0x09a2731d,0x87c2d7d2),LL(0x89ebca03,0x1f3344b9),LL(0x19158d11,0x3b4eb6aa),
   LL(0x4aea58b5,0xc2bc01be),LL(0x19f12b28,0xf7d53cd7),LL(0xdf3da230,0x475fcd61), LL(0x09990316,0xeb3dd742),LL(0x28ea4a85,0xb1c62ec2),LL(0x71770efc,0xd9ca3e02),
   LL(0x285cb4ae,0x92099143),LL(0x0d8f3f4f,0xfac421c8),LL(0x31e4afb4,0xb20b6d02), LL(0x41735275,0xb896f118),LL(0xe7a40b45,0x9f8d1d92),LL(0xc574911e,0x578117a2),
   LL(0xc8ab5906,0xe049b850),LL(0x90ebe14c,0x3dc8761d),LL(0x3d9b21b1,0x01b959ad), LL(0xa1be26d8,0x69e0edfb),LL(0x90bffa45,0x23061d51),LL(0x5432258c,0x12c49398),
   LL(0xec1ab63d,0xcf6318e5),LL(0x78baa5c1,0xfabf5dcc),LL(0x6529af85,0x6fe5da1b), LL(0x0d696bed,0xc02b8cbe),LL(0xc83d0478,0x5865de25),LL(0xf422a123,0xaef1328d),
   LL(0x700b22a4,0x0895f5d0),LL(0xf78d8bb2,0xb52e7e2d),LL(0xe0972d87,0x5771d2c8), LL(0xa67e68d5,0x45181514),LL(0x29446b80,0x11ad6a2c),LL(0x3c18bc56,0x17b2b4e7),
   LL(0x075ee3b2,0x3d4755d1),LL(0x0fcec9d9,0x1505789c),LL(0x398f0c91,0x6e00b31e), LL(0x57048d0c,0x54cea710),LL(0x6443fcff,0xf40ba9be),LL(0x05b2a1b2,0x04075b8f),
   LL(0xf3aa488f,0xcb23f2f6),LL(0xb3aa9bab,0xd48b8a3e),LL(0xa8b58c26,0xe1dfa993), LL(0x58d04b7c,0xe2641538),LL(0x32ad4a3c,0x1159caa2),LL(0x00256f11,0xa19210d5),
   LL(0xeb897a63,0x52787df1),LL(0x3afae814,0xe4f787d2),LL(0x9a665638,0x3c391a2a), LL(0x49d18f8c,0xa7cbf5cb),LL(0xd351d0bd,0xb159a6c2),LL(0x57b9faae,0xe501bee6),
   LL(0x7ffa6fb7,0xfd21c745),LL(0x6eb966ff,0x9f082577),LL(0x6cf4bbb6,0x7bc7fd6a), LL(0xadae7e60,0x01fbbec5),LL(0xdb5201fd,0x8ede95fa),LL(0x505dc527,0xfe027d78),
   LL(0x4ab8d622,0x556427f0),LL(0x2d8a1b65,0xac06be29),LL(0xc1517a5a,0x24ffe89e), LL(0xbb687bbe,0x8ed6619f),LL(0x5e6229f5,0xc391ab29),LL(0x6668e363,0xc6accdea),
   LL(0x2236abab,0xaa1a8aae),LL(0xbbbb5c34,0x7ff0f90e),LL(0xcdbbe595,0x342421e1), LL(0x158fa276,0x3a43eb45),LL(0xb1c8d9e2,0x3f206b59),LL(0xd1fb731b,0x926495ee),
   LL(0x772ae335,0x64dee795),LL(0x27bb58cf,0x5536e14e),LL(0x68065d10,0x559824cb), LL(0x5fca59f5,0x6780d28c),LL(0xc5672db7,0xb1f51778),LL(0xe64c87fb,0xc295e9dc),
   LL(0x6a759ba8,0x43aa71ca),LL(0xed6e67a5,0x296cde1c),LL(0x2481f6c7,0x97eff2f8), LL(0xee4bf211,0x311c89d7),LL(0x20c5e8fb,0xa03cf7db),LL(0xa30c9a13,0xf181425c),
   LL(0xef8041bd,0xe9164a69),LL(0xf14e75d4,0x10616709),LL(0x31432c48,0x3d0fda80), LL(0xedfce733,0xff34f02e),LL(0x97785106,0x47e53d79),LL(0x0c42b429,0xb73df841),
   LL(0x16d00d56,0x430590ae),LL(0x0b6db145,0xb36bf8a4),LL(0x631b8af0,0xf0fbf1b8), LL(0x7c97735a,0xd285813a),LL(0x233bbcb4,0x3be81538),LL(0xece4421e,0x5dce4f9e),
   LL(0x115acbd2,0xdf3aafe8),LL(0x4e168fd2,0x01352bc5),LL(0xafcfd633,0x7948b02b), LL(0x559eebfe,0xd8976862),LL(0x8de4fafd,0x752a202a),LL(0x47600461,0xf9964960),
   LL(0x45629c12,0xd8e2a4f5),LL(0x0e8d9c56,0x53d8b1f9),LL(0xce8a7e4b,0x69f85fa3), LL(0x50234689,0x78e35e04),LL(0x8ffbcfd7,0xe0274e82),LL(0x0c75b636,0x3bed2d8f),
   LL(0x373fcc37,0x5ad6f7e7),LL(0x347b8ead,0x3ae4bbd9),LL(0xdf9414dd,0x3ef24805), LL(0xed9355e0,0x5d655d51),LL(0x12d96970,0x93e831a1),LL(0xa1cc5116,0x9d02f97d),
   LL(0xc8216057,0xad5bc00d),LL(0xa7efc99a,0x7aa8806c),LL(0x63fe1b91,0x1abc67a2), LL(0xe3315539,0xdcef61b3),LL(0xbd3f1ea2,0x31a563f8),LL(0xb76036c6,0x86e84857),
   LL(0xc967c01d,0x22a2bca9),LL(0x6184d879,0x15300b29),LL(0x24018b6c,0xcb4eece3), LL(0x4bebca46,0xee715ca1),LL(0x0bbbd9c5,0x8d9daf6a),LL(0xd918c1bd,0x683a7647),
   LL(0x53eb0ffd,0xeda60e7d),LL(0x71a51e14,0x537aefa1),LL(0xcd7ff005,0x7927da15), LL(0xf7f04011,0x6191e2df),LL(0x4bba4547,0x7bedb9cb),LL(0x89b2a74a,0xa50562a9),
   LL(0x606c84ad,0x17eb0b47),LL(0x7c97dbb9,0xab88aa51),LL(0xd1818d8b,0x07ea03ae), LL(0x9374b72b,0xf73e2f04),LL(0x86858702,0xcb29129c),LL(0x98608a71,0x3620afda),
   LL(0x716a5874,0xb810b51e),LL(0xa889add3,0x9ec17162),LL(0x19089d83,0xfd276227), LL(0xbb1c7343,0x063aee05),LL(0xe784bbe9,0x87cb9fc6),LL(0xd7489e40,0x782d098e),
   LL(0x8f090898,0x7c43b0a9),LL(0xbdca71ed,0xf8f3b085),LL(0x3f9f4e6e,0xc9b19fd1), LL(0xc47961e6,0x6e42f7f0),LL(0x31020f60,0x45aaf72a),LL(0x2550f424,0x661616e1),
   LL(0x20db7e4a,0xb16f561f),LL(0xa4281dd3,0x83f2a22b),LL(0x43f5d715,0x31f11530), LL(0x1a9f9532,0xbb3cc73e),LL(0xf1fe503c,0xe86e672b),LL(0x6997be05,0xf4c9b96c),
   LL(0xe3cb641a,0x4b0bb45e),LL(0xb5ddb389,0x741adc8d),LL(0xbc920f0d,0xd87c5699), LL(0x15b6ef33,0x55bcf8a4),LL(0x1245ec9c,0x68eb1b09),LL(0x4c03f712,0x7b692b42),
   LL(0x829adb45,0x835194bf),LL(0xe4bd2b50,0x5f3fed6a),LL(0xfffe5c2e,0x32e24c5e), LL(0xb9a052b8,0xed361d44),LL(0x0b43353b,0x512fea09),LL(0xd5a71bbb,0x1d35cd1f),
   LL(0x05b5e6d9,0xba48388b),LL(0x3b0b203b,0x07136b21),LL(0x765b8607,0x8ab4af31), LL(0x74816f5a,0x5107c05c),LL(0x4a5d27b7,0x8d7cfad6),LL(0xd9c2aa8d,0xbc507c9c),
   LL(0x467fe1f2,0x6ed919c9),LL(0xfb1bf823,0x9a75c404),LL(0xfefebd90,0x5eff17b8), LL(0x521a3668,0xa8b6ed77),LL(0x251c6e8c,0x0f8b8ae7),LL(0x631669c0,0xe45c3453),
},
/* digit=27 base_pwr=2^189 */
{
   LL(0x65fd193b,0x443763e3),LL(0x98d2ccd9,0x23523a88),LL(0xff0d967e,0x0a6f8f92), LL(0x009db3b0,0x1263e468),LL(0xbd562f47,0xee4dc8c1),LL(0xafc849c5,0x76f03c5c),
   LL(0x99bd9bfe,0x8d976c26),LL(0x58ed4760,0xb19a85a7),LL(0x1e34868e,0x13dca814), LL(0x54cc6d8a,0x056e40ad),LL(0x8fcf3241,0x861dc2cf),LL(0xd3afc6cd,0xa6a8a21d),
   LL(0x7b77442c,0xe6eeae3d),LL(0x8aa82aef,0x057453a1),LL(0x9a9a0869,0x5a1dd863), LL(0x8372157f,0x15b34a60),LL(0x892b8e9a,0xc35f2257),LL(0xcdd030f9,0x0476b29a),
   LL(0x490b43e5,0x1f623ab9),LL(0x6d3b5faa,0x0fb153e0),LL(0x126be417,0x11ecf7af), LL(0xf9d44cdb,0xc5588fe3),LL(0x80576c99,0x93bd13f4),LL(0xf7cca387,0x458f50aa),
   LL(0x05cadd86,0x33e0840f),LL(0x1288d2ab,0xa4ceb96c),LL(0x2574cecd,0xf18de0a1), LL(0xbd606235,0xb1eecb58),LL(0x243a5639,0x46f1c0e3),LL(0x2f8617f4,0xfbf9ada6),
   LL(0xfdc33e8a,0x29778c36),LL(0x7c7a4b81,0xe96d19b3),LL(0x87bd6ef2,0x011f2a94), LL(0x7b36e1c8,0xd84b0388),LL(0x6e6d91ea,0x5f21366d),LL(0x670039fe,0x0c884a47),
   LL(0x5485e744,0xdff19ddb),LL(0x826523ab,0xb626116c),LL(0x10d50cd3,0x2f4f0fd8), LL(0x24125350,0x5dfd1acb),LL(0x53f2ab5e,0x09c128b0),LL(0x4f9b2256,0x9f940ced),
   LL(0x917ef295,0x347c7bb6),LL(0xe7525a0c,0x3578fbbf),LL(0xdaade6c5,0xd64bdb43), LL(0xeff35295,0xdb9014a1),LL(0x8bd89301,0x1c18a193),LL(0x8573ec51,0xccd8059d),
   LL(0xff89a745,0xd263620d),LL(0x8b578c94,0x485093e4),LL(0x831165c3,0x6cc8df06), LL(0x72a1e4bb,0x623c1f86),LL(0x1ae4359b,0x1857c157),LL(0xf8c83db9,0x51252b88),
   LL(0xf3eaa85f,0x75c0f153),LL(0x296331ba,0x8524e8a3),LL(0xf153b5e3,0xbdfb652d), LL(0xf3645389,0x650adac3),LL(0xe5150ccd,0x59382d93),LL(0x40e57d00,0xf3310865),
   LL(0x3135c8cc,0x43c89d80),LL(0x8ef6e51f,0x5f0b989e),LL(0xe67cd4d4,0xc6b0f722), LL(0x2433fada,0x35eb83b7),LL(0x688247c4,0xb18db37b),LL(0x51703d00,0x0a444f19),
   LL(0x57840670,0x8361e5ab),LL(0x42af05a2,0x4f46eb2d),LL(0x4520d223,0x34632e05), LL(0x66bdbe8a,0x4aa80691),LL(0xb5742363,0x467d5d3a),LL(0x0894ba88,0x5af9783e),
   LL(0x0cb7f96b,0x84644ff9),LL(0x522d6aaf,0x0ae33ac3),LL(0x2cc8944b,0x017c4b01), LL(0x6a2ddda6,0x9923bca6),LL(0x8c9b621e,0x6627fcc2),LL(0x17bdfa4f,0xa29f1b1e),
   LL(0x1cad89b6,0x534ff925),LL(0xc2042459,0x0863e52b),LL(0xaf25d3a6,0x5306b082), LL(0xbbbd9f5b,0xabb1a0b0),LL(0xb65cde23,0x7e3bd9f2),LL(0xc228a0ec,0x001c50b1),
   LL(0xb9339a10,0x86cb78d0),LL(0x1caa6cdf,0xa4b51028),LL(0x21d335bb,0xc52bad81), LL(0xc2fe7de5,0x4301dba7),LL(0x7d6b198e,0xc7b39fd3),LL(0x2a96f21b,0x28e0fdea),
   LL(0xbabe8679,0x59373e25),LL(0x9fd9ed30,0xd44cef09),LL(0x58210099,0xfafb37c0), LL(0x1174c9f1,0x19aea5a6),LL(0xaa0d14c2,0x95527d4f),LL(0xabaf1d12,0x3f19e7e2),
   LL(0xcb8a82f6,0xe6d77312),LL(0x63ded2df,0x1e5c2945),LL(0xa2eb913f,0xe6eba908), LL(0xd2a8cdaf,0xe7a562f9),LL(0xa209420a,0xc659bdc9),LL(0xc59d4edf,0x9d016dd4),
   LL(0x74da1b13,0x4bdd6da2),LL(0xe9df9fa8,0xcdee393f),LL(0x2e6bac46,0x64890274), LL(0xb2d61427,0x8843a6bd),LL(0x6efc5b51,0xfd6c2073),LL(0x035128e4,0x12029497),
   LL(0xbe06abde,0x680d1bf9),LL(0x452f6c1c,0xb37decb2),LL(0x6ec32edc,0x981e840c), LL(0x84360e72,0xd5257167),LL(0x18300091,0x558abf57),LL(0x72ecaad4,0x3a7b592d),
   LL(0xf7312883,0xcdc53ab7),LL(0x0571634d,0x69af637c),LL(0xc7cc70ca,0xec10a5e8), LL(0x424399ce,0xa90c48d4),LL(0x341ddfbc,0xe05856ae),LL(0xd3e4c39c,0x3a1976b2),
   LL(0x90220209,0x9ed5893d),LL(0x81b6ced6,0x45ae0a2a),LL(0x546a8d70,0xe929ce12), LL(0x455cdddf,0x267c3d5b),LL(0xcea5169b,0x4d9fa92f),LL(0xe94a3b42,0xb925af46),
   LL(0xe118cce2,0xe404af85),LL(0x8fc5766e,0xd5a91ab7),LL(0x5e924307,0x71c251eb), LL(0xf7a25052,0xe3cb6d55),LL(0x18b850c9,0x968fed5f),LL(0x2174cb4b,0x623893ba),
   LL(0xd3d0e873,0x67d096e1),LL(0x2541a85b,0x4487615c),LL(0xfa518d4c,0xe87a5c0f), LL(0x8371fee6,0xb0b2a717),LL(0xf0d1d1fa,0x0c2e5455),LL(0x11189a5d,0x42cf786c),
   LL(0x0075ca1a,0x6477d9cb),LL(0x0bada61a,0xff28d896),LL(0x7d671caf,0xea81a421), LL(0xe5e640ea,0x40762d23),LL(0xd43891af,0xfda94df7),LL(0x527b4662,0x6349fb9b),
   LL(0x3b2603b3,0x3b5ff95f),LL(0xf23c44e6,0x6b633ea1),LL(0xe90ab3cb,0xb54db154), LL(0x37f6bcde,0xa92deb1d),LL(0x4eb88b8c,0x7b328bf5),LL(0xde5545e4,0x4ce336ee),
   LL(0x1750434d,0x072846ed),LL(0xef744765,0xef46faee),LL(0xf439980c,0x7a858840), LL(0x6aecf70c,0xec75222e),LL(0x1a4ef06e,0x2ece8f59),LL(0x3d233bad,0x82784973),
   LL(0x3b3b6956,0xaee6f0e5),LL(0xf4eb1f77,0x04ceab91),LL(0xdcbb09d4,0xc6977800), LL(0xc8203c8b,0x8948d95e),LL(0x4f31a369,0xa354e6e3),LL(0x42785a63,0x10e7ebf6),
   LL(0x426a2da6,0x5c233e59),LL(0x5aff8031,0xfc7c21ac),LL(0xae2b7618,0x713ae9f6), LL(0x8ffa178c,0x958babfe),LL(0x404c1ece,0xb21fe414),LL(0x1e4f7c28,0x1fd8cf6a),
   LL(0x482a1f04,0xb68bc919),LL(0xfbee302f,0x7ff1f4be),LL(0x6f98c545,0x5d510b76), LL(0x04a84a52,0x0a54942b),LL(0x8d306d4d,0xdc70958c),LL(0x638a365c,0xd5564cd0),
   LL(0x295ed5b7,0x7275aaed),LL(0xa7a642b1,0xc6c05b8a),LL(0xa499ac12,0x5572818c), LL(0x038c961d,0xdf0a6fa3),LL(0x798ec56d,0x7ed5630f),LL(0x4acc0009,0x206c39e3),
   LL(0x63cc9e62,0xc371e275),LL(0x6e06e663,0x425ae991),LL(0x155a4978,0xa2aa623a), LL(0x82b7846e,0x021e0916),LL(0x07ccc4b1,0x0500f013),LL(0x891bba46,0xd62661e5),
   LL(0xd612756b,0x38909c56),LL(0x090f7ae5,0xd362d7a3),LL(0x9a871772,0xaf3ba39c), LL(0x575c94d9,0x28dfce6b),LL(0xfaf575ae,0xb556c52c),LL(0x710956bf,0x6c0131b7),
   LL(0x071961ab,0x33ce3aed),LL(0x69b35991,0x226aa630),LL(0x2398b3af,0xc42f173d), LL(0xfc2e6625,0x5d4d9cca),LL(0x4790b7b4,0x767e6911),LL(0x184e1458,0x81a0782f),
   LL(0x4a451326,0x77aab96a),LL(0xe916fe1d,0x8ba50ba3),LL(0xea83134c,0x4e839eec), LL(0x10e0b57b,0xc1f0f693),LL(0xe2b978ff,0xfd32f86a),LL(0x207746dd,0xa4737150),
   LL(0xa3db3bbd,0x454c0df4),LL(0xb0f36f88,0x0edad6a0),LL(0x5dcf2b92,0xe233816f), LL(0x53dab969,0x25098373),LL(0x102254b6,0x436f8133),LL(0x4eedcc6f,0x9ce8f3ae),
   LL(0x4b60ac27,0xf64e0fad),LL(0xe3f9cebf,0xea44c9f8),LL(0x84fe9a3a,0x719d958b), LL(0x27bf9bd3,0x5a5e47e8),LL(0xf50de48a,0x6f8c6ee1),LL(0x1971c6f2,0x5dc81c5f),
   LL(0xe435f073,0x1e41a11d),LL(0x49ef9817,0x7b20a534),LL(0x111d8548,0x1ab1fb5a), LL(0x43450972,0x40d481ef),LL(0xddcf8585,0x28382f8b),LL(0x1b0f6b65,0x5a75f9e6),
   LL(0xb1f8052b,0x41ac6def),LL(0xff4c9377,0xc417b22b),LL(0x3ba0240f,0x1c7a1067), LL(0xc99883cb,0x61005fe1),LL(0x6433295e,0xde900b52),LL(0x97d9ccb1,0x1eb1f666),
   LL(0x65928ad1,0x86410ced),LL(0x05383f3a,0x281a9d46),LL(0x11cdecb0,0xa2b718a9), LL(0x6fb1cbee,0x67043aa6),LL(0x94dc2a31,0x5e2de7f3),LL(0x18e2d13b,0x7796cacb),
   LL(0x413f1989,0x18e05e2d),LL(0x3e5cc915,0x2b2b27e9),LL(0x8836a712,0xa32587cf), LL(0x34c64b87,0x0cf77bb0),LL(0x6cafe545,0xf4d7f7ca),LL(0xa18bc173,0xf2191e2f),
   LL(0xaf600892,0xc338b3de),LL(0x35ee6915,0x43c76ec6),LL(0x5564fc33,0x4fdad82e), LL(0xd93b474f,0x089a676d),LL(0x62b06515,0x44a02780),LL(0x2067dba5,0x65fe8051),
   LL(0xf6549736,0xf1c24fd9),LL(0xc063acd4,0xf9d94656),LL(0x95a57c69,0x20c24772), LL(0x8993633e,0x91f7d4d6),LL(0x4444d45b,0xe9d75b4d),LL(0xb5bcc9cf,0x3b5a8bf8),
   LL(0x1b093e28,0xf93e1494),LL(0xfcd88fcf,0x8ce5f30d),LL(0xbfcff34f,0x8fe20824), LL(0x7eec26d2,0x23a22164),LL(0x72404851,0x9b0f8456),LL(0x0e6c550f,0x135ff4a9),
   LL(0x6ef7ad8c,0x70a42efc),LL(0xd82f0cec,0x2f6af7cd),LL(0x7a83b569,0xa0e0549b), LL(0xd3adf095,0xf580e86c),LL(0x8712a121,0x3c2d891a),LL(0xb9d07525,0xf1f2cd22),
   LL(0x461ab8fe,0x7e90bc98),LL(0xc3e7e4df,0xf27a52f3),LL(0xe71da883,0x99813684), LL(0x5c5d56d6,0x2bc98a5a),LL(0x9d789345,0x17248209),LL(0x7dd92590,0x3312dfe6),
   LL(0x26c45ea7,0xfeb5bd64),LL(0xea91cffa,0xfd9bbc06),LL(0xb59f6e6f,0x8be51e9c), LL(0x12663168,0x2082f3c9),LL(0xb78ce9a5,0xe57a2832),LL(0xe4f54287,0xb0d7aac6),
   LL(0x338a630c,0xdad61bc3),LL(0xe66f4efa,0xcffda1bb),LL(0xa61cc9ef,0xccb2e20e), LL(0x3e09551f,0x8d2efcc4),LL(0x617bb37e,0x4e8f46b8),LL(0x2d9b4751,0xc516bc0f),
   LL(0x8992f590,0x753b4f58),LL(0x994a51af,0x55461379),LL(0x0d16de56,0x9f0a37e6), LL(0x204a2e7c,0xb346ff69),LL(0xce235822,0xfaef890d),LL(0x98b923c7,0x5b046b64),
   LL(0x29e950a8,0xb55daedf),LL(0xd3a1ff37,0x79b58c3a),LL(0xec3ed694,0x72c9d2d9), LL(0x9ddf8225,0xd119d76b),LL(0xce66c9a2,0x330952af),LL(0xb0e5e0dc,0xe062e16a),
   LL(0x2d1cc8bd,0xe8cd1c24),LL(0x799a5052,0xf2f0da35),LL(0xbe0be8af,0xe52e337a), LL(0x6a2cbabf,0xeae205cf),LL(0x0acc8002,0xdddc1a94),LL(0x3d1031df,0xf559ec0f),
   LL(0xefc11888,0x5d22ffc2),LL(0x4ff02455,0xdbb1c1ef),LL(0xc63b8f6f,0x543b4752), LL(0x883a7c99,0xd08ef2e5),LL(0xf38b3b3c,0x58ac93a0),LL(0xd8cac2ad,0x3c6c8105),
   LL(0xdf1587c5,0x538bffc6),LL(0xa1a2eb88,0x0d5a4dd0),LL(0xc155f055,0xb9730932), LL(0xb9b8cd61,0xa21c36d2),LL(0x5320c28d,0xc435c223),LL(0x38120687,0x3230ce02),
   LL(0xc37ce54a,0xea0530df),LL(0x06b232a0,0xb1894861),LL(0xf018336a,0xca2122b1), LL(0x38a380e9,0x08135d85),LL(0xe3028bd7,0xf256c6e5),LL(0x663fdf24,0xdacc9271),
   LL(0x053cff11,0x31da52c7),LL(0xcc61252b,0x39051bad),LL(0x317a0ba8,0xf380f3e8), LL(0xd42451b7,0x212de129),LL(0x98e4747e,0x146f3c25),LL(0x2acac2c4,0x844da479),
   LL(0x4cabc97f,0x9b8a6fb0),LL(0x0fbc8135,0xba6da1e3),LL(0x561f0e02,0xb076c93d), LL(0xf54d9d7d,0x2f940cf1),LL(0xec9daef2,0x14a542cb),LL(0x19d5ffa1,0x0579ad14),
   LL(0xd4065d34,0xf8f179dd),LL(0x30ff0d6d,0x355fc75d),LL(0xc8fac2e6,0x91f6f5fb), LL(0x3e3fd917,0x8baf55f3),LL(0x077d4cef,0x6c4a7378),LL(0x192e1ce6,0x16be9fa8),
   LL(0x92e5aec0,0xa3a9011c),LL(0xf37a5ab0,0x4fa0764f),LL(0xb1e1e01a,0x75ffe5ac), LL(0x51756cb8,0x3cd4ee19),LL(0xe2636498,0xc557a98c),LL(0x2c691a68,0x53a49769),
   LL(0x752dbe26,0xdcdf8e6f),LL(0x7564740f,0xeaa9a260),LL(0x31a5df7f,0x9298d7e2), LL(0xa1b1f37a,0x47cdc94f),LL(0x3c25a836,0x26578ece),LL(0xec817acd,0x0aab1d74),
   LL(0x5911f2ae,0xdd0678c2),LL(0x126b83a4,0xb903d7a7),LL(0xecd46d83,0x4efeef39), LL(0xb0d49ea1,0xbb4b0af7),LL(0xb6c2ae9c,0xc62fa8f1),LL(0x3e453617,0x897a8d06),
   LL(0x54820219,0x335e7a19),LL(0x9104367e,0x8ae7926b),LL(0xcf0f8e5e,0xae2225b5), LL(0xd3fddb3b,0xc276e1af),LL(0x6a507a40,0x7b05028e),LL(0xef73bdac,0x85d2bd18),
   LL(0xc71dd981,0x6953e0e2),LL(0x0e9a352f,0x7f739d63),LL(0xbe280f30,0xb6620388), LL(0xd020979e,0x89d5450f),LL(0x66d59c07,0x0370df8a),LL(0x0e91d327,0x76a60330),
   LL(0x12f802fb,0x56161a39),LL(0x2ca0151f,0xe9e9e58c),LL(0xde14a0f7,0xc6da98d6), LL(0xe70f9647,0xdc21514c),LL(0x523f5cb8,0xa95c863d),LL(0x25f1c818,0x3e1cc20c),
   LL(0x8d6f8ca1,0xeb6c9c82),LL(0xb9d65ec1,0xc57e3312),LL(0x0f121596,0x8b4328c3), LL(0x067036a8,0x1affe768),LL(0xe3091467,0x42586aa8),LL(0x30b7ad74,0x12de5d18),
   LL(0x34933969,0x06705f70),LL(0x7267ab88,0x4ce99cd6),LL(0x1f916ab1,0x2bc4cfb9), LL(0xcd9b8de7,0xe458fea7),LL(0xedafc50a,0x59778ef8),LL(0x86ea0de9,0x58651c07),
}
};
#endif /* _DISABLE_ECP_192R1_HARDCODED_BP_TBL_ */
#endif  /* _IPP_DATA */


IPP_OWN_DEFN (const cpPrecompAP*, gfpec_precom_nistP192r1_fun, (void))
{
   static cpPrecompAP t = {
      /* w */                  7,
      /* select function */    p192r1_select_ap_w7,
      /* precomputed data */   (BNU_CHUNK_T*)ec_p192r1_precomputed
   };
   return &t;
}
