// ext3grep -- An ext3 file system investigation and undelete tool
//
//! @file debug.cc
//! @brief This file contains the definitions of debug related objects and functions.
//
// Copyright (C) 2008, by
// 
// Carlo Wood, Run on IRC <carlo@alinoe.com>
// RSA-1024 0x624ACAD5 1997-01-26                    Sign & Encrypt
// Fingerprint16 = 32 EC A7 B6 AC DB 65 A6  F6 F6 55 DD 1C DC FF 61
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef USE_PCH
#include "sys.h"                        // Needed for platform-specific code
#endif

#ifdef CWDEBUG

#ifndef USE_PCH
#include <cctype>                       // Needed for std::isprint
#include <iomanip>                      // Needed for setfill
#include <map>
#include <string>
#include <sstream>
#include <signal.h>
#include "debug.h"
#ifdef USE_LIBCW
#include <libcw/memleak.h>		// memleak_filter
#endif
#endif // USE_PCH

namespace debug {
  namespace channels {	// namespace DEBUGCHANNELS
    namespace dc {

#ifndef DOXYGEN
#define DDCN(x) (x)
#endif
      // Add new debug channels here.

      //channel_ct custom DDCN("CUSTOM");	//!< This debug channel is used for ...

    } // namespace dc
  } // namespace DEBUGCHANNELS

  // Anonymous namespace, this map and its initialization functions are private to this file
  // for Thead-safeness reasons.
  namespace {

    /*! @brief The type of rcfile_dc_states.
     * @internal
     */
    typedef std::map<std::string, bool> rcfile_dc_states_type;

    /*! @brief Map containing the default debug channel states used at the start of each new thread.
     * @internal
     *
     * The first thread calls main, which calls debug::init which will initialize this
     * map with all debug channel labels and whether or not they were turned on in the
     * rcfile or not.
     */
    rcfile_dc_states_type rcfile_dc_states;

    /*! @brief Set the default state of debug channel \a dc_label.
     * @internal
     *
     * This function is called once for each debug channel.
     */
    void set_state(char const* dc_label, bool is_on)
    {
      std::pair<rcfile_dc_states_type::iterator, bool> res =
          rcfile_dc_states.insert(rcfile_dc_states_type::value_type(std::string(dc_label), is_on));
      if (!res.second)
        Dout(dc::warning, "Calling set_state() more than once for the same label!");
      return;
    }

    /*! @brief Save debug channel states.
     * @internal
     *
     * One time initialization function of rcfile_dc_state.
     * This must be called from debug::init after reading the rcfile.
     */
    void save_dc_states(void)
    {
      // We may only call this function once: it reflects the states as stored
      // in the rcfile and that won't change.  Therefore it is not needed to
      // lock `rcfile_dc_states', it is only written to by the first thread
      // (once, via main -> init) when there are no other threads yet.
      static bool second_time = false;
      if (second_time)
      {
        Dout(dc::warning, "Calling save_dc_states() more than once!");
	return;
      }
      second_time = true;
      ForAllDebugChannels( set_state(debugChannel.get_label(), debugChannel.is_on()) );
    }

  } // anonymous namespace

  /*! @brief Returns the the original state of a debug channel.
   * @internal
   *
   * For a given \a dc_label, which must be the exact name (<tt>channel_ct::get_label</tt>) of an
   * existing debug channel, this function returns \c true when the corresponding debug channel was
   * <em>on</em> at the startup of the application, directly after reading the libcwd runtime
   * configuration file (.libcwdrc).
   *
   * If the label/channel did not exist at the start of the application, it will return \c false
   * (note that libcwd disallows adding debug channels to modules - so this would probably
   * a bug).
   */
  bool is_on_in_rcfile(char const* dc_label)
  {
    rcfile_dc_states_type::const_iterator iter = rcfile_dc_states.find(std::string(dc_label));
    if (iter == rcfile_dc_states.end())
    {
      Dout(dc::warning, "is_on_in_rcfile(\"" << dc_label << "\"): \"" << dc_label << "\" is an unknown label!");
      return false;
    }
    return iter->second;
  }

  /*! @brief Initialize debugging code from new threads.
   *
   * This function needs to be called at the start of each new thread,
   * because a new thread starts in a completely reset state.
   *
   * The function turns on all debug channels that were turned on
   * after reading the rcfile at the start of the application.
   * Furthermore it initializes the debug ostream, its mutex and the
   * margin of the default debug object (Dout).
   */
  void init_thread(void)
  {
    // Turn on all debug channels that are turned on as per rcfile configuration.
    ForAllDebugChannels(
        if (!debugChannel.is_on() && is_on_in_rcfile(debugChannel.get_label()))
	  debugChannel.on();
    );

    // Turn on debug output.
    Debug( libcw_do.on() );
#if LIBCWD_THREAD_SAFE
    Debug( libcw_do.set_ostream(&std::cout, &cout_mutex) );
#else
    Debug( libcw_do.set_ostream(&std::cout) );
#endif

    static bool first_thread = true;
    if (!first_thread)			// So far, the application has only one thread.  So don't add a thread id.
    {
      // Set the thread id in the margin.
      char margin[12];
      sprintf(margin, "%-10lu ", pthread_self());
      Debug( libcw_do.margin().assign(margin, 11) );
    }
  }

  /*! @brief Initialize debugging code from main.
   *
   * This function initializes the debug code.
   */
  void init(void)
  {
#if CWDEBUG_ALLOC && defined(USE_LIBCW)
    // Tell the memory leak detector which parts of the code are
    // expected to leak so that we won't get an alarm for those.
    {
      std::vector<std::pair<std::string, std::string> > hide_list;
      hide_list.push_back(std::pair<std::string, std::string>("libdl.so.2", "_dlerror_run"));
      hide_list.push_back(std::pair<std::string, std::string>("libstdc++.so.6", "__cxa_get_globals"));
      // The following is actually necessary because of a bug in glibc
      // (see http://sources.redhat.com/bugzilla/show_bug.cgi?id=311).
      hide_list.push_back(std::pair<std::string, std::string>("libc.so.6", "dl_open_worker"));
      memleak_filter().hide_functions_matching(hide_list);
    }
    {
      std::vector<std::string> hide_list;
      // Also because of http://sources.redhat.com/bugzilla/show_bug.cgi?id=311
      hide_list.push_back(std::string("ld-linux.so.2"));
      memleak_filter().hide_objectfiles_matching(hide_list);
    }
    memleak_filter().set_flags(libcwd::show_objectfile|libcwd::show_function);
#endif

    // The following call allocated the filebuf's of cin, cout, cerr, wcin, wcout and wcerr.
    // Because this causes a memory leak being reported, make them invisible.
    Debug(set_invisible_on());

    // You want this, unless you mix streams output with C output.
    // Read  http://gcc.gnu.org/onlinedocs/libstdc++/27_io/howto.html#8 for an explanation.
    //std::ios::sync_with_stdio(false);

    // Cancel previous call to set_invisible_on.
    Debug(set_invisible_off());

    // This will warn you when you are using header files that do not belong to the
    // shared libcwd object that you linked with.
    Debug( check_configuration() );

    Debug(
      libcw_do.on();		// Show which rcfile we are reading!
      ForAllDebugChannels(
        while (debugChannel.is_on())
	  debugChannel.off()	// Print as little as possible though.
      );
      read_rcfile();		// Put 'silent = on' in the rcfile to suppress most of the output here.
      libcw_do.off()
    );
    save_dc_states();

    init_thread();
  }

#if CWDEBUG_LOCATION
  /*! @brief Return call location.
   *
   * @param return_addr The return address of the call.
   */
  std::string call_location(void const* return_addr)
  {
    libcwd::location_ct loc((char*)return_addr + libcwd::builtin_return_address_offset);
    std::ostringstream convert;
    convert << loc;
    return convert.str();
  }
#endif

} // namespace debug

#endif // CWDEBUG

//-----------------------------------------------------------------------------
//
// Debug stuff

#include "debug.h"

// Define our own assert.
#ifdef DEBUG
#include <iostream>
#include "backtrace.h"
#include <signal.h>
void assert_fail(char const* expr, char const* file, int line, char const* function)
{
  std::cout << file << ':' << line << ": " << function << ": Assertion `" << expr << "' failed." << std::endl;
  std::cout << "Backtrace:\n";
  dump_backtrace_on(std::cout);
  raise(6);
}
#endif

#if EXTERNAL_BLOCK
#if 0
// Ian Jacobi's '\\' file.
uint32_t someones_inode_count = std::numeric_limits<uint32_t>::max();
unsigned char someones_block[SOMEONES_BLOCK_SIZE] = {
  0x21, 0x47, 0xf1, 0x00, 0x0c, 0x00, 0x01, 0x02, 0x2e, 0x00, 0x00, 0x00, 0x20, 0x47, 0xf1, 0x00,
  0x0c, 0x00, 0x02, 0x02, 0x2e, 0x2e, 0x00, 0x00, 0x22, 0x47, 0xf1, 0x00, 0xe8, 0x0f, 0x01, 0x02,
  0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0
};
#else
// siegelring -- directory, not detected as directory.
uint32_t someones_inode_count = 1465920;
unsigned char someones_block[SOMEONES_BLOCK_SIZE] = {
  0x34, 0xba, 0x08, 0x00, 0x0c, 0x00, 0x01, 0x02, 0x2e, 0x00, 0x00, 0x00, 0x2d, 0xba, 0x08, 0x00,
  0x0c, 0x00, 0x02, 0x02, 0x2e, 0x2e, 0x00, 0x00, 0x35, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01,
  0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30, 0x2e, 0x66, 0x6f, 0x72, 0x6d, 0x00, 0x00, 0x00,
  0x36, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01, 0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30,
  0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00, 0x00, 0x00, 0x37, 0xba, 0x08, 0x00, 0x24, 0x00, 0x1b, 0x01,
  0x46, 0x61, 0x68, 0x72, 0x74, 0x65, 0x6e, 0x62, 0x75, 0x63, 0x68, 0x2d, 0x61, 0x75, 0x66, 0x72,
  0x75, 0x66, 0x2d, 0x6a, 0x61, 0x76, 0x61, 0x2e, 0x74, 0x78, 0x74, 0x00, 0x38, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x0b, 0x01, 0x46, 0x62, 0x31, 0x24, 0x34, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00,
  0x39, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0f, 0x01, 0x54, 0x61, 0x62, 0x6c, 0x65, 0x44, 0x65, 0x6d,
  0x6f, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x3a, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0e, 0x01,
  0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00,
  0x3b, 0xba, 0x08, 0x00, 0x10, 0x00, 0x08, 0x01, 0x2e, 0x6e, 0x62, 0x61, 0x74, 0x74, 0x72, 0x73,
  0x3c, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0e, 0x01, 0x4a, 0x61, 0x72, 0x4d, 0x61, 0x6b, 0x65, 0x72,
  0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x3d, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x14, 0x01,
  0x54, 0x61, 0x62, 0x6c, 0x65, 0x43, 0x6f, 0x6c, 0x75, 0x6d, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x2e,
  0x6a, 0x61, 0x76, 0x61, 0x3e, 0xba, 0x08, 0x00, 0x14, 0x00, 0x09, 0x01, 0x46, 0x62, 0x31, 0x2e,
  0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00, 0x3f, 0xba, 0x08, 0x00, 0x10, 0x00, 0x08, 0x01,
  0x46, 0x62, 0x31, 0x2e, 0x66, 0x6f, 0x72, 0x6d, 0x40, 0xba, 0x08, 0x00, 0x10, 0x00, 0x08, 0x01,
  0x46, 0x62, 0x31, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x41, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01,
  0x46, 0x62, 0x31, 0x24, 0x35, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x42, 0xba, 0x08, 0x00,
  0x10, 0x00, 0x08, 0x01, 0x46, 0x62, 0x33, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x43, 0xba, 0x08, 0x00,
  0x18, 0x00, 0x0e, 0x01, 0x66, 0x62, 0x5f, 0x4f, 0x6b, 0x74, 0x6f, 0x62, 0x65, 0x72, 0x2e, 0x64,
  0x61, 0x74, 0x00, 0x00, 0x44, 0xba, 0x08, 0x00, 0x24, 0x00, 0x1a, 0x01, 0x70, 0x72, 0x69, 0x6e,
  0x74, 0x62, 0x75, 0x74, 0x74, 0x6f, 0x6e, 0x24, 0x4d, 0x79, 0x42, 0x75, 0x74, 0x74, 0x6f, 0x6e,
  0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x45, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x12, 0x01,
  0x4a, 0x4d, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x4c, 0x6f, 0x61, 0x64, 0x65, 0x72, 0x2e, 0x6a, 0x61,
  0x76, 0x61, 0x00, 0x00, 0x46, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01, 0x46, 0x62, 0x31, 0x24,
  0x36, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x47, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0e, 0x01,
  0x54, 0x61, 0x62, 0x6c, 0x65, 0x44, 0x65, 0x6d, 0x6f, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00, 0x00,
  0x48, 0xba, 0x08, 0x00, 0x20, 0x00, 0x16, 0x01, 0x46, 0x62, 0x31, 0x24, 0x4d, 0x79, 0x54, 0x61,
  0x62, 0x6c, 0x65, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00,
  0x49, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01, 0x66, 0x62, 0x5f, 0x41, 0x75, 0x67, 0x75, 0x73,
  0x74, 0x2e, 0x64, 0x61, 0x74, 0x00, 0x00, 0x00, 0x4a, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0f, 0x01,
  0x66, 0x62, 0x5f, 0x4e, 0x6f, 0x76, 0x65, 0x6d, 0x62, 0x65, 0x72, 0x2e, 0x64, 0x61, 0x74, 0x00,
  0x4b, 0xba, 0x08, 0x00, 0x14, 0x00, 0x09, 0x01, 0x46, 0x62, 0x33, 0x2e, 0x63, 0x6c, 0x61, 0x73,
  0x73, 0x00, 0x00, 0x00, 0x4c, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x13, 0x01, 0x42, 0x72, 0x6f, 0x77,
  0x73, 0x65, 0x72, 0x4d, 0x61, 0x69, 0x6e, 0x54, 0x61, 0x62, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00,
  0x4d, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x13, 0x01, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x62, 0x75, 0x74,
  0x74, 0x6f, 0x6e, 0x24, 0x31, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x4e, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x09, 0x01, 0x55, 0x74, 0x69, 0x6c, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00, 0x00, 0x00,
  0x4f, 0xba, 0x08, 0x00, 0x18, 0x00, 0x10, 0x01, 0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30,
  0x24, 0x31, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x50, 0xba, 0x08, 0x00, 0x18, 0x00, 0x10, 0x01,
  0x66, 0x62, 0x5f, 0x53, 0x65, 0x70, 0x74, 0x65, 0x6d, 0x62, 0x65, 0x72, 0x2e, 0x64, 0x61, 0x74,
  0x51, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01, 0x66, 0x6f, 0x6e, 0x74, 0x74, 0x65, 0x73, 0x74,
  0x2e, 0x66, 0x6f, 0x72, 0x6d, 0x00, 0x00, 0x00, 0x52, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01,
  0x66, 0x6f, 0x6e, 0x74, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00, 0x00, 0x00,
  0x53, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x11, 0x01, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x62, 0x75, 0x74,
  0x74, 0x6f, 0x6e, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00, 0x54, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x0b, 0x01, 0x4d, 0x41, 0x4e, 0x49, 0x46, 0x45, 0x53, 0x54, 0x2e, 0x4d, 0x46, 0x00,
  0x55, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01, 0x46, 0x62, 0x31, 0x24, 0x31, 0x2e, 0x63, 0x6c,
  0x61, 0x73, 0x73, 0x00, 0x56, 0xba, 0x08, 0x00, 0x24, 0x00, 0x1c, 0x01, 0x54, 0x61, 0x62, 0x6c,
  0x65, 0x44, 0x65, 0x6d, 0x6f, 0x24, 0x4d, 0x79, 0x54, 0x61, 0x62, 0x6c, 0x65, 0x4d, 0x6f, 0x64,
  0x65, 0x6c, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x57, 0xba, 0x08, 0x00, 0x20, 0x00, 0x15, 0x01,
  0x54, 0x61, 0x62, 0x6c, 0x65, 0x43, 0x6f, 0x6c, 0x75, 0x6d, 0x6e, 0x54, 0x65, 0x73, 0x74, 0x2e,
  0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00, 0x58, 0xba, 0x08, 0x00, 0x18, 0x00, 0x10, 0x01,
  0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30, 0x24, 0x32, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73,
  0x59, 0xba, 0x08, 0x00, 0x18, 0x00, 0x10, 0x01, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x62, 0x75, 0x74,
  0x74, 0x6f, 0x6e, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x5a, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0d, 0x01,
  0x4a, 0x61, 0x72, 0x4d, 0x61, 0x6b, 0x65, 0x72, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x00, 0x00, 0x00,
  0x5b, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0c, 0x01, 0x66, 0x62, 0x5f, 0x41, 0x70, 0x72, 0x69, 0x6c,
  0x2e, 0x64, 0x61, 0x74, 0x5c, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0f, 0x01, 0x66, 0x61, 0x68, 0x72,
  0x74, 0x65, 0x6e, 0x62, 0x75, 0x63, 0x68, 0x2e, 0x64, 0x61, 0x74, 0x00, 0x5d, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x0c, 0x01, 0x66, 0x62, 0x5f, 0x4d, 0x61, 0x65, 0x72, 0x7a, 0x2e, 0x64, 0x61, 0x74,
  0x5e, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01, 0x46, 0x62, 0x31, 0x24, 0x32, 0x2e, 0x63, 0x6c,
  0x61, 0x73, 0x73, 0x00, 0x5f, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x11, 0x01, 0x54, 0x61, 0x62, 0x6c,
  0x65, 0x44, 0x65, 0x6d, 0x6f, 0x24, 0x31, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
  0x60, 0xba, 0x08, 0x00, 0x18, 0x00, 0x10, 0x01, 0x56, 0x65, 0x72, 0x73, 0x75, 0x63, 0x68, 0x30,
  0x24, 0x33, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x61, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0c, 0x01,
  0x46, 0x62, 0x50, 0x72, 0x69, 0x6e, 0x74, 0x2e, 0x6a, 0x61, 0x76, 0x61, 0x62, 0xba, 0x08, 0x00,
  0x1c, 0x00, 0x13, 0x01, 0x4a, 0x4d, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x4c, 0x6f, 0x61, 0x64, 0x65,
  0x72, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x63, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x13, 0x01,
  0x66, 0x62, 0x5f, 0x46, 0x65, 0x62, 0x72, 0x75, 0x61, 0x72, 0x5f, 0x32, 0x30, 0x30, 0x34, 0x2e,
  0x64, 0x61, 0x74, 0x00, 0x64, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x13, 0x01, 0x66, 0x62, 0x5f, 0x46,
  0x65, 0x62, 0x72, 0x75, 0x61, 0x72, 0x5f, 0x32, 0x30, 0x30, 0x35, 0x2e, 0x64, 0x61, 0x74, 0x00,
  0x65, 0xba, 0x08, 0x00, 0x1c, 0x00, 0x12, 0x01, 0x66, 0x62, 0x5f, 0x4a, 0x61, 0x6e, 0x75, 0x61,
  0x72, 0x5f, 0x32, 0x30, 0x30, 0x34, 0x2e, 0x64, 0x61, 0x74, 0x00, 0x00, 0x66, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x0a, 0x01, 0x55, 0x74, 0x69, 0x6c, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00, 0x00,
  0x67, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01, 0x66, 0x62, 0x5f, 0x4a, 0x75, 0x6c, 0x69, 0x2e,
  0x64, 0x61, 0x74, 0x00, 0x68, 0xba, 0x08, 0x00, 0x18, 0x00, 0x0f, 0x01, 0x66, 0x62, 0x5f, 0x44,
  0x65, 0x7a, 0x65, 0x6d, 0x62, 0x65, 0x72, 0x2e, 0x64, 0x61, 0x74, 0x00, 0x69, 0xba, 0x08, 0x00,
  0x14, 0x00, 0x0b, 0x01, 0x46, 0x62, 0x31, 0x24, 0x33, 0x2e, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x00,
  0x6a, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0b, 0x01, 0x66, 0x62, 0x5f, 0x4a, 0x75, 0x6e, 0x69, 0x2e,
  0x64, 0x61, 0x74, 0x00, 0x6b, 0xba, 0x08, 0x00, 0x14, 0x00, 0x0a, 0x01, 0x66, 0x62, 0x5f, 0x4d,
  0x61, 0x69, 0x2e, 0x64, 0x61, 0x74, 0x00, 0x00, 0x6c, 0xba, 0x08, 0x00, 0xc8, 0x0a, 0x01, 0x01,
  0x9b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0
};
#endif
#else	// !EXTERNAL_BLOCK
// Not used, except in compiler check.
uint32_t someones_inode_count;
unsigned char someones_block[SOMEONES_BLOCK_SIZE];
#endif	// !EXTERNAL_BLOCK
