/*
cdcd - Command Driven CD player
Copyright (C) 1998, 1999 Tony Arcieri
Copyright (C) 2001, 2002, 2003 Fabrice Bauzac

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

  TODO

  Contact a Bash/Readline programmer and report the history_tokenize()
  bug: it returns NULL (should return a pointer to NULL) when it's got
  an empty line.
  
  Set the `char history_comment_char' variable to `#' (?).

  Set history_quotes_inhibit_expansion to 1 (?).

  Take more time to classify functions in files.

*/

#include <stdio.h>
#include <signal.h>
#include <locale.h>
#include <time.h>

#include <mycdaudio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include <errno.h>

/* Reminder: we have to make sure that the getopt files are
   up-to-date.  */
#include "getopt.h"

#include "rlhist.h"
#include "cdcd.h"
#include "conf.h"
#include "cmd_sites.h"
#include "cmd_cdcd.h"
#include "cmd_access.h"
#include "cmd_edit.h"
#include "str.h"
#include "cmdline.h"
#include "interface.h"
#include "global.h"

const char *cdcdrc_verbose_var = "verbose";
const char *cdcdrc_device_var = "device";
const char *cdcd_cddb_host = "freedb.freedb.org";
const char *cdcd_cdindex_host = "www.cdindex.org";
const char *author_email = "noon@users.sourceforge.net";


/* Generic functions.  */

static void
bug ()
{
  fprintf (stderr, "Please submit a bug report to <%s>.\n", author_email);
  exit (1);
}

void
memory_exhausted ()
{
  fputs ("Virtual memory exhausted\n", stderr);
  exit (1);
}

void *
xmalloc (size_t size)
{
  void *value = malloc (size);
  if (!value)
    memory_exhausted ();
  return value;
}

void *
xrealloc (void *ptr, size_t size)
{
  void *value = realloc (ptr, size);
  if (!value)
    memory_exhausted ();
  return value;
}

void
xfree (void *ptr)
{
  free (ptr);
}


/* The more I look at this function, the more I feel like it's
   useless.  I wonder...  -- fb.  */
RETSIGTYPE
signal_handler (int signal)
{
  static int recursive_segv = 0;

  switch (signal)
    {
    case SIGSEGV:
      /* If there's a bug below which causes a segmentation violation,
         this will prevent the infinite recursion.  */
      if (recursive_segv++)
	exit (1);

      printf ("\n"
	      "SIGSEGV caught: Segmentation violation\n"
	      "Please report what you were doing when this occurred to:\n"
	      "%s", author_email);
      putchar ('\n');
      exit (1);
    case SIGTERM:
      putchar ('\n');
      exit (0);
    }
}

void
print_version ()
{
  char dispbuffer[4096];
  cd_version (dispbuffer, 4095);
  dispbuffer[4095] = 0;		/* I'm not sure whether this is done in
				   cd_version() or not, so it's done here.  */
  printf ("cdcd %s\n"
	  "Copyright (C) 1998, 1999 Tony Arcieri.\n"
	  "Copyright (C) 2001, 2002, 2003 Fabrice Bauzac.\n", VERSION);
  pprintf (get_width () - 1,
	   "cdcd is free software, covered by the GNU General "
	   "Public Licence, and you are welcome to change it "
	   "and/or distribute copies of it under certain conditions.  "
	   "There is absolutely no warranty for cdcd.  See the "
	   "file COPYING for details.");
  putchar ('\n');
  printf ("Using %s.\n", dispbuffer);
}

void
print_cmd_line_help ()
{
  puts ("Usage: cdcd [options] [commands]");
  fputs ("Options:\n"
	 "  -h, --help           display this help and exit\n"
	 "  -v, --version        output version information and exit\n"
	 "  -d, --device=DEVICE  use DEVICE instead of /dev/cdrom\n", stdout);
  print_all_commands (cmds, 1);
}

int
main (int argc, char **argv)
{
  signal (SIGSEGV, signal_handler);
  signal (SIGTERM, signal_handler);
  srand (time (0));
  setlocale (LC_ALL, "");

  init_cmdline ();

  init_cmd_cdcd ();
  init_cmd_sites ();
  init_cmd_access ();
  init_cmd_edit ();

  {
    struct cdcdrc conf;
    cdcdrc_read (&conf);
    verbosity = conf.verbosity;
    strncpy (device, conf.device, CDCDRC_DEVICE_LEN - 1);
    device[CDCDRC_DEVICE_LEN - 1] = 0;
  }

  {
    int opt;
    struct option o[] = {
      {"help", no_argument, NULL, 'h'},
      {"version", no_argument, NULL, 'v'},
      {"device", required_argument, NULL, 'd'},
      {NULL}
    };

    while ((opt = getopt_long (argc, argv, "hvd:", o, NULL)) != -1)
      {
	switch (opt)
	  {
	  case 'd':
	    strncpy (device, optarg, CDCDRC_DEVICE_LEN - 1);
	    device[CDCDRC_DEVICE_LEN - 1] = 0;
	    break;
	  case 'v':
	    print_version ();
	    return 0;
	  case 'h':
	    print_cmd_line_help ();
	    return 0;
	  case '?':
	    exit (1);
	  default:
	    fprintf (stderr, "%s:%d: bogus argument parsing\n",
		     __FILE__, __LINE__);
	    bug ();
	  }
      }
  }

  if ((cd_desc = cd_init_device (device)) < 0)
    {
      if (errno == EBUSY)
	{
	  perror ("cdcd");
	  printf ("cdcd: %s appears to be already mounted.\n", device);
	}
      else
	{
	  fputs ("cdcd: cannot initialize ", stderr);
	  perror (device);
	}
      exit (1);
    }

  if (optind != argc)
    {
      char **p;
      p = cv2v0 (argc - optind, argv + optind);
      cmd_cdcd_execute (p);
      freev0 (p);
      exit (0);
    }
  else
    {
      print_version ();
      puts ("Enter ? for help.");
      cmd_cdcd_mainloop ();
      cd_finish (cd_desc);
    }
  return 0;
}
