/* $Cambridge: hermes/src/prayer/session/role.c,v 1.3 2008/09/16 09:59:58 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_session.h"

/* Little support routine, seems to turn up all over the place! */

static char *maybe_strdup(struct pool *p, char *s)
{
    return ((s) ? pool_strdup(p, s) : "");
}

/* role_add() ***********************************************************
 *
 * Add role to list
 *        rl: Role list
 *  personal: Personal name
 *      from: From address
 *  reply_to: Reply to address
 *       fcc: Default Fcc
 * signature: Signature
 *
 ***********************************************************************/

static void
role_add(struct list *rl,
         char *name,
         char *personal,
         char *from, char *reply_to, char *fcc, char *signature)
{
    struct role *new = pool_alloc(NIL, sizeof(struct role));

    new->next = NIL;
    new->personal = maybe_strdup(rl->pool, personal);
    new->from = maybe_strdup(rl->pool, from);
    new->reply_to = maybe_strdup(rl->pool, reply_to);
    new->fcc = maybe_strdup(rl->pool, fcc);
    new->signature = maybe_strdup(rl->pool, signature);

    list_push(rl, (struct list_item *) new, name);
}

/* role_update() ********************************************************
 *
 * Update existing role, or add new role to next of list if none exists.
 *        rl: Role list
 *  personal: Personal name
 *      from: From address
 *  reply_to: Reply to address
 *       fcc: Default Fcc
 * signature: Signature
 *
 ***********************************************************************/

void
role_update(struct list *rl,
            char *name,
            char *personal,
            char *from, char *reply_to, char *fcc, char *signature)
{
    struct list_item *li;

    if (!name)
        return;

    for (li = rl->head; li; li = li->next) {
        struct role *current = (struct role *) li;

        if (!strcmp(current->name, name)) {
            /* Replace existing entry */
            current->name = maybe_strdup(rl->pool, name);
            current->personal = maybe_strdup(rl->pool, personal);
            current->from = maybe_strdup(rl->pool, from);
            current->reply_to = maybe_strdup(rl->pool, reply_to);
            current->fcc = maybe_strdup(rl->pool, fcc);
            current->signature = maybe_strdup(rl->pool, signature);
            break;
        }
    }
    if (li == NIL)
        role_add(rl, name, personal, from, reply_to, fcc, signature);
}

/* ====================================================================== */

/* role_delete() ********************************************************
 *
 * Delete role
 *    rl: Role list
 *  name: Name of role to delete
 *
 ***********************************************************************/

void role_delete(struct list *rl, char *name)
{
    list_remove_byname(rl, name);
}

/* role_find() **********************************************************
 *
 * Lookup role
 *    rl: Role list
 *  name: Name of role to find
 *
 ***********************************************************************/

struct role *role_find(struct list *rl, char *name)
{
    return ((struct role *) list_lookup_byname(rl, name));
}

/* ====================================================================== */

/* role_parse_line() ****************************************************
 *
 * Parse a role from line of user preferences file
 *      rl: role list
 *    line: Line to parse
 * session: For logging purposes only
 *
 ***********************************************************************/

void role_parse_line(struct list *rl, char *line, struct session *session)
{
    char *list[6];
    char *name, *personal, *from, *reply_to, *fcc, *signature;

    if (!((list[0] = options_decode(string_get_token(&line))) &&
          (list[1] = options_decode(string_get_token(&line))) &&
          (list[2] = options_decode(string_get_token(&line))) &&
          (list[3] = options_decode(string_get_token(&line)))))
        return;

    if (line[0]) {
        list[4] = options_decode(string_get_token(&line));
        list[5] = options_decode(string_get_token(&line));

        name = list[0];
        personal = list[1];
        from = list[2];
        reply_to = list[3];
        fcc = list[4];
        signature = list[5];

    } else {
        name = list[0];         /* Backwards compatibility with old format */
        personal = list[1];
        from = "";
        reply_to = list[2];
        fcc = "";
        signature = list[3];

        session->options->save = T;
    }

    role_add(rl, name, personal, from, reply_to, fcc, signature);
}

/* role_print_options() *************************************************
 *
 * Convert role list to textual representation suitable for user
 * preferences file on IMAP server.
 *   rl: Role list to print
 *    b: Output buffer
 *
 ***********************************************************************/

void role_print_options(struct list *rl, struct buffer *b)
{
    struct list_item *li;
    struct role *role;

    for (li = rl->head; li; li = li->next) {
        unsigned long offset = 0L;
        role = (struct role *) li;

        options_print_token(b, role->name, &offset);
        options_print_token(b, role->personal, &offset);
        options_print_token(b, role->from, &offset);
        options_print_token(b, role->reply_to, &offset);
        options_print_token(b, role->fcc, &offset);
        options_print_token(b, role->signature, &offset);
        bputs(b, "" CRLF);
    }
}
