/* $Cambridge: hermes/src/prayer/cmd/cmd_redirect.c,v 1.5 2009/04/03 16:39:07 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_session.h"

static void generate_form(struct session *session,
                          BOOL redirect_enabled,
                          char *redirect_address,
                          BOOL redirect_copy)
{
    struct template_vals *tvals = session->template_vals;
    struct request *request = session->request;
    struct buffer *b = request->write_buffer;

    if (redirect_enabled)
        template_vals_ulong(tvals, "use_redirect", 1);

    if (redirect_address)
        template_vals_string(tvals, "redirect_addr", redirect_address);

    if (redirect_copy)
        template_vals_ulong(tvals, "redirect_copy", 1);

    session_seed_template(session, tvals);
    template_expand("redirect", tvals, b);
    response_html(request, 200);        /* Success */
}

static void generate_error(struct session *session)
{
    struct template_vals *tvals = session->template_vals;
    struct account *account = session->account;
    struct request *request = session->request;
    char *msg = account_fetch_message(account);
    struct buffer *b = request->write_buffer;

    if (!(msg && msg[0]))
        msg = "Unable to check mail processing status";

    template_vals_string(tvals, "msg", msg);
    session_seed_template(session, tvals);
    template_expand("redirect_fail", tvals, b);
    response_html(request, 200);
}

void cmd_redirect(struct session *session)
{
    struct request *request = session->request;
    struct assoc *h = NIL;
    struct account *account = session->account;
    char *s;
    BOOL rc = T;
    ADDRESS *addr = NIL;
    BOOL redirect_enabled;
    char *redirect_address = "";
    BOOL redirect_copy;

    if (request->method != POST) {
        if (!account_mail_check(account, request->pool))
            generate_error(session);
        else
            generate_form(session,
                          account->redirect_enabled,
                          account->redirect_address,
                          account->redirect_copy);
        return;
    }

    request_decode_form(request);
    h = request->form;

    if (!assoc_lookup(h, "sub_apply")) {
        session_redirect(session, request, "manage");
        return;
    }

    /* Store state in temporary variables until validated */
    redirect_enabled = (assoc_lookup(h, "use_redirect")) ? T : NIL;
    redirect_copy    = (assoc_lookup(h, "save_copy"))    ? T : NIL;

    s = assoc_lookup(h, "redirect_addr");
    redirect_address = (s) ? s : "";

    if (redirect_enabled) {
        addr = addr_parse(request->pool, redirect_address, "");

        if (!addr) {
            session_message(session, "Redirection Address invalid: %s",
                            ml_errmsg());
            rc = NIL;
        } else {
            if ((addr->next) || (addr->personal && addr->personal[0]) ||
                (!(addr->host && addr->host[0]))) {
                session_message(session,
                                ("Redirection Address must be "
                                 "single, simple "
                                 "and fully qualified email address"));
                rc = NIL;
            }
            mail_free_address(&addr);
        }

        if (rc == NIL) {
            generate_form(session,
                          redirect_enabled,
                          redirect_address,
                          redirect_copy);
            return;
        }
    }

    /* Commit and update */
    account->redirect_enabled = redirect_enabled;
    account->redirect_copy    = redirect_copy;
    string_strdup(&account->redirect_address, redirect_address);

    if (account_mail_update(account, request->pool)) {
        session_message(session, "Updated redirection");
    } else {
        char *msg = account_fetch_message(account);
        
        session_message(session, "Unable to update redirection: %s", msg);
        session_log(session,
                    "[cmd_redirect] Unable to update redirection: %s", msg);
    }

    session_redirect(session, request, "manage");
}
