/******************************************************************************
 *
 * Jacksum version 1.7.0 - checksum utility in Java
 * Copyright (C) 2001-2006 Dipl.-Inf. (FH) Johann Nepomuk Loefflmann,
 * All Rights Reserved, http://www.jonelo.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * E-mail: jonelo@jonelo.de
 *
 *****************************************************************************/

package jonelo.jacksum.ui;

import java.io.File;
import java.util.Date;
import java.util.Properties;
import java.util.StringTokenizer;
import jonelo.jacksum.JacksumAPI;
import jonelo.jacksum.cli.Jacksum;
import jonelo.sugar.util.Version;

/**
 * This is the Meta Information used by a Jacksum generated file list.
 */
public class MetaInfo {
    private boolean _r = false;
    private boolean _p = false;
    private boolean _A = false;
    private String timestampFormat = null;
    private char filesep = File.separatorChar;
    private int grouping = 0;
    private char groupChar = ' ';
    private String algorithm = null;
    private String encoding = ""; // default
    private String separator = null;
    private String commentchars = CheckFile.COMMENTDEFAULT;
    private Properties props = null;

    /** The prefix for a MetaInfo */
    public final static String METAINFO = JacksumAPI.NAME + ": Meta-Info: ";

    /**
     * Constructs an empty MetaInfo object.
     */
    public MetaInfo() {
        props = new Properties();
    }


    /**
     * Constructs a MetaInfo object by a String.
     *
     * @param pareseString a String containing MetaInfo
     * @exception MetaInfoVersionException if MetaInfo's version is incompatible
     */
    public MetaInfo(String parseString) throws MetaInfoVersionException {
        props = new Properties();
        parse(parseString);
    }

    /**
     * Parses a metaInfo String and fills the object.
     *
     * @param thisLine a String containing MetaInfo
     * @exception MetaInfoVersionException if MetaInfo's version is incompatible
     */
    public void parse(String thisLine) throws MetaInfoVersionException {
        StringTokenizer stsub = null;
        StringTokenizer st = new StringTokenizer(thisLine.substring(METAINFO.length()),";");
        while (st.hasMoreTokens()) {
           stsub = new StringTokenizer(st.nextToken(), "=");
           props.setProperty(stsub.nextToken(), (stsub.hasMoreTokens() ? stsub.nextToken():""));
        }
        // version check
        if (new Version(props.getProperty("version")).compareTo(JacksumAPI.getVersion()) > 0) {
           throw new MetaInfoVersionException(
             "The file was generated by a newer version of Jacksum.\n"+
             "Upgrade to the latest Jacksum release, at least to version "+props.getProperty("version"));
        }
        // default field separator has changed (since 1.6.0 it is dependent on the algorithm),
        // keep compatibility with older versions
        if (new Version(props.getProperty("version")).compareTo(new Version("1.6.0")) < 0) {
            props.setProperty("fieldsep","\t");
        }


        _A = false;
        _r = false;
        _p = false;
        _r = false;
        String flags = props.getProperty("flags","");
        for (int i = 0; i < flags.length(); i++) {
           if (flags.charAt(i)=='A') {
              _A = true;
           } else
           if (flags.charAt(i)=='x') { // compatibility with Jacksum < 1.6.0
               encoding = "hex";
           } else
           if (flags.charAt(i)=='X') { // compatibility with Jacksum < 1.6.0
               encoding = "hexup";
           } else
           if (flags.charAt(i)=='r') {
               _r = true;
           } else
           if (flags.charAt(i)=='p') {
               _p = true;
           }
        }

        // since Jacksum 1.6.0
        if ((props.getProperty("encoding")==null || props.getProperty("encoding").length()==0)) {
            encoding = null;
        } else {
            encoding = props.getProperty("encoding");
        }

        // algorithm
        algorithm = props.getProperty("algorithm","sha1");
        // compatibility with versions < 1.6.0
        if (
           algorithm.equals("whirlpool") &&
           (new Version(props.getProperty("version")).compareTo(new Version("1.6.0")) < 0)
        ) {
            algorithm = "whirlpool1";
        }

        if (props.getProperty("fieldsep")==null || props.getProperty("fieldsep").length()==0) {
            separator = null;
        } else {
            separator = props.getProperty("fieldsep");
        }


        if (props.getProperty("group")==null || props.getProperty("group").length()==0) {
            grouping = 0;
        } else { // yes, a group number has been set
            try {
               grouping = Integer.parseInt(props.getProperty("group"));
            } catch (NumberFormatException nfe) {
               System.err.println("Jacksum: Warning: Ignoring the group value, because the value is invalid.");
               grouping = 0;
            }
        }

        if (props.getProperty("groupChar")==null || props.getProperty("groupChar").length()==0) {
            groupChar = ' '; // default
        } else { // yes, a group char has been set
            groupChar = props.getProperty("groupChar").charAt(0);
        }


        if (props.getProperty("comment")==null || props.getProperty("comment").length()==0)
               setCommentchars(CheckFile.COMMENTDEFAULT);
        else {
               setCommentchars(props.getProperty("comment"));
        }

        if (props.getProperty("filesep")==null || props.getProperty("filesep").length()==0)
            filesep = File.separatorChar;
        else
            filesep = props.getProperty("filesep").charAt(0);


        // is there a timeformat?
        if (props.getProperty("tformat")==null || props.getProperty("tformat").length()==0)
               setTimestampFormat(null);
        else { // yes, a timeformat has been set

            if (props.getProperty("tformat").equals("null")) { // bugfix for sourceforge-bug #948070 (problem with files generated with Jacksum 1.4.0)
                System.err.println("Jacksum: Can't determine timeformat (tformat=null), trying default (tformat="+Jacksum.TIMESTAMPFORMAT_DEFAULT+")");
                setTimestampFormat(Jacksum.TIMESTAMPFORMAT_DEFAULT);
            } else {
                setTimestampFormat(props.getProperty("tformat"));
            }
        }
    }

    /**
     * Sets the algorithm.
     *
     * @param algorithm the code for the algorithm
     */
    public void setAlgorithm(String algorithm) {
        this.algorithm = algorithm;
    }

    /**
     * Gets the algorithm.
     *
     * @return the code for the algorithm
     */
    public String getAlgorithm() {
        return algorithm;
    }

    /**
     * Sets the version of the meta info.
     *
     * @param version of the meta info
     */
    public void setVersion(String version) {
        props.setProperty("version",version);
    }

    /**
     * Gets the version of the meta info.
     *
     * @param version of the meta info
     */
    public String getVersion() {
        return props.getProperty("version");
    }

    /**
     * Sets a flag if there is recursive processing.
     *
     * @param _r if recursive processing has been specified
     */
    public void setRecursive(boolean _r) {
        this._r = _r;
    }

    /**
     * Is recursive specified?
     *
     * @return is recursive specified?
     */
    public boolean isRecursive() {
        return _r;
    }

    /**
     * Sets the encoding of the checksum
     *
     * @param encoding of the checksum
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * Gets the encoding of the checksum
     *
     * @return the encoding of the checksum
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * Is encoding wanted?
     *
     * @return is encoding wanted?
     */
    public boolean isEncoding() {
        return encoding != null;
    }

    public void setPathInfo(boolean _p) {
        this._p = _p;
    }

    public boolean isPathInfo() {
        return _p;
    }

    public void setAlternate(boolean _A) {
        this._A = _A;
    }

    public boolean isAlternate() {
        return _A;
    }

    public void setTimestampFormat(String timestampFormat) {
       this.timestampFormat = timestampFormat;
    }

    public String getTimestampFormat() {
        return timestampFormat;
    }

    public boolean isTimestampFormat() {
        return timestampFormat != null;
    }

    public void setGrouping(int grouping) {
        this.grouping = grouping;
    }

    public int getGrouping() {
        return grouping;
    }

    public boolean isGrouping() {
        return grouping > 0;
    }

    public void setGroupChar(char groupChar) {
        this.groupChar = groupChar;
    }

    public char getGroupChar() {
        return groupChar;
    }

    public void setSeparator(String separator) {
        this.separator = separator;
    }

    public String getSeparator() {
        return separator;
    }

    public boolean isSeparatorWanted() {
        return separator != null;
    }

    public void setCommentchars(String commentchars) {
        this.commentchars = commentchars;
    }

    public String getCommentchars() {
        return commentchars;
    }

    public void setFilesep(char filesep) {
        this.filesep = filesep;
    }

    public char getFilesep() {
        return filesep;
    }

    /**
     * Gets a comment (Homepage, OS, Architecture, etc.).
     *
     * @returns a multiple line comment
     */
    public String getComment() {
        // avoid the string concatenation operator +
        StringBuffer sb = new StringBuffer(240);

        sb.append(commentchars);
        sb.append(" created with ");
        sb.append(JacksumAPI.NAME);
        sb.append(" ");
        sb.append(JacksumAPI.getVersionString());
        sb.append(", http://jacksum.sourceforge.net");
        sb.append(System.getProperty("line.separator"));

        sb.append(commentchars);
        sb.append(" created on ");
        sb.append(new Date());
        sb.append(System.getProperty("line.separator"));

        sb.append(commentchars);
        sb.append(" os name=");
        sb.append(System.getProperty("os.name"));
        sb.append(";os version=");
        sb.append(System.getProperty("os.version"));
        sb.append(";os arch=");
        sb.append(System.getProperty("os.arch"));
        sb.append(System.getProperty("line.separator"));

        sb.append(commentchars);
        sb.append(" jvm vendor=");
        sb.append(System.getProperty("java.vm.vendor"));
        sb.append(";jvm version=");
        sb.append(System.getProperty("java.vm.version"));
        sb.append(System.getProperty("line.separator"));

        sb.append(commentchars);
        sb.append(" user dir=");
        sb.append(System.getProperty("user.dir"));

        return sb.toString();
    }

    /**
     * A String representation of the Meta Information.
     *
     * @return a String representing the Meta Information
     */
    public String toString() {

        StringBuffer buf = new StringBuffer(100);
        buf.append(METAINFO);
        buf.append("version=");
        buf.append(getVersion());
        buf.append(";");

        buf.append("algorithm=");
        buf.append(algorithm);
        buf.append(";");

        buf.append("filesep=");
        buf.append(filesep);
        buf.append(";");


        StringBuffer flags = new StringBuffer(5);
        if (_r) flags.append('r');
        if (_p) flags.append('p');
        if (_A) flags.append('A');
        if (flags.toString().length()>0) {
          buf.append("flags=");
          buf.append(flags.toString());
          buf.append(";");
        }

        if (timestampFormat != null && timestampFormat.length() != 0) {
          buf.append("tformat=");
          buf.append(timestampFormat);
          buf.append(";");
        }

        if (grouping > 0) {
          buf.append("group=");
          buf.append(Integer.toString(grouping));
          buf.append(";");
          if (groupChar != ' ') {
            buf.append("groupChar=");
            buf.append(groupChar);
            buf.append(";");
          }
        }

        if (encoding != null && encoding.length() > 0) {
            buf.append("encoding=");
            buf.append(encoding);
            buf.append(";");
        }

        if (separator != null && !separator.equals("\t")) {
            buf.append("fieldsep="); // field spearator
            buf.append(separator);
            buf.append(";");
        }

        if (commentchars.length() > 0 && !commentchars.equals(CheckFile.COMMENTDEFAULT)) {
            buf.append("comment=");
            buf.append(commentchars);
            buf.append(";");
        }

        return buf.toString();
    }


}
