#include "toolspage.h"
#include "chrootbindguard.h"
#include "chrootprocess.h"
#include "pluginnavgation.h"
#include <DListView>
#include <DStandardItem>
#include <DApplication>

#include <QTimer>
#include <QHBoxLayout>
#include <QPushButton>
#include <QDebug>
#include <QApplication>
#include <QTranslator>
#include <QProcess>
#include <QThread>
DWIDGET_USE_NAMESPACE

DCORE_USE_NAMESPACE
const QString chroot_hook_script = "/usr/lib/deepin-repair-tools/chroot_hook.sh";

const RunResult execAsChrootSynchronous(const QString &root, const QString &script, const QStringList &args)
{
    QProcess &process = *execAsChrootAsynchronous(root, script, args);
    process.start();
    process.waitForFinished(-1);

    const RunResult r { process.exitCode(), process.readAllStandardOutput(), process.readAllStandardError() };

    qInfo().noquote() << Q_FUNC_INFO
                      << "exitCode =" << r.exitCode
                      << "output =" << r.standardOutput
                      << "error =" << r.standardError;

    process.deleteLater();

    return r;
}

QProcess* execAsChrootAsynchronous(const QString &root, const QString &script, const QStringList &args)
{
    qInfo().noquote() << Q_FUNC_INFO
                      << root
                      << script
                      << args;

    ChrootProcess *process = new ChrootProcess(root);
    process->setProgram("/bin/sh");
    process->setArguments(QStringList() << chroot_hook_script << root << script << args);

    return process;
}

ToolsPage::ToolsPage(QWidget *parent)
    : QWidget(parent)
    , m_navButtonsGroup(new QButtonGroup)
    , m_navigationLayout(new QVBoxLayout)
    , m_pluginsLayout(new QStackedLayout)
{
    m_navButtonsGroup->setExclusive(true);
    m_navigationLayout->setMargin(0);
    m_navigationLayout->setSpacing(10);

    QWidget *navWidget = new QFrame;
    navWidget->setObjectName("NavWidget");
    navWidget->setLayout(m_navigationLayout);

    QWidget *pluginWidget = new QWidget;
    pluginWidget->setObjectName("PluginWidget");
    pluginWidget->setLayout(m_pluginsLayout);

    QHBoxLayout *centralLayout = new QHBoxLayout;
    centralLayout->addWidget(navWidget);
    centralLayout->addWidget(pluginWidget);
    centralLayout->setSpacing(0);
    centralLayout->setContentsMargins(0, 0, 0, 40);

    setLayout(centralLayout);
    setStyleSheet("#NavWidget {"
                  "border: solid #eee;"
                  "border-right-width: 1px;"
                  "}"
                  ""
                  "#NavWidget > QPushButton {"
                  "margin: 0;"
                  "text-align: left;"
                  "padding: 5px 15px;"
                  "border: none;"
                  "}"
                  ""
                  "#NavWidget > QPushButton:checked {"
                  "color: #2ca7f8;"
                  "font-weight: 500;"
                  "background-color: #d5edfe;"
                  "border: 1px solid rgba(44, 167, 248, .1);"
                  "border-left: none;"
                  "border-right: none;"
                  "}"
                  ""
                  "#PluginWidget QComboBox {"
                  "width: 180px;"
                  "}"
                  ""
                  "#PluginWidget QPushButton {"
                  "width: 152px;"
                  "height: 26px;"
                  "background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #FFFFFF, stop:1 #FBFBFB);"
                  "border-color: rgba(0, 131, 255, 0.4);"
                  "border-radius: 4;"
                  "border-style: solid;"
                  "border-width: 1;"
                  "outline: none;"
                  "padding: 3px 24px 4px 24px;"
                  "color: #0699ff;"
                  "font-size: 14px;"
                  "}"
                  ""
                  "#PluginWidget QPushButton:hover {"
                  "background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #8ccfff, stop:1 #4bb8ff);"
                  "color: #FFFFFF;"
                  "border-color: rgba(0, 117, 243, 0.2);"
                  "}"
                  ""
                  "#PluginWidget QPushButton:pressed {"
                  "background: qlineargradient(x1:0, y1:0, x2:0, y2:1, stop:0 #0b8cff, stop:1 #0aa1ff);"
                  "color: #FFFFFF;"
                  "border-color: rgba(29, 129, 255, 0.3);"
                  "}"
                  ""
                  "#PluginWidget QPushButton:disabled {"
                  "border-color: rgba(0, 131, 255, 0.2);"
                  "color: rgba(6, 153, 255, .4);"
                  "}"
                  ""
                  "#PluginWidget QLineEdit {"
                  "width: 210px;"
                  "height: 25px;"
                  "}"
                  ""
                  "#PluginWidget #TipsLabel {"
                  "font-size: 9pt;"
                  "font-weight: 500;"
                  "}"
                  ""
                  "#PluginWidget Dtk--Widget--DLinkButton {"
                  "font-size: 12px;"
                  "}"
                  ""
                  "#PluginWidget QLabel#PluginIcon {"
                  "margin: 70px 0 0 0;"
                  "}"
                  "");

    m_languageView = new QComboBox(this);
    m_languageView->move(560,15);
    m_languageView->setFixedWidth(150 - 15);
    m_languageView->setFocusPolicy(Qt::TabFocus);
    m_languageView->setIconSize(QSize(32, 32));
    m_languageView->setContextMenuPolicy(Qt::NoContextMenu);
    m_languageModel=new QStandardItemModel(m_languageView);
    m_languageView->setModel(m_languageModel);
    m_languageView->show();

    appendLanguageitem();

    connect(m_languageView, &QComboBox::currentTextChanged, this, &ToolsPage::languageCurrentTextChanged);
    m_languageView->setCurrentIndex(1);
}

ToolsPage::~ToolsPage()
{

}

void ToolsPage::setModel(const QPointer<ToolsModel> &model)
{
    m_model = model;

    connect(model, &ToolsModel::pluginsLoadFinished, this, &ToolsPage::refreshNavBar);

    m_model->initPlugins(this);
}

void ToolsPage::refreshNavBar()
{
    // TODO:
    // clear old widgets

    m_navigationLayout->addStretch();
    for (int i(0); i != m_model->rowCount(QModelIndex()); ++i)
    {
        QPushButton *b = new PluginNavgation;
        b->setCheckable(true);
        b->setText(m_model->data(m_model->index(i), ToolsModel::ToolsName).toString());
        m_navigationLayout->addWidget(b);
        m_navButtonsGroup->addButton(b);

        connect(b, &QPushButton::clicked, this, [=] { onNavButtonClicked(i); });
    }
    m_navigationLayout->addStretch();

    // FIXME: select first page for default
    m_navButtonsGroup->buttons().first()->click();
}

void ToolsPage::setNavgationEnabled(const bool enabled)
{
    for (auto *button : m_navButtonsGroup->buttons())
        button->setEnabled(enabled);
}

void ToolsPage::onNavButtonClicked(const int index)
{
    QWidget *w = m_model->data(m_model->index(index), ToolsModel::ToolsWidget).value<QWidget *>();
    Q_ASSERT(w);

    m_pluginsLayout->insertWidget(0, w);
    m_pluginsLayout->setCurrentWidget(w);
}

void ToolsPage::languageCurrentTextChanged(const QString &)
{
    static QTranslator * translator = new QTranslator;
    qApp->removeTranslator(translator);

    QString translatorPath = "/usr/share/deepin-repair-tools/translations/deepin-repair-tools_";
    translatorPath += m_lang_list.at(m_languageView->currentIndex()).locale;
    translatorPath += ".qm";

    if (!translator->load(translatorPath))
        return ;

    if (!qApp->installTranslator(translator))
        return ;
}

void ToolsPage::changeEvent(QEvent* event)
{
    if (event->type() == QEvent::LanguageChange) {
        QList<QAbstractButton*> buttonList = m_navButtonsGroup->buttons();
        QStringList tetxList;
        for (int i(0); i != m_model->rowCount(QModelIndex()); ++i)
        {
           QString text = m_model->data(m_model->index(i), ToolsModel::ToolsName).toString();
           buttonList.at(i)->setText(::QObject::tr(text.toStdString().c_str()));

        }
    } else {
        QWidget::changeEvent(event);
    }
}

void ToolsPage::appendLanguageitem()
{
    m_lang_list = GetLanguageList();

    if (!m_languageSortList.isEmpty()) {
        qSort(m_lang_list.begin(), m_lang_list.end(), [&] (LanguageItem& left, LanguageItem& right) {
            return m_languageSortList.indexOf(left.name) < m_languageSortList.indexOf(right.name);
        });
    }
    else {
        qCritical() << "language sort list is empty";
    }

    m_languageModel->clear();

    for(auto it = m_lang_list.cbegin(); it!=m_lang_list.cend(); ++it) {
        if (m_languageSortList.indexOf((*it).name) < 0) {
            qInfo() << (*it).name << " is not exist in language sort file";
        }

        DStandardItem *item = new DStandardItem((*it).local_name);
        m_languageModel->appendRow(item);
    }
}
