# font_row.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

from typing import TYPE_CHECKING, Any, cast

from gi.repository import GLib, GObject, Gio, Gtk, Pango

from refine.logger import logger
from refine.widgets.action_row import ActionRow
from refine.widgets.base import BaseInterface

if TYPE_CHECKING:
    from gi.repository import Adw


@Gtk.Template.from_resource("/page/tesk/Refine/widgets/font-row.ui")
class FontRow(ActionRow):
    """
    An `ActionRow` for displaying and interacting with font.

    It is also used for launching the `Gtk.FontDialog`.
    """

    __gtype_name__ = "FontRow"

    revert_button = Gtk.Template.Child()
    revealer = Gtk.Template.Child()

    is_default = GObject.Property(type=bool, default=True)

    def _row_activated(self, *_args: Any) -> None:
        def choose_font_and_features_callback(
            _dialog: Gtk.FontDialog, result: Gio.AsyncResult
        ) -> None:
            try:
                font = font_dialog.choose_font_and_features_finish(result)
                font_string = font[1].to_string()
            except (IndexError, GLib.Error):
                return

            self.settings.set_string(self.key, font_string)
            self.value_set(self.settings.get_string(self.key))
            self.do_update()

        font_string = self.settings.get_string(self.key)
        font_description = Pango.FontDescription.from_string(font_string)
        font_dialog = Gtk.FontDialog()

        if window := cast("Adw.Window | None", self.get_root()):
            font_dialog.choose_font_and_features(
                window, font_description, callback=choose_font_and_features_callback
            )

    def _revert_button_clicked(self, *_args: Any) -> None:
        logger.debug("Revert button pressed")
        self.settings.reset(self.key)
        self.do_update()

    def do_update(self) -> None:
        font_string = self.settings.get_string(self.key)
        default_value = cast("GLib.Variant", self.settings.get_default_value(self.key))
        default_font_string = default_value.get_string()

        self.is_default = default_font_string != font_string

        font_description = Pango.FontDescription.from_string(font_string)
        font_family = font_description.get_family()

        self.set_subtitle(f"<span face='{font_family}'>{font_string}</span>")

    def do_setup(self) -> None:
        self.connect("activated", self._row_activated)
        self.bind_property(
            "is-default",
            self.revert_button,
            "visible",
            GObject.BindingFlags.SYNC_CREATE,
        )
        self.revert_button.connect("clicked", self._revert_button_clicked)
        BaseInterface.set_up_separator_revealer(self.revealer, self.revert_button)
