/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isString = require( '@stdlib/assert/is-string' ).isPrimitive;
var isObject = require( '@stdlib/assert/is-plain-object' );
var isFunction = require( '@stdlib/assert/is-function' );
var readFile = require( './../../read-file' );
var removeBOM = require( '@stdlib/string/remove-utf8-bom' );
var parseJSON = require( '@stdlib/utils/parse-json' );
var instanceOf = require( '@stdlib/assert/instance-of' );


// MAIN //

/**
* Asynchronously reads a file as JSON.
*
* @param {(string|Buffer|integer)} file - file path or file descriptor
* @param {(Options|string)} [options] - options
* @param {(string|null)} [options.encoding] - file encoding
* @param {string} [options.flag] - file status flag
* @param {Function} [options.reviver] - JSON reviver
* @param {Callback} clbk - callback
* @throws {TypeError} options argument must be either a string or an object
* @throws {TypeError} callback argument must be a function
*
* @example
* var resolve = require( 'path' ).resolve;
*
* readJSON( resolve( __dirname, '..', 'package.json' ), onJSON );
*
* function onJSON( error, data ) {
*     if ( error ) {
*         throw error;
*     }
*     console.dir( data );
* }
*/
function readJSON( file, options, clbk ) {
	var opts;
	var done;
	if ( arguments.length < 3 ) {
		opts = {};
		done = options;
	} else {
		if ( isString( options ) ) {
			opts = {
				'encoding': options
			};
		} else {
			if ( !isObject( options ) ) {
				throw new TypeError( 'invalid argument. Options argument must be either a string or an object. Value: `' + options + '`.' );
			}
			opts = options;
		}
		done = clbk;
	}
	if ( !isFunction( done ) ) {
		throw new TypeError( 'invalid argument. Callback argument must be a function. Value: `' + done + '`.' );
	}
	readFile( file, opts, onRead );

	/**
	* Callback invoked upon reading a file.
	*
	* @private
	* @param {(Error|null)} error - error object
	* @param {(Buffer|string)} file - file contents
	* @returns {void}
	*/
	function onRead( error, file ) {
		if ( error ) {
			return done( error );
		}
		file = file.toString();
		if ( opts.encoding === 'utf8' ) {
			file = removeBOM( file );
		}
		if ( opts.reviver ) {
			file = parseJSON( file, opts.reviver );
		} else {
			file = parseJSON( file );
		}
		if ( instanceOf( file, Error ) ) {
			return done( file );
		}
		done( null, file );
	}
}


// EXPORTS //

module.exports = readJSON;
