/* SPDX-License-Identifier: GPL-2.0-only */

#include <bootstate.h>
#include <arch/cpuid.h>
#include <cpu/x86/msr.h>
#include <soc/gpio.h>
#include <soc/ramstage.h>
#include "include/gpio.h"
#include "sch5555_ec.h"

void mainboard_silicon_init_params(FSP_SIL_UPD *params)
{
	gpio_configure_pads(gpio_table, ARRAY_SIZE(gpio_table));
}

#define FORM_FACTOR_MICRO	0
#define FORM_FACTOR_SFF		1
// Probably DT and MT
#define FORM_FACTOR_UNK2	2
#define FORM_FACTOR_UNK3	3

#define HWM_TAB_ADD_TEMP_TARGET	1
#define HWM_TAB_PKG_POWER_ANY	0xffff

struct hwm_tab_entry {
	uint16_t addr;
	uint8_t val;
	uint8_t flags;
	uint16_t pkg_power;
};

static const struct hwm_tab_entry HWM_TAB_MICRO_BASE[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x01b, 0x0f, 0, 0xffff },
	{ 0x057, 0xff, 0, 0xffff },
	{ 0x059, 0xff, 0, 0xffff },
	{ 0x05b, 0xff, 0, 0xffff },
	{ 0x05d, 0xff, 0, 0xffff },
	{ 0x05f, 0xff, 0, 0xffff },
	{ 0x061, 0xff, 0, 0xffff },
	{ 0x06e, 0x00, 0, 0xffff },
	{ 0x06f, 0x03, 0, 0xffff },
	{ 0x070, 0x03, 0, 0xffff },
	{ 0x071, 0x02, 0, 0xffff },
	{ 0x072, 0x02, 0, 0xffff },
	{ 0x073, 0x01, 0, 0xffff },
	{ 0x074, 0x06, 0, 0xffff },
	{ 0x075, 0x07, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x80, 0, 0xffff },
	{ 0x082, 0x80, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0xf1, 0, 0xffff },
	{ 0x086, 0x88, 0, 0xffff },
	{ 0x087, 0x61, 0, 0xffff },
	{ 0x088, 0x08, 0, 0xffff },
	{ 0x089, 0x00, 0, 0xffff },
	{ 0x08a, 0x73, 0, 0xffff },
	{ 0x08b, 0x73, 0, 0xffff },
	{ 0x08c, 0x73, 0, 0xffff },
	{ 0x090, 0x6d, 0, 0xffff },
	{ 0x091, 0x7e, 0, 0xffff },
	{ 0x092, 0x66, 0, 0xffff },
	{ 0x093, 0xa4, 0, 0xffff },
	{ 0x094, 0x7c, 0, 0xffff },
	{ 0x095, 0xa4, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x099, 0xa4, 0, 0xffff },
	{ 0x09a, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x2e, 0, 0xffff },
	{ 0x0a1, 0x00, 0, 0xffff },
	{ 0x0a2, 0x00, 0, 0xffff },
	{ 0x0ae, 0xa4, 0, 0xffff },
	{ 0x0af, 0xa4, 0, 0xffff },
	{ 0x0b0, 0xa4, 0, 0xffff },
	{ 0x0b1, 0xa4, 0, 0xffff },
	{ 0x0b2, 0xa4, 0, 0xffff },
	{ 0x0b3, 0xa4, 0, 0xffff },
	{ 0x0b6, 0x00, 0, 0xffff },
	{ 0x0b7, 0x00, 0, 0xffff },
	{ 0x0d1, 0xff, 0, 0xffff },
	{ 0x0d6, 0xff, 0, 0xffff },
	{ 0x0db, 0xff, 0, 0xffff },
	{ 0x0ea, 0x5c, 0, 0xffff },
	{ 0x0eb, 0x5c, 0, 0xffff },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x184, 0xff, 0, 0xffff },
	{ 0x186, 0xff, 0, 0xffff },
	{ 0x1a1, 0xce, 0, 0xffff },
	{ 0x1a2, 0x0c, 0, 0xffff },
	{ 0x1a3, 0x0c, 0, 0xffff },
	{ 0x1a6, 0x00, 0, 0xffff },
	{ 0x1a7, 0x00, 0, 0xffff },
	{ 0x1a8, 0xa4, 0, 0xffff },
	{ 0x1a9, 0xa4, 0, 0xffff },
	{ 0x1ab, 0x2d, 0, 0xffff },
	{ 0x1ac, 0x2d, 0, 0xffff },
	{ 0x1b1, 0x00, 0, 0xffff },
	{ 0x1bb, 0x00, 0, 0xffff },
	{ 0x1bc, 0x00, 0, 0xffff },
	{ 0x1bd, 0x00, 0, 0xffff },
	{ 0x1be, 0x01, 0, 0xffff },
	{ 0x1bf, 0x01, 0, 0xffff },
	{ 0x1c0, 0x01, 0, 0xffff },
	{ 0x1c1, 0x01, 0, 0xffff },
	{ 0x1c2, 0x01, 0, 0xffff },
	{ 0x280, 0x00, 0, 0xffff },
	{ 0x281, 0x00, 0, 0xffff },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x040, 0x01, 0, 0xffff },
};

static const struct hwm_tab_entry HWM_TAB_MICRO_TEMP80[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x01b, 0x0f, 0, 0xffff },
	{ 0x057, 0xff, 0, 0xffff },
	{ 0x059, 0xff, 0, 0xffff },
	{ 0x05b, 0xff, 0, 0xffff },
	{ 0x05d, 0xff, 0, 0xffff },
	{ 0x05f, 0xff, 0, 0xffff },
	{ 0x061, 0xff, 0, 0xffff },
	{ 0x06e, 0x00, 0, 0xffff },
	{ 0x06f, 0x03, 0, 0xffff },
	{ 0x070, 0x03, 0, 0xffff },
	{ 0x071, 0x02, 0, 0xffff },
	{ 0x072, 0x02, 0, 0xffff },
	{ 0x073, 0x01, 0, 0xffff },
	{ 0x074, 0x06, 0, 0xffff },
	{ 0x075, 0x07, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x80, 0, 0xffff },
	{ 0x082, 0x80, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0xf6, 0, 0xffff },
	{ 0x086, 0x88, 0, 0xffff },
	{ 0x087, 0x61, 0, 0xffff },
	{ 0x088, 0x08, 0, 0xffff },
	{ 0x089, 0x00, 0, 0xffff },
	{ 0x08a, 0x73, 0, 0xffff },
	{ 0x08b, 0x73, 0, 0xffff },
	{ 0x08c, 0x73, 0, 0xffff },
	{ 0x090, 0x6d, 0, 0xffff },
	{ 0x091, 0x86, 0, 0xffff },
	{ 0x092, 0x66, 0, 0xffff },
	{ 0x093, 0xa4, 0, 0xffff },
	{ 0x094, 0x7c, 0, 0xffff },
	{ 0x095, 0xa4, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x099, 0xa4, 0, 0xffff },
	{ 0x09a, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x2e, 0, 0xffff },
	{ 0x0a1, 0x00, 0, 0xffff },
	{ 0x0a2, 0x00, 0, 0xffff },
	{ 0x0ae, 0xa4, 0, 0xffff },
	{ 0x0af, 0xa4, 0, 0xffff },
	{ 0x0b0, 0xa4, 0, 0xffff },
	{ 0x0b1, 0xa4, 0, 0xffff },
	{ 0x0b2, 0xa4, 0, 0xffff },
	{ 0x0b3, 0xa4, 0, 0xffff },
	{ 0x0b6, 0x00, 0, 0xffff },
	{ 0x0b7, 0x00, 0, 0xffff },
	{ 0x0d1, 0xff, 0, 0xffff },
	{ 0x0d6, 0xff, 0, 0xffff },
	{ 0x0db, 0xff, 0, 0xffff },
	{ 0x0ea, 0x50, 0, 0xffff },
	{ 0x0eb, 0x50, 0, 0xffff },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x184, 0xff, 0, 0xffff },
	{ 0x186, 0xff, 0, 0xffff },
	{ 0x1a1, 0xce, 0, 0xffff },
	{ 0x1a2, 0x0c, 0, 0xffff },
	{ 0x1a3, 0x0c, 0, 0xffff },
	{ 0x1a6, 0x00, 0, 0xffff },
	{ 0x1a7, 0x00, 0, 0xffff },
	{ 0x1a8, 0xa4, 0, 0xffff },
	{ 0x1a9, 0xa4, 0, 0xffff },
	{ 0x1ab, 0x2d, 0, 0xffff },
	{ 0x1ac, 0x2d, 0, 0xffff },
	{ 0x1b1, 0x00, 0, 0xffff },
	{ 0x1bb, 0x00, 0, 0xffff },
	{ 0x1bc, 0x00, 0, 0xffff },
	{ 0x1bd, 0x00, 0, 0xffff },
	{ 0x1be, 0x01, 0, 0xffff },
	{ 0x1bf, 0x01, 0, 0xffff },
	{ 0x1c0, 0x01, 0, 0xffff },
	{ 0x1c1, 0x01, 0, 0xffff },
	{ 0x1c2, 0x01, 0, 0xffff },
	{ 0x280, 0x00, 0, 0xffff },
	{ 0x281, 0x00, 0, 0xffff },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x040, 0x01, 0, 0xffff },
};

static const struct hwm_tab_entry HWM_TAB_MICRO_EARLY_STEPPING[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x01b, 0x0f, 0, 0xffff },
	{ 0x057, 0xff, 0, 0xffff },
	{ 0x059, 0xff, 0, 0xffff },
	{ 0x05b, 0xff, 0, 0xffff },
	{ 0x05d, 0xff, 0, 0xffff },
	{ 0x05f, 0xff, 0, 0xffff },
	{ 0x061, 0xff, 0, 0xffff },
	{ 0x06e, 0x01, 0, 0xffff },
	{ 0x06f, 0x03, 0, 0xffff },
	{ 0x070, 0x03, 0, 0xffff },
	{ 0x071, 0x02, 0, 0xffff },
	{ 0x072, 0x02, 0, 0xffff },
	{ 0x073, 0x01, 0, 0xffff },
	{ 0x074, 0x06, 0, 0xffff },
	{ 0x075, 0x07, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x80, 0, 0xffff },
	{ 0x082, 0x80, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0xfd, 0, 0xffff },
	{ 0x086, 0x60, 0, 0xffff },
	{ 0x087, 0x50, 0, 0xffff },
	{ 0x088, 0x08, 0, 0xffff },
	{ 0x089, 0x00, 0, 0xffff },
	{ 0x08a, 0x73, 0, 0xffff },
	{ 0x08b, 0x73, 0, 0xffff },
	{ 0x08c, 0x73, 0, 0xffff },
	{ 0x090, 0x6d, 0, 0xffff },
	{ 0x091, 0x7a, 0, 0xffff },
	{ 0x092, 0x6b, 0, 0xffff },
	{ 0x093, 0xa4, 0, 0xffff },
	{ 0x094, 0x78, 0, 0xffff },
	{ 0x095, 0xa4, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x099, 0xa4, 0, 0xffff },
	{ 0x09a, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x2e, 0, 0xffff },
	{ 0x0a1, 0x00, 0, 0xffff },
	{ 0x0a2, 0x00, 0, 0xffff },
	{ 0x0ae, 0xa4, 0, 0xffff },
	{ 0x0af, 0xa4, 0, 0xffff },
	{ 0x0b0, 0xa4, 0, 0xffff },
	{ 0x0b1, 0xa4, 0, 0xffff },
	{ 0x0b2, 0xa4, 0, 0xffff },
	{ 0x0b3, 0xa4, 0, 0xffff },
	{ 0x0b6, 0x00, 0, 0xffff },
	{ 0x0b7, 0x00, 0, 0xffff },
	{ 0x0d1, 0xff, 0, 0xffff },
	{ 0x0d6, 0xff, 0, 0xffff },
	{ 0x0db, 0xff, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0xffff },
	{ 0x0eb, 0x64, 0, 0xffff },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x184, 0xff, 0, 0xffff },
	{ 0x186, 0xff, 0, 0xffff },
	{ 0x1a1, 0xce, 0, 0xffff },
	{ 0x1a2, 0x0c, 0, 0xffff },
	{ 0x1a3, 0x0c, 0, 0xffff },
	{ 0x1a6, 0x00, 0, 0xffff },
	{ 0x1a7, 0x00, 0, 0xffff },
	{ 0x1a8, 0xa4, 0, 0xffff },
	{ 0x1a9, 0xa4, 0, 0xffff },
	{ 0x1ab, 0x2d, 0, 0xffff },
	{ 0x1ac, 0x2d, 0, 0xffff },
	{ 0x1b1, 0x00, 0, 0xffff },
	{ 0x1bb, 0x00, 0, 0xffff },
	{ 0x1bc, 0x00, 0, 0xffff },
	{ 0x1bd, 0x00, 0, 0xffff },
	{ 0x1be, 0x01, 0, 0xffff },
	{ 0x1bf, 0x01, 0, 0xffff },
	{ 0x1c0, 0x01, 0, 0xffff },
	{ 0x1c1, 0x01, 0, 0xffff },
	{ 0x1c2, 0x01, 0, 0xffff },
	{ 0x280, 0x00, 0, 0xffff },
	{ 0x281, 0x00, 0, 0xffff },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff },
	{ 0x040, 0x01, 0, 0xffff },
};

static const struct hwm_tab_entry HWM_TAB_SFF[] = {
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x040, 0x01, 0, 0xffff },
	{ 0x072, 0x03, 0, 0xffff },
	{ 0x075, 0x06, 0, 0xffff },
	{ 0x07c, 0x00, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0x59, 0, 0xffff },
	{ 0x086, 0x6a, 0, 0xffff },
	{ 0x087, 0xc0, 0, 0xffff },
	{ 0x08a, 0x33, 0, 0xffff },
	{ 0x090, 0x77, 0, 0xffff },
	{ 0x091, 0x66, 0, 0xffff },
	{ 0x092, 0x94, 0, 0xffff },
	{ 0x093, 0x90, 0, 0xffff },
	{ 0x094, 0x68, 0, 0xffff },
	{ 0x096, 0xa4, 0, 0xffff },
	{ 0x097, 0xa4, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x099, 0xa4, 0, 0xffff },
	{ 0x09a, 0xa4, 0, 0xffff },
	{ 0x09b, 0xa4, 0, 0xffff },
	{ 0x0a0, 0x3e, 0, 0xffff },
	{ 0x0ae, 0x86, 0, 0xffff },
	{ 0x0af, 0x86, 0, 0xffff },
	{ 0x0b0, 0xa4, 0, 0xffff },
	{ 0x0b1, 0xa4, 0, 0xffff },
	{ 0x0b2, 0x90, 0, 0xffff },
	{ 0x0b6, 0x48, 0, 0xffff },
	{ 0x0b7, 0x48, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x1b1, 0x48, 0, 0xffff },
	{ 0x1b8, 0x00, 0, 0xffff },
	{ 0x1be, 0x95, 0, 0xffff },
	{ 0x1c1, 0x90, 0, 0xffff },
	{ 0x1c6, 0x00, 0, 0xffff },
	{ 0x1c9, 0x00, 0, 0xffff },
	{ 0x280, 0x68, 0, 0xffff },
	{ 0x281, 0x10, 0, 0xffff },
	{ 0x282, 0x03, 0, 0xffff },
	{ 0x283, 0x0a, 0, 0xffff },
	{ 0x284, 0x80, 0, 0xffff },
	{ 0x285, 0x03, 0, 0xffff}
};

static const struct hwm_tab_entry HWM_TAB_MT[] = {
	{ 0x005, 0x33, 0, 0xffff },
	{ 0x018, 0x2f, 0, 0xffff },
	{ 0x019, 0x2f, 0, 0xffff },
	{ 0x01a, 0x2f, 0, 0xffff },
	{ 0x080, 0x00, 0, 0xffff },
	{ 0x081, 0x00, 0, 0xffff },
	{ 0x082, 0x80, 0, 0xffff },
	{ 0x083, 0xbb, 0, 0xffff },
	{ 0x085, 0xb9, 0, 0x0010 },
	{ 0x086, 0xac, 0, 0x0010 },
	{ 0x087, 0x87, 0, 0x0010 },
	{ 0x08a, 0x51, 0, 0x0010 },
	{ 0x08b, 0x39, 0, 0x0010 },
	{ 0x090, 0x78, 0, 0xffff },
	{ 0x091, 0x6a, 0, 0xffff },
	{ 0x092, 0x8f, 0, 0xffff },
	{ 0x094, 0x68, 0, 0xffff },
	{ 0x095, 0x5b, 0, 0xffff },
	{ 0x096, 0x92, 0, 0xffff },
	{ 0x097, 0x86, 0, 0xffff },
	{ 0x098, 0xa4, 0, 0xffff },
	{ 0x09a, 0x8b, 0, 0xffff },
	{ 0x0a0, 0x0a, 0, 0xffff },
	{ 0x0a1, 0x26, 0, 0xffff },
	{ 0x0a2, 0xd1, 0, 0xffff },
	{ 0x0ae, 0x7c, 0, 0xffff },
	{ 0x0af, 0x7c, 0, 0xffff },
	{ 0x0b0, 0x9a, 0, 0xffff },
	{ 0x0b3, 0x7c, 0, 0xffff },
	{ 0x0b6, 0x08, 0, 0xffff },
	{ 0x0b7, 0x00, 0, 0xffff },
	{ 0x0ea, 0x64, 0, 0xffff },
	{ 0x0ef, 0xff, 0, 0xffff },
	{ 0x0f8, 0x15, 0, 0xffff },
	{ 0x0f9, 0x00, 0, 0xffff },
	{ 0x0f0, 0x30, 0, 0xffff },
	{ 0x0fd, 0x01, 0, 0xffff },
	{ 0x1a1, 0x99, 0, 0xffff },
	{ 0x1a2, 0x00, 0, 0xffff },
	{ 0x1a4, 0x00, 0, 0xffff },
	{ 0x1b1, 0x00, 0, 0xffff },
	{ 0x1be, 0x90, 0, 0xffff },
	{ 0x280, 0xc4, 0, 0xffff },
	{ 0x281, 0x09, 0, 0xffff },
	{ 0x282, 0x0a, 0, 0xffff },
	{ 0x283, 0x14, 0, 0xffff },
	{ 0x284, 0x01, 0, 0xffff },
	{ 0x285, 0x01, 0, 0xffff },
	{ 0x288, 0x94, 0, 0xffff },
	{ 0x289, 0x11, 0, 0xffff },
	{ 0x28a, 0x0a, 0, 0xffff },
	{ 0x28b, 0x14, 0, 0xffff },
	{ 0x28c, 0x01, 0, 0xffff },
	{ 0x28d, 0x01, 0, 0xffff },
	{ 0x294, 0x24, 0, 0xffff },
};

static uint8_t get_temp_target(void)
{
	uint8_t val = rdmsr(0x1a2).lo >> 8 & 0xff;
	if (!val)
		val = 20;
	return 0x95 - val;
}

static uint16_t get_pkg_power(void)
{
	const unsigned int pkg_power = rdmsr(0x614).lo & 0x7fff;
	const unsigned int power_unit = 1 << (rdmsr(0x606).lo & 0xf);
	if (pkg_power / power_unit > 65)
		return 32;
	else
		return 16;
}

static uint8_t get_core_cnt(void)
{
	// Intel describes this CPUID field as:
	// > Maximum number of addressable IDs for processor cores in the physical package
	if (cpuid(0).eax >= 4)
		return cpuid_ext(4, 0).eax >> 26;
	return 0;
}

static void apply_hwm_tab(const struct hwm_tab_entry *arr, size_t size)
{
	uint8_t temp_target = get_temp_target();
	uint16_t pkg_power = get_pkg_power();

	printk(BIOS_DEBUG, "Temp target = %#x\n", temp_target);
	printk(BIOS_DEBUG, "Package power = %#x\n", pkg_power);

	for (size_t i = 0; i < size; ++i) {
		// Skip entry if it doesn't apply for this package power
		if (arr[i].pkg_power != pkg_power &&
		    arr[i].pkg_power != HWM_TAB_PKG_POWER_ANY)
			continue;

		uint8_t val = arr[i].val;

		// Add temp target to value if requested (current tables never do)
		if (arr[i].flags & HWM_TAB_ADD_TEMP_TARGET)
			val += temp_target;

		// Perform write
		sch5555_mbox_write(1, arr[i].addr, val);
	}
}

static void sch5555_ec_hwm_init(void *arg)
{
	uint8_t form_fac_id, saved_2fc, core_cnt;

	printk(BIOS_DEBUG, "OptiPlex 3050 late HWM init\n");

	form_fac_id = gpio_get(GPP_G2) | gpio_get(GPP_G3) << 1;
	printk(BIOS_DEBUG, "Form Factor ID = %#x\n", form_fac_id);

	saved_2fc = sch5555_mbox_read(1, 0x2fc);
	sch5555_mbox_write(1, 0x2fc, 0xa0);
	sch5555_mbox_write(1, 0x2fd, 0x32);

	switch (form_fac_id) {
	case FORM_FACTOR_MICRO:
		// CPU stepping <= 3
		if ((cpuid(1).eax & 0xf) <= 3)
			apply_hwm_tab(HWM_TAB_MICRO_EARLY_STEPPING, ARRAY_SIZE(HWM_TAB_MICRO_EARLY_STEPPING));
		// Tjunction == 80
		else if ((rdmsr(0x1a2).lo >> 16 & 0xff) == 80)
			apply_hwm_tab(HWM_TAB_MICRO_TEMP80, ARRAY_SIZE(HWM_TAB_MICRO_TEMP80));
		else
			apply_hwm_tab(HWM_TAB_MICRO_BASE, ARRAY_SIZE(HWM_TAB_MICRO_BASE));
		break;
	case FORM_FACTOR_SFF:
		apply_hwm_tab(HWM_TAB_SFF, ARRAY_SIZE(HWM_TAB_SFF));
		break;
	default:
		apply_hwm_tab(HWM_TAB_MT, ARRAY_SIZE(HWM_TAB_MT));
		break;
	}

	core_cnt = get_core_cnt();
	printk(BIOS_DEBUG, "CPU Core Count = %#x\n", core_cnt);
	if (core_cnt > 2) {
		sch5555_mbox_write(1, 0x9e, 0x30);
		sch5555_mbox_write(1, 0xeb, sch5555_mbox_read(1, 0xea));
	}

	sch5555_mbox_write(1, 0x2fc, saved_2fc);
	sch5555_mbox_read(1, 0xb8);
}

BOOT_STATE_INIT_ENTRY(BS_POST_DEVICE, BS_ON_EXIT, sch5555_ec_hwm_init, NULL);
