################################################################################
# ---------------------- SMOOTHING PARAMETER SELECTION ----------------------- #
################################################################################

#' Smoothing Parameter Selection
#' 
#' @param y       vector with binned, variance stabilized and mirrored data
#' @param q       penalization order
#' @param evals   eigenvalues of the \code{q}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' @param method  to select the smoothing parameter of the smoothing spline
#' @param f.true  N-dim. vector with the true regression function evaluated at 
#'                equi-spaced points. Only required if \code{method} is set
#'                to "\code{ML-oracle}" or "\code{GCV-oracle}".
#' @param sigma   variance of the regression data. Only required if 
#'                \code{method} is set to "\code{ML-oracle}" or 
#'                "\code{GCV-oracle}".
#' 
#' @return lambda estimated smoothing parameter.
#' 
#' @keywords internal
#' @noRd
smoothpar_selection <- function(y, q, evals, method, f.true=NULL, sigma=NULL) {
  
  Duplicate <- function(x) {
    r <- length(x)
    return(c(x[2:(r-1)],x[r:1]))
  }
  if(method=="GCV")
    lambda <- GCV.opt(y = y, q = q, evals = evals)
  if(method=="GCV-oracle"){
    if (is.null(f.true)){
      stop("True spectral density is missing.")
    } else {
      check_numeric_vector(f.true)
    }
    if (is.null(sigma)){
      stop("True variance is missing.")
    }
    f.true <- Duplicate(f.true)
    log.sdf.true <- VST(f.true, m=1)
    lambda <- GCV.opt(y = NULL, q = q,  evals = evals, oracle = TRUE, f.true = log.sdf.true, sigma = sigma)
  }
  if(method=="ML")
    lambda <- ML.opt(y = y, q = q, evals = evals)
  if(method=="ML-oracle"){
    if (is.null(f.true)){
      stop("True spectral density is missing.")
    } else {
      check_numeric_vector(f.true)
    }
    if (is.null(sigma)){
      stop("True variance is missing.")
    }
    f.true <- Duplicate(f.true)
    log.sdf.true <- VST(f.true, m=1)
    lambda <- ML.opt(y = NULL, q = q, evals = evals, oracle = TRUE, f.true = log.sdf.true, sigma = sigma)
  }
  
  return(lambda)
}

#' Generalized Cross-Validation
#'
#' @param y       \code{N}-dim. vector with regression data points
#' @param q       penalization order
#' @param evals   eigenvalues of the \code{q}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' @param oracle  logical. If \code{TRUE}, then the \code{GCV-oracle} smoothing 
#'                parameter is calculated.
#' @param f.true  \code{N}-dim. vector with the true regression function 
#'                evaluated at equi-spaced points. Only required if 
#'                \code{oracle=TRUE}.
#' @param sigma   variance of the regression data.
#' 
#' @return lambda estimated smoothing parameter.
#' 
#' @keywords internal
#' @noRd
GCV.opt <- function(y, q, evals, oracle = FALSE, f.true = NULL, sigma = NULL){
  
  GCV <- function(h,y,q) {
    N <- length(y)
    term1 <- stats::fft(y[N:1])[N:1]*(h^(2*q)*evals/(1+h^(2*q)*evals))^2
    term1 <- c(y%*%Re(stats::fft(term1[N:1])[N:1])/N)                              #inverse=TRUE not needed for symmetric vector
    trace <- sum(1/(1+h^(2*q)*evals))
    score <- term1/(N*(1-trace/N)^2)
    return(score)
  }
  
  GCV.oracle <- function(h,f.true,q,sigma) {
    N=length(f.true)
    norm.term=stats::fft(f.true[N:1])[N:1]*h^(2*q)*evals/(1+h^(2*q)*evals)
    norm.term=Re(stats::fft(norm.term[N:1])[N:1])/N
    trace=sum(1/(1+h^(2*q)*evals)^2)
    score=(sum(norm.term^2)+trace*sigma)/N
    return(score)
  }
  
  if (oracle) {
    h <- stats::optimize(
      GCV.oracle,
      interval = c(.Machine$double.eps,1),
      f.true = f.true,
      q = q,
      sigma = sigma,
      tol = .Machine$double.eps
    )$minimum
  } else {
    h <- stats::optimize(
      GCV,
      interval = c(.Machine$double.eps, 1),
      y = y,
      q = q,
      tol = .Machine$double.eps
    )$minimum
  }
  
  lambda <- h^(2*q)
  
  return(lambda)
}

#' Maximum Likelihood method
#'
#' @param y       \code{N}-dim. vector with regression data points
#' @param q       penalization order
#' @param evals   eigenvalues of the \code{q}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' @param oracle  logical. If \code{TRUE}, then the \code{ML-oracle} smoothing 
#'                parameter is calculated.
#' @param f.true  \code{N}-dim. vector with the true regression function 
#'                evaluated at equi-spaced points. Only required if 
#'                \code{oracle=TRUE}.
#' @param sigma   variance of the regression data.
#' 
#' @return estimated smoothing parameter.
#' 
#' @keywords internal
#' @noRd
ML.opt <- function(y, q, evals, oracle = FALSE, f.true = NULL, sigma = NULL){
  
  ML <- function(h,y,q) {
    N=length(y)
    term1=stats::fft(y[N:1])[N:1]*(1/(1+h^(2*q)*evals)-1/(1+h^(2*q)*evals)^2)
    term1=c(y%*%Re(stats::fft(term1[N:1])[N:1])/N)                              #inverse=TRUE not needed for symmetric vector
    fac=(sum(1/(1+h^(2*q)*evals))-q)/(N-q)
    term2=stats::fft(y[N:1])[N:1]*(h^(2*q)*evals/(1+h^(2*q)*evals))
    term2=c(y%*%Re(stats::fft(term2[N:1])[N:1])/N)                              #inverse=TRUE not needed for symmetric vector
    score=(term1-term2*fac)/N
    return(score)
  }
  
  ML.oracle <- function(h,q,f.true,sigma) {
    N=length(f.true)
    term1=stats::fft(f.true[N:1])[N:1]*(1/(1+h^(2*q)*evals)-1/(1+h^(2*q)*evals)^2)
    term1=c(f.true%*%Re(stats::fft(term1[N:1])[N:1])/N)                         #inverse=TRUE not needed for symmetric vector
    term2=sum(1/(1+h^(2*q)*evals)^2)
    score=(term1-sigma*(term2-q))/N
    return(score)
  }
  if (oracle) {
    h <- stats::uniroot(
      ML.oracle,
      interval = c(.Machine$double.eps,1),
      f.true = f.true,
      q = q,
      sigma = sigma,
      tol = .Machine$double.eps, 
      extendInt = "yes"
    )$root 
  } else {
    h <- stats::uniroot(
      ML,
      interval = c(.Machine$double.eps,1),
      y = y,
      q = q,
      tol = .Machine$double.eps,
      extendInt = "yes"
    )$root
  }
  
  lambda <- h^(2*q)
  
  return(lambda)
}