% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/new.penalized.pls.R, R/penalized.pls.R,
%   R/penalized.pls.cv.R, R/penalized.pls.default.R, R/penalized.pls.kernel.R,
%   R/penalized.pls.select.R
\name{penalized.pls}
\alias{penalized.pls}
\alias{new.penalized.pls}
\alias{penalized.pls.cv}
\alias{penalized.pls.default}
\alias{penalized.pls.kernel}
\alias{penalized.pls.select}
\title{Predict New Data Using a Penalized PLS Model}
\usage{
new.penalized.pls(ppls, Xtest, ytest = NULL)

penalized.pls(
  X,
  y,
  P = NULL,
  ncomp = NULL,
  kernel = FALSE,
  scale = FALSE,
  blocks = 1:ncol(X),
  select = FALSE
)

penalized.pls.cv(
  X,
  y,
  P = NULL,
  lambda = 1,
  ncomp = NULL,
  k = 5,
  kernel = FALSE,
  scale = FALSE
)

penalized.pls.default(X, y, M = NULL, ncomp)

penalized.pls.kernel(X, y, M = NULL, ncomp)

penalized.pls.select(X, y, M = NULL, ncomp, blocks)
}
\arguments{
\item{ppls}{A fitted penalized PLS model, as returned by \code{\link{penalized.pls}}.}

\item{Xtest}{A numeric matrix of new input data for prediction.}

\item{ytest}{Optional. A numeric response vector corresponding to \code{Xtest}, for evaluating prediction error.}

\item{X}{A numeric matrix of centered (and optionally scaled) predictor variables.}

\item{y}{A centered numeric response vector.}

\item{P}{Optional penalty matrix. If \code{NULL}, ordinary PLS is computed (i.e., no penalization).}

\item{ncomp}{Integer. Number of PLS components to compute.}

\item{kernel}{Logical. If \code{TRUE}, uses the kernel representation of PPLS. Default is \code{FALSE}.}

\item{scale}{Logical. If \code{TRUE}, scales predictors in \code{X} to unit variance. Default is \code{FALSE}.}

\item{blocks}{An integer vector of length \code{ncol(X)} that defines the block structure of the variables. All variables sharing the same value in \code{blocks} belong to the same block.}

\item{select}{Logical. If \code{TRUE}, block-wise variable selection is applied in each iteration. Only one block contributes to the latent direction per component. Default is \code{FALSE}.}

\item{lambda}{A numeric vector of candidate penalty parameters. Default is \code{1}.}

\item{k}{Integer. Number of cross-validation folds. Default is \code{5}.}

\item{M}{Optional penalty transformation matrix \eqn{M = (I + P)^{-1}}. If \code{NULL}, no penalization is applied.}
}
\value{
A list containing:
\describe{
\item{ypred}{A numeric matrix of predicted responses. Each column corresponds to a different number of PLS components.}
\item{mse}{A numeric vector of mean squared errors, if \code{ytest} is provided. Otherwise \code{NULL}.}
}

A list with components:
\describe{
\item{intercept}{A numeric vector of intercepts for 1 to \code{ncomp} components.}
\item{coefficients}{A numeric matrix of size \code{ncol(X)} x \code{ncomp}, each column being the coefficient vector for the corresponding number of components.}
}

An object of class \code{"mypls"}, a list with the following components:
\describe{
\item{error.cv}{A matrix of mean squared errors. Rows correspond to different \code{lambda} values; columns to different numbers of components.}
\item{lambda}{The vector of candidate lambda values.}
\item{lambda.opt}{The lambda value giving the minimum cross-validated error.}
\item{index.lambda}{The index of \code{lambda.opt} in \code{lambda}.}
\item{ncomp.opt}{The optimal number of PLS components.}
\item{min.ppls}{The minimum cross-validated error.}
\item{intercept}{Intercept of the optimal model (fitted on the full dataset).}
\item{coefficients}{Coefficient vector for the optimal model.}
\item{coefficients.jackknife}{An array of shape \code{ncol(X) x ncomp x length(lambda) x k}, containing the coefficients from each CV split and parameter setting.}
}

A list with:
\describe{
\item{coefficients}{A matrix of size \code{ncol(X) x ncomp}, each column containing the regression coefficients for the first \eqn{i} components.}
}

A list with:
\describe{
\item{coefficients}{A matrix of size \code{ncol(X) x ncomp}, containing the estimated regression coefficients for each number of components.}
}

A list with:
\describe{
\item{coefficients}{A matrix of size \code{ncol(X) x ncomp}, containing the regression coefficients after block-wise selection.}
}
}
\description{
Given a fitted penalized PLS model and new test data, this function predicts the response for all components. If true response values are provided, it also returns the mean squared error (MSE) for each component.

Computes the regression coefficients for a Penalized Partial Least Squares (PPLS) model, using either a classical NIPALS algorithm or a kernel-based version. Optionally allows block-wise variable selection.

Performs k-fold cross-validation to evaluate and select the optimal penalization parameter \code{lambda} and the number of components \code{ncomp} in a PPLS model.

Computes the regression coefficients using the standard (NIPALS-based) version of Penalized PLS. This function is typically called internally by \code{\link{penalized.pls}}.

Computes the regression coefficients using the kernel-based version of Penalized PLS, especially useful when the number of predictors exceeds the number of observations (\code{p >> n}).

Computes the regression coefficients of a Penalized Partial Least Squares (PPLS) model using block-wise selection, where each component is restricted to use variables from only one block.
}
\details{
The fitted model \code{ppls} contains intercepts and regression coefficients for each number of components (from 1 to \code{ncomp}). The function computes:
\itemize{
\item the matrix of predicted values for each component (as columns),
\item and, if \code{ytest} is provided, a vector of mean squared errors for each component.
}

The prediction is performed as:
\deqn{\hat{y}^{(i)} = X_\text{test} \cdot \beta^{(i)} + \text{intercept}^{(i)},}
for each number of components \eqn{i = 1, \ldots, ncomp}.

This function centers \code{X} and \code{y}, and optionally scales \code{X}, then computes PPLS components using one of:
\itemize{
\item the classical NIPALS algorithm (\code{kernel = FALSE}), or
\item the kernel representation (\code{kernel = TRUE}), often faster when \code{p > n} (high-dimensional case).
}

When a penalty matrix \code{P} is supplied, a transformation \eqn{M = (I + P)^{-1}} is computed internally. The algorithm then maximizes the penalized covariance between \code{Xw} and \code{y}:
\deqn{\text{argmax}_w \; \text{Cov}(Xw, y)^2 - \lambda \cdot w^\top P w}

The block-wise selection strategy (when \code{select = TRUE}) restricts the weight vector \code{w} at each iteration to be non-zero in a single block, selected greedily.

The function splits the data into \code{k} cross-validation folds, and for each value of \code{lambda} and number of components up to \code{ncomp}, computes the mean squared prediction error.

The optimal parameters are selected as those minimizing the prediction error across all folds. Internally, for each fold and \code{lambda} value, the function calls \code{\link{penalized.pls}} to fit the model and \code{\link{new.penalized.pls}} to evaluate predictions.

The returned object can be further used for statistical inference (e.g., via jackknife) or prediction.

The method is based on iteratively computing latent directions that maximize the covariance with the response \code{y}. At each step:
\itemize{
\item A weight vector \eqn{w} is computed as \eqn{w = M X^\top y} (if penalization is used).
\item The latent component \eqn{t = X w} is extracted and normalized.
\item The matrix \code{X} is deflated orthogonally with respect to \code{t}.
}

The final regression coefficients are computed via a triangular system using the bidiagonal matrix \eqn{R = T^\top X W}, and backsolving:
\deqn{\beta = W L (T^\top y),}
where \eqn{L = R^{-1}}.

The kernel PPLS algorithm is based on representing the model in terms of the Gram matrix \eqn{K = X M X^\top} (or simply \eqn{K = X X^\top} if \code{M = NULL}). The algorithm iteratively computes orthogonal latent components \eqn{t_i} in sample space.

Steps:
\enumerate{
\item Initialize residual vector \eqn{u = y}, then normalize \eqn{t = Ku}.
\item Orthogonalize \eqn{t} with respect to previous components (if needed).
\item Repeat for \code{ncomp} components.
}

The regression coefficients are recovered as:
\deqn{\beta = X^\top A, \quad \text{where } A = UU L (T^\top y),}
with \eqn{UU} and \eqn{TT} the matrices of latent vectors and components, and \eqn{L = R^{-1}} the back-solved triangular system.

This function implements a sparse selection strategy inspired by sparse or group PLS. At each component iteration, it computes the penalized covariance between \code{X} and \code{y}, and selects the block \code{k} for which the mean squared weight of its variables is maximal:
\deqn{\text{score}_k = \frac{1}{|B_k|} \sum_{j \in B_k} w_j^2}

Only the weights corresponding to the selected block are retained, and all others are set to zero. The rest of the algorithm follows the classical NIPALS-like PLS with orthogonal deflation.

This procedure enhances interpretability by selecting only one block per component, making it suitable for structured variable selection (e.g., grouped predictors).
}
\examples{
set.seed(123)
X <- matrix(rnorm(50 * 200), ncol = 50)
y <- rnorm(200)

Xtrain <- X[1:100, ]
ytrain <- y[1:100]
Xtest <- X[101:200, ]
ytest <- y[101:200]

pen.pls <- penalized.pls(Xtrain, ytrain, ncomp = 10)
pred <- new.penalized.pls(pen.pls, Xtest, ytest)
head(pred$ypred)
pred$mse

## Example from Kraemer et al. (2008)
data(BOD)
X <- BOD[, 1]
y <- BOD[, 2]

Xtest <- seq(min(X), max(X), length = 200)
dummy <- X2s(X, Xtest, deg = 3, nknot = 20)  # Spline transformation
Z <- dummy$Z
Ztest <- dummy$Ztest
size <- dummy$sizeZ
P <- Penalty.matrix(size, order = 2)
lambda <- 200
number.comp <- 3

ppls <- penalized.pls(Z, y, P = lambda * P, ncomp = number.comp)
new.ppls <- new.penalized.pls(ppls, Ztest)$ypred

# Plot fitted values for 2 components
plot(X, y, lwd = 3, xlim = range(Xtest))
lines(Xtest, new.ppls[, 2], col = "blue")

set.seed(42)
X <- matrix(rnorm(20 * 100), ncol = 20)
y <- rnorm(100)

# Example with no penalty
result <- penalized.pls.cv(X, y, lambda = c(0, 1, 10), ncomp = 5)
result$lambda.opt
result$ncomp.opt
result$min.ppls

# Using jackknife estimation after CV
jack <- jack.ppls(result)
coef(jack)

set.seed(123)
X <- matrix(rnorm(20 * 50), nrow = 50)
y <- rnorm(50)
M <- diag(ncol(X))  # No penalty
coef <- penalized.pls.default(scale(X, TRUE, FALSE), scale(y, TRUE, FALSE),
  M, ncomp = 3)$coefficients
coef[, 1]  # coefficients for 1st component

set.seed(123)
X <- matrix(rnorm(100 * 10), nrow = 100)
y <- rnorm(100)
K <- X \%*\% t(X)
coef <- penalized.pls.kernel(X, y, M = NULL, ncomp = 2)$coefficients
head(coef[, 1])  # coefficients for 1st component

set.seed(321)
X <- matrix(rnorm(40 * 30), ncol = 40)
y <- rnorm(30)

# Define 4 blocks of 10 variables each
blocks <- rep(1:4, each = 10)
result <- penalized.pls.select(X, y, M = NULL, ncomp = 2, blocks = blocks)
result$coefficients[, 1]  # Coefficients for first component

}
\references{
N. Kraemer, A.-L. Boulesteix, and G. Tutz (2008). \emph{Penalized Partial Least Squares with Applications to B-Spline Transformations and Functional Data}. Chemometrics and Intelligent Laboratory Systems, 94(1), 60–69. \doi{10.1016/j.chemolab.2008.06.009}

N. Kraemer, A.-L. Boulesteix, and G. Tutz (2008). \emph{Penalized Partial Least Squares with Applications to B-Spline Transformations and Functional Data}. Chemometrics and Intelligent Laboratory Systems, 94(1), 60–69. \doi{10.1016/j.chemolab.2008.06.009}

N. Kraemer, A.-L. Boulesteix, and G. Tutz (2008). \emph{Penalized Partial Least Squares with Applications to B-Spline Transformations and Functional Data}. Chemometrics and Intelligent Laboratory Systems, 94(1), 60–69. \doi{10.1016/j.chemolab.2008.06.009}
}
\seealso{
\code{\link{penalized.pls}}, \code{\link{penalized.pls.cv}}, \code{\link{ppls.splines.cv}}

\code{\link{penalized.pls.cv}}, \code{\link{new.penalized.pls}}, \code{\link{ppls.splines.cv}}, \code{\link{Penalty.matrix}}

\code{\link{penalized.pls}}, \code{\link{new.penalized.pls}}, \code{\link{jack.ppls}}, \code{\link{ppls.splines.cv}}

\code{\link{penalized.pls}}, \code{\link{penalized.pls.kernel}}, \code{\link{normalize.vector}}

\code{\link{penalized.pls}}, \code{\link{penalized.pls.default}}, \code{\link{normalize.vector}}

\code{\link{penalized.pls}}, \code{\link{penalized.pls.cv}}, \code{\link{normalize.vector}}
}
\keyword{internal}
