% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/michaelis_menten_nl.R
\name{michaelis_menten_nl}
\alias{michaelis_menten_nl}
\title{Michaelis-Menten Kinetics (Nonlinear Form) with Inhibition Comparison}
\arguments{
\item{data}{A data.frame containing concentration and rate data.}

\item{conc_col}{Column name for substrate or drug concentration.}

\item{rate_col}{Column name for reaction rate or elimination rate.}

\item{group_col}{Column indicating different conditions (e.g., inhibitor levels).}

\item{inhibition_type}{Expected inhibition type:
"competitive", "noncompetitive", "uncompetitive", "multi-inhibition" or "none".}

\item{plot}{Logical; if TRUE, generates nonlinear fit comparison plot.}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{A data frame with estimated nonlinear
        Michaelis-Menten parameters (\code{Km} and \code{Vmax}) for each group or
        condition, obtained via nonlinear least squares fitting.}
  \item{\code{raw_data}}{A data frame containing the processed concentration
        and rate data used for model fitting and plotting, including group labels.}
  \item{\code{interpretation}}{A character string summarizing the expected
        inhibition pattern based on the specified \code{inhibition_type} and the
        comparison of \code{Km} and \code{Vmax} values across groups.}
}
}
\description{
Performs Michaelis-Menten kinetic analysis using nonlinear least squares
fitting to estimate Km and Vmax for one or more datasets.

This function fits the standard Michaelis-Menten equation:
\deqn{rate = (Vmax * concentration) / (Km + concentration)}
to each group separately and allows comparison of kinetic parameters
across conditions (e.g., inhibitors).

Interpretation of inhibition patterns:
\itemize{
  \item Competitive inhibition: Km increases, Vmax unchanged
  \item Noncompetitive inhibition: Vmax decreases, Km unchanged
  \item Uncompetitive inhibition: both Km and Vmax decrease
}
}
\examples{
# Example I: Single Michaelis-Menten dataset (no inhibition)
df1 <- data.frame(
  concentration = c(0.5, 1, 2, 4, 6, 8, 10),
  rate = c(0.48, 0.85, 1.45, 2.20, 2.70, 3.05, 3.25),
  group = "No Inhibitor"
)
michaelis_menten_nl(
  data = df1,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  inhibition_type = "none",
  plot = TRUE
)

# Example II: Two datasets compared (inhibition analysis)
df2 <- data.frame(
  concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 2),
  rate = c(
    # Reference (no inhibitor)
    0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
    # Condition B (possible inhibitor)
    0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55
  ),
  group = rep(c("No Inhibitor", "Inhibitor"), each = 7)
)
michaelis_menten_nl(
  data = df2,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  inhibition_type = "uncompetitive",
  plot = TRUE
)

# Example III: Six datasets compared (one reference, five test conditions)
df3 <- data.frame(
  concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 6),
  rate = c(
    # Reference
    0.50, 0.95, 1.80, 2.90, 3.60, 4.00, 4.30,
    # Competitive inhibitor
    0.35, 0.70, 1.35, 2.40, 3.05, 3.40, 3.65,
    # Mixed Noncompetitive inhibitor A
    0.30, 0.55, 1.00, 1.65, 2.05, 2.35, 2.50,
    # Uncompetitive inhibitor
    0.25, 0.50, 0.90, 1.40, 1.75, 2.00, 2.10,
    # Mixed Noncompetitive inhibitor high dose
    0.20, 0.40, 0.80, 1.50, 1.95, 2.25, 2.40,
    # Mixed Noncompetitive inhibitor B
    0.25, 0.45, 0.85, 1.35, 1.70, 1.95, 2.05
  ),
  group = rep(c(
    "Reference",
    "Competitive",
    "Noncompetitive (Mixed) A",
    "Uncompetitive",
    "Noncompetitive (Mixed) High",
    "Noncompetitive (Mixed) B"
  ), each = 7)
)
michaelis_menten_nl(
  data = df3,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  inhibition_type = "multi-inhibition",
  plot = TRUE
)
}
\references{
Michaelis, L. and Menten, M. (1913) Die kinetik der invertinwirkung.
Biochemistry Zeitung, 79, 333-369.
}
\author{
Paul Angelo C. Manlapaz
}
