\name{np.aov.test}
\alias{np.aov.test}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Nonparametric One-Way and RM ANOVA Tests
}
\description{
Assuming a one-way (fixed effects) ANOVA model of the form 
\deqn{Y_{ij} = \mu + \tau_j + \epsilon_{ij}}
or a one-way repeated measures ANOVA model of the form
\deqn{Y_{ij} = \mu + \beta_i + \tau_j + \epsilon_{ij}}
this function implements permutation tests of \eqn{H_0: (\forall j) \tau_j = \tau} versus \eqn{H_1: (\exists j) \tau_j \neq \tau}. 
Note that \eqn{\mu} is the overall mean/median ignoring block and group, \eqn{\beta_i} is the \eqn{i}-th subject's block effect, \eqn{\tau_j} is the \eqn{j}-th group's treatment effect,  and \eqn{\epsilon_{ij}} is an error term with mean/median zero.
}
\usage{
np.aov.test(x, groups, blocks = NULL, 
            var.equal = FALSE, median.test = FALSE, 
            R = 9999, parallel = FALSE, cl = NULL,
            perm.dist = TRUE, na.rm = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{
Numeric vector (or matrix) of data values (see Details).
}
  \item{groups}{
Factor vector giving the treatment group for each element/row of \code{x}.
}
  \item{blocks}{
Factor vector giving the block identification for each element/row of \code{x}.
}
  \item{var.equal}{
Logical indicating whether to treat the \eqn{k} group's variances as being equal.
}

  \item{median.test}{
Logical indicating whether the location test is for the median. Default is \code{FALSE}, i.e., \eqn{\mu} is the mean.
}
  \item{R}{
Number of resamples for the permutation test (positive integer). 
}
  \item{parallel}{
Logical indicating if the \code{\link{parallel}} package should be used for parallel computing (of the permutation distribution). Defaults to \code{FALSE}, which implements sequential computing.
}
  \item{cl}{
Cluster for parallel computing, which is used when \code{parallel = TRUE}. Note that if \code{parallel = TRUE} and \code{cl = NULL}, then the cluster is defined as \code{makeCluster(2L)} to use two cores. To make use of all available cores, use the code \code{cl = makeCluster(detectCores())}.
}
  \item{perm.dist}{
Logical indicating if the permutation distribution should be returned.  
}
  \item{na.rm}{
If \code{TRUE} (default), the arguments \code{x} and \code{groups} (and \code{blocks} if provided) are passed to the \code{\link{na.omit}} function to remove cases with missing data.
}
}
\details{
One-way ANOVA: the input \code{x} should be of length \eqn{N = \sum_{j=1}^k n_j} where \eqn{n_j} is size of \eqn{j}-th group. 

RM ANOVA: the input \code{x} should be of length \eqn{N = n k} where \eqn{n} is number of blocks and \eqn{k} is number of groups.

For multivariate models, the input \code{x} should be a matrix with \eqn{N} rows and \eqn{m} columns, where each column has \eqn{N = \sum_{j=1}^k n_j} or \eqn{N = n k} observations.
}
\value{
\item{statistic }{
Test statistic value.
}
  \item{p.value }{
p-value for testing \eqn{H_0: (\forall j) \tau_j = \tau}.
}
  \item{perm.dist }{
Permutation distribution of \code{statistic}. 
}
  \item{repeated }{
Repeated-measures ANOVA?
}
  \item{var.equal }{
Assuming equal variances?
}
  \item{median.test }{
Testing the median?
}
  \item{R }{
Number of resamples.
}
  \item{method }{
Method used for permutation test. See Examples.
}
  \item{ngroups }{
Number of groups = \code{nlevels(group)}  
}
  \item{nblocks }{
Number of blocks = \code{nlevels(blocks)} (if applicable)
}
}
\references{
Helwig, N. E. (2019a). Statistical nonparametric mapping: Multivariate permutation tests for location, correlation, and regression problems in neuroimaging. WIREs Computational Statistics, 11(2), e1457. doi: 10.1002/wics.1457

Helwig, N. E. (2019b). Robust nonparametric tests of general linear model coefficients: A comparison of permutation methods and test statistics. NeuroImage, 201, 116030. doi: 10.1016/j.neuroimage.2019.116030
}
\author{
Nathaniel E. Helwig <helwig@umn.edu>
}
\note{
For the one-way ANOVA, the number of elements of the exact (i.e., fully enumerated) permutation distribuion is given by the multinomial coefficient:
\deqn{\frac{N!}{n_1! n_2! \cdots n_k!}}
which will be quite large (much larger than typcailly choices for \code{R}) for any non-trivial sample sizes. Consequently, exact tests are **not** implemented by this function.
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{plot.np.aov.test}} S3 plotting method for visualizing the results
}
\examples{

###***###   ONE-WAY ANOVA   ###***###

# data generation design
N <- 90
k <- 3
g <- factor(rep(LETTERS[1:k], each = N/k))
tau <- c(-1/2, 0, 1/2)
sd <- c(1/2, 1, 2)

# generate data
set.seed(0)
x <- rnorm(N, mean = tau[g], sd = sd[g]) 

# mean test with unequal variances (robust W statistic)
set.seed(1)
np.aov.test(x, g)

# mean test with equal variances (classic F statistic)
set.seed(1)
np.aov.test(x, g, var.equal = TRUE)

# median test with unequal variances (robust Kruskal-Wallis statistic)
set.seed(1)
np.aov.test(x, g, median.test = TRUE)

# median test with equal variances (classic Kruskal-Wallis test)
set.seed(1)
np.aov.test(x, g, var.equal = TRUE, median.test = TRUE)

# Kruskal-Wallis test (asymptotic p-value)
kruskal.test(x, g)


\dontrun{

###***###   REPEATED MEASURES ANOVA   ###***###

# data generation design
N <- 90
k <- 3
n <- 30
g <- factor(rep(LETTERS[1:k], each = N/k))
b <- factor(rep(paste0("sub", 1:n), times = k),
            levels = paste0("sub", 1:n))
tau <- c(-1/2, 0, 1/2)
sd <- c(1/2, 1, 2)

# generate random block effects
set.seed(773)
beta <- runif(30, -1, 1)

# generate data
set.seed(0)
x <- rnorm(N, mean = tau[g] + beta[b], sd = sd[g]) 

# mean test with unequal variances (robust W statistic)
set.seed(1)
np.aov.test(x, g, b)

# mean test with equal variances (classic F statistic)
set.seed(1)
np.aov.test(x, g, b, var.equal = TRUE)

# median test with unequal variances (robust Friedman statistic)
set.seed(1)
np.aov.test(x, g, b, median.test = TRUE)

# median test with equal variances (classic Friedman test)
set.seed(1)
np.aov.test(x, g, b, var.equal = TRUE, median.test = TRUE)

# Friedman test (asymptotic p-value)
friedman.test(x, g, b)
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory (show via RShowDoc("KEYWORDS")):
\keyword{ htest }
\keyword{ nonparametric }
\keyword{ multivariate }
\keyword{ robust }
\keyword{ univar }
