#include <cmath>

#include "EMRPoint.h"
#include "naryn.h"
#include "NRPoint.h"

struct EMRPPointsSort {
    bool operator()(const EMRPoint *p1, const EMRPoint *p2) const { return *p1 < *p2; }
};

const char *NRPoint::COL_NAMES[NUM_PVAL_COLS] = { "id", "time", "ref", "value" };

SEXP NRPoint::convert_points(const vector<EMRPoint> &points, unsigned num_cols, bool null_if_empty, bool do_sort, vector<EMRPoint *> *ppoints)
{
    if (null_if_empty && !points.size())
        return R_NilValue;

    if (ppoints) {
        bool need_sort = false;

        ppoints->clear();
        ppoints->reserve(points.size());

        if (!points.empty()){
            ppoints->push_back((EMRPoint *)&points.front());

            for (vector<EMRPoint>::const_iterator ipoint = points.begin() + 1; ipoint < points.end(); ++ipoint) {
                ppoints->push_back((EMRPoint *)&*ipoint);
                need_sort = need_sort | (*ipoint < *(ipoint - 1));
            }
        }

        if (do_sort && need_sort){
            sort(ppoints->begin(), ppoints->end(), EMRPPointsSort());
        } else {
            ppoints = NULL;
        }
    }


    if (null_if_empty && points.size())
        return R_NilValue;

    SEXP answer;
    SEXP row_names;
    SEXP col_names;
    SEXP ids;
    SEXP times;
    SEXP refs;

    rprotect(answer = RSaneAllocVector(VECSXP, num_cols));
    rprotect(col_names = RSaneAllocVector(STRSXP, num_cols));
    rprotect(row_names = RSaneAllocVector(INTSXP, points.size()));
    rprotect(ids = RSaneAllocVector(INTSXP, points.size()));
    rprotect(times = RSaneAllocVector(INTSXP, points.size()));
    rprotect(refs = RSaneAllocVector(INTSXP, points.size()));

    for (uint64_t i = 0; i < points.size(); ++i)
        INTEGER(row_names)[i] = i + 1;

    for (int i = 0; i < NUM_POINT_COLS; i++)
        SET_STRING_ELT(col_names, i, Rf_mkChar(COL_NAMES[i]));

    if (ppoints) {
        for (vector<EMRPoint *>::const_iterator ippoint = ppoints->begin(); ippoint != ppoints->end(); ++ippoint) {
            uint64_t index = ippoint - ppoints->begin();
            INTEGER(ids)[index] = (*ippoint)->id;
            INTEGER(times)[index] = (*ippoint)->timestamp.hour();
            INTEGER(refs)[index] = (*ippoint)->timestamp.refcount() == EMRTimeStamp::NA_REFCOUNT ? -1 : (*ippoint)->timestamp.refcount();
        }
    } else {
        for (EMRPoints::const_iterator ipoint = points.begin(); ipoint != points.end(); ++ipoint) {
            uint64_t index = ipoint - points.begin();
            INTEGER(ids)[index] = ipoint->id;
            INTEGER(times)[index] = ipoint->timestamp.hour();
            INTEGER(refs)[index] = ipoint->timestamp.refcount() == EMRTimeStamp::NA_REFCOUNT ? -1 : ipoint->timestamp.refcount();
        }
    }

    SET_VECTOR_ELT(answer, ID, ids);
    SET_VECTOR_ELT(answer, TIME, times);
    SET_VECTOR_ELT(answer, REF, refs);

    Rf_setAttrib(answer, R_NamesSymbol, col_names);
    Rf_setAttrib(answer, R_ClassSymbol, Rf_mkString("data.frame"));
    Rf_setAttrib(answer, R_RowNamesSymbol, row_names);

    return answer;
}

void NRPoint::convert_rpoints(SEXP rpoints, vector<EMRPoint> *points, const char *error_msg_prefix)
{
    points->clear();

    if (TYPEOF(rpoints) == PROMSXP) {
        if (NARYN_PRENV(rpoints) == R_NilValue)
            rpoints = NARYN_PRVALUE(rpoints);
        else
            rpoints = eval_in_R(NARYN_PREXPR(rpoints), NARYN_PRENV(rpoints));
    }

    if (!Rf_isVector(rpoints))
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of id-time points", error_msg_prefix);

    SEXP colnames = Rf_getAttrib(rpoints, R_NamesSymbol);

    if (!Rf_isString(colnames) || Rf_length(colnames) < NUM_POINT_COLS - 1)
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of id-time points", error_msg_prefix);

    for (unsigned i = 0; i < NUM_POINT_COLS; i++) {
        if (i == REF) // reference column is optional
            continue;

        if (strcmp(CHAR(STRING_ELT(colnames, i)), COL_NAMES[i]))
            TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of id-time points", error_msg_prefix);
    }

    SEXP ids = VECTOR_ELT(rpoints, ID);
    SEXP hours = VECTOR_ELT(rpoints, TIME);
    SEXP refs = Rf_length(colnames) >= NUM_POINT_COLS && !strcmp(CHAR(STRING_ELT(colnames, REF)), COL_NAMES[REF]) ? VECTOR_ELT(rpoints, REF) : R_NilValue;
    unsigned num_points = (unsigned)Rf_length(ids);

    for (unsigned i = 1; i < NUM_POINT_COLS; i++) {
        if ((i != REF || (i == REF && refs != R_NilValue)) && Rf_length(VECTOR_ELT(rpoints, i)) != Rf_length(VECTOR_ELT(rpoints, i - 1))){
            TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of id-time points", error_msg_prefix);
        }
    }

    if ((!Rf_isReal(ids) && !Rf_isInteger(ids)) || (!Rf_isReal(hours) && !Rf_isInteger(hours)) || ((refs != R_NilValue) && !Rf_isReal(refs) && !Rf_isInteger(refs))){
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of id-time points", error_msg_prefix);
    }


    for (unsigned i = 0; i < num_points; i++) {
        if ((Rf_isReal(ids) && std::isnan(REAL(ids)[i])) || (Rf_isReal(hours) && std::isnan(REAL(hours)[i])) || ((refs != R_NilValue) && Rf_isReal(refs) && std::isnan(REAL(refs)[i]))){
            TGLError<NRPoint>(BAD_VALUE, "%sInvalid format of id-time points, row %d", error_msg_prefix, i + 1);
        }

        int id = Rf_isReal(ids) ? REAL(ids)[i] : INTEGER(ids)[i];
        int hour = Rf_isReal(hours) ? REAL(hours)[i] : INTEGER(hours)[i];
        int ref = -1;

        if (refs != R_NilValue)
            ref = Rf_isReal(refs) ? REAL(refs)[i] : INTEGER(refs)[i];

        if (Rf_isReal(ids) && REAL(ids)[i] != id)
            TGLError<NRPoint>(BAD_VALUE, "%sInvalid id at id-time points, row %d", error_msg_prefix, i + 1);

        if ((Rf_isReal(hours) && REAL(hours)[i] != hour) || hour < 0 ||
            (EMRTimeStamp::Hour)hour > EMRTimeStamp::MAX_HOUR)
            TGLError<NRPoint>(BAD_VALUE, "%sInvalid time at id-time points, row %d", error_msg_prefix, i + 1);

        if ((refs != R_NilValue && Rf_isReal(refs) && REAL(refs)[i] != ref )|| ref < -1 || ref > EMRTimeStamp::MAX_REFCOUNT)
            TGLError<NRPoint>(BAD_VALUE, "%sInvalid reference at id-time points, row %d", error_msg_prefix, i + 1);

		points->push_back(EMRPoint(id, EMRTimeStamp((EMRTimeStamp::Hour)hour, (EMRTimeStamp::Refcount)ref)));
    }
}

SEXP NRPoint::convert_ids(const vector<unsigned> &ids, unsigned num_cols, bool null_if_empty)
{
    if (null_if_empty && !ids.size())
        return R_NilValue;

    SEXP answer;
    SEXP rids;
    SEXP row_names;
    SEXP col_names;

    rprotect(answer = RSaneAllocVector(VECSXP, num_cols));
    rprotect(rids = RSaneAllocVector(INTSXP, ids.size()));
    rprotect(col_names = RSaneAllocVector(STRSXP, num_cols));
    rprotect(row_names = RSaneAllocVector(INTSXP, ids.size()));

    for (vector<unsigned>::const_iterator iid = ids.begin(); iid != ids.end(); ++iid) {
        uint64_t index = iid - ids.begin();
        INTEGER(rids)[index] = *iid;
        INTEGER(row_names)[index] = index + 1;
    }

    SET_STRING_ELT(col_names, 0, Rf_mkChar("id"));
    SET_VECTOR_ELT(answer, 0, rids);

    Rf_setAttrib(answer, R_NamesSymbol, col_names);
    Rf_setAttrib(answer, R_ClassSymbol, Rf_mkString("data.frame"));
    Rf_setAttrib(answer, R_RowNamesSymbol, row_names);

    return answer;
}

void NRPoint::convert_rids(SEXP rids, vector<unsigned> *ids, const char *error_msg_prefix)
{
    ids->clear();

    if (TYPEOF(rids) == PROMSXP) {
        if (NARYN_PRENV(rids) == R_NilValue)
            rids = NARYN_PRVALUE(rids);
        else
            rids = eval_in_R(NARYN_PREXPR(rids), NARYN_PRENV(rids));
    }

    if (!Rf_isVector(rids))
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of ids", error_msg_prefix);

    SEXP colnames = Rf_getAttrib(rids, R_NamesSymbol);

    if (!Rf_isString(colnames) || Rf_length(colnames) < 1 || strcmp(CHAR(STRING_ELT(colnames, 0)), "id"))
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of ids", error_msg_prefix);

    SEXP rvids = VECTOR_ELT(rids, 0);

    if (Rf_isReal(rvids)) {
        for (int i = 0; i < Rf_length(rvids); ++i) {
            double id = REAL(rvids)[i];
            if (id < 0 || (double)(int)id != id)
                TGLError<NRPoint>(BAD_VALUE, "%sInvalid id at row %d", error_msg_prefix, i + 1);
            ids->push_back((unsigned)id);
        }
    } else if (Rf_isInteger(rvids)) {
        for (int i = 0; i < Rf_length(rvids); ++i) {
            int id = INTEGER(rvids)[i];
            if (id < 0)
                TGLError<NRPoint>(BAD_VALUE, "%sInvalid id at row %d", error_msg_prefix, i + 1);
            ids->push_back((unsigned)id);
        }
    } else
        TGLError<NRPoint>(BAD_FORMAT, "%sInvalid format of ids", error_msg_prefix);
}
