#' Fit Longitudinal Mixed-Effects Models for Each Biomarker
#'
#' @description
#' Stage 1 of the two-stage estimation: fits a random intercept-slope model
#' \eqn{y_{ij}(t) = (\beta_{0j} + b_{0ij}) + (\beta_{1j} + b_{1ij})t + \epsilon_{ij}(t)}
#' for each biomarker using \code{nlme::lme}.
#'
#' @param long_data Data frame with columns \code{patient_id},
#'   \code{visit_time_years}, \code{biomarker}, \code{value}.
#' @param markers Character vector of biomarker names to fit. If \code{NULL},
#'   all unique biomarkers in \code{long_data} are used.
#' @param verbose Logical; print progress. Default \code{TRUE}.
#'
#' @return Named list of \code{nlme::lme} objects, one per biomarker.
#'   Failed fits are \code{NULL}.
#'
#' @export
fit_longitudinal <- function(long_data, markers = NULL, verbose = TRUE) {

  if (is.null(markers)) markers <- unique(long_data$biomarker)

  lme_fits <- list()
  for (mk in markers) {
    if (verbose) message("  Fitting LME for: ", mk)

    df_mk <- long_data[long_data$biomarker == mk, ]
    df_mk$y <- df_mk$value
    df_mk$time <- df_mk$visit_time_years
    df_mk$id <- factor(df_mk$patient_id)

    lme_fits[[mk]] <- tryCatch(
      nlme::lme(y ~ time, random = ~ time | id, data = df_mk,
                control = nlme::lmeControl(opt = "optim", maxIter = 300)),
      error = function(e) {
        if (verbose) message("    WARNING: LME failed for ", mk, ": ", e$message)
        NULL
      }
    )

    if (!is.null(lme_fits[[mk]]) && verbose) {
      fe <- nlme::fixef(lme_fits[[mk]])
      message("    Intercept = ", round(fe[1], 2),
              ", Slope = ", round(fe[2], 3))
    }
  }

  lme_fits
}


#' Compute BLUP-Based Latent Longitudinal Summaries
#'
#' @description
#' Extracts subject-specific BLUPs from fitted \code{lme} models and computes
#' the latent trajectory \eqn{\eta_{ij}(t) = \hat\beta_{0j} + \hat b_{0ij} +
#' (\hat\beta_{1j} + \hat b_{1ij})t} at specified time points.
#'
#' @param lme_fits Named list of \code{nlme::lme} objects (from \code{fit_longitudinal}).
#' @param patient_ids Numeric vector of patient IDs.
#' @param times Numeric vector of evaluation times.
#' @param markers Character vector of biomarker names. If \code{NULL}, uses
#'   names of \code{lme_fits}.
#'
#' @return Data frame with columns \code{patient_id}, \code{time}, and one
#'   \code{eta_*} column per biomarker.
#'
#' @export
compute_blup_eta <- function(lme_fits, patient_ids, times, markers = NULL) {

  if (is.null(markers)) markers <- names(lme_fits)

  ## Build grid
  grid <- expand.grid(patient_id = patient_ids, time = times,
                      stringsAsFactors = FALSE)

  for (mk in markers) {
    if (is.null(lme_fits[[mk]])) next
    cm <- coef(lme_fits[[mk]])
    mk_clean <- gsub("[^A-Za-z0-9]", "", mk)
    pid_char <- as.character(grid$patient_id)
    available_ids <- rownames(cm)
    matched <- pid_char %in% available_ids

    grid[[paste0("eta_", mk_clean)]] <- NA_real_
    if (any(matched)) {
      grid[[paste0("eta_", mk_clean)]][matched] <-
        cm[pid_char[matched], 1] + cm[pid_char[matched], 2] * grid$time[matched]
    }
  }

  grid
}
