/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Cpkg.h"      
#include "H5Eprivate.h"  
#include "H5SLprivate.h" 

#ifndef NDEBUG

herr_t
H5C_dump_cache(H5C_t *cache_ptr, const char *cache_name)
{
    H5C_cache_entry_t *entry_ptr;
    H5SL_t            *slist_ptr = NULL;
    int                i;                   
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cache_ptr != NULL);
    assert(cache_name != NULL);

    
    if (NULL == (slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTCREATE, FAIL, "can't create skip list");

    
    for (i = 0; i < H5C__HASH_TABLE_LEN; i++) {
        entry_ptr = cache_ptr->index[i];

        while (entry_ptr != NULL) {
            if (H5SL_insert(slist_ptr, entry_ptr, &(entry_ptr->addr)) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "can't insert entry in skip list");

            entry_ptr = entry_ptr->ht_next;
        } 
    }     

    

    Rfprintf(Rstdout, "\n\nDump of metadata cache \"%s\"\n", cache_name);

    
    Rfprintf(Rstdout, "Entry ");
    Rfprintf(Rstdout, "|       Address      ");
    Rfprintf(Rstdout, "|         Tag        ");
    Rfprintf(Rstdout, "|  Size ");
    Rfprintf(Rstdout, "| Ring ");
    Rfprintf(Rstdout, "|              Type              ");
    Rfprintf(Rstdout, "| Prot/Pin/Dirty");
    Rfprintf(Rstdout, "\n");

    Rfprintf(Rstdout, "--------------------------------------------------------------------------------------"
                    "--------------------------\n");

    i         = 0;
    entry_ptr = (H5C_cache_entry_t *)H5SL_remove_first(slist_ptr);
    while (entry_ptr != NULL) {
        
        Rfprintf(Rstdout, "%s%5d ", cache_ptr->prefix, i);
        Rfprintf(Rstdout, "  0x%16llx ", (long long)(entry_ptr->addr));
        if (NULL == entry_ptr->tag_info)
            Rfprintf(Rstdout, "    %16s ", "N/A");
        else
            Rfprintf(Rstdout, "  0x%16llx ", (long long)(entry_ptr->tag_info->tag));
        Rfprintf(Rstdout, "  %5lld ", (long long)(entry_ptr->size));
        Rfprintf(Rstdout, "    %d  ", (int)(entry_ptr->ring));
        Rfprintf(Rstdout, "  %2d %-32s ", (int)(entry_ptr->type->id), (entry_ptr->type->name));
        Rfprintf(Rstdout, " %d", (int)(entry_ptr->is_protected));
        Rfprintf(Rstdout, " %d", (int)(entry_ptr->is_pinned));
        Rfprintf(Rstdout, " %d", (int)(entry_ptr->is_dirty));
        Rfprintf(Rstdout, "\n");

        
        entry_ptr = (H5C_cache_entry_t *)H5SL_remove_first(slist_ptr);

        i++;
    } 

    Rfprintf(Rstdout, "\n\n");

    
    assert(H5SL_count(slist_ptr) == 0);

done:
    
    if (slist_ptr)
        H5SL_close(slist_ptr);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG

herr_t
H5C_dump_cache_LRU(H5C_t *cache_ptr, const char *cache_name)
{
    H5C_cache_entry_t *entry_ptr;
    int                i = 0;

    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache_ptr != NULL);
    assert(cache_name != NULL);

    Rfprintf(Rstdout, "\n\nDump of metadata cache LRU \"%s\"\n", cache_name);
    Rfprintf(Rstdout, "LRU len = %d, LRU size = %d\n", cache_ptr->LRU_list_len,
            (int)(cache_ptr->LRU_list_size));
    Rfprintf(Rstdout, "index_size = %d, max_cache_size = %d, delta = %d\n\n", (int)(cache_ptr->index_size),
            (int)(cache_ptr->max_cache_size),
            (int)(cache_ptr->max_cache_size) - (int)(cache_ptr->index_size));

    
    Rfprintf(Rstdout, "Entry ");
    Rfprintf(Rstdout, "|       Address      ");
    Rfprintf(Rstdout, "|         Tag        ");
    Rfprintf(Rstdout, "|  Size ");
    Rfprintf(Rstdout, "| Ring ");
    Rfprintf(Rstdout, "|              Type              ");
    Rfprintf(Rstdout, "| Dirty");
    Rfprintf(Rstdout, "\n");

    Rfprintf(Rstdout, "--------------------------------------------------------------------------------------"
                    "--------------------------\n");

    entry_ptr = cache_ptr->LRU_head_ptr;
    while (entry_ptr != NULL) {
        
        Rfprintf(Rstdout, "%s%5d ", cache_ptr->prefix, i);
        Rfprintf(Rstdout, "  0x%16llx ", (long long)(entry_ptr->addr));

        if (NULL == entry_ptr->tag_info)
            Rfprintf(Rstdout, "    %16s ", "N/A");
        else
            Rfprintf(Rstdout, "  0x%16llx ", (long long)(entry_ptr->tag_info->tag));

        Rfprintf(Rstdout, "  %5lld ", (long long)(entry_ptr->size));
        Rfprintf(Rstdout, "    %d  ", (int)(entry_ptr->ring));
        Rfprintf(Rstdout, "  %2d %-32s ", (int)(entry_ptr->type->id), (entry_ptr->type->name));
        Rfprintf(Rstdout, " %d", (int)(entry_ptr->is_dirty));
        Rfprintf(Rstdout, "\n");

        i++;
        entry_ptr = entry_ptr->next;
    } 

    Rfprintf(Rstdout, "--------------------------------------------------------------------------------------"
                    "--------------------------\n");

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
#endif 

#ifndef NDEBUG
herr_t
H5C_dump_cache_skip_list(H5C_t *cache_ptr, char *calling_fcn)
{
    herr_t             ret_value = SUCCEED; 
    int                i;
    H5C_cache_entry_t *entry_ptr = NULL;
    H5SL_node_t       *node_ptr  = NULL;

    FUNC_ENTER_NOAPI_NOERR

    assert(cache_ptr != NULL);
    assert(calling_fcn != NULL);

    Rfprintf(Rstdout, "\n\nDumping metadata cache skip list from %s.\n", calling_fcn);
    Rfprintf(Rstdout, " slist %s.\n", cache_ptr->slist_enabled ? "enabled" : "disabled");
    Rfprintf(Rstdout, "	slist len = %" PRIu32 ".\n", cache_ptr->slist_len);
    Rfprintf(Rstdout, "	slist size = %llu.\n", (unsigned long long)(cache_ptr->slist_size));

    if (cache_ptr->slist_len > 0) {

        
        Rfprintf(Rstdout, "Num:    Addr:               Len: Prot/Pind: Dirty: Type:\n");

        i = 0;

        node_ptr = H5SL_first(cache_ptr->slist_ptr);
        if (node_ptr != NULL)
            entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
        else
            entry_ptr = NULL;

        while (entry_ptr != NULL) {
            Rfprintf(Rstdout, "%s%d       0x%016llx  %4lld    %d/%d       %d    %s\n", cache_ptr->prefix, i,
                    (long long)(entry_ptr->addr), (long long)(entry_ptr->size),
                    (int)(entry_ptr->is_protected), (int)(entry_ptr->is_pinned), (int)(entry_ptr->is_dirty),
                    entry_ptr->type->name);
            Rfprintf(Rstdout, "		node_ptr = %p, item = %p\n", (void *)node_ptr, H5SL_item(node_ptr));

            
            node_ptr = H5SL_next(node_ptr);
            if (node_ptr != NULL)
                entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
            else
                entry_ptr = NULL;

            i++;
        } 
    }     

    Rfprintf(Rstdout, "\n\n");

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5C_set_prefix(H5C_t *cache_ptr, char *prefix)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL || prefix == NULL || strlen(prefix) >= H5C__PREFIX_LEN)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad param(s) on entry");

    strncpy(&(cache_ptr->prefix[0]), prefix, (size_t)(H5C__PREFIX_LEN));

    cache_ptr->prefix[H5C__PREFIX_LEN - 1] = '\0';

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_stats(H5C_t *cache_ptr, const char *cache_name,
          bool
#if !H5C_COLLECT_CACHE_STATS
              H5_ATTR_UNUSED
#endif 
                  display_detailed_stats)
{
#if H5C_COLLECT_CACHE_STATS
    int     i;
    int64_t total_hits                     = 0;
    int64_t total_misses                   = 0;
    int64_t total_write_protects           = 0;
    int64_t total_read_protects            = 0;
    int64_t max_read_protects              = 0;
    int64_t total_insertions               = 0;
    int64_t total_pinned_insertions        = 0;
    int64_t total_clears                   = 0;
    int64_t total_flushes                  = 0;
    int64_t total_evictions                = 0;
    int64_t total_take_ownerships          = 0;
    int64_t total_moves                    = 0;
    int64_t total_entry_flush_moves        = 0;
    int64_t total_cache_flush_moves        = 0;
    int64_t total_size_increases           = 0;
    int64_t total_size_decreases           = 0;
    int64_t total_entry_flush_size_changes = 0;
    int64_t total_cache_flush_size_changes = 0;
    int64_t total_pins                     = 0;
    int64_t total_unpins                   = 0;
    int64_t total_dirty_pins               = 0;
    int64_t total_pinned_flushes           = 0;
    int64_t total_pinned_clears            = 0;
    int32_t aggregate_max_accesses         = 0;
    int32_t aggregate_min_accesses         = 1000000;
    int32_t aggregate_max_clears           = 0;
    int32_t aggregate_max_flushes          = 0;
    size_t  aggregate_max_size             = 0;
    int32_t aggregate_max_pins             = 0;
    double  hit_rate;
    double  prefetch_use_rate;
    double  average_successful_search_depth                   = 0.0;
    double  average_failed_search_depth                       = 0.0;
    double  average_entries_skipped_per_calls_to_msic         = 0.0;
    double  average_dirty_pf_entries_skipped_per_call_to_msic = 0.0;
    double  average_entries_scanned_per_calls_to_msic         = 0.0;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (NULL == cache_ptr || NULL == cache_name)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr or cache_name");

#if H5C_COLLECT_CACHE_STATS
    for (i = 0; i <= cache_ptr->max_type_id; i++) {
        total_hits += cache_ptr->hits[i];
        total_misses += cache_ptr->misses[i];
        total_write_protects += cache_ptr->write_protects[i];
        total_read_protects += cache_ptr->read_protects[i];
        if (max_read_protects < cache_ptr->max_read_protects[i])
            max_read_protects = cache_ptr->max_read_protects[i];
        total_insertions += cache_ptr->insertions[i];
        total_pinned_insertions += cache_ptr->pinned_insertions[i];
        total_clears += cache_ptr->clears[i];
        total_flushes += cache_ptr->flushes[i];
        total_evictions += cache_ptr->evictions[i];
        total_take_ownerships += cache_ptr->take_ownerships[i];
        total_moves += cache_ptr->moves[i];
        total_entry_flush_moves += cache_ptr->entry_flush_moves[i];
        total_cache_flush_moves += cache_ptr->cache_flush_moves[i];
        total_size_increases += cache_ptr->size_increases[i];
        total_size_decreases += cache_ptr->size_decreases[i];
        total_entry_flush_size_changes += cache_ptr->entry_flush_size_changes[i];
        total_cache_flush_size_changes += cache_ptr->cache_flush_size_changes[i];
        total_pins += cache_ptr->pins[i];
        total_unpins += cache_ptr->unpins[i];
        total_dirty_pins += cache_ptr->dirty_pins[i];
        total_pinned_flushes += cache_ptr->pinned_flushes[i];
        total_pinned_clears += cache_ptr->pinned_clears[i];
#if H5C_COLLECT_CACHE_ENTRY_STATS
        if (aggregate_max_accesses < cache_ptr->max_accesses[i])
            aggregate_max_accesses = cache_ptr->max_accesses[i];
        if (aggregate_min_accesses > aggregate_max_accesses)
            aggregate_min_accesses = aggregate_max_accesses;
        if (aggregate_min_accesses > cache_ptr->min_accesses[i])
            aggregate_min_accesses = cache_ptr->min_accesses[i];
        if (aggregate_max_clears < cache_ptr->max_clears[i])
            aggregate_max_clears = cache_ptr->max_clears[i];
        if (aggregate_max_flushes < cache_ptr->max_flushes[i])
            aggregate_max_flushes = cache_ptr->max_flushes[i];
        if (aggregate_max_size < cache_ptr->max_size[i])
            aggregate_max_size = cache_ptr->max_size[i];
        if (aggregate_max_pins < cache_ptr->max_pins[i])
            aggregate_max_pins = cache_ptr->max_pins[i];
#endif 
    }  

    if ((total_hits > 0) || (total_misses > 0))
        hit_rate = 100.0 * ((double)(total_hits)) / ((double)(total_hits + total_misses));
    else
        hit_rate = 0.0;

    if (cache_ptr->successful_ht_searches > 0)
        average_successful_search_depth = ((double)(cache_ptr->total_successful_ht_search_depth)) /
                                          ((double)(cache_ptr->successful_ht_searches));

    if (cache_ptr->failed_ht_searches > 0)
        average_failed_search_depth =
            ((double)(cache_ptr->total_failed_ht_search_depth)) / ((double)(cache_ptr->failed_ht_searches));

    Rfprintf(Rstdout, "\n%sH5C: cache statistics for %s\n", cache_ptr->prefix, cache_name);

    Rfprintf(Rstdout, "\n");

    Rfprintf(Rstdout, "%s  hash table insertion / deletions   = %ld / %ld\n", cache_ptr->prefix,
            (long)(cache_ptr->total_ht_insertions), (long)(cache_ptr->total_ht_deletions));

    Rfprintf(Rstdout, "%s  HT successful / failed searches    = %ld / %ld\n", cache_ptr->prefix,
            (long)(cache_ptr->successful_ht_searches), (long)(cache_ptr->failed_ht_searches));

    Rfprintf(Rstdout, "%s  Av. HT suc / failed search depth   = %f / %f\n", cache_ptr->prefix,
            average_successful_search_depth, average_failed_search_depth);

    Rfprintf(Rstdout, "%s  current (max) index size / length  = %ld (%ld) / %lu (%lu)\n", cache_ptr->prefix,
            (long)(cache_ptr->index_size), (long)(cache_ptr->max_index_size),
            (unsigned long)(cache_ptr->index_len), (unsigned long)(cache_ptr->max_index_len));

    Rfprintf(Rstdout, "%s  current (max) clean/dirty idx size = %ld (%ld) / %ld (%ld)\n", cache_ptr->prefix,
            (long)(cache_ptr->clean_index_size), (long)(cache_ptr->max_clean_index_size),
            (long)(cache_ptr->dirty_index_size), (long)(cache_ptr->max_dirty_index_size));

    Rfprintf(Rstdout, "%s  current (max) slist size / length  = %ld (%ld) / %lu (%lu)\n", cache_ptr->prefix,
            (long)(cache_ptr->slist_size), (long)(cache_ptr->max_slist_size),
            (unsigned long)(cache_ptr->slist_len), (unsigned long)(cache_ptr->max_slist_len));

    Rfprintf(Rstdout, "%s  current (max) PL size / length     = %ld (%ld) / %lu (%lu)\n", cache_ptr->prefix,
            (long)(cache_ptr->pl_size), (long)(cache_ptr->max_pl_size), (unsigned long)(cache_ptr->pl_len),
            (unsigned long)(cache_ptr->max_pl_len));

    Rfprintf(Rstdout, "%s  current (max) PEL size / length    = %ld (%ld) / %lu (%lu)\n", cache_ptr->prefix,
            (long)(cache_ptr->pel_size), (long)(cache_ptr->max_pel_size), (unsigned long)(cache_ptr->pel_len),
            (unsigned long)(cache_ptr->max_pel_len));

    Rfprintf(Rstdout, "%s  current LRU list size / length     = %ld / %lu\n", cache_ptr->prefix,
            (long)(cache_ptr->LRU_list_size), (unsigned long)(cache_ptr->LRU_list_len));

#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
    Rfprintf(Rstdout, "%s  current clean LRU size / length    = %ld / %lu\n", cache_ptr->prefix,
            (long)(cache_ptr->cLRU_list_size), (unsigned long)(cache_ptr->cLRU_list_len));

    Rfprintf(Rstdout, "%s  current dirty LRU size / length    = %ld / %lu\n", cache_ptr->prefix,
            (long)(cache_ptr->dLRU_list_size), (unsigned long)(cache_ptr->dLRU_list_len));
#endif 

    Rfprintf(Rstdout, "%s  Total hits / misses / hit_rate     = %ld / %ld / %f\n", cache_ptr->prefix,
            (long)total_hits, (long)total_misses, hit_rate);

    Rfprintf(Rstdout, "%s  Total write / read (max) protects  = %ld / %ld (%ld)\n", cache_ptr->prefix,
            (long)total_write_protects, (long)total_read_protects, (long)max_read_protects);

    Rfprintf(Rstdout, "%s  Total clears / flushes             = %ld / %ld\n", cache_ptr->prefix,
            (long)total_clears, (long)total_flushes);

    Rfprintf(Rstdout, "%s  Total evictions / take ownerships  = %ld / %ld\n", cache_ptr->prefix,
            (long)total_evictions, (long)total_take_ownerships);

    Rfprintf(Rstdout, "%s  Total insertions(pinned) / moves   = %ld(%ld) / %ld\n", cache_ptr->prefix,
            (long)total_insertions, (long)total_pinned_insertions, (long)total_moves);

    Rfprintf(Rstdout, "%s  Total entry / cache flush moves    = %ld / %ld\n", cache_ptr->prefix,
            (long)total_entry_flush_moves, (long)total_cache_flush_moves);

    Rfprintf(Rstdout, "%s  Total entry size incrs / decrs     = %ld / %ld\n", cache_ptr->prefix,
            (long)total_size_increases, (long)total_size_decreases);

    Rfprintf(Rstdout, "%s  Ttl entry/cache flush size changes = %ld / %ld\n", cache_ptr->prefix,
            (long)total_entry_flush_size_changes, (long)total_cache_flush_size_changes);

    Rfprintf(Rstdout, "%s  Total entry pins (dirty) / unpins  = %ld (%ld) / %ld\n", cache_ptr->prefix,
            (long)total_pins, (long)total_dirty_pins, (long)total_unpins);

    Rfprintf(Rstdout, "%s  Total pinned flushes / clears      = %ld / %ld\n", cache_ptr->prefix,
            (long)total_pinned_flushes, (long)total_pinned_clears);

    Rfprintf(Rstdout, "%s  MSIC: (make space in cache) calls  = %lld\n", cache_ptr->prefix,
            (long long)(cache_ptr->calls_to_msic));

    if (cache_ptr->calls_to_msic > 0)
        average_entries_skipped_per_calls_to_msic =
            (((double)(cache_ptr->total_entries_skipped_in_msic)) / ((double)(cache_ptr->calls_to_msic)));

    Rfprintf(Rstdout, "%s  MSIC: Average/max entries skipped  = %lf / %ld\n", cache_ptr->prefix,
            (double)average_entries_skipped_per_calls_to_msic,
            (long)(cache_ptr->max_entries_skipped_in_msic));

    if (cache_ptr->calls_to_msic > 0)
        average_dirty_pf_entries_skipped_per_call_to_msic =
            (((double)(cache_ptr->total_dirty_pf_entries_skipped_in_msic)) /
             ((double)(cache_ptr->calls_to_msic)));

    Rfprintf(Rstdout, "%s  MSIC: Average/max dirty pf entries skipped  = %lf / %ld\n", cache_ptr->prefix,
            average_dirty_pf_entries_skipped_per_call_to_msic,
            (long)(cache_ptr->max_dirty_pf_entries_skipped_in_msic));

    if (cache_ptr->calls_to_msic > 0)
        average_entries_scanned_per_calls_to_msic =
            (((double)(cache_ptr->total_entries_scanned_in_msic)) / ((double)(cache_ptr->calls_to_msic)));

    Rfprintf(Rstdout, "%s  MSIC: Average/max entries scanned  = %lf / %ld\n", cache_ptr->prefix,
            (double)average_entries_scanned_per_calls_to_msic,
            (long)(cache_ptr->max_entries_scanned_in_msic));

    Rfprintf(Rstdout, "%s  MSIC: Scanned to make space(evict) = %lld\n", cache_ptr->prefix,
            (long long)(cache_ptr->entries_scanned_to_make_space));

    Rfprintf(Rstdout, "%s  MSIC: Scanned to satisfy min_clean = %lld\n", cache_ptr->prefix,
            (long long)(cache_ptr->total_entries_scanned_in_msic - cache_ptr->entries_scanned_to_make_space));

    Rfprintf(Rstdout, "%s  slist/LRU/index scan restarts   = %lld / %lld / %lld.\n", cache_ptr->prefix,
            (long long)(cache_ptr->slist_scan_restarts), (long long)(cache_ptr->LRU_scan_restarts),
            (long long)(cache_ptr->index_scan_restarts));

    Rfprintf(Rstdout, "%s  cache image creations/reads/loads/size = %d / %d /%d / %" PRIuHSIZE "\n",
            cache_ptr->prefix, cache_ptr->images_created, cache_ptr->images_read, cache_ptr->images_loaded,
            cache_ptr->last_image_size);

    Rfprintf(Rstdout, "%s  prefetches / dirty prefetches      = %lld / %lld\n", cache_ptr->prefix,
            (long long)(cache_ptr->prefetches), (long long)(cache_ptr->dirty_prefetches));

    Rfprintf(Rstdout, "%s  prefetch hits/flushes/evictions    = %lld / %lld / %lld\n", cache_ptr->prefix,
            (long long)(cache_ptr->prefetch_hits), (long long)(cache_ptr->flushes[H5AC_PREFETCHED_ENTRY_ID]),
            (long long)(cache_ptr->evictions[H5AC_PREFETCHED_ENTRY_ID]));

    if (cache_ptr->prefetches > 0)
        prefetch_use_rate = 100.0 * ((double)(cache_ptr->prefetch_hits)) / ((double)(cache_ptr->prefetches));
    else
        prefetch_use_rate = 0.0;

    Rfprintf(Rstdout, "%s  prefetched entry use rate          = %lf\n", cache_ptr->prefix, prefetch_use_rate);

#if H5C_COLLECT_CACHE_ENTRY_STATS

    Rfprintf(Rstdout, "%s  aggregate max / min accesses       = %d / %d\n", cache_ptr->prefix,
            (int)aggregate_max_accesses, (int)aggregate_min_accesses);

    Rfprintf(Rstdout, "%s  aggregate max_clears / max_flushes = %d / %d\n", cache_ptr->prefix,
            (int)aggregate_max_clears, (int)aggregate_max_flushes);

    Rfprintf(Rstdout, "%s  aggregate max_size / max_pins      = %d / %d\n", cache_ptr->prefix,
            (int)aggregate_max_size, (int)aggregate_max_pins);

#endif 

    if (display_detailed_stats) {
        for (i = 0; i <= cache_ptr->max_type_id; i++) {
            Rfprintf(Rstdout, "\n");

            Rfprintf(Rstdout, "%s  Stats on %s:\n", cache_ptr->prefix, ((cache_ptr->class_table_ptr))[i]->name);

            if ((cache_ptr->hits[i] > 0) || (cache_ptr->misses[i] > 0))
                hit_rate = 100.0 * ((double)(cache_ptr->hits[i])) /
                           ((double)(cache_ptr->hits[i] + cache_ptr->misses[i]));
            else
                hit_rate = 0.0;

            Rfprintf(Rstdout, "%s    hits / misses / hit_rate       = %ld / %ld / %f\n", cache_ptr->prefix,
                    (long)(cache_ptr->hits[i]), (long)(cache_ptr->misses[i]), hit_rate);

            Rfprintf(Rstdout, "%s    write / read (max) protects    = %ld / %ld (%d)\n", cache_ptr->prefix,
                    (long)(cache_ptr->write_protects[i]), (long)(cache_ptr->read_protects[i]),
                    (int)(cache_ptr->max_read_protects[i]));

            Rfprintf(Rstdout, "%s    clears / flushes               = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->clears[i]), (long)(cache_ptr->flushes[i]));

            Rfprintf(Rstdout, "%s    evictions / take ownerships    = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->evictions[i]), (long)(cache_ptr->take_ownerships[i]));

            Rfprintf(Rstdout, "%s    insertions(pinned) / moves     = %ld(%ld) / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->insertions[i]), (long)(cache_ptr->pinned_insertions[i]),
                    (long)(cache_ptr->moves[i]));

            Rfprintf(Rstdout, "%s    entry / cache flush moves      = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->entry_flush_moves[i]), (long)(cache_ptr->cache_flush_moves[i]));

            Rfprintf(Rstdout, "%s    size increases / decreases     = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->size_increases[i]), (long)(cache_ptr->size_decreases[i]));

            Rfprintf(Rstdout, "%s    entry/cache flush size changes = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->entry_flush_size_changes[i]),
                    (long)(cache_ptr->cache_flush_size_changes[i]));

            Rfprintf(Rstdout, "%s    entry pins / unpins            = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->pins[i]), (long)(cache_ptr->unpins[i]));

            Rfprintf(Rstdout, "%s    entry dirty pins/pin'd flushes = %ld / %ld\n", cache_ptr->prefix,
                    (long)(cache_ptr->dirty_pins[i]), (long)(cache_ptr->pinned_flushes[i]));

#if H5C_COLLECT_CACHE_ENTRY_STATS

            Rfprintf(Rstdout, "%s    entry max / min accesses       = %d / %d\n", cache_ptr->prefix,
                    cache_ptr->max_accesses[i], cache_ptr->min_accesses[i]);

            Rfprintf(Rstdout, "%s    entry max_clears / max_flushes = %d / %d\n", cache_ptr->prefix,
                    cache_ptr->max_clears[i], cache_ptr->max_flushes[i]);

            Rfprintf(Rstdout, "%s    entry max_size / max_pins      = %d / %d\n", cache_ptr->prefix,
                    (int)(cache_ptr->max_size[i]), (int)(cache_ptr->max_pins[i]));

#endif 

        } 
    }     

    Rfprintf(Rstdout, "\n");

#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
#ifndef NDEBUG
H5C_stats__reset(H5C_t *cache_ptr)
#else 
#if H5C_COLLECT_CACHE_STATS
H5C_stats__reset(H5C_t *cache_ptr)
#else  
H5C_stats__reset(H5C_t H5_ATTR_UNUSED *cache_ptr)
#endif 
#endif 
{
#if H5C_COLLECT_CACHE_STATS
    int i;
#endif 

    assert(cache_ptr);

#if H5C_COLLECT_CACHE_STATS
    for (i = 0; i <= cache_ptr->max_type_id; i++) {
        cache_ptr->hits[i]                     = 0;
        cache_ptr->misses[i]                   = 0;
        cache_ptr->write_protects[i]           = 0;
        cache_ptr->read_protects[i]            = 0;
        cache_ptr->max_read_protects[i]        = 0;
        cache_ptr->insertions[i]               = 0;
        cache_ptr->pinned_insertions[i]        = 0;
        cache_ptr->clears[i]                   = 0;
        cache_ptr->flushes[i]                  = 0;
        cache_ptr->evictions[i]                = 0;
        cache_ptr->take_ownerships[i]          = 0;
        cache_ptr->moves[i]                    = 0;
        cache_ptr->entry_flush_moves[i]        = 0;
        cache_ptr->cache_flush_moves[i]        = 0;
        cache_ptr->pins[i]                     = 0;
        cache_ptr->unpins[i]                   = 0;
        cache_ptr->dirty_pins[i]               = 0;
        cache_ptr->pinned_flushes[i]           = 0;
        cache_ptr->pinned_clears[i]            = 0;
        cache_ptr->size_increases[i]           = 0;
        cache_ptr->size_decreases[i]           = 0;
        cache_ptr->entry_flush_size_changes[i] = 0;
        cache_ptr->cache_flush_size_changes[i] = 0;
    } 

    cache_ptr->total_ht_insertions              = 0;
    cache_ptr->total_ht_deletions               = 0;
    cache_ptr->successful_ht_searches           = 0;
    cache_ptr->total_successful_ht_search_depth = 0;
    cache_ptr->failed_ht_searches               = 0;
    cache_ptr->total_failed_ht_search_depth     = 0;

    cache_ptr->max_index_len        = 0;
    cache_ptr->max_index_size       = (size_t)0;
    cache_ptr->max_clean_index_size = (size_t)0;
    cache_ptr->max_dirty_index_size = (size_t)0;

    cache_ptr->max_slist_len  = 0;
    cache_ptr->max_slist_size = (size_t)0;

    cache_ptr->max_pl_len  = 0;
    cache_ptr->max_pl_size = (size_t)0;

    cache_ptr->max_pel_len  = 0;
    cache_ptr->max_pel_size = (size_t)0;

    cache_ptr->calls_to_msic                          = 0;
    cache_ptr->total_entries_skipped_in_msic          = 0;
    cache_ptr->total_dirty_pf_entries_skipped_in_msic = 0;
    cache_ptr->total_entries_scanned_in_msic          = 0;
    cache_ptr->max_entries_skipped_in_msic            = 0;
    cache_ptr->max_dirty_pf_entries_skipped_in_msic   = 0;
    cache_ptr->max_entries_scanned_in_msic            = 0;
    cache_ptr->entries_scanned_to_make_space          = 0;

    cache_ptr->slist_scan_restarts = 0;
    cache_ptr->LRU_scan_restarts   = 0;
    cache_ptr->index_scan_restarts = 0;

    cache_ptr->images_created  = 0;
    cache_ptr->images_read     = 0;
    cache_ptr->images_loaded   = 0;
    cache_ptr->last_image_size = (hsize_t)0;

    cache_ptr->prefetches       = 0;
    cache_ptr->dirty_prefetches = 0;
    cache_ptr->prefetch_hits    = 0;

#if H5C_COLLECT_CACHE_ENTRY_STATS
    for (i = 0; i <= cache_ptr->max_type_id; i++) {
        cache_ptr->max_accesses[i] = 0;
        cache_ptr->min_accesses[i] = 1000000;
        cache_ptr->max_clears[i]   = 0;
        cache_ptr->max_flushes[i]  = 0;
        cache_ptr->max_size[i]     = (size_t)0;
        cache_ptr->max_pins[i]     = 0;
    } 

#endif 
#endif 
} 

#ifndef NDEBUG
herr_t
H5C_flush_dependency_exists(H5C_t *cache_ptr, haddr_t parent_addr, haddr_t child_addr, bool *fd_exists_ptr)
{
    bool               fd_exists = false; 
    H5C_cache_entry_t *parent_ptr;        
    H5C_cache_entry_t *child_ptr;         
    bool               ret_value = false; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(cache_ptr);
    assert(H5_addr_defined(parent_addr));
    assert(H5_addr_defined(child_addr));
    assert(fd_exists_ptr);

    H5C__SEARCH_INDEX(cache_ptr, parent_addr, parent_ptr, FAIL);
    H5C__SEARCH_INDEX(cache_ptr, child_addr, child_ptr, FAIL);

    if (parent_ptr && child_ptr) {
        if (child_ptr->flush_dep_nparents > 0) {
            unsigned u; 

            assert(child_ptr->flush_dep_parent);
            assert(child_ptr->flush_dep_parent_nalloc >= child_ptr->flush_dep_nparents);

            for (u = 0; u < child_ptr->flush_dep_nparents; u++) {
                if (child_ptr->flush_dep_parent[u] == parent_ptr) {
                    fd_exists = true;
                    assert(parent_ptr->flush_dep_nchildren > 0);
                    break;
                } 
            }     
        }         
    }             

    *fd_exists_ptr = fd_exists;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5C_validate_index_list(H5C_t *cache_ptr)
{
    H5C_cache_entry_t *entry_ptr = NULL;
    uint32_t           len       = 0;
    int32_t            index_ring_len[H5C_RING_NTYPES];
    size_t             size       = 0;
    size_t             clean_size = 0;
    size_t             dirty_size = 0;
    size_t             index_ring_size[H5C_RING_NTYPES];
    size_t             clean_index_ring_size[H5C_RING_NTYPES];
    size_t             dirty_index_ring_size[H5C_RING_NTYPES];
    int                i;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(cache_ptr);

    for (i = 0; i < H5C_RING_NTYPES; i++) {
        index_ring_len[i]        = 0;
        index_ring_size[i]       = 0;
        clean_index_ring_size[i] = 0;
        dirty_index_ring_size[i] = 0;
    } 

    if (((cache_ptr->il_head == NULL) || (cache_ptr->il_tail == NULL)) &&
        (cache_ptr->il_head != cache_ptr->il_tail))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index list pointer validation failed");

    if ((cache_ptr->index_len == 1) &&
        ((cache_ptr->il_head != cache_ptr->il_tail) || (cache_ptr->il_head == NULL) ||
         (cache_ptr->il_head->size != cache_ptr->index_size)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index list pointer sanity checks failed");

    if ((cache_ptr->index_len >= 1) &&
        ((cache_ptr->il_head == NULL) || (cache_ptr->il_head->il_prev != NULL) ||
         (cache_ptr->il_tail == NULL) || (cache_ptr->il_tail->il_next != NULL)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index list length sanity checks failed");

    entry_ptr = cache_ptr->il_head;
    while (entry_ptr != NULL) {
        if ((entry_ptr != cache_ptr->il_head) &&
            ((entry_ptr->il_prev == NULL) || (entry_ptr->il_prev->il_next != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index list pointers for entry are invalid");

        if ((entry_ptr != cache_ptr->il_tail) &&
            ((entry_ptr->il_next == NULL) || (entry_ptr->il_next->il_prev != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index list pointers for entry are invalid");

        assert(entry_ptr->ring > 0);
        assert(entry_ptr->ring < H5C_RING_NTYPES);

        len++;
        index_ring_len[entry_ptr->ring] += 1;

        size += entry_ptr->size;
        index_ring_size[entry_ptr->ring] += entry_ptr->size;

        if (entry_ptr->is_dirty) {
            dirty_size += entry_ptr->size;
            dirty_index_ring_size[entry_ptr->ring] += entry_ptr->size;
        } 
        else {
            clean_size += entry_ptr->size;
            clean_index_ring_size[entry_ptr->ring] += entry_ptr->size;
        } 

        entry_ptr = entry_ptr->il_next;
    } 

    if ((cache_ptr->index_len != len) || (cache_ptr->il_len != len) || (cache_ptr->index_size != size) ||
        (cache_ptr->il_size != size) || (cache_ptr->clean_index_size != clean_size) ||
        (cache_ptr->dirty_index_size != dirty_size) || (clean_size + dirty_size != size))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index, clean and dirty sizes for cache are invalid");

    size       = 0;
    clean_size = 0;
    dirty_size = 0;
    for (i = 0; i < H5C_RING_NTYPES; i++) {
        size += clean_index_ring_size[i] + dirty_index_ring_size[i];
        clean_size += clean_index_ring_size[i];
        dirty_size += dirty_index_ring_size[i];
    } 

    if ((cache_ptr->index_size != size) || (cache_ptr->clean_index_size != clean_size) ||
        (cache_ptr->dirty_index_size != dirty_size))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Index, clean and dirty sizes for cache are invalid");

done:
    if (ret_value != SUCCEED)
        assert(0);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5C_get_entry_ptr_from_addr(H5C_t *cache_ptr, haddr_t addr, void **entry_ptr_ptr)
{
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cache_ptr);
    assert(H5_addr_defined(addr));
    assert(entry_ptr_ptr);

    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, FAIL);

    if (entry_ptr == NULL)
        
        *entry_ptr_ptr = NULL;
    else {
        *entry_ptr_ptr = entry_ptr;

        
        (cache_ptr->get_entry_ptr_from_addr_counter)++;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
bool
H5C_get_serialization_in_progress(const H5C_t *cache_ptr)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(cache_ptr);

    FUNC_LEAVE_NOAPI(cache_ptr->serialization_in_progress)
} 
#endif 

#ifndef NDEBUG
bool
H5C_cache_is_clean(const H5C_t *cache_ptr, H5C_ring_t inner_ring)
{
    H5C_ring_t ring      = H5C_RING_USER;
    bool       ret_value = true; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(cache_ptr);
    assert(inner_ring >= H5C_RING_USER);
    assert(inner_ring <= H5C_RING_SB);

    while (ring <= inner_ring) {
        if (cache_ptr->dirty_index_ring_size[ring] > 0)
            HGOTO_DONE(false);

        ring++;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifndef NDEBUG
herr_t
H5C_verify_entry_type(H5C_t *cache_ptr, haddr_t addr, const H5C_class_t *expected_type, bool *in_cache_ptr,
                      bool *type_ok_ptr)
{
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cache_ptr);
    assert(H5_addr_defined(addr));
    assert(expected_type);
    assert(in_cache_ptr);
    assert(type_ok_ptr);

    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, FAIL);

    if (entry_ptr == NULL)
        
        *in_cache_ptr = false;
    else {
        *in_cache_ptr = true;

        if (entry_ptr->prefetched)
            *type_ok_ptr = (expected_type->id == entry_ptr->prefetch_type_id);
        else
            *type_ok_ptr = (expected_type == entry_ptr->type);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

void
H5C_def_auto_resize_rpt_fcn(H5C_t *cache_ptr,
#ifndef NDEBUG
                            int32_t version,
#else
                            int32_t H5_ATTR_UNUSED version,
#endif
                            double hit_rate, enum H5C_resize_status status, size_t old_max_cache_size,
                            size_t new_max_cache_size, size_t old_min_clean_size, size_t new_min_clean_size)
{
    assert(cache_ptr != NULL);
    assert(version == H5C__CURR_AUTO_RESIZE_RPT_FCN_VER);

    switch (status) {
        case in_spec:
            Rfprintf(Rstdout, "%sAuto cache resize -- no change. (hit rate = %lf)\n", cache_ptr->prefix,
                    hit_rate);
            break;

        case increase:
            assert(hit_rate < cache_ptr->resize_ctl.lower_hr_threshold);
            assert(old_max_cache_size < new_max_cache_size);

            Rfprintf(Rstdout, "%sAuto cache resize -- hit rate (%lf) out of bounds low (%6.5lf).\n",
                    cache_ptr->prefix, hit_rate, cache_ptr->resize_ctl.lower_hr_threshold);
            Rfprintf(Rstdout, "%scache size increased from (%llu/%llu) to (%llu/%llu).\n", cache_ptr->prefix,
                    (unsigned long long)old_max_cache_size, (unsigned long long)old_min_clean_size, 
                    (unsigned long long)new_max_cache_size, (unsigned long long)new_min_clean_size);
            break;

        case flash_increase:
            assert(old_max_cache_size < new_max_cache_size);

            Rfprintf(Rstdout, "%sflash cache resize(%d) -- size threshold = %llu.\n", cache_ptr->prefix,
                    (int)(cache_ptr->resize_ctl.flash_incr_mode), 
                    (unsigned long long)(cache_ptr->flash_size_increase_threshold));
            Rfprintf(Rstdout, "%s cache size increased from (%llu/%llu) to (%llu/%llu).\n", cache_ptr->prefix,
                    (unsigned long long)old_max_cache_size, (unsigned long long)old_min_clean_size, 
                    (unsigned long long)new_max_cache_size, (unsigned long long)new_min_clean_size);
            break;

        case decrease:
            assert(old_max_cache_size > new_max_cache_size);

            switch (cache_ptr->resize_ctl.decr_mode) {
                case H5C_decr__off:
                    Rfprintf(Rstdout, "%sAuto cache resize -- decrease off.  HR = %lf\n", cache_ptr->prefix,
                            hit_rate);
                    break;

                case H5C_decr__threshold:
                    assert(hit_rate > cache_ptr->resize_ctl.upper_hr_threshold);

                    Rfprintf(Rstdout, "%sAuto cache resize -- decrease by threshold.  HR = %lf > %6.5lf\n",
                            cache_ptr->prefix, hit_rate, cache_ptr->resize_ctl.upper_hr_threshold);
                    Rfprintf(Rstdout, "%sout of bounds high (%6.5lf).\n", cache_ptr->prefix,
                            cache_ptr->resize_ctl.upper_hr_threshold);
                    break;

                case H5C_decr__age_out:
                    Rfprintf(Rstdout, "%sAuto cache resize -- decrease by ageout.  HR = %lf\n",
                            cache_ptr->prefix, hit_rate);
                    break;

                case H5C_decr__age_out_with_threshold:
                    assert(hit_rate > cache_ptr->resize_ctl.upper_hr_threshold);

                    Rfprintf(Rstdout,
                            "%sAuto cache resize -- decrease by ageout with threshold. HR = %lf > %6.5lf\n",
                            cache_ptr->prefix, hit_rate, cache_ptr->resize_ctl.upper_hr_threshold);
                    break;

                default:
                    Rfprintf(Rstdout, "%sAuto cache resize -- decrease by unknown mode.  HR = %lf\n",
                            cache_ptr->prefix, hit_rate);
            }

            Rfprintf(Rstdout, "%s    cache size decreased from (%llu/%llu) to (%llu/%llu).\n", cache_ptr->prefix,
                    (unsigned long long)old_max_cache_size, (unsigned long long)old_min_clean_size, 
                    (unsigned long long)new_max_cache_size, (unsigned long long)new_min_clean_size);
            break;

        case at_max_size:
            Rfprintf(Rstdout, "%sAuto cache resize -- hit rate (%lf) out of bounds low (%6.5lf).\n",
                    cache_ptr->prefix, hit_rate, cache_ptr->resize_ctl.lower_hr_threshold);
            Rfprintf(Rstdout, "%s    cache already at maximum size so no change.\n", cache_ptr->prefix);
            break;

        case at_min_size:
            Rfprintf(Rstdout, "%sAuto cache resize -- hit rate (%lf) -- can't decrease.\n", cache_ptr->prefix,
                    hit_rate);
            Rfprintf(Rstdout, "%s    cache already at minimum size.\n", cache_ptr->prefix);
            break;

        case increase_disabled:
            Rfprintf(Rstdout, "%sAuto cache resize -- increase disabled -- HR = %lf.", cache_ptr->prefix,
                    hit_rate);
            break;

        case decrease_disabled:
            Rfprintf(Rstdout, "%sAuto cache resize -- decrease disabled -- HR = %lf.\n", cache_ptr->prefix,
                    hit_rate);
            break;

        case not_full:
            assert(hit_rate < cache_ptr->resize_ctl.lower_hr_threshold);

            Rfprintf(Rstdout, "%sAuto cache resize -- hit rate (%lf) out of bounds low (%6.5lf).\n",
                    cache_ptr->prefix, hit_rate, cache_ptr->resize_ctl.lower_hr_threshold);
            Rfprintf(Rstdout, "%s    cache not full so no increase in size.\n", cache_ptr->prefix);
            break;

        default:
            Rfprintf(Rstdout, "%sAuto cache resize -- unknown status code.\n", cache_ptr->prefix);
            break;
    }
} 

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
herr_t
H5C__validate_lru_list(H5C_t *cache_ptr)
{
    int32_t            len       = 0;
    size_t             size      = 0;
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (((cache_ptr->LRU_head_ptr == NULL) || (cache_ptr->LRU_tail_ptr == NULL)) &&
        (cache_ptr->LRU_head_ptr != cache_ptr->LRU_tail_ptr))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU list head/tail check failed");

    if ((cache_ptr->LRU_list_len == 1) &&
        ((cache_ptr->LRU_head_ptr != cache_ptr->LRU_tail_ptr) || (cache_ptr->LRU_head_ptr == NULL) ||
         (cache_ptr->LRU_head_ptr->size != cache_ptr->LRU_list_size)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU list sanity check failed");

    if ((cache_ptr->LRU_list_len >= 1) &&
        ((cache_ptr->LRU_head_ptr == NULL) || (cache_ptr->LRU_head_ptr->prev != NULL) ||
         (cache_ptr->LRU_tail_ptr == NULL) || (cache_ptr->LRU_tail_ptr->next != NULL)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU list sanity check failed");

    entry_ptr = cache_ptr->LRU_head_ptr;
    while (entry_ptr != NULL) {
        if ((entry_ptr != cache_ptr->LRU_head_ptr) &&
            ((entry_ptr->prev == NULL) || (entry_ptr->prev->next != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if ((entry_ptr != cache_ptr->LRU_tail_ptr) &&
            ((entry_ptr->next == NULL) || (entry_ptr->next->prev != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if (entry_ptr->is_pinned || entry_ptr->pinned_from_client || entry_ptr->pinned_from_cache)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "invalid entry 'pin origin' fields");

        len++;
        size += entry_ptr->size;
        entry_ptr = entry_ptr->next;
    }

    if ((cache_ptr->LRU_list_len != (uint32_t)len) || (cache_ptr->LRU_list_size != size))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "LRU list length/size check failed");

done:
    if (ret_value != SUCCEED)
        assert(0);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
herr_t
H5C__validate_pinned_entry_list(H5C_t *cache_ptr)
{
    int32_t            len       = 0;
    size_t             size      = 0;
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (((cache_ptr->pel_head_ptr == NULL) || (cache_ptr->pel_tail_ptr == NULL)) &&
        (cache_ptr->pel_head_ptr != cache_ptr->pel_tail_ptr))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "pinned list head/tail check failed");

    if ((cache_ptr->pel_len == 1) &&
        ((cache_ptr->pel_head_ptr != cache_ptr->pel_tail_ptr) || (cache_ptr->pel_head_ptr == NULL) ||
         (cache_ptr->pel_head_ptr->size != cache_ptr->pel_size)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "pinned list sanity check failed");

    if ((cache_ptr->pel_len >= 1) &&
        ((cache_ptr->pel_head_ptr == NULL) || (cache_ptr->pel_head_ptr->prev != NULL) ||
         (cache_ptr->pel_tail_ptr == NULL) || (cache_ptr->pel_tail_ptr->next != NULL)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "pinned list sanity check failed");

    entry_ptr = cache_ptr->pel_head_ptr;
    while (entry_ptr != NULL) {
        if ((entry_ptr != cache_ptr->pel_head_ptr) &&
            ((entry_ptr->prev == NULL) || (entry_ptr->prev->next != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if ((entry_ptr != cache_ptr->pel_tail_ptr) &&
            ((entry_ptr->next == NULL) || (entry_ptr->next->prev != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if (!entry_ptr->is_pinned)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "pinned list contains unpinned entry");

        if (!(entry_ptr->pinned_from_client || entry_ptr->pinned_from_cache))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "invalid entry 'pin origin' fields");

        len++;
        size += entry_ptr->size;
        entry_ptr = entry_ptr->next;
    }

    if ((cache_ptr->pel_len != (uint32_t)len) || (cache_ptr->pel_size != size))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "pinned list length/size check failed");

done:
    if (ret_value != SUCCEED)
        assert(0);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
herr_t
H5C__validate_protected_entry_list(H5C_t *cache_ptr)
{
    int32_t            len       = 0;
    size_t             size      = 0;
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (((cache_ptr->pl_head_ptr == NULL) || (cache_ptr->pl_tail_ptr == NULL)) &&
        (cache_ptr->pl_head_ptr != cache_ptr->pl_tail_ptr))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "protected list head/tail check failed");

    if ((cache_ptr->pl_len == 1) &&
        ((cache_ptr->pl_head_ptr != cache_ptr->pl_tail_ptr) || (cache_ptr->pl_head_ptr == NULL) ||
         (cache_ptr->pl_head_ptr->size != cache_ptr->pl_size)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "protected list sanity check failed");

    if ((cache_ptr->pl_len >= 1) &&
        ((cache_ptr->pl_head_ptr == NULL) || (cache_ptr->pl_head_ptr->prev != NULL) ||
         (cache_ptr->pl_tail_ptr == NULL) || (cache_ptr->pl_tail_ptr->next != NULL)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "protected list sanity check failed");

    entry_ptr = cache_ptr->pl_head_ptr;
    while (entry_ptr != NULL) {
        if ((entry_ptr != cache_ptr->pl_head_ptr) &&
            ((entry_ptr->prev == NULL) || (entry_ptr->prev->next != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if ((entry_ptr != cache_ptr->pl_tail_ptr) &&
            ((entry_ptr->next == NULL) || (entry_ptr->next->prev != entry_ptr)))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "entry has bad prev/next pointers");

        if (!entry_ptr->is_protected)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "protected list contains unprotected entry");

        if (entry_ptr->is_read_only && (entry_ptr->ro_ref_count <= 0))
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "read-only entry has non-positive ref count");

        len++;
        size += entry_ptr->size;
        entry_ptr = entry_ptr->next;
    }

    if ((cache_ptr->pl_len != (uint32_t)len) || (cache_ptr->pl_size != size))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "protected list length/size check failed");

done:
    if (ret_value != SUCCEED)
        assert(0);

    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

#ifdef H5C_DO_SLIST_SANITY_CHECKS
bool
H5C__entry_in_skip_list(H5C_t *cache_ptr, H5C_cache_entry_t *target_ptr)
{
    H5SL_node_t *node_ptr;
    bool         in_slist;
    bool         ret_value;

    FUNC_ENTER_PACKAGE

    
    assert(cache_ptr);
    assert(cache_ptr->slist_ptr);

    node_ptr = H5SL_first(cache_ptr->slist_ptr);
    in_slist = false;
    while ((node_ptr != NULL) && (!in_slist)) {
        H5C_cache_entry_t *entry_ptr;

        entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);

        assert(entry_ptr);
        assert(entry_ptr->is_dirty);
        assert(entry_ptr->in_slist);

        if (entry_ptr == target_ptr)
            in_slist = true;
        else
            node_ptr = H5SL_next(node_ptr);
    }

    
    ret_value = in_slist;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
#if H5C_COLLECT_CACHE_STATS
H5C__image_stats(H5C_t *cache_ptr, bool print_header)
#else  
H5C__image_stats(H5C_t *cache_ptr, bool H5_ATTR_UNUSED print_header)
#endif 
{
#if H5C_COLLECT_CACHE_STATS
    int     i;
    int64_t total_hits   = 0;
    int64_t total_misses = 0;
    double  hit_rate;
    double  prefetch_use_rate;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (NULL == cache_ptr)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr");

#if H5C_COLLECT_CACHE_STATS
    for (i = 0; i <= cache_ptr->max_type_id; i++) {
        total_hits += cache_ptr->hits[i];
        total_misses += cache_ptr->misses[i];
    } 

    if ((total_hits > 0) || (total_misses > 0))
        hit_rate = 100.0 * ((double)(total_hits)) / ((double)(total_hits + total_misses));
    else
        hit_rate = 0.0;

    if (cache_ptr->prefetches > 0)
        prefetch_use_rate = 100.0 * ((double)(cache_ptr->prefetch_hits)) / ((double)(cache_ptr->prefetches));
    else
        prefetch_use_rate = 0.0;

    if (print_header) {
        Rfprintf(Rstdout, "\nhit     prefetches      prefetch              image  pf hit\n");
        Rfprintf(Rstdout, "rate:   total:  dirty:  hits:  flshs:  evct:  size:  rate:\n");
    } 

    Rfprintf(Rstdout, "%3.1lf    %5lld   %5lld   %5lld  %5lld   %5lld   %5lld   %3.1lf\n", hit_rate,
            (long long)(cache_ptr->prefetches), (long long)(cache_ptr->dirty_prefetches),
            (long long)(cache_ptr->prefetch_hits), (long long)(cache_ptr->flushes[H5AC_PREFETCHED_ENTRY_ID]),
            (long long)(cache_ptr->evictions[H5AC_PREFETCHED_ENTRY_ID]),
            (long long)(cache_ptr->last_image_size), prefetch_use_rate);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
