#' @title Function to solve for a group sequential critical value at a given stage
#' @description Computes the critical boundary value at the current analysis stage by solving a probability
#' equation under a multivariate normal distribution. The boundary is chosen so that the incremental Type I error spent
#' at the current stage equals a specified target value, conditional on previously determined boundaries.
#'
#' @param previous.c Numeric value giving the critical boundary from the previous analysis stage.
#' @param pi Target incremental Type I error to be spent at the current stage.
#' @param corr Correlation matrix of the joint multivariate normal distribution of test statistics across analyses.
#' @importFrom mvtnorm pmvnorm
#' @importFrom stats uniroot
#'
#' @return A numeric value giving the critical boundary \code{c} at the current analysis stage that satisfies the
#' specified Type I spending constraint.
#' @export
#'
find.c <- function(previous.c, pi, corr){

  g <- function(c){
    prob1 = pmvnorm(lower = c(-Inf, -previous.c), upper = c(-c, previous.c), mean = rep(0, length = dim(corr)[1]),
                    corr = corr)
    prob2 = pmvnorm(lower = c(c, -previous.c), upper = c(Inf, previous.c), mean = rep(0, length = dim(corr)[1]),
                    corr = corr)
    p = prob1[1] + prob2[1]
    return(p - pi)
  }
  solution = uniroot(g, c(0, 10))$root
  return(solution)
}
