# Note: Full report generation requires LaTeX for PDF, so we test components
# rather than full rendering to avoid CRAN issues

test_that("generate_audit_report validates inputs", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  expect_error(generate_audit_report("not a data frame"))
  expect_error(generate_audit_report(data, output_format = "invalid"))
})

test_that("generate_audit_report pipeline components work", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000),
    gender = sample(c("M", "F"), 100, replace = TRUE)
  )
  
  # Test that scoring works (core of report generation)
  scored_data <- score_anomaly(data, method = "iforest", contamination = 0.05)
  flagged_data <- flag_top_anomalies(scored_data, contamination = 0.05)
  
  expect_true("anomaly_score" %in% names(scored_data))
  expect_true("is_anomaly" %in% names(flagged_data))
  
  # Test metadata extraction
  metadata <- attr(scored_data, "metadata")
  expect_false(is.null(metadata))
  expect_true("numeric_cols" %in% names(metadata))
})

test_that("generate_audit_report handles ground truth", {
  set.seed(123)
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000),
    is_error = sample(c(0, 1), 100, replace = TRUE, prob = c(0.95, 0.05))
  )
  
  # Test that benchmarking works in the pipeline
  scored_data <- score_anomaly(data, ground_truth_col = "is_error")
  metrics <- attr(scored_data, "benchmark_metrics")
  
  expect_false(is.null(metrics))
})

test_that("generate_audit_report handles different methods", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  # Test iforest
  scored_if <- score_anomaly(data, method = "iforest")
  expect_true("anomaly_score" %in% names(scored_if))
  
  # Test lof
  scored_lof <- score_anomaly(data, method = "lof")
  expect_true("anomaly_score" %in% names(scored_lof))
})

test_that("generate_audit_report handles edge cases", {
  # Small dataset
  data_small <- data.frame(
    patient_id = 1:10,
    age = rnorm(10, 50, 15)
  )
  
  scored_small <- score_anomaly(data_small)
  expect_true(is.data.frame(scored_small))
  
  # All numeric
  data_numeric <- data.frame(
    patient_id = 1:50,
    age = rnorm(50, 50, 15),
    cost = rnorm(50, 10000, 5000),
    length_stay = rpois(50, 5)
  )
  
  scored_numeric <- score_anomaly(data_numeric)
  expect_true("anomaly_score" %in% names(scored_numeric))
})

