#' Read and Process Universal PAM Data
#'
#' Reads a standardized CSV file containing PAR and yield data for photosystem I and/or II, calculates electron transport rates (ETR), and returns a cleaned and validated dataset. The function is device-agnostic but requires a predefined column structure.
#'
#' @param csv_path File path to the CSV file.
#' @param etr_factor Numeric. Factor for ETR calculation. Default is \code{0.84}.
#' @param fraction_photosystem_I Numeric. Relative distribution of absorbed PAR
#' to photosystem I. Default is \code{0.5}.
#' @param fraction_photosystem_II Numeric. Relative distribution of absorbed PAR
#' to photosystem II. Default is \code{0.5}.
#'
#' @details
#' Calculates ETR using:
#' \deqn{\text{ETR} = \text{PAR} \cdot \text{ETR-Factor} \cdot \text{Fraction of Photosystem (I or II)} \cdot \text{Yield (I or II)}}
#'
#' A detailed documentation can be found under \url{https://github.com/biotoolbox/pam?tab=readme-ov-file#read_universal_data}
#'
#' @return A \code{data.table} containing:
#' \itemize{
#'   \item \code{par}: Photosynthetically active radiation.
#'   \item \code{yield_1}: Yield for photosystem I.
#'   \item \code{yield_2}: Yield for photosystem II.
#'   \item \code{etr_1}: Calculated ETR for photosystem I.
#'   \item \code{etr_2}: Calculated ETR for photosystem II.
#' }
#'
#' @references{
#'   Heinz Walz GmbH. (2024). \emph{DUAL-PAM-100 DUAL-PAM/F MANUAL, 5th Edition, April 2024, Chapter 7 (pp. 162-172).}
#'   Heinz Walz GmbH, Effeltrich, Germany.
#'   Available at: \url{https://www.walz.com/files/downloads/dualpamed05.pdf}
#' }
#' @examples
#' path <- file.path(system.file("extdata", package = "pam"), "universal_data", "universal_data.csv")
#' data <- read_universal_data(path)
#' @export
read_universal_data <- function(csv_path,
                                etr_factor = 0.84,
                                fraction_photosystem_I = 0.5,
                                fraction_photosystem_II = 0.5) {
  if (fraction_photosystem_I + fraction_photosystem_II != 1) {
    stop("The sum of fraction_photosystem_I and fraction_photosystem_II must be equal 1.")
  }

  data <- utils::read.csv(csv_path, sep = ";", dec = ".")
  data <- data.table::as.data.table(data)
  validate_raw_intermediate_csv(data)

  result <- data.table::data.table(
    par = numeric(),
    yield_1 = numeric(),
    yield_2 = numeric(),
    etr_1 = numeric(),
    etr_2 = numeric()
  )
  for (i in seq_len(nrow(data))) {
    row <- data[i, ]
    current_par <- row$par

    yield_1 <- row$yield_1
    recalc_etr_1 <- calc_etr(yield_1, current_par, etr_factor, fraction_photosystem_I)

    yield_2 <- row$yield_2
    recalc_etr_2 <- calc_etr(yield_2, current_par, etr_factor, fraction_photosystem_II)

    new_row <- list(
      par = current_par,
      yield_1 = yield_1,
      yield_2 = yield_2,
      etr_1 = recalc_etr_1,
      etr_2 = recalc_etr_2
    )
    result <- rbind(result, new_row)
  }

  validate_data(result)
  return(result)
}

#' Read and Process DualPAM Data
#'
#' Reads raw CSV files generated by DualPAM software, calculates electron transport rate (ETR) values, and returns a universal dataset.
#'
#' @param csv_path File path to the CSV file.
#' @param remove_recovery Logical. Removes recovery measurements if \code{TRUE}. Default is \code{TRUE}.
#' @param etr_factor Numeric. Factor for ETR calculation. Default is \code{0.84}.
#' @param fraction_photosystem_I Numeric. Relative distribution of absorbed PAR to photosystem I. Default is \code{0.5}.
#' @param fraction_photosystem_II Numeric. Relative distribution of absorbed PAR to photosystem II. Default is \code{0.5}.
#'
#' @details
#' Calculates ETR using:
#' \deqn{\text{ETR} = \text{PAR} \cdot \text{ETR-Factor} \cdot \text{Fraction of Photosystem (I or II)} \cdot \text{Yield (I or II)}}
#'
#' A detailed documentation can be found under \url{https://github.com/biotoolbox/pam?tab=readme-ov-file#read_dual_pam_data}
#'
#' @return A \code{data.table} containing:
#' \itemize{
#'   \item \code{par}: Photosynthetically active radiation.
#'   \item \code{yield_1}: Yield for photosystem I.
#'   \item \code{yield_2}: Yield for photosystem II.
#'   \item \code{etr_1}: Calculated ETR for photosystem I.
#'   \item \code{etr_2}: Calculated ETR for photosystem II.
#' }
#'
#' @references{
#'   Heinz Walz GmbH. (2024). \emph{DUAL-PAM-100 DUAL-PAM/F MANUAL, 5th Edition, April 2024, Chapter 7 (pp. 162-172).}
#'   Heinz Walz GmbH, Effeltrich, Germany.
#'   Available at: \url{https://www.walz.com/files/downloads/dualpamed05.pdf}
#' }
#' @examples
#' path <- file.path(system.file("extdata", package = "pam"), "20240925.csv")
#' data <- read_dual_pam_data(path)
#' @export
read_dual_pam_data <- function(
    csv_path,
    remove_recovery = TRUE,
    etr_factor = 0.84,
    fraction_photosystem_I = 0.5,
    fraction_photosystem_II = 0.5) {
  if (fraction_photosystem_I + fraction_photosystem_II != 1) {
    stop("The sum of fraction_photosystem_I and fraction_photosystem_II must be equal 1.")
  }

  tryCatch(
    {
      data <- utils::read.csv(csv_path, sep = ";", dec = ".")
      data <- data.table::as.data.table(data)

      validate_dual_pam_data(data)
      data <- data[data$ID == "SP", ]

      date_time_col_values <- c()
      for (i in seq_len(nrow(data))) {
        row <- data[i, ]

        date_time_row_value <- as.POSIXct(
          paste(row$Date, row$Time, sep = " "),
          tz = "GMT", "%d.%m.%y %H:%M:%S"
        )
        date_time_col_values <- c(date_time_col_values, date_time_row_value)
      }

      data$DateTime <- date_time_col_values
      data <- data[order(data$DateTime), ]

      pm_det_row <- pm_det_row <- subset(data, data$PAR == 0 & data$Action == "Pm.-Det.")
      yield_1_first <- pm_det_row$Y.I.
      recalc_etr_1 <- calc_etr(yield_1_first, 0, etr_factor, fraction_photosystem_I)

      fm_det_row <- subset(data, data$PAR == 0 & data$Action == "Fm-Det.")
      yield_2_first <- fm_det_row$Y.II.
      recalc_etr_2 <- calc_etr(yield_2_first, 0, etr_factor, fraction_photosystem_II)


      result <- data.table::data.table(
        par = numeric(),
        yield_1 = numeric(),
        yield_2 = numeric(),
        etr_1 = numeric(),
        etr_2 = numeric()
      )
      new_row <- list(
        par = 0,
        yield_1 = yield_1_first,
        yield_2 = yield_2_first,
        etr_1 = recalc_etr_1,
        etr_2 = recalc_etr_2
      )
      result <- rbind(result, new_row)

      last_par <- as.numeric(0)
      for (i in seq_len(nrow(data))) {
        row <- data[i, ]
        current_par <- row$PAR

        if (row$Action != "P.+F. SP") {
          next
        }

        if (remove_recovery && last_par != 0 && current_par < last_par) {
          break
        }

        yield_1 <- row$Y.I.
        recalc_etr_1 <- calc_etr(yield_1, current_par, etr_factor, fraction_photosystem_I)

        yield_2 <- row$Y.II.
        recalc_etr_2 <- calc_etr(yield_2, current_par, etr_factor, fraction_photosystem_II)

        new_row <- list(
          par = current_par,
          yield_1 = yield_1,
          yield_2 = yield_2,
          etr_1 = recalc_etr_1,
          etr_2 = recalc_etr_2
        )
        result <- rbind(result, new_row)

        last_par <- current_par
      }

      validate_data(result)
      return(result)
    },
    warning = function(w) {
      stop("Warning in file: ", csv_path, " Warning: ", w)
    },
    error = function(e) {
      stop("Error in file: ", csv_path, " Error: ", e)
    }
  )
}

calc_etr <- function(yield, par, etr_factor, p_ratio) {
  if (is.na(yield)) {
    return(NA_real_)
  }

  if (!is.numeric(yield)) {
    stop("yield is not numeric")
  }

  if (!is.numeric(par)) {
    stop("par is not numeric")
  }

  if (!is.numeric(etr_factor)) {
    stop("etr_factor is not numeric")
  }

  if (!is.numeric(p_ratio)) {
    stop("p_ratio is not numeric")
  }

  return(yield * par * etr_factor * p_ratio)
}

#' Read and Process Junior PAM Data
#'
#' Reads raw CSV files generated by Junior PAM software, calculates electron transport rate (ETR) values, and returns a cleaned dataset. Customization may be needed for non-DualPAM devices.
#'
#' @param csv_path File path to the CSV file.
#' @param remove_recovery Logical. Removes recovery measurements if \code{TRUE}. Default is \code{TRUE}.
#' @param etr_factor Numeric. Factor for ETR calculation. Default is \code{0.84}.
#' @param fraction_photosystem_I Numeric. Relative distribution of absorbed PAR to photosystem I. Default is \code{0.5}.
#' @param fraction_photosystem_II Numeric. Relative distribution of absorbed PAR to photosystem II. Default is \code{0.5}.
#'
#' @details
#' Calculates ETR II using:
#' \deqn{\text{ETR II} = \text{PAR} \cdot \text{ETR-Factor} \cdot \text{Fraction of Photosystem (II)} \cdot \text{Yield (II)}}
#'
#' A detailed documentation can be found under \url{https://github.com/biotoolbox/pam?tab=readme-ov-file}
#'
#' @return A \code{data.table} containing:
#' \itemize{
#'   \item \code{par}: Photosynthetically active radiation.
#'   \item \code{yield_1}: Yield for photosystem I.
#'   \item \code{yield_2}: Yield for photosystem II.
#'   \item \code{etr_1}: Calculated ETR for photosystem I.
#'   \item \code{etr_2}: Calculated ETR for photosystem II.
#' }
#'
#' @references{
#'   Heinz Walz GmbH. (2024). \emph{DUAL-PAM-100 DUAL-PAM/F MANUAL, 5th Edition, April 2024, Chapter 7 (pp. 162-172).}
#'   Heinz Walz GmbH, Effeltrich, Germany.
#'   Available at: \url{https://www.walz.com/files/downloads/dualpamed05.pdf}
#' }
#' @examples
#' path <- file.path(system.file("extdata/junior_pam", package = "pam"), "junior_pam_20250613.csv")
#' data <- read_junior_pam_data(path)
#' @export
read_junior_pam_data <- function(
    csv_path,
    remove_recovery = TRUE,
    etr_factor = 0.84,
    fraction_photosystem_I = 0.5,
    fraction_photosystem_II = 0.5) {
  if (fraction_photosystem_I + fraction_photosystem_II != 1) {
    stop("The sum of fraction_photosystem_I and fraction_photosystem_II must be equal 1.")
  }

  tryCatch(
    {
      data <- utils::read.csv(csv_path, sep = ";", dec = ".", skip = 1, header = TRUE)
      data <- data.table::as.data.table(data)

      validate_junior_pam_data(data)

      par_col <- grep("PAR", names(data), value = TRUE)
      if (length(par_col) == 1 && par_col != "PAR") {
        data.table::setnames(data, old = par_col, new = "PAR")
      }

      yield_2_col <- grep("Y..II.", names(data), value = TRUE)
      if (length(yield_2_col) == 1 && yield_2_col != "Y..II.") {
        data.table::setnames(data, old = yield_2_col, new = "Y.II.")
      }

      data <- data[data$Type == "FO" | data$Type == "F", ]
      data$Datetime <- as.POSIXct(data$Datetime, format = "%Y-%m-%d %H:%M:%OS", tz = "GMT")

      date_time_col_values <- c()
      for (i in seq_len(nrow(data))) {
        row <- data[i, ]

        date_time_row_value <- as.POSIXct(
          paste(row$Date, row$Time, sep = " "),
          tz = "GMT", "%d.%m.%y %H:%M:%S"
        )
        date_time_col_values <- c(date_time_col_values, date_time_row_value)
      }
      data$DateTime <- date_time_col_values
      data <- data[order(data$Datetime), ]

      result <- data.table::data.table(
        par = numeric(),
        yield_1 = numeric(),
        yield_2 = numeric(),
        etr_1 = numeric(),
        etr_2 = numeric()
      )
      last_par <- as.numeric(0)
      for (i in seq_len(nrow(data))) {
        row <- data[i, ]
        current_par <- row$PAR

        if (remove_recovery && last_par != 0 && current_par < last_par) {
          break
        }

        yield_2 <- row$Y.II.
        recalc_etr_2 <- calc_etr(yield_2, current_par, etr_factor, fraction_photosystem_II)

        new_row <- list(
          par = current_par,
          yield_1 = NA_real_,
          yield_2 = yield_2,
          etr_1 = NA_real_,
          etr_2 = recalc_etr_2
        )
        result <- rbind(result, new_row)

        last_par <- current_par
      }

      return(result)
    },
    warning = function(w) {
      stop("Warning in file: ", csv_path, " Warning: ", w)
    },
    error = function(e) {
      stop("Error in file: ", csv_path, " Error: ", e)
    }
  )
}
