% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/copulas.R
\name{dcopula}
\alias{dcopula}
\title{Joint density under a bivariate copula}
\usage{
dcopula(d1, d2, p1, p2, copula = cgaussian(0), log = FALSE)
}
\arguments{
\item{d1, d2}{Marginal density values. If \code{log = TRUE}, supply the log-density.
If \code{log = FALSE}, supply the raw density.}

\item{p1, p2}{Marginal CDF values. Need not be supplied on log scale.}

\item{copula}{A function of two arguments (\code{u}, \code{v}) returning
the log copula density \eqn{\log c(u,v)}.
You can either construct this yourself or use the copula constructors available (see details)}

\item{log}{Logical; if \code{TRUE}, return the log joint density. In this case,
\code{d1} and \code{d2} must be on the log scale.}
}
\value{
Joint density (or log-density) under the bivariate copula.
}
\description{
Computes the joint density (or log-density) of a bivariate distribution
constructed from two arbitrary margins combined with a specified copula.
}
\details{
The joint density is
\deqn{f(x,y) = c(F_1(x), F_2(y)) \, f_1(x) f_2(y),}
where \eqn{F_i} are the marginal CDFs, \eqn{f_i} are the marginal densities,
and \eqn{c} is the copula density.

The marginal densities \code{d1}, \code{d2} and CDFs \code{p1}, \code{p2}
must be differentiable for automatic differentiation (AD) to work.

Available copula constructors are:
\itemize{
\item \code{\link{cgaussian}} (Gaussian copula)
\item \code{\link{cclayton}} (Clayton copula)
\item \code{\link{cgumbel}} (Gumbel copula)
\item \code{\link{cfrank}} (Frank copula)
}
}
\examples{
# Normal + Exponential margins with Gaussian copula
x <- c(0.5, 1); y <- c(1, 2)
d1 <- dnorm(x, 1, log = TRUE); d2 <- dexp(y, 2, log = TRUE)
p1 <- pnorm(x, 1); p2 <- pexp(y, 2)
dcopula(d1, d2, p1, p2, copula = cgaussian(0.5), log = TRUE)

# Normal + Beta margins with Clayton copula
x <- c(0.5, 1); y <- c(0.2, 0.8)
d1 <- dnorm(x, 1, log = TRUE); d2 <- dbeta(y, 2, 1, log = TRUE)
p1 <- pnorm(x, 1); p2 <- pbeta(y, 2, 1)
dcopula(d1, d2, p1, p2, copula = cclayton(2), log = TRUE)

# Normal + Beta margins with Gumbel copula
x <- c(0.5, 1); y <- c(0.2, 0.4)
d1 <- dnorm(x, 1, log = TRUE); d2 <- dbeta(y, 2, 1, log = TRUE)
p1 <- pnorm(x, 1); p2 <- pbeta(y, 2, 1)
dcopula(d1, d2, p1, p2, copula = cgumbel(1.5), log = TRUE)

# Normal + Exponential margins with Frank copula
x <- c(0.5, 1); y <- c(1, 2)
d1 <- dnorm(x, 1, log = TRUE); d2 <- dexp(y, 2, log = TRUE)
p1 <- pnorm(x, 1); p2 <- pexp(y, 2)
dcopula(d1, d2, p1, p2, copula = cfrank(2), log = TRUE)
}
\seealso{
\code{\link[=ddcopula]{ddcopula()}}, \code{\link[=dmvcopula]{dmvcopula()}}
}
