#' @title Harmonic Mean (HM)
#' @description
#' This function computes Harmonic Mean (HM) using any traits (like yield) under stress and non-stress conditions. The lower values of HM indicates greater tolerance. For more details see Bidinger et al. (1987) <doi:10.1071/AR9870037>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A data frame with \code{Gen}, \code{HM}.
#'
#' \deqn{HM = \frac{2 \times YN \times YS}{YN + YS}}
#'
#' @references Bidinger, F.R., Mahalakshmi, V. and Rao, G.D.P. (1987). <doi:10.1071/AR9870037>.
#' @examples
#' out = HM(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
HM <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  denom <- YN + YS
  out <- 2 * (YN * YS) / denom
  out[denom == 0] <- NA_real_
  return(data.frame(Gen = as.character(Gen), HM = out, row.names = NULL))
}
