#
#  growth : A Library of Normal Distribution Growth Curve Models
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     elliptic(response, model="linear", times=NULL, dose=NULL,
#	ccov=NULL, tvcov=NULL, nest=NULL, torder=0, interaction=NULL,
#	transform="identity", link="identity", autocorr="exponential",
#	pell=NULL, preg=NULL, pvar=1, varfn=NULL, par=NULL, pre=NULL,
#	delta=NULL, print.level=0, ndigit=10, gradtol=0.00001,
#	steptol=0.00001, iterlim=100, fscale=1,
#	stepmax=10*sqrt(theta%*%theta),typsiz=abs(c(theta)))
#
#  DESCRIPTION
#
#    Function to fit the multivariate elliptical distribution with
# various autocorrelation functions, one or two levels of random
# effects, and nonlinear regression.

elliptic <- function(response, model="linear", times=NULL, dose=NULL,
	ccov=NULL, tvcov=NULL, nest=NULL, torder=0, interaction=NULL,
	transform="identity", link="identity", autocorr="exponential",
	pell=NULL, preg=NULL, pvar=1, varfn=NULL, par=NULL, pre=NULL,
	delta=NULL, print.level=0, ndigit=10, gradtol=0.00001,
	steptol=0.00001, iterlim=100, fscale=1,
	stepmax=10*sqrt(theta%*%theta),typsiz=abs(c(theta))){
plra <- function(theta){
	if(mdl==2)mu <- model(theta)
	if(cvar==1)var <- varfn(theta[n1:n2])
	z <- .Fortran("plra",
		theta=as.double(theta),
		like=double(1),
		as.double(rxl),
		x=as.double(times),
		as.double(y),
		tvcov=as.double(zna$tvcov$tvcov),
		ccov=as.double(zna$ccov$ccov),
		dose=as.double(dose),
		nobs=as.integer(zna$response$nobs),
		nest=as.integer(zna$response$nest),
		lnest=as.integer(lnest),
		dev=double(nm),
		nind=as.integer(nind),
		nld=as.integer(nld),
		nxrl=as.integer(nxrl),
		np=as.integer(np),
		npell=as.integer(npell),
		npv=as.integer(npv),
		npvl=as.integer(npvl),
		nccov=as.integer(nccov),
		npvar=as.integer(npvar),
		cvar=as.integer(cvar),
		npre=as.integer(npre),
		npar=as.integer(npar),
		link=as.integer(lnk),
		torder=as.integer(torder),
		inter=as.integer(interaction),
		model=as.integer(mdl),
		ar=as.integer(ar),
		tvc=as.integer(tvc),
		beta=as.double(beta),
		betacov=betacov,
		v=double(nld*nld),
		sigsq=double(nld),
		ey=double(nld),
		tb=double(npvl),
		as.double(mu),
		as.double(var))
	list(like=z$like,res=z$dev,beta=z$beta,betacov=z$betacov)}
plral <- function(theta){
	if(mdl==2)mu <- model(theta)
	if(cvar==1)var <- varfn(theta[n1:n2])
	z <- .Fortran("plra",
		theta=as.double(theta),
		like=double(1),
		as.double(rxl),
		x=as.double(times),
		as.double(y),
		tvcov=as.double(zna$tvcov$tvcov),
		ccov=as.double(zna$ccov$ccov),
		dose=as.double(dose),
		nobs=as.integer(zna$response$nobs),
		nest=as.integer(zna$response$nest),
		lnest=as.integer(lnest),
		dev=double(nm),
		nind=as.integer(nind),
		nld=as.integer(nld),
		nxrl=as.integer(nxrl),
		np=as.integer(np),
		npell=as.integer(npell),
		npv=as.integer(npv),
		npvl=as.integer(npvl),
		nccov=as.integer(nccov),
		npvar=as.integer(npvar),
		cvar=as.integer(cvar),
		npre=as.integer(npre),
		npar=as.integer(npar),
		link=as.integer(lnk),
		torder=as.integer(torder),
		inter=as.integer(interaction),
		model=as.integer(mdl),
		ar=as.integer(ar),
		tvc=as.integer(tvc),
		beta=as.double(beta),
		betacov=betacov,
		v=double(nld*nld),
		sigsq=double(nld),
		ey=double(nld),
		tb=double(npvl),
		as.double(mu),
		as.double(var))
	z$like}
call <- sys.call()
if(!is.function(model)&&!missing(model))
	model <- match.arg(model,c("linear","logistic","pkpd"))
tmp <- c("exponential","gaussian","cauchy","spherical","IOU")
ar <- match(autocorr <- match.arg(autocorr,tmp),tmp)
transform <- match.arg(transform,c("identity","exp","square","sqrt","log"))
tmp <- c("identity","exp","square","sqrt","log")
lnk <- match(link <- match.arg(link,tmp),tmp)
mu <- NULL
var <- NULL
n1 <- length(preg)+1
n2 <- length(c(preg,pvar))
npell <- !missing(pell)
if(npell&&pell<=0)stop("The elliptic power parameter must be positive.")
npvar <- length(pvar)
if(!missing(pre))npre <- length(pre)
else npre <- 0
npar <- ifelse(!missing(par),1,0)
tvc <- ifelse(!missing(tvcov),1,0)
if(is.function(model)){
	if(is.null(preg))stop("when model is a function, preg must be supplied")
	mdl <- 2}
else if(model=="linear"){
	mdl <- 1
	torder <- torder+1}
else if(model=="logistic")mdl <- 3
else if(model=="pkpd")mdl <- 4
linear <- NULL
if(missing(ccov)){
	nccov <- 0
	zc <- NULL}
else {
	if(inherits(ccov,"tccov"))zc <- ccov
	else {
		if(is.vector(ccov,mode="double")||(is.matrix(ccov)&&is.null(colnames(ccov))))
			ccname <- paste(deparse(substitute(ccov)))
		else ccname <- NULL
		zc <- tcctomat(ccov,names=ccname)}
	nccov <- ncol(zc$ccov)}
if(mdl==1&&!missing(interaction)){
	if(length(interaction)!=nccov)
		stop(paste(nccov,"interactions with time must be specified"))
	else if(any(interaction>torder-1))
		stop(paste("Interactions can be at most of order ",torder-1))
	else interaction <- interaction+1}
else interaction <- rep(1,nccov)
if(mdl==4&&tvc==0&&missing(dose))stop("Doses required for PKPD model")
full <- !is.null(ccov)
if(inherits(response,"response"))zr <- response
else zr <- restovec(response,times,nest=nest,delta=delta)
if(!missing(tvcov)){
	if(inherits(tvcov,"tvcov"))zt <- tvcov
	else {
		if(is.matrix(tvcov)||is.data.frame(tvcov)||
			(is.list(tvcov)&&is.null(colnames(tvcov[[1]]))))
			tvcname <- paste(deparse(substitute(tvcov)))
		else tvcname <- NULL
		zt <- tvctomat(tvcov,names=tvcname)}
	tvc <- ncol(zt$tvcov)}
else {
	tvc <- 0
	zt <- NULL}
zna <- rmna(response=zr, tvcov=zt, ccov=zc)
rm(zr,zt,zc)
y <- zna$response$y
times <- zna$response$times
nind <- length(zna$response$nobs)
nld <- max(zna$response$nobs)
if(!is.null(zna$response$nest))lnest <- max(zna$response$nest)
else {
	lnest <- 0
	zna$response$nest <- rep(1,length(y))}
if(mdl==2&&length(model(preg))!=sum(zna$response$nobs))
	stop("The mean function must provide an estimate for each observation")
if(mdl==1){
	ave <- mean(times)
	times <- times-ave}
else ave <- 0
if(full){
	rxl <- NULL
	for(i in 1:nind){
		tmp <- 1
		if(full)for(j in 1:ncol(zna$ccov$ccov))tmp <- tmp+zna$ccov$ccov[i,j]*2^(j-1)
		rxl <- c(rxl,tmp)}
	nxrl <- max(rxl)
	p <- preg}
else {
	nxrl <- 1
	rxl <- matrix(1,ncol=1,nrow=nind)
	p <- NULL}
nm <- sum(zna$response$nobs)
if(transform=="identity")jacob <- 0
else if(transform=="exp"){
	jacob <- -sum(y)
	y <- exp(y)}
else {
	if(any(y<0))stop("Nonpositive response values: invalid transformation")
	else if(transform=="square"){
		jacob <- -sum(log(y[y>0]))
		y  <- y^2}
	else if(transform=="sqrt"){
		jacob <- sum(log(y[y>0]))/2
		y <- sqrt(y)}
	else if(any(y==0))stop("Zero response values: invalid transformation")
	else if(transform=="log"){
		jacob <- sum(log(y))
		y <- log(y)}}
if(!is.null(zna$response$delta)){
	if(length(zna$response$delta)==1)jacob <- jacob-nm*log(zna$response$delta)
	else jacob <- jacob -sum(log(zna$response$delta))}
if(mdl==1){
	if(lnk==1&&missing(varfn)){
		npvl <- torder+sum(interaction)+tvc
		npv <- 0
		beta <- double(npvl)
		betacov <- matrix(0,ncol=npvl,nrow=npvl)}
	else {
		npv <- torder+sum(interaction)+tvc
		npvl <- 0
		betacov <- NULL
		if(missing(preg)){
			if(lnk!=1)stop(paste("Initial regression estimates must be supplied for the linear model with ",link,"link"))
			else stop("Initial regression estimates must be supplied for the linear model when there is a variance function")}
		else beta <- preg}}
else {
	npvl <- 0
	beta <- 0
	betacov <- NULL
	if(mdl==2)npv <- length(preg)
	else if(mdl==3)npv <- ifelse(tvc==0,4*nxrl,4+nxrl-1)
	else if(mdl==4){
		if(!is.function(varfn))npv <- 2+nxrl
		else npv <- 3}}
if(length(preg)!=npv)
	stop(paste(npv,"initial parameter estimates for growth curve must be supplied"))
if(missing(varfn))cvar <- 0
else if(is.function(varfn)){
	cvar <- 1
	if(length(varfn(pvar))!=sum(zna$response$nobs))
		stop("The variance function must provide an estimate for each observation")}
else if(varfn=="identity"){
	if(any(y<0))warning("identity variance function not recommended with negative responses")
	cvar <- 2}
else if(varfn=="square")cvar <- 3
else stop("Unknown variance function: choices are identity and square")
if(!(mdl==4&npvar==4)&&!cvar==1){
	if(any(pvar<=0))stop("All variance parameters must be positive")
	pvar <- log(pvar)}
theta <- c(preg,pvar)
if(npre>0){
	if(any(pre<=0))stop("All variance components must be positive")
	theta <- c(theta,log(pre))}
if(npar>0){
	if(par<=0|(par>=1&ar!=5))
		stop("Estimate of autocorrelation must lie between 0 and 1")
	theta <- if(ar!=5)c(theta,log(par/(1-par)))
		else c(theta,log(par))}
if(npell)theta <- c(theta,log(pell))
np <- npv+npvl*(mdl==1)+npvar+npre+npar+npell
if(fscale==1)fscale <- plral(theta)
z0 <- nlm(plral, p=theta, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
p <- z0$estimate
z <- plra(p)
like <- z$like+nm*log(pi)/2+jacob
if(np-npvl==1){
	nlcov <- 1/z0$hessian
	nlse <- sqrt(nlcov)}
else {
	if(any(is.na(z0$hessian)))a <- 0
	else a <- qr(z0$hessian)$rank
	if(a==np-npvl)nlcov <- solve(z0$hessian)
	else nlcov <- matrix(NA,ncol=np-npvl,nrow=np-npvl)
	nlse <- sqrt(diag(nlcov))}
if(length(nlse)>1)nlcorr <- nlcov/(nlse%o%nlse)
else nlcorr <- as.matrix(nlcov/nlse^2)
dimnames(nlcorr) <- list(seq(1,np-npvl),seq(1,np-npvl))
betase <- betacorr <- NULL
if(mdl==1){
	if(lnk==1&&cvar==0){
		bt <- exp(p[length(p)])
		if(npell>0)z$betacov <- z$betacov/(npvl*gamma(npvl/(2*bt)))*
			2^(1/bt)*gamma((npvl+2)/(2*bt))
		if(npvl==1)betase <- sqrt(z$betacov)
		else if(npvl>1)betase <- sqrt(diag(z$betacov))
		if(npvl>1){
			betacorr <- z$betacov/(betase%o%betase)
			dimnames(betacorr) <- list(seq(1,npvl),seq(1,npvl))}}
	else {
		betase <- nlse[1:npv]
		betacov <- nlcov[1:npv,1:npv]
		if(npvl>1){
			betacorr <- nlcorr[1:npv,1:npv]
			dimnames(betacorr) <- list(seq(1,npv),seq(1,npv))}}}
sigsq <- p[(npv+1):(npv+npvar)]
if(!(mdl==4&npvar==4))sigsq <- exp(sigsq)
if(npre>0)tausq <- exp(p[(npv+npvar+1):(npv+npvar+npre)])
else tausq <- 0
if(npar>0){
	rho <- exp(p[npv+npvar+npre+1])
	if(ar!=5)rho <- rho/(1+rho)}
else rho <- 0
z <- list(
	call=call,
	model=model,
	autocorr=autocorr,
	response=zna$response,
	transform=transform,
	torder=torder-1,
	interaction=interaction-1,
	ccov=zna$ccov,
	tvcov=zna$tvcov,
	full=full,
	link=link,
	maxlike=like,
	aic=like+np,
	df=nm-np,
	np=np,
	npell=npell,
	npv=npv,
	npvl=npvl,
	npvar=npvar,
	varfn=varfn,
	npar=npar,
	npre=npre,
	coefficients=p,
	beta=z$beta,
	betacov=z$betacov,
	betacorr=betacorr,
	betase=betase,
	nlse=nlse,
	nlcov=nlcov,
	nlcorr=nlcorr,
	sigsq=sigsq,
	tausq=tausq,
	rho=rho,
	residuals=z$res,
	grad=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z) <- "elliptic"
return(z)}

coefficients.elliptic <- function(z) z$coefficients
deviance.elliptic <- function(z) 2*z$maxlike
residuals.elliptic <- function(z) z$residuals

print.elliptic <- function(z, digits = max(3, .Options$digits - 3)) {
	if(!is.null(z$ccov$ccov))nccov <- ncol(z$ccov$ccov)
	else nccov <- 0
	if(z$npell==0)cat("\nMultivariate normal distribution\n")
	else cat("\nMultivariate elliptically contoured distribution\n")
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	if(z$code>2)cat("Warning: no convergence - error",z$code,"\n\n")
	cat("Number of subjects    ",length(z$response$nobs),"\n")
	cat("Number of observations",length(z$response$y),"\n")
	if(!is.function(z$model)){
		if(z$model=="linear"){
			if(z$torder>0){
				cat("\nPolynomial model\n")
				cat("Times centred at  ",mean(z$response$times),"\n\n")}
			else cat("\nLinear model\n\n")}
		else if(z$model=="logistic")cat("\nGeneralized logistic model\n\n")
		else if(z$model=="pkpd")cat("\nPKPD model\n\n")}
	cat("Transformation:",z$trans,"\n")
	cat("Link function: ",z$link,"\n\n")
	cat("-Log likelihood   ",z$maxlike,"\n")
	cat("Degrees of freedom",z$df,"\n")
	cat("AIC               ",z$aic,"\n")
	cat("Iterations        ",z$iterations,"\n\n")
	tvc <- !is.null(z$tvcov)
	if(!is.function(z$model))cat("Location parameters\n")
	if(is.language(z$ccov$linear))
		cat("Formula: ",deparse(z$ccov$linear),"\n")
	if(is.function(z$model)){
		t <- deparse(z$model)
		cat("Parameters for location function\n")
		cat(t[2:length(t)],"",sep="\n")
		coef.table <- cbind(z$coef[1:z$npv], z$nlse[1:z$npv])
		cname <- paste("p",1:z$npv,sep="")
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)}
	else if(z$model=="linear"){
		if(z$npell==1&&z$link=="identity"&&!is.function(z$varfn))
			cat("(Approximate s.e.)\n")
		tord <- z$torder+1+nccov
		if(tvc)tord <- tord+1
		coef.table <- cbind(z$beta,z$betase)
		cname <- "(Intercept)"
		if(z$torder>0)cname <- c(cname,paste("t^",1:z$torder,sep=""))
		if(nccov>0)for(i in 1:nccov){
			cname <- c(cname,colnames(z$ccov$ccov)[i])
			if(z$interaction[i]>0){
				cname <- c(cname,paste(colnames(z$ccov$ccov)[i],".t^",1:z$interaction[i],sep=""))}}
		if(tvc)cname <- c(cname,colnames(z$tvcov$tvcov))
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)
		if(z$npvl>1&&z$link=="identity"&&is.null(z$varfn)){
			cat("\nCorrelation matrix of linear parameters\n")
			print.default(z$betacorr, digits=digits)}}
	else if(z$model=="logistic"){
		coef.table <- cbind(z$coef[1:4], z$nlse[1:4])
		if(tvc)cname  <- c("kappa1","kappa3","kappa4","beta")
		else cname <- c("kappa1","kappa2","kappa3","kappa4")
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)
		if(z$full){
			if(tvc){
				coef.table <- cbind(z$coef[5:z$npv],z$nlse[5:z$npv])
				cname <- colnames(z$ccov$ccov)
				dimnames(coef.table) <- list(cname,c("estimate","se"))
				print.default(coef.table,digits=digits,print.gap=2)}
			else {
				for(i in 1:nccov){
					cat("   ",colnames(z$ccov$ccov)[i],"\n")
					coef.table <- cbind(z$coef[(i*4+1):((i+1)*4)],z$nlse[(i*4+1):((i+1)*4)])
					dimnames(coef.table) <- list(cname, c("estimate", "se"))
					print.default(coef.table, digits=digits, print.gap=2)}}}}
	else if(z$model=="pkpd"){
		coef.table <- cbind(z$coef[1:3], z$nlse[1:3])
		cname <- c("log k_a","log k_e","log V")
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)
		if(z$full){
			for(i in 1:nccov){
				cat("   ",colnames(z$ccov$ccov)[i],"\n")
				coef.table <- cbind(z$coef[i+3],z$nlse[i+3])
				dimnames(coef.table) <- list(cname[3], c("estimate", "se"))
				print.default(coef.table, digits=digits, print.gap=2)}}}
	if(is.function(z$varfn)){
		cat("\nVariance parameters for variance function\n")
		t <- deparse(z$varfn)
		cat(t[2:length(t)],"",sep="\n")
		coef.table <- cbind(z$coef[(z$npv+1):(z$npv+z$npvar)], z$nlse[(z$npv+1):(z$npv+z$npvar)])
		cname <- paste("p",1:z$npvar,sep="")
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)}
	else if(!is.function(z$model)&&z$model=="pkpd"&&z$npvar==4){
		cat("\nVariance parameters\n")
		coef.table <- cbind(z$sigsq, z$nlse[(z$npv+1):(z$npv+4)])
		cname <- c("log k_a","log k_e","log V","power")
		dimnames(coef.table) <- list(cname, c("estimate", "se"))
		print.default(coef.table, digits=digits, print.gap=2)}
	else {
		if(z$npvar==1)cat("\nVariance\n")
		else cat("\nVariance parameters\n")
		if(!is.null(z$varfn)){
			cat(z$varfn,"function of the location parameter\n")
			vname <- "factor"
			if(z$npvar==2)vname <- c("constant",vname)}
		else if(z$npvar>1)
			vname <- c("(Intercept)",paste("t^",1:(z$npvar-1),sep=""))
		else vname <- ""
		coef.table <- cbind(z$coef[(z$npv+1):(z$npv+z$npvar)],
			z$nlse[(z$npv+1):(z$npv+z$npvar)],z$sigsq)
		dimnames(coef.table) <- list(vname,c("estimate","se","sigsq"))
		print.default(coef.table, digits=digits, print.gap=2)}
	if(z$npre>0){
		cat("\nVariance components\n")
		coef.table <- cbind(z$coef[(z$npv+z$npvar+1):(z$npv+z$npvar+z$npre)],z$nlse[(z$npv+z$npvar+1):(z$npv+z$npvar+z$npre)],z$tausq)
		if(z$npre==1)cname <- "tausq"
		else cname <- c("Level 1","Level 2")
		dimnames(coef.table) <- list(cname, c("estimate","se",""))
		print.default(coef.table, digits=digits, print.gap=2)}
	if(z$rho!=0){
		cat("\n",z$autocorr," autocorrelation\n",sep="")
		coef.table <- cbind(z$coef[z$npv+z$npvar+z$npre+1],z$nlse[z$npv+z$npvar+z$npre+1],z$rho)
		dimnames(coef.table) <- list("rho",c("estimate","se",""))
		print.default(coef.table, digits=digits, print.gap=2)}
	if(z$npell==1){
		cat("\nElliptical distribution power parameter\n")
		coef.table <- cbind(z$coef[z$np-z$npvl],z$nlse[z$np-z$npvl],exp(z$coef[z$np-z$npvl]))
		dimnames(coef.table) <- list(1,c("estimate","se","power"))
		print.default(coef.table, digits=digits, print.gap=2)}
	if(z$np-z$npvl>1){
		cat("\nCorrelation matrix of nonlinear parameters\n")
		print.default(z$nlcorr, digits=digits)}
	invisible(z)}
