# LogisRayleigh.R
#' Logistic-Rayleigh Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Rayleigh distribution.
#'
#' The Logistic-Rayleigh distribution is parameterized by the parameters
#' \eqn{\alpha > 0} and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Rayleigh distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda) =
#'  1 - \frac{1}{{1 + {{\left( {{e^{(\lambda {x^2}/2)}} - 1}
#' \right)}^\alpha }}}   \, ; \quad x \geq 0.
#' }
#'
#' where \eqn{\alpha} and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.rayleigh()} — Density function
#'   \item \code{plogis.rayleigh()} — Distribution function
#'   \item \code{qlogis.rayleigh()} — Quantile function
#'   \item \code{rlogis.rayleigh()} — Random generation
#'   \item \code{hlogis.rayleigh()} — Hazard function
#' }
#'
#' @usage
#' dlogis.rayleigh(x, alpha, lambda, log = FALSE)
#' plogis.rayleigh(q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.rayleigh(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.rayleigh(n, alpha, lambda)
#' hlogis.rayleigh(x, alpha, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.rayleigh}: numeric vector of (log-)densities
#'   \item \code{plogis.rayleigh}: numeric vector of probabilities
#'   \item \code{qlogis.rayleigh}: numeric vector of quantiles
#'   \item \code{rlogis.rayleigh}: numeric vector of random variates
#'   \item \code{hlogis.rayleigh}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary, A.K., & Kumar, V. (2020). 
#' The Logistic - Rayleigh Distribution with Properties and Applications. 
#' \emph{International Journal of Statistics and Applied Mathematics}, 
#' \bold{5(6)}, 12--19. 
#' \doi{10.22271/maths.2020.v5.i6a.603}  
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dlogis.rayleigh(x, 2.0, 5.0)
#' plogis.rayleigh(x, 2.0, 5.0)
#' qlogis.rayleigh(0.5, 2.0, 5.0)
#' rlogis.rayleigh(10, 2.0, 5.0)
#' hlogis.rayleigh(x, 2.0, 5.0)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=2.6967, lambda=0.0291)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.rayleigh, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.rayleigh, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.rayleigh, pfun=plogis.rayleigh, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisRayleigh
#' @aliases dlogis.rayleigh plogis.rayleigh qlogis.rayleigh rlogis.rayleigh hlogis.rayleigh
#' @export
NULL

#' @rdname LogisRayleigh
#' @usage NULL
#' @export   
dlogis.rayleigh <- function (x, alpha, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- exp((lambda * x[valid] * x[valid])/2.0)
    num <- lambda * alpha * u * ((u - 1.0)^(alpha - 1.0)) * x[valid]
    deno <- (1.0 + (u - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisRayleigh
#' @usage NULL
#' @export 
plogis.rayleigh <- function (q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp((lambda * q[valid] * q[valid])/2.0)
    tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0 - (1.0/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)   
}

#' @rdname LogisRayleigh
#' @usage NULL
#' @export 
qlogis.rayleigh <- function (p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
        qtl[valid] <- ((2.0/lambda)*(log(1.0 +((p[valid]/(1.0-p[valid])) ^ (1.0/alpha))))) ^ 0.5      
	}
    return(qtl)  
}

#' @rdname LogisRayleigh
#' @usage NULL
#' @export 
rlogis.rayleigh <- function(n, alpha, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")

    u <- runif(n)
    q <- qlogis.rayleigh(u, alpha, lambda)
    return(q)
}

#' @rdname LogisRayleigh
#' @usage NULL
#' @export 
hlogis.rayleigh <- function (x, alpha, lambda)
{
    if (!all(sapply(list(x, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dlogis.rayleigh(x, alpha, lambda) 
    surv <- 1.0 - plogis.rayleigh(x, alpha, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 
