#' Create a Lee-Carter Model with a Neural Network
#'
#' Utility function to initialize a \code{NNMoMo} object representing a neural
#' network Lee-Carter model. Important settings of the neural network can be
#' specified here.
#'
#' @param activation Sets the activation function of the \eqn{k_t}-subnet of the
#' neural network. The first activation function is set by this parameter, the
#' second is linear by default. "tanh" defines a tangens hyperbolicus activation
#' function, "relu" is a Rectified Linear Unit function and "linear" sets a
#' linear activation function.
#' @param model_type Defines the feed-forward architecture of the neural
#' network. "FCN" sets a fully connected network for the first layer of the
#' \eqn{k_t}-subnet, "CNN" sets a convolutional neural network layer for this
#' subset and "LCN" defines a locally connected network layer (note that
#' the "LCN" option has some stability issues and therefore the use of
#' "FCN" or "CNN" is recommended).
#' @param loss_type Defines the type of the loss function. As a default value,
#' "MSE" (Mean Squared Error) is set. This function minimizes the squared
#' difference between the observed and estimated mortality rates \eqn{m_{x,t}}.
#' "Poisson" evaluates the likelihood of the observed deaths.
#' @param q_e Sets the dimension of the embedding vectors for country and sex
#' speratly. The dimension of the full embedding vector is hence 2*q_e. A value
#' between 5 and 15 is recommended.
#' @param q_z1 Sets the dimension of the first layer of the \eqn{k_t}-subnet. A
#' value between 5 and 50 is recommended.
#' @return An object of class \code{NNMoMo}.
#' @examples
#' # Sets up the neural network with the recommended settings:
#' model_1 <- lcNN()
#'
#' # Sets up a fully customized netural network:
#' model_2 <- lcNN(activation = "tanh",
#'               model_type = "CNN",
#'               loss_type = "Poisson",
#'               q_e = 5,
#'               q_z1 = 25)
#'
#' # These models can be used for fitting.
#' @export
lcNN <- function(activation = c("linear", "tanh", "relu"),
                 model_type = c("FCN", "LCN", "CNN"),
                 loss_type = c("MSE", "Poisson"),
                 q_e = 10,
                 q_z1 = 10) {

  obj = list(
    activation = match.arg(activation),
    model_type = match.arg(model_type),
    loss_type = match.arg(loss_type),
    q_e = q_e,
    q_z1 = q_z1
  )

  class(obj) = "NNMoMo"

  return(obj)
}
