#' @title Simulates Correlation matrix by a given SEM model.
#' @description
#' This function generates the model-implied covariance and correlation matrices based on a specified structural equation model (SEM).
#' The function returns the implied covariance and correlation matrices.
#'
#' @author Fatih Orçan
#' @importFrom lavaan lavInspect sem
#' @importFrom utils write.table
#' @importFrom stats cov2cor
#' @param Model A lavaan model object specifying the measurement and structural components of the SEM.
#' @param nobs  An integer indicating the number of observed indicators (xs) in the model.
#' @return Returns model implied covariance and correlation matrices.
#'
#' @export
#' @examples
#' LavaanM <- '
#' # Measurement model (fixed factor loading)
#' F1 =~ 0.7*x1 + 0.7*x2 + 0.7*x3
#' F2 =~ 0.7*x4 + 0.7*x5 + 0.7*x6
#' F3 =~ 0.7*x7 + 0.7*x8 + 0.7*x9
#' # Structural regressions
#' F2 ~ 0.4*F1
#' F3 ~ 0.6*F1
#' # Fix latent variances
#' F1 ~~ 1*F1
#' # Residual variances
#' F2 ~~ 1*F2
#' F3 ~~ 1*F3
#' # Correlated residuals
#' F2 ~~ 0.5*F3
#' '
#'
#' cov.mtx(Model=LavaanM, nobs=9)

cov.mtx<-function(Model, nobs){

  if (!is.numeric(nobs) || nobs < 1) stop("nobs must be a positive integer.")

  nx<-nobs

  # Create empty covariance matrix (identity).
  x.names <- paste0("x", 1:nx)
  emp.cov <- diag(1,nx)
  rownames(emp.cov) <- colnames(emp.cov) <- x.names

  # Fit model without data (just to extract implied matrices)
  sonuc <- sem(model = Model, sample.cov = emp.cov, sample.nobs = 1000)

  B    <- lavInspect(sonuc, "coef")$beta    # regression paths among latent vars
  Psi  <- lavInspect(sonuc, "coef")$psi     # residual covariances among latent vars

  # Compute implied latent covariance: Sigma_eta = (I-B)^-1 * Psi * (I-B)^-T
  I <- diag(nrow(B))
  ifelse(is.null(B), fcov<-Psi, fcov<-solve(I - B) %*% Psi %*% t(solve(I - B)))

   # fcov <- solve(I - B) %*% Psi %*% t(solve(I - B))

  # Covariance and correlation matrices


  fcor<-cov2cor(fcov)
  fcor<-round(fcor,4)
  fcov<-round(fcov,4)

  message("The covariance matrix:")
  print(fcov)
  message("The correlation matrix:")
  print(fcor)

  return(list(cov = fcov, cor = fcor))
}
